use warnings;
use strict;

use utf8;

package NotificationTools;

require Exporter;

our @ISA = qw(Exporter);
our @EXPORT = qw(
    validate_date_interval
    get_emails_by_cids
    get_emails_by_uids
    get_emails_by_logins
    show_mail_logs
    show_sms_logs
    get_java_and_perl_email_templates
);

use DateTime;
use List::MoreUtils qw/ uniq /;
use PrimitivesIds qw/ get_uids /;
use Settings;
use ShardingTools;
use EnvTools qw/ is_beta /;
use User qw/ get_user_data /;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::ListUtils qw/ xuniq /;
use Yandex::Overshard qw/ overshard /;
use Yandex::MailTemplate qw/ get_email_template_list /;

=pod

    $Id$

=head1 NAME

    NotificationTools

=head1 DESCRIPTION

    Cобрание разнообразных вспомогательных функций для работы с нотификациями

=head1 METHODS

=head2 validate_date_interval

    Функция для проверки дат начала и окончания интервала, а также его длительности

    Параметры:
        date_from  - дата в формате 'YYYY-MM-DD' или 'YYYMMDD', необязательный, если не передан, то начало вчерашнего дня
        date_to    - дата в формате 'YYYY-MM-DD' или 'YYYMMDD', необязательный, если не передан, то начало завтрашнего дня
        limit_days - максимальная длительность интервала в днях, число, необязательный

    Результат:
        Даты начала и окончания интервала в виде объектов DateTime

=cut

sub validate_date_interval {
    my ( %params ) = @_;

    my $date_from;
    if ( defined $params{date_from} ) {
        $date_from = eval { Yandex::DateTime::date( $params{date_from} ) };
        die sprintf("Начальная дата (%s) задана не корректно. Требуемый формат: YYYY-MM-DD\n", $params{date_from}) if $@;
    } else {
        $date_from = Yandex::DateTime::now(); 
        $date_from->subtract(days => 1)->truncate(to => 'day');
    }

    my $date_to;
    if ( defined $params{date_to} ) {
        $date_to = eval { Yandex::DateTime::date( $params{date_to} ) };
        die sprintf("Конечная дата (%s) задана не корректно. Требуемый формат: YYYY-MM-DD\n", $params{date_to}) if $@;
        $date_to->add(days => 1); # set next day, because in SELECT will be used operator '<'
    } else {
        $date_to = Yandex::DateTime::now(); 
        $date_to->add(days => 1)->truncate(to => 'day');
    }

    if ( DateTime->compare( $date_to, $date_from ) < 0 ) {
        die sprintf("Конечная дата '%s' меньше начальной '%s'\n", $date_to->ymd(), $date_from->ymd());
    }

    if ( defined( $params{limit_days} ) and $date_to->delta_days( $date_from )->in_units('days') > $params{limit_days} ) {
        die "Задан слишком большой интервал времени\n";
    }

    return ( $date_from, $date_to );
}

=head2 get_emails_by_cids

    Функция для получения email по id кампании, на входе - ссылка на массив id кампаний, на выходе - ссылка на массив email

=cut

sub get_emails_by_cids {
    my ( $cids ) = @_;

    return [] unless $cids;

    my @cids = uniq @$cids;

    my @emails;
    push @emails, @{ get_one_column_sql( PPC(cid => \@cids), [ "SELECT email FROM camp_options", WHERE => { cid => SHARD_IDS } ] ) };
    push @emails, @{ get_one_column_sql( PPC(cid => \@cids), [ "SELECT email FROM campaigns JOIN users USING(uid)", WHERE => { cid =>  SHARD_IDS } ] ) };

    return \@emails;
}

=head2 get_emails_by_uids

    Функция для получения email по id пользователя, на входе - ссылка на массив id пользователей, на выходе - ссылка на массив email

=cut

sub get_emails_by_uids {
    my ( $uids ) = @_;

    return [] unless $uids;

    return get_one_column_sql( PPC(uid => [ uniq @$uids ]), [ "SELECT email FROM users", WHERE => { uid => SHARD_IDS } ] );
}

=head2 get_emails_by_logins

    Функция для получения email по login пользователя, на входе - ссылка на массив login пользователей, на выходе - ссылка на массив email

=cut

sub get_emails_by_logins {
    my ( $logins ) = @_;

    return [] unless $logins;

    my @emails;

    my @client_ids;
    foreach my $uid ( @{ get_uids(login => [ uniq @$logins ]) } ) {
        my $user = get_user_data( $uid, [qw/email ClientID/] ) || {};
        push @emails, $user->{email} if $user->{email};

        # from reps
        push @client_ids, $user->{ClientID} if $user->{ClientID};
    }

    if ( @client_ids ) {

        @client_ids = uniq @client_ids;

        push @emails, @{ get_one_column_sql( PPC(ClientID => \@client_ids), [ "SELECT email FROM users", WHERE => { ClientID => SHARD_IDS } ] ) };

        # from campaigns
        push @emails, @{ get_one_column_sql( PPC(ClientID => \@client_ids),
                                                [ "SELECT
                                                        co.email
                                                    FROM
                                                        camp_options co
                                                            JOIN campaigns c USING(cid)
                                                            JOIN users u USING(uid)
                                                    ",
                                                    WHERE => { 'u.ClientID' => SHARD_IDS }
                                                ]
                                            )
                        };
    }

    return \@emails;
}


=head2 show_mail_logs

    Функция для получения информации об отправленных письмах из лога.

    На вход получает ссылку на хеш с параметрами:
        emails         - ссылка на массив email-ов,
        template_names - ссылка на массив названий шаблонов писем, которые требуются
        date_from      - начало периода, за который выбирается информация, дата в виде объекта DateTime, обязательный параметр
        date_to        - окончание периода, за который выбирается информация, дата в виде объекта DateTime, обязательный параметр

    Должен быть указан хотя бы одни из аргументов emails и template_names.

    Если писем, подпадающих по заданные условия, не найдено - возвращает ссылку на пустой массив, иначе структуру вида
        [
            {
                email            => ...,
                template_name    => ...,
                subject          => ...,
                content          => ..., # в UTF-8
                logtime_format   => ..., # в формате '%d.%m.%Y %T'
            },
            ...
        ]

=cut

sub show_mail_logs {
    my ( $params ) = @_;

    my ( $date_from, $date_to, $emails, $templates ) = @$params{qw/ date_from date_to emails template_names /};

    my $clh = Tools::get_clickhouse_handler('cloud');

    my @queries;
    if ( $emails ) {
        foreach my $email ( xuniq { lc $_ } sort @$emails ) {
            push @queries, "lower(email) LIKE ". $clh->quote_str( lc($email) .'%' );
        }
    }

    my @logs;
    if ( @queries || @$templates ) {

        my @where;

        if ( @queries ) {
            push @where, {OR => \@queries};
        }

        if ( $templates ) {
            push @where, {template_name__in => $templates};
        }

        if ( EnvTools::is_beta() && $Settings::CONFIGURATION ne 'test' ) {
            # host проставляется только из perl на ppcdev'ах, письма из java сюда не попадут и надо с этим что-то сделать.
            push @where, 'host LIKE' . $clh->quote_str("%:beta.%.$Settings::CONFIGURATION");
        } elsif ( $Settings::CONFIGURATION eq 'test' ) {   # ТС или бета, "повёрнутая" на ТС
            # считаем, что в тестинге всё, что не обозначено бетой -- это ТС
            push @where, {OR => ['host NOT LIKE ' . $clh->quote_str('%:beta.%'), 'host LIKE ' . $clh->quote_str('%:beta.%.test')]};
        }

        my $limit_sql = @queries ? '' : "LIMIT 100";

        my $date = $date_from;
        # Выбираем за каждую дату отдельно, потому что раньше в mysql ppclog были отдельные таблицы на каждый день, и хотелось обратной совместимости с поведением, когда за какую-нибудь дату работает LIMIT.
        while ($date <= $date_to) {
            my @sql_parts = ("SELECT email
                                , template_name
                                , subject
                                , content
                                , client_id
                                , formatDateTime(log_time, '%d.%m.%Y&nbsp;%T') AS logtime_format
                                , toUnixTimestamp(log_time) AS logtime_unixtime FROM",
                            "mails",
                            WHERE => {
                                log_date => $date->strftime('%Y-%m-%d'),
                                AND => \@where,
                            },
                            "ORDER BY log_time",
                            $limit_sql,
                            "FORMAT JSON");
            my $result = $clh->query(\@sql_parts);
            push @logs, @{ $result->json->{data} }; 
            $date = $date->add(days => 1)->truncate(to => 'day');
        }
    }

    return \@logs;
}

=head2 show_sms_logs

    Функция для получения данных об отправленных смс из очереди на отправку.

    На вход получает ссылку на хеш с параметрами:
        cids        - ссылка на массив id кампаний
        uids        - ссылка на массив id пользователей
        date_from   - начало периода, за который выбирается информация, дата в виде объекта DateTime, обязательный параметр
        date_to     - окончание периода, за который выбирается информация, дата в виде объекта DateTime, обязательный параметр
        by_add_time - если указан, то выборка идет по датам в колонке add_time, иначе sent_time

    Должен быть указан хотя бы одни из аргументов cids и uids.

    Если смс, подпадающих по заданные условия, не найдено - возвращает ссылку на пустой массив, иначе структуру вида
        [
            {
                cid         => ...,
                uid         => ...,
                sms_text    => ...,
                send_time   => ..., # в формате '%d.%m.%Y %T' 
                send_status => ...,
            },
            ...
        ]

=cut

sub show_sms_logs {
    my ( $params ) = @_;

    my ( $date_from, $date_to, $cids, $uids ) = @$params{qw/ date_from date_to cids uids /};

    my %cond;
    $cond{cid} = $cids if defined $cids;
    $cond{uid} = $uids if defined $uids;

    my $logs = [];
    if ( %cond ) {
        # если заполнено только поле логины, то показываем все записи с указанными uid
        # если заполнено только поле кампании, то показываем все записи по cid
        # если заполнены оба поля, то показываем записи, где есть и cid и uid

        if ( $params->{by_add_time} ) {
            $cond{add_time__ge} = $date_from->ymd();
            $cond{add_time__lt} = $date_to->ymd();
        } else {
            $cond{send_time__ge} = $date_from->ymd();
            $cond{send_time__lt} = $date_to->ymd();
        }

        $logs = get_all_sql(
            PPC( choose_shard_param( \%cond, [qw(uid cid)], allow_shard_all => 0, set_shard_ids => 1 ) ),
            [
                "SELECT sms_time, cid, sms_text, uid, send_status FROM sms_queue",
                WHERE => \%cond,
            ]
        );

        $logs = overshard( order => ['cid:num', 'send_time'], $logs );
    }

    return $logs;
}

=head2 get_java_and_perl_email_templates
    
    Функция для получения шаблонов писем с их описаниями на определенном языке. 
    
    Параметры:
        lang - код языка
 
    Результат:
        Ссылка на хеш название шаблона -> описание шаблона

=cut

sub get_java_and_perl_email_templates {
    my ($lang) = @_;
    my %templates = map { $_->{name} => $_->{description} } grep { $_->{name} !~ /^i_/ } @{ get_email_template_list($lang) };
    if ($lang eq "ru") {
        $templates{"deal_notification"} = "Уведомление по сделке";
    }
    return \%templates;
}

1;
