package OrgDetails;

# $Id$

=head1 NAME

    OrgDetails

=head1 DESCRIPTION

    Модуль с функциями для работы с детализированной информацией 

=cut

use strict;
use warnings;
use POSIX qw/floor/;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use PrimitivesIds;
use Yandex::ScalarUtils;
use Yandex::HashUtils;
use List::MoreUtils qw/any all/;

use base qw/Exporter/;
our @EXPORT = qw/
        get_user_org_details
        get_org_details
        is_empty_org_details
        find_org_details_by_fields
        add_org_details
        compare_org_details
        validate_ogrn
        clean_org_details

        expand_org_details
        separate_org_details
/;

my @VALUABLE_DETAILED_FIELDS = qw/ogrn/;     # Если хотябы одно из полей не пусто, запись стоит оставить в таблице. Иначе - удалить

my @DETAILD_FIELDS_DB = (@VALUABLE_DETAILED_FIELDS,
                       qw/org_details_id
                          uid/);

my $DETAILD_FIELDS_STR = join ", ", @DETAILD_FIELDS_DB;


=head2 get_user_org_details (uid)

    Возвращает список реквизитов, которые используются пользователям в его визитках.

=cut
sub get_user_org_details($) {
    my $uid = shift;
    return [] if ! defined($uid);
    return get_all_sql(PPC(uid => $uid), ["SELECT $DETAILD_FIELDS_STR FROM org_details", where => {uid => SHARD_IDS}]);
}

=head2 find_org_details_by_fields
    
    Возвращает org_details_id для набора детальных параметров, если такой присутствует в базе у пользователя   
        undef - в противном случае

=cut
sub find_org_details_by_fields($) {
    my ($details) = @_;

    my $where_condition =  {uid => $details->{uid}};
    hash_merge $where_condition, {map {$_."__is", $details->{$_}} @VALUABLE_DETAILED_FIELDS};

    return get_one_field_sql(PPC(uid => $details->{uid}), ["SELECT org_details_id FROM org_details", where => $where_condition ]);
}

=head2 get_org_details (org_details_id)

    Возвращает реквизиты организации по id номеру.

=cut
sub get_org_details($) {
    my $org_details_id = shift;
    return undef if ! defined($org_details_id);

    return get_one_line_sql(PPC(org_details_id => $org_details_id), [
        "SELECT $DETAILD_FIELDS_STR FROM org_details",
        where => {org_details_id => $org_details_id}
    ]);
}

=head2 get_org_details_multi (org_detail_ids)

    Возвращает реквизиты организаций по массиву id

=cut

sub get_org_details_multi {
    
    my $org_detail_ids = shift;
    return get_all_sql(PPC(org_details_id => $org_detail_ids), [
        "SELECT $DETAILD_FIELDS_STR FROM org_details",
        where => {org_details_id => SHARD_IDS}
    ]);
}


=head2 compare_org_details($new_details, $old_details)

    Сравнивает 2 набора детальной информации по значимым полям
    Поля сравниваются как строки
   
    Возвращяет 
        0 - Детальная информация одинаковая
        1 - Детальная информация разная

=cut
sub compare_org_details($$) {
    my ($details_1, $details_2) = @_;

    return ! all { str($details_1->{$_}) eq str($details_2->{$_})} @VALUABLE_DETAILED_FIELDS;
}

=head2 is_empty_org_details($details)

    Проверяет, есть ли в параметрах непустое значимое поле.
    Если есть, то данные надо сохранить в БД. 

=cut
sub is_empty_org_details($) {
    my $details = shift;

    return not any {$_} values %{hash_cut $details, @VALUABLE_DETAILED_FIELDS}; 
}

=head2 add_org_details (\%details)

    Добавляет в БД новый ОГРН если такого ещё не существует.
    Params
        $details->{uid}  - uid пользователя
        $details->{ogrn} - собственно ОГРН
    Return
        Идентификатор ОГРН в таблице (org_details_id)

=cut
sub add_org_details($) {
    my $details = shift;
    _check_org_details($details);
    return $details->{org_details_id} if ($details->{org_details_id});

    return undef if is_empty_org_details($details);

    # Проверяем есть ли уже такой ОГРН у данного пользователя. Если есть, то не добавляем, а используем уже имеющийся.
    my $existed_org_details_id = find_org_details_by_fields($details);
    return $existed_org_details_id if $existed_org_details_id;

    $details->{org_details_id} = get_new_id('org_details_id', uid => $details->{uid});
    do_insert_into_table(PPC(uid => $details->{uid}), 'org_details', $details);
    return $details->{org_details_id};
}

=head2 clean_org_details({uid => $uid} или {vcard_id => $vcard_id})

    Удалить все реквизиты, которые нигде не используются,
    т.е. ни одна vcard не имеет org_details_id, который есть в org_details для пользователя с uid.
    В случае, если не указан uid, но есть vcard_id, то uid берется из визитки с vcard_id

=cut
sub clean_org_details(%) {
    my %OPT = @_;

    my $uid;
    $uid = $OPT{uid};

    $uid = $uid || get_owner(vcard_id => $OPT{vcard_id}) if $OPT{vcard_id};

    return if !$uid;

    my $empty_orgs = get_one_column_sql(PPC(uid => $uid), ['
            SELECT od.org_details_id
              FROM org_details od
                   LEFT JOIN vcards vc ON vc.uid = od.uid
                                          AND vc.org_details_id = od.org_details_id
        ', where => {
            'od.uid' => $uid,
            'vc.org_details_id__is_null' => 1,
        }
    ]) || [];
    if (@$empty_orgs) {
        do_delete_from_table(PPC(uid => $uid), 'org_details', where => {org_details_id => $empty_orgs});
        delete_shard(org_details_id => $empty_orgs);
    }
}

=head2 validate_ogrn(ogrn)

    Проверяет ОГРН/ОГРНИП значение.
    Основной государственный регистрационный номер состоит из 13 цифр. Последняя, тринадцатая, и есть контрольная. 
    Делим число, состоящее из первых 12 знаков, на 11. Потом целую часть полученного результата умножаем на 11 
    и сравниваем с первоначальным 12-значным числом. Разница должна совпадать с 13-й цифрой ОГРН.

    Допустим, приходит регистрационный номер 1037556120338.

    Проверяем:

    103755612033 : 11 = 9432328366,636
    9432328366 x 11 = 103755612026.
    103755612033 – 103755612026 = 7.

    Последняя цифра в заявленном ОГРН равна 8. Значит такого ОГРН не существует.
    Примеры существующих ОГРН:
    1087746113344
    1087746385320
    1055406282513
    1067760833810 (внимание, они настоящие, взятые из интернета)

    ОГРНИП - основной государственный регистрационный номер индивидуального предпринимателя. 
    Состоит из 15 цифр, последняя - контрольная. От ОГРН отличается тем, что:
    1. Под номер записи в реестре выделено семь, а не пять цифр
    2. Контрольная цифра равна последней цифре остатка от деления на 13,а не на 11, предыдущего 14-значного числа
    Примеры существующих ОГРНИП:
    304540707500034
    304540220800032
    309774611900857
    310253706100022 (внимание, они настоящие, взятые из интернета)

=cut

sub validate_ogrn($) {
    my $ogrn = shift;
    #Неправильно, если ОГРН содержит не только цифры или если длина не равна 13 или 15 цифрами.
    return 0 if ($ogrn !~ /\d{13}\d{0,2}/ || !(length($ogrn) == 13 || length($ogrn)== 15));
    my $first_num = substr( $ogrn, 0, 1 );
    
    # Первая цифра ОГРН может быть только цифрами 1, 2, 3, 5
    return 0 if ($first_num !~ /[1235]/);

    my $last_num = substr( $ogrn, -1 );

    my $mult = (length($ogrn)== 13)?11:13;
    my $t1 = substr( $ogrn, 0, -1 );
    my $div = floor( int($t1) / $mult );
    my $t2 = $div * $mult;

    my $diff = abs($t1 - $t2);

    # Разница значений должна совпадать.
    # иногда разница может быть больше 9, тогда надо брать только последний символ для сравнения.
    return ($last_num eq substr($diff, -1 ))?1:0;

}


sub _check_org_details($) {
    my $details = shift;

    my @required_keys = qw/uid ogrn/;
    for my $key (@required_keys) {
        die "Required key [$_] is missed in hashref" unless exists $details->{$key};
    }

    my @optional_keys = qw/org_details_id/;

    my $allowed_keys_hash = {map {$_ => 1} @required_keys, @optional_keys};
    die sprintf "forbiden keys found: [%s]", join(',', grep {!exists $allowed_keys_hash->{$_}} keys %$details) if any {!exists $allowed_keys_hash->{$_}} keys %$details;
    
}


=head2 expand_org_details

    в визитке вместо поля org_details_id делает хеш с деталями по ОГРН

    Параметр -- ссылка на хеш с визиткой, он модифицируется in-place. 

    Если org_details_id в визитке нет -- ничего не делает.

    Возвращаемого значения нет.

=cut
sub expand_org_details($) {
    my ($vcard) = @_;

    return unless $vcard->{org_details_id};

    $vcard->{org_details} = hash_cut get_org_details($vcard->{org_details_id}), qw/ogrn org_details_id/;  
    delete $vcard->{org_details_id};

    return;
}

=head2 separate_org_details

    в плоской визитке делает хеш с деталями по ОГРН

    Параметр -- ссылка на хеш с визиткой, он модифицируется in-place. 

    Если ogrn в визитке нет -- ничего не делает.

    Возвращаемого значения нет.

=cut
sub separate_org_details($) {
    my ($vcard) = @_;

    return unless $vcard->{ogrn};

    $vcard->{org_details}->{$_} = delete $vcard->{$_} for qw/ogrn org_details_id/;

    return;
}

=head2 copy_org_details ($old_org_details_id, $old_chief_uid, $new_chief_uid)

    Копирует детали организации от одно пользователя - к другому. Вынесено из Campaign::Copy::copy_camp
    Параметры:
        old_org_details_id  - id записи, которую нужно скопировать
        old_chief_uid       - uid владельца записи (нажно для определения шарда для выборки записи)
        new_chief_uid       - uid пользователя, которому копируем "org details"

    Возвращает id новой (или имеющейся подходящей) записи или undef (например, если старая запись не найдена)

=cut
sub copy_org_details($$$) {
    my ($old_org_details_id, $old_chief_uid, $new_chief_uid) = @_;

    my $old_org_details = get_one_line_sql(PPC(uid => $old_chief_uid),
        ["SELECT $DETAILD_FIELDS_STR FROM org_details", where => { org_details_id => $old_org_details_id }]
    );

    if ($old_org_details && %$old_org_details) {
        # Функция сама проверит - есть ли уже в org_details существующая запись с такими uid и ogrn, и если нет - добавит новую и вернет ее id
        return add_org_details(hash_merge({uid => $new_chief_uid}, hash_cut($old_org_details, @VALUABLE_DETAILED_FIELDS)));
    } else {
        return undef;
    }
}

1;
