package PSGIApp::JSONRPC;

# $Id$

=head1 NAME


=head1 DESCRIPTION


=cut


use strict;
use warnings;

use Plack::Builder;
use Plack::Response;

use JSON::RPC::Dispatch;
use Router::Simple::Declare;

use EnvTools;
use Intapi;
use Rbac;
use RBACDirect;
use Yandex::I18n ();
use TvmChecker;
use SentryTools;

use utf8;

our $prefix = 'Intapi';

sub get_app
{
    my (%O) = @_;
    my $group = $O{group} || '';
    my $baseurl = $O{baseurl} || '';
    $O{profile_type} ||= '';

    # secret-jsonrpc предназначен только для dev-серверов + ТС
    # и в production не работает
    if ( is_production() && $Intapi::GROUP_DISABLED_IN_PRODUCTION{$group} ) {
        return sub { die "$group is disabled in production" };
    }

    my %DISPATCH;
    my $cmd_params = {};
    
    my $jsonrpc_commands = Intapi::filter_cmd(type => 'jsonrpc', group => $O{group});
 
    for my $resource (keys %$jsonrpc_commands){
        next if $Intapi::cmd{$resource}->{sandbox_only} && !is_sandbox();

        my $router = router {
            for my $method (@{$Intapi::cmd{$resource}->{methods}}){
                connect $method => {
                    handler => "+$Intapi::cmd{$resource}->{module}",
                    action  => $method,
                };
            }
        };
        
        $DISPATCH{$resource} = JSON::RPC::Dispatch->new(
            prefix => "",
            router => $router,
        );

        my $acl_name = is_production() ? 'allow_to' : 'allow_to_in_testing'; 
        if (exists $Intapi::cmd{$resource}->{$acl_name}){
            $cmd_params->{$resource}->{allow_to} = $Intapi::cmd{$resource}->{$acl_name};
        }

        my $prod = is_production();
        my $sb = is_sandbox();

        my ($tvm_list_name, $skip_tvm2_check_name);
        if ($prod) {
            if ($sb) {
                ($tvm_list_name, $skip_tvm2_check_name) = ('tvm2_allow_ids_sandbox', 'skip_tvm2_check_sandbox');
            } else {
                ($tvm_list_name, $skip_tvm2_check_name) = ('tvm2_allow_ids', 'skip_tvm2_check');
            }
        } else {
            if ($sb) {
                ($tvm_list_name, $skip_tvm2_check_name) = ('tvm2_allow_ids_sandbox_in_testing', 'skip_tvm2_check_sandbox_in_testing');
            } else {
                ($tvm_list_name, $skip_tvm2_check_name) = ('tvm2_allow_ids_in_testing', 'skip_tvm2_check_in_testing');
            }
        }
        if (exists $Intapi::cmd{$resource}->{$tvm_list_name}){
            $cmd_params->{$resource}->{tvm2_allow_ids} = $Intapi::cmd{$resource}->{$tvm_list_name};
            if (exists $Intapi::cmd{$resource}->{$skip_tvm2_check_name}) {
                $cmd_params->{$resource}->{skip_tvm2_check} = $Intapi::cmd{$resource}->{$skip_tvm2_check_name};
            }
        }
    }

    # основное приложение
    my $app = sub {
        my $env = shift;
        my $resource = $env->{resource};
        my $dispatch = $DISPATCH{$resource} or die "can't dispatch '$resource'";

        my $lang_guard = Yandex::I18n::init_i18n_guard(Intapi::INTAPI_LANG);

        local $Yandex::TVM2::APP_ID;      $Yandex::TVM2::APP_ID = $Settings::TVM2_APP_ID{intapi};
        local $Yandex::TVM2::SECRET_PATH; $Yandex::TVM2::SECRET_PATH = $Settings::TVM2_SECRET_PATH{intapi};
        no warnings 'once';
        local $Yandex::Blackbox::BLACKBOX_USE_TVM_CHECKER = \&TvmChecker::use_tvm;
        local $SIG{__DIE__} = SentryTools::prepare_die_handler();
        my $ret = $dispatch->handle_psgi($env);
        if (SentryTools::get_last_error()){
            SentryTools::send_last_exception(SentryTools::get_last_error(),
                skip => [qw/main PSGIApp::JSONRPC JSON::RPC::Dispatch Try::Tiny/],
                skip_re => [qr|^JSON::RPC::|, qr|^Plack::|],
            );
        }

        return fix_utf8_header($ret);
    };

    return builder {
        enable 'ExtractJsonrpcResource', cmd_params => $cmd_params;
        enable 'MemLimit', limit => $Settings::APACHE_MAX_PROC_SIZE;
        enable 'JsonrpcLogger';
        enable 'Trace', cmd_type => 'direct.jsonrpc', use_headers => 1;
        enable 'ACL', networks_dir => $Settings::NETWORKS_DIR,
                      error_log => [log_file_name => 'intapi_check_acl_errors.log', date_suf => '%Y%m%d'];
        enable 'CheckTVM2Ticket', app_id => $Settings::TVM2_APP_ID{intapi}, tvmtool_token_file => $Settings::TVMTOOL_TOKEN_FILE;
        enable sub {
            # нешардированные методы работают только с первым шардом
            my $_app = shift;
            sub {
                my $env = shift;
                Yandex::DBShards::clear_cache();
                Rbac::clear_cache();
                RBACDirect::clear_cache();
                my $one_shard = exists $Intapi::cmd{ $env->{resource} } && $Intapi::cmd{$env->{resource}}->{one_shard} ? 1 : 0;
                local $Yandex::DBShards::STRICT_SHARD_DBNAMES = $one_shard ? 0 : 1;
                my $res = $_app->($env);
                return $res;
            };
        };
        $app;
    };
}

=head2 fix_utf8_header

    В JSON::RPC::Dispatch кодировка указана не по стандерту - utf8 вместо UTF-8
    https://metacpan.org/release/JSON-RPC/source/lib/JSON/RPC/Dispatch.pm#L233
    Фиксим по месту.

=cut
sub fix_utf8_header {
    my $resp = shift;
    if (ref($resp) eq 'ARRAY' && ref($resp->[1]) eq 'ARRAY') {
        my $headers = $resp->[1];
        for(my $i = 0; $i < @$headers - 1; $i+=2) {
            if ($headers->[$i] eq 'Content-Type') {
                $headers->[$i+1] =~ s/charset=utf8/charset=UTF-8/;
            }
        }
    }
    return $resp;
}

1;
