package PSGIApp::LogJS;

=head1 NAME

    PSGIApp::LogJS

=head1 DESCRIPTION

    Конструктор psgi-приложения для логгирования данных из браузера, пришедших http-запросом

    Новый вариант - метаданные передаются в url: /log-js/<type>/<cmd>/<reqid>
    В теле POST запроса приходит json, который записывается в лог
    Поддерживаются запросы:
        render - скорость отрисовки страницы
        errors - js-ошибки на странице
        csp - отчёт Content-Security-Policy
        model-errors - лог неправильных присвоений полей моделей (DIRECT-46836)
        info - неспециализированный лог для временной отсылки статистики/дебага с продакшена; если какие-то данные оказываются там надолго -- стоит подумать про отдельный тип лога

    Старый вариант - данные передаются параметрами:
    Принимает параметры:
    - reqid - uint64 - id запроса, страница которого сгенерировала запись в лог
    - type - тип записи:
        render - скорость отрисовки страницы
        errors - js-ошибки на странице
    - data - валидный json, размером не больше $MAX_DATA_SIZE 


    Возврвщает:
    {"status": "ok"}

=cut


use strict;
use warnings;

use Yandex::Log;

use Settings qw//;
use Direct::ResponseHelper qw/respond_json/;
use DoCmd::Base;
use Plack::UTF8Request;
use Encode;

use utf8;

our $MAX_DATA_SIZE ||= 20 * 1024;
our %ADDITIONAL_CMDS = map {$_=>1} qw/
    index
    direct_ui
    PPCLoginBox
/;

sub get_app
{
    # основное приложение
    my $app = sub {
        my $env = shift;
        my $r = Plack::UTF8Request->new($env);
        if (my ($type, $cmd, $reqid) = $r->path_info =~ m{ ^ /* (csp|render|errors|info|model-errors) /+ ([\w\-\.]{0,128}) /+ (\d{0,30}) $ }x) {
            if (!exists $DoCmd::Base::cmds{$cmd} && !exists $ADDITIONAL_CMDS{$cmd}) {
                return [ 400, [ 'Content-Type' => 'text/html' ], [] ];
            }
            my $log_name = "js_$type";
            my $metadata = "$cmd/$reqid";
            my $data = Encode::decode_utf8($r->raw_body);
            if (!$data) {
                return [ 400, [ 'Content-Type' => 'text/html' ], [] ];
            }
            if (length($data) > $MAX_DATA_SIZE) {
                return [ 400, [ 'Content-Type' => 'text/html' ], [] ];
            }

            my $log = Yandex::Log->new(log_file_name => "$log_name.log", date_suf => '%Y%m%d');
            $log->out("$metadata\t$data");
        } else {
            return [ 404, [ 'Content-Type' => 'text/html' ], [] ];
        }

        return Direct::ResponseHelper::respond_json($r, {status => 'ok'});
    };

    return $app;
}

1;
