package PSGIApp::Public;

# $Id$

=head1 NAME

    PSGIApp::Public

=head1 DESCRIPTION

    Конструктор psgi-приложения для публичных методов DoCmd

=cut


use strict;
use warnings;

use UNIVERSAL;

use Plack::Builder;
use Plack::CaptchaChecker;
use Plack::Response;
use Plack::UTF8Request;
use Plack::NormalizeResponse;

use EnvTools qw/is_production/;
use GeoTools;
use HttpTools;
use geo_regions;

use Settings;
use DoCmd::Public;

use utf8;

our $CAPTCHA_MAX_REQ ||= 8;
our $CAPTCHA_INTERVAL ||= 8 * 3600;
our $CAPTCHA_FREQ ||= 20;

# хитрое вычисление лимитов для отображения капчи:
# для России иставляем как есть, для "стран присутствий" - уменьшаем в 4 раза, для остальных (Азия и прочее) - в 16
sub captcha_limits {
    my ($env) = @_;
    
    my $r = Plack::UTF8Request->new($env);

    if ( !is_production() && get_cookie($r, 'do_not_show_captcha') ) {
        # проставляем нереальные параметры лимита, чтобы не сработала проверка на показ капчи
        # Описание параметров:
        # за captcha_interval секунд пользователь может совершить captcha_max_req запросов,
        # после этого начинаем показывать капчу на каждый captcha_freq запрос
        return {
            captcha_max_req => 1000000000,
            captcha_interval => 1,
            captcha_freq => 1,
        };
    }

    my $ip = $r->address;
    my $region = get_direct_region_from_geo(get_geo_from_ip($ip));

    # понижающий "карму" коэффицент
    my $geo_k = is_targeting_in_region($region, $geo_regions::RUS, {host => http_server_host($r)})
        ? 1
        : is_targeting_in_region($region, join(',', $geo_regions::UKR, $geo_regions::KAZ, $geo_regions::BY, $geo_regions::TR))
        ? 2
        : 4;

    # для некритичных методов лимит значительно больше (нужно перейти на использование аттрибута :Captcha)
    my $req_k = ($r->param('cmd')||'') =~ /^(ajaxGetSuggestion|ajaxPhraseStat)$/ ? 1 : 300;
    # ajaxInfoblockRequest вызывается на каждой странице Директа, и у больших агентств с одним IP адресом на нем появляется капча
    # поэтому еще увеличиваем лимит
    if (($r->param('cmd') // '') eq 'ajaxInfoblockRequest') {
        $req_k *= 2;
    }

    my $limits = {
        captcha_max_req => int($req_k * $CAPTCHA_MAX_REQ / $geo_k),
        captcha_interval => int($CAPTCHA_INTERVAL * $geo_k),
        captcha_freq => $CAPTCHA_FREQ,
    };
    # use Data::Dumper; print STDERR Dumper [$ip, $region, $geo_k, $limits];

    return $limits;
}   

sub get_app
{
    # основное приложение
    my $app = sub {
        my $env = shift;

        my $r = Plack::UTF8Request->new($env);
        my $multiform = $r->parameters;

        my $response = eval {
            DoCmd::Public::do_direct_public_cmd($r, $multiform);
        };

        my $e = $@;
        if ($e && UNIVERSAL::isa($e, 'Direct::HttpResponse')) {
            my $body = $e->{body_sub} ? $e->{body_sub}->($r->env->{reqid}) : $e->{body};
            return [$e->{status} || 200, ["Content-Type" => $e->{type} || 'text/html'], [$body]];
        } elsif ($e) {
            return [500, ['Content-Type' => 'text/html'], []]; 
        }
        
        return $response;
    };

    # набор оберток, характеризующий intapi
    return builder {
        enable 'Trace', cmd_type => 'direct.perl.web', cmd => '-';
        enable 'SetAccelInfo';
        enable 'MemLimit', limit => $Settings::APACHE_MAX_PROC_SIZE;
        #enable 'MailFatals', subject => '500 error';
        enable 'Sentry';
        enable 'Captcha',
            checker => Plack::CaptchaChecker->new(
                prefix => 'public',
                md_servers => $Settings::MEMCACHED_SERVERS,
                captcha_limits_sub => \&captcha_limits,
            ),
            display_captcha => \&DoCmd::Public::display_captcha;
        enable 'Direct::LogCmd';
        $app;
    };
}

1;
