package PSGIApp::Storage;

=head1 DESCRIPTION

PSGI-приложение для адреса /storage/ - доступ файлам, хранящимся в MDS, из web-интерфейса

Q: Почему используем приватный MDS Storage и проксируем файлы через себя,
вместо того, чтобы завести публичный mds namespace?
A: Хотим больше контроля за доступом: легко дать/закрыть доступ к неймспейсу
или даже отдельному файлу; можно проверять ClientID

=cut

use strict;
use warnings;

use Plack::Builder;
use Plack::UTF8Request;

use Direct::Storage;
use Direct::Storage::Types;
use Yandex::Validate qw/is_valid_id/;
use EnvTools qw/is_beta/;

use utf8;

=head2 get_app

=cut

sub get_app {
    my $app = sub {
        my $env = shift;
        my $r = Plack::UTF8Request->new($env);
        
        my ($client_id, $type, $name) = ($r->uri =~ m!/storage/(\d+)/([^/]+)/(.*)!);

        return respond_storage_file($client_id, $type, $name);
    };
    return builder {
        enable 'MemLimit', limit => $Settings::APACHE_MAX_PROC_SIZE;
        enable 'Trace', cmd_type => 'direct.storage', cmd => 'get';
        $app;
    };
}

=head2 respond_storage_file

=cut

sub respond_storage_file
{
    my ($client_id, $type, $name) = @_;

    my $DEBUG = is_beta();

    unless ($client_id && $type && $name) {
        if ($DEBUG) {
            warn "404: invalid param ($client_id) ($type) ($name)";
        }
        return [404, ['Content-Type' => 'text/html'], []];
    }
    
    my $storage = Direct::Storage->new();

    my $orig_name = $name;

    $name = mds_preprocess_filename($type, $name);

    if (!is_valid_mds_type($type) || !is_valid_id($client_id) || !$name) {
        if ($DEBUG) {
            warn "401: ($type) ($client_id) ($name)";
        }
        return [ 401, [ 'Content-Type' => 'text/html' ], [] ];
    }
    
    if (!mds_check_access($client_id, $type, $name, $orig_name)) {
        if ($DEBUG) {
            warn "403: ($client_id) ($type) ($name)";
        }
        return [ 403, [ 'Content-Type' => 'text/html' ], [] ];
    }
    
    my $mime = mds_get_mime_type($client_id, $type, $name, $orig_name) || 'application/octet-stream';

    my %req_opt = (
        filename => $name,
    );
    unless (Direct::Storage::Types::mds_check_type_trait($type, 'empty_client_id')) {
        $req_opt{ClientID} = $client_id;
    }
    my $file = $storage->get_file($type, %req_opt);
    unless ($file) {
        if ($DEBUG) {
            warn "404: get_file ($type) ($client_id) ($name)";
        }
        return [ 404, [ 'Content-Type' => 'text/html' ], [] ];
    }
    
    return [ 200, [ 'X-Accel-Redirect' => $file->internal_redirect_url( 'Content-Type' => $mime ) ], [] ];
}

1;
