package PdfReport::Queue;

=encoding utf8

=head1 NAME

    PdfReport::Queue

=head1 DESCRIPTION

    Модуль содержит в себе описания "очередей" построения PDF-отчетов

=cut

use Direct::Modern;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils;

use Settings;

=head1 VARIABLES

=head2 $HEAVY_PHRASES

    Количество фраз в отчете, начиная с которого считаем его тяжелым

=cut

our $HEAVY_PHRASES = 60000;

=head2 $HEAVY_PERIOD_DAYS

    период в днях, выше которого считаем отчёт "тяжёлым"

=cut

our $HEAVY_PERIOD_DAYS = 100;

=head2 $HEAVY_CIDS_COUNT

    считаем запрос тяжёлым, если отчёт требуется одновременно
    больше чем по этому количеству кампаний

=cut

our $HEAVY_CIDS_COUNT = 4;

=head2 $ATTEMPTS

    max attempts to complete the report

=cut

our $ATTEMPTS = 2;

=head2 $MAX_QUEUE_LENGTH

    Максимальная длина очереди отчетов для одного клиента

=cut

our $MAX_QUEUE_LENGTH = 5;

=head2 $MAX_REPORT_LIFETIME

    через сколько дней после создания удалять отчеты

=cut

our $MAX_REPORT_LIFETIME = 90;

=head2 INTERNAL VARIABLES

=head3 $base_conditon

    Общая часть sql-условия, т.к. очереди пока только две ("тяжелая" и "не тяжелая")
=cut

my $base_condition = {
    _OR => [
        phrases_count__ge => $HEAVY_PHRASES,
        _TEXT => "TIMESTAMPDIFF(day, date_from, date_to) >= $HEAVY_PERIOD_DAYS",
        _TEXT => "TIMESTAMPDIFF(day, date_from, date_to) IS NULL",
        # +1, т.к. считаем запятые между кампаниями, а нужно кол-во кампаний
        _TEXT => "LENGTH(cids)-LENGTH(REPLACE(cids,',',''))+1 > $HEAVY_CIDS_COUNT",
    ],
};

=head3 $PROCESSING_TIMEOUT

    Maximum time of beeing in processing state

=cut

my $PROCESSING_TIMEOUT = 30;

=head2 %QUEUES

    Хеш с перечислением доступных очередей и соответствующих им SQL-условий для отбора отчетов

=cut

our %QUEUES = (
    std => {
        _NOT => $base_condition,
    },
    heavy => {
        _AND => $base_condition,
    },
    heavy2 => {
        _AND => $base_condition,
    }
);

=head1 SUBROUTINES

=head2 getPDFQueueLength

    Получить длину очереди заказов на PDF отчёты для заданного uid

=cut

sub getPDFQueueLength($) {
    my $uid = shift;
    my $queue_length = get_one_field_sql(PPC(uid => $uid), "
            SELECT count(*)
              FROM pdf_reports
             WHERE uid=?
               AND not(rank>=? AND status='new' OR rank>? AND status='processing')
               AND status!='ready'
              ", $uid, $ATTEMPTS, $ATTEMPTS);

    return $queue_length;
}

=head2 recycleGarbageReports

    finds all orders that are in 'processing' state longer then for $PROCESSING_TIMEOUT
    and checks if the connection_id is still active. other way the order marked as 'new'

=cut

sub recycleGarbageReports {
    my %opt = @_;

    #   get all hanging orders
    my $orders = get_all_sql(PPC(shard => $opt{shard}), "select connect_id, id FROM pdf_reports
                                     WHERE status=?
                                       AND rank<=?
                                       AND processing_time < TIMESTAMPADD(SECOND,?, NOW())",
                                           "processing", $ATTEMPTS, -$PROCESSING_TIMEOUT)
        or return _err($opt{log}, 'Failed to get list of expired pdf_report orders');

    #   Get list of active connections
    my $processlist = get_all_sql(PPC(shard => $opt{shard}), "show full processlist")
        or return _err($opt{log}, 'Failed to get processlist');

    my %connects = ();
    map { $connects{$_->{Id}} = $_->{Id} } @$processlist;

    #   find closed connections
    my @bad_orders = map {$_->{id}} grep { not exists $connects{$_->{connect_id}} } @$orders;

    return '0E0' unless scalar @bad_orders;   #   nil but true
    #   recycle hanging orders with closed connections
    do_update_table(PPC(shard => $opt{shard}),
        'pdf_reports',
        { status => "new", processing_time => 0, create_time__dont_quote => "NOW()" },
        where => {status => "processing", id => \@bad_orders}
    ) or return _err($opt{log}, 'Failed to recycle bad pdf_report orders');

    return 1;
}

=head2 popReportOrder

    This fun "gets an order" marked as "new" from the report_orders table in DB
    i.e. it marks it as "processing" so 

    Then the order is processed and PDF file is generated, filename is written to DB
    and the order status became "ready"

    Возвращаемые значения:
        undef           при ошибке
        0E0             при отсутствии в очереди отчетов для обработки
        {               при успехе - ссылка на хеш с данными о сгенерированном отчете
            id => NN,
            create_time => 'YYYY-MM-DD HH:MM:SS'
        }

=cut

sub popReportOrder{
    my ($queue, $vars) = @_;
    return _err($vars->{log}, "No PDF making proc is given") unless (defined $vars->{pdf_proc} and ref $vars->{pdf_proc} eq 'CODE');

    my $connection_id = get_one_field_sql(PPC(shard => $vars->{shard}), "select CONNECTION_ID()");

    my $row =  get_one_line_sql(PPC(shard => $vars->{shard}), [
        "SELECT id, create_time, date_from, date_to, cids, lang, date_group as `group`, uid
        FROM pdf_reports",
        WHERE => {
            status => "new",
            rank__le => $ATTEMPTS,
            %{ $QUEUES{$queue} },
        },
        "ORDER BY id",
        "LIMIT 1"]) or do {
            _err(undef, 'Failed to pop an order (empty queue?)');
            return '0E0';
        };

    my $affected_before = do_update_table(PPC(shard => $vars->{shard}), 'pdf_reports', {status => "processing", 
                                                         processing_time__dont_quote => "NOW()", 
                                                         connect_id => $connection_id, "rank__dont_quote" => "IFNULL(rank,0)+1"},
                                                        where => {id => $row->{id}, cids => $row->{cids}, status => "new"}) || 0;

    if ($affected_before eq '0E0') {    #   Error, no line affected
        return _err($vars->{log}, "Error updating report's rank (before process)");
    } elsif ($affected_before eq '1') { #   OK

        my $result = eval { $vars->{pdf_proc}->($row) };
        _err($vars->{log}, "Error creating report file for report with id $row->{id} for cids $row->{cids}: " . ($@ // '') ) unless defined $result->{output} and !$result->{no_data};

        # Не используем Yandex::DBTools::do_update_table, потому что он не поддерживает бинарные данные. Поле pdf_data имеет тип mediumblob.
        my $affected_report = do_sql(PPC(shard => $vars->{shard}), "UPDATE pdf_reports
                                      SET status=?, pdf_data=?, ready_time=now(), date_from=?, date_to=?, status_no_data=?
                                      WHERE id=? AND cids=? AND status=?",
                                      "ready", $result->{output}, $row->{date_from}, $row->{date_to}, $result->{no_data}?"Yes":"No",
                                      $row->{id}, $row->{cids}, "processing")||0;

        return _err($vars->{log}, "Error updating report (after process)") unless $affected_report eq '1';
        return hash_cut($row, qw/id create_time/);
    } else {                            #   Error
        return _err($vars->{log}, "Error updating report's rank (before process) - strange result");
    }

    return undef 
}

=head2 pushReportOrder

    Добавить отчёт в очередь

=cut

sub pushReportOrder($){
    my ($vars) = @_;

    my $qty = get_one_field_sql(PPC(uid => $vars->{uid}), ['SELECT COUNT(*) FROM bids',
        WHERE => {cid => [split /\D/, $vars->{cids}], PhraseID__gt => 0}]);

    do_insert_into_table(PPC(uid => $vars->{uid}), 'pdf_reports', {
                                              id         => get_new_id('report_id'),
                                              uid        => $vars->{uid},
                                              cids       => $vars->{cids},
                                              date_from  => $vars->{date_from},
                                              date_to    => $vars->{date_to},
                                              status     => "new",
                                              date_group => $vars->{group},
                                              lang       => $vars->{lang},
                                              phrases_count => $qty
                                             });
}

=head2 get_data($uid, $id)

Возвращает ссылку на скаляр (файл с отчётом) или undef

=cut

sub get_data {
    my ($uid, $id) = @_;

    my $data = get_one_field_sql(PPC(uid => $uid),
        "SELECT pdf_data FROM pdf_reports
        WHERE uid=? AND id=? AND status_no_data='No' AND status='ready'",
        $uid, $id,
    );

    return undef if !$data;
    return \$data;
}

=head2 get_list($uid)

Возвращает список отчётов пользователя

=cut

sub get_list {
    my ($uid) = @_;

    my $reports = get_all_sql(PPC(uid => $uid),
        "SELECT id, cids, date_from, date_to, status, status_no_data,
            ready_time, create_time, date_group as `group`, rank,
            (
                pdf_data IS NULL AND status='ready'
                OR rank>=? AND status='new'
                OR rank>? AND status='processing'
            ) as is_fail
        FROM pdf_reports
        WHERE uid=?
        ORDER BY id DESC",
        $ATTEMPTS,
        $ATTEMPTS,
        $uid,
    );

    return $reports;
}

=head2 requeue_order($uid, $id)

Перезапрос отчёта

=cut

sub requeue_order {
    my ($uid, $id) = @_;

    do_sql(PPC(uid => $uid),
        "UPDATE pdf_reports
        SET rank=0
        WHERE uid=? AND id=? AND (rank>=? AND status='new' OR rank>? AND status='processing')",
        $uid, $id,
        $ATTEMPTS,
        $ATTEMPTS,
    );

    return;
}

=head2 delete_data($uid, $id)

Удаляет файл с отчётом

=cut

sub delete_data {
    my ($uid, $id) = @_;

    do_delete_from_table(PPC(uid => $uid), 'pdf_reports',
        where => { uid => $uid, id => $id },
    );

    return;
}

=head2 clearOldPdfReports

    удаляет Pdf отчёты старше 3х месяцев

=cut

sub clearOldPdfReports {
    my (%opt) = @_;
    do_delete_from_table(PPC(shard => $opt{shard}), 'pdf_reports', where => {create_time__lt__dont_quote => "NOW() - INTERVAL $MAX_REPORT_LIFETIME DAY"});
}

=head2 INTERNAL SUBROUTINES

=head3 _err($log)

    если $log умеет out, сделать shift->out(@_),
    иначе напечатать на STDERR если в переменных окружения истинны DEBUG или LOG_TEE

=cut

sub _err {
    my $log = shift;
    if ($log && ref $log && UNIVERSAL::can($log, 'out')) {
        $log->out(@_);
    } elsif ($ENV{DEBUG} || $ENV{LOG_TEE}) {
        print STDERR "ERROR:", join('\n',map { s/\n+$//r } @_), "\n";
    }

    return undef;
}

1;
