

##################################################
#
#  Direct.Yandex.ru
#
#  Phrase
#      Работа с фразами
#
#
#  $Id$
#
# (c) 2011, Yandex
#
##################################################

use strict;
use warnings;
use utf8;

package Phrase;

=head1 NAME

Phrase

=head1 DESCRIPTION

  Работа с фразами

=cut

use List::Util qw/max min/;
use List::MoreUtils qw/uniq part/;

use Settings;
use Primitives;
use PrimitivesIds;
use ModerateChecks;
use MinusWords;
use PhraseText;
use Direct::PhraseTools qw/polish_phrase_text stop_word_add_plus/;
use TextTools;
use BannerTemplates;
use GeoTools qw/refine_geoid/;
use Lang::Unglue;

use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate qw/is_valid_float/;
use Yandex::ScalarUtils;

use Models::Phrase;
use Models::AdGroup qw/get_auto_adgroup_name/;

use base 'Exporter';
our @EXPORT = qw/
    validate_phrase_one_href_param
    update_group_phrases
    update_phrases
    fixate_stopwords
    fixate_stopwords_lite
    process_phrases
/;


# приориеты для фраз в автобюджете
our %PRIORITY_VALUES = (Low => 1, Medium => 3, High => 5);
our %PRIORITY_REVERSE_VALUES = reverse %PRIORITY_VALUES;

# Сравниваются старые и новые фразы, записываются в базу

=head2 update_group_phrases

Сохраняем фразы группы

Именнованные опции %OPT:
    ignore_minus_words -- (0|1) не апдейтить минус слова
    is_different_places -- в кампании раздельное размещение
    dont_change_translocal_geo -- (0|1) не модифицировать geo из транслокального в апи-шный, например если он пришел напрямую из базы, а не от пользователя
    from_stat -- (0|1) изменения фраз приехали со страниц статистики (для логгирования)
    campaign -- (strategy, ContextPriceCoef)  для расчета логгируемой ставки в сети

=cut
sub update_group_phrases($$$$$$;%) {
    my ($campaign, $uid, $ClientID, $group, $old, $UID, %OPT ) = @_;

    my $cid = $campaign->{cid};
    my ($need_moderate_phrases, $changes) = check_moderate_phrases($group, $old);

    my ($p_statusModerate, $is_new_group, $geo_updated);

    unless ($OPT{dont_change_translocal_geo}) {
        $group->{geo} = GeoTools::modify_translocal_region_before_save($group->{geo}, {ClientID => $ClientID});
    }

    my $geo = refine_geoid($group->{geo}, undef, {ClientID => $ClientID});

    my $pid = $group->{pid};
    unless ($pid) {
        $is_new_group = 1;
        # create adgroup
        ($p_statusModerate, $pid) = update_phrases($pid, $group->{cid},
                                                   $group->{statusModerate}, $OPT{statusEmpty}||'No', $uid, $geo);
        $geo_updated = 1;
        $group->{pid} = $pid; 
    }

    my ($norm_to_id, $indicators) = Models::Phrase::save_phrases($campaign, $group, $old, %OPT, uid => $uid, UID => $UID);

    # если были условия ретаргетинга или включали/отключали фразы, то нужно послать в БК
    # TODO: ^^^ почему ? ^^^ - кажется, что этот момент должен решаться при работе с условиями ретаргетинга
    my $send_to_bs = $indicators->{delete_params}
                        || $indicators->{changed_suspend_state}
                        || $indicators->{changed_phrase_data}
                        || exists $group->{retargetings};

    $need_moderate_phrases = 1 if $indicators->{added_new_phrases} || $indicators->{moderate_new};
    if ($OPT{moderate_declined}) {
        $need_moderate_phrases = 1 if $old->{statusModerate} eq 'No';
    }
    if ($need_moderate_phrases && $need_moderate_phrases == 1 && !$is_new_group) {
        ($p_statusModerate) = update_phrases($pid, $group->{cid},
                                             $group->{statusModerate}, $OPT{statusEmpty}||'No', $uid, $geo);
    }

    my ($banner_values, $group_values) = ({'b.LastChange__dont_quote' => 'LastChange'}, {});

    # Обновляем LastChange только если в группе поменялись тексты фраз (в т.ч. были добавлены или удалены), фразы были
    # включеные/отключены или изменились пользовательские параметры (param1, param2). Иначе оставляем всё как есть.
    if (!$indicators->{phrase_set_changed} && !$indicators->{params_updated} && !$indicators->{changed_suspend_state}) {
        $group_values->{LastChange__dont_quote} = 'LastChange';
    }

    my $bids_for_update = [];
    # Если группа пустая (флаг выставляется ранее до вызова функции, так как внутри этой функции невозможно это определить),
    # то переотправляем ее в BS, чтобы остановить показы баннеров группы 
    if ($group->{is_empty_need_bs_sync}) {
        $group_values->{statusBsSynced} = 'No';

    } elsif (
        $need_moderate_phrases && ($changes->{new_phrases} || $changes->{changed_phrases})
        && (!$p_statusModerate || $p_statusModerate ne 'New') && $group->{pid}
    ) {
        hash_merge $banner_values, {
            'b.statusModerate' => 'Ready',
            'b.phoneflag__dont_quote' => 'IF(b.vcard_id IS NOT NULL,"Ready","New")', # отправляем на модерацию визитку
            'b.statusSitelinksModerate__dont_quote' => 'IF(b.sitelinks_set_id IS NOT NULL,"Ready","New")', # отправляем на модерацию сайтлинки
            'b.statusPostModerate__dont_quote' => "IF(b.statusPostModerate = 'Rejected', 'Rejected', 'No')",
            'b.LastChange__dont_quote' => 'NOW()',
            'bim.statusModerate' => 'Ready', # отправляем на модерацию картинку
            'bdh.statusModerate' => 'Ready',
        };
        hash_merge $group_values, { statusModerate => 'Ready', 
                                    statusPostModerate__dont_quote => "IF(statusPostModerate = 'Rejected', 'Rejected', 'No')" };
        foreach my $banner (@{$group->{banners}}) {
            # Отсылаем шаблонный баннер на модерацию, если изменились фразы, но при этом не только удалены
            if( is_template_banner( $banner ) ) {
                push @$bids_for_update, $banner->{bid};
            }
        }
        do_update_table(PPC(ClientID => $ClientID), 
            'banners b
            LEFT JOIN banner_images bim on bim.bid = b.bid
            LEFT JOIN banner_display_hrefs bdh on bdh.bid = b.bid',
            $banner_values,
            where => {
                'b.bid' => $bids_for_update,
            }
        ) if @$bids_for_update;
        Direct::Banners::delete_minus_geo(bid => $bids_for_update, status => $banner_values->{statusModerate});

    }
    
    # иначе geo не во всех случаях обновляется
    my $need_schedule_forecast = 0;
    if (! $geo_updated && ($group->{geo} || '') ne ($old->{geo} || '') ) {
        $group_values->{geo} = $group->{geo};
        $need_schedule_forecast = 1;
    }

    if (!$is_new_group && ($indicators->{phrase_set_changed} || $need_schedule_forecast)) {
        # планируем обновление showsForecast
        $group_values = hash_merge $group_values, { statusShowsForecast => 'New' };
    }
    
    if ( $send_to_bs || $need_moderate_phrases || $indicators->{params_updated} || $need_schedule_forecast) {
        schedule_forecast($cid) if $need_moderate_phrases || $need_schedule_forecast;
        if ( $send_to_bs || $need_moderate_phrases || $indicators->{params_updated} ) {
            # когда есть параметры фразы для обновления - отправляем в БК и все условие т.к. иначе параметры не переотправятся
            $group_values->{statusBsSynced} = 'No';
        }
    }
    if (keys %$group_values) {
        do_update_table(PPC(ClientID => $ClientID), 'phrases', $group_values, where => {pid => $pid});
    }

    # DIRECT-23711: В процедуре save_banner_minus_words сбрасывается флаг statusBsSynced. Поэтому предвартельно проверим, действительно ли изменились banner_minus_words
    if (!$OPT{ignore_minus_words}) {
        my $old_minus_words = MinusWords::polish_minus_words($old->{minus_words});
        my $new_minus_words = MinusWords::polish_minus_words($group->{minus_words});
        MinusWords::save_banner_minus_words($pid, $group->{minus_words}) if $group->{is_new} || $group->{is_copy} || $new_minus_words ne $old_minus_words;
    }

    return { norm_to_id => $norm_to_id, pid => $pid };
}

=head2 update_phrases($pid, $cid, $status, $statusEmpty, $uid, $geo)

    Обновляет/добавляет(автоматически определяет необходимость) условие показа для указанного баннера
    Обновляет только следующие поля: statusModerate, statusBsSynced, LastChange.
    Добавляет еще и geo.

    $geo должен уже быть нормализованным через refine_geoid()

=cut

sub update_phrases {
    my ($pid, $cid, $status, $statusEmpty, $uid, $geo) = @_;

    my $statusModerate = $statusEmpty ne "Yes" && $status ne "New" ? "Ready" : "New";

    my %banner;
    
    if ($pid) {
        do_sql(PPC(pid => $pid), "update phrases
                      set statusModerate = ?
                            , statusBsSynced = 'No'
                            , statusPostModerate = IF(statusPostModerate = 'Rejected', 'Rejected', 'No')
                      where pid = ?", $statusModerate, $pid);
        clear_banners_moderate_flags(get_bids(pid => $pid));
    } else {
        $pid = get_new_id('pid', cid => $cid);

        do_insert_into_table(PPC(pid => $pid), 'phrases', {
            pid => $pid,
            cid => $cid,
            statusModerate => $statusModerate,
            geo => $geo,
            statusPostModerate => 'No',
            group_name => Models::AdGroup::get_auto_adgroup_name({pid => $pid}, uid => $uid),
        });
        # ad groups     
        $banner{pid} = $pid;
    }

    if (%banner) {
        $banner{LastChange__dont_quote} = 'NOW()';
    }
    
    # API: ad_groups_v4
    
    # TODO: обновляем только 1 баннер, а не все баннеры группы
    if (%banner) {
        do_update_table(PPC(pid => $pid), 'banners', \%banner, where => {pid => $pid});
    }
    
    return wantarray ? ($statusModerate, $pid) : $statusModerate;
}

=head2 fixate_stopwords

Фиксация стоп-слов в баннере (добавить + перед стоп-словами из словаря)
Параметры:
    $phrases - массив фраз [ { phrase => '...' }, ]. Если фраза была изменена, phrase меняется на новую
    $ph_hash - Необязательный выходной параметр, хеш, который будет выглядеть как 
    {
        # новая фраза
        '+моя фраза полностью' => {
            original => 'моя фраза полностью', # фраза до того, как мы добавили свои плюсики
            fixation => [ ['+моя фраза','моя фраза'], ['+еще одна фраза','еще одна фраза'] ], # список подсветки для интерфейса
        }
    }
    В списке подсветки указаны только те части фразы, которые совпали со словарными фразами (из ppcdict.stopword_fixation)

Возвращает 1 или 0 - были ли сделаны какие-то изменения

=cut

sub fixate_stopwords
{
    my ($phrases, $ph_hash) = @_;
    $ph_hash ||= {};
    my $fixed = 0;
    for my $p (@$phrases) {
        my $hl_list = [];
        my $new_phr = fixate_stopword_phrase($p->{phrase}, $hl_list);
        if ($new_phr ne $p->{phrase}) {
            $ph_hash->{$new_phr}->{original} = $p->{phrase};
            $ph_hash->{$new_phr}->{fixation} = $hl_list;
            $p->{phrase} = $new_phr;
            $p->{fixation} = $hl_list;
            $fixed = 1;
        }
    }
    return $fixed;
}

=head2 fixate_stopwords_lite

Функция работает с массивом строк, модифицируя строки в этом массиве (добавляет + перед стоп словами, см. fixate_stopwords)
Ничего не возвращает

=cut

sub fixate_stopwords_lite
{
    my ($phrases) = @_;
    for my $p (@$phrases) {
        my $new_phr = fixate_stopword_phrase($p);
        if ($new_phr ne $p) {
            $p = $new_phr;
        }
    }
}

=head2 fixate_stopword_phrase($phrase, $hl_list)

Если во фразе встречается подстрока из словаря, добавляется плюс перед стоп-словом, например:
fixate_stopword('love is') => 'love +is'
В $hl_list будет добавлена запись ['love +is', 'love is']

Также, фиксируем слова вида стоп_слово \d+ (а 15 -> +а 15)

=cut

{
my $list_regexp;
sub fixate_stopword_phrase
{
    my ($str, $hl_list) = @_;
    my $profile = Yandex::Trace::new_profile('phrase:fixate_stopword_phrase');
    # фиксируем стоп-слова, если фраза состоит только из стоп-слов и чисел (и минус-слов)
    if ($str =~ /\d/) {
        my ($words, $minus) = part { /^-/ ? 1 : 0 } split /\s+/, $str;
        if ($words && @$words == 1 && $words->[0] =~ /^ (\w+(\-\d+)+) | ((\d+\-)+\w+) $/x) {
            # особый случай - слово и числа, разделенные дефисом
            my $old_str_orig = $words->[0];
            my $old_str = $old_str_orig =~ s/\-/ /gr;
            my $new_str = stop_word_add_plus($old_str);
            $new_str =~ s/ /-/g;
            push @$hl_list, [ $new_str, $old_str_orig ];
            if ($minus and @$minus) {
                $new_str .= ' '.(join ' ', @$minus);
            }
            return $new_str;
        }
        if (List::MoreUtils::all { is_valid_float($_) or _is_stopword($_) } @$words) {
            my $old_str = join ' ', @$words;
            my $new_str = stop_word_add_plus($old_str);
            push @$hl_list, [ $new_str, $old_str ];
            if ($minus and @$minus) {
                $new_str .= ' '.(join ' ', @$minus);
            }
            return $new_str;
        }
    }
    $list_regexp ||= _load_fix_list();
    my $new_str = '';
    for my $str_part ($str =~ /( [^\[\"]+ | \[[^]]*\]? | "[^"]*"? )/gx) {
        if ($str_part !~ /^\s*[\["]/) {
            for my $re ( qr/([!+ ]*\b[!+ ]*(?:$list_regexp))\b/i) {
                $str_part =~ s/$re/
                my $old = $1;
                my $new = stop_word_add_plus($old);
                $old =~ s!^\s+!!;
                my $new_hl = $new;
                $new_hl =~ s!^\s+!!;
                push @$hl_list, [ $new_hl, $old ];
                $new;
                /ige;
            }
        }
        $new_str .= $str_part;
    }
    return $new_str;
}
}

# превращаем список фраз в список регулярок, чтобы находить фразы с +
sub _load_fix_list
{
    my $list = get_one_column_sql(PPCDICT, "select phrase from stopword_fixation") || [];
    return join '|',
        (map { 
            my @r = split /\s+/, $_;
            my $r = (quotemeta shift @r); # перед первым словом будет добавлено [+ ]* (в fixate_stopword_phrase)
            $r .= join '', map { "[!+ ]+\Q$_\E" } @r; # перд остальными словами добавляем [+ ]+
            $r
        } @$list);
}

# is_stopword для слов со спец-символами
sub _is_stopword
{
    my ($w) = @_;
    $w =~ s!\W!!g;
    return Yandex::MyGoodWords::is_stopword($w);
}

=head2 process_phrases

    Процессит массив фраз:
        склеивает дубликаты, минус слова, добавляет нормальную форму и т.д.

=cut

# TODO: можно заменить на этот вызов везде, где встречаются эти 2 операции подряд
sub process_phrases
{
    my $phrases = shift;
    my $result = [glue_minus_words($phrases)];

    add_plus_to_minus_stop_words($result);

    for my $p (@{$result}) {
        $p = hash_merge $p, PhraseText::get_phrase_props($p->{phrase});
    }

    return $result;
}

=head2 get_phrases_form(%opt)

    Обработка фраз, пришедших из формы:
    получение нормализованной фразы, простановка флагов.

    Именованные параметры:
        group
        old_phrases
        new_phrases
        is_groups_copy_action

    Результат: пробрасывается результат fixate_stopwords

=cut

sub get_phrases_form {
    my %OPT = @_;
    my ($group, $old_phrases, $edited_phrases, $is_groups_copy_action) =
       ($OPT{group}, $OPT{old_phrases}, $OPT{new_phrases}, $OPT{is_groups_copy_action});

    $old_phrases = [] unless defined $old_phrases;
    my $phrases_hash = {map {$_->{id} => $_} grep {$_->{id}} @$edited_phrases};
    my @new_phrases = grep {!$_->{id}} @$edited_phrases;
    my @result_phrases;

    # Обработаем существующие фразы
    foreach my $old_ph (grep { $phrases_hash->{$_->{id}} } @$old_phrases) {
        my $new_phrase = $phrases_hash->{$old_ph->{id}};
        next unless defined $new_phrase->{phrase} && length($new_phrase->{phrase});

        my $old_norm_words = Yandex::MyGoodWords::norm_words($old_ph->{phrase});
        if ($is_groups_copy_action) {
            $old_ph->{is_suspended} = 0;
            $old_ph->{nobsdata} = 0;
        }

        # Обработаем операторы "()|"
        my $phrase_texts_after_process = [$new_phrase->{phrase}];
        PhraseText::process_phrase_brackets($phrase_texts_after_process);
        if (@$phrase_texts_after_process > 1) {
            for my $subphrase (@$phrase_texts_after_process) {
                if (Yandex::MyGoodWords::norm_words($subphrase) eq $old_norm_words) {
                    $old_ph->{phrase} = $old_ph->{phr} = $subphrase;
                    push @result_phrases, $old_ph;
                } else {
                    push @new_phrases, {id => 0, phrase => $subphrase};
                }
            }
        } else {
            # update phrase
            $old_ph->{phrase} = $old_ph->{phr} = $new_phrase->{phrase};
            $old_ph->{norm_phrase} = Yandex::MyGoodWords::norm_words($new_phrase->{phrase});
            push @result_phrases, $old_ph;
        }
    }

    # Обработаем новые фразы
    foreach my $phrase (@new_phrases) {
        # чтобы при удалении дубликатов удалились новые записи, а не старые
        $phrase->{is_new} = 1;

        $phrase->{rank} = $Settings::DEFAULT_RANK;

        # Обработаем операторы во фразе
        my $phrase_texts_after_process = [$phrase->{phrase}];
        PhraseText::process_phrase_brackets($phrase_texts_after_process);

        push @result_phrases, (map { +{
            %$phrase,
            phrase => $_,
        } } @$phrase_texts_after_process);
    }
    # Ряд манипуляций с фразами:
    polish_phrase_text(smartstrip($_->{phrase})) foreach (@result_phrases);
    # добавляем + к некоторым стоп-словам перед склейкой минус-слов
    my $fixated_phrases = {};
    my $is_stopword_fixated = fixate_stopwords(\@result_phrases, $fixated_phrases);
    
    @result_phrases = glue_minus_words(\@result_phrases);
    add_plus_to_minus_stop_words(\@result_phrases);
    for (@result_phrases) {
        if (exists $fixated_phrases->{$_->{phrase}}) {
            $_->{fixation} = $fixated_phrases->{$_->{phrase}}->{fixation};
        }
    }

    $group->{statusShowsForecast} = 'New';

    $group->{phrases} = \@result_phrases;
    return $is_stopword_fixated;
}



1;
