package PhraseDoubles;

# $Id: $

=head1 NAME

    PhraseDoubles

=head1 DESCRIPTION

    Работа с фразами и дублями. 

=cut

use warnings;
use strict;

use Settings;
use Tools;

use Yandex::MyGoodWords;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xsort/;

use List::MoreUtils qw/uniq/;
use List::Util qw/sum/;


use utf8;

use base qw/Exporter/;
our @EXPORT = qw/

    get_all_doubles

/;

#Максимальное количество фраз, которое мы готовы обрабатывать.
our $MAX_PHRASES_LIMIT = 10_000_000;

my $CIDS_CHUNK = 100;

=head2 get_all_doubles (phrases, %O)

    Принимает на вход список фраз, которые надо проверить на дубликаты
    Возвращает список дублей в виде:
    '<фраза>' => [массив фраз-дублей]
    Возвращаются только фразы, которые имеют дубли, фразы без дублей опускаются.

    Параметры:
        phrases - указатель на список фраз
        O - опции:
            use_domain_geo - делать проверку с учетом домена и региона.
            phrases_for_check - список фраз, которые не принадлежат каким-либо кампаниям(не сохранен в базу), но их тоже надо посмотреть на предмет дублей.

=cut

sub get_all_doubles {
    my ($phrases, %O) = @_;

    my $phrases2 = $O{phrases_for_check}||[];

    my (%checked_lemmas, %checked_formas, %checked_phrases);

    # Если указаны phrases_for_check, то значит надо пересечь первые фразы со вторыми.
    # Если не указаны, то значит надо было сделать пересечения между первыми фразами, 
    my %checked_phrases2;
    if ($O{phrases_for_check}) {
        foreach my $phrase (@$phrases2) {
            add_phrase_to_doubles( phrase          => $phrase,
                                   checked_lemmas  => \%checked_lemmas,
                                   checked_formas  => \%checked_formas,
                                   checked_phrases => \%checked_phrases2,
                                   bids_domain_geo => $O{bids_domain_geo},
                                 );

        }
    }
    # Добавляем к пересечениям фразы из БД.
    foreach my $phrase (@$phrases) {
        add_phrase_to_doubles( phrase          => $phrase,
                               checked_lemmas  => \%checked_lemmas,
                               checked_formas  => \%checked_formas,
                               checked_phrases => \%checked_phrases,
                               bids_domain_geo => $O{bids_domain_geo},
                             );

    }

    # В конец списка дубликатов для фраз не из БД добавляем список дублей для фраз для этих фраз из БД
    if ($O{phrases_for_check}) {
        foreach my $d_g (keys %checked_phrases2) {
            foreach my $ph (keys %{$checked_phrases2{$d_g}}) {
                push @{$checked_phrases2{$d_g}->{$ph}}, @{$checked_phrases{$d_g}->{$ph}} 
                if $checked_phrases{$d_g} && $checked_phrases{$d_g}->{$ph} && scalar @{$checked_phrases{$d_g}->{$ph}};
            }
        }
        %checked_phrases =  %checked_phrases2;
    }

    # После окончания раскладывания по полочками дубли удаляем все фразы и домены_гео, которые не имеют дубли
    foreach my $d_g (keys %checked_phrases) {
        foreach (keys %{$checked_phrases{$d_g}}) {
            delete $checked_phrases{$d_g}->{$_} if (scalar @{$checked_phrases{$d_g}->{$_}} < 2);
        }
        delete $checked_phrases{$d_g} unless %{$checked_phrases{$d_g}};
    }

    # В случае, если не надо использовать различия по домену и региону мы ранее использовали ключ "". 
    # Но он не нужен в выводе, поэтому поднимаем хеш на один уровень.
    %checked_phrases = %{delete $checked_phrases{""}} if (!$O{bids_domain_geo} && $checked_phrases{""});
    return \%checked_phrases;
}

=head2 add_phrase_to_doubles

    Внутренняя функция. Добавляет новую фразу к списку фраз выбирая дублем какой фразы она является.

=cut
sub add_phrase_to_doubles {
    my %O = @_;
    my $phrase = $O{phrase};

    # Если показано использовать различия по доменам и регионам, то узнаем какой баннер имеет какой домен и регион.    
    my $domain_geo = $O{bids_domain_geo}->{$phrase->{bid} || $phrase->{mbid} || 0} || "";
    my ($lemmas, $formas) = ($phrase->{phrase} !~ m/\"/) ? Yandex::MyGoodWords::get_lemmas_formas_list_words($phrase->{phrase}, str_result => 1)
                                                         : ($phrase->{phrase}, $phrase->{phrase});

    my $prev_lemmas =  $O{checked_lemmas}->{$domain_geo}->{$lemmas};
    my $prev_formas =  $O{checked_formas}->{$domain_geo}->{$formas};

    if (defined($prev_lemmas)) {
        # если раньше мы находили такую же лемму, то добавляем фразу в список дубликатов фразы
        push @{$O{checked_phrases}->{$domain_geo}->{$prev_lemmas->{phrase}}}, $phrase;
    } elsif (defined($prev_formas)) {
        # если раньше мы находили такую же форму, то добавляем фразу в список дубликатов фразы
        push @{$O{checked_phrases}->{$domain_geo}->{$prev_formas->{phrase}}}, $phrase;
    } else {
        # а если НЕ находили, то добавляем новую лемму и форму.            
        $O{checked_lemmas}->{$domain_geo}->{$lemmas} = $O{checked_formas}->{$domain_geo}->{$formas} = $phrase;
        push @{$O{checked_phrases}->{$domain_geo}->{$phrase->{phrase}}}, $phrase;
    }
}

=head2 add_search_phrase_doubles_queue (uid, cids, words)

    Кладет в очередь заявок на поиск дублей новую заявку.
    Параметры:
    uid - идентификатор пользователя
    cids - указатель на список номеров кампаний
    words - указатель на список слов, которые должны проверяться

=cut
sub add_search_phrase_doubles_queue {
    my ($uid, $cids, $words) = @_;
    do_insert_into_table(PPC(uid => $uid), 'phrases_doubles_queue', {request_id => get_new_id('doubles_request_id'),
                                                                     uid   => $uid, 
                                                                     status =>'New', 
                                                                     params => encode_json_and_compress({cids=>$cids, words=>$words})});
}

=head2 get_search_phrase_doubles_requests (uid)

    Возвращает список всех заявок на поиск дублей для пользователя.

=cut
sub get_search_phrase_doubles_requests {
    my $uid = shift;
    my $request_list = get_all_sql(PPC(uid => $uid), "SELECT request_id, result, create_time, status,
                                            (SELECT COUNT(*) FROM phrases_doubles_queue pdq2 WHERE status='New' AND pdq2.request_id<pdq1.request_id) AS queue_num
                                          FROM phrases_doubles_queue pdq1 WHERE uid = ? ORDER BY create_time DESC", $uid);
    prepare_doubles_result_data($_) foreach (@$request_list);

    return $request_list;
}

=head2 get_search_phrase_doubles_result_info (request_id)

    Возвращает детальную информацию по результатам поиска дублей по заявке.

=cut
sub get_search_phrase_doubles_result_info {
    my ($request_id, $uid) = @_;
    my $request = get_one_line_sql(PPC(uid => $uid), "SELECT request_id, result, create_time, status FROM phrases_doubles_queue WHERE request_id = ?", $request_id);
    prepare_doubles_result_data($request);
    my $ph_ids = [uniq map { @{$_->{ph_ids}} } grep {$_->{ph_ids}} (@{$request->{double_phrases}},@{$request->{single_phrases}})];
    my $ph_id_cid = get_hashes_hash_sql(PPC(uid => $uid), ["SELECT id, cid FROM bids", where=>{id=>$ph_ids}]);
    foreach my $phrase (@{$request->{double_phrases}}) {
        my $cids = [uniq map {$ph_id_cid->{$_}->{cid}} @{$phrase->{ph_ids}}];
        $phrase->{cids} = $cids;
    }
    return $request;
}

=head2 prepare_doubles_result_data (request)

    Внутренняя функция, обрабатывает извлеченный из БД результат по заявке(раскладывает на дубли и не дубли)
    Изменяет содержимое входящего объекта request.

=cut
sub prepare_doubles_result_data {
    my $request = shift;
    if ($request->{status} eq 'Done' && defined $request->{result}) {
        $request->{result} = decode_json_and_uncompress($request->{result});
        my $result = $request->{result};
        my $double_phrases = [];
        my $single_phrases = [];
        foreach my $phrase (keys %{$result}) {
            my $arr = ($result->{$phrase}->{is_double}) ? $double_phrases : $single_phrases;
            delete $result->{$phrase}->{is_double};
            push @$arr, hash_merge {phrase=>$phrase}, $result->{$phrase};
        }
        foreach my $phrases ($double_phrases, $single_phrases) {
            $phrases = [xsort {$_->{phrase}} @{$phrases}];
        }
        hash_merge $request, {double_phrases => $double_phrases, single_phrases => $single_phrases};
    }
    # поле уже переработано, разобрано, больше не нужно.
    delete $request->{result};
}

=head2 search_phrase_doubles_for_request

    Ищет все дубли для запроса.
    Запрос - поле params в таблице phrases_doubles_queue
    Возвращает хеш с результатами поиска дублей.

=cut
sub search_phrase_doubles_for_request {
    my $params = shift;

    my $phrases_for_check = [map { {phrase=>$_} } @{$params->{words} || []}];

    my $phrases = [];
    my @chunks = sharded_chunks(cid => $params->{cids}, $CIDS_CHUNK);
    foreach my $chunk (@chunks) {
        push @$phrases, @{get_all_sql(PPC(shard => $chunk->{shard}), 
                                      ["SELECT phrase, b.bid, bi.id FROM bids bi JOIN banners b ON b.pid=bi.pid
                                                              JOIN phrases p ON p.pid=b.pid", 
                                       where => {'p.cid' => $chunk->{cid}, statusArch => 'No'}])};
    };

    my $doubles = get_all_doubles($phrases, phrases_for_check=>$phrases_for_check);
    my $result = {};
    
    foreach my $phrase (keys(%$doubles)) {
        my $phrase_doubles = $doubles->{$phrase};
        my @ph_ids = map {$_->{id}} grep {defined $_->{id}} @$phrase_doubles;
        my @new_phrases = map {$_->{phrase}} grep {! defined $_->{id}} @$phrase_doubles;
        # Если фраза является дублем каких-то фраз из кампаний, тогда записываем id тех фраз, дублями которых она является
        # Устанавливаем флаг is_double = 1
        if (@ph_ids) {
            $result->{$phrase}->{ph_ids} = \@ph_ids;
            $result->{$phrase}->{is_double} = 1;
        }
        # Для фраз, которые оказались дублями, но были взяты из списка для проверки тоже устанавливаем флаг is_double = 1, но список id фраз отсутствует
        foreach (@new_phrases) {
            $result->{$_}->{is_double} = 1;
        }
        # В случае, если фразы из списка сами себе дубли, то первый элемент надо считать НЕ дублем. 
         $result->{$new_phrases[0]}->{is_double} = 0 if @new_phrases && !@ph_ids;
    }
    # Для тех фраз, которые вообще не оказались ни чьими дублями устанавливаем флаг is_double = 0
    foreach (@$phrases_for_check) {
        if (!defined($result->{$_->{phrase}})){
            $result->{$_->{phrase}}->{is_double} = 0;
        }
    }

    return $result;
}

=head2 is_valid_phrases_limit

    Проверяет не превышает ли количество фраз в выбранных кампаниях максимально допустимого значения.

=cut
sub is_valid_phrases_limit {
    my $cids = shift;
    my $counts = get_one_column_sql(PPC(cid => $cids), ["SELECT count(distinct id) FROM bids bi JOIN banners b USING (pid)", 
                                        where=> {'bi.cid' => SHARD_IDS, statusArch => 'No'}]) || [];
    my $res_count = sum 0, @$counts;

    return ($res_count < $MAX_PHRASES_LIMIT) ? 1 : 0;
}

1;
