package PhrasePrice;

use Direct::Modern;

use List::Util qw/min max/;
use List::MoreUtils qw/first_index/;
use Readonly;

use Yandex::Validate;
use Yandex::I18n;

use Currencies qw/get_geo_list get_frontpage_min_price get_currency_constant currency_price_rounding get_min_price/;
use Currency::Format;
use Pokazometer;
use PlacePrice;
use Yandex::HashUtils;
use Yandex::Interpolate qw/interpolate_linear_fast/;

use parent qw/Exporter/;
our @EXPORT = qw/
    phrase_camp_price_context
    phrase_price_context
    validate_phrase_price
    validate_cpm_price
    validate_autobudget_priority
    validate_frontpage_price
    is_bids_limited_by_day_budget
/;
our @EXPORT_OK = qw/
    get_available_places
    get_next_place
/;

Readonly our @AVAILABLE_PLACES_API => (
    map { $PlacePrice::PLACES{$_} } qw/PREMIUM1 PREMIUM2 PREMIUM3 PREMIUM4 GUARANTEE1 GUARANTEE4/
);

Readonly our @AVAILABLE_PLACES_FRONT => (
    map { $PlacePrice::PLACES{$_} } qw/PREMIUM1 PREMIUM2 PREMIUM4 GUARANTEE1 GUARANTEE4/
);

Readonly our @AVAILABLE_PLACES_FRONT_MOBILE_CONTENT => (
    map { $PlacePrice::PLACES{$_} } qw/PREMIUM1 PREMIUM4 GUARANTEE1 GUARANTEE4/
);

Readonly our @AVAILABLE_PLACES_DEFAULT => @AVAILABLE_PLACES_FRONT;

=head2 get_available_places($camp_auto_price_params)

    Возвращает список позиций, которые можно выставить через мастер цен. Список зависит от того, кто вызывает мастер цен
    и это определяется по переданным в него параметрам

=cut
sub get_available_places {
    my ($params) = @_;

    if ($params->{for} && $params->{for} eq 'API') {
        return \@AVAILABLE_PLACES_API;
    }
    if ($params->{for} && $params->{for} eq 'front') {
        return $params->{mobile_content} ? \@AVAILABLE_PLACES_FRONT_MOBILE_CONTENT : \@AVAILABLE_PLACES_FRONT;
    }
    return \@AVAILABLE_PLACES_DEFAULT;
}

=head2 get_next_place($camp_auto_price_params, $place)

    Возвращает следующую по возрастанию позицию после $place
    Используется в мастере цен для выставления ставки какой-то позиции + % от разницы между следующей и этой позицией

=cut
sub get_next_place {
    my ($params, $place) = @_;

    my $all_places = get_available_places($params);
    my $place_index = first_index { $_ == $place } @$all_places;
    if (!defined($place_index) || $place_index == 0) {
        return undef;
    }

    # это странное условие здесь для сохранения поведения после рефакторинга позиций. В DIRECT-68253 нужно убрать
    if ($params->{for} && $params->{for} eq 'API' && $place == $PlacePrice::PLACES{PREMIUM4}) {
        return $PlacePrice::PLACES{PREMIUM2};
    }

    return $all_places->[$place_index - 1];
}

=head2 phrase_camp_price_context($camp, $phrase)

    Цена фразы за контекст
    Для отдельного размещения возвращает цену как есть,
    для остальных стратегий пересчитывает из ставки на поиске и коэффициента
    Используется только в экспорте в БК

=cut

sub phrase_camp_price_context {
    
    my ($camp, $phrase) = @_;

    my $strategy = $camp->{strategy};
    my $currency = $camp->{currency};

    my $price_context = (($strategy // '') eq 'different_places')
        ? $phrase->{price_context}
        : phrase_price_context($phrase->{price}, $camp->{ContextPriceCoef}, $currency);
    return $price_context;
}

=head2 phrase_price_context($price, $ContextPriceCoef, $currency)

    Цена фразы в РСЯ, в проценте от цены на поиске

=cut

sub phrase_price_context {
    my ($price, $ContextPriceCoef, $currency) = @_;

    if (!$ContextPriceCoef || $ContextPriceCoef == 100) {
        return $price;
    } else {
        my $price_context = $price * $ContextPriceCoef / 100;
        return currency_price_rounding($price_context, $currency, up => 1);
    }
}

=head2 auto_price_for_different_places($phrases, $params)

    Расчет цены фразы для кампаниии со стратегией "независимое управление".
    Параметры расчета задаются через мастер цен($params)
    Если мастером цен не меняется ставка на поиске/в сетях, то в $prices->{$id}{price/price_context} возвращается текущее значение ставки из БД.

=cut

sub auto_price_for_different_places {
    my ($phrases, $params) = @_;
    my $currency = $params->{currency};
    die 'no currency given' unless $currency;

    my %prices;
    my $code = $params->{single} ? \&_auto_price_single : \&_auto_price_wizard;
    my $have_to_update_search_price = $params->{strategy}->{search}->{name} ne 'stop';
    foreach my $phrase (@$phrases) {
        my ($price, $price_context) = $code->($phrase, $params);
        # здесь ожидается что $code->() вернет либо обе ставки, либо никаких.
        # если в мастере цен не выставляется какая-то из ставок, возвращается текущая ставка на фразе
        next unless defined($price) && defined($price_context);

        my $id = $phrase->{id};
        $prices{$id} = hash_cut $phrase, qw/premium guarantee/;

        if ($have_to_update_search_price) {
            $price = max($price // $phrase->{price}, get_currency_constant($currency, 'MIN_PRICE'));
        }

        $prices{$id}->{price} = $price;
        $prices{$id}->{price_context} = $price_context;
    }
    return \%prices;
}

sub _auto_price_single {
    my ($phrase, $wizard_params) = @_;
    return (
        $wizard_params->{single}->{price} || $phrase->{price},
        $wizard_params->{single}->{price_ctx} || $phrase->{price_context},
    );
}

sub _auto_price_wizard {
    my ($phrase, $params) = @_;

=head2 about currency_price_rounding for $price
    
    Логика про округление цены всегда должна находится после определения цены,
    поэтому если появятся другие места, изменяющие или определяющие цену на поиске $price,
    то код округления должен быть перемещен дальше или добавлен по месту.

    При этом важно не потерять тот факт, что если вместо цены установили клиентское ограничение (max_price) - то округлять его не следует - DIRECT-28915

=cut
    my $price;
    if (ref $params->{on_search}) {

        my $search = $params->{on_search};
        if ($phrase->{rank} && $search->{update_phrases}) {
            # Если заполнены не все позиции, то не продолжать.
            return if @{$phrase->{premium}} < $PlacePrice::PREMIUM_PLACE_COUNT || @{$phrase->{guarantee}} < $PlacePrice::GUARANTEE_PLACE_COUNT;

            if ($search->{use_position_ctr_correction}) {
                # не меняем ставку, если торги недоступны ($price = undef)
                $price = get_bid_for_position_ctr_correction($phrase, $search);
                
            } else {

                $price = PlacePrice::get_bid_price_by_place($phrase, $search->{price_base}) / 1_000_000;
                my $next_place = get_next_place($params, $search->{price_base});
                my $next_price = defined($next_place) ? PlacePrice::get_bid_price_by_place($phrase, $next_place) / 1_000_000 : $price;
                my $diff = $next_price - $price;

                $diff = 0 if $diff < 0;
                if ($search->{proc}) {
                    $price = $search->{proc_base} eq 'value'
                        ? $price + $price * $search->{proc} / 100
                        : $price + $diff * $search->{proc} / 100;
                }

                # ставим дефолтную цену, если торги недоступны
                $price = get_currency_constant($params->{currency}, 'DEFAULT_PRICE')  if $phrase->{nobsdata};
            }

            if ($search->{max_price} && defined($price) && $price > $search->{max_price}) {
                $price = $search->{max_price};
            } elsif (defined $price && $params->{strategy}->{search}->{name} ne 'stop') {
                # если показы на поиске не отключены
                # это $have_to_update_search_price из auto_price_for_different_places - условия должны совпадать!
                $price = currency_price_rounding($price, $params->{currency}, up => 1);
            }
        }
    }

=head2 about currency_price_rounding for $price_ctx
    
    Округление цены должны быть везде, где цена есть и это не max_price
    При появлении новых мест - дополнить округлением.

=cut
    my $price_ctx;
    if (ref $params->{on_ctx} && $phrase->{pokazometer_data}
        && (!defined $params->{on_ctx}->{update_phrases} || $params->{on_ctx}->{update_phrases})) {

        my $context = $params->{on_ctx};
        my $pokazometr_price = get_price_for_coverage($phrase->{pokazometer_data}, $context->{scope} * 0.01);
        if (defined $pokazometr_price) {
            $price_ctx = $pokazometr_price;
            $price_ctx += $price_ctx * $context->{proc} / 100 if $context->{proc};
            if($context->{max_price} && $price_ctx > $context->{max_price}) {
                $price_ctx = $context->{max_price};
            } else {
                $price_ctx = currency_price_rounding($price_ctx, $params->{currency}, up => 1);
            }

        } elsif (!$context->{auto_max_price}) { # флаг, что параметр автоматически назначен API, где он необязателен
            $price_ctx = $context->{max_price} || get_currency_constant($params->{currency}, 'DEFAULT_PRICE'); 
        }
        undef $price_ctx unless $price_ctx;
    }

    return (
        $price || $phrase->{price},
        $price_ctx || $phrase->{price_context}
   );
}


=head2 auto_price_for_other($bid, $phrases, $params)

    Расчет цены фразы для кампаниии со стратегией отличной от "независимое управление".
    Параметры расчета задаются через мастер цен($params)

=cut

sub auto_price_for_other {
    
    my ($phrases, $params) = @_;

    my $currency = $params->{currency};
    die 'no currency given' unless $currency;

    my $prices = {};
    for my $phrase (@$phrases) {

        my $id = $phrase->{id};

        # calc price .................................
        my ($price, $price_context);
        
        if (defined $params->{single_price}) {
            my $platform = $params->{platform}||'search';
            if ( $platform eq 'search' ) {
                $price = $params->{single_price};
            } elsif ( $platform eq 'both' ) {
                $price = $params->{single_price};
            }
        } 
        else {
            # TODO: не пытаться обновлять ставки, если $phrase->{no_bs_data}

            if($phrase->{rank} == 0 and $phrase->{context_stop_flag} == 0 
                and defined $params->{platform} and $params->{platform} =~ /^(network|both)$/
            ) {
                if ($phrase->{pokazometer_data}) {
                    my $price_for_coverage = (get_price_for_coverage($phrase->{pokazometer_data}, $params->{scope}*0.01) || 0);
                    $price_context = min ($price_for_coverage, $params->{max_price}) ;
                }
                if (defined $price_context && $params->{proc}) {
                    $price_context += $price_context * $params->{proc} / 100;
                }
            }
            elsif($phrase->{rank} && $phrase->{rank} != 0 and (not defined $params->{platform} or $params->{platform} ne 'network')  ) {
                # пропускаем фразы или рубрики если не был выставлен соотв. чекбокс
                next if ! $params->{update_phrases}; # пропускаем фразы

                # Если заполнены не все позиции, то не продолжать.
                next if @{$phrase->{premium}} < $PlacePrice::PREMIUM_PLACE_COUNT || @{$phrase->{guarantee}} < $PlacePrice::GUARANTEE_PLACE_COUNT;

                if ($params->{use_position_ctr_correction}) {
                    # не меняем ставку, если торги недоступны ($price = undef)
                    $price = get_bid_for_position_ctr_correction($phrase, $params);
                    
                } else {
 
                    $price = PlacePrice::get_bid_price_by_place($phrase, $params->{price_base}) / 1_000_000;
                    my $next_place = get_next_place($params, $params->{price_base});
                    my $next_price = defined($next_place) ? PlacePrice::get_bid_price_by_place($phrase, $next_place) / 1_000_000 : $price;
                    my $diff = $next_price - $price;

                    $diff = 0 if $diff < 0;

                    # Прибавляем проценты
                    if ($params->{proc_base} eq 'value') {
                        $price = $price + $price * $params->{proc} / 100;
                    } elsif ($params->{proc_base} eq 'diff') {
                        $price = $price + $diff * $params->{proc} / 100;
                    }
                }

                # Сравниваем с максимальной ценой
                if (defined($price) && $price > $params->{max_price}) {
                    $price = $params->{max_price}; 
                }
            }

            for my $p ($price, $price_context) {
                next unless defined $p;
                # не округляем, если это максимальная ставка, указанная пользователем
                next if $p == $params->{max_price};
                $p = currency_price_rounding($p, $currency, up => 1);
            }
        }

        if (!defined $price and !defined $price_context ) {
            # ничего не делаем, если нет данных о новых ценах
            next;
        }

        # set new price ..............................
        $prices->{$id} = hash_cut $phrase, qw/price premium guarantee/;
        $prices->{$id}->{price} = $price if defined $price; # Если $price была отдельно инициализирована
        $prices->{$id}->{price_context} = $price_context if defined $price_context;
    }

    return $prices;
}

=head2 get_bid_for_position_ctr_correction($phrase, $params) 

    Расчет ставки для фразы в мастере цен
    По заданному х ($params->{position_ctr_correction}) и данным от торгов ($phrase->{clickometer})
    вычисляет значение bid_price,
    прибавляет к найденной ставке проценты ($params->{proc})
    Если нет данных, то вернет undef 

=cut

sub get_bid_for_position_ctr_correction {
    my ($phrase, $params) = @_;
    return undef if $phrase->{nobsdata};

    my $bid;
    # для mcbanner торги ставку записывают в поле price_for_mcbanner (одномерный массив с единственным элементом)
    if ($params->{mcbanner}) {
        if ($phrase->{price_for_mcbanner} && @{ $phrase->{price_for_mcbanner} } && 
            $phrase->{price_for_mcbanner}->[0]->{bid_price}) {
            $bid = $phrase->{price_for_mcbanner}->[0]->{bid_price} / 1_000_000;
        } else { # $bid = undef
        }
    } else {
        my $traffic_volume = $phrase->{traffic_volume} // {};
        my $data = [map { {position_ctr_correction => $_, bid_price => $traffic_volume->{$_}->{bid_price}} } keys %$traffic_volume];
        $bid = _get_interpolated_bid($data, $params->{position_ctr_correction});
    }

    if (defined($bid) && $params->{proc}) {
        $bid += $bid * $params->{proc} / 100;
    }
    return $bid;
}

sub _get_interpolated_bid {
    my ($data, $position_ctr_correction) = @_;

    my @sorted_data = sort { $a->{x} <=> $b->{x} or $a->{y} <=> $b->{y} } 
                       map { {x => $_->{position_ctr_correction}, y => $_->{bid_price}} } 
                      grep { defined($_->{bid_price}) && $_->{bid_price} > 0 } 
                      grep { defined($_->{position_ctr_correction}) && $_->{position_ctr_correction} > 0 } 
                      @{ $data // [] };

    if (! (defined($position_ctr_correction) && @sorted_data) ) {
        return undef;
    }
    my $bid; 
    # если больше максимального, то берем ставку максимального
    if (($position_ctr_correction eq 'max') || 
        ($position_ctr_correction > $sorted_data[ @sorted_data - 1 ]->{x})) {
        $bid = $sorted_data[ @sorted_data - 1 ]->{y};

    # если меньше минимального, то берем ставку минимального
    } elsif ($position_ctr_correction < $sorted_data[0]->{x}) {
        $bid = $sorted_data[0]->{y};

    # линейная интерполяция
    } else {
        $bid = interpolate_linear_fast($position_ctr_correction, \@sorted_data);
    }
    return $bid / 1_000_000;
}


=head2 is_bids_limited_by_day_budget

    Определить, доступна ли опция ограничения ставок дневным бюджетом

    Результат:
        0/1 -  доступна ли опция bs_bids_limited_by_day_budget

=cut

sub is_bids_limited_by_day_budget {
    state $is_bids_limited_by_day_budget_by_property = Property->new('bs_bids_limited_by_day_budget');

    return $is_bids_limited_by_day_budget_by_property->get(60);
}

=head2 validate_phrase_price

    Валидация цены размещения по фразе/в каталоге

    $error = validate_phrase_price($price, $currency, %O);
    $error => undef|"текст ошибки"

    %O = (
        dont_support_comma => 1,
        dont_validate_min_price => 1,
    );

=cut

sub validate_phrase_price {
    my ($price, $currency, %O) = @_;

    die "no currency given" unless $currency;

    if (defined $price && !$O{dont_support_comma}) {
        $price =~ s/,/./;
    }

    if (defined $price && is_valid_float($price)) {
        if ($price < get_currency_constant($currency, 'MIN_PRICE') && !$O{dont_validate_min_price} || $price < 0) {
            return iget('Цена не может быть меньше %s', format_const($currency, 'MIN_PRICE'));
        } elsif ($price > get_currency_constant($currency, 'MAX_PRICE')) {
            return iget('Цена не может быть больше %s', format_const($currency, 'MAX_PRICE'));
        }
    } else {
        my $price_str = Yandex::ScalarUtils::str($price);
        return iget("Некорректная цена: '$price_str'");
    }

    return undef;
}

=head2 validate_cpm_price

    Валидация цены за тыс. показов

    $error = validate_cpm_price($price, $currency);

=cut

sub validate_cpm_price {
    my ($price, $currency, %options) = @_;

    die "no currency given" unless $currency;

    $price =~ s/,/./ if defined $price && !$options{dont_support_comma};

    if (!defined $price || !is_valid_float($price)) {
        return iget("Некорректная цена за тыс. показов: '".Yandex::ScalarUtils::str($price)."'");
    } elsif ($price < get_currency_constant($currency, 'MIN_CPM_PRICE') || $price < 0) {
        return iget('Цена за тыс. показов не может быть меньше %s', format_const($currency, 'MIN_CPM_PRICE'));
    } elsif ($price > get_currency_constant($currency, 'MAX_CPM_PRICE')) {
        return iget('Цена за тыс. показов не может быть больше %s', format_const($currency, 'MAX_CPM_PRICE'));
    }

    return;
}

=head2 validate_frontpage_price

    Валидация цены за тыс. показов на Главной

    $error = validate_frontpage_price($price, $currency, $camp_geo, $pages, %options);

=cut

sub validate_frontpage_price {
    my ($price, $currency, $frontpage_types, $camp_geo, $client_id, %options) = @_;

    die "no currency given" unless $currency;
    $price =~ s/,/./ if defined $price && !$options{dont_support_comma};

    if (!defined $price || !is_valid_float($price)) {
        return iget("Некорректная цена за тыс. показов: '".Yandex::ScalarUtils::str($price)."'");
    } else {
        my $min_price = get_min_price($currency, $camp_geo, $frontpage_types, $client_id);

        if ($price < $min_price || $price < 0) {
            return iget('Цена за тыс. показов не может быть меньше %s %s', $min_price, format_currency($currency));
        } elsif ($price > get_currency_constant($currency, 'MAX_CPM_PRICE')) {
            return iget('Цена за тыс. показов не может быть больше %s', format_const($currency, 'MAX_CPM_PRICE'));
        }
    }

    return;
}

=head2 validate_autobudget_priority($autobudget_priority)

    Проверка приоритета автобюджета для фразы
    
    Результат:
        текстовая строка с ошибкой или undef в случае корректности приоритета

=cut

sub validate_autobudget_priority {
	
	my ($autobudget_priority) = @_;
	
	if (!defined $autobudget_priority) {
		return iget('Необходимо задать приоритет');
	} 	
	if ($autobudget_priority !~ /^(1|3|5)$/) {
		return iget('Задан некорректный приоритет');		
	}
	
	return undef;
}

1;
