package PhraseText;

=head1 NAME
    
    PhraseText -- модуль для работы с текстами фраз

=head1 DESCRIPTION

    Не работает с БД, использует только утилиты для работы с текстом. 

    $Id$

=cut

use strict;
use warnings;

use Yandex::HashUtils;
use Yandex::DBTools;
use Direct::PhraseTools qw/ polish_phrase_text /;

use Settings;
use GeoTools qw/get_geo_numbers/;
use HashingTools;
use TextTools;
use Yandex::HighlightWords;
use Yandex::I18n;
use Yandex::MyGoodWords;
use List::MoreUtils qw/all uniq/;

use base qw/Exporter/;
our @EXPORT = qw(
    validate_key_phrase
    cmp_phrases

    ensure_phrase_have_props
    get_phrase_props
    get_phrase_hash

    process_phrase_brackets
    split_phrase_with_normalize

    add_plus_to_minus_stop_words
    );

use utf8; 

#-----------------------------------------------------------

=head2 split_phrase_with_normalize

    разделяем фразу на слова (нормализовав их), оставляем стоп-слова, если фраза в кавычках

=cut

sub split_phrase_with_normalize {
    my ($phrase, %options) = @_;

    # если фраза закавычена, стопслова считаются обычными словами и не выкидываются
    my $preserve_stopwords = $options{preserve_stopwords} || ($phrase =~ /^".+"$/) ? 1 : 0;

    # помним что далее $phrase_norm теряет минус-слова
    my $phrase_norm = join ' ', map {$_->[1]} @{Yandex::MyGoodWords::_norm_words($phrase, undef, undef, undef, $preserve_stopwords)||[]};

    smartstrip($phrase_norm);
    $phrase_norm =~ s/(^\"|\"$)//;

    return split(/\s+/, $phrase_norm);
}

=head2 norm_phrase($text)

    Разбивка(пробелами) и нормализация слов из фразы

=cut

sub norm_phrase {
    
    my $text = shift;
    my %words;
    foreach (split /\s+/, $text) {
        next if /^[\-\+\~\!\&\%\$\^\_\"\]\[]$/;
        m/^(\-?)(.*)$/;
        my $_norm_words = Yandex::MyGoodWords::_norm_words($2);
        $_norm_words = [[]] if ref $_norm_words eq 'ARRAY' && !@$_norm_words;
        $words{@{$_norm_words||[[]]}[0]->[1]} = $1
    }
    return \%words;
}

=head2 validate_key_phrase

    проверяет слова ключевой фразы, по которым показывается объявление, на "непустоту"
    под этим понимается, что минус-слова не должны "уничтожать" плюс-слова :)

    для точных плюс-слов проверяется два условия:
        нет такого же точного минус-слова;
        леммы этого слова не пересекаются с леммами всех (!) неточных минус-слов;
    для неточных плюс-слов одно условие:
        хотя бы одна лемма каждого слова не имеет соответствующей леммы неточного минус-слова;

    (А не надо ли в леммы неточных минус-слов добавить все точные минус-слова?)

    * в Forecast::Autobudget::get_moneymeter_data полагаемся на то, что данная функция не учитывает взаимосвязи минус-слов между собой

    Ответ: ссылка на массив минус-слов, которые не прошли валидацию

=cut

sub validate_key_phrase {
    my ($input_phrase, $input_minus_words) = @_;

    my @pluswords;
    my (%precise_minuswords, %minuswords, %minus_lemma, @mwords);
    my @result = ();

    # prepare
    foreach my $word ( split /\s+/, $input_phrase ) {
        $word =~ s/\"//g;
        my $is_minus = $word =~ /^\-/ ? 1 : 0;
        $word =~ s/^\-//;

        # точные слова типа "!в" тоже проходят эту проверку, т.к. в словарике их, очевидно, нет
        next if Yandex::MyGoodWords::is_stopword($word);
        
        if ( $is_minus ) {
            push @mwords, $word;
        } else {
            push @pluswords, $word; # can contain `!`
        }
    }

    foreach my $word (@mwords, @$input_minus_words) {
        # Мы не хотим править полученные слова в input_minus_words так как они могут использоваться вовне
        next if Yandex::MyGoodWords::is_stopword($word);

        my $wordcopy = $word;
        $wordcopy =~ s/^(!|\+)//;

        if ( $word =~ /^!/ && !Yandex::MyGoodWords::is_stopword($wordcopy) ) {
            $precise_minuswords{$wordcopy} = $word;
        } else {
            my ($lemmas, $forms) = Yandex::MyGoodWords::get_lemmas_formas_list_words($wordcopy);
            @minuswords{@$forms} = ($word) x @$forms;
            @minus_lemma{@$lemmas} = ($word) x @$lemmas
        }
    }

    my @minus_lemmas = keys %minus_lemma;
    foreach my $plusword ( @pluswords ) {
        $plusword =~ s/^(!|\+)//;
        my $precise = $1 && $1 eq '!';

        my ($lemmas, $forms) = Yandex::MyGoodWords::get_lemmas_formas_list_words($plusword);
        my %plus_lemmas = map { $_ => 1 } @$lemmas;
        if ( $precise && !Yandex::MyGoodWords::is_stopword($plusword) ) {
            if ( $precise_minuswords{$plusword} ) {
                push @result, $precise_minuswords{$plusword};
                next;
            }

            if ( grep { $plus_lemmas{$_} } @minus_lemmas ) {
                push @result, @minus_lemma{@$lemmas};
                next;
            }

        } else {
            my @bad_lemmas = grep {$minus_lemma{$_}} @$lemmas;
            if (@bad_lemmas > 0 && @bad_lemmas == @$lemmas) {
                push @result, uniq map {$minus_lemma{$_}} @bad_lemmas;
                next;
            }

            my @badwords = grep {$minuswords{$_}} @$forms;
            if (@badwords == @$forms) {
                push @result, uniq map {$minuswords{$_}} @badwords;
                next;
            }

        }
    }

    return \@result;
}

=head2 cmp_phrases

=cut 

sub cmp_phrases
{
    my $norm_words_hash = shift;
    my $word = shift; # no norm word
    my $use_only_first_norm_form = shift;

    # ref array
    my $nwords = $word =~ /^!/ ? [$word] : Yandex::MyGoodWords::get_whole_list_norm_words( $word ); 
    $#{ $nwords } = 0 if $use_only_first_norm_form;

    foreach my $k ( @{ $nwords } ) {
        return 1 if defined $norm_words_hash->{ $k };
    }
    return 0;
}



=head2 ensure_phrase_have_props

    Обеспечивает наличие в данных о фразе следующих ключей:
        norm_phrase — нормализованная фраза 
        numword     — количество слов, ограниченное числом $Settings::MAX_WORDS_IN_KEYPHRASE
        md5         — md5 хэш нормальной формы в hex
        norm_hash   — число, 64-ёх битный хэш нормальной формы (только если указано with_norm_hash)
    На вход принимает ссылку на хеш с данными, описывающими фразу. В хеше обязательно должен быть ключ
    phrase с текстом самой фразы. Если каких-то ключей из списка выше не хватает, они будут добавлены.
    Также принимает опциональные именованные параметры:
        with_norm_hash — обеспечивать также наличие ключа norm_hash
    ВНИМАНИЕ: модифицирет переданный хеш in-place. Возвращает ссылку на него же.

    $phrase = {phrase => 'окна пвх'};
    ensure_phrase_have_props($phrase, with_norm_hash => 1);
    $phrase => {
        phrase => 'окна пвх',
        numword => 2,
        norm_phrase => 'окно пвх',
        norm_hash => '9595796445660914630',
        md5 => '852b1e06d09323c6ec2e5346f34d3017',
    };

=cut

sub ensure_phrase_have_props {
    my ($phrase_hash, %O) = @_;

    die 'bad phrase data' unless $phrase_hash && ref($phrase_hash) eq 'HASH' && (exists $phrase_hash->{phrase} || exists $phrase_hash->{phr});

    my $phrase = exists $phrase_hash->{phrase} ? $phrase_hash->{phrase} : $phrase_hash->{phr};
    my @fields = qw/numword norm_phrase norm_hash md5/;
    # если не указан with_norm_hash, то наличие norm_hash не проверяем
    # но если уж не хватает чего-то ещё, то копируем в том числе и norm_hash, раз уж get_phrase_props его считает
    if ( grep { !defined $phrase_hash->{$_} && ($_ ne 'norm_hash' || $O{with_norm_hash}) } @fields ) {
        my $phrase_props = get_phrase_props($phrase);
        hash_copy $phrase_hash, $phrase_props, @fields;
    }

    return $phrase_hash;
}

=head2 get_phrase_props

    По фразе получить норм. форму, хеши, numword
    Параметры:
        текст фразы
    Результат:
        ссылка на хэш с ключами
            phrase - немного причесанная исходная фраза
            norm_phrase - нормальная форма
            numword - количество слов, ограниченное числом $Settings::MAX_WORDS_IN_KEYPHRASE
            md5 - md5 хэш нормальной формы в hex
            norm_hash - число, 64-ёх битный хэш нормальной формы
        если фраза undef или пустая - возвращаем undef

=cut

sub get_phrase_props {
    my $phrase = shift;

    $phrase =~ s/\\*//g;
    $phrase =~ s/\-\s+/-/g;
    $phrase =~ s/^\s+//g;
    $phrase =~ s/\s+$//g;
    
    my $norm_phrase = Direct::PhraseTools::get_norm_phrase($phrase);
    return undef if !defined $norm_phrase;

    my @Words = split(/ /, $norm_phrase);
    my $numword = @Words > $Settings::MAX_WORDS_IN_KEYPHRASE ? $Settings::MAX_WORDS_IN_KEYPHRASE : @Words;

    my $md5 = md5_hex_utf8($norm_phrase);

    return {
        phrase      => $phrase, 
        norm_phrase => $norm_phrase, 
        numword     => $numword,
        md5         => $md5,
        norm_hash   => half_md5hex_hash($md5),
    };
}

=head2 process_phrase_brackets

  замена скобок на несколько фраз
  удаление лишних пробелов
  удаление "+" в фразах в кавычках

  process_phrase_brackets($b->{phrases});
  return error or false

  args:
    'ph1, ph2, ph3'  or ['ph1','ph2','ph3']

=cut
sub process_phrase_brackets
{
    my $phrases = $_[0] or return; 
    my @phs = ();

    if( ref $phrases eq 'ARRAY') {
        @phs = @$phrases;
    } else {
        @phs = split /\s*,\s*/, $phrases;
    }

    my $flag = 1;
    while ($flag) {
        $flag = 0;
        @phs = grep {defined $_} @phs; # remove deleted items (with "(a|b)")
        for my $i (0 .. $#phs) {
            if ($phs[$i] =~ / \( ( [^()]+ ) \) /x) {
                my $variants = $1;

                # check on error in use '|'; must "a_word | b_word | c_word"
                return iget("Ошибочное использование фраз внутри скобок") unless $variants =~ /^ \s* [^|]+ ( \s* \| \s* [^|]+)+ \s* $/x;

                # replace one old phrase with variants, on multiple new
                for my $word (split /\|/, $variants) {
                    $word =~ s/^\s+//;
                    $word =~ s/\s+$//;
                    return iget("Ошибочное использование фраз внутри скобок") if length($word) == 0;
                    my $new_ph = $phs[$i];
                    $new_ph =~ s/\( [^()]+ \)/ $word /x;
                    push @phs, $new_ph;
                    return iget("Превышен лимит по количеству символов в поле ключевых фраз при использовании оператора (|)") if $#phs > 1000;
                }

                splice @phs, $i, 1; # old phrase with "(a|b)"
                $flag = 1;
                last;
            }
        }
    }

    return iget("Ошибка в использовании оператора, возможно, использованы непарные скобки") if grep {m/[()|]/} @phs;

    @phs = map {polish_phrase_text($_)} @phs;
    
    if( ref $phrases eq 'ARRAY') {
        $_[0] = [@phs];
    } else {
        $_[0] = join(',', @phs);
    }
    
    return;
}

=head2 add_plus_to_minus_stop_words (phrases)

    Функция просматривает все минус-слова в фразах и в случае минусования стоп-слова, добавляет к нему символ "!".
    На входе: массив объектов вида: {phrase => 'фраза'....}
    В процессе работы изменяет исходные данные: поле phrase

=cut
sub add_plus_to_minus_stop_words {
    my $phrases = shift;
    for my $phrase (@{$phrases}) {
        my @words = split " ", $phrase->{phrase};
        for my $word (@words) {
            my ($wo_minus_word) = $word =~ /^-(.+)/;
            # Интересуют только минус слова, которые являются стоп-словами.
            next unless ($wo_minus_word && Yandex::MyGoodWords::is_stopword($wo_minus_word));
            $word = "-!$wo_minus_word";
        }
        $phrase->{phrase} = join " ", @words;
    }
}


=head2 get_phrase_geo_id

    Сопоставляем фразе geo_id, если можем
    Смотрим на явное вхождение топонима

=cut

{
my $synonims = {
    'питер' => 'Санкт-Петербург',
    'санкт' => 'Санкт-Петербург',
    'петербург' => 'Санкт-Петербург',
    'нижний' => 'Нижний Новгород',
    'подмосковье' => 'Москва и область',
};

my $highlighter;
sub get_phrase_geo_id
{
    my $phrase = shift;

    $highlighter ||= Yandex::HighlightWords->new(
        {
            files => ['towns.txt'], # из исходного файла удалены слова "Нижний", "Новгород"
            filespath => $Settings::ROOT.'/protected/data/suggestions',
            phrases => ['подмосковье'],
        }
    );
    my $res = $highlighter->highlight($phrase, {}, {not_change_phrase => 1});
    if ($res->{flag}) {
        my @toponims = map {(undef, my $t) = split '_', $_, 2; $t} keys %{ $res->{sources} };
        for my $toponim (@toponims) {
            if (my $synonim = $synonims->{lc($toponim)}) {
                return get_geo_numbers($synonim);
            } elsif (my $geo_id = get_geo_numbers($toponim)) {
                return $geo_id;
            }
        }
    }
    return 0;
}
}

=head2 get_phrase_hash
    Получить хэш от фразы -- например, для подсказок
    Все минус-слова выкидываются, кавычки и восклицательные знаки удаляются
=cut
sub get_phrase_hash {
    my $phrase = shift;

    $phrase =~ s/[!\"+]//g;
    $phrase =~ s/ -.*//;
    
    return url_hash_utf8( Yandex::MyGoodWords::norm_words($phrase) ); 
}

=head2 

    Разбираем ключевую фразу (с минус-словами) в хеш: 

    На входе строка: "стиральная машина -вятка -малютка" 
    На выходе хеш: {phrase_plus=> 'стиральная машина', minus_words=>['вятка', 'малютка'] } 

=cut
sub phrase_text2detail_hash
{
    my ($text ) = @_;

    $text =~ /^(.*?)(?: (-.*))?$/;
    my $detail = {
        phrase_plus => $1, 
        minus_words => [map {s/^-//; $_} split(" ", $2||'')],
    };

    return $detail;
}

=head2 mass_get_ctr_data

=cut

sub mass_get_ctr_data {
    my ($norm_hashes) = @_;

    return get_hashes_hash_sql(PPCDICT, ['SELECT hash, ctr, pctr AS p_ctr FROM forecast_ctr', WHERE => { hash => $norm_hashes }]);
}

=head2 get_ctr_data

    Возвращает среднесистемный CTR по фразе с указанным хешем

    $ctr_data = get_ctr_data($norm_hash);
    $ctr_data => {
        ctr =>
        p_ctr =>
    }

=cut

sub get_ctr_data {
    my ($norm_hash) = @_;

    my $mass_ctr_data = mass_get_ctr_data([$norm_hash]);
    return $mass_ctr_data->{$norm_hash};
}

=head2 get_norm_hash_for_ctr_data

=cut

sub get_norm_hash_for_ctr_data {
    my ($norm_phrase) = @_;

    # нормальная форма без точных совпадений
    (my $norm_unex = $norm_phrase) =~ s/[!"+]//g;
    $norm_unex = Yandex::MyGoodWords::norm_words($norm_unex);
    my $norm_hash = url_hash_utf8( $norm_unex );

    return $norm_hash;
}

=head2 mass_get_norm_phrases_ctr_data

    Получает по переданному списку нормализованных текстов фраз данные об их прогнозном CTRе

    $norm_phrase2ctr_data = mass_get_norm_phrases_ctr_data(\@norm_phrases);
    $norm_phrase2ctr_data => {
        $norm_phrase1 => {
            ctr =>,
            p_ctr =>,
        },
        ...
    }

=cut

sub mass_get_norm_phrases_ctr_data {
    my ($norm_phrases) = @_;

    my %norm_hash2norm_phrases;
    for my $norm_phrase (@$norm_phrases) {
        my $norm_hash = get_norm_hash_for_ctr_data($norm_phrase);
        push @{$norm_hash2norm_phrases{$norm_hash}}, $norm_phrase;
    }

    my $ctr_data = get_all_sql(PPCDICT, ['
        SELECT hash, ctr, pctr AS p_ctr
        FROM forecast_ctr
        WHERE', { hash => [ keys %norm_hash2norm_phrases ] },
    ]);

    my %norm_phrase2ctr_data;
    for my $ctr_data_row (@$ctr_data) {
        my $norm_hash = $ctr_data_row->{hash};
        my $ctr_data = hash_cut $ctr_data_row, qw(ctr p_ctr);
        my $norm_phrases = $norm_hash2norm_phrases{$norm_hash};
        for my $norm_phrase (@$norm_phrases) {
            $norm_phrase2ctr_data{ $norm_phrase } = $ctr_data;
        }
    }

    return \%norm_phrase2ctr_data;
}

1;
