use strict;
use warnings;
use utf8;

package PhrasesAuction;

=head1 NAME
    
    PhraseAuction - работа с данными аукциона по фразам

=head1 DESCRIPTION

    Объединяет работу с БД + аукцион (Phrase.pm + BsAuction.pm) 

=cut

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use List::UtilsBy qw/partition_by/;
use List::MoreUtils qw/uniq/;

use Settings;
use TextTools;
use BS::TrafaretAuction qw/trafaret_auction/;
use TimeTarget;
use PhrasePrice;
use Pokazometer;
use Models::AdGroup ();

=head2 get_auction_phrases(%filters)

    Выборка фраз не архивных кампаний c получением данных аукциона.

    Параметры:
        pid => id групп объявлений (число или массив чисел pid || [pid1, pid2, ...]) (обязательный параметр)
        phid => id ключевых фраз (число или массив чисел phid || [phid1, phid2, ...])
        only_auction_phrases => 1 | 0 выбирать только группы объявлений за которые можно торговаться (фактически по фразам группы есть аукцион)
        skip_pokazometer => 1 | 0 - не ходить в показометр
        exclude_price => текущая цена фразы, фразы с такой ценой пропускаются при выборке из БД
        exclude_price_context => текущая цена фразы для контекста, фразы с такой ценой пропускаются при выборке из БД

    Если заданы и exclude_price, и exclude_price_context, пропускаются только фразы, у которых совпадают обе цены
         
    Результат:
        $phrases - [$phrase1, $phrase2, $phrase3] массив фраз
        Каждая фраза - это ссылка на хеш {} c данными аукциона
        {
            phrase =>
            price =>  
            price_context => 
            id => 
            pid => 
            PhraseID => 
            is_suspended => 1|0
            nobsdata => 1 # данные аукциона не полученны 
        }
=cut

sub get_auction_phrases {
    my %filters = @_;
    return _get_auction_phrases(\%filters, no_pokazometer => $filters{skip_pokazometer} // 0);
}

sub _get_auction_phrases {
    my ($filters, %opt) = @_; 

    my %condition = %{hash_cut $filters, qw/pid phid/};
    my $groups = get_groups_for_auction(
        %condition,
        only_auction_phrases => $filters->{only_auction_phrases},
        exclude_price => $filters->{exclude_price},
        exclude_price_context => $filters->{exclude_price_context}
    );
    Models::AdGroup::update_phrases_shows_forecast($groups);
    trafaret_auction($groups);
    safe_pokazometer($groups, net => 'both', get_all_phrases => 1) unless $opt{no_pokazometer};

    my @phrases;
    foreach my $group (@$groups) {
        if ( exists $group->{pokazometer_failed} ) {
            for my $phrase ( @{ $group->{phrases} } ) {
                $phrase->{pokazometer_failed} = $group->{pokazometer_failed};
            }
        }

        if ($group->{strategy} && $group->{strategy} eq 'different_places') {
            push @phrases, @{$group->{phrases}};
            next;
        }

        foreach my $phrase (@{$group->{phrases}}) {
            # для фраз:
            # без цены_для_контекста только что отключенные на поиске.
            # с ценой для контекста, заново запущенные на поиске.
            # работающие на поиске, с нулевой ценой для контекста.
            $phrase->{price_context} = phrase_price_context($phrase->{price}, $group->{ContextPriceCoef}, $group->{currency});
        }

        push @phrases, @{$group->{phrases}};
    }

    return \@phrases;
}



=head2 get_groups_for_auction (%filters)
    
    Выборка групп объявлений с необходимыми данными для запроса торгов в БК
    
    Параметры:
        pid => id групп объявлений (число или массив чисел pid || [pid1, pid2, ...]) (обязательный параметр) 
        phid => id ключевых фраз (число или массив чисел phid || [phid1, phid2, ...])
        only_auction_phrases => 1 | 0 выбирать только группы объявлений за которые можно торговаться (фактически по фразам группы есть аукцион)
        exclude_price => текущая цена фразы, фразы с такой ценой пропускаются при выборке из БД
        exclude_price_context => текущая цена фразы для контекста, фразы с такой ценой пропускаются при выборке из БД

    Если заданы и exclude_price, и exclude_price_context, пропускаются только фразы, у которых совпадают обе цены

    Результат:
        $groups - [$group1, $group2] массив групп объявлений
        Каждая группа - это ссылка на хэш с полями
        {
            pid => ,
            geo => 
            filter_domain => # данные 
            phone => 
            title => 
            body => 
            BannerID =>  
            banners_quantity => # число баннеров в группе
            image_BannerID => 
            statusShowsForecast =>
            phrases => []
        }
        
        данные filter_domain phone title body BannerID image_BannerID относятся к одному случайно
        выбранному баннеру из группы
            
=cut

sub get_groups_for_auction {
    
    my %filters = @_;
    my $where = {};
    $where->{'bi.pid'} = SHARD_IDS;
    
    $where->{'bi.id'} = $filters{phid} if $filters{phid};
    if ($filters{exclude_price} && $filters{exclude_price_context}) {
        $where->{_OR} = {
            'bi.price__ne' => $filters{exclude_price},
            'bi.price_context__ne' => $filters{exclude_price_context},
        };
    } elsif ($filters{exclude_price}) {
        $where->{'bi.price__ne'} = $filters{exclude_price};
    } elsif ($filters{exclude_price_context}) {
        $where->{'bi.price_context__ne'} = $filters{exclude_price_context};
    }
    if ($filters{only_auction_phrases}) {
        $where->{"p.is_bs_rarely_loaded"} = 0;
    }

    my $phrases = get_all_sql(PPC(pid => $filters{pid}), [
        "SELECT
            bi.phrase, bi.phrase AS phr, bi.price, bi.price_context, bi.id,
            bi.cid, bi.PhraseID, bi.pid, bi.is_suspended, bph.phraseIdHistory,
            bi.showsForecast
        FROM phrases p
            JOIN bids bi on bi.cid = p.cid and bi.pid = p.pid
            LEFT JOIN bids_phraseid_history bph ON bi.cid = bph.cid AND bi.id = bph.id",
        WHERE => $where
    ]);
    my %group_phrases = partition_by {$_->{pid}} @$phrases;
    my @cids = uniq map {$_->{cid}} @$phrases;

    # выберем один случайный баннер из группы
    # при отсутсвии статистики на группу в БК, торги будет разыгранны на 
    # этот случайный баннер
    # если мастер цен работает по единой цене - выбираем все фразы, т.к. в торги ходить не нужно
    # иначе нужно выбрать только объявления с заголовком и телом (не графические)
    my $group_sql;
    if ($filters{only_auction_phrases}) {
        $group_sql = 
            ["SELECT
                p.pid, p.geo, p.cid, p.is_bs_rarely_loaded,
                IFNULL(fd.filter_domain, b.domain) filter_domain, mc.publisher_domain_id,
                vc.phone,
                b.title, b.body, b.BannerID,
                count(*) banners_quantity,
                MAX(bi.BannerID) image_BannerID,
                b.banner_type,
                p.adgroup_type, p.statusShowsForecast,
                amc.device_type_targeting, mc.os_type
            FROM phrases p
                JOIN banners b USING(pid)
                LEFT JOIN filter_domain fd ON fd.domain = b.domain
                LEFT JOIN adgroups_mobile_content amc USING(pid)
                LEFT JOIN mobile_content mc USING(mobile_content_id)
                LEFT JOIN vcards vc ON vc.vcard_id = b.vcard_id
                LEFT JOIN banner_images bi ON bi.bid = b.bid",
            WHERE => {
                'p.pid' => SHARD_IDS,
                'b.banner_type__not_in' => [qw/image_ad cpm_banner cpc_video cpm_yndx_frontpage/],
            },
            "GROUP BY p.pid"];
    } else {
        $group_sql =
            ["SELECT t.*, count(*) banners_quantity, MAX(t.BannerID) image_BannerID
            FROM (
                SELECT
                    p.pid, p.geo, p.cid, p.is_bs_rarely_loaded,
                    IFNULL(fd.filter_domain, b.domain) filter_domain, mc.publisher_domain_id,
                    vc.phone,
                    IF(b.banner_type = 'image_ad', NULL, b.title) AS title,
                    IF(b.banner_type = 'image_ad', NULL, b.body) AS body,
                    b.BannerID, b.banner_type,
                    p.adgroup_type, p.statusShowsForecast,
                    amc.device_type_targeting, mc.os_type
                FROM phrases p
                    JOIN banners b USING(pid)
                    LEFT JOIN filter_domain fd ON fd.domain = b.domain
                    LEFT JOIN adgroups_mobile_content amc USING(pid)
                    LEFT JOIN mobile_content mc USING(mobile_content_id)
                    LEFT JOIN vcards vc ON vc.vcard_id = b.vcard_id
                    LEFT JOIN banner_images bi ON bi.bid = b.bid",
                WHERE => {
                    'p.pid' => SHARD_IDS,
                },
                "ORDER BY p.pid, IF(b.banner_type = 'image_ad' or b.banner_type = 'cpc_video', 2, 1)",
            ") AS t",
            "GROUP BY t.pid"];
    }
    my $groups = get_all_sql(PPC(pid => [keys %group_phrases]), $group_sql);

    my $campaigns = get_hashes_hash_sql(PPC(cid => \@cids),
                        ["SELECT
                            c.cid,
                            IFNULL(c.currency, 'YND_FIXED') AS currency,
                            co.fairAuction = 'Yes' AS fairAuction,
                            c.OrderID, c.autobudget,
                            s.strategy_data->>'\$.bid' as autobudget_bid,
                            c.ContextPriceCoef,
                            IF(s.type='no_premium', s.strategy_data->>'\$.place', NULL) as strategy_no_premium,
                            c.timeTarget, c.timezone_id, co.strategy, co.device_targeting,
                            c.sum + IFNULL(wc.sum, 0) AS sum,
                            c.sum_spent + IFNULL(wc.sum_spent, 0) AS sum_spent,
                            FIND_IN_SET('no_extended_geotargeting', c.opts)>0 as no_extended_geotargeting
                        FROM campaigns c
                            JOIN camp_options co USING(cid)
                            LEFT JOIN strategies s on c.strategy_id = s.strategy_id
                            LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid AND wc.uid = c.uid",
                        WHERE => {'c.cid' => SHARD_IDS}]);

    foreach my $camp (values %$campaigns) {
        $camp->{timetarget_coef} = TimeTarget::timetarget_current_coef($camp->{timeTarget}, $camp->{timezone_id});
        $camp->{camp_rest} = round2s($camp->{sum} - $camp->{sum_spent});
    }                        

    my @domain_ids = grep {$_} map {$_->{publisher_domain_id}} @$groups;
    my $domain_dict = !@domain_ids ? {} : get_hash_sql(PPCDICT, [
            'SELECT domain_id, domain
            FROM domains_dict',
            WHERE => {domain_id => \@domain_ids},
        ]);

    foreach my $group (@$groups) {
        hash_merge $group, $campaigns->{$group->{cid}};
        $group->{phrases} = $group_phrases{$group->{pid}};

        my $domain_id = delete $group->{publisher_domain_id};
        $group->{filter_domain} = $domain_dict->{$domain_id}  if $domain_id;
        
        # device_type_targeting / os_type приводим к формату аналогичному Direct::Model::AdGroupMobileContent
        if ($group->{adgroup_type} eq 'mobile_content') {
            $group->{device_type_targeting} = [ split '\s*,\s*', $group->{device_type_targeting} ];
            $group->{mobile_content} = { os_type => delete $group->{os_type} };
        }
    }

    return $groups;
}

1;
