package PlacePrice;

# $Id$

=head1 NAME
    
    PlacePrice

=head1 DESCRIPTION

    Функции про расчет позиций и цен
    В некотором роде -- обертка над BSAuction и AutoBroker;

=cut

use Direct::Modern;

use Readonly;

use Settings;
use List::MoreUtils qw/any/;

use base qw/Exporter/;
our @EXPORT = qw/
    calcPlace
    get_usedto_show_places
    get_data_by_place
    PLACES
    PREMIUM_PLACE_COUNT
    GUARANTEE_PLACE_COUNT
/;

our @EXPORT_OK = qw/
    get_premium_entry_place
    get_guarantee_entry_place
    INTERFACE_PREMIUM_PLACES
    INTERFACE_GUARANTEE_PLACES
    INTERFACE_PLACES
    ALL_PLACES_ORDERED
    get_places_for_js
    get_entry_places_for_js
/;

=head2 PLACES

    Хеш позиций.

=cut

Readonly our %PLACES => (
    # старые позиции входа в СР, первой гарантии и входа в гарантию,
    # могут встречаться в bids.place у старых фраз
    PREMIUM  => 3,
    FIRST    => 2,
    GARANT   => 1,
    # позиции в спецразмещении
    PREMIUM1 => 10,
    PREMIUM2 => 11,
    PREMIUM3 => 12,
    PREMIUM4 => 13,
    # позиции в гарантии
    GUARANTEE1 => 20,
    GUARANTEE2 => 21,
    GUARANTEE3 => 22,
    GUARANTEE4 => 23,

    ROTATION => 0
);

Readonly our %POSITION_CTR_CORRECTION_BY_PLACE => (
    $PLACES{PREMIUM1} => 100,
    $PLACES{PREMIUM2} => 85,
    $PLACES{PREMIUM3} => 73,
    $PLACES{PREMIUM4} => 62,
    $PLACES{GUARANTEE1} => 9,
    $PLACES{GUARANTEE2} => 7,
    $PLACES{GUARANTEE3} => 7,
    $PLACES{GUARANTEE4} => 5,
);

Readonly our %ADVQ_TO_FORECAST_POSITION => (
    P11 => 'first_premium',
    P12 => 'second_premium',
    P13 => 'third_premium',
    P14 => 'fourth_premium',
    P21 => 'first_place',
    P22 => 'second_place',
    P23 => 'third_place',
    P24 => 'fourth_place',
);

=head2 PREMIUM_PLACE_COUNT

    количество позиций в Спецразмещении (всех)
    Столько ставок мы ожидаем получить от торгов

=cut
Readonly our $PREMIUM_PLACE_COUNT => 4;

=head2 GUARANTEE_PLACE_COUNT

    количество позиций в Гарантии (всех)
    Столько ставок мы ожидаем получить от торгов

=cut
Readonly our $GUARANTEE_PLACE_COUNT => 4;

Readonly our %BSAUCTION_PLACE_LAYOUT => (
    $PLACES{PREMIUM1} => {type => 'premium', index => 0},
    $PLACES{PREMIUM2} => {type => 'premium', index => 1},
    $PLACES{PREMIUM3} => {type => 'premium', index => 2},
    $PLACES{PREMIUM4} => {type => 'premium', index => 3},
    $PLACES{GUARANTEE1} => {type => 'guarantee', index => 0},
    $PLACES{GUARANTEE2} => {type => 'guarantee', index => 1},
    $PLACES{GUARANTEE3} => {type => 'guarantee', index => 2},
    $PLACES{GUARANTEE4} => {type => 'guarantee', index => 3},
);


Readonly our @PRESET_TRAFFIC_VOLUME => (5, 15, 75, 100, );
Readonly our %PLACE_BY_POSITION_CTR_CORRECTION = (
    1_000_000 => {
        type  => 'premium',
        index => 0,
    },
    850_000   => {
        type  => 'premium',
        index => 1,
    },
    730_000   => {
        type  => 'premium',
        index => 2,
    },
    620_000   => {
        type  => 'premium',
        index => 3,
    },

    85_000    => {
        type  => 'guarantee',
        index => 0,
    },
    72_000    => {
        type  => 'guarantee',
        index => 1,
    },
    67_000    => {
        type  => 'guarantee',
        index => 2,
    },
    50_000    => {
        type  => 'guarantee',
        index => 3,
    },
);

Readonly our %PREMIUM_POSITION_CTR_CORRECTION = map { my $p = $PLACE_BY_POSITION_CTR_CORRECTION{$_}; ($p->{type} eq 'premium') ? ($p->{index} => $_) : () } keys %PLACE_BY_POSITION_CTR_CORRECTION;
Readonly our %GUARANTEE_POSITION_CTR_CORRECTION = map { my $p = $PLACE_BY_POSITION_CTR_CORRECTION{$_}; ($p->{type} eq 'guarantee') ? ($p->{index} => $_) : () } keys %PLACE_BY_POSITION_CTR_CORRECTION;

=head2 get_premium_entry_place

    позиция входа в Спецразмещение

=cut
sub get_premium_entry_place {
    my (%O) = @_;
    return ($O{forecast_style} ? 'fourth_premium' : $PLACES{PREMIUM4});
}

=head2 get_guarantee_entry_place

    позиция входа в Гарантию

=cut
sub get_guarantee_entry_place {
    my (%O) = @_;
    return ($O{forecast_style} ? 'fourth_place' : $PLACES{GUARANTEE4});
}

=head2 get_entry_places_for_js

    Хешик с позициями входа в спецразмещение/гарантию для шаблонов

=cut

sub get_entry_places_for_js {
    return {
        PREMIUM => get_premium_entry_place(),
        GUARANTEE => get_guarantee_entry_place(),
    }
}

=head2 INTERFACE_PREMIUM_PLACES

    позиции Спецразмещения, которые показываются в интерфейсе. Отсортированы по убыванию

=cut
Readonly our @INTERFACE_PREMIUM_PLACES => (
    $PLACES{PREMIUM1},
    $PLACES{PREMIUM2},
    get_premium_entry_place(),
);

=head2 INTERFACE_GUARANTEE_PLACES

    позиции Гарантии, которые показываются в интерфейсе. Отсортированы по убыванию

=cut
Readonly our @INTERFACE_GUARANTEE_PLACES => (
    $PLACES{GUARANTEE1},
    get_guarantee_entry_place(),
);

=head2 INTERFACE_PLACES

    позиции, которые показываются в интерфейсе. Отсортированы по убыванию

=cut
Readonly our @INTERFACE_PLACES => (
    @INTERFACE_PREMIUM_PLACES, @INTERFACE_GUARANTEE_PLACES
);

Readonly our @ALL_PREMIUM_PLACES_ORDERED => (
    $PLACES{PREMIUM4},
    $PLACES{PREMIUM3},
    $PLACES{PREMIUM2},
    $PLACES{PREMIUM1},
);

Readonly our @ALL_GUARANTEE_PLACES_ORDERED => (
    $PLACES{GUARANTEE4},
    $PLACES{GUARANTEE3},
    $PLACES{GUARANTEE2},
    $PLACES{GUARANTEE1},
);

Readonly our @ALL_PLACES_ORDERED => (
    @ALL_GUARANTEE_PLACES_ORDERED, @ALL_PREMIUM_PLACES_ORDERED
);

=head2 calcPlace

    По ставке и предложенным ценам подбирает позицию.
    На входе:
        price - ставка
        guarantee - цены для гарантированных показов
        premium - цены для спецразмещения
    на выходе:
        число - позиция, одно из значений %PLACES

=cut

sub calcPlace {
    my ($price, $guarantee, $premium) = @_;

    return $PLACES{ROTATION} if !$price || !$guarantee || !$premium;

    my @p_prices = map { $_->{bid_price}/1e6 } @$premium;
    my @g_prices = map { $_->{bid_price}/1e6 } @$guarantee;
    if ( @p_prices > 0 && $price >= $p_prices[0] ) {
        # 1 Спец-размещение
        return $PLACES{PREMIUM1};
    } elsif ( @p_prices > 1 && $price >= $p_prices[1]) {
        # 2 Спец-размещение
        return $PLACES{PREMIUM2};
    } elsif ( @p_prices > 2 && $price >= $p_prices[2] ) {
        # 3 Спец-размещение
        return $PLACES{PREMIUM3};
    } elsif ( @p_prices > 3 && $price >= $p_prices[3] ) {
        # 4 Спец-размещение
        return $PLACES{PREMIUM4};
    } elsif ( @g_prices > 0 && $price >= $g_prices[0] ) {
        # 1 Гарантированные показы
        return $PLACES{GUARANTEE1};
    } elsif ( @g_prices > 3 && $price >= $g_prices[3] ) {
        # 4 Гарантированные показы
        return $PLACES{GUARANTEE4};
    }

    return $PLACES{ROTATION};
}

=head2 set_new_place_style

    На период перехода на новые позиции (DIRECT-41669), преобразовываем старое место в современное.

=cut
sub set_new_place_style {
    my $place = shift;
    return unless defined $place;
    state $map_old_places = {
        $PLACES{PREMIUM} => get_premium_entry_place(),
        $PLACES{FIRST}   => $PLACES{GUARANTEE1},
        $PLACES{GARANT}  => get_guarantee_entry_place(),
        # Позиции ниже не поддерживаем
        $PLACES{GUARANTEE2} => get_guarantee_entry_place(),
        $PLACES{GUARANTEE3} => get_guarantee_entry_place(),
        };

    $place = $map_old_places->{$place} if $map_old_places->{$place};
    return $place;
}

=head2 get_data_by_place

    Взять данные по позиции по номеру.

=cut
sub get_data_by_place {
    my ($row, $place) = @_;
    state $place_rules = {
        $PLACES{PREMIUM1} => {type => 'premium', index => 0},
        $PLACES{PREMIUM2} => {type => 'premium', index => 1},
        $PLACES{PREMIUM3} => {type => 'premium', index => 2},
        $PLACES{PREMIUM4} => {type => 'premium', index => 3},
        $PLACES{GUARANTEE1} => {type => 'guarantee', index => 0},
        $PLACES{GUARANTEE4} => {type => 'guarantee', index => 3},
    };
    $place = set_new_place_style($place);
    my $rule = $place_rules->{$place};
    unless ($rule) {
        return undef;
    }
    return $row->{$rule->{type}}->[$rule->{index}];
}

=head2 get_bid_price_by_place($row, $place)

    Взять bid_place на позиции $place из строки аукциона $row. $place должна быть одним из значений %PLACES

=cut
sub get_bid_price_by_place {
    my ($row, $place) = @_;

    my $data = get_data_by_place($row, $place);
    return undef unless defined $data;
    return $data->{bid_price};
}

=head2 get_usedto_show_places

    Список позиций, который показываются в интерфейсе.
    PREMIUM1, PREMIUM2, вход в СР, GUARANTEE1, вход в Гарантию - обычно
    GUARANTEE1 - если передынный аргумент истинен (для mcbanner кампаний)

=cut
sub get_usedto_show_places {
    my $only_first_guarantee = shift;
    if ($only_first_guarantee) {
        return [ $PLACES{GUARANTEE1} ];
    } else {
        return \@INTERFACE_PLACES;
    };
}

=head2 get_warnplace_value
    
    При отправке уведомлений используем только вытеснения с 2х позиций: вход в спецразмещения и гарантию
    Поэтому, все позиции СР сводим ко входу в СР (PREMIUM4), 
    все позиции гарантии сводим ко входу в гарантию (GUARANTEE4)

=cut
sub get_warnplace_value {
    my $place = shift;
    $place = PlacePrice::get_premium_entry_place() if any {$place == $_} @ALL_PREMIUM_PLACES_ORDERED;
    $place = PlacePrice::get_guarantee_entry_place() if any {$place == $_} @ALL_GUARANTEE_PLACES_ORDERED;
    return $place;
}

1;
