package Plack::Middleware::IntapiAutoLogger;
use strict;
use warnings;

=head1 NAME

    Plack::Middleware::IntapiAutoLogger

=head1 DESCRIPTION

    Логгирует запросы в Intapi в syslog. Этот модуль логгирует запросы к не-jsonrpc-ручкам, для
    jsonrpc-ручек есть JsonrpcLogger.

=cut

use parent 'Plack::Middleware';

use JSON;

use Plack::UTF8Request;

use Intapi;

my $MAX_CONTENT_LENGTH = 10240;

=head1 SUBROUTINES/METHODS

=head2 call

=cut

sub call {
    my ( $self, $env ) = @_;

    my $cmd = $env->{cmd};
    my $cmd_spec = $cmd ? $Intapi::cmd{$cmd} : undef;

    return $self->app->($env) if $cmd_spec && $cmd_spec->{custom_log};

    my $log = Intapi::request_logger($cmd);

    my $req = Plack::UTF8Request->new($env);
    my $req_dump = {
        method => $req->method,
        query_params => $req->query_parameters->as_hashref,
        content_type => $req->content_type,
        content_length => $req->content_length,
        content => _dump_content($req),
        reqid => $env->{reqid},
    };
    my $trace_parent_id = $env->{trace} ? $env->{trace}->parent_id() : undef;
    if ($trace_parent_id) {
        $req_dump->{parent_id} = $trace_parent_id;
    }

    $log->out($req_dump);

    return $self->app->($env);
}

=head2 _dump_content($req)

    Приводит содержимое ответа в вид, который потом можно задампить в JSON и записать в лог.
    Действует в зависимости от типа содержимого: содержимое формы или JSON распаковывается и возвращается;
    содержимое в неизвестных форматах при необходимости обрезается до $MAX_CONTENT_LENGTH и возвращается
    строчкой.

=cut

sub _dump_content {
    my ($req) = @_;

    return undef unless defined $req->content;

    my $type = $req->content_type || 'application/octet-stream';

    if ( $type eq 'application/x-www-form-urlencoded' || $type eq 'multipart/form-data' ) {
        return $req->body_parameters->as_hashref;
    }

    if ( $type eq 'application/json' ) {
        my $decoded_content = eval { decode_json($req->content) };
        return $decoded_content if $decoded_content;
    }

    if ( length( $req->content ) < $MAX_CONTENT_LENGTH ) {
        return $req->content;
    }

    return substr( $req->content, 0, $MAX_CONTENT_LENGTH ) . ' (truncated)';
}

1;
