package Pokazometer;

#   $Id$
#

=head1

    Работа с сервером показометра - получение данных об охвате аудитории в РСЯ и/или на поиске в зависимости от цены клика

=cut

use strict;
use warnings;

use Settings;
use Yandex::Interpolate;
use Yandex::ListUtils;
use Tools;
use EnvTools;
use TextTools;
use Yandex::Trace;
use Currency::Rate;
use Currencies;

use Storable qw/dclone/;
use List::Util qw/min sum/;

use Pokazometer::RPC qw/ get_pokazometer_call_result get_pokazometer_call_results_parallel /;

use base 'Exporter';

our @EXPORT = qw(
    safe_pokazometer
    get_pokazometer_data
    get_price_for_coverage
    get_coverage_for_price
);

use utf8;

# показы вычислять слишком дорого, просто умножаем клики на коэффициент
# сам коэфиициент ни на что влиять не должен, по историческим причинам оставлен
# 7 - (показы отдаются за сутки, нормируем к неделе)
our $SHOWS_COEF = 7;

our $POKAZOMETER_CURRENCY = 'YND_FIXED';

=head2 get_pokazometer_data

    Для вызова из Common|DoCmd лучше использовать safe_pokazometer

    Получить данные показометра для массива фраз. 
   
    get_pokazometer_data (\@groups)


    Входные данные
    ссылка на массив групп

    groups: [
        {
            phrases => [
                {
                    phrase: => "окна",
                    rank    => 0,
                },
                {
                    phrase: => "холодильник",
                    rank    => 11234,
                },
            ]
            geo     => "225, -3",
            currency => 'RUB',
        },
    ]
    
    rank = 0        => Идём в показометр если фраза отключена на поиске
    rank = undef   : считаем что указанные поля не заполнены, значения по умолчанию
    def_rank = 0

    поле rank добавляется при обращении к торгам. Если предварительно не спросить у БК цены (и ранк), то данные показометра добавятся ко всем фразам.


    функция добавляет данные от показометра к фразам, у которых !defined rank || rank == 0

    Параметры именованные
        get_all_phrases -- флаг, "игнорировать остановленность на Поиске, обработать все фразы"
        period
        net -- 'both' (по умолчанию), 'search', 'context' (получать данные суммарные/по поисковым площадкам/по контекстным)
        pokazometer_field -- имя поля, в которое будут записаны данные от Показометра (умолчание -- 'pokazometer_data'). 



    В pokazometer_data есть списки: 
        shows_list   -- массив элементов { cost => ..., cnt => ...}, cnt -- кол-во показов
        clicks_list  -- массив элементов { cost => ..., cnt => ...}, cnt -- кол-во кликов
        complete_list -- массив элементов { cost => ..., shows => ..., clicks => ... }
        Важно: "ступеньки" цен в этих списках (cost) могут, но не обязаны совпадать ни по значениям, ни даже по количеству

    фраза с данными показометра выглядит так:
    {
            phrase: => "магазин москве",
            rank    => 0,
            pokazometer_data    => {
                'lookup_time' => '0.031711000000000003',
                'phrases_cnt' => '7004',
                'query' => "магазин москве",
                'shows_cnt' => 648002,
                'shows_list' => [
                  {
                    'cnt' => 131022,
                    'cost' => 10000
                  },
                  {
                    'cnt' => 296740,
                    'cost' => 40000
                  },
                  {
                    'cnt' => 533507,
                    'cost' => 210000
                  },
                  {
                    'cnt' => 647180,
                    'cost' => 2290000
                  },
                  {
                    'cnt' => 648002,
                    'cost' => 6350000
                  }
                ],
                clicks_list => [
                  {
                    'cost' => 10000,
                    'cicks' => 1010,
                  },
                  {
                    'cost' => 20000,
                    'clicks' => 4750,
                  },
                  {
                    'cost' => 110000,
                    'clicks' => 18911,
                  },
                ],
                complete_list =>[
                  {
                    'cost' => 10000,
                    'shows' => 131022,
                    'cicks' => 1010,
                  },
                  {
                    'cost' => 50000,
                    'shows' => 315580,
                    'clicks' => 6950,
                  },
                  {
                    'cost' => 280000,
                    'shows' => 586793,
                    'clicks' => 30871,
                  },
                ],
                'total_time' => '0.032039165496826172'
            }
    }

    Фукнкцию можно вызывать и отлаживать из командной строки: 
    perl -Iprotected -ME -MPokazometer -Mutf8 -le '$groups = [ {phrases=>[{phrase=>"установка кондиционера"}],currency=>"RUB"} ]; get_pokazometer_data($groups); p $groups;'

=cut

sub get_pokazometer_data($;%) {
    my ($groups, %OPT) = @_;

    my $pokazometer_field = $OPT{pokazometer_field} || 'pokazometer_data';
    my %requests;
    my %phrase_keys_by_queries;
    for my $i (0..$#{$groups}) {
        my $group = $groups->[$i];
        die 'No currency given' unless $group->{currency};

        my ($phrases, $geo) = (
            $group->{phrases} || $group->{Phrases},
            $group->{geo}
        );
        
        my $phrase_keys_by_query = {};
        for my $j  (0..$#{$phrases}) {
            my $phrase_hash = $phrases->[$j];

            # если не указано, что нужны все фразы -- оставляем только отключенные на поиске
            if ( ! $OPT{get_all_phrases} && ! $group->{get_all_phrases} && ( $phrase_hash->{rank} || $phrase_hash->{context_stop_flag} ) ) {
                next;
            }
            
            (my $q = $phrase_hash->{phrase}) =~ s/\[|\]//g;
            unless ($OPT{consider_minus_words} || $group->{consider_minus_words}) {
                my @words = grep {! m/^-/} split /\s+/, $q;
                if (@words > $Settings::MAX_WORDS_IN_KEYPHRASE) {
                    @words = @words[0..$Settings::MAX_WORDS_IN_KEYPHRASE-1];
                }
                $q = join " ", @words;
                unless ($q) {
                    die "Empty phrase after clear minus words in pokazometer for phrase: $phrase_hash->{phrase}\n";
                }
            }
            push @{$phrase_keys_by_query->{$q}}, $j;
        }

        next unless scalar keys %$phrase_keys_by_query;
        $requests{$i} = {
            phrases => [ keys %$phrase_keys_by_query ],
            regions => [ split /[^-\d]/, ( $geo || '')  ],
            currency => $group->{currency},
        };
        $phrase_keys_by_queries{$i} = $phrase_keys_by_query;
    }
    my $data_groups = _request_pokazometer_data_parallel( \%requests );
    
    for my $i (0..$#{$groups}) {
        next unless $requests{$i};
        my $group = $groups->[$i];
        my $phrases = $group->{phrases} || $group->{Phrases};
        my $data = $data_groups->{$i};
        my $phrase_keys_by_query = $phrase_keys_by_queries{$i};
        for my $item (@$data) {
            _resample_lists_for_item($item);
            
            unless( exists $phrase_keys_by_query->{$item->{query}} ) {
                die "Unknown query in the pokazometer response: '$item->{query}'\n";
            }
            my $keys = $phrase_keys_by_query->{$item->{query}};
            for my $key (@$keys) {
                $phrases->[$key]->{$pokazometer_field} = dclone($item);
            }
            delete $phrase_keys_by_query->{$item->{query}};
        }
        if(scalar keys %$phrase_keys_by_query) {
            die "No pokazometer response for phrases: ". join(", ", keys %$phrase_keys_by_query)."\n";
        }
    }

}

sub _request_pokazometer_data_parallel {
    my ( $requests ) = @_;

    my $total_phrases_count = sum ( map { scalar @{ $_->{phrases} } } values %$requests );
    my $profile = Yandex::Trace::new_profile('pokazometer:_request_pokazometer_data_parallel', obj_num => $total_phrases_count);

    # Сразу перед вызовом get_pokazometer_call_results_parallel форсируем числовой контекст для параметров, которые хотим передать как числа, иначе они будут закодированы в json как строки
    Tools::force_number_recursive($_->{regions}) for values %$requests; 
    my $res_clicks_by_group =  get_pokazometer_call_results_parallel('ClicksDistribution', $requests);

    # Показометр пока не может подсчитывать распределение показов(не справляется с нагрузкой), поэтому заглушка - копируем клики
    my $res_shows_by_group = { %$res_clicks_by_group };
#    $res_shows_by_group = get_pokazometer_call_results_parallel('ShowsDistribution', $requests);
    
    my %data_groups;
    while (my ($id, $request) = each %$requests) {
        my $res_shows = $res_shows_by_group->{$id}->{distribution};
        my $res_clicks = $res_clicks_by_group->{$id}->{distribution};
        
        my $currency = $request->{currency};
        die "No currency given" unless $currency;

        my $phrases = $request->{phrases};
        my $phrases_count = scalar @$phrases;

        my $res_clicks_count = scalar @$res_clicks;
        die "Wrong Pokazometer answer length for clicks: $res_clicks_count, $phrases_count" unless $res_clicks_count == $phrases_count;

        my $res_shows_count = scalar @$res_shows;
        die "Wrong Pokazometer answer length for shows: $res_shows_count, $phrases_count"   unless $res_shows_count  == $phrases_count;
        
        my @data;
        for my $i (0..$#{$phrases}) {
            my $ph = $phrases->[$i];
            my $item = {
                query => $ph,
            };

            my $clicks = $res_clicks->[$i];
            my $shows = $res_shows->[$i];

            $item->{clicks_list} = [ map { {cost => convert_currency($_->{cost}, $POKAZOMETER_CURRENCY, $currency), cnt => $_->{count}} }     @$clicks ]; 
            $item->{shows_list} =  [ map { {cost => convert_currency($_->{cost}, $POKAZOMETER_CURRENCY, $currency), cnt => $_->{count} * $SHOWS_COEF} } @$shows  ]; # Показы отдаются за сутки, нормируем к неделе
            $item->{shows_cnt} = @{$item->{shows_list}} > 0 ? $item->{shows_list}->[-1]->{cnt} : 0;
            #$item->{clicks_cnt} = $item->{clicks_list}->{-1}->{cnt};
            $item->{currency} = $currency;
            push @data, $item;
        }
        $data_groups{$id} = \@data;
    }
    return \%data_groups;
}

=head2 _rough_resample(n, arr)

    грубое прореживание массива до n элементов,
    для более точных результатов нужно использовать Yandex::Interpolate::resample

=cut
sub _rough_resample {
    my ($n, $arr) = @_;
    return $arr if @$arr <= $n;
    return $arr->[0] if $n == 1;
    my $ret = [map {$arr->[int($#$arr * $_ / ($n - 1))]} 0 .. $n-1];
    return $ret;
}


=head2 _resample_lists_for_item

    Улучшает списки кликов и показов, полученные от Показометра: 
    * уменьшает количество опорных точек (resample)
    * составляет сводный список, в котором на каждую опорную точку указаны и клики, и показы (complete_list)

=cut 

sub _resample_lists_for_item
{
    my ($item) = @_;

    $item->{clicks_list} = _rough_resample(5,
        [ sort {$a->{cost} <=> $b->{cost}} xuniq {$_->{cost}}  @{$item->{clicks_list}} ], 
    ) if ref $item->{clicks_list} eq 'ARRAY';

    # поскольку сейчас shows_list = clicks_list, копируем всё грязными хаками
    $item->{shows_list} = [map { {cost => $_->{cost}, cnt => $SHOWS_COEF * $_->{cnt}} } @{$item->{clicks_list}}];
    $item->{complete_list} = [map { +{cost => $_->{cost}, shows => $SHOWS_COEF * $_->{cnt}, clicks => $_->{cnt}} } @{$item->{clicks_list}}];

=pod
    $item->{shows_list} = _rough_resample(5,
        [ sort {$a->{cost} <=> $b->{cost}} xuniq {$_->{cost}}  @{$item->{shows_list}} ], 
    ) if ref $item->{shows_list} eq 'ARRAY';
    
    $item->{complete_list} = [];
    if (ref $item->{clicks_list} eq 'ARRAY' && @{$item->{clicks_list}} > 0) {
        my @clicks_list_for_interpolation = map { {y => $_->{cnt}, x => $_->{cost}} } @{$item->{clicks_list}};
        for my $shows_point ( @{$item->{shows_list}} ){
            my $complete_point = {
                cost => $shows_point->{cost},
                shows => $shows_point->{cnt},
            };
            $complete_point->{clicks} = int(interpolate_linear($shows_point->{cost}, @clicks_list_for_interpolation) + 0.5);
            # из-за погрешностей статистики и интерполяции у начальных точек может оказаться кликов больше, чем показов
            # пока для таких точки просто сбрасываем клики
            $complete_point->{clicks} = 0 if $complete_point->{clicks} > $complete_point->{shows};
            push @{$item->{complete_list}}, $complete_point;
        }
    }
=cut

    return;
}

=head2 get_price_for_coverage

    Найти цену, при которой достигается заданное покрытие
    Покрытие задаётся в долях, а не процентах

    $price = get_price_for_coverage($pokazometer_data, $scope);

=cut

sub get_price_for_coverage {
    my ($data, $scope) = @_;

    return undef unless scalar @{$data->{shows_list}};
    my $cnt = $data->{shows_cnt}*$scope;
    my $price = interpolate_linear($cnt, map { {'x' => $_->{cnt}, 'y' => $_->{cost}} } @{$data->{shows_list}}) / 1e6;
    $price = round_price_to_currency_step($price, $data->{currency}, up => 1);
    return $price;
}

=head2 get_coverage_for_price

    Найти покрытие при заданной цене клика
    возвращает результат в долях, чтобы получить проценты надо домножить на 100

    $coverage = get_coverage_for_price($pokazometer_data, $price);

=cut

sub get_coverage_for_price {
    my ($data, $price) = @_;

    return undef unless scalar @{$data->{shows_list}}; 

    $price = $price * 1e6;
    # если все показы были с ценой выше нашей, то покрытие 0%
    if ($price < min(map {$_->{cost}} @{$data->{shows_list}})) {
        return 0;
    }

    return interpolate_linear($price, map { {'y' => $_->{cnt}, 'x' => $_->{cost}} } @{$data->{shows_list}})/ ($data->{shows_cnt}||1);
}


=head3 safe_pokazometer

    Безопасная обёртка для get_pokazometer_data($banners)

    Дополнительно расчитывает context_coverage

=cut

sub safe_pokazometer($;%) {
    my ($groups, %options) = @_;

    # Получаем данные от показометра
    eval {
        get_pokazometer_data($groups, %options);
        for my $group (@$groups) {
            
            foreach my $ph (@{$group->{phrases}}) {
                unless ($ph->{pokazometer_data}) {
                    $ph->{no_pokazometer_stat} = 1;
                    next;
                }
                my $coverage = get_coverage_for_price($ph->{pokazometer_data}, $ph->{price_context});
                $ph->{context_coverage} = sprintf "%d", 100*$coverage if defined $coverage;
                # цена контекста в зависимости от охвата аудитории
                $ph->{price_for_coverage} = {map {
                    $_ => sprintf '%0.02f', round2s(get_price_for_coverage($ph->{pokazometer_data}, $_ / 100) || 0)
                } 100, 50, 20};
            }
        }
    };

    if ($@) {
        # Если это был таимаут - снова используем die, чтобы дойти до того, кто его установил
        die $@ if $@ =~ /^\Q$Settings::TIMEOUT_ALERT_MESSAGE\E/;
        warn "FAILED to get pokazometer data: $@";
        for my $group (@$groups) {
            $group->{pokazometer_failed} = 1;
        }
    }

    return $groups;
}

# --------------------------------------------------------------------

=head2 _request_pokazometer_data_totalhitsbycost

Сколько было показов по определенной цене в РСЯ по региону

=cut

sub _request_pokazometer_data_totalhitsbycost($$) {
    my ($cost, $geo) = @_;

    $geo ||= '';
    my $profile = Yandex::Trace::new_profile('pokazometer:_request_pokazometer_data_totalhitsbycost');

    my $params = {
        cost => $cost * 1_000_000,
        regions => [ split(/[^-\d]/, $geo) ],
    };
    
    Tools::force_number_recursive($params->{regions});   # эта строка должна располагаться непосредственно перед вызовом get_pokazometer_call_result, чтобы гарантировать, что параметры правильно закодируются
    my $result = get_pokazometer_call_result('TotalHitsByCost', $params);

    return $result->{shows};
}

# --------------------------------------------------------------------

=head2 get_total_hits_by_cost_context

Сколько было показов по определенной цене в РСЯ по региону

    $total_hits_by_cost = get_total_hits_by_cost_context(0.33, '225');

=cut

{
    # кешируем на час значения, TotalHitsByCost пересчитываются раз в сутки, значений должно быть не много (все сочетания цен/регионов)
    my $pokazometer_cache = {};
sub get_total_hits_by_cost_context($$) {
    my ($cost, $geo) = @_;

    my $key = "$cost/$geo";

    if (exists $pokazometer_cache->{$key}) {
        if (time() - $pokazometer_cache->{$key}->{time} < 3600) {
            return $pokazometer_cache->{$key}->{res};
        } else {
            delete $pokazometer_cache->{$key};
        }
    }

    my $total_hits_by_cost = eval { _request_pokazometer_data_totalhitsbycost($cost, $geo) } || 0;
    unless ($@) {
        $pokazometer_cache->{$key} = {
            "res" => $total_hits_by_cost,
            "time" => time(),
        };
    } elsif (is_beta()) {
        warn $@;
    }

    return $total_hits_by_cost;
}}

# --------------------------------------------------------------------

=head2 get_hits_by_cost_context_table

Возвращаем разкладку кликов, в зависимости от цены (по региону)

  my $table = get_hits_by_cost_context_table('225');
  $table: [
    {price => 0.01, shows => 123323},
    {price => 0.02, shows => 133323},
    {price => 0.10, shows => 233323},
    {price => 1.00, shows => 333323},
    {price => 2.00, shows => 333323},
    {price => 10.00, shows => 1333323},
    ...
    {price => 50.00, shows => 7333323}
  ]

  тестирование:
  PERL5LIB=./yandex-lib/pokazometer/lib:./protected perl -ME -MPokazometer -e 'p Pokazometer::get_hits_by_cost_context_table("225")'

=cut

{
    my @predefined_prices = (
        {from => 0.01,  to => 0.09,  step => 0.01},
        {from => 0.10,  to => 0.90,  step => 0.01},
        {from => 1.00,  to => 9.00,  step => 0.10},
        {from => 10.00, to => 50.00, step => 10.00},
    );
sub get_hits_by_cost_context_table($) {
    my $geo = shift;

    my @result_table;
    my ($prev_shows, $prev_price);

    ALL_PRICES_LOOP:
    for my $prices (@predefined_prices) {
        for (my $price = $prices->{from}; $price <= $prices->{to}; $price += $prices->{step}) {
            $price = round2s($price);
            my $shows = get_total_hits_by_cost_context($price, $geo);

            # клики перестали увеличиваться
            last ALL_PRICES_LOOP if defined $prev_shows && $shows > 0 && $shows == $prev_shows;

            # новый прирост кликов слишком дорог (см. prepare_transitions_for_advanced_forecast())
            last ALL_PRICES_LOOP if defined $prev_shows
                                    && $shows > $prev_shows
                                    && ($shows * $price - $prev_shows * $prev_price) / ($shows - $prev_shows) > 20;

            push @result_table, {price => $price, shows => $shows};

            $prev_shows = $shows;
            $prev_price = $price;
        }
    }

    return \@result_table;
}}

1;

