package Pokazometer::RPC;

=head1

    Интерфейс для вызова методов на сервере показометра через RPC
    В этой реализации используется протокол JSON-RPC.

=cut

use strict;
use warnings;

use Yandex::HTTP qw( http_parallel_request );
use JSON::RPC2::Client;
use Settings;

use base 'Exporter';

our @EXPORT_OK = qw(
    get_pokazometer_call_result
    get_pokazometer_call_results_parallel
);

use utf8;

use feature 'state';

# старый Показометр
#our $POKAZOMETER_URL ||= 'http://ipv6.whale-http.yandex.net:80';
# новый Показометр-RTC
our $POKAZOMETER_URL ||= 'http://pokazometer.yandex.ru:80';
our $POKAZOMETER_TIMEOUT ||= 16;
our $POKAZOMETER_PARALLEL_LEVEL ||= 4;

=head2 get_pokazometer_call_result

    Вызывает метод API показометра.

    get_pokazometer_call_result ($pokazometer_method, $params)


    Входные данные:
        $pokazometer_method -- имя метода (например 'ClicksDistribution')
        $params     -- ссылка на хэш с параметрами метода.
            

     B<ВАЖНО:> параметры, которые имеют численный тип, должны быть использованы в численном контексте перед вызовом этой функции, иначе они будут перекодированы в json как строки.

     Возвращаемое значение:
        ссылка на хэш, содержащий структуру, возвращённую методом. Например,

     Пример вызова:

        my $params = {
                        phrases => [
                                        'купить крокодила',
                                        'нильский крокодил'
                                   ],
                        regions => [
                                        1, 2
                                   ],
                     };
        my $result = get_pokazometer_call_result('ClicksDistribution', $params);

        $result:
            {    
                distribution => [
                    {
                        cost => 42,
                        cnt => 111
                    },
                    {
                        cost => 24,
                        cnt => 11
                    }
                ],
                ref_days => 7
            }
    
    Однострочник, который можно использовать для отладки:
    perl -Iprotected -ME -MPokazometer::RPC -Mutf8 -le 'p Pokazometer::RPC::get_pokazometer_call_result("ClicksDistribution", {phrases => [ "установка кондиционера" ], regions => [ 1 ]});'

=cut

sub get_pokazometer_call_result($;$) {
    my ($pokazometer_method, $params) = @_;

    return get_pokazometer_call_results_parallel($pokazometer_method, {1 => $params})->{1};
}

=head2 get_pokazometer_call_results_parallel

    Вызывает метод API показометра для нескольких групп параметров, отправляя параллельные HTTP-запросы.

    get_pokazometer_call_results_parallel($pokazometer_method, $requests)


    Входные данные:

        $pokazometer_method -- имя метода (аналогично get_pokazometer_call_result)
        $requests -- ссылка на хэш значениями которого являются параметры метода.

    Выходные данные:
        
        Ссылка на хэш, значениями которого являются результаты вызова метода для каждой группы параметров в соответствии с ключами %$requests.
    
    Однострочник для отладки:
    perl -Iprotected -ME -MPokazometer::RPC -Mutf8 -le 'p Pokazometer::RPC::get_pokazometer_call_results_parallel("ClicksDistribution",{"1"=>{phrases=>["установка кондиционера"],regions=>[1]},"2"=>{phrases=>["купить кондиционер"],regions=>[1]}});'
=cut


sub get_pokazometer_call_results_parallel {
    my ($pokazometer_method, $requests) = @_;

    state $client = JSON::RPC2::Client->new();

    my %tasks;

    while (my ($id, $request) = each %$requests) {
        my $json_string = $client->call_named($pokazometer_method, %$request);
        $tasks{$id} = { body => $json_string, url => $POKAZOMETER_URL };
    }

    my $pokazometer_responses = http_parallel_request( 
        POST => \%tasks, 
        timeout => $POKAZOMETER_TIMEOUT,
        max_req => $POKAZOMETER_PARALLEL_LEVEL,
        num_attempts => 2,
        soft_timeout => $POKAZOMETER_TIMEOUT / 4,
        connect_timeout => 0.3,
    );

    my %call_results;
    while (my ($id, $res) = each %$pokazometer_responses) {
        my $json_string = $res->{content};
        my ($failed, $result, $error, $call) = $client->response($json_string);

        if (defined($failed)) {
            die "Remote method call failed: $failed"
                . "\nStatus line: $res->{headers}->{Status} $res->{headers}->{Reason}"
                . "\nContent: ". substr($res->{content}, 0, 1024)
                ;
        }
        die "Remote method call error: $error->{message}" if defined($error);
        $call_results{$id} = $result;
    }

    return \%call_results;
}

1;
