package Primitives;

# $Id$

=head1 NAME
    
    Primitives

=head1 DESCRIPTION

    Модуль для простых низкоуровневых функций, для Директовских "примитивов".
    В основном не для изменения данных, а для удобного доступа к данным из БД/смежных сервисов

    Здешние функции не делают ничего особнно умного (но делают удобно), 
    используются повсюду, и ради них не хочется и не надо загружать Common или другие высокоуровневые модули.

=cut

use strict;
use warnings;
use feature 'state';

use Carp qw/croak/;
use Encode;
use List::Util qw/sum max/;
use List::MoreUtils qw/uniq any none/;
use Params::Validate qw/:all/;
use JSON;
use YAML;

use Yandex::ListUtils qw/xuniq xminus xflatten chunks/;
use Settings;
use Yandex::Blackbox;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;

use TextTools;
use PrimitivesIds;
use Campaign::Types;
use PlacePrice;
use Moderate::ReModeration;

use base qw/Exporter/;
our @EXPORT = qw/

    detect_strategy
    detect_search_strategy
    detect_context_strategy
    detect_autobudget_strategy
    get_attribution_model_or_default_by_type
    get_attribution_model_default
    get_attribution_model_internal_stat_default
    get_attribution_model_for_stat_request
    get_client_campaigns_attribution_type_or_default
    get_common_stat_default_attribution

    get_fio_by_uid
    get_user_info
    get_users_list_info
    get_lim_rep_info
    get_lim_rep_users_list_info
    get_client_reps_list_info
    get_uid_by_login2
    get_wallet_camp
    get_all_wallet_camps

    get_uid_by_login
    get_login_by_uid_passport
    get_info_by_uid_passport
    check_direct_sid

    get_agency_client_relations
    get_agency_clients_relations_data
    get_allow_agency_bind_client
    mass_get_allow_create_scamp_by_subclient
    get_allow_create_scamp_by_subclient

    get_manager_email

    get_domain_for_stat_by_bid
    get_domains_for_stat_by_bids

    product_info
    validate_priority
    get_campaign_warnplace
    is_package_mcb
    is_regional_mcb
    is_turkish_mcb

    clear_banners_moderate_flags
    reverse_domain
    schedule_forecast
    schedule_forecast_multi

    get_auto_servicing_manager_uid
    get_other_manager_uid
    get_idm_primary_manager_uid

    query_special_user_option
    mass_query_special_user_option
    set_special_user_option
    query_all_api_special_options

    get_wallets_by_cid

    is_first_camp_under_wallet

    filter_archived_campaigns

    get_domain2domain_id

    get_camp_options
    set_camp_options
    update_camp_options

    content_lang_to_view
    content_lang_to_save

    get_freelancer_info
    get_adgroup_type

    get_bannerid2bid

    get_internal_product_names_by_client_ids
/;

our @EXPORT_OK = qw/
    get_main_banner_ids_by_pids
    get_goal_type_by_goal_id
/;

use utf8; 


=head2 detect_strategy

    определить используемую стратегию

    my $scalar_text = detect_strategy($vals);

        default                => Наивысшая доступная позиция
        autobudget             => Недельный бюджет
        autobudget_avg_click   => Средняя цена клика (за неделю)
        autobudget_avg_cpa     => Средняя цена конверсии (за неделю)
        autobudget_avg_cpi     => Средняя цена установки
        autobudget_week_bundle => Недельный пакет кликов
        autobudget_roi         => Рентабельность рекламы
        different_places       => Независимое управление для разных типов площадок 
        strategy_no_premium    => Показ под результатами поиска

    $vals - хеш кампании

    options:
        without_different_places - не учитывать "Отдельное размещение в сети"
        api - отдавать токены стратегий для API (cpa_optimizer, strategy_no_premium_highest)

=cut

sub detect_strategy {
    my ($vals, %options) = @_;

    my $type = $vals->{type} || $vals->{mediaType} || 'text';
    return undef if $type eq 'wallet' || $type eq 'billing_aggregate';

    if ($vals->{strategy} && $vals->{strategy} ne 'different_places') {
        return $vals->{strategy};
    }


    my $is_different_places = $vals->{strategy} && $vals->{strategy} eq 'different_places';
    if ($is_different_places) {
        return 'different_places' if !$options{without_different_places};
        return 'stop'  if $options{api} && $vals->{platform} eq 'context';
    }

    my $is_autobudget = ($vals->{autobudget} || '') =~ /^(Yes|1|True)$/i;
    if ($is_autobudget) {
        return 'autobudget_media'  if $type eq 'mcb';
        return detect_autobudget_strategy($vals);
    }

    # TODO DIRECT-67186 после миграции, которая сделает min_price в базе невозможным, регэкс надо поменять на eq 'highest_place'
    my $strategy_data = $vals->{strategy_decoded} || from_json $vals->{strategy_data};
    if ($strategy_data->{name} eq 'no_premium') {
        # TODO эта ветка выполнения только про API4 и устарела, после удаления кода API4 надо удалить её тоже
        if ($options{api}) {
            return 'strategy_no_premium_highest';
        }

        return 'strategy_no_premium';
    }

#    return 'stop' if $vals->{platform} eq 'context';
    return 'default';
}


=head2 detect_search_strategy($campaign)

    Поисковая стратегия на кампанию

    Возвращаемые значения:
        default                => Наивысшая доступная позиция
        strategy_no_premium
        stop                   => Показы на поиске остановлены 
        undef                  => Кампания не имеет поисковой стратегии 

    options: 
        api - отдавать токены стратегий для API

=cut

sub detect_search_strategy {
    my ($campaign, %options) = @_;

    return 'stop' if ($campaign->{platform} || '') eq 'context';

    return $campaign->{strategy} && $campaign->{strategy} eq 'different_places'
        ? detect_strategy($campaign, without_different_places => 1, %options)
        : undef;
}


=head2 detect_context_strategy($campaign)

    Поисковая стратегия на кампанию 

    Возвращаемые значения:
        default                => Дефолтная стратегия для сети (процент от ставок на поиске)
        maximum_coverage       => максимальный доступный охват 
        stop                   => Показы в сети остановлены  

=cut

sub detect_context_strategy {
    my ($vals, %options) = @_;
    my $autobudget = $vals->{autobudget} || '';

    if ($vals->{platform} eq 'context' && $autobudget =~ /^(Yes|1|True)$/i) {
        return detect_autobudget_strategy($vals);
    }

    if ($vals->{strategy} && $vals->{strategy} eq 'different_places') {
        return 'maximum_coverage';
    } elsif ($vals->{platform} eq 'search') {
        return 'stop';
    }

    return 'default';
}


=head2 detect_autobudget_strategy($campaign)

    Если используется стратегия с автобюджетом определяет какая именно
    
    Параметры:
        $campaign - {} хэш кампании, обязательные поля
                        mediaType - тип кампании
        
    Результат:        
    
    Название в коде         - Название в API            - Название в интерфейсе
    autobudget_roi          - ROIOptimization           - Рентабельность рекламы
    autobudget_avg_cpa      - AverageCPAOptimization    - Средняя цена конверсии
    autobudget_avg_cpi      - TODO                      - Средняя цена установки
    autobudget_avg_click    - AverageClickPrice         - Средняя цена клика
    autobudget_week_bundle  - WeeklyPacketOfClicks      - Недельный пакет кликов
    cpa_optimizer           - CPAOptimizer              - Недельный бюджет: максимальная конверсия
    autobudget              - WeeklyBudget              - Недельный бюджет: максимум кликов
    

=cut

sub detect_autobudget_strategy($) {
    my $vals = shift;

    my $type = $vals->{type} || $vals->{mediaType} || 'text';
    return undef if $type eq 'wallet' || $type eq 'billing_aggregate';
    my $strategy = $vals->{strategy_decoded} || from_json($vals->{strategy_data});
    my $strategy_name = $strategy->{name};

    if ($strategy_name eq 'autobudget' && defined $strategy->{goal_id}) {
        $strategy_name = 'cpa_optimizer';
    }

    return $strategy_name;
}

=head2 get_attribution_model_or_default_by_type

    определить используемую модель атрибуции
    для новых типов кампаний дефолт LAST_YANDEX_DIRECT_CLICK

=cut

sub get_attribution_model_or_default_by_type {
    my ($vals) = @_;

    if ($vals->{attribution_model}) {
        return $vals->{attribution_model};
    }

    return get_attribution_model_default();
}

=head2 get_attribution_model_for_stat_request

    определить используемую модель атрибуции для запросов в статистику БК
    для новых типов кампаний дефолт LAST_YANDEX_DIRECT_CLICK

=cut

sub get_attribution_model_for_stat_request {
    my ($camp) = @_;

    if ($camp->{type} eq 'mobile_content') {
        return 'last_click';
    }

    return $camp->{attribution_model} || get_attribution_model_default();
}

=head2 get_attribution_model_default

    Возвращает модель атрибуции по умолчанию

=cut

sub get_attribution_model_default {
    state $cross_device_default_attribution_type_enabled = Property->new('CROSS_DEVICE_DEFAULT_ATTRIBUTION_TYPE_ENABLED');
    if ($cross_device_default_attribution_type_enabled->get(60) // 0) {
        return 'last_yandex_direct_click_cross_device';
    }
    return 'last_yandex_direct_click';
}

=head2 get_attribution_model_internal_stat_default

    Возвращает модель атрибуции по умолчанию для внутренней рекламы в статистике

=cut

sub get_attribution_model_internal_stat_default {

    return 'last_significant_click';
}

=head2 get_common_stat_default_attribution

    Возвращает модель атрибуции по умолчанию для запросов общей статистики в БК.
    Todo: выпилить через год, когда накопится статистика по LYDC: DIRECT-104090

=cut

sub get_common_stat_default_attribution {

    return 'last_significant_click';
}

=head2 get_client_campaigns_attribution_type_or_default

   Проверяет модель атрибуции на кампаниях клиента.
   Если на всех кампаниях клиента установлена одна модель атрибуции, возрващает ее
   В противном случае - значение по-умолчанию.

=cut

sub get_client_campaigns_attribution_type_or_default {
    my $client_id = shift;

    my $camps_attribution_models = get_one_column_sql(PPC(ClientID => $client_id), ["select distinct attribution_model from campaigns"
        , where => {
             ClientID => $client_id,
             type__not_in => [qw/mcb geo wallet billing_aggregate/],
             statusEmpty => 'No',
             archived => 'No'
        }
    ]) || [];

    return (@$camps_attribution_models == 1) ?
         shift @$camps_attribution_models
             : get_attribution_model_default();
}

=head1 Функции про пользователей

=cut

=head2 get_user_info

    Возвращает данные о пользователе по UIDу.
    Используется там, где нельзя использовать User::get_user_data из-за зависимостей.

    $data = get_user_info($uid);
    $data => {
        ClientID => 12345,
        login => 'holodilnikru',
        email => 'holod@ilnkir.ru',
        # правильно использовать только fio
        # FIO оставлено для совместимости со старым кодом
        fio => 'Холодильник Иван Никифорович',
        FIO => 'Холодильник Иван Никифорович',
        phone => '+78122128506',
        uid => 456789,
        statusBlocked => 'No,
        emaillink => '<a href="mailto:holod@ilnkir.ru">Холодильник Иван Никифорович (holodilnikru)</a>',
        emaillink_for_client => '<a href="mailto:holod@ilnkir.ru">Холодильник Иван Никифорович &lt;holod@ilnkir.ru&gt;</a>',
    };

=cut
sub get_user_info($)
{
    my ($uid) = @_;
    return undef if !defined $uid;

    my $info = get_users_list_info([$uid])->{$uid};
    _enrich_user_info($info);
    return $info;
}

sub _enrich_user_info {
    my ($info) = @_;

    if ($info) {
        $info->{fio} = $info->{FIO};    # legacy
        $info->{email} = '' unless defined $info->{email};
        $info->{login} = '' unless defined $info->{login};
        $info->{phone} = '' unless defined $info->{phone};
        $info->{emaillink} = qq|<a href="mailto:$info->{email}">|.($info->{fio} || '').qq| ($info->{login})</a>|;
        $info->{emaillink_for_client} = qq|<a href="mailto:$info->{email}">| . ($info->{fio} || '') . qq| &lt;$info->{email}&gt;</a>|;
    }

    return;
}

=head2 get_users_list_info

    информация из users, clients по списку uid-ов, опционально с дополнительной фильтрацией

    get_users_list_info($uids_array_ref, %where_options);

    my $info = get_users_list_info([$uid1, $uid2, ...]);
    $info:
    {
        $uid1 => {ClientID => 123, login => 'login1', FIO => 'FIO1', email => 'login1@ya.ru', phone => 1344234, uid => $uid1, statusBlocked => 'No', hidden => 'No'},
        $uid2 => {ClientID => 123, login => 'login2', FIO => 'FIO2', email => 'login2@ya.ru', phone => 4567245, uid => $uid2, statusBlocked => 'Yes', hidden => 'Yes'},
        ...
    }

    my $info_not_arch_only = get_users_list_info([$uid1, $uid2, ...], statusArch => 'No');

=cut

sub get_users_list_info {
    my ($uids, %opt_where) = @_;

    return get_hashes_hash_sql(PPC(uid => $uids),
        ["SELECT u.uid, u.ClientID, u.login, u.FIO, u.email, u.phone, u.verified_phone_id, u.statusBlocked
               , u.hidden, u.rep_type
               , cl.country_region_id
               , cl.connect_org_id
          FROM users u
          LEFT JOIN clients cl using(ClientID)
         "
         , where => {uid => SHARD_IDS, %opt_where}
        ]
    );
}

=head2 get_lim_rep_info($uid)

    Возвращаем хэш с данными ограниченного представителя агентства для заданного uid'а

=cut

sub get_lim_rep_info {
    my ($uid) = @_;
    return undef if !$uid;

    my $lim_rep_info = get_one_line_sql(PPC(uid => $uid),
        [q/SELECT u.uid, u.ClientID, u.login, u.FIO, u.email, u.phone, u.verified_phone_id, u.statusBlocked
               , u.hidden
               , cl.country_region_id
               , cl.connect_org_id
               , IFNULL(ua.lim_rep_type, "legacy") lim_rep_type
          FROM users u
          LEFT JOIN clients cl using(ClientID)
          LEFT JOIN users_agency ua using(uid)/
         , where => {uid => SHARD_IDS}
        ]
    );

    _enrich_user_info($lim_rep_info);

    return $lim_rep_info;

}

=head2 get_lim_rep_users_list_info

    информация из users, clients и users_agency по списку uid-ов ограниченных представителей агентства, опционально с дополнительной фильтрацией

=cut

sub get_lim_rep_users_list_info {
    my ($uids, %opt_where) = @_;

    return get_hashes_hash_sql(PPC(uid => $uids),
        ["SELECT u.uid, u.ClientID, u.login, u.FIO, u.email, u.phone, u.verified_phone_id, u.statusBlocked
               , u.hidden
               , cl.country_region_id
               , cl.connect_org_id
               , ua.group_id as lim_rep_group_id, ua.lim_rep_type
          FROM users u
          LEFT JOIN clients cl using(ClientID)
          LEFT JOIN users_agency ua using(uid)
         "
         , where => {uid => SHARD_IDS, %opt_where}
        ]
    );
}


# --------------------------------------------------------------------

=head2 get_client_reps_list_info

    По списку ClientID или uid отдаем информацию по всем представителям клиента из users и clients

    my $clients_info = get_client_reps_list_info(ClientID => [123, 124]);
    my $clients_info = get_client_reps_list_info(uid => [12123, 12124]);

    Результат хеш с uid ключами:
    {
        uid1 => {
            uid => 12123,
            ClientID => 123,
            user_fio => 'user FIO',
            client_name => 'client name',
            ...
        },
        uid2 => {
            ...
        }
    }

=cut

sub get_client_reps_list_info {
    my %params = @_;

    my ($result, $shard);
    if ($params{ClientID}) {
        $shard = PPC(ClientID => $params{ClientID});
    } elsif ($params{uid}) {
        $shard = PPC(uid => $params{uid});
    } else {
        die "get_client_reps_list_info params is invalid";
    }

    my $where = {};
    $where->{'cl.ClientID'} = $params{ClientID} if defined $params{ClientID};
    $where->{'u.uid'} = $params{uid} if defined $params{uid};

    $result = get_hashes_hash_sql($shard, [
        "select u.uid
              , u.ClientID
              , u.FIO as user_fio
              , cl.name as client_name
         from clients cl
           join users u using(ClientID)
        ",
        where => $where
    ]);

    return $result;
}

sub get_fio_by_uid($)
{
    my $uids = shift;
    my $fio = get_one_field_sql(PPC(uid => $uids), ["select FIO from users", where => {uid => SHARD_IDS }]);
    return $fio;
}


# Посмотреть пользователя сначала в локальной базе, потом в
# passportdb
sub get_uid_by_login2($)
{
    my $login = shift;
    return get_uid(login => $login) || get_uid_by_login($login);
}


=head1 Работа с Черным Ящиком

=head2 get_uid_by_login

    Получает UID из Паспорта по логину
    Возвращает undef, если пользователя не найдено
    ПДДшных пользователей рассматривает только при with_pdd => 1
    Социальную авторизацию (без дорегистрации) рассматривает только при with_social => 1

    $uid = get_uid_by_login($login);
    $uid = get_uid_by_login($login, with_pdd => 1);
    $uid = get_uid_by_login($login, with_pdd => 1, with_social => 1);

=cut

sub get_uid_by_login
{
    my ($login, %O) = @_;

    my $bb = bb_userinfo({login => $login}, $ENV{REMOTE_ADDR} || '127.0.0.1', "direct.yandex.ru");
    if ($bb) {
        if (!$bb->{hosted}) {
            if ($bb->{login}) {
                return $bb->{uid};
            } elsif ($O{with_social}) {
                return $bb->{uid};
            } else {
                return undef;
            }
        } elsif ($O{with_pdd}) {
            return $bb->{uid};;
        } else {
            return undef;
        }
    } else {
        return undef;
    }
}


sub get_login_by_uid_passport($)
{
    my ($uid) = shift;
    return get_info_by_uid_passport($uid)->{login};
}


sub get_info_by_uid_passport($) {
    my ($uid) = @_;
    my $bb_res = bb_userinfo($uid, $ENV{REMOTE_ADDR} || '127.0.0.1', "direct.yandex.ru"
                                  , [$BB_LOGIN, $BB_EMAIL, $BB_FIO, $BB_REG_DATE], 'getdefault');

    return {
        login => $bb_res->{dbfield}{$BB_LOGIN},
        email => $bb_res->{'address-list'}->{'address-list'}->[0] || $bb_res->{dbfield}{$BB_EMAIL},
        fio => $bb_res->{dbfield}{$BB_FIO},
        karma => $bb_res->{karma}{content},
        reg_date => $bb_res->{dbfield}{$BB_REG_DATE},
    };
}

#-----------------------------------------------------------

=head2 check_direct_sid

  проверяем что логин подписан на 14 sid (логин использует директ, его нельзя удалять)

=cut

sub check_direct_sid($) {
    my ($login) = @_;
    my $bb_res = bb_userinfo({login => $login}, $ENV{REMOTE_ADDR} || '127.0.0.1', "direct.yandex.ru"
                                  , [$BB_SUID_DIRECT]);
    return $bb_res->{dbfield}{$BB_SUID_DIRECT};
}

#-----------------------------------------------------------

=head2 get_agency_client_relations

    Есть ли обслуживание клиента у агентства

    # информация по 1 клиенту
    $relation_hashref = get_agency_client_relations($agency_ClientID, $client_ClientID);
    # информация по списку клиентов
    $relation_hashref = get_agency_client_relations($agency_ClientID, [$client_ClientID1, $client_ClientID2]);

    возвращает хеш результатов
    $relation_hashref: {
        client_ClientID1 => {relation => 1                              # есть или возможно обслуживание на заданном агентстве
                             , agencies_count => 1                      # всего агентств у клиента
                             , allow_create_scamp_by_subclient => 1     # клиенту разрешено создвать самоходные кампании
                             , servicing_types_count => 2               # кол-во типов обслуживания
                             , client_description => 'описание клиента' # описание клиента заданое агентством
                             , client_archived => 1                     # статус архивности клиента у агентства (0 or 1)
                            },

        client_ClientID2 => {...},
        ...
    }

=cut

sub get_agency_client_relations {
    my ($agency_client_id, $client_client_ids) = @_;

    my %freedom_by_client;
    for my $fd (@{get_agency_clients_relations_data($client_client_ids)}) {
        push @{$freedom_by_client{$fd->{client_client_id}}}, $fd;
    }

    my $allow_create_scamp_by_subclient = get_all_sql(PPC(ClientID => $client_client_ids),
        ["select ClientID, allow_create_scamp_by_subclient from clients"
        , where => {ClientID => SHARD_IDS}
        ]

    );
    $allow_create_scamp_by_subclient = {map {$_->{ClientID} => $_->{allow_create_scamp_by_subclient} eq 'Yes' ? 1 : 0} @$allow_create_scamp_by_subclient};

    my $result = {};
    my @all_client_client_ids = ref($client_client_ids) ne 'ARRAY' ? $client_client_ids : @$client_client_ids;

    for my $ClientID (@all_client_client_ids) {
        my $freedom = $freedom_by_client{$ClientID} || [];

        # список по клиенту и всем агентствам с разрешениями
        my @freedom_data_for_client = grep {$_->{bind} eq 'Yes'} @$freedom;

        # есть запись с разрешениями по данному агентству
        my $this_agency_record = scalar(grep {$_->{agency_client_id} == $agency_client_id} @$freedom);
        # дано разрешение данному агентству
        my $this_agency_bind = scalar(grep {$_->{agency_client_id} == $agency_client_id} @freedom_data_for_client);
        # всего агентств с разрешениями на клиенте
        my $agencies_count = scalar(xuniq {$_->{agency_client_id}} @freedom_data_for_client);

        # свойства клиента у агентства, берем первое и единственное - см. первичный ключ
        my ($agency_client_settings) = grep {$_->{agency_client_id} == $agency_client_id} @$freedom;

        # клиенту разрешено создавать самоходные/сервисируемые кампании
        $allow_create_scamp_by_subclient->{ $ClientID } ||= 0;

        $result->{ $ClientID }->{allow_create_scamp_by_subclient} = $allow_create_scamp_by_subclient->{ $ClientID };
        $result->{ $ClientID }->{agencies_count} = $agencies_count;
        $result->{ $ClientID }->{servicing_types_count} = $agencies_count + $allow_create_scamp_by_subclient->{ $ClientID };

        # описание клиента у агентства, не зависимо от разрешения на обслуживание
        $result->{ $ClientID }->{client_description} = $agency_client_settings->{client_description};
        # статус архивности клиента у агентства, не зависимо от разрешения на обслуживание
        $result->{ $ClientID }->{client_archived} = $agency_client_settings->{client_archived} && $agency_client_settings->{client_archived} eq 'Yes' ? 1 : 0;

        # или должна быть запись на агентство-клиента или на клиенте не должно быть записей (и не должно быть разрешено создавать самоходные кампании)
        $result->{ $ClientID }->{relation} = $this_agency_bind || ($agencies_count == 0 && ! $allow_create_scamp_by_subclient->{ $ClientID }) ? 1 : 0;

        # есть явная привязка агентства к клиенту
        $result->{ $ClientID }->{agency_bind} = $this_agency_bind;
        # есть явная привязка агентства к клиенту, но клиент снят с обслуживания агентством
        $result->{ $ClientID }->{agency_unbind} = $this_agency_record && !$this_agency_bind ? 1 : 0;
    }

    return $result;
}

=head2 get_agency_clients_relations_data

    Raw data from table `agency_client_relations` for array of client_id

    my $arr_ref = get_agency_clients_relations_data(array_client_ids);

    Result:
        $arr_ref = [
            {
                agency_client_id => __
                , client_client_id => __
                , bind => __
                , client_description => __
                , client_archived => __
            }, ...
        ]

=cut

sub get_agency_clients_relations_data
{
    my ($client_client_ids) = @_;
    
    my $res = get_all_sql(PPC(ClientID => $client_client_ids),
        ["select agency_client_id
               , client_client_id
               , bind
               , client_description
               , client_archived
          from agency_client_relations"
         , where => {
             client_client_id => SHARD_IDS
         }
        ]
    );
    
    return $res;
}

=head2 mass_get_client_first_agency

    Находит по ClientID клиентов первые попавшиеся агентства, с которыми клиенты сейчас связаны.

    $client_clientid2agency_clientid = mass_get_client_first_agency(\@clientids);

=cut

sub mass_get_client_first_agency {
    my ($client_ids) = @_;

    return get_hash_sql(PPC(ClientID => $client_ids), ['
        SELECT client_client_id, MAX(agency_client_id)
        FROM agency_client_relations
    ',  WHERE => {
            client_client_id => SHARD_IDS,
            bind => 'Yes',
            client_archived => 'No',
    }, 'GROUP BY client_client_id']);
}

=head2 get_client_first_agency

    Находит по указанному ClientID первое попавшееся агентство, с которым связан клиент.

    $agency_client_id = get_client_first_agency($client_id);

=cut

sub get_client_first_agency {
    my ($client_id) = @_;

    my $clid2agid = mass_get_client_first_agency([$client_id]) || {};
    return $clid2agid->{$client_id};
}

#-----------------------------------------------------------

=head2 get_allow_agency_bind_client

  проверяем может ли агентство обслуживать клиента:
  либо должен быть доступ в agency_client_relations
  либо у клиента не должно быть других агентств кроме текущего (и не должно быть возможности создавать самоходные кампании)

  $result = get_allow_agency_bind_client($agency_id, $client_id);

=cut

sub get_allow_agency_bind_client {
    my ($agency_id, $client_id) = @_;

    my $relations = get_agency_client_relations($agency_id, $client_id);
    return $relations->{$client_id}->{relation} && !$relations->{$client_id}->{agency_unbind};
}

#-----------------------------------------------------------

=head2 mass_get_allow_create_scamp_by_subclient

    может ли субклиент создавать самоходные/сервисируемые кампании

    $allow_create_scamp_by_subclient = mass_get_allow_create_scamp_by_subclient(\@uids);

    $allow_create_scamp_by_subclient = {
        1231231 => 0,
        1231234 => 1,
    }

=cut

sub mass_get_allow_create_scamp_by_subclient {
    my ($uids) = @_;

    return get_hash_sql(PPC(uid => $uids), ["
            SELECT u.uid, IF(IFNULL(cl.allow_create_scamp_by_subclient, 'No') = 'Yes', 1, 0)
            FROM users u
                LEFT JOIN clients cl ON cl.ClientID = u.ClientID
         ", WHERE => {
                'u.uid' => SHARD_IDS
            }]);
}

=head2 get_allow_create_scamp_by_subclient

    может ли субклиент создавать самоходные/сервисируемые кампании

    $allow_create_scamp_by_subclient = get_allow_create_scamp_by_subclient($uid);

=cut

sub get_allow_create_scamp_by_subclient {
    my $uid = shift;

    return mass_get_allow_create_scamp_by_subclient([$uid])->{$uid} // 0;
}

#-----------------------------------------------------------

=head2 get_manager_email

    достаем email менеджера с учетом того использует ли он CRM.
    если использует то это internal_users.manager_private_email (и если он задан),
    иначе это users.email

    $manager_email = get_manager_email($manager_uid);

=cut

sub get_manager_email($) {
    my $manager_uid = shift;

    return get_one_field_sql(PPC(uid => $manager_uid), "select IF(manager_use_crm = 'Yes' and iu.manager_private_email is not null, iu.manager_private_email, u.email)
                                   from users u
                                     left join users_options uo using (uid)
                                     left join internal_users iu using (uid)
                                   where u.uid = ?", $manager_uid
                            );
}

=head2 get_domain_for_stat_by_bid

    возвращает домен баннера или, если домена нет, конструирует псевдодомен на основе телефона из визитки

=cut

sub get_domain_for_stat_by_bid($)
{
    my $bid = shift;
    
    my $domains = get_domains_for_stat_by_bids([$bid]);
    
    return $domains->{$bid};
}

sub get_domains_for_stat_by_bids(@)
{
    my $bids = shift;

    my $domains_by_bid_data = get_hashes_hash_sql(PPC(bid => $bids), ["
                                                SELECT b.bid
                                                      , ifnull(fd.filter_domain, b.domain) AS domain
                                                      , domains.domain AS dynamic_main_domain
                                                      , v.phone
                                                  FROM banners b
                                                  JOIN phrases p ON p.pid = b.pid
                                                       LEFT JOIN adgroups_dynamic ad ON p.adgroup_type = 'dynamic'
                                                                                        AND ad.pid = p.pid
                                                       LEFT JOIN domains ON domains.domain_id = ad.main_domain_id
                                                       LEFT JOIN vcards v ON v.vcard_id = b.vcard_id
                                                       LEFT JOIN filter_domain fd ON fd.domain = b.domain
                                                           ", where => { 'b.bid' => SHARD_IDS } ] );

    my $domains = undef;

    foreach my $bid (keys %$domains_by_bid_data) {
        my $domain_data = $domains_by_bid_data->{$bid};

        if ( $domain_data->{dynamic_main_domain} ) {
            # динамический баннер
            $domains->{$bid} = $domain_data->{dynamic_main_domain};
        } elsif ( $domain_data->{domain} ) {
            # обычный текстовый баннер
            $domains->{$bid} = $domain_data->{domain};
        } elsif ( !$domain_data->{domain} && $domain_data->{phone} ) {
            # если домена нет - то сохраняем телефон в виде псевдодоменного имени
            $domains->{$bid} = phone_domain($domain_data->{phone});
        }
    }

    return $domains;

}

=head2 product_info

по cid кампании или типу продукта вернуть информацию о продукте
без аргументов возвращает все существующие продукты (хэш по type)
если указан параметр hash_by_id - возвращаем все продукты в хэше по ProductID

product_info(type => 'text')
product_info(type => 'text', currency => 'USD')    # валюта обязательна!
product_info(cid => $cid)
product_info(ProductID => $product_id)
product_info(type => 'mcb_pkg')
{
    'EngineID' => '7',
    'NDS' => '1',
    'Price' => '1.000000',
    'ProductID' => '1475',
    'Rate' => '1',
    'UnitName' => 'Bucks',
    'currency' => 'YND_FIXED',
    'daily_shows' => undef,
    'packet_size' => undef,
    'product_type' => 'text',
    'type' => 'text',
}

product_info() => {
    text => product_info(type => 'text'),
    mcb  => product_info(type => 'mcb'),
    ...
}

=cut
{
my $productid_cache;
sub product_info {
    validate(@_, {
        cid => 0,
        type => 0,
        hash_by_id => 0,
        currency => 0,
        ProductID => 0,
        quasi_currency => 0,
    });
    my (%opt) = @_;

    if ($opt{ProductID}) {
        return $productid_cache->{$opt{ProductID}} ||= 
            get_one_line_sql(PPCDICT, ["SELECT type, ProductID, Price, NDS, UnitName, EngineID, Rate,
                                  prod.type as product_type, prod.daily_shows, prod.packet_size,
                                  prod.currency
                                  FROM products prod", WHERE => {'prod.ProductID' => $opt{ProductID}}]);
    } elsif ($opt{type}) {
        my $currency = $opt{currency};
        my $quasi_currency_flag = $opt{quasi_currency} // 0;
        my $info = product_info_by_type($opt{type}, $currency, $quasi_currency_flag);
        die "Unknown campaign type: $opt{type} for currency $currency and quasi_currency_flag $quasi_currency_flag" unless $info;
        return $info;
    } elsif ($opt{hash_by_id}) {
        return get_hashes_hash_sql(PPCDICT, "SELECT ProductID, type, Price, NDS, UnitName, EngineID, Rate, daily_shows, currency, packet_size
                                           FROM products
                                          WHERE Price IS NOT NULL and not (ProductID = 1475 and type = 'geo')");
    } elsif ($opt{cid}) {
        return product_info_by_cid($opt{cid});
    } else {
        return get_hashes_hash_sql(PPCDICT, "SELECT type, ProductID, Price, NDS, UnitName, EngineID, Rate, daily_shows, currency
                                           FROM products 
                                          WHERE Price IS NOT NULL");
    }
}
}

# -----------------------------------------------------------------------------

=head2 product_info_by_cid

Получить product_info по номеру cid.

=cut

sub product_info_by_cid
{
    my ($cid) = @_;
    my $product_id = get_one_field_sql(PPC(cid => $cid), 'select ProductID from campaigns c where cid = ?', $cid);
    return product_info(ProductID => $product_id);
}

=head2 validate_priority

    Провалидировать приоритет по ценам, если не правильный - вернуть приоритет по-умолчанию

=cut
sub validate_priority {
    my $val = shift;
    if ( defined $val && $val =~ /^(1|3|5)$/ ) {
        return $val;
    } else {
        return 3;
    }
}

=head2 product_info_by_type

Функция возвращает описание продукта по его типу (product_type) и валюте

=cut

sub product_info_by_type
{
    my ($type, $currency, $quasi_currency_flag) = @_;

    die 'no currency given' unless $currency;

    $type = 'text' if (camp_kind_in(type => $type, 'text_campaign_in_balance'));

    $currency = 'YND_FIXED' if none { $type eq $_ } qw/text mcb_turkish cpm_banner cpm_deals cpm_video
        internal_autobudget internal_distrib internal_free
        cpm_yndx_frontpage content_promotion cpm_price/; # валютный бывает только Директ и турецкий МКБ

    my %cond = (type => $type);
    $cond{currency} = $currency if $currency;
    if ($currency ne 'YND_FIXED' && (camp_kind_in(type => $type, 'text_campaign_in_balance', 'cpm') || $type eq 'cpm_video')) {
        # квазивалютность работает только для валютных кампаний Директа
        if($quasi_currency_flag) {
            $cond{UnitName} = 'QuasiCurrency';
        } else {
            $cond{UnitName} = 'Bucks';
        }
    }
    return get_one_line_sql(PPCDICT, ["SELECT type, ProductID, Price, NDS, UnitName, EngineID, Rate,
                                  prod.type as product_type, prod.daily_shows, prod.packet_size,
                                  prod.currency
                                  FROM products prod", WHERE => \%cond]);
}

=head2 product_info_by_product_types(\@product_types, $currency, $quasi_currency_flag)

Возвращает информацию по продуктам @product_types в валюте $currency.
Флаг $quasi_currency_flag указывает, что клиент в квазивалюте. Работает только для тенге (KZT)

=cut
sub product_info_by_product_types
{
    my ($product_types, $currency, $quasi_currency_flag) = @_;

    my $unit_name = 'Bucks';
    if ($currency eq 'KZT' && $quasi_currency_flag) {
        # Для клиентов в тенге испольузем квазивалютный продукт, если стоит флаг
        $unit_name = 'QuasiCurrency';
    }

    my $result = get_hashes_hash_sql(PPCDICT, ["SELECT type, ProductID, Price, NDS, UnitName, EngineID, Rate,
                                  prod.type as product_type, prod.daily_shows, prod.packet_size,
                                  prod.currency
                                  FROM products prod",
                                  WHERE => [
                                    'prod.type' => $product_types,
                                    'prod.currency' => $currency,
                                    'prod.UnitName' => $unit_name,
                                  ]
                                ]);
    my @products_missing = @{xminus $product_types, [keys %$result]};
    if (@products_missing) {
        croak "unknown products: ".join(',', @products_missing);
    }
    return $result;
}

# -----------------------------------------------------------------------------


=head2 get_campaign_warnplace

    возвращаем по кампании данные по фразам и баннерам которые попали в мониторинг по позициям

    на входе cid
    на выходе:
    {
        warnplace0   => {id1 => 1, id2 => 1, ...},
        warnplace1   => {id3 => 1, id4 => 1, ...},
        warnplace2   => {id5 => 1, id6 => 1, ...},
        show_adgroups => {pid1 => 1, pid2 => 1}
    }

=cut

sub get_campaign_warnplace($) {
    my $cid = shift;

    my $warnplace = get_all_sql(PPC(cid => $cid), "select id, bid, pid, old_place
                                      from warnplace
                                      where done = 'No'
                                        and cid = ?", $cid);
    $_->{old_place} = PlacePrice::set_new_place_style($_->{old_place}) foreach @$warnplace;
    $_->{old_place} = PlacePrice::get_warnplace_value($_->{old_place}) foreach @$warnplace;
    return {
        warnplace0   => {map {$_->{id} => 1} grep {$_->{old_place} == PlacePrice::get_guarantee_entry_place()} @$warnplace},
        warnplace2   => {map {$_->{id} => 1} grep {$_->{old_place} == PlacePrice::get_premium_entry_place()} @$warnplace},
        show_adgroups => {map {$_->{pid} => 1} @$warnplace}
    };
}

sub is_package_mcb
{
    return ($_[0] || '') eq 'mcb_pkg';
}

sub is_regional_mcb
{
    return ($_[0] || '') eq 'mcb_regional';
}

sub is_turkish_mcb
{
    return ($_[0] || '') eq 'mcb_turkish';
}

=head2 clear_banners_moderate_flags

    Удаляет для массива баннеров флажки постмодерации и автомодерации
    Принимает один позиционный параметр - массив id баннеров

=cut
sub clear_banners_moderate_flags {
    my ($bids) = validate_pos(@_, {type => ARRAYREF});
    return if !@$bids;

    for my $tbl ('post_moderate') {
        foreach_shard bid => $bids, sub {
            my ($shard, $bids_chunk) = @_;

            my $bids_to_del = get_one_column_sql(PPC(shard => $shard), ["SELECT bid FROM $tbl", WHERE => {bid => $bids_chunk}]) || [];
            if (@$bids_to_del) {
                do_delete_from_table(PPC(shard => $shard), $tbl, where => {bid => $bids_to_del});
            }
        }
    }

    Moderate::ReModeration->try_to_remove_auto_moderate_flag($bids);

}

=head2 reverse_domain

    переворачивает строку домена. Используется в таблице banners, для лучшей индексации.

=cut
sub reverse_domain {
    return scalar reverse( shift );
}



=head2 schedule_forecast(@cids)

  Запланировать перерасчёт прогноза

=cut

=head2 schedule_forecast

  Запланировать перерасчёт прогноза

=cut

sub schedule_forecast {
    my ( $cid ) = @_;

    schedule_forecast_multi([ $cid ]);
}

=head2 schedule_forecast_multi

  Запланировать перерасчёт прогноза для нескольких кампаний

=cut

sub schedule_forecast_multi($) {
    my $cids = shift; # []

    do_update_table(
        PPC(cid => $cids), 'campaigns',
        { autobudgetForecastDate__dont_quote => 'NULL' },
        where => { cid => SHARD_IDS, autobudgetForecastDate__is_not_null => 1 }
    );
}


=head2 get_auto_servicing_manager_uid

    Достаёт UID менеджера для кампаний клиента в случае, если его можно автоматически перевести на сервисируемость.
    Если менеджер только один, то вернется его UID, в противном случае результат будет 0

=cut

sub get_auto_servicing_manager_uid {
    my $uid = shift;
    return get_one_field_sql(PPC(uid => $uid), [
        "SELECT CASE COUNT(DISTINCT(ManagerUID)) WHEN 1 THEN ManagerUID ELSE 0 END FROM campaigns",
        WHERE => {
            ManagerUID__is_not_null => 1,
            statusEmpty => 'No',
            type => get_camp_kind_types("web_edit_base"),
            archived => 'No',
            uid => $uid,
        }
    ]);
}


=head2 get_other_manager_uid

=cut

sub get_other_manager_uid {
    my ($uid, $media_type) = @_;

    $media_type = 'text' unless defined $media_type;

    return get_one_field_sql(PPC(uid => $uid),
        ["SELECT MAX(ManagerUID) FROM campaigns",
        where => {
            uid => SHARD_IDS,
            ManagerUID__is_not_null => 1,
            statusEmpty => 'No',
            type => $media_type,
            archived => 'No'
        }]);
}

=head2 get_idm_primary_manager_uid

    Получение "главного менеджера" по Тирной схеме.
    Он задаётся через IDM

=cut

sub get_idm_primary_manager_uid {
    my ($client_id) = @_;

    return get_one_field_sql(PPC(ClientID => $client_id),
        ["SELECT primary_manager_uid FROM clients",
            where => {
                ClientID => SHARD_IDS,
                primary_manager_set_by_idm => 1
            }]);
}

=head2 query_special_user_option(ClientID, key)

    Получение особых пользовательских ограничений и прочих настроек

=cut

sub query_special_user_option($$) {
    my ($client_id, $option) = @_;
    return mass_query_special_user_option($client_id, $option)->{$client_id};
}

sub mass_query_special_user_option($$) {
    my ($client_ids, $option) = @_;
    return get_hash_sql(PPC(ClientID => $client_ids), ["
                    SELECT ClientID, `value`
                      FROM api_special_user_options"
                , where => {ClientID => SHARD_IDS, keyname => $option}
            ]);
}

=head2 set_special_user_option(ClientID, key, value)

    Установка и удаление особых пользовательских ограничений и прочих настроек

=cut

sub set_special_user_option($$$) {
    my ($client_id, $option, $value) = @_;

    if (!defined $value || $value eq '' || $value =~ /^-/) {
        return do_delete_from_table(PPC(ClientID => $client_id), 'api_special_user_options', where => {ClientID => $client_id, keyname => $option});
    }
    return do_insert_into_table(PPC(ClientID => $client_id), 'api_special_user_options', {ClientID => $client_id, keyname => $option, value => $value},
        on_duplicate_key_update => 1);
}

=head2 query_all_api_special_options(ClientID)

    Получение всех особых прользовательских настроек:
        ссылки на хеш всех ограничений по вызовам метода,
        числа одновременных запросов к API

=cut

sub query_all_api_special_options($) {
    my $client_id = shift;

    my $hash = get_hash_sql(PPC(ClientID => $client_id), ["select `keyname`, `value` from api_special_user_options", where => {ClientID => SHARD_IDS}]);

    my $rethash = {};
    while (my($key, $value) = each %$hash) {
        if ($key =~ /^(.+)_calls_limit$/) {
            $rethash->{method_limits}{$1} = $value;
        } else {
            $rethash->{$key} = $value;
        }
    }

    return $rethash;
}


=head2 get_wallet_camp

    Возрвращаем cid, и другие параметры общего счета по типу сервисирования (агентство/не агентство) и валюте
    на входе uid шефа и ClientID агентства (или undef)

    my $result = get_wallet_camp($chief_uid, undef, $currency);
    my $result = get_wallet_camp($chief_uid, $agency_client_id, $currency);

    $result:
        {
            wallet_cid => 1234,
            is_enabled => 1,
            ...
        }

    Поле total - с учетом отрицательных остатков на кампаниях под счетом (включая кампании отключеные от счета но совпадающие по AgencyID)

=cut

sub get_wallet_camp($$$) {
    my ($chief_uid, $agency_client_id, $currency) = @_;

    my ($agency_reps, %cond);
    if ($agency_client_id) {
        $agency_reps = get_client_reps_list_info(ClientID => $agency_client_id);

        %cond = (
            'c.AgencyUID' => [keys %$agency_reps],
            _AND => {_TEXT => 'IFNULL(c.ManagerUID, 0) = 0'},
        );
    } else {
        %cond = (
            _AND => {_TEXT => 'IFNULL(c.AgencyUID, 0) = 0'},
        );
    }

    my $result = get_one_line_sql(PPC(uid => $chief_uid), [
            "select c.cid as wallet_cid
                  , c.uid
                  , c.AgencyUID
                  , IFNULL(c.currency, 'YND_FIXED') AS currency
                  , IF(COUNT(IF(text_camp.wallet_cid, text_camp.cid, NULL)) > 0, 1, 0) as is_enabled
                  , wc.last_change
                  , IF(wc.onoff_date IS NULL OR wc.onoff_date < DATE_SUB(NOW(), INTERVAL ? DAY), 1, 0) as allow_enable_wallet
                  , TIMEDIFF(DATE_ADD(wc.onoff_date, INTERVAL 1 DAY), NOW()) as time_to_enable
                  , IFNULL(wc.onoff_date, DATE_SUB(NOW(), INTERVAL 2 DAY)) as onoff_date
                  , c.sum - c.sum_spent + SUM(IF(text_camp.cid AND text_camp.sum - text_camp.sum_spent < 0, text_camp.sum - text_camp.sum_spent, 0)) as total
                  , c.sum_to_pay
                  , c.sum_last
                  , co.sms_flags
                  , co.sms_time
                  , co.email
                  , co.money_warning_value
                  , c.OrderID
                  , c.day_budget
                  , c.day_budget_show_mode
                  , wc.total_sum
             from campaigns c USE INDEX (i_uid)
               left join wallet_campaigns wc on wc.wallet_cid = c.cid
               left join camp_options co on c.cid = co.cid
               left join campaigns text_camp on text_camp.uid = c.uid
                                            -- по условию на AgencyID приджойниваются все кампании, даже не висящие на общем счёте
                                            -- это нужно, чтобы посчитать недоперенесённые деньги после отключения общего счёта
                                            and (text_camp.wallet_cid = c.cid OR text_camp.AgencyID = c.AgencyID)
                                            and text_camp.statusEmpty = 'No'
                                            and ", {'text_camp.type' => get_camp_kind_types('web_edit_base') },
                                            " and IFNULL(text_camp.currency, 'YND_FIXED') = IFNULL(c.currency, 'YND_FIXED')
             ", where => {
                 'c.type' => 'wallet'
                 , 'c.uid' => $chief_uid
                 , 'c.statusEmpty' => 'No',
                 , 'c.currency' => $currency
                 , %cond
             },
             "group by c.cid",
             'limit 1',
        ], $Settings::MIN_WALLET_DAYS_INTERVAL);

    $result = {
        wallet_cid => 0
        , uid => $chief_uid
        , is_enabled => 0
        , allow_enable_wallet => 1
        , sum => 0
        , sum_spent => 0
        , currency => $currency // 'YND_FIXED'
    } unless defined $result;

    # добавляем информацию по агентству
    if ($agency_client_id && $result->{AgencyUID} && exists $agency_reps->{ $result->{AgencyUID} }) {
        $result->{agency_name} = $agency_reps->{ $result->{AgencyUID} }->{client_name};
        $result->{agency_fio} = $agency_reps->{ $result->{AgencyUID} }->{user_fio};
        $result->{agency_client_id} = $agency_reps->{ $result->{AgencyUID} }->{ClientID};
    }

    return $result;
}

# --------------------------------------------------------------------

=head2 get_all_wallet_camps

    Возрвращаем cid, и другие параметры общих счетов клиента
    на входе uid шефа

    my $result = get_all_wallet_camps(client_chief_uid => $chief_uid);
    my $result = get_all_wallet_camps(client_client_id => $ClientID);

    $result:
        [{
            wallet_cid => 1234,
            is_enabled => 1,
            agency_client_id => 735,
            ...
         },
         {...}, ...
        ]

=cut

sub get_all_wallet_camps(%) {
    my (%OPT) = @_;

    my $where = {'c.type' => 'wallet'};           
    my ($shard, $join);

    if ($OPT{client_chief_uid}) {
        $where->{'c.uid'} = $OPT{client_chief_uid};
        $shard = PPC(uid => $OPT{client_chief_uid});
        $join = '';
    } elsif ($OPT{cid}) {
        $where->{'c.cid'} = $OPT{cid};
        $shard = PPC(cid => $OPT{cid});
        $join = '';
    } elsif ($OPT{client_client_id}) {
        $where->{'u.ClientID'} = $OPT{client_client_id};
        $shard = PPC(ClientID => $OPT{client_client_id});
        $join = 'join users u on u.uid = c.uid';
    } else {
        die "get_all_wallet_camps failed: need client param";
    }

    my $result = get_all_sql($shard, [
        "select c.AgencyID as agency_client_id
               , c.cid as wallet_cid
               , IF(COUNT(text_camp.cid) > 0, 1, 0) as is_enabled
               , wc.last_change
               , IF(wc.onoff_date IS NULL OR wc.onoff_date < DATE_SUB(NOW(), INTERVAL ? DAY), 1, 0) as allow_enable_wallet
               , TIMEDIFF(DATE_ADD(wc.onoff_date, INTERVAL 1 DAY), NOW()) as time_to_enable
               , IFNULL(wc.onoff_date, DATE_SUB(NOW(), INTERVAL 2 DAY)) as onoff_date
               , c.sum - sum(IF(text_camp.cid, IF(text_camp.sum_spent > text_camp.sum, text_camp.sum_spent - text_camp.sum, 0), 0)) as wallet_total
               , c.sum
               , c.sum_spent
               , c.sum_to_pay
               , c.sum_last
               , IFNULL(c.currency, 'YND_FIXED') AS currency
               , c.uid
               , co.sms_flags
               , co.sms_time
               , co.email
               , co.money_warning_value
         from campaigns c
           left join wallet_campaigns wc on wc.wallet_cid = c.cid
           left join camp_options co on c.cid = co.cid
           left join campaigns text_camp on text_camp.uid = c.uid and text_camp.wallet_cid = c.cid and text_camp.statusEmpty = 'No'
           $join
        ", where => $where,
        'group by c.AgencyID, c.cid'
    ], $Settings::MIN_WALLET_DAYS_INTERVAL);

    return $result;
}


# --------------------------------------------------------------------

=head2 get_wallets_by_cid

Получаем кампании общие счета по списку wallet_cid-ов

    my $wallets = get_wallets_by_cid([123, 435])

=cut

sub get_wallets_by_cid($) {
    my ($cids) = @_;

    do_sql(PPC(cid => $cids), 'SET SESSION group_concat_max_len = 100000');
    my $result = get_all_sql(PPC(cid => $cids), [
        "select c.AgencyID as agency_client_id
                , c.AgencyUID
                , c.cid as wallet_cid
                , c.uid
                , c.sum
                , c.sum_spent
                , c.sum_to_pay
                , c.sum_last
                , IFNULL(c.currency, 'YND_FIXED') AS currency
                , co.sms_flags
                , co.sms_time
                , co.email
                , co.money_warning_value
                , IF(COUNT(text_camp.cid) > 0, 1, 0) as is_enabled
                , COUNT(text_camp.cid) as text_camp_count
                , GROUP_CONCAT(text_camp.cid) as text_camp_cids
                , wc.last_change
                , IF(wc.onoff_date IS NULL OR wc.onoff_date < DATE_SUB(NOW(), INTERVAL ? DAY), 1, 0) as allow_enable_wallet
                , TIMEDIFF(DATE_ADD(wc.onoff_date, INTERVAL 1 DAY), NOW()) as time_to_enable
                , IFNULL(wc.onoff_date, DATE_SUB(NOW(), INTERVAL 2 DAY)) as onoff_date
         from campaigns c
            left join wallet_campaigns wc on wc.wallet_cid = c.cid
            left join camp_options co on c.cid = co.cid
            left join campaigns text_camp on text_camp.uid = c.uid and text_camp.wallet_cid = c.cid and text_camp.statusEmpty = 'No'
          ",
          WHERE => {'c.type' => 'wallet'
                     , 'c.cid' => SHARD_IDS
                   },
          'group by c.AgencyID, c.cid'
    ], $Settings::MIN_WALLET_DAYS_INTERVAL);

    if (@$result) {
        my $agency_reps = get_client_reps_list_info(ClientID => [map {$_->{agency_client_id}} @$result]);

        for my $row (@$result) {
            # добавляем информацию по агентству
            if ($row->{AgencyUID} && exists $agency_reps->{ $row->{AgencyUID} }) {
                $row->{agency_name} = $agency_reps->{ $row->{AgencyUID} }->{client_name};
                $row->{agency_fio} = $agency_reps->{ $row->{AgencyUID} }->{user_fio};
            }
        }
    }

    return $result;
}

# --------------------------------------------------------------------

=head2 is_first_camp_under_wallet(cid, client_id)
   
    Есть ли у клиента с client_id кампании помимо той, что передана в cid

=cut

sub is_first_camp_under_wallet($$) {
    my ($cid, $client_id) = @_;
    return get_one_field_sql(PPC(ClientID => $client_id), [
                "select 1
                 from campaigns
                ", where => {
                    ClientID => $client_id
                    , cid__ne => $cid
                    , type => get_camp_kind_types('under_wallet')
                    , statusEmpty => 'No'
                }, limit => 1
            ]) ? 0 : 1;
}

# --------------------------------------------------------------------

=head2 get_servicing_info_by_cids

Возвращаем информацию о сервисируемости по списку cid-ов: AgencyUID, AgencyID, ManagerUID

    my $info = get_servicing_info_by_cids([123, 456]);

    $info: {
        123 => {
            ManagerUID => 12123,
            AgencyID => 123,
            AgencyUID => 0,
        }, ...
    }

=cut

sub get_servicing_info_by_cids($) {
    my $cids = shift;

    return get_hashes_hash_sql(
        PPC(cid => $cids),
        ["select c.cid, c.ManagerUID, c.AgencyUID, c.AgencyID
          from campaigns c
         ", where => {
             "c.cid" => SHARD_IDS
         }
        ]
    );
}

=head2 filter_archived_campaigns(cids; client_id)

    Из списка кампаний вернуть только те, что заархивированы;
    если передан client_id, смотреть только в шарде этого клиента

=cut

sub filter_archived_campaigns($;$) {
    my ($cids, $client_id) = @_;
    unless ($client_id) {
        return get_one_column_sql(
            PPC(cid => $cids),
            [
                "select cid from campaigns", 
                where => {
                    cid => SHARD_IDS,
                    archived => 'Yes'
                }
            ]
        );
    }
    return get_one_column_sql(
        PPC(ClientID => $client_id),
        [
            "select cid from campaigns", 
            where => {
                cid => $cids,
                archived => 'Yes'
            }
        ]
    );
}

=head2 _normalize_domain

    Нормализует домен до вида, в котором он хранится в domains_dict:
    - в нижнем регистре

=cut

sub _normalize_domain
{
    return lc($_[0]);
}

=head3 _internal_domain2domain_id($domains, %params)

    Общий код для функций get_domain2domain_id и get_existing_domain2domain_id

    Параметры позиционные:
        $domains    - ссылка на массив доменов
    Параметры именованные:
        only_existsing  - не добавлять новые домены в базу, возвращать только существующие domain_id
    Результат:
        $domains_dict   - ссылка на хеш, где возможные ключи - домены из оригинального массива,
                          значения - domain_id

=cut

sub _internal_domain2domain_id {
    my ($domains, %params) = @_;

    die "no domains array given" unless $domains && ref($domains) eq 'ARRAY';
    return {} unless @$domains;

    my %domain2norm_domain = map { $_ => _normalize_domain($_) } grep { $_ } @$domains;
    my @norm_domains = uniq values %domain2norm_domain;

    my $existent_domains2id = {};
    for my $chunk (chunks(\@norm_domains, 4_000)) {
        my $sth = exec_sql(PPCDICT, ["SELECT domain, domain_id FROM domains_dict", where => { domain => $chunk }]);
        while (my ($key, $value) = $sth->fetchrow_array()) {
            $key = '' unless defined $key;
            $existent_domains2id->{$key} = $value;
        }
    }

    if (!$params{only_existsing}) {
        my @domains_to_add = @{xminus \@norm_domains, [keys %$existent_domains2id]};
        if (@domains_to_add) {
            do_mass_insert_sql(PPCDICT, "INSERT IGNORE INTO domains_dict (domain) VALUES %s", [map { [$_] } @domains_to_add]);
            my $added_domains2id = get_hash_sql(PPCDICT, ["SELECT domain, domain_id FROM domains_dict", where => { domain => \@domains_to_add }]);
            $existent_domains2id = hash_merge $existent_domains2id, $added_domains2id;
        }
    }

    # возвращаем домены в том же виде, в каком их спрашивали
    my %result_domains;
    while (my ($domain, $norm_domain) = each %domain2norm_domain) {
        if ($params{only_existsing} && !exists $existent_domains2id->{$norm_domain}) {
            next;
        }
        $result_domains{$domain} = $existent_domains2id->{$norm_domain};
    }

    return \%result_domains;
}

=head2 get_domain2domain_id($domains)

    Принимает ссылку на массив доменов
    Возвращает пары domain => domain_id из таблицы ppcdict.domains_dict
    Eсли домена нет, то добавляет

    $domain2domain_id = get_domain2domain_id(\@domains);

=cut

sub get_domain2domain_id($) {
    return _internal_domain2domain_id(shift);   
}

=head2 get_camp_options

    Достаем параметры в camp_secondary_options.

    $camp_options = get_camp_options($cid);

=cut

sub get_camp_options($) {
    my $cid = shift;

    my $option = get_one_field_sql(PPC(cid => $cid), "select options from camp_secondary_options where cid = ? AND `key` = 'default'", $cid);
    return $option ? YAML::Load($option): {};
}

=head2 set_camp_options

    Сохраняем параметры в camp_secondary_options.

    $camp_options = set_camp_options($cid, {$field_name => $field_value, ...});

=cut

sub set_camp_options($$) {
    my ($cid, $option) = @_;

    $option = (defined $option && %$option) ? YAML::Dump($option) : undef;
    do_sql(PPC(cid =>  $cid), "
        INSERT INTO camp_secondary_options set cid = ?, `key` = 'default', options = ?
        ON DUPLICATE KEY UPDATE options = values(options)
    ", $cid, $option);
}

=head2 update_camp_options

    Изменяет значения указанных параметров в camp_secondary_options. Не указанные параметры оставляет неизменными.
    Возвращает набор опций, который получился после изменения.

    $new_camp_options = update_camp_options($cid, {$field_name => $field_value, ...});

=cut

sub update_camp_options($$) {
    my ($cid, $changes) = @_;

    my $camp_options = get_camp_options($cid);
    hash_merge $camp_options, $changes;
    set_camp_options($cid, $camp_options);

    return $camp_options;
}

=head2 content_lang_to_view (lang)
    
    Преобразовывает язык к вариату как видят пользователи:
    Казахский: пользователи видят kz, хотя в БД хранится как kk

=cut
sub content_lang_to_view {
    my $lang = shift;
    $lang = 'kz' if $lang eq 'kk';
    $lang = 'ua' if $lang eq 'uk';
    return $lang;
}

=head2 content_lang_to_save (lang)

    Преобразовывает язык к вариату как хранится в БД:
    Казахский: пользователи видят kz, хотя в БД хранится как kk

=cut
sub content_lang_to_save {
    my $lang = shift;
    $lang = 'kk' if defined $lang && $lang eq 'kz';
    $lang = 'uk' if defined $lang && $lang eq 'ua';
    return $lang;
}

=head2 get_main_banner_ids_by_pids(@pids)

    Для списка AdGroupID возвращает ссылку на хэш с главным баннером
    для каждой группы
    { pid => main_banner_bid, .. }

=cut

sub get_main_banner_ids_by_pids {
    my @pids = @_ == 1 && ref($_[0]) eq 'ARRAY' ? @{$_[0]} : @_;

    # получаем для каждой группы первый баннер отвечающий критериям показа
    my $pid2bid = get_hash_sql(PPC(pid => \@pids), ["
        SELECT STRAIGHT_JOIN b.pid, MIN(b.bid) bid
        FROM banners b
            LEFT JOIN banner_turbolandings bt ON (bt.bid = b.bid)
            JOIN phrases p ON (p.pid = b.pid)
        ",
        WHERE => { 'b.pid' => SHARD_IDS },
        "
            AND b.statusShow = 'Yes'
            AND (
                b.statusActive = 'Yes'
                OR (
                    b.statusPostModerate = 'Yes'
                    AND p.statusPostModerate = 'Yes'
                    AND (
                        IFNULL(b.href, '') <> ''
                        OR b.phoneflag = 'Yes'
                        OR bt.tl_id IS NOT NULL
                    )
                )
            )
        GROUP BY b.pid"]);

    # для групп, для которых не удалось получить главный баннер, берем первый
    my @futile_pids = grep { ! $pid2bid->{$_} } @pids;

    if(scalar @futile_pids) {
        my $futile_pid2bid = get_hash_sql(PPC(pid => \@futile_pids), ["
            SELECT pid, MIN(bid) bid
            FROM banners",
            WHERE => { pid => SHARD_IDS },
            "GROUP BY pid"
        ]);
        hash_merge($pid2bid, $futile_pid2bid);
    }

    return $pid2bid;
}

=head2 get_goal_type_by_goal_id

По goal_id вычисляем тип данных: цель, сегмент метрики, сегмент аудитории

0      .. (1*10^9)-1 -> цели метрики
1*10^9 .. (2*10^9)-1 -> сегменты метрики
2*10^9 .. (3*10^9)-1 -> сегменты аудиторий
3*10^9 .. (3.9*10^9)-1 -> виртуальные ecommerce-цели
4*10^9 ..            -> счетчики

Полная таблица https://wiki.yandex-team.ru/jurijjgalickijj/raznoe/goalid/

=cut

use constant BRANDSAFETY_LOWER_BOUND => 4_294_967_296; # 2^32
use constant BRANDSAFETY_UPPER_BOUND => BRANDSAFETY_LOWER_BOUND + 1_000;
use constant CONTENT_CATEGORY_UPPER_BOUND => BRANDSAFETY_LOWER_BOUND + 3_000;
use constant CONTENT_GENRE_UPPER_BOUND => BRANDSAFETY_LOWER_BOUND + 5_000;
use constant HOST_LOWER_BOUND => 19_000_000_000;
use constant HOST_UPPER_BOUND => 19_900_000_000;

sub get_goal_type_by_goal_id {
    my $goal_id = shift;
    return undef if $goal_id < 0;
    return 'goal' if $goal_id < 1_000_000_000;
    return 'segment' if $goal_id < 1_500_000_000;

    return 'lal_segment' if $goal_id < 1_900_000_000;
    return 'mobile' if $goal_id < 2_000_000_000;

    # Изначальный диапазон audience [2_000_000_000; 3_000_000_000) был разбит на несколько
    # см. https://wiki.yandex-team.ru/users/aliho/projects/direct/crypta/#diapazony
    return 'audience' if $goal_id < 2_499_000_000;
    return 'social_demo' if $goal_id < 2_499_000_100;
    return 'family' if $goal_id < 2_499_000_200;
    return 'behaviors' if $goal_id < 2_499_001_100;
    return 'interests' if $goal_id < 2_499_990_000;
    return 'audio_genres' if $goal_id < 2_500_000_000;
    return 'ab_segment' if $goal_id < 2_600_000_000;
    return 'cdp_segment' if $goal_id < 3_000_000_000;
    return 'ecommerce' if $goal_id < 3_900_000_000;
    return 'goal' if $goal_id < BRANDSAFETY_LOWER_BOUND;
    return 'brandsafety' if $goal_id < BRANDSAFETY_UPPER_BOUND;
    return 'content_category' if $goal_id < CONTENT_CATEGORY_UPPER_BOUND;
    return 'content_genre' if $goal_id < CONTENT_GENRE_UPPER_BOUND;
    return 'host' if $goal_id >= HOST_LOWER_BOUND && $goal_id < HOST_UPPER_BOUND;

    # По идее, если goal_id >= 4_000_000_000, то это счетчик.
    # Но так как функция используется для заполнения данных в retargeting_goals, то там таких не должно быть.
    # Если вдруг потребуется, то за параметром надо будет открыть следующую строку.
    # return 'counter' if $goal_id >= 4_000_000_000;
    # А пока будем считать их целями
    return 'goal';
}

=head2 is_crypta_goal

Проверить является ли цель сегментом в Крипте. Признаки:
- наличие поля bb_keyword у интересов аудитории
- вычисляемый тип цели host

=cut

sub is_crypta_goal {
    my $goal = shift;
    if (exists $goal->{bb_keyword} || get_goal_type_by_goal_id($goal->{goal_id}) eq 'host') {
        return 1;
    }
    return 0;
}

=head2 get_freelancer_info

По переданному uid фрилансера достаем user_info и добавляем данные из карточки фрилансера
Поддерживаемые опции:
image - тип в mds для картинки, используемой для аватара фрилансера. По умолчанию - size180

Возвращает hashref на структуру, аналогичную выдаваемой user_info, в которую добавлен
avatar_url и поля 
icq, town, email, phone, siteUrl, telegram, whatsApp 
заполнены данными из карточки фрилансера

=cut

sub get_freelancer_info
{
    my ($uid, %opt) = @_;

    my $info = get_user_info($uid);

    my $image_suffix = $opt{image} // 'size180';
    my $card_info = get_one_line_sql(PPC(ClientID => $info->{ClientID}), [q/
            SELECT a.host_config_name, a.external_id, c.contacts, brief, first_name, second_name, rating FROM freelancers_card c
            JOIN freelancers f ON (c.ClientID = f.ClientID)
            LEFT JOIN clients_avatars a ON (a.avatar_id = c.avatar_id)/,
            WHERE => {'c.ClientID' => SHARD_IDS, 'c.status_moderate' => 'accepted', 'c.is_archived' => 0},  'ORDER BY' => 'c.freelancers_card_id' => 'DESC', LIMIT => 1]);

    $info->{avatar_url} = sprintf('https://%s/get-%s/'.$image_suffix, @$card_info{qw/host_config_name external_id/}) if $card_info->{external_id};
    $info = hash_merge($info, hash_cut($card_info, qw/first_name second_name brief rating/));
    $info = hash_merge($info, from_json($card_info->{contacts})) if $card_info->{contacts};

    return $info;
}

=head2 get_operator_info_for_do_cmd

    По переданному uid достаем набор данных оператора, используемый в DoCmd

=cut

sub get_operator_info_for_do_cmd {
    my ($uid) = @_;

    my $operator_info = get_one_line_sql(PPC(uid => $uid), "
        select u.uid
             , u.email, u.valid, u.LastChange, u.FIO, u.rep_type
             , u.phone, u.sendNews, u.sendWarn, u.createtime
             , u.ClientID, u.login, u.hidden, u.sendAccNews
             , u.not_resident, u.statusArch, u.statusBlocked
             , u.description, u.lang, u.captcha_freq
             , u.allowed_ips, u.statusYandexAdv, u.showOnYandexOnly
             , IFNULL(cl_ab.is_autobanned, 'No') as is_autobanned
             , iu.is_developer
             , iu.is_super_manager
             , uo.passport_karma
             , uo.options
             , uo.opts
             , uag.is_no_pay
             , uag.disallow_money_transfer
             , uag.lim_rep_type
             , cl.country_region_id
             , co.is_ya_agency_client
        from users u
            left join clients cl using(ClientID)
            left join clients_autoban cl_ab using(ClientID)
            left join clients_options co using(ClientID)
            left join internal_users iu using(uid)
            left join users_options uo using(uid)
            left join users_agency uag using(uid)
        where uid = ?", $uid) || {};

    return $operator_info;
}


=head2 get_adgroup_type($cid, $pid)

    Возвращает тип группы $pid в кампании $cid

=cut
sub get_adgroup_type {
    my ($cid, $pid) = @_;

    return get_one_field_sql(PPC(cid => $cid), ["
        SELECT adgroup_type FROM phrases",
        WHERE => [pid => $pid, cid => $cid]
    ]);
}

=head2 get_bannerid2bid
    
    Получить хэш соответствий BannerID => bid
    Возможные значения key:
    BannerID
=cut
sub get_bannerid2bid($$$) {
    my ($banner_ids, $sharding_key, $sharding_value) = @_;
    if ($sharding_key eq 'OrderID') {
        return get_hash_sql(PPC(OrderID => $sharding_value), [
            'SELECT b.BannerID, b.bid FROM banners b',
            WHERE => {'b.BannerID' => $banner_ids}]
        );
    } else {
        croak "Unsupported sharding key $sharding_key";
    }
}

=head2 get_internal_product_names_by_client_ids($client_ids)

    Возвращает названия продуктов внутренней рекламы по ClientID

=cut
sub get_internal_product_names_by_client_ids {
    my ($client_ids) = @_;

    return get_hash_sql(PPC(ClientID => $client_ids), [
        'SELECT ClientID, product_name FROM internal_ad_products',
        WHERE => { ClientID => $client_ids },
    ]);
}

1;
