
use strict;
use warnings;
use utf8;

package PrimitivesIds;

=head1 NAME
    
    PrimitivesIds

=head1 DESCRIPTION

    Модуль для простых низкоуровневых функций по преобразованию id.
    Интерфейсы всех функций должны стандарту, описанному ниже.
    Функции дописываются по мере необходимости.

    Все варианты функций должны покрываться тестами в PrimitivesIds/all.t

    **Названия идентификаторов**

    ClientID
    chief
    uid
    cid
    OrderID
    pid - phrases.pid, AdGroupID
    bid - banners.bid (не рассматриваем banner_images.image_id)
    BannerID - banners.BannerID (не рассматриваем banner_images.BannerID)

    **Методы:**
    
    my $id = get_id(ID => Value)
    по одиночному id получить соотвествующий ему другой id
    при этом, не все комбинации являются допустимыми, get_bid(cid => 88700) должен падать - непонятно какой именно bid возвращать
    если данные не нашлись в базе - возвращаем undef
    Примеры:
        my $pid = get_pid(bid => 123);
        get_pid(cid => 123); # !!croak!!


    my @ids = get_ids(ID => Value|[Values])
    по одиночному id или списку возвращает ссылку на cписок(возможно пустой) соотвествующих им уникальных id
    если данных для каких-то id не нашлось - соответствующие им id не вернутся 
    Примеры:
        my $pids = get_pids(cid => 123);
        my $cids = get_cids(pid => [123,123,2453,213,345]);
    
    
    my $id2id_hash = get_id2id(ID => Value|[Values])
    по одиночному id или списку возвращает ссылку на хэш id->id
    если данных для каких-то id не нашлось данных - в хэше не будет таких ключей
    Примеры:
        my $bid2pid = get_bid2pid(bid => [123,435,65]);
        my $bid2pid = get_bid2pid(cid => 123);
    
    
    my $id2ids_hash = get_id2ids(ID => Value|[Values]);
    по одиночному id или списку возвращает ссылку на хэш id->[id, …]
    Примеры:
        my $pid2bids = get_pid2bids(cid => 24323)

=cut

use Carp;
use List::MoreUtils qw/uniq/;

use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards;

use Settings;

use TextTools qw/normalize_login/;

use base qw/Exporter/;
our @EXPORT = qw/
    get_cid
    get_cids
    get_orderid2cid
    get_pid2cid
    get_bid2cid

    get_pid
    get_pids
    get_bid2pid

    get_bids
    get_pid2bids

    get_bannerids

    get_orderid
    get_orderids
    get_cid2orderid

    get_uid get_owner
    get_uids
    get_cid2uid
    get_login2uid

    get_login
    get_logins
    get_uid2login
    get_clientid2logins

    get_clientid
    get_clientids
    get_key2clientid
    get_cid2clientid
    get_uid2clientid
    get_login2clientid
    get_retcondids

    get_chief
/;


=head2 get_cid(key => val)

    Получить номер кампании по одному id
    Возможные значения key:
    bid pid media_bid media_gid mediaplan_bid OrderID

=cut
sub get_cid($$) {
    my ($key, $val) = @_;

    if (ref $val) {
        croak "Incorrect val in get_cid, scalar expected";
    } elsif (defined $key && grep {$_ eq $key} qw/bid pid media_bid media_gid mediaplan_bid OrderID/) {
        return get_cids($key, $val)->[0];
    } else {
        croak "Incorrect key in get_cid: $key";
    }
}


=head2 get_cids(key => val|[vals])

    Получить массив из номеров кампаний по одному или нескольким id:
    uid bid pid media_bid media_gid mediaplan_bid OrderID vcard_id

=cut
sub get_cids($$) {
    my ($key, $vals) = @_;

    my $cids;
    if ($key eq 'bid') {
        $cids = get_one_column_sql(PPC(bid => $vals), ["select cid from banners", where => {bid => SHARD_IDS}]);
    } elsif ($key eq 'vcard_id') {
        $cids = get_one_column_sql(PPC(vcard_id => $vals), ["select cid from vcards", WHERE => {vcard_id => SHARD_IDS}]);
    } elsif ($key eq 'pid') {
        $cids = get_one_column_sql(PPC(pid => $vals), ["select cid from phrases", where => {pid => SHARD_IDS}]);
    } elsif ($key eq 'media_bid') {
        $cids = get_one_column_sql(PPC(mbid => $vals), ["select mg.cid from media_banners mb join media_groups mg using(mgid)", WHERE => {mbid => SHARD_IDS}]);
    } elsif ($key eq 'media_gid') {
        $cids = get_one_column_sql(PPC(pid => $vals), ["select cid from media_groups", WHERE => {mgid => SHARD_IDS}]);
    } elsif ($key eq 'mediaplan_bid') {
        $cids = get_one_column_sql(PPC(mediaplan_bid => $vals), ["select cid from mediaplan_banners", where => {mbid => SHARD_IDS}]);
    } elsif ($key eq 'OrderID') {
        $cids = get_one_column_sql(PPC(OrderID => $vals), ["select cid from campaigns", where => {OrderID => SHARD_IDS}]);
    } elsif ($key eq 'uid') {
        $cids = get_one_column_sql(PPC(uid => $vals), ["select cid from campaigns", where => {uid => SHARD_IDS}]);
    } elsif ($key eq 'ClientID') {
        $cids = get_one_column_sql(PPC(ClientID => $vals), ["select cid from campaigns", where => {ClientID => SHARD_IDS}]);
    } else {
        croak "Incorrect key=$key in get_cids";
    }
    return [uniq @$cids];
}


=head2 get_orderid(key => val)

    Получить номер заказа в БК по одному id
    Возможные значения key:
    cid pid bid

=cut
sub get_orderid($$) {
    my ($key, $val) = @_;

    if (ref $val) {
        croak "Incorrect val in get_orderid, scalar expected";
    } elsif (defined $key && grep {$_ eq $key} qw/cid pid bid/) {
        return get_orderids($key, $val)->[0];
    } else {
        croak "Incorrect key in get_cid: $key";
    }
}


=head2 get_orderids(key => val|[vals])

    Получить массив из номеров заказов в БК по одному или нескольким id:
    ClientID uid cid pid bid

=cut
sub get_orderids($$) {
    my ($key, $vals) = @_;

    my $orderids;
    if ($key eq 'ClientID') {
        $orderids = get_one_column_sql(PPC(ClientID => $vals), ["select c.OrderID from campaigns c ", where => {'c.ClientID' => SHARD_IDS, 'c.OrderID__gt' => 0}]);
    } elsif ($key eq 'uid') {
        $orderids = get_one_column_sql(PPC(uid => $vals), ["select c.OrderID from campaigns c", where => {'c.uid' => SHARD_IDS, 'c.OrderID__gt' => 0}]);
    } elsif ($key eq 'cid') {
        $orderids = get_one_column_sql(PPC(cid => $vals), ["select c.OrderID from campaigns c", where => {'c.cid' => SHARD_IDS, 'c.OrderID__gt' => 0}]);
    } elsif ($key eq 'pid') {
        $orderids = get_one_column_sql(PPC(pid => $vals), ["select c.OrderID from phrases p join campaigns c on c.cid = p.cid", where => {'p.pid' => SHARD_IDS, 'c.OrderID__gt' => 0}]);
    } elsif ($key eq 'bid') {
        $orderids = get_one_column_sql(PPC(bid => $vals), ["select c.OrderID from banners b join campaigns c on c.cid = b.cid", where => {'b.bid' => SHARD_IDS, 'c.OrderID__gt' => 0}]);
    } else {
        croak "Incorrect key=$key in get_orderids";
    }
    return [uniq @$orderids];
}


=head2 get_cid2orderid(key => [vals])

    Получить хэш соответствий cid => OrderID
    Возможные значения key:
    cid

=cut
sub get_cid2orderid($$) {
    my ($key, $vals) = @_;
    if ($key eq 'cid') {
        return get_hash_sql(PPC(cid => $vals),
                                    ["SELECT c.cid, c.OrderID FROM campaigns c",
                                     WHERE => {'c.OrderID__gt' => 0, 'c.cid' => SHARD_IDS}
                                    ]);
    } else {
        croak "Unsupported key $key";
    }
}

=head2 get_orderid2cid(OrderID => [1,2,3])
    
    Получить хэш соответствий OrderID => cid
    Возможные значения key:
    uid OrderID cid

=cut
sub get_orderid2cid($$) {
    my ($key, $vals) = @_;
    if ($key eq 'uid') {
        return get_hash_sql(PPC(uid => $vals),
                            ["SELECT OrderID, cid FROM campaigns", WHERE => {OrderID__gt => 0, uid => SHARD_IDS}]
            );
    } elsif ($key eq 'cid') {
        return get_hash_sql(PPC(cid => $vals),
                            ["SELECT OrderID, cid FROM campaigns", WHERE => {OrderID__gt => 0, cid => SHARD_IDS}]
            );
    } elsif ($key eq 'OrderID') {
        return get_hash_sql(PPC(OrderID => $vals),
                            ["SELECT OrderID, cid FROM campaigns", WHERE => {OrderID => SHARD_IDS}]
            );
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_pid2cid(key => vals)
    
    Получить хэш соответствий pid => cid
    Возможные значения key:
    pid

=cut
sub get_pid2cid($$) {
    my ($key, $vals) = @_;
    if ($key eq 'pid') {
        return get_hash_sql(PPC(pid => $vals),
                            ["SELECT pid, cid FROM phrases", WHERE => {pid => SHARD_IDS}]
            );
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_bid2cid(key => vals)
    
    Получить хэш соответствий bid => cid
    Возможные значения key:
    bid

=cut
sub get_bid2cid($$) {
    my ($key, $vals) = @_;
    if ($key eq 'bid') {
        return get_hash_sql(PPC(bid => $vals),
                            ["SELECT bid, cid FROM banners", WHERE => {bid => SHARD_IDS}]
            );
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_pid(key => val)

    Получить номер группы объявлений по одному id
    Возможные значения key:
    bid

=cut
sub get_pid($$) {
    my ($key, $val) = @_;

    if (ref $val) {
        croak "Incorrect val in get_pid, scalar expected";
    } elsif (defined $key && grep {$_ eq $key} qw/bid/) {
        return get_pids($key, $val)->[0];
    } else {
        croak "Incorrect key in get_pid: $key";
    }
}


=head2 get_pids(key => val|[vals])

    Получить массив из номеров групп объавлений по одному или нескольким id:
    bid cid

=cut
sub get_pids($$) {
    my ($key, $vals) = @_;

    my $pids;
    if ($key eq 'bid') {
        $pids = get_one_column_sql(PPC(bid => $vals), ["select pid from banners", where => {bid => SHARD_IDS}]);
    } elsif ($key eq 'cid') {
        return get_one_column_sql(PPC(cid => $vals), ["select pid from phrases", where => {cid => SHARD_IDS}]);
    } else {
        croak "Incorrect key=$key in get_pids";
    }
    return [uniq @$pids];
}


=head2 get_bids(key => val|[vals])

    Получить массив из номеров групп объавлений по одному или нескольким id:
    cid pid

=cut
sub get_bids($$) {
    my ($key, $vals) = @_;

    if ($key eq 'pid') {
        return get_one_column_sql(PPC(pid => $vals), ["select bid from banners", where => {pid => SHARD_IDS}]);
    } elsif ($key eq 'cid') {
        return get_one_column_sql(PPC(cid => $vals), ["select bid from banners", where => {cid => SHARD_IDS}]);
    } elsif ($key eq 'creative_id') {
        return get_one_column_sql(PPC(creative_id => $vals), ["select bid from banners_performance", where => {'creative_id' => SHARD_IDS}]);
    } else {
        croak "Incorrect key=$key in get_bids";
    }
}


=head2 get_bid2pid(key => [vals])

    Получить хэш соответствий bid => pid
    Возможные значения key:
    cid pid bid

=cut
sub get_bid2pid($$) {
    my ($key, $vals) = @_;
    if ($key =~ /^(cid|pid|bid)$/) {
        return get_hash_sql(PPC($key => $vals),
                                    ["SELECT bid, pid FROM banners",
                                     WHERE => {$key => SHARD_IDS}
                                    ]);
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_pid2bids(key => [vals])

    Получить хэш соответствий pid => [bid1, ...]
    Возможные значения key:
    pid
    cid

=cut
sub get_pid2bids($$) {
    my ($key, $vals) = @_;

    my $pid2bids = {};
    if ($key eq 'pid' || $key eq 'cid') {
        my $sth = exec_sql(PPC($key => $vals), ["SELECT pid, bid FROM banners",
                                     WHERE => {$key => SHARD_IDS}
                           ]);
        while(my ($pid, $bid) = $sth->fetchrow_array) {
            push @{$pid2bids->{$pid}}, $bid;
        }
    } else {
        croak "Unsupported key $key";
    }
    return $pid2bids;
}

=head2 get_bannerids(key => val|[vals])

    Получить массив BannerID по одному или нескольким id:
    bid image_bid

=cut
sub get_bannerids($$) {
    my ($key, $vals) = @_;

    if ($key eq 'bid') {
        return get_one_column_sql(PPC($key => $vals), ["select BannerID from banners ", where => {$key => SHARD_IDS, BannerID__gt => 0}]);
    } elsif ($key eq 'image_bid') {
        return get_one_column_sql(PPC(bid => $vals), ["select BannerID from banner_images ", where => {bid => SHARD_IDS, BannerID__gt => 0}]);
    } else {
        croak "Incorrect key=$key in get_bannerids";
    }
}

=head2 get_clientid(key => val)

    Получить ClientID по одному id
    Возможные значения key:
    uid login cid OrderID pid bid ret_cond_id

=cut
sub get_clientid($$) {
    my ($key, $val) = @_;

    if (ref $val) {
        croak "Incorrect val in get_clientid, scalar expected";
    } elsif (defined $key && grep {$_ eq $key} qw/uid login cid OrderID pid bid/) {
        return get_clientids($key, $val)->[0];
    } else {
        croak "Incorrect key in get_clientid: $key";
    }
}


=head2 get_clientids(key => val|[vals])

    Получить массив ClientID по одному или нескольким id:
    uid login cid OrderID pid bid ret_cond_id

=cut
sub get_clientids($$) {
    my ($key, $vals) = @_;

    return [uniq values %{get_key2clientid($key, $vals)}];
}

=head2 get_key2clientid(key => val|[vals])

    Получить хэш соотсветствий key => ClientID по одному или нескольким id:
    uid login cid OrderID pid bid ret_cond_id

=cut
sub get_key2clientid($$) {
    my ($key, $vals) = @_;

    if ($key eq 'uid') {
        return get_hash_sql(PPC(uid => $vals), ["select u.uid, u.ClientID from users u", where => {'u.uid__int' => SHARD_IDS}]);
    } elsif ($key eq 'login') {
        return get_hash_sql(PPC(login => _norm_logins($vals)), ["select u.login, u.ClientID from users u", where => {'u.login' => SHARD_IDS}]);
    } elsif ($key eq 'cid') {
        return get_hash_sql(PPC(cid => $vals), ["select c.cid, c.ClientID from campaigns c", where => {'c.cid__int' => SHARD_IDS}]);
    } elsif ($key eq 'OrderID') {
        return get_hash_sql(PPC(OrderID => $vals), ["select c.OrderID, c.ClientID from campaigns c", where => {'c.OrderID__int' => SHARD_IDS}]);
    } elsif ($key eq 'pid') {
        return get_hash_sql(PPC(pid => $vals), ["select p.pid, c.ClientID from phrases p join campaigns c on c.cid = p.cid", where => {'p.pid__int' => SHARD_IDS}]);
    } elsif ($key eq 'bid') {
        return get_hash_sql(PPC(bid => $vals), ["select b.bid, c.ClientID from banners b join campaigns c on c.cid = b.cid", where => {'b.bid__int' => SHARD_IDS}]);
    } elsif ($key eq 'ret_cond_id') {
        return get_hash_sql(PPCDICT, ["select ret_cond_id, ClientID from shard_inc_ret_cond_id", where => {'ret_cond_id' => $vals}]);
    } else {
        croak "Incorrect key=$key in get_clientids";
    }
}


=head2 get_uid2clientid(key => [vals])

    Получить хэш соответствий uid => ClientID
    Возможные значения key:
    uid ClientID

=cut
sub get_uid2clientid($$) {
    my ($key, $vals) = @_;
    if ($key =~ /^(uid|ClientID)$/) {
        return get_hash_sql(PPC($key => $vals),
                                    ["SELECT uid, ClientID FROM users",
                                     WHERE => {$key => SHARD_IDS}
                                    ]);
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_login2clientid(key => [vals])

    Получить хэш соответствий login => ClientID

    Логины могут быть ненормализованы (такими же они будут в ответе)

    Возможные значения key:
    login

=cut
sub get_login2clientid($$) {
    my ($key, $vals) = @_;
    if ($key eq 'login') {
        my $login2norm = {};
        for my $login (ref $vals ? @$vals : $vals) {
            $login2norm->{$login} //= normalize_login($login);
        }
        my $norm2uid = hash_grep {defined} get_shard_multi(login => [values %$login2norm], 'uid');
        my $uid2clientid = get_uid2clientid(uid => [values %$norm2uid]);

        my %ret;
        while(my ($login, $norm) = each %$login2norm) {
            my $uid = $norm2uid->{$norm} || next;
            my $ClientID = $uid2clientid->{$uid} || next;
            $ret{$login} = $ClientID;
        }
        return \%ret;
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_login2uid(key => [vals])

    Получить хэш соответствий login => uid

    Логины могут быть ненормализованы (такими же они будут в ответе)

    Возможные значения key:
    login

=cut
sub get_login2uid($$) {
    my ($key, $vals) = @_;
    if ($key eq 'login') {
        my $login2norm = {};
        for my $login (ref $vals ? @$vals : $vals) {
            $login2norm->{$login} //= normalize_login($login);
        }
        my $norm2uid = hash_grep {defined} get_shard_multi(login => [values %$login2norm], 'uid');

        my %ret;
        while(my ($login, $norm) = each %$login2norm) {
            my $uid = $norm2uid->{$norm} || next;
            $ret{$login} = $uid;
        }
        return \%ret;
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_uid

    Находит uid владельца по одному из id
    login cid bid vcard_id

    Есть алиас - get_owner, лучше использовать get_uid

=cut
sub get_owner($$) { return get_uid($_[0], $_[1]); }
sub get_uid($$) {
    my ($key, $val) = @_;

    if (ref $val) {
        croak "Incorrect val in get_uid, scalar expected";
    } elsif (defined $key && grep {$_ eq $key} qw/ClientID login cid bid vcard_id/) {
        return get_uids($key, $val)->[0];
    } else {
        croak "Incorrect key in get_uid: $key";
    }
}


=head2 get_uids(key => val|[vals])

    Получить массив из номеров пользователей(uid) по одному или нескольким id:
    ClientID login cid OrderID bid vcard_id

=cut
sub get_uids($$) {
    my ($key, $vals) = @_;

    my $uids;
    if ($key eq 'ClientID') {
        $uids = get_one_column_sql(PPC(ClientID => $vals), ["select uid from users", where => {'ClientID__int' => SHARD_IDS}]);
    } elsif ($key eq 'login') {
        $uids = get_one_column_sql(PPC(login => _norm_logins($vals)), ["select max(uid) from users", where => {'login' => SHARD_IDS}, " group by login"]);
    } elsif ($key eq 'cid') {
        $uids = get_one_column_sql(PPC(cid => $vals), ["select c.uid from campaigns c", where => {'c.cid__int' => SHARD_IDS}]);
    } elsif ($key eq 'OrderID') {
        $uids = get_one_column_sql(PPC(OrderID => $vals), ["select uid from campaigns", where => {'OrderID__int' => SHARD_IDS}]);
    } elsif ($key eq 'bid') {
        $uids = get_one_column_sql(PPC(bid => $vals), ["select c.uid from banners b join campaigns c using (cid)", where => {'b.bid__int' => SHARD_IDS}]);
    } elsif ($key eq 'vcard_id') {
        $uids = get_one_column_sql(PPC(vcard_id => $vals), ["select c.uid from vcards vc join campaigns c using (cid)", where => {'vc.vcard_id__int' => SHARD_IDS}]);
    } else {
        croak "Incorrect key=$key in get_uids";
    }
    return [uniq @$uids];
}


=head2 get_login

    Находит login по одному из id
    uid

=cut
sub get_login($$) {
    my ($key, $val) = @_;

    if (ref $val) {
        croak "Incorrect val in get_uid, scalar expected";
    } elsif (defined $key && grep {$_ eq $key} qw/uid/) {
        return get_logins($key, $val)->[0];
    } else {
        croak "Incorrect key in get_uid: $key";
    }
}


=head2 get_logins

    Находит список login-ов по одному или нескольким id
    uid
    ClientID

=cut
sub get_logins($$) {
    my ($key, $vals) = @_;

    my $logins;
    if ($key eq 'uid') {
        $logins = get_one_column_sql(PPC($key => $vals), ["SELECT login FROM users", WHERE => {$key => SHARD_IDS}]);
    } elsif ($key eq 'ClientID') {
        $logins = [ uniq @{ get_one_column_sql(PPC($key => $vals), ["SELECT DISTINCT login FROM users", WHERE => {$key => SHARD_IDS}]) } ];
    } else {
        croak "Incorrect key in get_logins: $key";
    }
    return [uniq @$logins];
}


=head2 get_uid2login(key => val|[vals])

    Получить хэш соответствий uid => login
    Возможные значения key:
    uid

=cut
sub get_uid2login($$) {
    my ($key, $vals) = @_;

    if ($key eq 'uid') {
        return get_hash_sql(PPC($key => $vals), ["select uid, login from users", where => {'uid__int' => SHARD_IDS}]);
    } else {
        croak "Incorrect key=$key in get_uid2login";
    }
}


=head2 get_clientid2logins(key => val|[vals])

    Получить хэш соответствий ClientID => [login1, ...]
    Возможные значения key:
    ClientID

=cut
sub get_clientid2logins($$) {
    my ($key, $vals) = @_;

    if ($key eq 'ClientID') {
        my $sth = exec_sql(PPC(ClientID => $vals), ["select ClientID, login from users", where => {'ClientID__int' => SHARD_IDS}]);
        my %ret;
        while(my ($ClientID, $login) = $sth->fetchrow_array) {
            push @{$ret{$ClientID}}, $login;
        }
        return \%ret;
    } else {
        croak "Incorrect key=$key in get_clientid2logins";
    }
}


=head2 get_cid2uid(key => [vals])

    Получить хэш соответствий cid => uid
    Возможные значения key:
    cid

=cut
sub get_cid2uid($$) {
    my ($key, $vals) = @_;
    if ($key =~ /^(cid)$/) {
        return get_hash_sql(PPC($key => $vals),
                                    ["SELECT c.cid, c.uid FROM campaigns c",
                                     WHERE => {"c.$key" => SHARD_IDS}
                                    ]);
    } else {
        croak "Unsupported key $key";
    }
}


=head2 get_cid2clientid(key => [vals])

    Получить хэш соответствий cid => ClientID
    Возможные значения key:
    cid

=cut
sub get_cid2clientid($$) {
    my ($key, $vals) = @_;
    if ($key =~ /^(cid)$/) {
        return get_key2clientid($key => $vals);
    } else {
        croak "Unsupported key $key";
    }
}

=head2 get_chief

    Получить uid шефа
    ОСТОРОЖНО: до окончательного переезда RBAC в PPC лучше не использовать

=cut
sub get_chief($$) {
    my ($key, $val) = @_;
    if (ref $val) {
        croak "Incorrect val in get_chief, scalar expected";
    } elsif (defined $key && $key eq 'ClientID') {
        return get_one_field_sql(PPC(ClientID => $val), "SELECT chief_uid FROM clients WHERE ClientID = ?", $val);
    } elsif (defined $key && $key eq 'uid') {
        return get_one_field_sql(PPC(uid => $val), "SELECT c.chief_uid FROM users u JOIN clients c ON c.ClientID = u.ClientID WHERE u.uid = ?", $val);
    } else {
        croak "Incorrect key in get_clientid: $key";
    }
}

=head2 get_retcondids

    Находит список id-шников условий ретаргетинга по одному или нескольким id
    ClientID

=cut

sub get_retcondids($$) {
    my ($key, $vals) = @_;

    my $ret_cond_ids;
    if ($key eq 'ClientID') {
        $ret_cond_ids = get_one_column_sql(PPCDICT, ["SELECT ret_cond_id FROM shard_inc_ret_cond_id", WHERE => {$key => $vals}]);
    } else {
        croak "Incorrect key in get_retcondids: $key";
    }
    return \@$ret_cond_ids;
}

sub _norm_logins($) {
    my ($vals) = @_;
    if (!ref $vals) {
        return normalize_login($vals);
    } else {
        return [uniq map {normalize_login($_)} @$vals];
    }
}
    
1;
