package Promocodes;

=head1 NAME

    Promocodes

=head1 DESCRIPTION

    Модуль для вычисления необходимости предоставления клиенту возможности введения промокода в Балансе
    (10 существительных подряд, если не считать предлоги :)

=cut

use Direct::Modern;

use Yandex::MirrorsTools::Hostings qw/strip_domain strip_www/;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::IDN;
use MirrorsTools;
use Campaign::Types;

use List::MoreUtils qw/any uniq/;

use Settings;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    mass_check_is_domain_used
    is_promocode_entry_available
    get_promocode_domains
/;


=head2 mass_check_is_domain_used($domains)

    Возвращает хеш, в котором каждому домену ставится в соответствие факт его
    присутствия в статистике за 365 дней

    - $domains - arrayref со списком доменов

=cut

sub mass_check_is_domain_used
{
    my $domains = shift;

    push @$domains, map {Yandex::IDN::idn_to_ascii($_)} @$domains;
    push @$domains, map {Yandex::IDN::idn_to_unicode($_)} @$domains;
    $domains = [uniq @$domains];

    my $nrows = get_hash_sql(PPCDICT, ["SELECT DISTINCT filter_domain, 1 FROM api_domain_stat",
            WHERE => {filter_domain => $domains, shows_approx__gt => 0}]);

    my $res = {map {$_ => 0} @$domains};
    return hash_merge $res, $nrows;
}


=head2 is_promocode_entry_available($uid)

    Проверить, может ли пользователь вводить промокод:
        - На логине не было кампаний с деньгами
        - На логине нет РМП кампаний
        - На логине есть ровно один домен второго уровня (различие в www. - не учитывается)
        - Домен, а так же его зеркала отсутствуют в статистике показов за год.

=cut

sub is_promocode_entry_available
{
    my $uid = shift;

    my $kinds = get_camp_kind_types('text_campaign_in_balance', 'cpm');
    my $paid = get_one_field_sql(PPC(uid => $uid),
        ["SELECT 1 FROM campaigns",
         WHERE => {
             uid => $uid,
             type => $kinds,
             _OR => {
                 sum__gt => 0,
                 sum_last__gt => 0,
                 sum_spent__gt => 0,
             },
         },
         LIMIT => 1]);
    return 0 if $paid;

    my $banners_data = get_all_sql(PPC(uid => $uid),
        ["SELECT DISTINCT c.type as camp_type, b.domain, d.domain as dyn_domain
           FROM campaigns c
             JOIN phrases p ON p.cid = c.cid
             JOIN banners b ON b.pid = p.pid
             LEFT JOIN adgroups_dynamic ad ON p.pid = ad.pid
             LEFT JOIN domains d ON ad.main_domain_id = d.domain_id",
         WHERE => {
             "c.uid" => $uid,
             "c.type" => $kinds,
         }]);

    my $mt = MirrorsTools->new(dont_load_file => 1, use_db => 1);
    my $has_2_level_domains = 0;
    my $has_mobile_content_camps = 0;
    my %domains;             # домены без учета www в объявлениях пользователя (без учета доменов в РМП кампаниях)
    my %domains_and_mirrors; # домены из %domains + их зеркала для проверки статистики показов
    for my $banner (@$banners_data) {
        if ($banner->{camp_type} eq 'mobile_content') {
            $has_mobile_content_camps = 1;
        } else {
            for my $key (grep { $_ ne 'camp_type' } keys %$banner) {
                _add_domain($banner->{$key}, \%domains_and_mirrors, $mt, \$has_2_level_domains, \%domains);
            }
        }
    }

    if (  !$has_2_level_domains
        || $has_mobile_content_camps
        || (scalar keys %domains != 1) )
    {
        return 0;
    }

    my $domain_usage = mass_check_is_domain_used([keys %domains_and_mirrors]);
    if (any {$_} values %{$domain_usage}) {
        return 0;
    }

    return 1;
}


=head2 get_promocode_domains

    Получает список доменов, на которые действуют ограничения по действующему промокоду.
    Если www значимый, то список состоит из одного домена, если незначимый - то из двух.

=cut

sub get_promocode_domains {
    my $cid = shift;

    my @result_domains = ();

    my $wallet_cid = get_one_field_sql(PPC(cid => $cid), ["SELECT wallet_cid FROM campaigns", WHERE => {cid => $cid}]);
    my $cid_to_search = $wallet_cid == 0 ? $cid : $wallet_cid;

    my $domain = get_one_field_sql(PPC(cid => $cid_to_search), ["SELECT restricted_domain FROM camp_promocodes", WHERE => {cid => $cid_to_search}]);

    if ($domain) {
        push @result_domains, $domain;
        # считаем, что если домен с www, то мы уже знаем про него, что он значащий
        if ($domain !~ /^www\./) {
            my $with_www = "www.$domain";
            if (strip_www($with_www) eq $domain) {
                push @result_domains, $with_www;
            }
        }

        push @result_domains, map { Yandex::IDN::idn_to_ascii($_) } @result_domains;
    }

    return [uniq @result_domains];
}


=head2 _add_domain

    Если домен задан, добавить в список его, его гланое зеркало и проверить
    домен на то, является ли он доменом второго уровня

=cut

sub _add_domain {
    my ($domain, $storage, $mt, $has_2_level_domains, $different_domains) = @_;
    if (defined $domain) {
        $storage->{$domain} = 1;
        my $stripped = strip_www(Yandex::IDN::idn_to_ascii($domain));
        $different_domains->{$stripped} = 1;
        $storage->{$stripped} = 1;
        $$has_2_level_domains = 1 if (!$$has_2_level_domains && _is_2_level_domain($domain));

        my $mirror = $mt->get_main_mirror($domain);
        $storage->{$mirror} = 1 if $mirror;
    }
}


=head2 _is_2_level_domain($domain)

    Отбросить www, если оно есть в домене и проверить является ли домен доменом
    второго уровня или доменом третьего уровня, в котором первый и второй уровень - это
    зона хостера или региона (например, spb.ru)

=cut

sub _is_2_level_domain {
    my $domain = shift;

    # strip_www и strip_domain не умеют работать с кирилличискими доменами
    $domain = Yandex::IDN::idn_to_ascii($domain);
    $domain = strip_www($domain);
    return ($domain eq strip_domain($domain)) ? 1 : 0;
}

1;
