package Property;

=pod
    $Id$
    Работа с набором свойств, сохраняемых в базе
=cut

use strict;
use warnings;

use Time::HiRes;

use Settings;
use Yandex::DBTools;

use Carp qw/croak/;

my $MAX_PROPERTY_NAME_LENGTH = 100;

sub new {
    my ($class, $propname) = @_;
    if (!defined $propname) {
        die "Property name is not defined";
    } elsif ($propname !~ /^([a-zA-Z0-9_]{1,$MAX_PROPERTY_NAME_LENGTH})$/) {
        die "Incorrect property name";
    }
    bless {
        propname => $propname,
        last_value_ts => undef,
    };
}

=head2 my $res = get( ?$cache_time );

    получить свойство, если cache_time > 0 - позволяем использовать ранне полученные данные,
    возрастом не более $cache_time секунд
    
    NB! для корректной работы кеша объект Property должен храниться в переменной уровня пакета или объявленной как state
    код вида Property->new()->get(NN) ничего кешировать не может

=cut
sub get {
    my ($self, $cache_time) = @_;
    if ($cache_time && $self->{last_value_ts} && Time::HiRes::time - $self->{last_value_ts} < $cache_time) {
        return $self->{last_value};
    }

    if (my $prop_row = get_one_line_sql(PPCDICT, "SELECT value FROM ppc_properties WHERE name = ?", $self->{propname})) {
        $self->{last_value} = $prop_row->{value};
    } else {
        delete $self->{last_value};
    }
    $self->{last_value_ts} = Time::HiRes::time;        
    return $self->{last_value};
}

# установить свойство
sub set {
    my ($self, $value) = @_;
    do_sql(PPCDICT, "INSERT INTO ppc_properties (name, value)
                                     VALUES (?, ?)
                                     ON DUPLICATE KEY UPDATE value = values(value)",
                                     $self->{propname}, $value);
    $self->{last_value} = $value;
    $self->{last_value_ts} = Time::HiRes::time;        
}

# проверить что значение свойства не изменилось с момента последнего доступа, и в случае успеха - установить свойство
sub cas {
    my ($self, $value) = @_;
    
    croak "last value is not determined" if !defined $self->{last_value_ts};

    my $ok;
    if (!exists $self->{last_value}) {
        $ok = 0+do_sql(PPCDICT, "INSERT IGNORE INTO ppc_properties 
                                                  (name, value)
                                           VALUES (?, ?) ",
                                                  $self->{propname}, $value);
    } else {
        $ok = 0+do_sql(PPCDICT, "UPDATE ppc_properties SET value = ? 
                                                   WHERE name = ? and (value = ? OR (value IS NULL AND ? IS NULL))",
                                                         $value, $self->{propname}, $self->{last_value}, $self->{last_value});
    }
    if ($ok) {
        $self->{last_value} = $value;
        $self->{last_value_ts} = Time::HiRes::time;
    }
    return $ok;
}

# удалить свойство
sub delete {
    my ($self) = @_;
    do_sql(PPCDICT, "DELETE FROM ppc_properties WHERE name = ?", $self->{propname});
    delete $self->{last_value};
    $self->{last_value_ts} = Time::HiRes::time;
}

1;
