=head1 NAME

RBAC2::DirectChecks

=head1 DESCRIPTION

Проверки для контроллеров веб-интерфейса и АПИ
Из других мест (моделей) не стоит вызывать эти методы

=cut

package RBAC2::DirectChecks;

use Direct::Modern;

use Carp qw/carp/;
use List::MoreUtils qw/uniq any all/;

use Yandex::HashUtils;
use Yandex::ListUtils qw/xminus/;
use Yandex::Validate qw/is_valid_int is_valid_id/;
use Yandex::Balance;
use Yandex::I18n;
use Yandex::DBTools;
use Yandex::Log::Messages;
use Yandex::DBQueue;

use Settings;
use RBACElementary;
use Rbac qw/:const/;
use RbacRolePerms;
use RBACDirect;
use Primitives;
use PrimitivesIds;
use IpTools;
use EnvTools;
use HashingTools;
use Campaign::Types;
use Agency ();
use TextTools qw/get_num_array_by_str/;
use Client qw/get_client_data/;

use Reports::Offline::Postview qw//;

use API::Limits qw/has_spec_limits get_spec_limit/;

use utf8;

use base qw/Exporter/;
our @EXPORT = qw(
    rbac_login_check
    rbac_do_cmd
);


#======================================================================

=head2 Errors from rbac_do_cmd()

=cut

our %CMD_ERRORS = (
    1  => iget_noop("Нет прав для выполнения данной операции!"),
    2  => iget_noop("Ошибка проверки прав доступа"),
    3  => iget_noop("Доступ к интерфейсу работы с файлами табличных форматов временно приостановлен."), # текст ошибки шаблонизируется в bemtree.xjst DIRECT-42399, а эта строка попадает в тайтл страницы
    10 => iget_noop("У менеджера есть клиенты или агентства"),
    11 => iget_noop("Ошибка: у агентства есть клиенты"),
    12 => iget_noop("Ошибка: данный логин уже занят"),
    13 => iget_noop("Ошибка: данный логин является простым клиентом"),
    14 => iget_noop("Ошибка: данный логин является субклиентом"),
    15 => iget_noop("Ошибка: данный логин должен быть менеджером"),
    16 => iget_noop("У тимлидера есть подчиненные"),
    17 => iget_noop("Невозможно создать медиаплан в кампании, не обслуживаемой менеджером, либо без заявки на оптимизацию"),
    # код 18 уже не выдаём, но может попадаться в старых записях
    18 => iget_noop("Операция не разрешена в выбранном интерфейсе"),
    19 => iget_noop("Ошибка: логин не существует"),
    20 => iget_noop("У начальника отдела есть подчиненные"),
    21 => iget_noop("Ошибка: Перенос для кампании запрещен"),
    22 => iget_noop("Ошибка: Действие не разрешено для данного типа кампании"),
    23 => iget_noop("Ошибка: доступ к веб-интерфейсу ограничен"),
    24 => iget_noop("Менеджер является руководителем группы или отдела"),   
    25 => iget_noop("Ошибка: перенос денег недоступен."),
);

=head1 FUNCTIONS

=cut

#======================================================================

=head2 rbac_do_cmd

exec checks on cmd

=cut

=head2 %cmds 
    
    Хэш для сбора информации о проверках команд
    
    Возможные значения в %cmds описаны в DoCmd::Base
       ссылка на хеш: {
           Code      => \&rbac_cmd_allow_all, # функция для проверки
           Perm      => 'Unserv',
           PermClear => 'SystemReport',
           Role      => 'super',
       }

=cut

our %cmds;

sub rbac_do_cmd
{
    my ($rbac, $cur_step, $vars, $rbac_rights, $user_info) = @_;

    if (exists $cmds{$cur_step}) {

        my $check = $cmds{$cur_step};
        return rbac_check(  rbac => $rbac, check => $check, rights => $rbac_rights, vars => $vars, 
                            comment => $cur_step, user_info => $user_info);

    } else {
        warn "RBAC2::DirectChecks::rbac_cmd_* not found for: $cur_step";
        $$rbac_rights = {};
        return 0;
    }
}


=head2 rbac_check

    Проверка, разрешено ли пользователю делать что-то

    Параметры именованные 
        rbac  -- $rbac
        check -- ссылка на хеш с условиями, при выполнении которых действие разрешено. Подробное описание см. в DoCmd::Base, атрибут :Rbac
        rights -- ссылка на $rbac_rights
        comment -- строка для отладки. Включается в сообщение при die
        vars  -- ссылка на хеш с данными про действие и пользователя
            {
                rights => $login_rights, # обязательно!
            }

    Результат 
        0 -- порядок, можно 
        не-0 -- нельзя, код ошибки. Расшифровку кодов см. выше в %CMD_ERRORS

    В cmd_* вызов может быть (если вдруг понадобится) таким: 
        my $rbac_check_error = RBACDirect::rbac_check(
                rbac => $rbac,
                rights => \$rights,
                check => $check,
                vars => { rights => $login_rights }, 
                comment => "cmd_myCoolFeature",
            );
        error("действие не разрешено") if $rbac_check_error;

=cut

sub rbac_check
{
    my %PARAM = @_;

    my ($rbac, $check, $rbac_rights, $vars, $comment, $user_info) = @PARAM{
      qw/rbac   check   rights        vars   comment   user_info/};

    die "Rbac check is not defined".($comment ? " ($comment)" : "" ) if !grep {/^(?:Code|Perm|PermClear|Role)$/} keys %$check;

    my $res;
    # Проверяем все условия по порядку
    if (exists $check->{Perm}) {
        $res = rbac_check_by_rights($rbac, $vars, $check->{Perm});
        $res = 0 if $res && $check->{AllowDevelopers} && $vars->{rights}->{is_developer};
    }
    if (!$res && exists $check->{PermClear}) {
        $res = rbac_check_by_rights_clear($rbac, $vars, $check->{PermClear});
    }

    my $ClientID = $vars->{rights}->{ClientID};
    if (! $res
        && $ClientID
        && has_spec_limits($ClientID)
        && get_spec_limit($ClientID, 'web_interface_disallow')
       )
    {
        if (! exists $check->{AllowForLimitedClients}) {
            $res = 23;
        }
    }

    # если у текущего логина есть право is_developer -  то по ограничениям Role/ExceptRole/Code пропускаем
    if (!$res && exists $check->{Role}) {
        $res = rbac_check_by_role($rbac, $vars, $check->{Role});
        $res = 0 if $res && $check->{AllowDevelopers} && $vars->{rights}->{is_developer};
    }

    if (!$res && exists $check->{ExceptRole}) {
        $res = rbac_check_by_except_role($rbac, $vars, $check->{ExceptRole});
        $res = 0 if $res && $check->{AllowDevelopers} && $vars->{rights}->{is_developer};
    }

    if (!$res && exists $check->{Code}) {
        for my $rbac_code_ref (@{ $check->{Code} }) {
            $res = $rbac_code_ref->($rbac, $vars, $rbac_rights, $user_info, allow_readonly_reps => $check->{AllowReadonlyReps});
            $res = 0 if $res && $check->{AllowDevelopers} && $vars->{rights}->{is_developer};
            last if $res;
        }
    }

    if (!$res && exists $check->{CampKind}) {
        $res = rbac_check_by_camp_kind($rbac, $vars, $check->{CampKind});
    }

    return $res;
}

#======================================================================

=head2 rbac_check_by_rights

 get access by rights
     0 -- allow
     1 -- deny

=cut

sub rbac_check_by_rights
{
    my ($rbac, $vars, $perm) = @_;

    if (! $vars->{rights}{media_control}) {
        my $owner_res = rbac_cmd_by_owners($rbac, $vars);
        return $owner_res if $owner_res;
    }

    return ! $vars->{rights}{$perm};
}

sub rbac_check_by_rights_clear {
    my ($rbac, $vars, $perm) = @_;

    return ! $vars->{rights}{$perm};
}

sub rbac_check_by_role {
    my ($rbac, $vars, $role) = @_;

    return ! grep {$vars->{rights}->{role} eq $_} ref($role) eq 'ARRAY' ? @$role : ($role);
}

sub rbac_check_by_except_role {
    my ($rbac, $vars, $role) = @_;

    my %except_roles = map {$_ => 1} (ref($role) eq 'ARRAY' ? @$role : ($role));
    my $result = $except_roles{$vars->{rights}->{role}} ? 1 : 0;

    # псевдо-роль суперменеджера, если есть отдельный запрет на неё
    if ($except_roles{"super_manager"} && $vars->{rights}->{is_super_manager}) {
        $result = 1;
    }

    return $result;
}


=head2 rbac_check_by_camp_kind

   проверяем доступно ли пользователю выполнять действие с кампанией определенного типа
   права выставляються в атрибуте контролера
   :Rbac(..., CampKind => {web_edit => 1, web_view_super => [super]})

=cut

sub rbac_check_by_camp_kind
{
    my ($rbac, $vars, $kind_desc) = @_;

    return 0 unless $vars->{cid} && keys $kind_desc;

    my $cids = ref($vars->{cid}) eq 'ARRAY' ? $vars->{cid} : [split /\s*,\s*/, $vars->{cid}];

    my $role = $vars->{rights}->{role};

    for my $kind (sort keys %$kind_desc) {
        if (camp_kind_in(cid => $cids, $kind)) {
            my $rule = $kind_desc->{$kind};
            if (!ref $rule) {
                return $rule ? 0 : 22;
            } else {
                return 0 if grep {$_ eq $role} @$rule;
            }
        }
    }
    return 22;
}

#======================================================================

=head2 rbac_cmd_by_owners

   return 0 if UID owner of uid or cid (in $vars)
   внимание! суперридеры и медиапланеры являются владельцами всех кампаний/клиентов, на модифицирующих контролерах нужно добавлять Rbac => (ExceptRole => [superreader, media])

=cut

sub rbac_cmd_by_owners
{
    my ($rbac, $vars, $rights, $user_info, %opts) = @_;

    return 1 if $user_info->{is_readonly_rep} && !$opts{allow_readonly_reps}; # обрабатываем readonly-представителя

    # небольшой "rbac hack": суперам даем право на uid в любом случае (надо в тех случаях, когда пользователь еще не заведен в rbac'е и rbac_is_owner не срабатывает)
    if ($vars->{uid} 
            && $vars->{UID} != $vars->{uid} 
            && ! $vars->{rights}{super_control} 
            && ! ($vars->{cmd} && $vars->{cmd} eq 'showCamps' && $vars->{rights}{support_control})
            && ! ($vars->{cmd} && $vars->{cmd} eq 'showManagerMyClients' && $vars->{rights}{placer_control})
            && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid})) {
        return 1;
    }

    # проверяем право оператора на массив uid'ов (используется в API)
    if ($vars->{uids}) {
        return 1 unless $vars->{rights}{super_control}  
                            || rbac_mass_is_owner($rbac, $vars->{UID}, $vars->{uids});
    }

    my @cid_list;
    my @cid_list_media;

    if ($vars->{cid}) {
        if (ref($vars->{cid}) eq 'ARRAY') {
            push @cid_list, @{ $vars->{cid} }; # already splitted in DoCmd::FormCheck::check_required_params()
        } elsif ($vars->{cid} =~ /,/) {
            push @cid_list, uniq split /\s*,\s*/, $vars->{cid};
        } else {
            push @cid_list, $vars->{cid};
        }
    }

    if ( $vars->{bid} ) {

        my @bids;
        if (ref($vars->{bid}) eq 'ARRAY') {
            @bids = @{ $vars->{bid} };
        } else {
            @bids = uniq split /\s*,\s*/, $vars->{bid};
        }

        if ( defined $vars->{media} && $vars->{media} == 1 ) {
            # для медиаплановских баннеров рассчитываем что cid всегда приходит вместе с bid
            my $mp_cids = get_cids(mediaplan_bid => \@bids);
            return 1 if @{ xminus($mp_cids, \@cid_list) };
        } else {
            push @cid_list, @{get_cids(bid => \@bids)};
        }
    }

    # adgroup_id[s] используется для передачи параметров с клиента вместо pid[s]
    # нужно оставить только adgroup_id[s]  
    if ($vars->{pid} || $vars->{adgroup_ids} || $vars->{adgroup_id}) {
        my @pids;
        foreach my $field (qw/pid adgroup_ids adgroup_id/) {
            next unless $vars->{$field};
            push @pids, ref $vars->{$field} eq 'ARRAY'
                            ? @{$vars->{$field}} : uniq split /\s*,\s*/, $vars->{$field};
        }
        push @cid_list, @{get_cids(pid => \@pids)};
    }

    if ( $vars->{mbid} ) {
        push @cid_list_media, @{get_cids(media_bid => $vars->{mbid})};
    }

    if ( $vars->{mgid} ) {
        push @cid_list_media, @{get_cids(media_gid => $vars->{mgid})};
    }

    if ( $vars->{vcard_id} ) {
        push @cid_list, @{get_cids(vcard_id => $vars->{vcard_id})};
    }

    @cid_list = uniq grep {defined $_} @cid_list, @cid_list_media;

    # нельзя одновременно менять mbid/mgid из разных kампаний (с) gerich@
    if (@cid_list_media && 1 < scalar @cid_list) {
        return 2;
    }

    return 1 if @cid_list && ! rbac_is_owner_of_camps($rbac, $vars->{uid}, [@cid_list]);

    return 0;
}

#======================================================================

=head2 rbac_cmd_by_editing_owners

   вызывать для проверок как rbac_cmd_by_owners для контроллеров, которые редактируют
   данные (потому что у суперридера, например, owner = 1, а доступа на редактирование нет)

=cut

sub rbac_cmd_by_editing_owners {
    my ($rbac, $vars, $user_rights, $user_info, %opts) = @_;

    if ( my $status = rbac_cmd_by_owners($rbac, $vars, $user_rights, $user_info, %opts) ) {
        return $status;
    }

    my $rights = $vars->{rights};
    if ( $rights->{superreader_control} || $rights->{media_control} ) {
        return 1;
    }

    if ( $rights->{internal_ad_manager_control} ) {
        my $operator_client_id = $rights->{ClientID};
        my $user_perminfo = Rbac::get_perminfo( uid => $vars->{uid} );

        if (!Rbac::has_perm( $user_perminfo, 'internal_ad_product' )) {
            return 1;
        }

        my $user_client_id = $user_perminfo->{ClientID};
        my $access_types_by_client_id = InternalAdManagers::filter_accessible_client_ids( $operator_client_id, [$user_client_id] );
        unless ( exists $access_types_by_client_id->{$user_client_id} &&
            $access_types_by_client_id->{$user_client_id} eq 'internal_ad_publisher')
        {
            return 1;
        }
    }

    if ( $rights->{internal_ad_superreader_control} ) {
        return 1;
    }

    return 0;
}

#======================================================================

=head2 rbac_cmd_super_and_super_media

   rights for reseting first aid responses

=cut

sub rbac_cmd_super_and_super_media {
    my ($rbac, $vars) = @_;
    return not scalar grep { $vars->{rights}{$_} } qw/is_super_media_planner super_control/;
}

#======================================================================

=head2 rbac_cmd_allow_all

   allow all

=cut

sub rbac_cmd_allow_all
{
    return 0;
}

#======================================================================

=head2 rbac_cmd_forbid_all

   forbid all

=cut

sub rbac_cmd_forbid_all
{
    return 3;
}

#======================================================================

=head2 rbac_login_check

 login check on RBAC

    $vars = {
        UID =>,
        user_info => {
            is_developer, is_super_manager, login, passport_karma
        },
        ClientID =>,
        is_internal_ip => is_internal_ip($ENV{REMOTE_ADDR}),
    };
    $rbac_error = rbac_login_check($rbac, $vars, \$rbac_login_rights);

=cut

sub rbac_login_check
{
    my ($rbac, $vars, $result) = @_;
    my $res;

    my $perminfo = Rbac::get_perminfo(uid => $vars->{UID}) || {};
    my $role_hash = rbac_who_is_detailed(undef, $vars->{UID}) || {};

    my $role = $role_hash->{role};

    # from internal network only
    if (rbac_is_internal_user(undef, role => $role) && ! $vars->{is_internal_ip}) {
        warn "internal network vialotion for $role: $ENV{REMOTE_ADDR}";
        return 3;
    }

    if ($role eq 'empty' && is_balance_subclient($vars->{UID})) {
        $role = 'client';
    }

    $res->{super_control}     = 1 if $role eq 'super';
    $res->{placer_control}    = 1 if $role eq 'placer';
    $res->{agency_control}    = 1 if $role eq 'agency';
    $res->{manager_control}   = 1 if $role eq 'manager';
    $res->{media_control}     = 1 if $role eq 'media';
    $res->{support_control}   = 1 if $role eq 'support';
    $res->{limited_support_control} = 1 if $role eq 'limited_support';
    $res->{superreader_control} = 1 if $role eq 'superreader';
    $res->{freelancer_control} = 1 if RBACDirect::is_freelancer($vars->{UID});
    $res->{internal_ad_admin_control} = 1 if $role eq 'internal_ad_admin';
    $res->{internal_ad_manager_control} = 1 if $role eq 'internal_ad_manager';
    $res->{internal_ad_superreader_control} = 1 if $role eq 'internal_ad_superreader';

    $res->{is_developer} = 1 if $role eq 'superreader'
                                && $vars->{user_info}
                                && $vars->{user_info}->{is_developer}
                                && $vars->{user_info}->{is_developer} eq 'Yes';
    $res->{is_devops} = 1 if $res->{is_developer} && exists $Settings::DEVOPS_LOGINS{$vars->{user_info}->{login}};

    $res->{is_super_manager} = 1 if $role eq 'superreader'
                                    && $vars->{user_info}
                                    && $vars->{user_info}->{is_super_manager};

    $res->{is_freelancer} = 1 if $res->{freelancer_control};

    $res->{mcc_client_ids} = { map { $_ => 1 } @{$perminfo->{mcc_client_ids}} } if @{$perminfo->{mcc_client_ids} // []};
    $res->{is_mcc} = 1 if $res->{mcc_client_ids};

    # есть ли у клиента агентства, аналог бывшего *subclient_control (пока не открыли свободу полностью)
    $res->{client_have_agency} = $role eq 'client' && $perminfo->{agency_client_id} ? 1 : 0;
    if ($res->{client_have_agency}) {
        # клиент имеет агентские кампании, но без прав на редактирование
        $res->{dont_show_new_camp_link_in_menu} = Rbac::has_perm($perminfo, $Rbac::PERM_SUPER_SUBCLIENT) || get_allow_create_scamp_by_subclient($vars->{UID}) ? 0 : 1;
    }

    $res->{role} = $role;
    $res->{ClientID} = $vars->{ClientID};

    # Позволено ли пользователю платить деньги.
    $res->{user_allow_pay}    = 1 if $role_hash->{is_any_client} || $res->{agency_control}
        || $res->{super_control} || $res->{manager_control} || $res->{support_control} || $res->{limited_support_control};

    # Заблокирован ли пользователь по спам карме в XLS и API
    # Блокируем только несервесируемых самоходов без агентстких кампаний
    $res->{can_block_karma} = $role =~ /client/ && ! $res->{client_have_agency} 
                              && ! scalar @{rbac_get_managers_of_client(undef, $vars->{UID})} ? 1 : 0;
    $res->{passport_karma_block} = $res->{can_block_karma} 
                                   && defined $vars->{user_info}->{passport_karma}
                                   && $vars->{user_info}->{passport_karma} > $Settings::KARMA_API_AUTOBLOCK
                                   ? 1 : 0;

    # another system rights
    eval {
        hash_merge $res, rbac_system_client_rights($rbac, $vars->{UID}, $vars->{ClientID});
    };
    if ($@) {
        carp "rbac error: $@";
        return 2;
    }

    hash_merge $res, $role_hash;
    
    $$result = $res;
    return 0;
}


=head2 rbac_system_client_rights

    Получение системных прав и прав на ClientID из старого rbac

=cut
sub rbac_system_client_rights {
    my ($rbac, $UID, $ClientID) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $UID);

    my $ret = RbacRolePerms::system_perms($perminfo);
    if ($ClientID) {
        my $client_perminfo = Rbac::get_perminfo(ClientID => $ClientID);
        hash_merge $ret, RbacRolePerms::onclient_perms($UID, $perminfo, $client_perminfo);
    }

    return $ret;
}


#======================================================================
#======================================================================
#======================================================================

=head1 rbac_cmd_ ... - callback functions for each cmd_'s (return RBAC rights)

 return:
  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

#======================================================================

=head2 rbac_cmd_showCamp

=cut

sub rbac_cmd_showCamp
{
    # RBAC_SubClientAccess - redirect to statistic
    # RBAC_ManageNSCampaign - old rule
    my (undef, $vars, $result) = @_;
    my $rights = [];
    my $errcode;

    my $cid;
    if (defined $vars->{bid} && $vars->{bid} =~ /^\d+$/) {
        if( defined $vars->{media} && $vars->{media} == 1 ) {
            # для медиаплановских баннеров рассчитываем что cid всегда приходит вместе с bid
            $cid = $vars->{cid};
            return 1 unless get_cid('mediaplan_bid', $vars->{bid}) == $cid;
        } else {
            $cid = get_cid(bid => $vars->{bid});
        }

        # unless bid in cid
        if ( defined $vars->{cid} && $vars->{cid} =~ /^\d+$/ && $vars->{cid} != $cid ) {
            return 1;
        }

    } elsif (defined $vars->{bid} && $vars->{bid} =~ /^\d+(,\d+)*$/ && defined $vars->{cid} && $vars->{cid} =~ /^\d+$/ ) { # multi-bids cmds
        my $old_cid = $vars->{cid};
        for my $bid (split /,/, $vars->{bid}) {
            $cid = get_cid(bid => $bid);
            return 1 if defined $old_cid && $old_cid != $cid;
            $old_cid = $cid;
        }
    } elsif (defined $vars->{cid} && $vars->{cid} =~ /^\d+$/) {
        $cid = $vars->{cid};
    } else {
        return 1;
    }

    if (defined $vars->{rights}->{role} && $vars->{rights}->{role} eq 'media') {
        # Нехорошая проверка. Подумать, как сделать правильнее!!!
        return 1 if defined $vars->{cmd} &&
                    $vars->{cmd} =~ /^(delBanner|
                                       stopCamp|
                                       resumeCamp|
                                       editBannerPrice|
                                       saveBannerPrice|
                                       updPrices|
                                       saveCampAutobudget|
                                       updateBannerPrice|
                                       delBanner|
                                       stopBanner|
                                       resumeBanner|
                                       archiveBanner|
                                       unarchiveBanner)$/x;

        return 1 if rbac_who_is(undef, $vars->{UID}) =~ m/((super)?sub)?client/ &&
                    defined $vars->{cmd} &&
                    is_media_camp(cid => $vars->{cid}) &&
                    $vars->{cmd} =~ /^(stopCamp|
                                       resumeCamp)$/x;

        if (RBACDirect::is_user_allowed_for_action_on_camps('show', $vars->{UID}, [$cid])->{$cid}) {
            return 0;
        } else {
            return 1;
        }
    }

    my $perm = $vars->{cmd} =~ /showCamp/ ? 'show' : 'edit';
    if (RBACDirect::is_user_allowed_for_action_on_camps($perm, $vars->{UID}, [$cid])->{$cid}) {
        return 0;
    }
    return 1;
}


sub rbac_cmd_showCompetitors 
{
    my ($rbac, $vars) = @_;
    # со временем нужно переделать вёрстку на получение параметра cid вместо nocid - тогда этот код можно выпилить
    local $vars->{cid} = $vars->{cid} || $vars->{nocid};
    return rbac_cmd_by_owners($rbac, $vars);
}

#--------------------------------------------------------------
# утверждение медиаплана - только своим менеджером, тимлидером этого менеджера или супервизором или агентством
#
sub rbac_cmd_acceptMediaplan
{
    my ($rbac, $vars, $result) = @_;
    my ( $cid, $to_cid, $client_uid ) = (0, 0, 0);

    if ( $vars->{cid} && $vars->{cid} =~ /^\d+$/ ) {
        $cid = $vars->{cid};
    } else {
        return 1;
    }

    if( $vars->{to_camp} eq 'current' ) {
        $to_cid = $cid;
    } elsif ( $vars->{to_camp} && $vars->{to_camp}  eq 'other'
                # && $vars->{client_login} && validate_login( $vars->{client_login} )
                && $vars->{to_cid} && $vars->{to_cid} =~ /^\d+$/ ) {
        # $client_uid = get_uid_by_login( $vars->{client_login} );
        # return 1 if !$client_uid;
        $to_cid = $vars->{to_cid};
    } else {
        # return 1;
    }

    if (($vars->{rights}->{ModifyMediaplan} || $vars->{rights}->{is_any_client} || $vars->{rights}{agency_control}) && rbac_is_owner_of_camp($rbac, $vars->{UID}, $cid)) {
        return 0;
    }
    return 2;
}

#--------------------------------------------------------------
# создание медиаплана/редактирвоание
#

=c

Можно создавать медиаплан - суперпользователю, процессингу
Менеджеру в своей кампании.
Медиапланеру - в любой сервисируемой кампании, либо кампании на которую есть заявка на оптимизацию.

=cut

sub rbac_cmd_createMediaplan
{
    my ($rbac, $vars, $result) = @_;
    my $cid;
    if (defined $vars->{cid} && $vars->{cid} =~ /^\d+$/) {
        $cid = $vars->{cid};
    } else {
        return 1;
    }

    # проверяем наличие заявки на оптимизацию кампании
    my $optimize_camp_flag = get_one_field_sql(PPC(cid => $cid), ["select count(*) from optimizing_campaign_requests",
                                                                  where => {cid => $cid, status__not_in => [qw/Accepted AcceptDeclined/]}]);
    if( $vars->{rights}->{ModifyMediaplan}
        && rbac_is_owner_of_camp( $rbac, $vars->{UID}, $vars->{cid} )
        || $vars->{rights}->{super_control}) {
        
        if (rbac_is_scampaign($rbac, $vars->{cid})
            || rbac_is_agencycampaign($rbac, $vars->{cid})
            || $optimize_camp_flag
            || $vars->{rights}->{super_control}
            || $vars->{rights}->{media_control}
        ) {
            return 0;
        } else {
            return 17;
        }
    }
    return 2;
}

#--------------------------------------------------------------
# просмотр медиаплана, а также согласие/несогласие с фразами
#
sub rbac_cmd_showMediaplan
{
    my ($rbac, $vars, $result) = @_;
    my $rights = [];

    # если не задан, либо задан неправильно номер кампании
    return 1 unless defined $vars->{cid} && $vars->{cid} =~ /^\d+$/;
    
    # если есть право на просмотр медиаплана и есть права на кампанию
    return 0 if rbac_is_owner_of_camp( $rbac, $vars->{UID}, $vars->{cid} );

    return 2;
}

#======================================================================

=head2 rbac_cmd_showCampStat

=cut

sub rbac_cmd_showCampStat
{
    my ($rbac, $vars, $result) = @_;
    my $rights = [];
    my $errcode;

    return 1 if $vars->{uid} && $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    # отчет по всем агентствам доступен только внутренним ролям
    return 1 if $vars->{stat_type}
                && $vars->{stat_type} eq 'by_agencies'
                && ! ($vars->{rights}{manager_control}
                      || $vars->{rights}{support_control}
                      || $vars->{rights}{super_control}
                      || $vars->{rights}->{superreader_control}
                     );

    # by_clients|by_managers
    if (defined $vars->{stat_type} && $vars->{stat_type} =~ /^by_clients|by_managers|by_agency_clients$/) {

        return 0 if ($vars->{rights}{manager_control} || $vars->{rights}{agency_control}) && $vars->{stat_type} eq 'by_agency_clients';
        return 1 if ! ($vars->{rights}{manager_control} || $vars->{rights}{support_control} || $vars->{rights}{super_control} || $vars->{rights}{superreader_control});
        return 1 if $vars->{stat_type} eq 'by_managers' && ! $vars->{rights}{is_teamleader} && ! $vars->{rights}{is_superteamleader} && ! $vars->{rights}{super_control} && ! $vars->{rights}{support_control};
        return 1 if $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});
    }

    if ($vars->{cid}) {
        my @cids = uniq split /\s*,\s*/, $vars->{cid};
        return 2 if grep {! is_valid_id($_)} @cids;
        my $can_show_stat = RBACDirect::rbac_check_allow_show_stat_camps($rbac, $vars->{UID}, \@cids);
        return 1 if grep {! $can_show_stat->{$_}} @cids;
        return 0;
    } elsif ( !$vars->{cid} ) {
        # если не указан cid, позже будем получать статистику только для разрешённых кампаний
        return 0;
    }
}

#======================================================================

=head2 rbac_cmd_showStat

=cut

sub rbac_cmd_showStat
{
    my ($rbac, $vars, $result) = @_;
    my $rights = [];
    my $errcode;

    return 1 if $vars->{uid} && $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    if ($vars->{single_camp} && $vars->{cid}) {
        return 2 if ! is_valid_id($vars->{cid});
        my $can_show_stat = RBACDirect::rbac_check_allow_show_stat_camps($rbac, $vars->{UID}, [$vars->{cid}]);
        return 1 if ! $can_show_stat->{$vars->{cid}};
        return rbac_check_by_camp_kind($rbac, $vars, { stat_stream_ext => 1});
    } else {
        # если не указан cid, позже будем получать статистику только для разрешённых кампаний
        return 0;
    }
}

#======================================================================

=head2 rbac_cmd_pdf_report

=cut

sub rbac_cmd_pdf_report
{
    my ($rbac, $vars, $result) = @_;
    my $rights = [];
    my $errcode;

    my @camps;
    my $cid = $vars->{cid} || $vars->{cids};
    if ( defined $cid && $cid =~ /^\D*(?:(\d+)\D*)+$/ ) {
        @camps = grep { $_ =~ /^\d+$/} split /\D+/, $cid;
    } elsif ( !$cid and $vars->{cmd} =~ /^newPdfReport$/ ) {
        # если не указан cid, позже будем получать статистику только для разрешённых кампаний
        return $vars->{rights}{ViewPdfReports} || $vars->{rights}->{media_control} ? 0 : 1;
    } elsif ( $vars->{cmd} =~ /^addPdfReport$/ ) {
        @camps = grep { defined $_ } map { (defined $_ and m/^cid_(\d+)$/) ? $1 : undef } keys %$vars;
    } elsif ( defined $vars->{rid} and $vars->{rid} =~ /^\d+$/) {
        if ( $vars->{type} eq 'offline' ) {
            @camps = ( get_one_field_sql(PPCDICT, "SELECT cid FROM xls_reports WHERE id=?", $vars->{rid}) // () );
        } elsif ($vars->{type} eq 'pdf') {
            @camps = split /\D+/, (get_one_field_sql(PPC(uid => $vars->{uid}), 
                    "SELECT cids FROM pdf_reports WHERE id=? and uid = ?", $vars->{rid}||'', $vars->{uid}));
        } elsif ($vars->{type} eq 'video') {
            my $job = Yandex::DBQueue->new(PPC(uid => $vars->{uid}), 'video_stat_report')
                ->find_jobs(ClientID => get_clientid(uid => $vars->{uid}), job_id => [$vars->{rid}])->[0];
            @camps = @{$job->args->{cids}};
        } elsif ($vars->{type} eq 'dynamic') {
            @camps = @{ Reports::Offline::Dynamic->new()
                ->get_report_options($vars->{uid}, $vars->{rid})->{CampaignIds} // [] };
        } elsif ($vars->{type} eq 'performance') {
            @camps = @{ Reports::Offline::Performance->new()
                ->get_report_options($vars->{uid}, $vars->{rid})->{CampaignIds} // [] };
        } elsif ($vars->{type} eq $Reports::Offline::Postview::POSTVIEW_REPORT_TYPE) {
            @camps = @{Reports::Offline::Postview->new()->get_report_options($vars->{uid}, $vars->{rid})->{cids} // [] };
        } else {
            die "unknown report type '$vars->{type}'";
        }
    }


    $errcode = rbac_cmd_showCampStat( $rbac, {UID => $vars->{UID}, cid => join(',', @camps)} );
    if ( $errcode ) {
        return 2;
    }

    #Закрываем addPdfReport для суперридеров: https://st.yandex-team.ru/DIRECT-104769
    if ($vars->{cmd} =~ /^addPdfReport$/) {
        return 2 if $vars->{rights}->{superreader_control};
    }

    return $vars->{rights}{ViewPdfReports}
    		|| $vars->{rights}->{placer_control}
    		|| $vars->{rights}->{media_control} && ($vars->{cmd} =~ /^(loadPdfReport|listOfflineReports)$/ || 
                                                        $vars->{cmd} =~ /^(listPdfReport|newPdfReport|addPdfReport|delPdfReport)$/ && 
                                                        all { rbac_is_scampaign($rbac, $_) || rbac_is_agencycampaign($rbac, $_) } @camps  ) ? 0 : 1;
}

#======================================================================

=head2 rbac_cmd_createTeamleader

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

sub rbac_cmd_createTeamleader
{
    my ($rbac, $vars, $result) = @_;

    return 19 unless $vars->{'login'};
    my $uid = get_uid_by_login($vars->{'login'});
    return 19 unless $uid;

    if ($uid && ($vars->{rights}{UsersAdministration})) {
        return 15 if rbac_who_is($rbac, $uid) !~ /^manager$/;
        return 15 if rbac_is_superteamleader($rbac, $uid) || rbac_is_teamleader($rbac, $uid);
        return 13 if get_one_field_sql(PPC(uid => $uid), "select count(*) from campaigns where uid = ? and statusEmpty = 'No'", $uid) > 0;
        return 13 if is_balance_subclient($uid);
        return 0;
    } else {
        return 2;
    }
}

#======================================================================

=head2 rbac_cmd_showClients

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

sub rbac_cmd_showClients
{
    my ($rbac, $vars, $result) = @_;

    return 1 unless $vars->{uid};
    return 1 if rbac_who_is($rbac, $vars->{uid}) ne 'agency';
    return 0 if $vars->{rights}->{super_control} || $vars->{rights}->{support_control} || $vars->{rights}->{media_control} || $vars->{rights}->{placer_control};
    return 1 if $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});
    return 0;
}

=head2 rbac_can_massMoveToRealMoney

  0 - OK
  1 - Error

=cut

sub rbac_can_massMoveToRealMoney
{
    my ($rbac, $vars, $result) = @_;

    return 1 unless $vars->{uid};
    return 1 if rbac_who_is($rbac, $vars->{uid}) ne 'agency';

    if ($vars->{rights}->{super_control}) {
        return 0;
    } elsif ($vars->{rights}->{support_control} || $vars->{rights}->{superreader_control}) {
        # можно только смотреть, но не заказывать
        return ($vars->{move}) ? 1 : 0;
    } elsif ($vars->{rights}->{manager_control}) {
        return (rbac_is_owner($rbac, $vars->{UID}, $vars->{uid})) ? 0 : 1;
    } elsif ($vars->{rights}->{agency_control}) {
        return ($vars->{rights}->{is_agency_chief}) ? 0 : 1;
    } else {
        return 1;
    }
}

#======================================================================

=head2 rbac_cmd_acceptServicing

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

sub rbac_cmd_acceptServicing
{
    my ($rbac, $vars, $result) = @_;

    my ($errorcode, $rights);
    my $uid = $vars->{uid};
    return 1 if $vars->{rights}->{role} eq 'super';

    $errorcode = rbac_bind_manager($rbac, $vars->{UID}, $uid);
    return 2 if $errorcode;
    return 0;
}

#======================================================================

=head2 rbac_cmd_deleteUser

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

sub rbac_cmd_deleteUser
{
    my ($rbac, $vars, $result) = @_;

    return 2 unless ($vars->{role}
        && rbac_is_owner($rbac, $vars->{UID}, $vars->{uid})
        && ($vars->{rights}{UsersAdministration}
            || $vars->{rights}{is_superteamleader} && ($vars->{role} eq 'teamleader' || $vars->{role} eq 'manager')
           )
       );

    return 0;
}

#======================================================================

=head2 rbac_cmd_pay

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

sub rbac_cmd_pay
{
    my ($rbac, $vars, $result) = @_;

    return 2 if $vars->{uid} && $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    my %cids_for_check;
    if ($vars->{cid}) {
        my $cids = [uniq split /\s*,\s*/, $vars->{cid}];
        $cids_for_check{$_}++ for @$cids;
    }

    for my $cid_in_form (keys %$vars) {
        if ($cid_in_form && $cid_in_form =~ /^sum_(\d+)$/) {
            $cids_for_check{$1}++;
        }
    }

    my $UID_for_check = $vars->{UID};
    my $cids = [keys %cids_for_check];
    if ($vars->{rights}->{manager_control} && scalar(@$cids) == 1 && is_wallet_camp(cid => $cids->[0])) {
        my $servicing_info = Primitives::get_servicing_info_by_cids([$cids->[0]]);
        # для менеджеров проверяем права на общий счет через клиента (только для самоходных клиентов)
        if (exists $servicing_info->{$cids->[0]} && ! $servicing_info->{$cids->[0]}->{AgencyUID} && ! $servicing_info->{$cids->[0]}->{ManagerUID}) {
            $UID_for_check = $vars->{uid};
        }
    }

    if (%cids_for_check) {
	my $can_edit_camp = rbac_user_allow_edit_camp($rbac, $UID_for_check, $cids);
	my $can_transfer_money = rbac_check_allow_transfer_money_camps(undef, $UID_for_check, $cids);
        return 2 unless $can_edit_camp || all { $can_transfer_money->{ $_ } } @$cids;
    }

    return 2 if $vars->{rights}->{agency_limited_no_pay};

    return 0;
}

#======================================================================

=head2 rbac_cmd_pay_api

  Отличается от rbac_cmd_pay тем, что в АПИ субклиент должен иметь право на редактирование кампаний,
  чтобы оплачивать их.

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut

sub rbac_cmd_pay_api
{
    my ($rbac, $vars, $result) = @_;

    return 2 if $vars->{uid} && $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    my %cids_for_check;
    if ($vars->{cid}) {
        my $cids = [uniq split /\s*,\s*/, $vars->{cid}];
        $cids_for_check{$_}++ for @$cids;
    }

    for my $cid_in_form (keys %$vars) {
        if ($cid_in_form && $cid_in_form =~ /^sum_(\d+)$/) {
            $cids_for_check{$1}++;
        }
    }

    my $UID_for_check = $vars->{UID};
    my $cids = [keys %cids_for_check];
    if ($vars->{rights}->{manager_control} && scalar(@$cids) == 1 && is_wallet_camp(cid => $cids->[0])) {
        my $servicing_info = Primitives::get_servicing_info_by_cids([$cids->[0]]);
        # для менеджеров проверяем права на общий счет через клиента (только для самоходных клиентов)
        if (exists $servicing_info->{$cids->[0]} && ! $servicing_info->{$cids->[0]}->{AgencyUID} && ! $servicing_info->{$cids->[0]}->{ManagerUID}) {
            $UID_for_check = $vars->{uid};
        }
    }

    if (%cids_for_check) {
        return 2 unless rbac_user_allow_edit_camp($rbac, $UID_for_check, $cids);
    }

    return 2 if $vars->{rights}->{agency_limited_no_pay};

    return 0;
}


#======================================================================

=head2 rbac_cmd_direct_payment_instructions

  Проверяет, что пользователь имеет право смотреть инструкцию по оплате директовской кампании наличными через терминал.
  За исключением проверки сумм повторяет rbac_cmd_pay.

  0 - OK
  2 - Ошибка проверки прав доступа

=cut

sub rbac_cmd_direct_payment_instructions
{
    my ($rbac, $vars, $result) = @_;

    # на агентские кампании прямой платёж недоступен
    return 2 if $vars->{rights}{role} eq 'client' && rbac_is_agencycampaign($rbac, $vars->{cid});

    return 2 if $vars->{uid} && $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    if ($vars->{cid}) {
        my $cids = $vars->{cid};
        my $UID_for_check = $vars->{UID};
        if ($vars->{rights}->{manager_control} && scalar(@$cids) == 1 && is_wallet_camp(cid => $cids->[0])) {
            my $servicing_info = Primitives::get_servicing_info_by_cids([$cids->[0]]);
            # для менеджеров проверяем права на общий счет через клиента (только для самоходных клиентов)
            if (exists $servicing_info->{$cids->[0]} && ! $servicing_info->{$cids->[0]}->{AgencyUID} && ! $servicing_info->{$cids->[0]}->{ManagerUID}) {
                $UID_for_check = $vars->{uid};
            }
        }
        return 2 if ! rbac_user_allow_edit_camp($rbac, $UID_for_check, $cids);

        # агентские кампании нельзя оплачивать наличными
        return 2 if rbac_is_agencycampaign($rbac, $vars->{cid});
    }

    return 0;
}

#======================================================================

=head2 rbac_cmd_for_chief_agency

    Действия над предстваителями агентств:
      addAgRep: добавление представителя агентству
      deleteAgRep: удаление представителя агентства
      switchAgChief: переключение главного представиттеля у агентства
      manageClientsOfAgency, ajaxMoveClientToAgency: назначение клиентов ограниченному представителю агентства
      moveClientToMain: перенос клиента от ограниченного представителя агентства к главному

    1. позволенно только представителям-шефам агентств, менеджерам, (супер)тимлидерам, суперам
    2. всем тимлидерам и менеджерам не позволенно manageClientsOfAgency, ajaxMoveClientToAgency, moveClientToMain
    3. менеджерам не позволенно addAgRep, deleteAgRep, switchAgChief (могут только смотреть)

=cut

sub rbac_cmd_for_chief_agency
{
    my ($rbac, $vars, $result) = @_;

    return 1 unless rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    my $is_simple_manager = $vars->{rights}->{manager_control} && ! $vars->{rights}->{is_any_teamleader};

    return 1 if $is_simple_manager && $vars->{cmd} =~ /^(?:deleteAgRep|switchAgChief)$/;
    return 1 if $vars->{rights}->{manager_control} && $vars->{cmd} =~ /^(?:manageClientsOfAgency|ajaxMoveClientToAgency|moveClientToMain|ajaxSwapAgLimRepChiefs)$/;
    if (Client::ClientFeatures::has_new_lim_rep_schema_feature($vars->{rights}{ClientID})) {
        return 0 if $vars->{rights}->{is_agency_main} && $vars->{cmd} =~ /^(?:addAgRep|deleteAgRep|manageClientsOfAgency|ajaxMoveClientToAgency|ajaxSwapAgLimRepChiefs)$/;
        return 0 if $vars->{rights}->{is_agency_chief_lim_rep} && $vars->{cmd} =~ /^(?:manageClientsOfAgency|ajaxMoveClientToAgency)$/;
    }
    return 1 unless $vars->{rights}->{is_agency_chief}
                    || $vars->{rights}->{manager_control}
                    || $vars->{rights}->{super_control};

    return 0;
}

#======================================================================

=head2 rbac_cmd_for_chief_or_main_agency

=cut

sub rbac_cmd_for_chief_or_main_agency
{
    my ($rbac, $vars, $result) = @_;

    return 1 unless rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    return 0 if $vars->{rights}->{is_agency_chief}
             || $vars->{rights}->{is_agency_main}
             || $vars->{rights}->{manager_control}
             || $vars->{rights}->{superreader_control}
             || $vars->{rights}->{super_control}
             || $vars->{rights}->{support_control}
             || $vars->{rights}->{limited_support_control};

    return 1;
}

#======================================================================

=head2 rbac_cmd_for_chief_client

=cut

sub rbac_cmd_for_chief_client
{
    my ($rbac, $vars, $result) = @_;

    # "чистым" логинам "для себя" разрешаем
    return 0 if $vars->{UID} == $vars->{uid} && $vars->{rights}->{role} eq 'empty';

    return 1 unless rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});

    # менеджерам разрешаем только добавлять, переключать, удалять и смотреть
    return 0 if $vars->{rights}->{manager_control} && $vars->{cmd} =~ /^(?:addClRep|showClReps|deleteClRep|addClientToMCC|deleteClientFromMCC)$/;

    my $client_perminfo = Rbac::get_perminfo(uid => $vars->{uid});
    return 1 if $client_perminfo && $vars->{rights}{is_mcc} && $vars->{rights}{mcc_client_ids}{$client_perminfo->{ClientID}};

    #фрилансерам не разрешаем добавлять/удалять представителей и привязывать/отвязывать МСС
    return 1 if $vars->{rights}->{is_freelancer} &&  $vars->{UID} != $vars->{uid} && $vars->{cmd} =~ /^(?:addClRep|deleteClRep|switchClChief|addClientToMCC|deleteClientFromMCC)$/;

    # тимлидерам разрешаем переключать
    return 0 if $vars->{rights}->{is_any_teamleader} && $vars->{cmd} =~ /^switchClChief$/;

    # представителю агентства разрешаем привязывать/отвязывать MCC
    return 0 if $vars->{rights}->{agency_control} && $vars->{cmd} =~ /^addClientToMCC|deleteClientFromMCC$/;

    # limited_support разрешаем смотреть
    return 0 if $vars->{rights}->{limited_support_control} && $vars->{cmd} =~ /^showClReps$/;

    # разрешаем всё шефам клиента, суперам и менеджерам только смотреть
    return 0 if $vars->{rights}->{is_client_chief}
             || $vars->{rights}->{super_control}
             || $vars->{rights}->{support_control}
             || $vars->{rights}->{superreader_control};

    return 1;
}

#======================================================================
#
=head2 rbac_cmd_report_unchecked_context

  0 - OK
  1 - System Error
  2 - No rights for objects

=cut

sub rbac_cmd_report_unchecked_context
{
    my ($rbac, $vars, $result) = @_;
    
    unless ($vars->{rights}->{ShowReportContext}) {
        return 1;
    } else {
        
        my $no_right = 0;
        if ($vars->{cid} && !rbac_is_owner_of_camp($rbac, $vars->{UID}, $vars->{cid})) {
            $no_right = 1;
        }
        
        # проверяем права на указанные агентства
        unless ($no_right) {
            if ($vars->{agency_uid}) {
                foreach my $uid (grep {/^(\d+)$/} split /,\s*/, $vars->{agency_uid}) {
                    next if $uid && rbac_is_owner($rbac, $vars->{UID}, $uid);
                    $no_right = 1;
                }
            }
        }
        
        # проверяем права на указанных менеджеров
        unless ($no_right) {
            if ($vars->{manager_uid}) {
                foreach my $uid (grep {/^(\d+)$/} split /,\s*/, $vars->{manager_uid}) {
                    next if $uid && rbac_is_owner($rbac, $vars->{UID}, $uid);
                    $no_right = 1;
                }
            }
        }
        
        # проверяем права на указанных клиентов
        unless ($no_right) {
            my @uids = ();
            if ($vars->{login}) {
                push @uids, @{get_uids(login => [split /,\s*/, $vars->{login}])}
            }
            
            if ($vars->{client_uid}) {
                push @uids, grep {/^(\d+)$/} split(/,\s*/, $vars->{client_uid});
            }
            
            foreach my $uid (@uids) {                        
                next if $uid && rbac_is_owner($rbac, $vars->{UID}, $uid);
                $no_right = 1;
            }
        }
        
        return $no_right ? 2 : 0;
    }
}

#======================================================================

=head2 rbac_userAllowEditCampsOrSuperReader

=cut

sub rbac_userAllowEditCampsOrSuperReader
{
    my ($rbac, $vars, $result) = @_;
    
    #Открываем showCampMultiEdit для суперридеров: https://st.yandex-team.ru/DIRECT-104769
    return 0 if $vars->{rights}->{superreader_control};
    return 0 if $vars->{rights}->{limited_support_control};

    return rbac_cmd_user_allow_edit_camps($rbac, $vars, $result);
}

=head2 rbac_cmd_searchDoubles

    0 - OK
    1 - forbidden
=cut

sub rbac_cmd_searchDoubles {
    my ($rbac, $vars, $result) = @_;
    
    return 0 if $vars->{rights}->{limited_support_control};
    return rbac_cmd_internal_user_only($rbac, $vars, $result);
}

#======================================================================

=head2 rbac_cmd_modifyUser

  for cmd_modifyUser

=cut

sub rbac_cmd_modifyUser
{
    my ($rbac, $vars, $result) = @_;

    return 1 if ! defined $vars->{uid} || ($vars->{UID} == $vars->{uid} && ! $vars->{rights}{super_control} && ! $vars->{rights}{agency_control} && ! $vars->{rights}{manager_control});

    # неограниченым представителям не даем редактировать шефов
    return 1 if $vars->{rights}{is_agency_main} && $vars->{UID} != $vars->{uid} && rbac_is_agency_chief_rep($vars->{uid});

    if ($vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid})) {
        return 1;
    }

    return 0;
}

#======================================================================

=head2 rbac_cmd_transfer

  for cmd_transfer

=cut

sub rbac_cmd_transfer
{
    my (undef, $vars, $result) = @_;

    my $uid_from = $vars->{client_from} ? get_uid_by_login2( $vars->{client_from} ) : $vars->{uid};
    my $uid_to   = $vars->{client_to}   ? get_uid_by_login2( $vars->{client_to} )   : $vars->{uid};

    return 1  if ! $vars->{rights}->{user_allow_pay};
    return 25 if $vars->{rights}->{agency_disallow_money_transfer};
    
    return 1 if ! rbac_mass_is_owner( undef, $vars->{UID}, [ $uid_from, $uid_to ] );

    return 1 if $vars->{transfer_to}   && $vars->{transfer_to}   !~ /^\d+$/;
    return 1 if $vars->{transfer_from} && $vars->{transfer_from} !~ /^\d+$/;

    my %cids;
    $cids{ $vars->{transfer_to}   }++ if $vars->{transfer_to}   && $vars->{transfer_to}   > 0;
    $cids{ $vars->{transfer_from} }++ if $vars->{transfer_from} && $vars->{transfer_from} > 0;

    for my $cid ( map { /^(from|to)__(\d+)$/ ? ( $2 ) : () } keys %$vars ) {
        $cids{ $cid }++ if   ( $vars->{"from__$cid"} && $vars->{"from__$cid"} > 0 )
                          || ( $vars->{"to__$cid"}   && $vars->{"to__$cid"}   > 0 );
    }

    # проверяем что у клиентов есть права на перенос денег на агентских кампаниях
    my @cids = keys %cids;
    if ( @cids ) {
        my $uid_from_cids = rbac_check_allow_show_camps( undef, $uid_from, \@cids );
        my $uid_to_cids   = rbac_check_allow_show_camps( undef, $uid_to, \@cids );

        # если есть кампании, владельцем которых не являются ни $uid_from ни $uid_to
        return 1 if any { ! $uid_from_cids->{ $_ } && ! $uid_to_cids->{ $_ } } @cids;

        my $operator_perminfo = Rbac::get_perminfo( uid => $vars->{UID} );

        if (   $operator_perminfo
            && ( $operator_perminfo->{role} // '' ) eq $ROLE_CLIENT
            && $operator_perminfo->{agency_client_id}
        ) {
            my $agency_camps = rbac_mass_is_agency_campaign( \@cids );
            return 1 if scalar( uniq values %$agency_camps ) > 1;
            return 21 if ( any { $_ } values %$agency_camps ) && ! Rbac::has_perm( $operator_perminfo, $PERM_MONEY_TRANSFER );
        }
    }

    return 0;
}

#======================================================================

=head2 rbac_cmd_showDiag

  for cmd_showDiag

=cut

sub rbac_cmd_showDiag
{
    my ($rbac, $vars, $result) = @_;

    my $cid_by_bid;
    if (exists $vars->{creative_rejection_reason_ids}) {
        # для причин отклонения креатива передаются только id непосредственных причин
        return 0;
    } elsif ($vars->{mbid}) {
        $cid_by_bid = get_cid(media_bid => $vars->{mbid});
    } elsif ($vars->{mgid}) {
        $cid_by_bid = get_cid(media_gid => $vars->{mgid});
    } elsif ($vars->{bid}) {
        $cid_by_bid = get_cid(bid => $vars->{bid});
    } else {
        return 1;
    }

    return 1 if ! $cid_by_bid || ! rbac_is_owner_of_camp($rbac, $vars->{UID}, $cid_by_bid);

    return 0;
}

#======================================================================

=head2 rbac_cmd_newCamp

  for cmd_editCamp

  правильно было бы переименовать в rbac_cmd_editCamp

=cut

sub rbac_cmd_newCamp
{
    my ($rbac, $vars, $result) = @_;

    my $cid = $vars->{campaign_old} || $vars->{cid} || '';
    if ($cid) {
        my $res = rbac_cmd_by_owners($rbac, $vars, $result) || rbac_cmd_user_allow_edit_camps($rbac, $vars, $result);
        return $res if $res;
    }

    my $role = rbac_who_is($rbac, $vars->{UID});

    # если нет полной свободы, агентским клиентам запрещаем создавать самоходные или сервисируемые кампании
    if ($vars->{cmd} eq 'saveNewCamp' && ! $vars->{campaign_old} && $vars->{rights}->{is_any_client}) {
        return 1 if rbac_has_agency($rbac, $vars->{UID}) && ! $vars->{for_agency} && ! get_allow_create_scamp_by_subclient($vars->{UID});
    }

    if ( $role eq 'client' and
         0 == get_one_field_sql(PPC(uid => $vars->{uid}), ["
             SELECT count(*)
               FROM campaigns
             ", where  => {
                 uid => $vars->{uid}
                , statusEmpty => 'No'
                , type__not_in => get_camp_kind_types("web_edit_base")
             }])
       )
    {
        if ($vars->{cmd} =~ m/^(newCampType)$/ or ! camp_kind_in(type => $vars->{mediaType} || 'text', "web_edit_base")) {
            return 1;
        }
    }

    return 0;
}

#======================================================================

=head2 rbac_cmd_for_cert

    Проверяем доступ к управлению сертификатами.
    Даем доступ суперам и пользователям с включенным API

=cut

sub rbac_cmd_for_cert {
    my ($rbac, $vars, $result) = @_;

    return rbac_can_use_api($rbac, $vars) || $vars->{rights}{super_control} ? 0 : 1;
}

=head2 rbac_cmd_fakeadm
    Модификации тестовой базы
=cut

sub rbac_cmd_fakeadm {
    my ($rbac, $vars, $result) = @_;

    return 1 unless $vars->{rights}->{super_control} || $vars->{rights}->{is_developer};
    return 1 if is_production();
    return 0;
}

=head2 rbac_cmd_internal_networks_only

    Разрешаем доступ только из внутренних сетей

    0 - OK
    1 - forbidden

=cut

sub rbac_cmd_internal_networks_only {
    my ($rbac, $vars, $result) = @_;

    if (is_ip_in_list($ENV{REMOTE_ADDR}, $Settings::INTERNAL_NETWORKS)) {
        return 0;
    } else {
        return 1;
    }
}

=head2 rbac_cmd_internal_user_only

    Разрешаем доступ только для внутренних пользователей

    0 - OK
    1 - forbidden

=cut

sub rbac_cmd_internal_user_only {
    my ($rbac, $vars, $result) = @_;

    return !$vars->{rights}{is_internal_user};
}

=head2 rbac_cmd_specific_super

    В продакшене разрешаем доступ только для определенных суперов из внутренней сети.
    На бетах доступ разрешен всем суперам.

    0 - OK
    1 - forbidden

=cut

sub rbac_cmd_specific_super {
    my ($rbac, $vars, $result) = @_;

    return 1 if !$vars->{is_internal_ip};
    my $allowed_logins = [qw/
        alaien
        yndx-anna-khozina-super
        yndx-chdm
        yndx-elizavet-super
        yndx-kaluga-cat-super
        yndx-kildyashov-support
        yndx-kondratiev-support
        yndx-linazk-super
        yndx-nadiano-super-1
        yndx-pantukhina-super
        yndx-s-udalov-super
        yndx-sivkov-super
        yndx-usyakaya-super
        yndx-vnkorzhov-super
    /];
    my $login = Primitives::get_login_by_uid_passport($vars->{UID});

    return ($vars->{rights}->{role} eq 'super') && (is_beta() || any { $login eq $_ } @$allowed_logins)
        ? 0 : 1;
}

=head2 rbac_cmd_all_super_specific_users

    Разрешаем доступ из внутренней сети всем суперам и некоторым разработчикам.

    0 - OK
    1 - forbidden

=cut

sub rbac_cmd_all_super_specific_users {
    my ($rbac, $vars, $result) = @_;

    return 1 if !$vars->{is_internal_ip};
    my $allowed_developers = [qw/
        yndx-ppalex-superreader
        yndx-gukserg-superreader
    /];
    my $login = Primitives::get_login_by_uid_passport($vars->{UID});

    return ($vars->{rights}->{role} eq 'super') || (any { $login eq $_ } @$allowed_developers)
        ? 0 : 1;
}

=head2 rbac_cmd_specific_super_or_support_or_sustain

    В проде разрешаем доступ только для определённых разработчиков, суперов и саппортов из внутренней сети.
    На бетах доступ также разрешён всем суперам.

    0 - OK
    1 - forbidden

=cut

sub rbac_cmd_specific_super_or_support_or_sustain {
    my ($rbac, $vars, $result) = @_;

    return 1 if !$vars->{is_internal_ip};
    my $allowed_logins = [qw/
        yndx-ibolit
        yndx-rogozin-m-super
        yndx-elizavet-super
        yndx-pantukhina-super
        yndx-s-udalov-super-1
        yndx-kildyashov-support
        yndx-kondratiev-support
        yndx-linazk-super
        yndx-usyakaya-super
        yndx-vnkorzhov-super
        alaien
        yndx-asianina-super
        yndx-sivkov-super
        yndx-chechevaxenia
        yndx-ygmediaplan-support
        yndx-ppalex-superreader
        yndx-gukserg-superreader
    /];
    my $login = Primitives::get_login_by_uid_passport($vars->{UID});

    return (is_beta() && $vars->{rights}->{role} eq 'super') || (any { $login eq $_ } @$allowed_logins)
        ? 0 : 1;
}

#======================================================================

=head2 rbac_cmd_importClampXls

  for XLS import/export iface

=cut

sub rbac_cmd_importCampXls
{
    my ($rbac, $vars, $result, $user_info) = @_;

    my $role = $vars->{rights}{role};
    my $ClientID = $vars->{rights}{ClientID};

    return 2 if $role eq 'empty';
    return 1 if $user_info->{is_readonly_rep};
    return 1 if $vars->{UID} != $vars->{uid} && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid});
    return 1 if defined $vars->{cid} and !rbac_is_owner_of_camp($rbac, $vars->{UID}, $vars->{cid});
    return 3 if !Client::ClientFeatures::do_not_check_passport_karma_for_xls_access($ClientID) && !rbac_check_spam_karma($vars, $user_info);
    return 1 if $vars->{rights}{superreader_control};
    return 0 if !$vars->{rights}{is_any_client} or !$vars->{rights}{client_have_agency};

    my $rbac_error = rbac_can_use_xls($rbac, $vars->{UID}, $vars->{uid});
    return $rbac_error if $rbac_error;

    if ($vars->{cid}) {
        return rbac_get_allow_xls_import_to_campaign($rbac, $vars->{UID}, $vars->{uid}, $vars->{cid});
    } elsif($vars->{destination_camp} && $vars->{destination_camp} eq 'new') {
        my $agency_login = $vars->{for_agency};
        my $agency_clientid;
        if ($agency_login) {
            my $agency_uid = get_uid_by_login2($agency_login);
            $agency_clientid = rbac_get_agency_clientid_by_uid( $agency_uid);
            # проверяем, что у субклиента есть право импорта XLS под агентство
            # rbac_can_import_xls_into_new_camp этот случай корректно не отрабатывает
            if ($agency_clientid && $vars->{rights}{is_any_client} && RBACDirect::rbac_get_allow_import_xls($rbac, $agency_uid, $ClientID)) {
                return 0;
            }
        }
        return RBACDirect::rbac_can_import_xls_into_new_camp($rbac, $vars->{UID}, $vars->{uid}, $agency_clientid) ? 0 : 1;
    } else {
        return 0;
    }
}

=head2 rbac_cmd_exportClampXls

  for XLS export iface

=cut

sub rbac_cmd_exportCampXls
{
    my ($rbac, $vars, $result, $user_info) = @_;

    return 0 if $vars->{rights}{superreader_control};
    return rbac_cmd_importCampXls($rbac, $vars, $result, $user_info);
}

=head2 rbac_check_spam_karma
 
 Проверить и записать при необходимости запись о спам-карме
 Возвращает 0 если запись найдена (плохая спам-карма), иначе 1

 Входные параметры: хеш вида {
    fuid => $fuid,
    yandexuid => $yandexuid,
    uid => $uid,
    rights => { passport_karma_block => 1|0, },
    ...
 }

 Функция записывает в таблицу bad_karma_clients поля yandexuid и fuid01 из кукисов для клиентов с плохой кармой.
 В следующий раз, даже если паспорт вернул хорошую карму, но мы ранее записали хотя бы одно из этих полей
 в свою таблицу - помечаем у себя этого клиента плохим и запоминаем новые данные из кукисов

=cut

sub rbac_check_spam_karma
{
    my ($vars, $user_info) = @_;
    return 1 if !$vars->{rights}{can_block_karma};
    my $uid = $vars->{uid};
    my $yuid = $vars->{yandexuid};
    my $fuid = $vars->{fuid};
    return 0 if !$yuid and !$fuid;
    $fuid = url_hash($fuid);

    my $log = Yandex::Log::Messages->new();
    $log->msg_prefix("bad_karma");

    if ($vars->{rights}{passport_karma_block}) {
        do_insert_into_table(PPCDICT, 'bad_karma_clients', { uid => $uid, yandexuid => $yuid, fuid => $fuid }, ignore => 1);
        $log->out("insert: uid=$uid, yandexuid=$yuid, fuid=$fuid");
        return 0;
    }
    return 1 if $user_info->{passport_karma} == -1; # manual reset
    my ($yuid2, $fuid2) = get_one_line_array_sql(PPCDICT, ["select yandexuid, fuid from bad_karma_clients ",
        where => {
            yandexuid => $yuid,
        }, 
        OR => {
            fuid => $fuid,
        }
    ]);
    if (!$yuid2 and !$fuid2) {
        return 1; 
    }
    do_insert_into_table(PPCDICT, 'bad_karma_clients', { uid => $uid, yandexuid => $yuid, fuid => $fuid }, ignore => 1);

    # временно! (EXP_DATE: 25.08.2012) - смотрим на логи с флагом marked - сколько пользователей заражается друг от друга
    # do_update_table(PPC, 'users_options', { passport_karma => 101 }, where => { uid => $uid});
    $log->out("marked: uid=$uid, yandexuid=$yuid, fuid=$fuid, new_karma=101");
    return 0;
}

#======================================================================

=head2 rbac_cmd_user_allow_edit_camps

  0 - OK
  1 - System Error
  2 - User not found
  3 - User already manager

=cut


sub rbac_cmd_user_allow_edit_camps
{
    my ($rbac, $vars, $result) = @_;

    # $vars->{cid} already splitted in DoCmd::FormCheck::check_required_params()
    my @cids = uniq grep {$_} ref($vars->{cid}) eq 'ARRAY' ? @{ $vars->{cid} } : ($vars->{cid});
    return 1 if @cids && !rbac_user_allow_edit_camp($rbac, $vars->{UID}, \@cids);

    return 0;
}

#======================================================================

=head2 rbac_cmd_user_allow_edit_extended

  0 - OK
  1 - System Error

=cut


sub rbac_cmd_user_allow_edit_camps_extended
{
    my ($rbac, $vars, $result) = @_;

    my $keys_by_fields = {
        cid => 'cid',
        bid => 'bid',
        pid => 'pid',
        adgroup_ids => 'pid',
        adgroup_id => 'pid'
    };

    my @cids;
    foreach my $field (keys %$keys_by_fields) {
        if ($vars->{$field}) {
            my @ids = uniq grep {$_} ref($vars->{$field}) eq 'ARRAY' ? @{ $vars->{$field} } : ($vars->{$field});
            push @cids, ($field eq 'cid')? @ids : @{get_cids($keys_by_fields->{$field} => \@ids)};
        }
    }
    return 1 if @cids && !rbac_user_allow_edit_camp($rbac, $vars->{UID}, [uniq @cids]);

    return 0;
}

#======================================================================

=head2 rbac_cmd_user_allow_edit_day_budget

  0 - OK
  1 - System Error

=cut

sub rbac_cmd_user_allow_edit_day_budget
{
    my ($rbac, $vars, $result) = @_;

    my $cid = $vars->{cid};

    if (rbac_is_agencycampaign($rbac, $cid)) {
        return 1 unless rbac_user_allow_edit_camp($rbac, $vars->{UID}, [$cid]) ||
                rbac_get_allow_transfer_money($rbac, $vars->{UID}, [$cid]);
    } else {
        return 1 if !rbac_user_allow_edit_camp($rbac, $vars->{UID}, [$cid]);
    }


    return 0;
}

#======================================================================

=head2 rbac_cmd_user_allow_show_camps

  0 - OK
  1 - No rights

=cut

sub rbac_cmd_user_allow_show_camps
{
    my ($rbac, $vars, $result) = @_;
    
    # $vars->{cid} already splitted in DoCmd::FormCheck::check_required_params()
    my @cids = grep {$_} ref($vars->{cid}) eq 'ARRAY' ? @{ $vars->{cid} } : ($vars->{cid});

    if (@cids) {
        my $res = RBACDirect::is_user_allowed_for_action_on_camps('show', $vars->{UID}, \@cids);
        return 1 unless all {$res->{$_}} @cids;
    }

    return 0;
}

#======================================================================

=head2 rbac_cmd_check_client_login

  check $FORM{client_login} for managers and agencies
  and fill $rights->{client_uids} = [client_uid1, client_uid2, client_uid3]

  0 - OK
  1 - forbidden

=cut

sub rbac_cmd_check_client_login
{
    my ($rbac, $vars, $result) = @_;

    return 1 if defined $vars->{client_login} && ! ($vars->{rights}{is_internal_user} || $vars->{rights}{agency_control});
    my @client_uids;
    if (defined $vars->{client_login}) {
        my @client_logins = split /\s*,\s*/, $vars->{client_login};
        for my $client_login (@client_logins) {

            my $client_uid = get_uid_by_login2($client_login);
            return 19 unless $client_uid;
            return 1 if $vars->{UID} == $client_uid;
            return 1 unless rbac_is_owner($rbac, $vars->{UID}, $client_uid);

            push @client_uids, $client_uid;
        }
    }
    $$result = {client_uids => [@client_uids]} if @client_uids;

    return 0;
}

#======================================================================

=head2 rbac_cmd_not_for_me

  check $UID != $uid

  0 - OK
  1 - forbidden

=cut

sub rbac_cmd_not_for_me
{
    my ($rbac, $vars, $result) = @_;

    return $vars->{UID} == $vars->{uid} ? 1 : 0;
}

#======================================================================

=head2 rbac_cmd_admShowAddHuman

  allow to super and superteamleaders for create managers

  0 - OK
  1 - forbidden

=cut

sub rbac_cmd_admShowAddHuman
{
    my ($rbac, $vars, $result) = @_;

    return 0 if $vars->{rights}{super_control};
    return 0 if $vars->{rights}{is_superteamleader} && $vars->{human} && $vars->{human} eq 'manager';
    return 1;
}

#======================================================================

=head2 rbac_cmd_managerTakeServicing

  управление клиентами, разрешаем суперам и тимлидерам

  0 - OK
  1 - forbidden

=cut

sub rbac_cmd_managerTakeServicing
{
    my ($rbac, $vars, $result) = @_;

    return 0 if $vars->{rights}->{super_control}
             || $vars->{rights}->{is_any_teamleader};
    return 1;
}

#======================================================================

=head2 rbac_cmd_showManagerLogs

  0 - OK
  1 - forbidden

=cut

sub rbac_cmd_showManagerLogs
{
    my ($rbac, $vars, $result) = @_;

    return 0 if $vars->{rights}->{super_control}
             || $vars->{rights}->{manager_control}
             || $vars->{rights}->{support_control}
             || $vars->{rights}->{limited_support_control}
             || $vars->{rights}->{superreader_control}
             || $vars->{rights}->{is_developer}
             || $vars->{rights}->{media_control}
             || $vars->{rights}->{placer_control};
    return 1;
}

#======================================================================

=head2 rbac_cmd_user_allow_edit_adgroups
    Проверяет, что в полях json_phrases и json_minus_words ключи - это bidы, которые позволено редактировать пользователю.

    Входные параметры: 
        $rbac,
        $vars - ссылка на хеш вида {json_phrases/json_minus_words/json_adgroup_retargetings => {...}, 
                                    json_stat_phrases => [{...},...],
                                    uid => $uid}

    На выходе:
        0 - OK
        1 - forbidden

=cut

sub rbac_cmd_user_allow_edit_adgroups
{
    my ($rbac, $vars) = @_;
    my @objects = map {$vars->{$_}} grep {defined $vars->{$_}} qw/json_phrases json_minus_words json_adgroup_retargetings/;

    if ($vars->{json_stat_phrases}) {
        push @objects, { map { $_->{pid} => undef } grep { $_->{pid} } @{$vars->{json_stat_phrases}} }
    }

    return 0 unless @objects;

    my @cid_list = ();
    foreach my $obj (@objects) {
        next if (ref $obj ne 'HASH');
        push @cid_list, @{get_cids(pid => [keys(%$obj)])};
    }
    return 1 if @cid_list && ! rbac_is_owner_of_camps($rbac, $vars->{uid}, [@cid_list]);

    return 0;
}

#======================================================================

=head2 rbac_cmd_by_owners_of_cid_through_uid

    Проверяем права как в rbac_cmd_by_owners, только права на cid проверяем как UID на uid нa cid
    Требуется когда менеджер должен иметь права на самоходную кампанию своего клиента.
    Даем так проверять только для менеджеров на кампании "общие счета"

    Входные параметры: 
        $rbac,
        $vars - форма + стандартное окружение

    На выходе:
        0 - OK
        1 - forbidden

=cut

sub rbac_cmd_by_owners_of_cid_through_uid {
    my ($rbac, $vars) = @_;

    my $UID_for_cid = $vars->{uid} != $vars->{UID}
                      && $vars->{rights}->{manager_control}
                      && is_wallet_camp(cid => $vars->{cid})
                      ? $vars->{uid}
                      : $vars->{UID};

    return rbac_is_owner($rbac, $vars->{UID}, $vars->{uid})
           && rbac_is_owner_of_camps($rbac, $UID_for_cid, [$vars->{cid}])
           ? 0
           : 1;
}


sub rbac_can_edit_api_settings {
    my ($rbac, $vars) = @_;

    if ( $vars->{rights}{agency_control} || $vars->{rights}{manager_control} || $vars->{rights}{role} =~ /client$/ ) {
        if ($vars->{UID} == $vars->{uid} || $vars->{UID} == $vars->{rights}{client_chief_uid}
                || RBACDirect::rbac_is_related_freelancer($rbac, $vars->{UID}, $vars->{uid})
        ) {
            return 0;
        }
    } elsif ($vars->{rights}{super_control} || $vars->{rights}{support_control}) {
        return 0;
    }

    return 1;
}


=head2 rbac_can_view_api_settings

    Разрешение на просмотр настроек API

    Входные параметры:
        $rbac,
        $vars - форма + стандартное окружение

    На выходе:
        0 - OK
        1 - forbidden

=cut

sub rbac_can_view_api_settings {
    my ($rbac, $vars) = @_;

    if ( $vars->{rights}{agency_control} || $vars->{rights}{manager_control} || $vars->{rights}{role} =~ /client$/ ) {
        if ($vars->{UID} == $vars->{uid} || $vars->{UID} == $vars->{rights}{client_chief_uid}
                || RBACDirect::rbac_is_related_freelancer($rbac, $vars->{UID}, $vars->{uid})
        ) {
            return 0;
        }
    } elsif ($vars->{rights}{super_control} || $vars->{rights}{support_control} ||
        $vars->{rights}{superreader_control} || $vars->{rights}{limited_support_control}) {
        return 0;
    }

    return 1;
}


=head2 rbac_allow_upload_subclient_docs

    Разрешение на загрузку электронных версий документов (Лицензии и сертификаты)
    Функционал доступен для ролей супер, саппорт, менеджер, агентство, клиент

    Входные параметры: 
        $rbac,
        $vars - форма + стандартное окружение

    На выходе:
        0 - OK
        1 - forbidden

=cut

sub rbac_allow_upload_subclient_docs {
    my ($rbac, $vars) = @_;
    
    if ($vars->{uid} 
            && ! rbac_is_owner($rbac, $vars->{UID}, $vars->{uid})
            || ! is_direct($vars->{hostname})) {
        return 1;
    }
    
    return 0;
}

#======================================================================

=head2 rbac_can_read_client_data($rbac, $vars)

Оператор имеет право просматривать данные клиента (например список креативов, список фидов)
Применяется на читающих методах. Право не дает возможность изменять данные клиентов.

=cut

sub rbac_can_read_client_data {
    my ($rbac, $vars) = @_;
    my $client_chief_uid = $vars->{rights}->{client_chief_uid} || $vars->{uid};
    my $client_id = get_clientid(uid => $client_chief_uid);

    my $operator_uid = $vars->{UID};
    my $role = rbac_who_is($rbac, $operator_uid);
    my $operator_has_access;
    if ($role eq 'agency') {
        my $agency_chief = rbac_get_chief_rep_of_agency_rep($operator_uid);
        my $agency_uids = rbac_get_agencies_of_client($rbac, $client_chief_uid);
        $operator_has_access = any { $_ == $agency_chief } @$agency_uids;
    } elsif ($role eq 'manager') {
        $operator_uid = rbac_replace_manager_by_teamlead($rbac, $operator_uid);
        $operator_has_access = rbac_manager_can_create_camp($rbac, $operator_uid, $client_chief_uid);
    } elsif ($role =~ /^(super|superreader|support|(super)?media|(super)?placer)$/) {
        $operator_has_access = rbac_who_is($rbac, $client_chief_uid) eq 'client';
    } elsif ($role eq $ROLE_LIMITED_SUPPORT) {
        $operator_has_access = RBACDirect::rbac_is_related_client_support($rbac, $operator_uid, $client_id);
    } elsif ($role eq 'client') {
        $operator_has_access = 
			rbac_get_chief_rep_of_client_rep($operator_uid) == $client_chief_uid
    }
    if (!$operator_has_access && RBACDirect::rbac_is_related_client($rbac, get_clientid(uid => $operator_uid), $client_id)){
        $operator_has_access = 1;
    }

    if ($operator_has_access) {
        my @feed_ids;
        foreach my $field (qw/feed_id feed_ids/) {
            next unless $vars->{$field};
            push @feed_ids, ref $vars->{$field} eq 'ARRAY'
                            ? @{$vars->{$field}} : @{get_num_array_by_str($vars->{$field})};
        }
        if (@feed_ids) {
            my $feed_clients = get_one_column_sql(PPC(ClientID => $client_id), ["SELECT DISTINCT ClientID FROM feeds", WHERE => {feed_id => \@feed_ids}]);
            $operator_has_access = all { $_ == $client_id } @$feed_clients;
        }
    }

    return $operator_has_access ? 0 : 1;
}

=head2 rbac_modify_feeds($rbac, $vars)

Право на изменение данных по фидам

=cut

sub rbac_modify_feeds {
    my ($rbac, $vars, $rights, $operator_info) = @_;

    # Не даем суперридеру загружать фиды (https://st.yandex-team.ru/DIRECT-104769)
    return 1 if $vars->{rights}->{superreader_control};
    return 1 if $vars->{rights}->{role} eq $ROLE_LIMITED_SUPPORT;
    return 1 if $operator_info->{is_readonly_rep}; # обрабатываем readonly-представителя

    my $can_read = !rbac_can_read_client_data($rbac, $vars);
    my $can_write = 0;
    if ($can_read) {
        $can_write = 1;

        my $operator_uid = $vars->{UID};
        if (rbac_who_is($rbac, $operator_uid) eq 'client') {
            my $client_chief_uid = $vars->{rights}->{client_chief_uid} || $vars->{uid};
            my $agency_uids = rbac_get_agencies_of_client($rbac, $client_chief_uid);
            # there isn't client's freedom
            if (@$agency_uids) {
                my $is_super_subclient
                    = get_client_data(get_clientid(uid => $client_chief_uid), ['allow_create_scamp_by_subclient'])->{allow_create_scamp_by_subclient};
                for my $agency_uid ($is_super_subclient ? () : @$agency_uids) {
                    my $agency_client_id = get_clientid(uid => $agency_uid); 
                    my $rights = rbac_get_subclients_rights($rbac, $agency_uid, $client_chief_uid)->{$agency_client_id};
                    if ($rights->{isSuperSubClient}) {
                        $is_super_subclient = 1;
                        last;
                    }
                }
                $can_write = 0 unless $is_super_subclient;
            }
        }
    } 

    return $can_write ? 0 : 1;
}

=head2 rbac_empty_check($rbac, $vars)

    Функция пустой проверки. Всегда возвращает 0.
    Внимание: применять исключительно в методах при переезде с perl на java при вызове intapi java ручки.
    Перед этим важно(!) убедиться, что в java все необходимые RBAC проверки проведены.

=cut

sub rbac_empty_check {
    return 0;
}
1;
