#######################################################################
#
#  Direct.Yandex.ru
#
#  RBACDirect
#  interface to RBAC
#
#  $Id$
#
#######################################################################

=head1 NAME

RBACDirect - interface to RBAC

=head1 DESCRIPTION

interface to RBAC

=cut

package RBACDirect;

use Direct::Modern;
use feature qw/state/;

use Carp qw/croak/;
use List::MoreUtils qw(uniq any all none);
use Attribute::Handlers;
use Storable qw/freeze thaw/;

use Yandex::Balance;
use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards qw/SHARD_IDS sharded_chunks/;
use Yandex::ListUtils qw/chunks/;
use Yandex::HashUtils qw/hash_map hash_merge hash_grep/;

use Settings;

use API::ClientOptions;
use Campaign::Types;
use Client qw//;
use InternalAdManagers;
use List::Util qw/first/;
use Primitives;
use PrimitivesIds;
use Property;
use Rbac qw/:const get_manager_data remove_manager_data has_perm/;
use RBACElementary;

use base qw/Exporter/;
our @EXPORT = qw(
    rbac_get_campaigns_actions

    rbac_create_client
    rbac_create_client_main_rep
    rbac_delete_campaign
    rbac_get_agencies_uids
    rbac_get_all_reps_agencies_of_manager
    rbac_get_subclients_clientids
    rbac_get_subclients_uids
    rbac_get_subclients_list_with_info
    rbac_get_subclients_all_reps_uids
    rbac_get_camps_for_servicing
    rbac_get_scamps
    rbac_get_agcamps
    rbac_get_scamps_of_managers_and_client_rep
    rbac_create_manager_client
    rbac_accept_servicing
    rbac_create_agency_subclient
    rbac_is_scampaign
    rbac_mass_is_scampaign
    rbac_is_agencycampaign
    rbac_mass_is_agency_campaign

    rbac_is_owner
    rbac_mass_is_owner
    rbac_is_owner_of_camp
    rbac_is_owner_of_camps
    rbac_check_owner_of_camps

    rbac_is_allow_edit_camp
    rbac_check_allow_edit_camps
    rbac_check_allow_show_camps
    rbac_check_allow_show_stat_camps
    rbac_check_allow_delete_camps
    rbac_check_allow_transfer_money_camps
    rbac_get_rights_on_campaigns
    rbac_get_owner_of_camp
    rbac_get_staff
    rbac_change_manager
    rbac_change_manager_of_agency
    rbac_move_scamp_to_nscamp
    rbac_move_nscamp_to_scamp
    rbac_role_name

    rbac_get_manager_of_agency
    rbac_mass_get_manager_of_agencies
    rbac_get_manager_of_agency_clientid
    rbac_mass_get_manager_of_agencies_clientids
    rbac_get_all_managers_of_agency_clientid
    rbac_get_all_managers_of_agencies_clientids

    rbac_assign_manager_to_agency
    rbac_withdraw_manager_from_agency

    rbac_get_managers_of_client
    rbac_get_managers_of_clients
    rbac_get_managers_of_clients_by_clientids
    rbac_get_primary_managers_of_clients_by_clientids
    rbac_get_managers_of_client_campaigns

    rbac_get_clients_of_manager
    rbac_get_limited_agency_rep_of_client
    rbac_get_limited_agencies_reps_of_clients_list
    rbac_get_agencies_of_client
    rbac_mass_get_agencies_of_clients
    rbac_has_agency
    rbac_mass_has_agency
    rbac_get_agencies_for_create_camp_of_client
    rbac_get_campaigns_for_edit
    rbac_create_agency
    rbac_create_agency_main_rep
    rbac_create_agency_limited_rep
    rbac_drop_agency_rep
    rbac_offer_to_servicing
    rbac_decline_servicing
    rbac_get_camp_wait_servicing
    rbac_is_super_placer
    rbac_unbind_manager
    rbac_unbind_agency
    rbac_bind_manager
    rbac_bind_agency
    rbac_add_agency_client_relation
    rbac_remove_agency_client_relation
    rbac_get_subclients_rights
    rbac_mass_get_subclients_rights
    rbac_get_allow_transfer_money_agencies

    rbac_get_all_superteamleaders
    rbac_get_all_teamleaders
    rbac_get_all_managers
    rbac_get_all_agencies
    rbac_get_all_agencies_chief_reps
    rbac_get_teamleaders_data
    rbac_get_teamleaders_data_with_idm
    rbac_get_teamleader_of_manager
    rbac_get_managers_of_teamleader
    rbac_get_managers_of_teamleader_with_idm
    rbac_add_manager_to_teamleader
    rbac_delete_manager_from_teamleader
    rbac_is_teamleader
    rbac_is_superteamleader
    rbac_get_team_of_superteamleader
    rbac_get_superteamleader_of_team
    rbac_get_superteamleaders_data
    rbac_move_team_between_superteamleaders
    rbac_add_team_to_superteamleader
    rbac_delete_team_from_superteamleader
    rbac_move_manager_between_teamleaders

    rbac_user_allow_edit_camp
    rbac_user_allow_edit_camps_detail

    rbac_make_super_subclient
    rbac_make_simple_subclient
    rbac_is_supersubclient
    rbac_set_allow_import_xls
    rbac_get_allow_agency_mass_advq
    rbac_set_allow_agency_mass_advq
    rbac_get_allow_transfer_money
    rbac_set_allow_transfer_money
    rbac_get_allow_xls_import_to_campaign
    rbac_get_allow_xls_import_camp_list

    rbac_create_manager
    rbac_create_superuser
    rbac_create_placer
    rbac_create_superplacer
    rbac_create_mediaplanner
    rbac_create_supermedia
    rbac_create_support
    rbac_create_teamleader
    rbac_create_superteamleader
    rbac_create_superreader
    rbac_create_internal_ad_admin
    rbac_create_internal_ad_manager
    rbac_create_internal_ad_superreader
    rbac_drop_client
    rbac_drop_placer
    rbac_drop_superplacer
    rbac_drop_supermedia
    rbac_drop_user

    rbac_is_super_media_planner
    rbac_get_my_managers
    rbac_get_my_managers_with_idm
    rbac_get_my_agencies

    rbac_move_client_to_limited_agency_rep
    rbac_move_client_to_main_agency_rep
    rbac_switch_agency_chief
    rbac_switch_client_chief
    rbac_drop_client_rep

    rbac_allow_edit_description
    rbac_get_clients_for_step_zero
    rbac_manager_can_create_camp

    rbac_can_use_api
    rbac_can_use_xls

    rbac_check_freedom

    rbac_agency_can_edit_user_data
    rbac_replace_manager_by_teamlead
    rbac_is_agency_client_id_of_client

    rbac_can_create_scamp_by_role
);

our @EXPORT_OK = qw(
    rbac_mass_get_teamleaders_of_managers
    rbac_mass_get_managers_of_teamleaders
);

#======================================================================

=head2 PERMISSIONS

=cut

our %PERMISSIONS = (
    on_campaign => [qw/
        RBAC_ClientControlNSC
        RBAC_ClientControlSC
        RBAC_ClientControlAC
        RBAC_ManageNSCampaign
        RBAC_ManageServicedCampaign
        RBAC_ManagerManageServicedCampaign
        RBAC_ManagerControlSCamps
        RBAC_AgencyManagerAccess
        RBAC_SubClientAccess
        RBAC_SuperSubClientAccess
        RBAC_AgencyAccess
        RBAC_Lookup
        RBAC_EditCamp
        RBAC_ShowStat
    /],

    on_clientid => [qw/
        RBAC_DropClient
        RBAC_DropAgency
        RBAC_Control
        RBAC_AgencyControl
        RBAC_ManagerControlAgency
        RBAC_ClientControlClient
    /],

    on_client_clientid => [qw/
        RBAC_DropClient
        RBAC_AgencyControlClient
        RBAC_ManagerControlClient
    /],

    on_agency_clientid => [qw/
        RBAC_DropAgency
        RBAC_ManagerControlAgency
        RBAC_AgencyControlAgencyLim
    /],

    on_login => [qw/
        RBAC_UserLookup
        RBAC_EditRep
        RBAC_CreateNSCampaign
        RBAC_LeadTheTeam
        RBAC_LeadTheSuperTeam
        RBAC_DropUbermensch
        RBAC_DropPlacer
        RBAC_DropSuperPlacer
        RBAC_DropSupport
        RBAC_DropMediaplanner
        RBAC_DropManager
        RBAC_DropTeamLeader
        RBAC_DropSuperTeamLeader
        RBAC_DropSuperReader
        RBAC_MoveManagerBetweenTeams
        RBAC_MoveManagerFromToTeam
        RBAC_AgencyControlRep
        RBAC_ManagerControlRep
        RBAC_ManagerControlManager
    /],

);

our %ROLE_NAMES = (
    super             => 'супер-пользователь',
    placer            => 'вешальщик',
    manager           => 'менеджер',
    agency            => 'агентство',
    media             => 'медиапланер',
    support           => 'саппорт',
    limited_support   => 'саппорт с ограниченным доступом',
    client            => 'клиент',
    superreader       => 'суперсмотритель',
    subclient         => 'субклиент',
    supersubclient    => 'супер-субклиент',
    teamleader        => 'тимлидер',
    superteamleader   => 'начальник отдела',
    internal_ad_admin => 'администратор внутренней рекламы',
    internal_ad_manager => 'менеджер внутренней рекламы',
    internal_ad_superreader => 'суперсмотритель внутренней рекламы',
);

# базовые роли (без тимлидеров...) - то что схраняется в $login_rights->{role}
our %IS_BASE_ROLE = map {$_ => 1} qw/
    super
    superreader
    support
    limited_support
    placer
    media
    manager
    agency
    client
    internal_ad_admin
    internal_ad_manager
    internal_ad_superreader
/;

my %CAN_CREATE_SERVICED_CAMPAIGNS = map {$_ => 1} qw/super support client manager/;

=head1 CACHE
    Кеширование вызовов методов
=cut

#Кеш, для сохранения результатов вызова методов
our $CACHE_OPTIONS //= {
    namespace => 'RBACDirect',
    max_size => 1024,
    default_expires_in => 60,
};

my $cache = Cache::SizeAwareMemoryCache->new($CACHE_OPTIONS);

=head2 Cacheable (:Cacheable)

 Атрибут, включающий кеширование результатов вызова метода.
 Если метод помечен как :Cacheable - при повторном вызове с теми же аргументами
 будет взят из кеша и возвращен результат предыдущего его вызова.
 По умолчанию результат вызова кешируется по строке, получаемой конкатенацией через ":"
 уникального для каждого метода префикса и скалярных значений всех его аргументов.

 Более сложная сериализация может быть настроена через опции.

 Дополнительные опции:
    ignore_args => [n1, n2, ...] - список индексов аргументов метода,
        которые нужно игнорировать при проверке совпадения значений аргументов.
        Например, если кешируемый метод принимает три аргумента, задано ignore_args => [0] и в кеше
        есть результат вызова method('aa','bb','cc'), при вызове method('zz','bb','cc') будет
        возвращен закешированный результат.

    serialize_by => CODEREF - ссылка на кастомную функцию сериализации агргументов.
        Кастомный сериализатор может использоваться в случаях, когда аргументы метода - не скалярные значения,
        а более сложные структуры/объекты, или если значения аргументов слишком большие, чтобы использовать их как ключ.
        Если кастомная функция сериализации возвращает undef - кеш проверяться и модифицироваться не будет -
        просто выполнентся вызов метода и возвращается его результат.

    debug => 0|1 - включает печать на STDERR информации о том взято ли значение из кеша или выполнялся вызов метода.

    label => 'строка' - используется совместно с debug, задает имя кешируемого метода в отладочных сообщениях.
=cut

sub Cacheable :ATTR(BEGIN) {
    my ($package, $symbol, $referent, $attr, $data, $phase, $filename, $linenum) = @_;

    my %opts = @{$data // []};
    my $prototype = prototype $referent;
    {
        no warnings 'redefine';
        *$symbol = eval "sub($prototype)".q/{return _find_cached_result_or_call_method($referent => \@_, ''.$referent, \%opts)}/;
    };

    return;
}

=head2 _find_cached_result_or_call_method
    Вспомогательный метод, реализующий логику кеширования
=cut

sub _find_cached_result_or_call_method {
    my ($orig_method, $orig_args, $prefix, $opts) = @_;

    my %ignored_args_indexes;
    if ($opts->{ignore_args}){
        %ignored_args_indexes = map {$_ => 1} @{$opts->{ignore_args}};
    }
    my ($key, $result);
    if (defined $opts->{serialize_by}){
        my $serializer = $opts->{serialize_by};
        die 'coderef required for serialize_by value' unless ref $serializer eq 'CODE';
        die 'jointly usage serialize_by and ignore_args forbidden' if $opts->{ignore_args};
        $key = &$serializer($prefix, $orig_args, $opts);
    }
    else{
        my $i = 0;
        $key = join ':', ($prefix, grep {!$ignored_args_indexes{$i++}} @$orig_args);
    }

    unless (defined $key){
        $result = &$orig_method(@$orig_args);
        warn(sprintf('method %s called, because serializer returned undef, result: %s', $opts->{label}//'', $result//'undef'))
            if ($opts->{debug});
    } else {
        $result = thaw($cache->get($key));
        unless (defined $result) {
            $result = &$orig_method(@$orig_args);
            $cache->set($key => freeze([\$result]));
            warn(sprintf('method %s called, result: %s', $opts->{label}//'', $result//'undef')) if ($opts->{debug});
        } else {
            $result = ${$result->[0]};
            warn(sprintf('cached result used for method %s, value: %s', $opts->{label}//'undef', $result//'undef')) if ($opts->{debug});
        }
     }

    return $result;
}

=head2 clear_cache()

    Сбрасывает кеш RBACDirect

=cut

sub clear_cache {
    $cache->clear();
}

=head1 FUNCTIONS

=head2 rbac_can_create_scamp_by_role

    check if user can create serviced campaign by his role

=cut

sub rbac_can_create_scamp_by_role {
    my (undef, $role) = @_;

    return exists $CAN_CREATE_SERVICED_CAMPAIGNS{ $role || '' };
}

=head2 rbac_get_campaigns_actions

    return actions in campaigns list
    for use from Common.pm

    Example of return value:
    {
        cid1 => {
            is_scampaign => 1,
            is_agencycampaign => undef,
            actions => [
                {text => "stop campaign", cmd => "stopCamp"},
                ...
            ],
        },
        ...
    }

=cut

sub rbac_get_campaigns_actions($$$)
{
    my ($uid, $is_super_manager, $campaigns) = @_;

    return {} unless ref($campaigns) eq 'ARRAY' && @$campaigns;

    $uid = rbac_replace_manager_by_teamlead( undef, $uid );

    my @cids;
    my $agency_clients_lim_reps = {};
    my %is_man_camp;
    my %is_ag_camp;
    my %camps;
    for ( @$campaigns ) {
        my $cid = $_->{cid};

        push @cids, $cid;
        $is_man_camp{ $cid } = $_->{ManagerUID};
        $is_ag_camp{ $cid }  = $_->{AgencyUID};
        $camps{ $cid }       = $_;
        $agency_clients_lim_reps->{$_->{ClientID}} = 1 if($_->{AgencyUID});
    }

    if (%$agency_clients_lim_reps) {
        $agency_clients_lim_reps = get_agency_clients_lim_reps([keys %$agency_clients_lim_reps]);
    }

    @cids = uniq @cids;

    my $role = rbac_who_is( undef, $uid );
    my $is_superuser   = $role eq $ROLE_SUPER ? 1 : 0;
    my $is_superreader = $role eq $ROLE_SUPERREADER ? 1 : 0;
    my $is_support     = $role eq $ROLE_SUPPORT ? 1 : 0;
    my $is_limited_support = $role eq $ROLE_LIMITED_SUPPORT ? 1 : 0;
    my $is_placer      = $role eq $ROLE_PLACER ? 1 : 0;
    my $is_media       = $role eq $ROLE_MEDIA ? 1 : 0;
    my $is_manager     = $role eq $ROLE_MANAGER ? 1 : 0;
    my $is_agency      = $role eq $ROLE_AGENCY ? 1 : 0;
    my $is_client      = $role eq $ROLE_CLIENT ? 1 : 0;

    my $operator_perms     = Rbac::get_perminfo( uid => $uid );

    my $can_show_camps     = rbac_check_allow_show_camps( undef, $uid, \@cids );
    my $can_edit_camps     = rbac_user_allow_edit_camps_detail( undef, $uid, \@cids );
    my $can_delete_camps   = rbac_check_allow_delete_camps( undef, $uid, \@cids );
    my $is_wait_serv_camp  = rbac_mass_get_camps_wait_servicing( \@cids );
    my $can_subclient_transfer_money = Rbac::has_perm( $operator_perms, $PERM_MONEY_TRANSFER );

    # set actions on campaigns
    my $result = {};
    my %client_currencies_cache;
    for my $cid ( @cids ) {

        $result->{ $cid } = {
            allow_edit_camp   => $can_edit_camps->{ $cid },
            is_scampaign      => $is_man_camp{ $cid },
            is_agencycampaign => $is_ag_camp{ $cid },
        };

        my @act;

        if ( $can_edit_camps->{ $cid } ) {
            push @act,  ($is_media ? () : {text => iget("Остановить"), cmd => "stopCamp"}),
                        {text => iget("Статистика"), cmd => "showCampStat"},
                        ($is_media ? {text => iget("Параметры"), cmd => "showCampSettings"} : {text => iget("Параметры"), cmd => "editCamp"} ),
                        ($is_media ? () : {text => iget("Включить"), cmd => "resumeCamp"}),
                        ($is_superuser || $is_client ? {text => "Копировать", cmd => "copyCampClient"} : ()),
                        {text => iget("Lookup"), cmd => "Lookup"};
        } elsif ($is_limited_support) {
            #limited_support на странице кампании показываем кнопку Остановить/Возобновить показы
            push @act, ($is_media ? () : {text => iget("Остановить"), cmd => "stopCamp"}),
                 ($is_media ? () : {text => iget("Включить"), cmd => "resumeCamp"}),
        }

        my $allow_pay =    $can_edit_camps->{ $cid }
                        || $is_superuser
                        || ( $is_placer && $is_man_camp{ $cid } )
                        || ( $is_ag_camp{ $cid } && $can_subclient_transfer_money );

        if ( ! $is_media && $allow_pay ) {
            push @act, {text => iget("Оплатить"), cmd => "pay"};
        }

        if ( $can_show_camps->{ $cid } && ! $is_media && ! $is_superreader ) {
            if ( ! _is_campaign_in_special_archive( $camps{ $cid }, \%client_currencies_cache ) ) {
                push @act, {text => iget("Архивировать"), cmd => "campArc"};
            }
        }

        if ( $can_delete_camps->{ $cid } ) {
            push @act, {text => iget("Удалить"), cmd => "delCamp"};
        }

        if (     $can_show_camps->{ $cid }
            && ! ( $is_super_manager && ( $is_man_camp{ $cid } || $is_ag_camp{ $cid } ) ) # суперменеджер умеет смотреть только самоходные кампании
           )
        {
            push @act, {text => "Lookup", cmd => "Lookup"},
                       {text => iget("Статистика"), cmd => "showCampStat"};
        }

        if ( $is_superreader || $is_limited_support || (
                   $can_show_camps->{ $cid }
                && ( ( $is_ag_camp{ $cid } && $is_client && ! Rbac::has_perm( $operator_perms, $PERM_SUPER_SUBCLIENT ) ) || ( $camps{ $cid }->{archived} // '') eq 'Yes' )
                && ! ( $is_super_manager && ( $is_man_camp{ $cid } || $is_ag_camp{ $cid } ) ) # суперменеджер умеет смотреть только самоходные кампании
        ) )  {
            push @act, {text => iget("Параметры"), cmd => "showCampSettings"};
        }

        # check AcceptServicing campaign
        if ( $is_wait_serv_camp->{ $cid } ) {
            push @act, {text => iget("перевод на обслуживание менеджером..."), cmd => "AcceptServicing"};
        }

        if (   ! $is_wait_serv_camp->{ $cid }
            && ! $is_ag_camp{ $cid }
            && ! $is_placer
            && ! $is_support
        ) {
            push @act, {text => iget("Перейти на обслуживание менеджером"), cmd => "offerServicing"};
        }

        # pseudo actions
        if ( $is_client && $is_man_camp{ $cid } ) {
            push @act, {text => "1", cmd => "serviced"};
        }

        if ( $is_ag_camp{ $cid } ) {
            push @act, {text => $is_ag_camp{ $cid }, cmd => "agency_serviced"};
            my $lim_reps = $agency_clients_lim_reps->{$camps{$cid}{ClientID}};
            push @act, {text => $lim_reps, cmd => "agency_serviced_lim_reps"} if $lim_reps;

            if ( $can_subclient_transfer_money ) {
                push @act, {text => 1, cmd => "allow_transfer_money_subclient"};
            }
        }

        if ( $is_man_camp{ $cid } && ! $is_ag_camp{ $cid } ) {
            push @act, {text => $is_man_camp{ $cid }, cmd => "other_manager_serviced"};
        }

        if ( $is_placer || $is_support || $is_limited_support || $is_superuser || $is_superreader || $is_manager ) {
            push @act, {text => "Перемодерировать", cmd => "remoderateCamp"};
        }

        $result->{ $cid }->{actions} = \@act;
    }

    return $result;
}

sub _is_campaign_in_special_archive {
    my ($camp, $client_currencies_cache) = @_;

    if ($camp->{currency} eq 'YND_FIXED'
        && $camp->{archived} eq 'Yes'
        && !camp_kind_in(type => $camp->{mediaType}, qw/non_currency_convert/)
    ) {
        # кампания могла быть архивна на момент перехода клиента к мультивалютности
        # и осталась в у. е., то разархивировать её не даём
        my $client_id = $camp->{ClientID};
        my $uid = $camp->{uid};
        $client_currencies_cache->{$client_id} ||= get_one_field_sql(PPC(ClientID => $client_id),
            ['SELECT work_currency FROM clients', WHERE => {ClientID => $client_id}]) || "YND_FIXED";
        if ($client_currencies_cache->{$client_id} && $client_currencies_cache->{$client_id} ne 'YND_FIXED') {
            return 1;
        }
    }

    return 0;
}

#======================================================================

=head2 rbac_create_client

=cut

sub rbac_create_client
{
    my ($rbac, $uid, $commit_after_create) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_create_client_main_rep

    my $error = rbac_create_client_main_rep($rbac, $client_id, $rep_uid, $dont_commit);

=cut

sub rbac_create_client_main_rep
{
    my ($rbac, $client_id, $rep_uid, $dont_commit) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_delete_campaign

  $errcode = rbac_delete_campaign(undef, $cid, $uid);
  $cid  - cid
  $uid  - client uid

=cut

sub rbac_delete_campaign
{
    my (undef, $cid, $uid) = @_;

    my $manager_uid = rbac_is_scampaign(undef, $cid);
    my $agency_uid = rbac_is_agencycampaign(undef, $cid);
    my $client_id = Rbac::get_rep2client([$uid], role => $ROLE_CLIENT)->{$uid} || die "ClientID for $uid not found";

    my $errorcode;

    # RBAC_ManageServicedCampaign
    if ($manager_uid && ! $agency_uid) {

        $errorcode = rbac_unbind_manager(undef, $manager_uid, $uid);
    }

    return 0;
}

#======================================================================

=head2 rbac_get_agencies_uids

  $agencies_uids_arrayref = rbac_get_agencies_uids($rbac, $manager_uid);

  Возвращаются uid-ы шефов агентств, привязанных к менеджеру

=cut

sub rbac_get_agencies_uids
{
    my (undef, $manager_uid) = @_;

    my $uids = get_one_column_sql(PPC(shard => 'all'), [
        'SELECT c.chief_uid FROM agency_managers am JOIN clients c ON (am.agency_client_id = c.ClientID)',
        WHERE => { 'am.manager_uid' => $manager_uid }
    ]);

    return [ uniq @$uids ];
}

#======================================================================

=head2 rbac_get_all_reps_agencies_of_manager

  $agencies_uids_arrayref = rbac_get_all_reps_agencies_of_manager($rbac, $manager_uid);

    Возвращаются uid-ы всех представителей агентств, привязанных к менеджеру

=cut

sub rbac_get_all_reps_agencies_of_manager
{
    my (undef, $manager_uid) = @_;

    my $uids = get_one_column_sql(PPC(shard => 'all'), [
        'SELECT u.uid FROM agency_managers am JOIN users u ON (am.agency_client_id = u.ClientID)',
        WHERE => { 'am.manager_uid' => $manager_uid }
    ]);

    return [ uniq @$uids ];
}

=head2 rbac_mass_get_agencies_clientids_of_managers

    $result = rbac_mass_get_agencies_clientids_of_managers([ manager_uid1, ... ])

    $result = {
        manager_uid1 => [ agency_client_id1, agency_client_id2, ... ],
        ...
    }

=cut

sub rbac_mass_get_agencies_clientids_of_managers {
    my ( $manager_uids ) = @_;

    my $rows = get_all_sql( PPC(shard => 'all'), [
        'SELECT manager_uid, agency_client_id FROM agency_managers',
        WHERE => { manager_uid => $manager_uids }
    ]);

    my %res;
    for my $row ( @$rows ) {
        push @{ $res{ $row->{manager_uid} } //= [] }, $row->{agency_client_id};
    }

    return \%res;
}

=head2 rbac_get_subclients_clientids

    $subclients_clientids = rbac_get_subclients_clientids($rbac, $agency_uid);
    $subclients_clientids = rbac_get_subclients_clientids($rbac, [$agency_uid1, $agency_uid2, ...]);
    $subclients_clientids = rbac_get_subclients_clientids($rbac, [$agency_uid1, $agency_uid2, ...], chunk_size => 10);
    $subclients_clientids => [$clientid1, $clientid2, ...]

=cut

sub rbac_get_subclients_clientids {
    my (undef, $agency_uid, %O) = @_;
    my $ag_uids = ref($agency_uid) eq 'ARRAY' ? $agency_uid : [$agency_uid];

    my $ag_perms = Rbac::get_key2perminfo(uid => $ag_uids);

    my (@ag_main_clientids, @ag_lim);
    for my $uid (@$ag_uids) {
        my $perms = $ag_perms->{$uid};
        next unless $perms && $perms->{role} eq $Rbac::ROLE_AGENCY;
        if ($perms->{rep_type} eq $Rbac::REP_CHIEF || $perms->{rep_type} eq $Rbac::REP_MAIN) {
            push @ag_main_clientids, $perms->{ClientID};
        } else {
            push @ag_lim, $uid;
        }
    }

    my @client_ids;
    if (@ag_main_clientids) {
        push @client_ids, @{get_one_column_sql(PPC(shard => 'all'), ["SELECT ClientID FROM clients", WHERE => {agency_client_id => \@ag_main_clientids}])};
    }
    if (@ag_lim) {
        push @client_ids, @{get_one_column_sql(PPC(shard => 'all'), ["SELECT ClientID FROM clients", WHERE => {agency_uid => \@ag_lim}])};
    }

    return [uniq @client_ids];
}

#======================================================================

=head2 rbac_get_subclients_uids

  $uids_arrref = rbac_get_subclients_uids($rbac, $agency_uid);

=cut

sub rbac_get_subclients_uids
{
    my (undef, $agency_uid) = @_;

    my $agency_perminfo = Rbac::get_perminfo(uid => $agency_uid);
    return [] unless $agency_perminfo->{role} eq $Rbac::ROLE_AGENCY;

    if ($agency_perminfo->{rep_type} eq $Rbac::REP_CHIEF || $agency_perminfo->{rep_type} eq $Rbac::REP_MAIN) {
        return get_one_column_sql(PPC(shard => 'all'), "SELECT chief_uid FROM clients WHERE agency_client_id = ?", $agency_perminfo->{ClientID});
    } else {
        return get_one_column_sql(PPC(shard => 'all'), "SELECT chief_uid FROM clients WHERE agency_uid = ? UNION SELECT chief_uid FROM agency_lim_rep_clients ac JOIN clients c USING(ClientID) WHERE ac.agency_uid = ?", $agency_uid, $agency_uid);
    }
}


=head2 rbac_get_subclients_list_with_info

  $subclients_hashref = rbac_get_subclients_list_with_info(undef, $agency_uid);
  for use from DoCmd::cmd_showClients()

  $subclients_hashref: {
      54325235 => {                       # представитель-шеф суб-клиента
          ClientID => 1234
          , is_super_subclient => 1       # является супер-субклиентом (право редактировать кампании)
          , is_SuperRightsToSSC => 0      # имеет право импортировать xls
          , allowTransferMoney => 1       # имеет право переносить деньги
          , limited_agency_uid => 7654321 # назначенный ограниченный представитель агентства этому клиенту
          , client_main_rep_uids => [1234, 5436, 2683, ...] # представители не-шефы клиента
      },
      34565464 => {...},
  }

=cut

sub rbac_get_subclients_list_with_info
{
    my ( undef, $agency_uid ) = @_;

    my $agency_rep_perms = Rbac::get_perminfo( uid => $agency_uid );

    return {} unless $agency_rep_perms && ($agency_rep_perms->{role} // '') eq $ROLE_AGENCY;

    my ($where, @sql_union_parts);
    if ( ($agency_rep_perms->{rep_type} // '') eq $REP_LIMITED ) {
        $where = { 'c.agency_uid' => $agency_uid };
        @sql_union_parts = ('UNION SELECT u.ClientID, u.uid FROM agency_lim_rep_clients alrc JOIN users u USING(ClientID)',
                        WHERE => { 'alrc.agency_uid' => $agency_uid });
    }
    else {
        $where = { 'c.agency_client_id' => $agency_rep_perms->{ClientID} };
    }

    my $rows = get_all_sql(PPC( shard => 'all' ), [
        'SELECT u.ClientID, u.uid FROM clients AS c JOIN users AS u USING (ClientID)',
        WHERE => $where,
        @sql_union_parts
    ]);

    return {} unless @$rows;

    my %subclient_ids_to_uids;
    for my $row ( @$rows ) {
        push @{ $subclient_ids_to_uids{ $row->{ClientID} } }, $row->{uid};
    }

    my $result = {};

    foreach my $chunk ( sharded_chunks( ClientID => [ keys %subclient_ids_to_uids ], chunk_size => 10_000 ) ) {

        my $client_ids = $chunk->{ClientID};

        my $perms_by_clientid = Rbac::get_key2perminfo( ClientID => $client_ids );

        my @agency_reps = uniq grep { $_ } map { $_->{agency_uid}, @{$_->{agency_uids} // []} } values %$perms_by_clientid;

        my $reps = Rbac::get_key2perminfo( uid => \@agency_reps );
        my %limited_reps = map { $_ => 1 } grep { ($reps->{ $_ }{rep_type} // '') eq $REP_LIMITED } keys %$reps;

        foreach my $perm ( values %$perms_by_clientid ) {

            my $reps = $subclient_ids_to_uids{ $perm->{ClientID} } // [];
            $result->{ $perm->{chief_uid} } = {
                ClientID             => $perm->{ClientID},
                is_super_subclient   => Rbac::has_perm( $perm, $PERM_SUPER_SUBCLIENT ),
                is_SuperRightsToSSC  => Rbac::has_perm( $perm, $PERM_XLS_IMPORT ),
                allowTransferMoney   => Rbac::has_perm( $perm, $PERM_MONEY_TRANSFER ),
                limited_agency_uid   => ( ( $perm->{agency_uid} && exists( $limited_reps{ $perm->{agency_uid} } ) ) ? $perm->{agency_uid} : undef ),
                limited_agency_uids => $perm->{agency_uids},
                client_main_rep_uids => [ grep { $_ != $perm->{chief_uid} } @$reps ]
            };
        }
    }

    return $result;
}

=head2 rbac_get_subclients_all_reps_uids

  $uids_arrref = rbac_get_subclients_all_reps_uids($rbac, $agency_uid);

=cut

sub rbac_get_subclients_all_reps_uids
{
    my ($rbac, $agency_uid) = @_;

    my $subclients_clientids = rbac_get_subclients_clientids(undef, [$agency_uid]);

    my @uids;
    for my $chunk (chunks($subclients_clientids, 1000)) {
        my $cl2reps = Rbac::get_reps_multi(ClientID => $chunk);
        push @uids, @$_ for values %$cl2reps;
    }

    return \@uids;
}

#======================================================================

=head2 rbac_get_camps_for_servicing

  @camps_cids = rbac_get_camps_for_servicing($rbac, $manager_uid);

=cut

sub rbac_get_camps_for_servicing
{
    return get_one_column_sql(PPC(shard => 'all'), 'SELECT cid FROM camps_for_servicing');
}

#======================================================================

=head2 rbac_create_manager_client

    Создание сервисируемого клиента

=cut

sub rbac_create_manager_client
{
    my ($rbac, $UID, $uid, $commit_after_create, %O) = @_;

    Rbac::clear_cache();

    return 2 if ! defined $uid
                || $UID == $uid
                || rbac_who_is($rbac, $UID) ne 'manager'
                || rbac_who_is($rbac, $uid) !~ /^(?:.*client|empty)$/;

    # если нет свободы создавать самоходные/сервисируемые кампании у субклиента
    return 'NOT_ENOUGH_FREEDOM' if (!$O{type} || $O{type} ne 'geo') && ! rbac_check_freedom($rbac, $uid);

    my $errcode = rbac_create_client($rbac, $uid);
    return $errcode if $errcode;

    my $errorcode = rbac_bind_manager($rbac, $UID, $uid);
    return 2 if $errorcode;

    return 0;
}

#======================================================================

=head2 rbac_bind_manager

  bind client to manager if need
  $result = rbac_bind_manager(undef, $manager_uid, $client_uid);

=cut

sub rbac_bind_manager
{
    my (undef, $manager_uid, $client_uid) = @_;

    my $client_id = rbac_get_client_clientid_by_uid($client_uid) || die "ClientID for $client_uid not found";

    Client::bind_manager($client_id, $manager_uid);

    return 0;
}

#======================================================================

=head2 rbac_accept_servicing

  rbac_accept_servicing(undef, $cid, $client_uid, $manager_uid);

=cut

sub rbac_accept_servicing
{
    my (undef, $cid, $client_uid, $manager_uid) = @_;

    do_delete_from_table(PPC(cid => [$cid]), 'camps_for_servicing', where => {cid => $cid});

    return 0;
}

#======================================================================

=head2 rbac_create_agency_subclient

  Создание агентского субклиента

=cut

sub rbac_create_agency_subclient
{
    my ($rbac, $UID, $uid, $commit_after_create) = @_;

    Rbac::clear_cache();

    my $perms = Rbac::get_perminfo(uid => $UID) || die "No perminfo fopr uid $UID";

    return 2 if ! defined $uid
                || $UID == $uid
                || $perms->{role} ne $Rbac::ROLE_AGENCY
                || rbac_who_is($rbac, $uid) !~ /^(?:.*client|empty)$/; # supersubclient, subclient, client or empty

    my $errorcode = rbac_create_client($rbac, $uid);
    return $errorcode if $errorcode;

    my $agency_id = $perms->{ClientID} || die "ClientID for $UID not found";
    my $agency_uid = ( $perms->{rep_type} eq $Rbac::REP_LIMITED ? $UID : $perms->{chief_uid} ) || die "No agency chief rep for $agency_id";
    my $client_id = rbac_get_client_clientid_by_uid($uid) || die "ClientID for $uid not found";

    return 'NOT_ENOUGH_FREEDOM' unless get_allow_agency_bind_client($agency_id, $client_id);

    my $client_relations = get_agency_clients_relations_data($client_id);
    # если обслуживание завершено, то при добавлении кампании не биндим клиента обратно к агентству
    if (none { $_->{agency_client_id} == $agency_id && $_->{bind} eq 'No' } @$client_relations) {
        # добавляем 1ое агентство клиенту автоматически
        if (none { $_->{bind} eq 'Yes' } @$client_relations) {
            do_insert_into_table(PPC(ClientID => $client_id)
                                 , 'agency_client_relations'
                                 , {agency_client_id => $agency_id
                                    , client_client_id => $client_id
                                    , bind => 'Yes'
                                   }
                                 , on_duplicate_key_update => 1
                                 , key => ['agency_client_id', 'client_client_id']
                                );
        }

        $errorcode = rbac_bind_agency($rbac, $agency_id, $client_id, $UID);
        return $errorcode if $errorcode;

        my $perminfo = Rbac::get_perminfo(ClientID => $client_id);
        if ($perminfo && (($perminfo->{agency_client_id} // -1) != $agency_id)) {
            Client::create_update_client({client_data => {
                ClientID => $client_id,
                agency_client_id => $agency_id,
                agency_uid => $agency_uid,
                }});
        }
    }

    return 0;
}

#======================================================================

=head2 rbac_bind_agency

  bind client to agency if need
  $result = rbac_bind_agency($rbac, $agency_id, $client_id, $agency_uid);

=cut

sub rbac_bind_agency
{
    my ($rbac, $agency_id, $client_id, $agency_uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_add_agency_client_relation

  $result = rbac_add_agency_client_relation($rbac, $agency_id, $client_id, $revert_campaigns);
  $revert_campaigns: set statusNoPay = 'No' for old agency-clients campaigns

=cut

sub rbac_add_agency_client_relation
{
    my ($rbac, $agency_id, $client_id, $agency_uid, $revert_campaigns) = @_;

    Rbac::clear_cache();

    do_insert_into_table(PPC(ClientID => $client_id),
        'agency_client_relations',{
            agency_client_id => $agency_id,
            client_client_id => $client_id,
            bind => 'Yes'
        },
        on_duplicate_key_update => 1,
        key => ['agency_client_id', 'client_client_id']
    );

    my $errorcode = rbac_bind_agency($rbac, $agency_id, $client_id, $agency_uid);
    return $errorcode if $errorcode;

    my $perminfo = Rbac::get_perminfo(ClientID => $client_id);
    if ($perminfo && $perminfo->{agency_client_id} != $agency_id) {
        Client::create_update_client({client_data => {
            ClientID => $client_id,
            agency_client_id => $agency_id,
            agency_uid => $agency_uid,
        }});
    }

    # если уже есть самоходные/серв. кампании, то автоматом даём свободу 3
    if (get_one_field_sql(PPC(ClientID => $client_id), "
            select count(*)
            from campaigns c
                join users u using(uid)
            where u.ClientID = ?
                and statusEmpty = 'No'
                and AgencyUID is null",
            $client_id)
    ) {
        do_sql(PPC(ClientID => $client_id), "insert into clients (ClientID, allow_create_scamp_by_subclient)
                     values (?, ?)
                     on duplicate key update
                       allow_create_scamp_by_subclient = values(allow_create_scamp_by_subclient)
                    ",
                    $client_id, 'Yes'
              );
    }

    if ($revert_campaigns) {
        do_sql(PPC(ClientID => $client_id),
                    "update campaigns c
                       join users u on c.uid = u.uid
                     set c.statusNoPay = 'No'
                     where c.statusEmpty = 'No'
                       and u.ClientID = ?
                       and c.AgencyID = ?
                    ", $client_id, $agency_id
              );
    }

    return 0;
}

#======================================================================

=head2 rbac_remove_agency_client_relation

  rbac_remove_agency_client_relation(undef, $agency_id, $client_id);
  + запрещаем оплату всех кампаний этого анентства-клиента

=cut

sub rbac_remove_agency_client_relation
{
    my (undef, $agency_id, $client_id) = @_;

    do_update_table(PPC(ClientID => $client_id)
                    , 'agency_client_relations'
                    , {bind => 'No'}
                    , where => {agency_client_id => $agency_id
                                , client_client_id => $client_id
                               }
                   );

    # запрещаем оплату на всех кампаниях данного агентства-клиента
    my $uids = get_uids('ClientID' => $client_id);
    do_update_table(PPC(ClientID => $client_id)
                    , "campaigns"
                    , {statusNoPay => 'Yes'}
                    , where => {uid => $uids, AgencyID => $agency_id}
        );

    return 0;
}

#======================================================================

=head2 rbac_get_subclients_rights

  my $all_rights = rbac_get_subclients_rights(undef, $operator_uid, $client_uid);
  $operator_uid - manager/agency/super uid for who show rights of client

  return:
  {
      agency_ClientID_1 => {
          isSuperSubClient   => 1, # является суперсубклиентом для agency_uid1
          allowImportXLS     => 1, # может импортировать xls в кампании agency_uid1
          allowTransferMoney => 1, # право суперсубклиенту переносить средства с кампании на кампанию (в агентстве agency_uid1)
      },

      agency_ClientID_2 => {
          ...
      }
  }

=cut

sub rbac_get_subclients_rights
{
    my (undef, $operator_uid, $client_uid) = @_;

    my $client_id = rbac_get_client_clientid_by_uid($client_uid) || die "ClientID for $client_uid not found";

    return rbac_mass_get_subclients_rights(undef, $operator_uid, {$client_uid => $client_id})->{$client_id};
}

=head2 rbac_mass_get_subclients_rights

  $operator_uid - manager/agency/super uid for who show rights of client
  $for_cur_agency_flag - считать оператора агентством и вернуть права клиента только для него
  $client_uids2client_ids - хэш uid => client_id клиентов

  my $all_rights = rbac_mass_get_subclients_rights(undef, $operator_uid, { $client_uid => $client_id }, 0);

  return:
  {
    client_ClientID_1 => {
        agency_ClientID_1 => {
            isSuperSubClient   => 1, # является суперсубклиентом для agency_uid1
            allowImportXLS     => 1, # может импортировать xls в кампании agency_uid1
            allowTransferMoney => 1, # право суперсубклиенту переносить средства с кампании на кампанию (в агентстве agency_uid1)
        },

        agency_ClientID_2 => {
            ...
        }
    },
    client_ClientID_2 => {
        agency_ClientID_1 => {
            isSuperSubClient   => 1, # является суперсубклиентом для agency_uid1
            allowImportXLS     => 1, # может импортировать xls в кампании agency_uid1
            allowTransferMoney => 1, # право суперсубклиенту переносить средства с кампании на кампанию (в агентстве agency_uid1)
        },

        agency_ClientID_2 => {
            ...
        }
    }
  }

=cut

sub rbac_mass_get_subclients_rights
{
    my (undef, $operator_uid, $client_uids2client_ids, %O) = @_;

    my $operator_perms = Rbac::get_perminfo(uid => $operator_uid);
    my $uid2perms      = Rbac::get_key2perminfo(uid => [ keys %$client_uids2client_ids ]);

    my @agencies_uids = grep { $_ } map { $_->{agency_uid}, @{$_->{agency_uids}//[]} } values %$uid2perms;
    my $is_owner_of_uid = rbac_is_owner_of_users( $operator_uid, \@agencies_uids );

    my $rights = {};

    for my $perms ( values %$uid2perms ) {
        my $ClientID         = $perms->{ClientID};
        my $agency_client_id = $perms->{agency_client_id} or next;

        next unless (any { $is_owner_of_uid->{$_} } ($perms->{agency_uid}, @{$perms->{agency_uids} // []})) || $O{not_filter_by_perm};

        $rights->{ $ClientID }->{ $agency_client_id } = {
            isSuperSubClient   => Rbac::has_perm( $perms, $Rbac::PERM_SUPER_SUBCLIENT) ? 1 : 0,
            allowTransferMoney => Rbac::has_perm( $perms, $Rbac::PERM_MONEY_TRANSFER)  ? 1 : 0,
            allowImportXLS     => Rbac::has_perm( $perms, $Rbac::PERM_XLS_IMPORT)      ? 1 : 0,
        };
    }

    return $rights;
}

#======================================================================

=head2 rbac_get_allow_transfer_money_agencies

    Для субклиента возвращаем список агентств (ClientID) которые разрешили переносить деньги

        $agencies_client_id_arrref = rbac_get_allow_transfer_money_agencies(undef, $ClientID);

=cut

sub rbac_get_allow_transfer_money_agencies {
    my (undef, $client_id) = @_;

    my $perm_info = Rbac::get_perminfo(ClientID => $client_id);

    return [] unless   $perm_info
                    && $perm_info->{agency_client_id}
                    && Rbac::has_perm( $perm_info, $PERM_MONEY_TRANSFER );

    return [ $perm_info->{agency_client_id} ];
}

#======================================================================

=head2 rbac_is_scampaign

  $manager_uid = rbac_is_scampaign(undef, $cid);

  return manager_uid if cid is serviced campaign (of any direct client)
  else return 0

=cut

sub rbac_is_scampaign
{
    my (undef, $cid) = @_;

    return rbac_mass_is_scampaign(undef, [$cid])->{$cid};
}

=head2 rbac_mass_is_scampaign

  $manager_uid_hashref = rbac_mass_is_scampaign(undef, $cid);

  return hashref to cid => manager_uid  || 0 if cid is serviced campaign
  (of any direct client) else return 0

=cut

sub rbac_mass_is_scampaign
{
    my (undef, $cids) = @_;

    my $data_hash = get_hash_sql(PPC(cid => $cids), [
        'SELECT cid, ManagerUID FROM campaigns',
        WHERE => { cid => SHARD_IDS }
    ]);

    foreach my $cid (@$cids) {
        if (!$data_hash->{$cid}) {
            $data_hash->{$cid} = 0;
        }
    }

    return $data_hash;
}

#======================================================================

=head2 rbac_is_agencycampaign

  $agency_uid = rbac_is_agencycampaign(undef, $cid);
  return agency_uid if cid is agency campaign (of any subclient)
  else return 0

=cut

sub rbac_is_agencycampaign
{
    my (undef, $cid) = @_;

    my $ag = get_one_field_sql(PPC(cid => $cid), "SELECT AgencyID from campaigns WHERE cid = ?", $cid)
        || return 0;
    return rbac_get_chief_rep_of_agency($ag) || 0;
}

=head2 rbac_mass_is_agency_campaign

  $cid_to_agency_id_hashref = rbac_mass_is_agency_campaign($cids);

=cut

sub rbac_mass_is_agency_campaign
{
    my ($cids) = @_;

    my $data_hash = get_hash_sql(PPC(cid => $cids), [
        'SELECT cid, AgencyID FROM campaigns',
        WHERE => {
            cid => SHARD_IDS,
            AgencyID__is_not_null => 1,
        }
    ]);

    foreach my $cid (@$cids) {
        if (!$data_hash->{$cid}) {
            $data_hash->{$cid} = 0;
        }
    }

    return $data_hash;
}

#======================================================================

=head2 rbac_get_staff

  $result = rbac_get_staff(undef);

  return all managers, placers and superusers uids
  {
    managers => [123, 345, 2323, ...],
    supports => [1231, 3451, 23231, ...],
    mediaplanners => [1232, 3452, 23232, ...],
    superusers => [123, 345, 23232, ...],
    ...
  }

=cut

sub rbac_get_staff
{
    my %mapping = (
        super             => { '' => 'superusers' },
        manager           => { '' => 'managers', teamleader => 'teamleaders', superteamleader => 'superteamleaders', },
        placer            => { '' => 'placers', 'superplacer' => 'superplacers' },
        media             => { '' => 'mediaplanners' },
        support           => { '' => 'supports' },
        superreader       => { '' => 'superreaders' },
        internal_ad_admin => { '' => 'internal_ad_admins'},
        internal_ad_manager => {'' => 'internal_ad_managers'},
        internal_ad_superreader => {'' => 'internal_ad_superreaders'},
        );

    my $sth = exec_sql(PPC(shard => 'all'), ["SELECT role, subrole, chief_uid FROM clients", WHERE => {role => [keys %mapping]}]);
    my $ret = {};
    while(my ($role, $subrole, $chief_uid) = $sth->fetchrow_array) {
        push @{$ret->{$mapping{$role}->{''}}}, $chief_uid;
        if ($subrole && $mapping{$role}->{$subrole}) {
            push @{$ret->{$mapping{$role}->{$subrole}}}, $chief_uid;
        }
    }

    return $ret;
}

#======================================================================

=head2 rbac_change_manager

  $result = rbac_change_manager($rbac, $old_muid, $new_muid, $cid);

=cut

sub rbac_change_manager
{
    my ($rbac, $old_muid, $new_muid, $cid) = @_;

    Rbac::clear_cache();

    my $perms = Rbac::get_perminfo(uid => $old_muid) // {};
    return 255 if   ( $perms->{role} // '') ne $ROLE_MANAGER
                 || ( ! rbac_user_allow_edit_camp(undef, $old_muid, $cid) )
                 || ( ! rbac_is_scampaign(undef, $cid) );

    my $client_uid = rbac_get_owner_of_camp($rbac, $cid);
    my $client_id = rbac_get_client_clientid_by_uid($client_uid) || die "ClientID for $client_uid not found";

    my $errorcode = rbac_bind_manager($rbac, $new_muid, $client_uid);
    return $errorcode if $errorcode;

    do_update_table(PPC(cid => $cid), 'campaigns', {
            ManagerUID => $new_muid,
            statusBsSynced => 'No'
        },
        where => { cid => $cid },
    );

    $errorcode = rbac_unbind_manager($rbac, $old_muid, $client_uid);
    return $errorcode;
}

#======================================================================

=head2 rbac_change_manager_of_agency

  $error_code = rbac_change_manager_of_agency($rbac, $agency_uid, $old_muid, $new_muid);
  move agency from $old_muid to $new_muid in RBAC
  return 0 - if no errors

=cut

sub rbac_change_manager_of_agency
{
    my ($rbac, $agency_uid, $old_muid, $new_muid) = @_;

    Rbac::clear_cache();

    my $agency_id = rbac_get_agency_clientid_by_uid( $agency_uid) || die "ClientID for $agency_uid not found";
    my $agency_perminfo = Rbac::get_perminfo(ClientID => $agency_id);
    return 1 if $agency_perminfo->{primary_manager_set_by_idm};

    do_in_transaction {
        # меняем также сохраненных директовских/баяновских/гео менеджеров у агентства
        do_sql(PPC(ClientID => $agency_id),
                    "update clients
                     set primary_manager_uid = ?
                     where primary_manager_uid = ?
                     and ClientID = ?", $new_muid, $old_muid, $agency_id);

        do_sql(PPC(ClientID => $agency_id),
                    "update clients
                     set primary_bayan_manager_uid = ?
                     where primary_bayan_manager_uid = ?
                       and ClientID = ?", $new_muid, $old_muid, $agency_id);

        Client::bind_manager_to_agency($agency_id, $new_muid);

        Client::unbind_manager_from_agency($agency_id, $old_muid);
    };

    return 0;
}

#======================================================================

=head2 rbac_move_scamp_to_nscamp

  $result = rbac_move_scamp_to_nscamp($rbac, $cid);

=cut

sub rbac_move_scamp_to_nscamp
{
    my ($rbac, $cid) = @_;

    Rbac::clear_cache();

    my $muid = rbac_is_scampaign($rbac, $cid);
    return 255 if ! defined $muid || $muid == 0;

    my $client_uid = get_uid(cid => $cid);

    my $errorcode;

    $errorcode = rbac_unbind_manager($rbac, $muid, $client_uid);
    return $errorcode if $errorcode;

    do_update_table(PPC(cid => $cid), 'campaigns', {
            ManagerUID => undef,
        },
        where => { cid => $cid },
    );

    return 0;
}

#======================================================================

=head2 rbac_move_nscamp_to_scamp

  $result = rbac_move_nscamp_to_scamp(undef, $dbh, $cid);
        $manager_uid

=cut

sub rbac_move_nscamp_to_scamp
{
    my (undef, $cid, $manager_uid, $client_uid) = @_;

    do_update_table(PPC(cid => $cid), 'campaigns',
                    {ManagerUID => $manager_uid, statusBsSynced => 'No'},
                    where => {cid => $cid});

    return 0;
}
#======================================================================

=head2 rbac_get_scamps

  $cids_arrayref = rbac_get_scamps(undef, $manager_uid);

  return all campaigns for manager

=cut

sub rbac_get_scamps
{
    my (undef, $manager_uid) = @_;

    return get_one_column_sql(PPC(shard => 'all'), "SELECT cid FROM campaigns WHERE ManagerUID = ? AND statusEmpty = 'No'", $manager_uid);
}

#======================================================================

=head2 rbac_get_scamps_of_managers_and_client_rep

  $cids_arrayref = rbac_get_scamps_of_managers_and_client_rep(undef, $manager_uid, [$client_uid1, $client_uid2, $client_uid3]);
  $cids_arrayref = rbac_get_scamps_of_managers_and_client_rep(undef, $manager_uid, $client_uid);

  $client_uids - uid or arrref of uids

  return all scampaigns for client and manager

=cut

sub rbac_get_scamps_of_managers_and_client_rep
{
    my (undef, $manager_uid, $client_uids) = @_;

    return get_one_column_sql(PPC(uid => $client_uids), [
                    "SELECT c.cid
                       FROM (SELECT distinct u.ClientID
                               FROM users u
                              ", WHERE => {'u.uid' => SHARD_IDS}, "
                            ) t JOIN campaigns c on c.ClientID = t.ClientID
                      ", WHERE => {'c.ManagerUID' => $manager_uid}
                    ]);
}

#======================================================================

=head2 rbac_get_agcamps

  $arrref_agcamps = rbac_get_agcamps(undef, $agency_uid);

  return all campaigns for agency else return []

=cut

sub rbac_get_agcamps
{
    my (undef, $agency_uid) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $agency_uid);
    return [] unless Rbac::has_role($perminfo, $ROLE_AGENCY);

    if ($perminfo->{rep_type} eq $REP_CHIEF || $perminfo->{rep_type} eq $REP_MAIN) {
        return get_one_column_sql(PPC(shard => 'all'), "
                        SELECT c.cid
                          FROM clients cl
                               JOIN campaigns c on c.ClientID = cl.ClientID
                         WHERE c.statusEmpty = 'No'
                           AND c.AgencyID = cl.agency_client_id
                           AND cl.agency_client_id = ?", $perminfo->{ClientID});
    } else {
        return get_one_column_sql(PPC(shard => 'all'), "
                        SELECT c.cid
                          FROM clients cl
                               JOIN campaigns c on c.ClientID = cl.ClientID
                         WHERE c.statusEmpty = 'No'
                           AND c.AgencyID = cl.agency_client_id
                           AND cl.agency_uid = ?
                        UNION
                         SELECT c.cid
                          FROM agency_lim_rep_clients alrc
                               JOIN clients cl USING(ClientID)
                               JOIN campaigns c USING(ClientID)
                         WHERE c.statusEmpty = 'No'
                           AND c.AgencyID = cl.agency_client_id
                           AND alrc.agency_uid = ?", $agency_uid, $agency_uid);
    }
}

#======================================================================

=head2 rbac_role_name

  по нику роли получить её название
  $role_name = rbac_role_name($role_str);
  $role_str is one of - 'super', 'superreader', 'placer', 'support', 'manager', 'agency', 'client', 'media'

=cut

sub rbac_role_name
{
    my $role = shift;

    return $ROLE_NAMES{$role};
}

#======================================================================

=head2 rbac_get_manager_of_agency

  $manager_uid = rbac_get_manager_of_agency(undef, $agency_uid, $interface_type);

  interface_type: 'text' or 'mcb' - for select primary manager on direct or bayan (undef - is direct)

=cut

sub rbac_get_manager_of_agency {
    my (undef, $agency_uid, $interface_type) = @_;

    my $agency_client_id = rbac_get_agency_clientid_by_uid( $agency_uid) || die "ClientID for $agency_uid not found";

    return rbac_mass_get_manager_of_agencies_clientids(undef, [$agency_client_id], $interface_type)->{$agency_client_id};
}

=head2 rbac_mass_get_manager_of_agencies

    my $manager_uid_by_agency_uid = rbac_mass_get_manager_of_agencies(undef, $agencies_uids, $interface_type);

    rbac_mass_get_manager_of_agencies(undef,[1,49,577]) => {
        1   => undef,       # если агентство или менеджер не найдены
        49  => 216476158,
        577 => 104737484
    }

=cut

sub rbac_mass_get_manager_of_agencies {
    my (undef, $agencies_uids, $interface_type) = @_;

    my $agency_client_id_by_uid = rbac_get_agencies_clientids_by_uids($agencies_uids);
    my $manager_uid_by_agency_client_id = rbac_mass_get_manager_of_agencies_clientids(undef,
                                                                                      [uniq values %$agency_client_id_by_uid],
                                                                                      $interface_type
                                                                                      );

    my %result;
    for my $agency_uid (@$agencies_uids) {
        my $manager_uid;

        if (my $agency_client_id = $agency_client_id_by_uid->{$agency_uid}) {
            if (defined $manager_uid_by_agency_client_id->{$agency_client_id}) {
                $manager_uid = $manager_uid_by_agency_client_id->{$agency_client_id};
            }
        }

        $result{$agency_uid} = $manager_uid;
    }

    return \%result;
}

#======================================================================

=head2 rbac_get_manager_of_agency_clientid

  $manager_uid = rbac_get_manager_of_agency_clientid(undef, $agency_client_id, $interface_type);

  interface_type: 'text' or 'mcb' - for select primary manager on direct or bayan (undef - is direct)

=cut

sub rbac_get_manager_of_agency_clientid {
    my (undef, $agency_client_id, $interface_type) = @_;

    return rbac_mass_get_manager_of_agencies_clientids(undef, [$agency_client_id], $interface_type)->{$agency_client_id};
}

=head2 rbac_mass_get_manager_of_agencies_clientids

    $manager_uid_by_agency_client_id = rbac_mass_get_manager_of_agencies_clientids(undef, $agencies_client_ids, $campaign_type);

    $campaign_type: 'text'/'mcb'/... - for select primary manager (undef - is direct)

    rbac_mass_get_manager_of_agencies_clientids(undef, [1,456868,505448]) => {
        1       => undef,       # агенстство с таким ClientID не существует или не удалось определить менеджера
        456868  => 216476158,
        505448  => 104737484
    }

=cut

sub rbac_mass_get_manager_of_agencies_clientids {
    my (undef, $agencies_client_ids, $campaign_type) = @_;

    $campaign_type //= 'text';

    my ($field, $alt_field);
    if (camp_kind_in(type => $campaign_type, 'media')) {
        $field = 'primary_bayan_manager_uid';
        $alt_field = 'primary_manager_uid';
    } elsif ($campaign_type eq 'geo') {
        $field = 'primary_geo_manager_uid';
    } elsif (camp_kind_in(type => $campaign_type, 'text_campaign_in_balance', 'billing_aggregate', 'cpm')) {
        $field = 'primary_manager_uid';
        # для ~ 5.5% агентств, создававшихся в Баяне, прописан только баяновский менеджер
        # при этом они могут и создают кампании в том числе в Директе, имя только баяновского менеджера
        $alt_field = 'primary_bayan_manager_uid';
    } else {
        die "unknown type $campaign_type";
    }

    my $manager_uids = get_hashes_hash_sql(PPC(ClientID => $agencies_client_ids), [
                                            "SELECT ClientID
                                                  , primary_manager_uid
                                                  , primary_bayan_manager_uid
                                                  , primary_geo_manager_uid
                                                  , primary_manager_set_by_idm
                                               FROM clients",
                                              WHERE => {ClientID => SHARD_IDS}
    ]);
    my $all_managers = rbac_get_all_managers_of_agencies_clientids(undef, $agencies_client_ids);

    my %result;
    for my $agency_client_id (@$agencies_client_ids) {
        my $manager_uid = $manager_uids->{$agency_client_id}->{$field}
                          || ($alt_field && $manager_uids->{$agency_client_id}->{$alt_field})
                          || $manager_uids->{$agency_client_id}->{primary_geo_manager_uid};

        my $agency_manager_uid;
        if ($manager_uid && any {$manager_uid == $_} @{ $all_managers->{$agency_client_id} // [] }) {
            $agency_manager_uid = $manager_uid;
        } elsif ($field eq 'primary_manager_uid' && $manager_uids->{$agency_client_id}->{primary_manager_set_by_idm}) {
            #Если менеджер назначен через IDM, считаем его наличие в agency_managers вторичным
            $agency_manager_uid = $manager_uid;
        }

        $result{$agency_client_id} = $agency_manager_uid;
    }

    return \%result;
}

#======================================================================

=head2 rbac_get_all_managers_of_agency_clientid

  $managers_uids_arrayref = rbac_get_all_managers_of_agency_clientid(undef, $agency_client_id);

=cut

sub rbac_get_all_managers_of_agency_clientid
{
    my (undef, $agency_client_id) = @_;

    die "ClientID not found" unless $agency_client_id;

    return rbac_get_all_managers_of_agencies_clientids(undef, [ $agency_client_id ])->{ $agency_client_id } // [];
}

=head2 rbac_get_all_managers_of_agencies_clientids(undef, $agencies_client_ids)

  $hashref = rbac_get_all_managers_of_agencies_clientids(undef, $agencies_client_ids_arr_ref);

  Результат:
      $hashref = { AgencyClientID1 => [ManagerUID1, ManagerUID2], AgencyClientID2 => [ManagerUID3], ... }

=cut

sub rbac_get_all_managers_of_agencies_clientids
{
    my (undef, $agencies_client_ids) = @_;

    return undef unless $agencies_client_ids && @$agencies_client_ids;

    my $uids = get_all_sql(PPC(ClientID => $agencies_client_ids), [
        'SELECT agency_client_id, manager_uid FROM agency_managers',
        WHERE => { agency_client_id => SHARD_IDS }
    ]);

    my %result;
    for my $row ( @$uids ) {
        push @{ $result{ $row->{agency_client_id} } //= [] }, $row->{manager_uid};
    }

    return \%result;
}

#======================================================================

=head2 rbac_assign_manager_to_agency

  $error = rbac_assign_manager_to_agency($rbac, $new_manager_uid, $agency_client_id);
  $error = rbac_assign_manager_to_agency($rbac, $new_manager_uid, $agency_client_id, 'text'); -- прописываем дополнительно менеджера в clients

=cut

sub rbac_assign_manager_to_agency
{
    my ($rbac, $new_manager_uid, $agency_client_id, $manager_type) = @_;

    Rbac::clear_cache();

    Client::bind_manager_to_agency($agency_client_id, $new_manager_uid);

    if ($manager_type) {
        my $primary_manager_type_field = {
            text  => 'primary_manager_uid'
            , mcb => 'primary_bayan_manager_uid'
            , geo => 'primary_geo_manager_uid'
        }->{$manager_type} || 'primary_manager_uid';

        do_update_table(PPC(ClientID => $agency_client_id), "clients", {$primary_manager_type_field => $new_manager_uid}, where => {ClientID => SHARD_IDS});
    }

    return 0;
}

#======================================================================

=head2 rbac_withdraw_manager_from_agency

  $error = rbac_withdraw_manager_from_agency($rbac, $old_manager_uid, $agency_client_id);

=cut

sub rbac_withdraw_manager_from_agency
{
    my ($rbac, $old_manager_uid, $agency_client_id) = @_;

    Rbac::clear_cache();

    Client::unbind_manager_from_agency($agency_client_id, $old_manager_uid);

    return 0;
}

#======================================================================

=head2 rbac_get_managers_of_client

  $managers_uid_arr_ref = rbac_get_managers_of_client(undef, $client_uid);

=cut

sub rbac_get_managers_of_client
{
    my (undef, $uid) = @_;

    my $res = rbac_get_managers_of_clients(undef, [$uid]);

    return $res->{$uid};
}

=head2 rbac_get_managers_of_clients

  $managers_uids_hash_ref = rbac_get_managers_of_client(undef, $client_uids);

  Результат:
      { ClientUID1 => [ManagerUID1], ClientUID2 => [ManagerUID1, ManagerUID2], ClientUID3 => [] }

=cut

sub rbac_get_managers_of_clients
{
    my (undef, $uids) = @_;

    my $arr_uids = ref($uids) eq 'ARRAY' ? $uids : [$uids];

    # инициализируем результат
    my $result = { map { $_ => [] } @$arr_uids };

    my $client_ids = rbac_get_client_clientids_by_uids($uids);
    my %clientid2uid = reverse %$client_ids;

    return $result if ! keys %clientid2uid;

    my $res = rbac_get_managers_of_clients_by_clientids(undef, [values %$client_ids]);

    foreach my $client_id (keys %$res) {
        my $uid = $clientid2uid{$client_id};
        push $result->{$uid}, @{ $res->{$client_id} };
    }

    return $result;
}

=head2 rbac_get_managers_of_clients_by_clientids

    $managers_client_ids_hash_ref = rbac_get_managers_of_clients_by_clientids(undef, $client_ids);

    Результат:
          { ClientID1 => [ManagerUID1], ClientID2 => [ManagerUID1, ManagerUID2], ... }

=cut

sub rbac_get_managers_of_clients_by_clientids
{
    my (undef, $client_ids) = @_;

    return {} unless @$client_ids;

    my $client_managers = get_all_sql(PPC(ClientID => $client_ids), [
                                          "SELECT ClientID, manager_uid FROM client_managers",
                                          WHERE => {ClientID => SHARD_IDS}
                                      ]);

    my %result;
    foreach my $r (@$client_managers) {
        push @{ $result{$r->{ClientID}} }, $r->{manager_uid};
    }

    return \%result;
}

=head2 rbac_get_primary_managers_of_clients_by_clientids

$client_id2primary_manager_uid = rbac_get_primary_managers_of_clients_by_clientids($client_ids);

Результат:
    {ClientID1 => PrimaryManagerUID1, ClientID2 => PrimaryManagerUID2, ...}

=cut

sub rbac_get_primary_managers_of_clients_by_clientids
{
    my ($client_ids) = @_;

    return {} unless @$client_ids;

    return get_hash_sql(PPC(ClientID => $client_ids), [
        "SELECT ClientID, primary_manager_uid FROM clients",
        WHERE => {ClientID => SHARD_IDS}
    ]);
}

=head2 rbac_get_managers_of_client_campaigns

$managers_of_client_campaigns = rbac_get_managers_of_client_campaigns($client_id);

Результат:
    [ManagerUID1, ManagerUID2, ...]

=cut

sub rbac_get_managers_of_client_campaigns
{
    my ($client_id) = @_;
    return get_one_column_sql(PPC(shard => 'all'), "SELECT ManagerUID FROM campaigns WHERE ClientID = ? AND statusEmpty = 'No' AND ManagerUID IS NOT NULL", $client_id);
}

#======================================================================

=head2 rbac_get_clients_of_manager

  $clients_uids_arr_ref = rbac_get_clients_of_manager(undef, $manager_uid);

=cut

sub rbac_get_clients_of_manager
{
    my (undef, $manager_uid) = @_;

    return get_one_column_sql(PPC(shard => 'all'), "
                SELECT u.uid
                  FROM client_managers cm
                       JOIN clients cl on cl.ClientID = cm.ClientID and cl.role = 'client'
                       JOIN users u ON u.ClientID = cl.ClientID
                 WHERE cm.manager_uid = ?", $manager_uid);
}

#======================================================================

=head2 rbac_get_limited_agency_rep_of_client

  $agency_uid = rbac_get_limited_agency_rep_of_client($agency_uid, $client_uid);

=cut

sub rbac_get_limited_agency_rep_of_client($$)
{
    my ($agency_uid, $client_uid) = @_;

    my $lim_ag_of_client = rbac_get_limited_agencies_reps_of_clients_list($agency_uid, [$client_uid]);
    return $lim_ag_of_client->{$client_uid} ? $lim_ag_of_client->{$client_uid}[0] : undef;
}

#======================================================================

=head2 rbac_get_limited_agencies_reps_of_clients_list

  назначенные ограниченные представители агентств по списку клиентов

  $lim_agencies = rbac_get_limited_agencies_reps_of_clients_list($agency_uid, [$client_uid1, $client_uid2]);

  $lim_agencies: {
      $client_uid1 => $lim_ag_rep1
      , $client_uid2 => $lim_ag_rep2
      , ...
  }

=cut

sub rbac_get_limited_agencies_reps_of_clients_list($$%)
{
    my ($agency_uid, $clients_uids, %opts) = @_;

    return {} unless @$clients_uids;

    my $agency_client_id = get_clientid(uid => $agency_uid) or die "ClientID not found for uid = $agency_uid";

    my $limited_uids = Rbac::get_reps(ClientID => $agency_client_id, rep_type => $REP_LIMITED);
    return {} unless @$limited_uids;

    my %limited = map {$_ => 1} @$limited_uids;

    my $clients_reps_raw = Rbac::get_agency_uid_multi(uid => $clients_uids);

    my $clients_reps;
    foreach my $client_uid (keys %$clients_reps_raw) {
        my @rep_uids = $opts{new_lim_rep_schema} ? map { @$_ } values %{$clients_reps_raw->{$client_uid}} : @{$clients_reps_raw->{$client_uid}{old_schema} // []};
        $clients_reps->{$client_uid} = [ grep { exists $limited{$_} } @rep_uids ];
    }
    return hash_grep { @$_ } $clients_reps;
}

#======================================================================

=head2 rbac_get_agencies_of_client

  $agency_uids_of_client_arrref = rbac_get_agencies_of_client(undef, $client_uid);

=cut

sub rbac_get_agencies_of_client
{
    my (undef, $client_uid) = @_;

    my $client_uid2agency_uids = rbac_mass_get_agencies_of_clients(undef, [$client_uid]);
    return $client_uid2agency_uids->{$client_uid};
}

=head2 rbac_mass_get_agencies_of_clients

    $client_uid2agency_uids = rbac_mass_get_agencies_of_clients(undef, \@client_uids);

    $client_uid2agency_uids => {
        $client_uid1 => [$agency_uid1, $agency_uid2, ...],
        $client_uid2 => [$agency_uid3, $agency_uid4, ...],
        $client_uid3 => [], # не связан с агентствами
        ...
    }

=cut

sub rbac_mass_get_agencies_of_clients
{
    my (undef, $client_uids) = @_;

    die "client uids should be an array" unless $client_uids && ref($client_uids) eq 'ARRAY';
    return {} unless @$client_uids;

    # получим назначенных клиенту представителей агентств (возможно, ограниченных)
    my $agency_rep_uid_by_client_rep_uid = Rbac::get_agency_uid_multi(uid => $client_uids);

    # получим всех не ограниченных представителей агентств и назначенного
    my $agency_reps_by_rep_uid = Rbac::get_agency_reps_with_same_rights_multi([ uniq map { @$_ } map {values %$_} values %$agency_rep_uid_by_client_rep_uid ]);

    my %result;
    for my $client_uid ( @$client_uids ) {
        my $rep_uids = [ map {@$_} values %{$agency_rep_uid_by_client_rep_uid->{ $client_uid }} ];

        if ( ! defined $rep_uids ) {
            $result{ $client_uid } = [];
            next;
        }

        $result{ $client_uid } = [ uniq map { @{ $agency_reps_by_rep_uid->{ $_ } // [] } } @$rep_uids ];
    }

    return \%result;
}

=head2 rbac_has_agency

    Для данного $uid'а возвращает логическое значение "есть ли у пользователя агентство"

=cut

sub rbac_has_agency
{
    my ( undef, $uid ) = @_;

    return rbac_mass_has_agency( undef, [ $uid ] )->{ $uid };
}

=head2 rbac_mass_has_agency

    $uid2has_agency = rbac_mass_has_agency(undef, \@uids);
    $uid2has_agency => {
        $agency_uid1 => 1,
        $agency_uid2 => 0,
        ...
    }

=cut

sub rbac_mass_has_agency {
    my ( undef, $uids ) = @_;

    my $client_uid2agency_uids = rbac_mass_get_agencies_of_clients( undef, $uids );

    my %result;
    while ( my ( $client_uid, $agency_uids ) = each %$client_uid2agency_uids ) {
        $result{ $client_uid } = ($agency_uids && @$agency_uids) ? 1 : 0;
    }

    return \%result;
}

#======================================================================

=head2 rbac_get_agencies_for_create_camp_of_client

  $agencies_of_client_arrref = rbac_get_agencies_for_create_camp_of_client(undef, $OPERATOR_UID, $client_uid, $for_import_xls);

  если вызавает менеджер - возвращаем все агентства менеджера
  если вызывает клиент - возвращаем все агентства которые дали право редактирование данному клиенту,

  у всех агентств должна быть прописана "свобода" на данного клиента,
  будут возвращены либо шефы, либо ограниченые представители, если они назначены

  $for_import_xls: 1 - возвращаем для клиента агентства в которые он может импортировать xls

=cut

sub rbac_get_agencies_for_create_camp_of_client
{
    my (undef, $UID, $client_uid, $for_import_xls) = @_;

    my $client_perminfo = Rbac::get_perminfo(uid => $client_uid);
    return [] unless $client_perminfo
        && $client_perminfo->{role} eq $ROLE_CLIENT
        && ($client_perminfo->{agency_uid} // -1) > 0;

    my $ag_uid = $client_perminfo->{agency_uid};

    return [] unless get_allow_agency_bind_client($client_perminfo->{agency_client_id}, $client_perminfo->{ClientID});

    my $operator = Rbac::get_perminfo(uid => $UID);
    if ($operator->{role} eq $ROLE_CLIENT) {
        if ($for_import_xls) {
            # субклиент имеет право импортировать xls
            return Rbac::has_perm($operator, $PERM_XLS_IMPORT) ? [$ag_uid] : [];
        }
        return Rbac::has_perm($operator, $PERM_SUPER_SUBCLIENT) ? [$ag_uid] : [];
    } else {
        my $lim_rep_uids = $client_perminfo->{agency_uids};
        if ($lim_rep_uids && @$lim_rep_uids) {
            my $lim_reps_ownership = rbac_is_owner_of_users($UID, $lim_rep_uids);
            return [ $ag_uid ] if any { $lim_reps_ownership->{$_} } @$lim_rep_uids;
        } else {
            return rbac_is_owner(undef, $UID, $ag_uid) ? [$ag_uid] : [];
        }
    }
}

#======================================================================

=head2 rbac_get_campaigns_for_edit

  $cids_arref = rbac_get_campaigns_for_edit(undef, $OPERATOR_UID, $client_uid);

  возвращаем все кампании которые может редактировать $UID и $uid

=cut

sub rbac_get_campaigns_for_edit
{
    my (undef, $UID, $uid) = @_;

    my $perms = Rbac::get_perminfo(uid => $uid);

    return [] unless $perms && $perms->{ClientID};

    my $cids = get_one_column_sql( PPC( ClientID => $perms->{ClientID} ), [
        'SELECT cid FROM campaigns',
        WHERE => {
            ClientID => SHARD_IDS,
        }
    ]);

    my $can_edit = is_user_allowed_for_action_on_camps( 'edit', $UID, $cids );

    return [ grep { $can_edit->{ $_ } } @$cids ];
}

#======================================================================

=head2 rbac_create_agency

  0 - OK
  1 - System Error
  2 - Trouble with uid

=cut

sub rbac_create_agency
{
    my ($rbac, $UID, $uid, $agency_id) = @_;

    Rbac::clear_cache();

    my $role = rbac_who_is(undef, $uid);
    if ($role !~ /^(client|empty)$/ || is_balance_subclient($uid)) {
        return 2;
    }

    Client::bind_manager_to_agency($agency_id, $UID);

    return 0;
}

#======================================================================

=head2 rbac_create_agency_main_rep

    my $error = rbac_create_agency_main_rep($rbac, $agency_id, $agency_uid);

=cut

sub rbac_create_agency_main_rep
{
    my ($rbac, $agency_id, $agency_uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_create_agency_limited_rep

    my $error = rbac_create_agency_limited_rep($rbac, $agency_id, $agency_uid);

=cut

sub rbac_create_agency_limited_rep
{
    my ($rbac, $agency_id, $agency_uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_drop_agency_rep

    my $error = rbac_drop_agency_rep($rbac, $agency_client_id, $agency_uid);

=cut

sub rbac_drop_agency_rep
{
    my ($rbac, $agency_client_id, $agency_uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_is_owner

  $is_owner = rbac_is_owner(undef, $UID, $uid, %opt);

=cut

sub rbac_is_owner
{
    my (undef, $UID, $uid, %opt) = @_;

    return 0 unless $UID && $uid;

    return rbac_is_owner_of_users( $UID, $uid, %opt )->{ $uid } ? 1 : 0;
}

#======================================================================

=head2 rbac_mass_is_owner

  $is_owner = rbac_mass_is_owner(undef, $UID, $uids, %opt);

  %opt - дополнительные флаги/опции:
        no_team_expand => 1 - не расширяем права менеджера до прав всех менеджеров в группе

  По проверкам повторяет - rbac_is_owner

=cut

sub rbac_mass_is_owner
{
    my (undef, $UID, $uids, %opt) = @_;

    my $res = rbac_is_owner_of_users( $UID, $uids, %opt );

    return ( keys %$res && all { $_ } values %$res ) ? 1 : 0;
}

#======================================================================

=head2 rbac_is_owner_of_users

  $is_owner = rbac_is_owner_of_users($UID, $uids, %opt);

  %opt - дополнительные флаги/опции:
        no_team_expand => 1 - не расширяем права менеджера до прав всех менеджеров в группе

  Результат { uid1 => 1, uid2 => 0, ... }

=cut

sub rbac_is_owner_of_users
{
    my ($UID, $uids, %opt) = @_;

    return {} unless $UID && $uids;

    $uids = [ $uids ] unless ref $uids;

    # для UID-а нашлась информация по права и у всех есть роль
    my $operator_perms = Rbac::get_perminfo(uid => $UID);
    return {} unless $operator_perms && $operator_perms->{role};

    my $uid2perms = Rbac::get_key2perminfo(uid => [ uniq @$uids ]);
    # для всех элементов из uids нашлась информация по права и у каждого есть роль
    return {} unless all { $uid2perms->{ $_ } && $uid2perms->{ $_ }{role} ne $ROLE_EMPTY } @$uids;

    my $operator_role      = $operator_perms->{role};
    my $operator_subrole   = $operator_perms->{subrole};
    my $operator_client_id = $operator_perms->{ClientID};

    my %result = map { $_ => 0 } @$uids;

    if (   $operator_role eq $ROLE_SUPER
        || $operator_role eq $ROLE_SUPERREADER
    ) {
        # суперы 'владеют' всеми
        $result{ $_ } = 1 for @$uids;
    }
    elsif ( $operator_role eq $ROLE_SUPPORT ) {
        # саппорт 'владеет' клиентами, агентствами и менеджерами
        for my $uid ( keys %$uid2perms ) {
            my $role = $uid2perms->{ $uid }{role};

            $result{ $uid } = 1 if $role eq $ROLE_MANAGER
                                || $role eq $ROLE_AGENCY
                                || $role eq $ROLE_CLIENT;
        }
    }
    elsif (    $operator_role eq $ROLE_PLACER
            || $operator_role eq $ROLE_MEDIA
    ) {
        # вешальщики и медипланнеры 'владеет' клиентами, агентствами
        for my $uid ( keys %$uid2perms ) {
            my $role = $uid2perms->{ $uid }{role};

            $result{ $uid } = 1 if $role eq $ROLE_AGENCY
                                || $role eq $ROLE_CLIENT;
        }
    }
    elsif ( $operator_role eq $ROLE_MANAGER ) {

        # Доступ через idm-группы
        my @client_ids = map {$_->{ClientID}} values %$uid2perms;
        my $owned_by_idm_client_ids_idx = { map {$_ => 1} @{rbac_filter_client_ids_by_manager_idm_access(undef, $operator_client_id, \@client_ids)} };

        #Исключим из дальнейших проверок клиентов к которым нашелся доступ через idm-группы
        if (keys %$owned_by_idm_client_ids_idx){
            my %tail;
            foreach my $uid (keys %$uid2perms) {
                if ( $owned_by_idm_client_ids_idx->{$uid2perms->{ $uid }->{ ClientID }} ){
                    $result{ $uid } = 1;
                } else {
                    $tail{ $uid } = $uid2perms->{ $uid };
                }
            }
            if (keys %tail == 0) {
                return \%result;
            }
            $uid2perms = \%tail;
        }

        # менеджер 'владеет' собой
        my %allowed_managers_uid = ( $UID => 1 );

        my $manager_data = get_manager_data( $operator_client_id ) // {};
        my $original_operator_client_id = $operator_client_id;

        # заменим менеджера на его тимлидера (если таковой имеется)
        # аналог $uid = rbac_replace_manager_by_teamlead($rbac, $uid);
        if (   ! $opt{no_team_expand}
            && ! $operator_subrole
            && $manager_data->{supervisor_client_id}
        ) {
            $operator_client_id = $manager_data->{supervisor_client_id};
            $UID = $manager_data->{supervisor_uid};
            $manager_data = get_manager_data( $operator_client_id );
            $allowed_managers_uid{ $UID } = 1;
        }

        for my $uid ( @{ $manager_data->{subordinates_uid} } ) {
            $allowed_managers_uid{ $uid } = 1;
        }

        my @clients_client_id;
        my @agencies_client_id;
        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };

            if ( $perms->{role} eq $ROLE_CLIENT ) {
                push @clients_client_id, $perms->{ClientID};
                push @agencies_client_id, $perms->{agency_client_id} if $perms->{agency_client_id};
            }
            elsif ( $perms->{role} eq $ROLE_AGENCY ) {
                push @agencies_client_id, $perms->{ClientID};
            }
        }
        if ( @agencies_client_id ) {
            #Проверим доступ через группы idm к агенствам и добавим доступные агентства в индекс доступных через idm client_id
            foreach my $agency_client_id (@{rbac_filter_client_ids_by_manager_idm_access(undef, $original_operator_client_id, \@agencies_client_id)}) {
                $owned_by_idm_client_ids_idx->{ $agency_client_id } = 1;
            }
        }

        my $client_managers = {};
        if ( @clients_client_id ) {
            $client_managers = rbac_get_managers_of_clients_by_clientids( undef, [ uniq @clients_client_id ] );
        }

        my $agency_managers = {};
        if ( @agencies_client_id ) {
            $agency_managers = rbac_get_all_managers_of_agencies_clientids( undef, [ uniq @agencies_client_id  ] );
        }

        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };

            if ( $perms->{role} eq $ROLE_CLIENT ) {
                #Если агенство доступно оператору через idm - клиенты агенства тоже доступны
                if ($perms->{agency_client_id} && $owned_by_idm_client_ids_idx->{ $perms->{agency_client_id} }){
                    $result{ $uid } = 1;
                } else {
                    my $client_managers_uid = $client_managers->{ $perms->{ClientID} };
                    my $agency_managers_uid = $perms->{agency_client_id} ? $agency_managers->{ $perms->{agency_client_id} } : undef;

                    $result{ $uid } = 1 if
                        # 'владеет' своими клиентами и клиентами своих подчиненных менеджеров
                        ( $client_managers_uid && any { $allowed_managers_uid{ $_ } } @$client_managers_uid )
                        # 'владеет' клиентами своих агентств и агентств своих подчиненных менеджеров
                        || ( $agency_managers_uid && any { $allowed_managers_uid{ $_ } } @$agency_managers_uid );
                }
            }
            elsif ( $perms->{role} eq $ROLE_AGENCY ) {
                # 'владеет' своими агентствами и агентствами своих подчиненных менеджеров
                my $managers_uid = $agency_managers->{ $perms->{ClientID} };
                $result{ $uid } = 1 if $managers_uid && any { $allowed_managers_uid{ $_ } } @$managers_uid;
            }
            elsif ( $perms->{role} eq $ROLE_MANAGER ) {
                # 'владеет' собой и подчиненными менеджерами
                $result{ $uid } = 1 if exists $allowed_managers_uid{ $uid }
            }
        }
    }
    elsif ( $operator_role eq $ROLE_AGENCY ) {
        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };

            if ( $perms->{role} eq $ROLE_AGENCY ) {
                # представитель агентства может 'владеть' только представителями своего агентства
                $result{ $uid } = 1 if $perms->{ClientID} eq $operator_client_id;
                if ($operator_perms->{rep_type} eq 'limited' && $uid ne $UID && !rbac_has_lim_rep_access_to_other_rep($UID, $uid)) {
                    $result{ $uid } = 0;
                }
            }
            elsif ( $perms->{role} eq $ROLE_CLIENT ) {
                # представитель агентства может 'владеть' только представителями клиентов этого агентства
                if ( $perms->{agency_client_id} && $perms->{agency_client_id} eq $operator_client_id ) {

                    # не ограниченные представители 'владеют' представителями всех клиентов
                    if ( $operator_perms->{rep_type} ne $REP_LIMITED ) {
                        $result{ $uid } = 1;
                    }
                    else {
                        # ограниченный представитель 'владеет' представителями только назначенных ему клиентов
                        $result{ $uid } = 1 if ($perms->{agency_uid} && ($perms->{agency_uid} eq $UID || rbac_has_lim_rep_access_to_client($UID, $perms->{ClientID})));
                    }
                }
            }
        }
    }
    elsif ( $operator_role eq $ROLE_CLIENT ) {

        my $is_freelancer = is_freelancer( $UID );
        my $mcc_client_ids = get_related_client_ids($operator_client_id);

        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };

            # представитель клиента может 'владеть' только представителями того же клиента
            $result{ $uid } = 1 if $perms->{ClientID} eq $operator_client_id;

            # фрилансер может 'владеть' представителями привязанных клиентов
            $result{ $uid } = 1 if $is_freelancer && rbac_is_related_freelancer( undef, $UID, $uid );

            # mcc владеет связанными клиентами
            $result{ $uid } = 1 if $mcc_client_ids && $mcc_client_ids->{$perms->{ClientID}}
        }
    }
    elsif ( $operator_role eq $ROLE_INTERNAL_AD_ADMIN ) {
        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };
            $result{ $uid } = 1 if Rbac::has_perm( $perms, 'internal_ad_product' );
        }
    }
    elsif ( $operator_role eq $ROLE_INTERNAL_AD_MANAGER ) {
        my @uids = keys %$uid2perms;
        my %uid2clientid = map { $_ => $uid2perms->{$_}->{ClientID} } @uids;
        my $access_types_by_client_id = InternalAdManagers::filter_accessible_client_ids( $operator_client_id, [ values %uid2clientid ] );
        $result{$_} = 1 for grep { exists $access_types_by_client_id->{ $uid2clientid{$_} } } @uids;
    }
    elsif ( $operator_role eq $ROLE_INTERNAL_AD_SUPERREADER ) {
        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };
            $result{ $uid } = 1 if Rbac::has_perm( $perms, 'internal_ad_product' );
        }
    }
    elsif ($operator_role eq $ROLE_LIMITED_SUPPORT) {
        my $related_fl_or_mcc_client_ids = mass_get_client_ids_of_related_freelancer_or_control_mcc([keys %$uid2perms]);

        for my $uid ( keys %$uid2perms ) {
            my $perms = $uid2perms->{ $uid };

            my $has_access_via_fl_or_mcc = 0;
            if ($related_fl_or_mcc_client_ids->{$uid}) {
                $has_access_via_fl_or_mcc = any { rbac_is_related_client_support(undef, $UID, $_) } @{$related_fl_or_mcc_client_ids->{$uid}};
            }

            $result{$uid} = $has_access_via_fl_or_mcc
                || ($perms->{agency_client_id} && rbac_is_related_client_support(undef, $UID, $perms->{agency_client_id}))
                || rbac_is_related_client_support(undef, $UID, $perms->{ClientID});
        }
    }

    return \%result;
}

#======================================================================

=head2 rbac_is_owner_of_camp

  $is_owner = rbac_is_owner_of_camp(undef, $uid, $cid);

=cut

sub rbac_is_owner_of_camp
{
    my (undef, $uid, $cid) = @_;

    return rbac_check_allow_show_camps(undef, $uid, [ $cid ])->{ $cid };
}

#======================================================================

=head2 rbac_is_owner_of_camps

  $is_owner = rbac_is_owner_of_camps(undef, $uid, [$cid, $cid2]);

=cut

sub rbac_is_owner_of_camps
{
    my (undef, $uid, $cids) = @_;

    my $result = rbac_check_allow_show_camps(undef, $uid, $cids);

    return ( all { $result->{ $_ } } @$cids ) ? 1 : 0;
}

#======================================================================

=head2 rbac_check_owner_of_camps

  $hash = rbac_check_owner_of_camps(undef, $uid, [$cid1, $cid2, $cid3]);

  Result:
      $hash = { $cid1 => 1, $cid3 => 1 } or {}

=cut

sub rbac_check_owner_of_camps
{
    my ( undef, $uid, $cids ) = @_;

    my $result = rbac_check_allow_show_camps( undef, $uid, $cids );

    return hash_grep( sub { $_[0] }, $result );
}

#======================================================================

=head2 rbac_is_allow_edit_camp

  $is_allow_edit_camp = rbac_is_allow_edit_camp(undef, $uid, $cid);

=cut

sub rbac_is_allow_edit_camp {
    my ( undef, $uid, $cid ) = @_;

    return 0 unless $cid;

    return is_user_allowed_for_action_on_camps( 'edit', $uid, [ $cid ] )->{ $cid } || 0;
}

#======================================================================

=head2 rbac_check_allow_edit_camps

  $cid_result = rbac_check_allow_edit_camps(undef, $uid, [$cid,...]);

  Result:
      $hash = { cid1 => 1, ...}

=cut

sub rbac_check_allow_edit_camps
{
    my ( undef, $uid, $cids ) = @_;

    return {} unless @$cids;

    my $res = is_user_allowed_for_action_on_camps( 'edit', $uid, $cids );

    return hash_grep sub { shift }, $res;
}

=head2 rbac_check_allow_show_stat_camps

  вернуть хеш с кампаниями по которым uid может смотреть статистику

  $cid_result = rbac_check_allow_show_stat_camps(undef, $uid, [$cid,...]);
  result: {cid1 => 1, ...}

=cut

sub rbac_check_allow_show_stat_camps
{
    my (undef, $uid, $cids) = @_;

    return rbac_check_allow_show_camps(undef, $uid, $cids);
}

#======================================================================

=head2 rbac_check_allow_delete_camps

    $results = rbac_check_allow_delete_camps(undef, $uid, [$cid, ...]);

    Result:
        $hash = { cid1 => 1, ... }

=cut

sub rbac_check_allow_delete_camps
{
    my (undef, $uid, $cids) = @_;

    return is_user_allowed_for_action_on_camps( 'delete', $uid, $cids );
}

#======================================================================

=head2 rbac_check_allow_transfer_money_camps

  $cid_result = rbac_check_allow_transfer_money_camps(undef, $uid, [$cid,...]);

  Result:
      $hash = { cid1 => 1, ...}

=cut

sub rbac_check_allow_transfer_money_camps
{
    my ( undef, $uid, $cids ) = @_;

    return {} unless @$cids;

    my $perminfo = Rbac::get_perminfo( uid => $uid );

    # считаем что $uid может быть только субклиента
    return {} unless $perminfo
                && $perminfo->{agency_client_id}
                && ( $perminfo->{role} eq $ROLE_CLIENT );

    my $camps_info = get_hashes_hash_sql( PPC( cid => $cids ), [
        'SELECT cid, ClientID, AgencyID FROM campaigns',
        WHERE => { cid => SHARD_IDS }
    ]);

    my $client_id          = $perminfo->{ClientID};
    my $agency_client_id   = $perminfo->{agency_client_id};
    my $can_transfer_money = Rbac::has_perm( $perminfo, $PERM_MONEY_TRANSFER );

    # клиент является владельцем кампании и, если кампания - агентская, то
    # агентство должно совпадать с текущим агентством клиента и у клиента
    # должно быть право на перенос денег от этого агентства
    my %res;
    for my $cid ( keys %$camps_info ) {
        my $info = $camps_info->{ $cid };


        next if $info->{ClientID} ne $client_id;
        next if $info->{AgencyID} && ( $info->{AgencyID} ne $agency_client_id || ! $can_transfer_money );

        $res{ $cid } = 1;
    }

    return \%res;
}

#======================================================================

=head2 rbac_get_rights_on_campaigns

  $camp_rights = rbac_get_rights_on_campaigns($uid, [123, 456, 789]);

  $camp_rights: {
      68232 => {
                 'DropSCampaign' => 1,
                 'EditCamp' => 1,
                 'ImportXlsBanners' => 1,
               },
      68663 => {
                 'ShowStat' => 1,
                 'UnserviceSCamp' => 1
               }
  }

=cut

sub rbac_get_rights_on_campaigns
{
    my ($uid, $cids) = @_;

    die "rbac_get_rights_on_campaigns: params is not valid" unless ref($cids) eq 'ARRAY' && @$cids;

    my $can_edit = is_user_allowed_for_action_on_camps('edit', $uid, $cids);

    my %result;
    for my $cid ( @$cids ) {
        $result{ $cid }{EditCamp} = 1 if $can_edit->{ $cid };
    }

    return \%result;
}

#======================================================================

=head2 rbac_get_owner_of_camp

  $uid = rbac_get_owner_of_camp(undef, $cid);

=cut

sub rbac_get_owner_of_camp
{
    my (undef, $cid) = @_;

    return get_one_field_sql(PPC(cid => $cid),
                    "SELECT cl.chief_uid
                       FROM campaigns c
                            JOIN clients cl on cl.ClientID = c.ClientID
                      WHERE c.cid = ?", $cid) || 0;
}

#======================================================================

=head2 get_users_by_role_subrole

=cut

sub get_users_by_role_subrole {
    my ( $role, $subrole ) = @_;

    my %where;
    $where{'c.role'}    = $role    if $role;
    $where{'c.subrole'} = $subrole if $subrole;

    return [] unless %where;

    return get_one_column_sql(PPC(shard => 'all'), [
        'SELECT u.uid FROM clients c JOIN users u USING (ClientID)',
        WHERE => \%where,
    ]);
}

=head2 rbac_get_all_teamleaders

  $arr_ref = rbac_get_all_teamleaders(undef);

=cut

sub rbac_get_all_teamleaders
{
    return get_users_by_role_subrole( $ROLE_MANAGER, $SUBROLE_TEAMLEADER );
}

#======================================================================

=head2 rbac_get_all_managers

  $uids_arr_ref = rbac_get_all_managers(undef);

=cut

sub rbac_get_all_managers
{
    return get_users_by_role_subrole( $ROLE_MANAGER );
}

#======================================================================

=head2 rbac_get_all_agencies

  $arr_ref = rbac_get_all_agencies(undef);

=cut

sub rbac_get_all_agencies
{
    return get_users_by_role_subrole( $ROLE_AGENCY );
}

=head2 rbac_get_all_agencies_chief_reps

  $arr_ref = rbac_get_all_agencies_chief_reps(undef);

  Вернуть список всех главных представителей агентств

=cut

sub rbac_get_all_agencies_chief_reps
{
    return get_one_column_sql(PPC(shard => 'all'), "SELECT chief_uid FROM clients WHERE role = ?", 'agency');
}

#======================================================================

=head2 rbac_get_teamleaders_data

  $data = rbac_get_teamleaders_data(undef, $is_superteamleader_uid_or_zero);
  $data:
  {
    teamleader1_uid => [manager1_uid, manager2_uid, manager3_uid...],
    teamleader2_uid => [manager4_uid, manager5_uid, manager6_uid...],
    ...
    'none'          => [manager7_uid, manager8_uid, manager9_uid...] # managers without teamleader
  }

=cut

sub rbac_get_teamleaders_data
{
    my (undef, $is_superteamleader_uid) = @_;

    return _rbac_get_teamleaders_data( undef, $is_superteamleader_uid)
}


#======================================================================

=head2 rbac_get_teamleaders_data_with_idm

  $data = rbac_get_teamleaders_data_with_idm(undef, $is_superteamleader_uid_or_zero);
  $data:
  {
    teamleader1_uid => [manager1_uid, manager2_uid, manager3_uid...],
    teamleader2_uid => [manager4_uid, manager5_uid, manager6_uid...],
    ...
    'none'          => [manager7_uid, manager8_uid, manager9_uid...] # managers without teamleader
  }

=cut

sub rbac_get_teamleaders_data_with_idm
{
    my (undef, $is_superteamleader_uid) = @_;

    return _rbac_get_teamleaders_data( undef, $is_superteamleader_uid, with_idm_groups => 1)
}

#======================================================================

sub _rbac_get_teamleaders_data
{
    my (undef, $is_superteamleader_uid, %opts) = @_;
    my $with_idm_groups = $opts{with_idm_groups} ? 1 : 0;

    my $tuids = rbac_get_all_teamleaders();

    my %result = map {$_ => []} @$tuids;
    my $muids = rbac_get_all_managers();
    my %managers = map {$_ => 'none'} grep {! rbac_is_superteamleader(undef, $_) && ! rbac_is_teamleader(undef, $_)} @$muids;


    my $teamleader_by_muid = rbac_mass_get_teamleaders_of_managers(undef, [ keys %managers ]);
    for my $muid (keys %managers) {
        my $tuid = $teamleader_by_muid->{ $muid };
        $managers{$muid} = $tuid if defined $tuid;
    }

    for my $muid (keys %managers) {
        push @{$result{ $managers{$muid} }}, $muid;
    }

    if ($is_superteamleader_uid) {
        for my $tuid (keys %result) {
            next if $tuid eq 'none';
            delete $result{$tuid} unless rbac_is_owner(undef, $is_superteamleader_uid, $tuid);
        }
    }

    if ( $with_idm_groups ) {
        #Добавим менеджеров, доступных через idm-группы
        my $tclient_id_by_uid = get_key2clientid(uid => $tuids);
        my $idm_managers_by_tclient_id = rbac_get_managers_with_same_groups(undef, [values %$tclient_id_by_uid]);

        for my $tuid (keys %result){
            my $idm_managers = $idm_managers_by_tclient_id->{$tclient_id_by_uid->{$tuid}} // [];
            $result{$tuid} = [ uniq @{$result{$tuid}}, @$idm_managers ];
        }
    }

    return \%result;
}

#======================================================================

=head2 rbac_get_teamleader_of_manager

  $tuid = rbac_get_teamleader_of_manager(undef, $muid);
  return undef if not found or error

=cut

sub rbac_get_teamleader_of_manager
{
    my (undef, $muid) = @_;

    return rbac_mass_get_teamleaders_of_managers(undef, [$muid])->{$muid};
}

#======================================================================

=head2 rbac_mass_get_teamleaders_of_managers

  $manager_uid2teamleader_uid = rbac_mass_get_teamleaders_of_managers(undef, $manager_uids);

=cut

sub rbac_mass_get_teamleaders_of_managers
{
    my (undef, $manager_uids) = @_;

    return get_hash_sql(PPC(uid => $manager_uids), [
                    "SELECT manager_uid, supervisor_uid FROM manager_hierarchy",
                      WHERE => {manager_uid => SHARD_IDS, supervisor_uid__gt => 0}]);
}

#======================================================================

=head2 rbac_replace_manager_by_teamlead

  $tuid = rbac_replace_manager_by_teamlead(undef, $uid);
  return teamlead_uid if $uid is manager and have teamlead, in other case - return $uid

=cut

sub rbac_replace_manager_by_teamlead ($$) {
    my (undef, $uid) = @_;

    # надеемся, что роль закеширована, и её бесплатно проверить
    my $perminfo = Rbac::get_perminfo(uid => $uid);
    if ($perminfo && $perminfo->{role} && $perminfo->{role} eq $Rbac::ROLE_MANAGER && !$perminfo->{subrole}) {
        return rbac_get_teamleader_of_manager(undef, $uid) || $uid;
    } else {
        return $uid;
    }
}

#======================================================================

=head2 rbac_replace_freelancer_by_client_via_cids

  $client_chief_uid = rbac_replace_freelancer_by_client_via_cids(undef, $uid, $cids);
  return client_chief_uid if freelancer, defined by uid has relation with client
  who owned one or more campaigns from list, in other case - return $uid

=cut

sub rbac_replace_freelancer_by_client_via_cids($$$) : Cacheable(
        label => 'rbac_replace_freelancer_by_client_via_cids',
        serialize_by => \&__srlz_explain_cids)
{
    my (undef, $uid, $cids) = @_;

    my $fl_client_id = get_freelancer_client_id_by_uid($uid);
    return $uid unless $fl_client_id;

    my $client_chief_uids = get_one_column_sql(PPC(cid => $cids), [
        q/SELECT DISTINCT c.chief_uid FROM clients c
            JOIN clients_relations r ON (c.ClientID = r.client_id_to AND r.type = 'freelancer')
            JOIN campaigns camp ON (camp.ClientID = c.ClientID)
            /,
        WHERE => {'camp.cid' => SHARD_IDS, 'r.client_id_from' => $fl_client_id}]);

    return $uid unless @$client_chief_uids;

    die sprintf('multiple uids found for freelancer %s, cids: [%s]', $fl_client_id, join(', ', @$cids))
        if @$client_chief_uids > 1;

    return $client_chief_uids->[0];
}

#======================================================================

=head2
    Serializator for rbac_replace_freelancer_by_client_via_cids

    Returns key for caching results of the method call when @$cids <= $MAX_CIDS
    and undef in other case
=cut
sub __srlz_explain_cids {
    state $MAX_CIDS = 10;
    my ($prefix, $args, $opts) = @_;
    my (undef, $uid, $cids) = @$args;

    return undef if (@$cids > $MAX_CIDS);

    return join ':', ($prefix, $uid, sort {$a <=> $b} @$cids);
}

#======================================================================

=head2 rbac_is_related_freelancer

    returns 1 when $UID belongs to client($uid) related freelancer, 0 in another case

=cut

sub rbac_is_related_freelancer($$$) :Cacheable(
        ignore_args => [0],
        label => 'rbac_is_related_freelancer')
{
    my (undef, $UID, $uid) = @_;

    my $fl_client_id = get_freelancer_client_id_by_uid($UID);

    return 0 unless $fl_client_id;
    return get_one_field_sql(PPC(uid => $uid), [
            q/SELECT 1 FROM users u JOIN clients_relations r ON (u.ClientID = r.client_id_to AND r.type = 'freelancer')/,
            WHERE => {'u.uid' => SHARD_IDS, 'r.client_id_from' => $fl_client_id}
    ]) // 0;
}

=head2 rbac_is_related_client

    возвращает флаг наличия связи между оператором и клиентом (фрилансер или управляющий аккаунт MCC)

=cut

sub rbac_is_related_client($$$) {
    return rbac_get_client_relation_type(@_) ? 1 : 0;
}

=head2 rbac_get_client_relation_type

    возвращает тип связи если $operator_client_id является связанным c клиентом (фрилансер или управляющий аккаунт MCC)

=cut

sub rbac_get_client_relation_type($$$) :Cacheable(
        ignore_args => [0],
        label => 'rbac_get_client_relation_type')
{
    my (undef, $operator_client_id, $client_id) = @_;

    return get_one_field_sql(PPC(ClientID => $client_id), [
            'SELECT type FROM clients_relations',
            WHERE => { client_id_to => SHARD_IDS, client_id_from => $operator_client_id, type => [qw/freelancer mcc/]}
    ]) // '';
}

#======================================================================

=head2 rbac_is_related_client_support

    returns 1 when $UID belongs to client($client_id) related clients support, 0 in another case

=cut

sub rbac_is_related_client_support($$$) :Cacheable(
        ignore_args => [0],
        label => 'rbac_is_related_client_support')
{
    my (undef, $UID, $client_id) = @_;

    if ( Property->new('enable_limited_support_read_all_clients')->get() ) {
        my $client_perminfo = Rbac::get_perminfo(ClientID => $client_id);
        my $result = ( Rbac::has_role($client_perminfo, $ROLE_CLIENT)
                    || Rbac::has_role($client_perminfo, $ROLE_AGENCY)
        ) ? 1 : 0;
        return $result;
    }

    my $operator_perm =  Rbac::get_perminfo(uid => $UID);

    return get_one_field_sql(PPC(ClientID => $client_id), [
            q/SELECT 1 FROM clients_relations r/,
            WHERE => {
                'r.type' => 'support_for_client',
                'r.client_id_to' => SHARD_IDS,
                'r.client_id_from' => $operator_perm->{ClientID}
            }]) // 0;
}

#======================================================================

=head2 rbac_get_uid_of_related_freelancer

    returns uid of client freelancer

=cut

sub rbac_get_uid_of_related_freelancer($$) :Cacheable(
    ignore_args => [0],
    label => 'rbac_get_uid_of_related_freelancer')
{
    my (undef, $uid) = @_;

    return mass_get_uid_of_related_freelancer([$uid])->{$uid};
}

#======================================================================

=head2 mass_get_uid_of_related_freelancer

    returns hashref
        {uid1 => freelancerUID1, uid2 => freelancerUID2, ... }

    If source uid hasn't related freelancer, result doesn't contain it.

=cut

sub mass_get_uid_of_related_freelancer($)
{
    my ($client_uids) = @_;

    my $fl_client_id_by_client_uid = get_hash_sql(PPC(uid => $client_uids), [
            q/SELECT u.uid, r.client_id_from FROM users u JOIN clients_relations r ON (u.ClientID = r.client_id_to AND r.type = 'freelancer')/,
            WHERE => {'u.uid' => SHARD_IDS}
    ]);

    return {} unless keys %$fl_client_id_by_client_uid;

    my $result;
    my $chief_uid_by_fl_client_id = Rbac::get_chiefs_multi(ClientID => [values %$fl_client_id_by_client_uid]);

    foreach my $uid (keys %$fl_client_id_by_client_uid) {
        $result->{$uid} = $chief_uid_by_fl_client_id->{$fl_client_id_by_client_uid->{$uid}}
    }

    return $result;
}

#======================================================================

=head2 mass_get_client_ids_of_related_freelancer_or_control_mcc

    функция возвращает списки идентификаторов клиента фрилансеров или управляющих МСС, связанных с заданными uid'ами

=cut

sub mass_get_client_ids_of_related_freelancer_or_control_mcc() {
    my ($uids) = @_;

    my $rows = get_all_sql(PPC(uid => $uids), [
            q/SELECT u.uid, r.client_id_from FROM users u JOIN clients_relations r ON u.ClientID = r.client_id_to/,
            WHERE => {'u.uid' => SHARD_IDS, 'r.type' => [qw/freelancer mcc/]}
    ]);

    my $result = {};

    foreach my $row (@$rows) {
        push @{$result->{$row->{uid}}}, $row->{client_id_from};
    }

    return $result;
}

#======================================================================

=head2 get_freelancer_client_id_by_uid

    returns freelancer client_id by given freelancers uid
    or 0 if client hasn't freelancer support

=cut

sub get_freelancer_client_id_by_uid($) :Cacheable(
        label => 'get_freelancer_client_id_by_uid')
{
    my ($uid) = @_;

    return get_one_field_sql(PPC(uid => $uid), [
            q/SELECT fl.ClientID FROM users u JOIN freelancers fl ON (u.ClientID = fl.ClientID)/,
            WHERE => {uid => SHARD_IDS}]);
}

#======================================================================

=head2 has_freelancer

    returns 1 when given client supported by freelancer, 0 in another case

=cut

sub has_freelancer($)
{
    my ($uid) = @_;

    return rbac_get_uid_of_related_freelancer(undef, $uid) ? 1 : 0;
}

#======================================================================

=head2 is_freelancer

    returns 1 when given freelancers uid and 0 in another case

=cut

sub is_freelancer($)
{
    my ($uid) = @_;

    return get_freelancer_client_id_by_uid($uid) ? 1 : 0;
}

=head2 get_related_client_ids

    возаращает хэш связанных client_id (для mcc) для управляющего client_id

=cut

sub get_related_client_ids($) :Cacheable(
        label => 'get_related_client_ids') {
    my ($client_id) = @_;

    return { map { $_ => 1 } @{get_one_column_sql(PPC(ClientID => $client_id), ["SELECT client_id_to FROM reverse_clients_relations",
        WHERE => {client_id_from => SHARD_IDS, type => ['mcc']}])}};
}

#======================================================================

=head2 rbac_get_managers_of_teamleader_with_idm

  $muid_arrref = rbac_get_managers_of_teamleader_with_idm(undef, $tuid);
  return undef if not found or error

=cut

sub rbac_get_managers_of_teamleader_with_idm
{
    my (undef, $tuid) = @_;

    return rbac_mass_get_managers_of_teamleaders( undef, [ $tuid ], with_idm_groups => 1 )->{ $tuid };
}

#======================================================================

=head2 rbac_get_managers_of_teamleader

  $muid_arrref = rbac_get_managers_of_teamleader(undef, $tuid);
  return undef if not found or error

=cut

sub rbac_get_managers_of_teamleader
{
    my (undef, $tuid) = @_;

    return rbac_mass_get_managers_of_teamleaders( undef, [ $tuid ])->{ $tuid };
}

#======================================================================

=head2 rbac_mass_get_managers_of_teamleaders

    $teamleader_uid2manager_uids = rbac_mass_get_managers_of_teamleaders(undef, \@teamleader_uids,  with_idm_groups => 0|1 );

    $teamleader_uid2manager_uids => {
        $teamleader_uid1 => [$manager_uid1, $manager_uid2, ...],
        ...
    }

=cut

sub rbac_mass_get_managers_of_teamleaders
{
    my (undef, $teamleader_uids, %opt) = @_;

    my $with_idm_groups = $opt{with_idm_groups} ? 1 : 0;
    my %teamleader_uid_by_client_id = reverse %{get_key2clientid(uid => $teamleader_uids)};
    my @client_ids = keys %teamleader_uid_by_client_id;

    my $manager_data = Rbac::mass_get_manager_data( \@client_ids );
    my $idm_managers_by_teamleader_client_id = $with_idm_groups ? rbac_get_managers_with_same_groups( undef, \@client_ids ) : {};

    my %result;
    for my $client_id (@client_ids) {
        my $uid = $teamleader_uid_by_client_id{$client_id};
        my $idm_manager_uids = $idm_managers_by_teamleader_client_id->{$client_id} // [];
        my ($manager_data_item, $manager_data_uids) = ($manager_data->{$client_id}, []);
        if (($manager_data_item->{subrole} // '' ) eq $Rbac::SUBROLE_TEAMLEADER) {
            $manager_data_uids = $manager_data_item->{subordinates_uid};
        }
        $result{$uid} = [ uniq @{$result{$uid} // []}, @$idm_manager_uids, @$manager_data_uids];
    }

    return \%result;
}

#======================================================================

=head2 rbac_add_manager_to_teamleader

  $errorcode = rbac_add_manager_to_teamleader($rbac, $muid, $tuid);
  return false if not error

=cut

sub rbac_add_manager_to_teamleader
{
    my ($rbac, $muid, $tuid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_delete_manager_from_teamleader

  $errorcode = rbac_delete_manager_from_teamleader($rbac, $muid, $tuid);
  return false if not error

=cut

sub rbac_delete_manager_from_teamleader
{
    my ($rbac, $muid, $tuid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_move_manager_between_teamleaders

  $errorcode = rbac_move_manager_between_teamleaders($rbac, $muid, $tuid_old, $tuid_new);
  return false if not error

=cut

sub rbac_move_manager_between_teamleaders
{
    my ($rbac, $muid, $tuid_old, $tuid_new) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_offer_to_servicing

  $errorcode = rbac_offer_to_servicing(undef, $cid);

=cut

sub rbac_offer_to_servicing
{
    my (undef, $cid) = @_;

    do_insert_into_table(PPC(cid => [$cid]), 'camps_for_servicing', {cid => $cid}, ignore => 1);

    return 0;
}

#======================================================================

=head2 rbac_decline_servicing

  $errorcode = rbac_decline_servicing($rbac, $cid);

=cut

sub rbac_decline_servicing
{
    my ($rbac, $cid) = @_;

    Rbac::clear_cache();

    do_delete_from_table(PPC(cid => [$cid]), 'camps_for_servicing', where => {cid => $cid});

    return 0;
}

#======================================================================

=head2 rbac_make_super_subclient

  rbac_make_super_subclient($rbac, $agency_client_id, $client_uid);

=cut

sub rbac_make_super_subclient
{
    my ($rbac, $agency_client_id, $client_uid) = @_;

    Rbac::clear_cache();

    my $client_id = rbac_get_client_clientid_by_uid($client_uid) || die "ClientID for $client_uid not found";

    Rbac::set_client_perm($client_id, $PERM_SUPER_SUBCLIENT => 1);

    return 0;
}

#======================================================================

=head2 rbac_make_simple_subclient

  rbac_make_simple_subclient($rbac, $agency_client_id, $client_uid);

=cut

sub rbac_make_simple_subclient
{
    my ($rbac, $agency_client_id, $client_uid) = @_;

    Rbac::clear_cache();

    my $client_id = rbac_get_client_clientid_by_uid($client_uid) || die "ClientID for $client_uid not found";

    Rbac::set_client_perm($client_id, $PERM_SUPER_SUBCLIENT => 0, $PERM_XLS_IMPORT => 0);

    return 0;
}

#======================================================================

=head2 rbac_is_supersubclient

  $is_supersubclient = rbac_is_supersubclient(undef, $agency_client_id, $client_client_id);

=cut

sub rbac_is_supersubclient
{
    my (undef, $agency_client_id, $client_client_id) = @_;

    my $perms = Rbac::get_perminfo(ClientID => $client_client_id);

    return   $perms->{role} && $perms->{role} eq $Rbac::ROLE_CLIENT
          && $perms->{agency_client_id} && $perms->{agency_client_id} eq $agency_client_id
          && $perms->{perms} && Rbac::has_perm($perms, $Rbac::PERM_SUPER_SUBCLIENT) ? 1 : 0;
}

#======================================================================

=head2 rbac_set_allow_import_xls

  rbac_set_allow_import_xls($rbac, $agency_client_id, $subclient_uid, $allow_import_xls);

=cut

sub rbac_set_allow_import_xls
{
    my ($rbac, $agency_client_id, $subclient_uid, $allow_import_xls) = @_;

    Rbac::clear_cache();

    my $client_id = rbac_get_client_clientid_by_uid($subclient_uid) || die "ClientID for $subclient_uid not found";

    my $perminfo = Rbac::get_perminfo(uid => $subclient_uid);

    return 1 unless    $perminfo
        && ($perminfo->{role} // '') eq $ROLE_CLIENT
        && ($perminfo->{agency_client_id} // 0) eq $agency_client_id
        && Rbac::has_perm($perminfo, $PERM_SUPER_SUBCLIENT);

    Rbac::set_client_perm($client_id, $PERM_XLS_IMPORT => $allow_import_xls);

    return 0;
}

=head2 rbac_get_allow_import_xls

    $res = rbac_get_allow_import_xls(undef, $agency_client_id, $subclient_uid);

    $res = true|false

=cut
sub rbac_get_allow_import_xls {
    my ( undef, $agency_uid, $subclient_clientid ) = @_;

    my $agency_perminfo = Rbac::get_perminfo( uid => $agency_uid );
    return 0 unless $agency_perminfo
                && ($agency_perminfo->{role} // '') eq $ROLE_AGENCY;

    my $client_perminfo = Rbac::get_perminfo( ClientID => $subclient_clientid );
    return 0 unless  $client_perminfo
                && ( ($client_perminfo->{role} // '') eq $ROLE_CLIENT )
                && ( ($client_perminfo->{agency_client_id} // -1) eq $agency_perminfo->{ClientID} );

    return Rbac::has_perm( $client_perminfo, $PERM_XLS_IMPORT );
}

#======================================================================

=head2 rbac_get_allow_agency_mass_advq

  my $allow_agency_mass_advq = rbac_get_allow_agency_mass_advq($agency_uid);

=cut

sub rbac_get_allow_agency_mass_advq
{
    my ($agency_uid) = @_;

    my $perminfo = Rbac::get_perminfo( uid => $agency_uid );

    return 0 unless $perminfo && ($perminfo->{role} // '') eq $ROLE_AGENCY;

    return Rbac::has_perm( $perminfo, $PERM_MASS_ADVQ );
}

#======================================================================

=head2 rbac_set_allow_agency_mass_advq

  rbac_set_allow_agency_mass_advq($rbac, $agency_uid, $allow_agency_mass_advq);

=cut

sub rbac_set_allow_agency_mass_advq
{
    my ($rbac, $agency_uid, $allow_agency_mass_advq) = @_;

    Rbac::clear_cache();

    my $agency_id = rbac_get_agency_clientid_by_uid( $agency_uid) || die "ClientID for $agency_uid not found";

    Rbac::set_client_perm($agency_id, $PERM_MASS_ADVQ => $allow_agency_mass_advq);

    return 0;
}

#======================================================================

=head2 rbac_unbind_manager

  unbind manager if manager not have campaigns for client

  rbac_unbind_manager(undef, $manager_uid, $client_uid);

=cut

sub rbac_unbind_manager
{
    my (undef, $manager_uid, $client_uid) = @_;

    # TODO: нужна ли проверка что оператор может отвязывать менеджера?
    # return 0 unless $rbac->HasAccess($rbac->{UID}, 'RBAC_UnbindManager', $manager_uid);

    my $client_id = rbac_get_client_clientid_by_uid($client_uid) || die "ClientID for $client_uid not found";

    my $all_clients_uids = rbac_get_client_uids_by_clientid($client_id);
    my $cids = rbac_get_scamps_of_managers_and_client_rep(undef, $manager_uid, $all_clients_uids);
    return 0 if @$cids;

   Client::unbind_manager($client_id, $manager_uid);

    return 0; # no rbac errors, but not unbind manager
}

#======================================================================

=head2 rbac_unbind_agency

    unbind agency if agency not have campaigns for client

    rbac_unbind_agency($rbac, $agency_uid, $client_uid);

    Используется в песочнице для сбрасывания логинов агентств

=cut

sub rbac_unbind_agency
{
    my ($rbac, $agency_uid, $client_uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_is_super_placer

  rbac_is_super_placer(undef, $uid);

=cut

sub rbac_is_super_placer
{
    my (undef, $uid) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $uid);

    return $perminfo && ($perminfo->{role}//'') eq $ROLE_PLACER && ($perminfo->{subrole}//'') eq $SUBROLE_SUPERPLACER ? 1 : 0;
}

#======================================================================

=head2 rbac_create_placer

    $result = rbac_create_placer($rbac, $uid);

=cut

sub rbac_create_placer
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $placer_client_id = get_clientid(uid => $uid) or die "ClientID for $uid not found";

    my $errcode = rbac_drop_client($rbac, $uid);
    return $errcode if $errcode;

    return 0;
}

#======================================================================

=head2 rbac_create_superplacer

  $result = rbac_create_superplacer($rbac, $uid);

=cut

sub rbac_create_superplacer
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $placer_client_id = get_clientid(uid => $uid) or die "ClientID for $uid not found";

    my $errcode = rbac_drop_client($rbac, $uid);
    return $errcode if $errcode;

    return 0;
}

#======================================================================

=head2 rbac_drop_placer

  $result = rbac_drop_placer($rbac, $uid);

=cut

sub rbac_drop_placer
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_drop_superplacer

  $result = rbac_drop_superplacer($rbac, $uid);

=cut

sub rbac_drop_superplacer
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_get_allow_transfer_money

  my $allow_transfer_money = rbac_get_allow_transfer_money(undef, $subclient_uid, $cid);

=cut

sub rbac_get_allow_transfer_money
{
    my (undef, $subclient_uid, $cid) = @_;

    return rbac_check_allow_transfer_money_camps(undef, $subclient_uid, [ $cid ])->{ $cid } ? 1 : 0;
}

#======================================================================

=head2 rbac_set_allow_transfer_money

  $result = rbac_set_allow_transfer_money($rbac, $agency_client_id, $subclient_uid, $allow_transfer_money);

=cut

sub rbac_set_allow_transfer_money
{
    my ($rbac, $agency_client_id, $subclient_uid, $allow_transfer_money) = @_;

    Rbac::clear_cache();

    my $subclient_client_id = rbac_get_client_clientid_by_uid($subclient_uid) || die "ClientID for $subclient_uid not found";

    Rbac::set_client_perm($subclient_client_id, $PERM_MONEY_TRANSFER => $allow_transfer_money);

    return 0;
}

#======================================================================

=head2 rbac_is_super_media_planner

  rbac_is_super_media_planner(undef, $uid);

=cut

sub rbac_is_super_media_planner
{
    my (undef, $uid) = @_;

    return 0 if !defined($uid) || $uid==0;

    my $perminfo = Rbac::get_perminfo(uid => $uid);

    return $perminfo && ($perminfo->{role}//'') eq $ROLE_MEDIA && ($perminfo->{subrole}//'') eq $SUBROLE_SUPERMEDIA ? 1 : 0;
}

#======================================================================

=head2 rbac_create_supermedia

  $result = rbac_create_supermedia($rbac, $uid);

=cut

sub rbac_create_supermedia
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $media_client_id = get_clientid(uid => $uid) or die "ClientID for $uid not found";

    my $errcode = rbac_drop_client($rbac, $uid);
    return $errcode if $errcode;

    return 0;
}

#======================================================================


=head2 rbac_drop_supermedia

  $result = rbac_drop_supermedia($rbac, $uid);

=cut

sub rbac_drop_supermedia
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_is_teamleader

  $is_teamleader = rbac_is_teamleader(undef, $uid);

=cut

sub rbac_is_teamleader
{
    my (undef, $uid) = @_;

    my $perms = Rbac::get_perminfo(uid => $uid);

    return   $perms->{role}    && $perms->{role}    eq $Rbac::ROLE_MANAGER
          && $perms->{subrole} && $perms->{subrole} eq $Rbac::SUBROLE_TEAMLEADER ? 1 : 0;
}

#======================================================================

=head2 rbac_is_superteamleader

  $is_superteamleader = rbac_is_superteamleader(undef, $uid);

=cut

sub rbac_is_superteamleader
{
    my (undef, $uid) = @_;

    my $perms = Rbac::get_perminfo(uid => $uid);

    return   $perms->{role}    && $perms->{role}    eq $Rbac::ROLE_MANAGER
          && $perms->{subrole} && $perms->{subrole} eq $Rbac::SUBROLE_SUPERTEAMLEADER ? 1 : 0;
}

#======================================================================

=head2 rbac_get_all_superteamleaders

  $arr_ref = rbac_get_all_superteamleaders(undef);

=cut

sub rbac_get_all_superteamleaders
{
    return get_users_by_role_subrole( $ROLE_MANAGER, $SUBROLE_SUPERTEAMLEADER );
}

#======================================================================

=head2 rbac_get_team_of_superteamleader

  $team_arrref = rbac_get_team_of_superteamleader(undef, $st_uid);

=cut

sub rbac_get_team_of_superteamleader
{
    my (undef, $st_uid) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $st_uid);
    return [] unless $perminfo && $perminfo->{role} eq $ROLE_MANAGER && ($perminfo->{subrole} // '') eq $SUBROLE_SUPERTEAMLEADER;

    return get_one_column_sql(PPC(shard => 'all'), "SELECT manager_uid FROM manager_hierarchy WHERE supervisor_uid = ?", $st_uid);
}

#======================================================================

=head2 rbac_get_superteamleader_of_team

  $superteamleader_uid = rbac_get_superteamleader_of_team(undef, $t_uid);

=cut

sub rbac_get_superteamleader_of_team
{
    my (undef, $tuid) = @_;

    my $supervisor_uid = get_one_field_sql(PPC(uid => $tuid), "SELECT supervisor_uid FROM manager_hierarchy WHERE manager_uid = ?", $tuid);
    return undef unless $supervisor_uid;

    my $perminfo = Rbac::get_perminfo(uid => $supervisor_uid);
    return $perminfo && $perminfo->{role} eq $ROLE_MANAGER && ($perminfo->{subrole} // '') eq $SUBROLE_SUPERTEAMLEADER ? $supervisor_uid : undef;
}

#======================================================================

=head2 rbac_get_superteamleaders_data

  $data = rbac_get_superteamleaders_data(undef);

  $data: like in rbac_get_superteamleaders_data()

=cut

sub rbac_get_superteamleaders_data
{
    my $tuids = rbac_get_all_superteamleaders();
    my %result = map {$_ => []} @$tuids;

    my $muids = rbac_get_all_teamleaders();
    my %managers = map {$_ => 'none'} @$muids;

    for my $muid (keys %managers) {
        my $tuid = rbac_get_superteamleader_of_team(undef, $muid);
        $managers{$muid} = $tuid if defined $tuid;
    }

    for my $muid (keys %managers) {
        push @{$result{ $managers{$muid} }}, $muid;
    }

    return \%result;
}

#======================================================================

=head2 rbac_move_team_between_superteamleaders

  $errorcode = rbac_move_team_between_superteamleaders($rbac, $t_uid, $stuid_old, $stuid_new);
  return false if not error

=cut

sub rbac_move_team_between_superteamleaders
{
    my ($rbac, $t_uid, $stuid_old, $stuid_new) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_add_team_to_superteamleader

  $errorcode = rbac_add_team_to_superteamleader($rbac, $tuid, $stuid);
  return false if not error

=cut

sub rbac_add_team_to_superteamleader
{
    my ($rbac, $tuid, $stuid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_delete_team_from_superteamleader

  $errorcode = rbac_delete_team_from_superteamleader($rbac, $tuid, $stuid);
  return false if not error

=cut

sub rbac_delete_team_from_superteamleader
{
    my ($rbac, $tuid, $stuid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_get_my_managers

  get managers of supers/super-teamleader/teamleader/manager

  $man_arrref = rbac_get_my_managers( undef, $UID );

=cut

sub rbac_get_my_managers
{
    my ( undef, $uid ) = @_;

    return _rbac_get_my_managers(undef, $uid);
}

#======================================================================

=head2 rbac_get_my_managers_with_idm

  get managers of supers/super-teamleader/teamleader/manager

  $man_arrref = rbac_get_my_managers_with_idm( undef, $UID );

=cut

sub rbac_get_my_managers_with_idm
{
    my ( undef, $uid ) = @_;

    return _rbac_get_my_managers(undef, $uid, with_idm_groups => 1 );
}

#======================================================================

sub _rbac_get_my_managers
{
    my ( undef, $uid, %opt ) = @_;
    my $with_idm_groups = $opt{with_idm_groups} ? 1 : 0;

    my $perminfo = Rbac::get_perminfo( uid => $uid );
    my $role     = $perminfo ? $perminfo->{role} : '';

    return [] if ! $role || $role eq $ROLE_EMPTY;

    my $managers = [];
    if ( $role eq $ROLE_SUPER || $role eq $ROLE_SUPERREADER || $role eq $ROLE_SUPPORT ) {
        $managers = rbac_get_all_managers();
    } elsif ( rbac_is_teamleader( undef, $uid ) ) {
        $managers = $with_idm_groups ? rbac_get_managers_of_teamleader_with_idm( undef, $uid )
            :  rbac_get_managers_of_teamleader( undef, $uid );
        push @$managers, $uid;
    } elsif ( rbac_is_superteamleader( undef, $uid ) ) {
        my $my_team = rbac_get_team_of_superteamleader( undef, $uid );
        for my $tuid ( @$my_team ) {
            push @$managers, $tuid;
            my $my_muids = $with_idm_groups ? rbac_get_managers_of_teamleader_with_idm( undef, $tuid )
                :  rbac_get_managers_of_teamleader( undef, $tuid );
            if ( $my_muids && @$my_muids ) {
                push @$managers, @$my_muids;
            }
        }
        push @$managers, $uid;
    } elsif ( $role eq $ROLE_MANAGER ) {
        push @$managers, $uid;
    }

    return $managers;
}

#======================================================================

=head2 rbac_get_my_agencies

  get agencies of supers/super-teamleader/teamleader/manager/agencies
  возвращаются все представители агентств

  $result = rbac_get_my_agencies( undef, $UID );

  Результат:
        ссылка на массив uid-ов представителей агентств

=cut

sub rbac_get_my_agencies
{
    my ( undef, $uid ) = @_;

    my $perminfo = Rbac::get_perminfo( uid => $uid );
    my $role     = $perminfo ? $perminfo->{role} : '';

    return [] if ! $role || $role eq $ROLE_EMPTY;

    my $agencies;
    if ( $role eq $ROLE_SUPER || $role eq $ROLE_SUPERREADER ) {
        $agencies = rbac_get_all_agencies();
    } elsif ( $role eq $ROLE_AGENCY ) {
        $agencies = [];
        if ( $perminfo->{rep_type} eq $REP_LIMITED ) {
            push @$agencies, $uid;
        } else {
            my $reps = Rbac::get_reps( ClientID => $perminfo->{ClientID}, role => $ROLE_AGENCY );
            push @$agencies, @$reps;
        }
    } else {
        my $manager_uids = rbac_get_my_managers( undef, $uid );
        return [] unless @$manager_uids;

        my $agencies_by_manager = rbac_mass_get_agencies_clientids_of_managers( $manager_uids );
        my $agencies_client_ids = [ uniq map { @$_ } values %$agencies_by_manager ];
        return [] unless @$agencies_client_ids;

        my $reps_by_agency = Rbac::get_reps_multi(
                                    ClientID => $agencies_client_ids,
                                    role => $ROLE_AGENCY
                                );
        $agencies = [ uniq map { @$_ } values %$reps_by_agency ];
    }

    return $agencies;
}

#======================================================================

=head2 rbac_drop_client

  drop empty client
  $error_code = rbac_drop_client($rbac, $client_uid);

=cut

sub rbac_drop_client
{
    my ($rbac, $client_uid) = @_;

    Rbac::clear_cache();

    if (rbac_who_is(undef, $client_uid) eq $ROLE_CLIENT) {

        # client must be empty (without campaigns)
        return 1 if get_one_field_sql(PPC(uid => $client_uid), "select count(*) from campaigns where uid = ? and statusEmpty = 'No'", $client_uid);

        # удаляем оставшиеся кампании из rbac-а (в директе они со statusEmpty = 'Yes')
        my $errcode;
        my $cids = rbac_get_campaigns_for_edit(undef, $client_uid, $client_uid);
        for my $cid (@$cids) {
            $errcode = rbac_delete_campaign($rbac, $cid, $client_uid);
            return $errcode if $errcode;
        }

        return 0;
    }

    return 0;
}

#======================================================================

=head2 rbac_get_camp_wait_servicing

  get RBAC_AcceptServicing perm for campaign
  $camp_is_wait_servising = rbac_get_camp_wait_servicing(undef, $cid);

=cut

sub rbac_get_camp_wait_servicing
{
    my (undef, $cid) = @_;

    return rbac_mass_get_camps_wait_servicing([ $cid ])->{ $cid } ? 1 : 0;
}

=head2 rbac_mass_get_camps_wait_servicing

  $camps_is_wait_servicing = rbac_mass_get_camps_wait_servicing($cids);

=cut

sub rbac_mass_get_camps_wait_servicing
{
    my ( $cids ) = @_;

    my $res = get_one_column_sql(PPC(cid => $cids), [
        'SELECT cid FROM camps_for_servicing',
        where => { cid => $cids }
    ]);

    return { map { $_ => 1 } @$res };
}

#======================================================================

=head2 rbac_get_allow_xls_import_to_campaign

    $allow_xls = rbac_get_allow_xls_import_to_campaign(undef, $UID, undef, $cid);

    Проверяет, что $UID может импортировать XLS файл в кампанию $cid, принадлежащую $uid.
    Возвращает 0 если импортировать можно, и код 1 если нельзя.

=cut

sub rbac_get_allow_xls_import_to_campaign
{
    my ( undef, $UID, undef, $cid ) = @_;

    # инвертируем результат тк is_user_allowed_for_action_on_camps возвращает 1
    # когда действие разрешено и 0 - когда действие запрещено, а в этой функции
    # нужно наоборот

    return ! is_user_allowed_for_action_on_camps( 'import_xls', $UID, [ $cid ] )->{ $cid };
}

#======================================================================

=head2 rbac_get_allow_xls_import_camp_list

    my $cids = rbac_get_allow_xls_import_camp_list(undef, $UID, $uid);

    возвращаем все кампании в которые можно импортировать xls
    return: arrayref of campaigns' ids

=cut

sub rbac_get_allow_xls_import_camp_list
{
    my ( undef, $UID, $uid ) = @_;

    my $perms = Rbac::get_perminfo(uid => $uid);

    return [] unless $perms && $perms->{ClientID};

    my $cids = get_one_column_sql( PPC( ClientID => $perms->{ClientID} ), [
        'SELECT cid FROM campaigns',
        WHERE => {
            ClientID    => SHARD_IDS,
            type        => Campaign::Types::get_camp_kind_types('xls'),
            statusEmpty => 'No',
            source__not_in  => ['uac', 'zen', 'widget']
        }
    ]);

    my $can_import_xls = is_user_allowed_for_action_on_camps( 'import_xls', $UID, $cids );

    return [ grep { $can_import_xls->{ $_ } } keys %$can_import_xls ];
}

#======================================================================
# работа с представителями
#======================================================================

=head2 rbac_move_client_to_limited_agency_rep

  $result = rbac_move_client_to_limited_agency_rep($rbac, $agency_uid, $client_client_id);

=cut

sub rbac_move_client_to_limited_agency_rep
{
    my ($rbac, $agency_uid, $client_client_id) = @_;

    Rbac::clear_cache();

    my $agency_client_id = rbac_get_agency_clientid_by_uid( $agency_uid) or die "ClientID not found for uid = $agency_uid";

    do_sql(PPC(ClientID => $client_client_id),
                "update campaigns c
                        join users u on c.uid = u.uid
                 set c.AgencyUID = ?
                 where u.ClientID = ?
                   and c.AgencyID = ?
                ", $agency_uid, $client_client_id, $agency_client_id);

    return 0;
}

#======================================================================

=head2 rbac_move_client_to_main_agency_rep

  $result = rbac_move_client_to_main_agency_rep($rbac, $agency_client_id, $agency_chief_uid, $client_client_id);

=cut

sub rbac_move_client_to_main_agency_rep
{
    my ($rbac, $agency_client_id, $agency_uid, $client_client_id) = @_;

    Rbac::clear_cache();

    do_sql(PPC(ClientID => $client_client_id),
                "update campaigns c
                        join users u on c.uid = u.uid
                 set c.AgencyUID = ?
                 where u.ClientID = ?
                   and c.AgencyID = ?
                ", $agency_uid, $client_client_id, $agency_client_id);

    return 0;
}

#======================================================================

=head2 rbac_switch_agency_chief

  $error = rbac_switch_agency_chief($rbac, $agency_rep_uid);

=cut

sub rbac_switch_agency_chief
{
    my ($rbac, $agency_rep_uid) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_switch_client_chief

  $error = rbac_switch_client_chief($rbac, $rep_uid, $dont_commit);

=cut

sub rbac_switch_client_chief
{
    my ($rbac, $rep_uid, $dont_commit) = @_;

    Rbac::clear_cache();

    if (is_freelancer($rep_uid)) {
        warn 'Main rep changing disallowed for freelancers';
        return 777;
    }

    return 0;
}

#======================================================================

=head2 rbac_drop_client_rep

  $result = rbac_drop_client_rep($rbac, $ClientID, $uid, $dont_commit);

=cut

sub rbac_drop_client_rep
{
    my ($rbac, $client_id, $uid, $dont_commit) = @_;

    Rbac::clear_cache();

    return 0;
}

#======================================================================

=head2 rbac_get_clients_for_step_zero

    my $clients = rbac_get_clients_for_step_zero(undef, $UID);
    $clients: [
        {uid => 1234567, ClientID => 123}
        {uid => 1234568, ClientID => 125}
        ...
    ]

    возвращаем всех подчиненных клиентов (представителей - шефов) для менеджера или агентства

=cut

sub rbac_get_clients_for_step_zero
{
    my (undef, $uid) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $uid);
    if ($perminfo && $perminfo->{role} eq $ROLE_MANAGER) {
        return get_all_sql(PPC(shard => 'all'), "
                    SELECT cl.chief_uid as uid, cl.ClientID
                      FROM client_managers cm
                           JOIN clients cl on cl.ClientID = cm.ClientID
                     WHERE cm.manager_uid = ?", $uid);
    } elsif ($perminfo && $perminfo->{role} eq $ROLE_AGENCY) {
        if ($perminfo->{rep_type} eq $REP_CHIEF || $perminfo->{rep_type} eq $REP_MAIN) {
            return get_all_sql(PPC(shard => 'all'), "
                        SELECT cl.chief_uid as uid, cl.ClientID
                          FROM clients cl
                         WHERE cl.agency_client_id = ?", $perminfo->{ClientID});
        } else {
            return get_all_sql(PPC(shard => 'all'), "
                        SELECT cl.chief_uid as uid, cl.ClientID
                          FROM clients cl
                         WHERE cl.agency_uid = ?", $uid);
        }
    } else {
        return [];
    }
}

=head2 rbac_manager_can_create_camp(undef, $manager_uid, $client_chief_uid)

    Менеджер имеют право на создание кампании пользователю?

    Параметры:
        $manager_uid - uid менеджера
        $client_chief_uid - uid главного представителя

    Результаты:
        1|0

=cut

sub rbac_manager_can_create_camp {
    my ( undef, $manager_uid, $client_chief_uid ) = @_;

    my $perminfo_by_uid = Rbac::get_key2perminfo( uid => [ $manager_uid, $client_chief_uid ] );

    my $manager_perminfo = $perminfo_by_uid->{ $manager_uid };
    return 0 unless $manager_perminfo && ( $manager_perminfo->{role} // '' ) eq $ROLE_MANAGER;

    my $user_perminfo = $perminfo_by_uid->{ $client_chief_uid };
    return 0 unless $user_perminfo && ( $user_perminfo->{role} // '' ) eq $ROLE_CLIENT;

    return 0 unless rbac_is_owner( undef, $manager_uid, $client_chief_uid );

    my $has_access = get_one_field_sql( PPC( uid => $client_chief_uid ), [
        'SELECT u.uid FROM users u LEFT JOIN campaigns c USING(uid)',
        WHERE => { 'u.statusArch' => 'No', 'u.uid' => $client_chief_uid },
        'GROUP BY uid',
        "HAVING $Client::NOT_GEOCONTEXT_HAVING_CONDITION",
        'LIMIT 1'
    ]);

    return $has_access ? 1 : 0;
}

#======================================================================

=head2 rbac_drop_user

    удаление ролей в rbac-е (всех кроме клиентов, для клиентов есть rbac_drop_client())

    my $errorcode = rbac_drop_user($rbac, $uid, $role, $dont_commit);
    $errorcode: 0 при успешном удалении
    если указан $dont_commit - то не коммитить изменения

=cut

sub rbac_drop_user
{
    my ($rbac, $uid, $role, $dont_commit) = @_;

    Rbac::clear_cache();

    my $errorcode = 2;
    if ($role eq 'super') {

        $errorcode = 0;

    } elsif ($role eq 'placer') {

        $errorcode = rbac_drop_placer($rbac, $uid);

    } elsif ($role eq 'media') {

        $errorcode = 0;

    } elsif ($role eq 'support') {

        $errorcode = 0;

    } elsif ($role eq 'superreader') {

        $errorcode = 0;

    } elsif ($role eq 'teamleader') {
        #При удалении тимлидера у него не должно быть подчиненых по старой схеме менеджеров,
        #но он может входить в IDM-группы
        my $muid_arrref = rbac_get_managers_of_teamleader($rbac, $uid);
        return 16 if @$muid_arrref;

        $errorcode = 0;

    } elsif ($role eq 'superteamleader') {

        my $tuid_arrref = rbac_get_team_of_superteamleader($rbac, $uid);
        return 20 if @$tuid_arrref;

        $errorcode = 0;

    } elsif ($role eq 'manager') {

        # нельзя удалить менеджера пока он является (супер)тимлидером
        return 24 if rbac_is_superteamleader($rbac, $uid) || rbac_is_teamleader($rbac, $uid);

        my $agencies_uids = rbac_get_agencies_uids($rbac, $uid);
        return 10 if @$agencies_uids; # error("Ошибка: у менеджера есть клиенты"); - in DoCmd

        my $cids_arrayref = rbac_get_scamps($rbac, $uid);

        # delete empty campaigns from RBAC
        if (defined $cids_arrayref && @{$cids_arrayref}) {
            my %all_status_empty = map {$_->{cid} => $_->{statusEmpty}}
                                   @{ get_all_sql(PPC(cid => $cids_arrayref), ["select cid, statusEmpty
                                                         from campaigns",
                                                         where => {cid => SHARD_IDS}
                                                       ])
                                    };
            my %deleted_cids;

            for my $cid (@{$cids_arrayref}) {

                my $statusEmpty = $all_status_empty{$cid};
                my $owner_of_camp_uid = rbac_get_owner_of_camp($rbac, $cid);
                next unless $owner_of_camp_uid;

                if (! defined $statusEmpty || $statusEmpty eq 'Yes') {
                    my $del_errcode = rbac_delete_campaign($rbac, $cid, $owner_of_camp_uid);
                    return $del_errcode if $del_errcode;
                    $deleted_cids{$cid} = 1;
                }
            }

            # remove deleted (empty) camps from $cids_arrayref
            @{$cids_arrayref} = grep {! exists $deleted_cids{$_}} @{$cids_arrayref};
        }

        return 10 if defined $cids_arrayref && @{$cids_arrayref}; # error("Ошибка: у менеджера есть клиенты"); - in DoCmd

        # отвязываем клиентов если остались без кампаний
        my $old_client_uids = rbac_get_clients_of_manager($rbac, $uid);
        for my $client_uid (@$old_client_uids) {
            next unless rbac_is_owner($rbac, $uid, $client_uid); # если уже отвязали одного представителя
            $errorcode = rbac_unbind_manager($rbac, $uid, $client_uid);
            return $errorcode if $errorcode;
        }

        # delete from teamleader
        my $tuid = rbac_get_teamleader_of_manager($rbac, $uid);
        rbac_delete_manager_from_teamleader($rbac, $uid, $tuid) if $tuid;

        remove_manager_data($uid);

        $errorcode = 0;
    } elsif ($role eq 'agency') {

        my $sub_cl_uids = rbac_get_subclients_uids($rbac, $uid);
        return 11 if @$sub_cl_uids; # error("Ошибка: у агентства есть клиенты"); - in DoCmd

        my $agency_id = rbac_get_agency_clientid_by_uid( $uid) || die "ClientID for $uid not found";
        $errorcode = 0;

    } elsif ($role eq 'internal_ad_admin') {

        $errorcode = 0;

    } elsif ($role eq 'internal_ad_manager') {

        $errorcode = 0;
    } elsif ($role eq 'internal_ad_superreader') {

        $errorcode = 0;
    }

    return $errorcode if $errorcode;

    # при удалении роли тимлидера, пользователь остаётся менеджером
    my $new_role = $role eq 'teamleader' || $role eq 'superteamleader' ? 'manager' : 'empty';
    Client::update_role(get_clientid(uid => $uid), $new_role, undef);

    return 0;
}

# --------------------------------------------------------------------

=head2 rbac_is_agency_client_id_of_client

    Проверяем, что агентство (по ClientID) является агентством клиента (по ClientID)
    my $AgencyID = rbac_is_agency_client_id_of_client(undef, $AgencyID, $uid);

=cut

sub rbac_is_agency_client_id_of_client($$$) {
    my (undef, $agency_client_id, $uid) = @_;

    my $agencies_id_of_client = rbac_get_agencies_clientids_by_uids( rbac_get_agencies_of_client(undef, $uid)) || {};

    if (grep {$agency_client_id =~ /^\d+$/ && $agency_client_id == $_} values %$agencies_id_of_client) {
        return $agency_client_id;
    } else {
        return undef;
    }
}

=head2 rbac_is_agency_of_client_multi($agency_uid, $uids)

    Проверка того, что agency_uid имеет права на клиентов (uids)

    Возвращается ссылка на хеш:
      uid => 1|0, если право есть|нет

=cut
sub rbac_is_agency_of_client_multi($$) {
    my ( $agency_uid, $uids ) = @_;

    my $operator_perms = Rbac::get_perminfo(uid => $agency_uid);
    return {} unless $operator_perms && $operator_perms->{role} eq $ROLE_AGENCY;

    return hash_grep { $_ } rbac_is_owner_of_users( $agency_uid, $uids );
}

#======================================================================

=head2 rbac_create_manager

    $result = rbac_create_manager($rbac, $uid);
    $dont_commit - не коммитить изменения (при создании тимлидеров)

=cut

sub rbac_create_manager
{
    my ($rbac, $uid, $dont_commit) = @_;

    Rbac::clear_cache();

    my $errorcode = rbac_drop_client($rbac, $uid);
    return $errorcode if $errorcode;

    return 0;
}

#======================================================================

=head2 rbac_create_superuser

    $result = rbac_create_superuser($rbac, $uid);

=cut

sub rbac_create_superuser
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $errorcode = rbac_drop_client($rbac, $uid);
    return $errorcode if $errorcode;

    return 0
}


=head2 rbac_create_mediaplanner

    $result = rbac_create_mediaplanner($rbac, $uid);

=cut

sub rbac_create_mediaplanner
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $errorcode = rbac_drop_client($rbac, $uid);
    return $errorcode if $errorcode;

    return 0
}


=head2 rbac_create_support

    $result = rbac_create_support($rbac, $uid);

=cut

sub rbac_create_support
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $errorcode = rbac_drop_client($rbac, $uid);
    return $errorcode if $errorcode;

    return 0;
}

#======================================================================

=head2 rbac_create_teamleader

    $result = rbac_create_teamleader($rbac, $uid);

=cut

sub rbac_create_teamleader
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $errorcode = rbac_drop_client($rbac, $uid);
    return $errorcode if $errorcode;

    return 0;
}

#======================================================================

=head2 rbac_create_superteamleader

    $result = rbac_create_superteamleader($rbac, $uid);

=cut

sub rbac_create_superteamleader
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $errorcode = rbac_drop_client($rbac, $uid);
    return $errorcode if $errorcode;

    return 0;
}

=head2 rbac_create_superreader

    $result = rbac_create_superreader($rbac, $uid);

=cut

sub rbac_create_superreader
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $error = rbac_drop_client($rbac, $uid);
    return $error if $error;

    return 0;
}

=head2 rbac_create_internal_ad_admin

    $result = rbac_create_internal_ad_admin($rbac, $uid);

=cut

sub rbac_create_internal_ad_admin
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $error = rbac_drop_client($rbac, $uid);
    return $error if $error;

    return 0
}

=head2 rbac_create_internal_ad_manager

    $result = rbac_create_internal_ad_manager($rbac, $uid);

=cut

sub rbac_create_internal_ad_manager
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $error = rbac_drop_client($rbac, $uid);
    return $error if $error;

    return 0
}

=head2 rbac_create_internal_ad_superreader

    $result = rbac_create_internal_ad_superreader($rbac, $uid);

=cut

sub rbac_create_internal_ad_superreader
{
    my ($rbac, $uid) = @_;

    Rbac::clear_cache();

    my $error = rbac_drop_client($rbac, $uid);
    return $error if $error;

    return 0
}

=head2 rbac_can_use_api

    Проверяем возможность получения доступа к управлению сертификатами для АПИ

=cut

sub rbac_can_use_api {
    my (undef, $vars) = @_;

    my $client_id = $vars->{ClientID} // get_clientid(uid => $vars->{uid});

    my $api_enabled = defined $vars->{api_enabled}
            ? $vars->{api_enabled}
            : API::ClientOptions::get_one($client_id, 'api_enabled');
    $api_enabled ||= 'Default';

    return 0 if $api_enabled eq 'No';

    return 1;
}

=head2 rbac_can_import_xls_into_new_camp

    Проверяет, может ли $UID импортировать из XLS новую кампанию пользователю $uid под агентство $agency_clientid.
    Или под хоть какое-нибудь агентство, если $agency_clientid не указан.
    Для создания кампании под агентство, проверяется наличие права импортировать из xls-файла.
    Для создания самостоятельной/сервисированной кампании проверяется свобода пользователя.
    Предполагает, что базовые проверки (можно менеджеру/агентству/суперу/самоходному клиенту и т.п.) уже выполнены.
    Возвращает 1, если импортировать можно, и 0 в противном случае.

    $can_import_xls_into_new_camp = rbac_can_import_xls_into_new_camp(undef, $UID, $uid, $agency_clientid);
    $can_import_xls_into_new_camp => 1|0

=cut

sub rbac_can_import_xls_into_new_camp {
    my ( undef, $UID, $uid, $agency_clientid ) = @_;

    $UID = $uid if rbac_is_related_freelancer( undef, $UID, $uid );

    my $rights = rbac_get_subclients_rights( undef, $UID, $uid );
    if ( $agency_clientid ) {
        return ( $rights && $rights->{ $agency_clientid } && $rights->{ $agency_clientid }{allowImportXLS} ) ? 1 : 0;
    } else {
        return 1 if $rights && (any { $rights->{ $_ }{allowImportXLS} } keys %$rights);
        return rbac_check_freedom( undef, $uid );
    }
}

=head2 rbac_can_use_xls

    Проверяет, есть ли у пользователя $UID права на использования интерфейса управления кампаниями пользователя $uid через XLS.
    Предполагает, что базовые проверки (можно менеджеру/агентству/суперу/самоходному клиенту и т.п.) уже выполнены.
    Возвращает 0, если может использовать и RBACовские коды ошибок в противном случае.

    $can_use_xls_rbac_error = rbac_can_use_xls($rbac, $UID, $uid);

=cut
sub rbac_can_use_xls {
    my ( undef, $UID, $uid ) = @_;

    return (rbac_can_import_xls_into_new_camp(undef, $UID, $uid) || @{rbac_get_allow_xls_import_camp_list(undef, $UID, $uid)}) ? 0 : 1;
}

=head2 rbac_user_allow_edit_camp

    имеет ли право uid редактировать кампанию или список кампаний

    $allow = rbac_user_allow_edit_camp(undef, $uid, $cid);
    $allow = rbac_user_allow_edit_camp(undef, $uid, [$cid1, $cid2, $cid3]);

=cut

sub rbac_user_allow_edit_camp
{
    my (undef, $uid, $cids) = @_;

    $cids = [$cids] unless ref $cids eq 'ARRAY';

    $uid = rbac_replace_freelancer_by_client_via_cids(undef, $uid, $cids);

    my $res = rbac_user_allow_edit_camps_detail( undef, $uid, $cids );

    return ( all { $_ } values %$res ) ? 1 : 0;
}

=head2 rbac_user_allow_edit_camps_detail

    имеет ли право юзер на редактирование списка кампаний

    $cid_result = rbac_user_allow_edit_camps_detail(undef, $uid, [$cid, ...]);
    Result:
        $hash = { cid1 => 1, cid2 => 0, ... }

        1 - права есть
        0 - прав нет

=cut

sub rbac_user_allow_edit_camps_detail
{
    my (undef, $uid, $cids) = @_;

    return is_user_allowed_for_action_on_camps( 'edit', $uid, $cids );
}

#======================================================================

=head2 rbac_check_allow_show_camps

    имеет ли право юзер на просмотр списка кампаний

    $cid_result = rbac_check_allow_show_camps(undef, $uid, [$cid, ...]);
    Result:
        $hash = { cid1 => 1, cid2 => 0, ... }

        1 - права есть
        0 - прав нет

=cut

sub rbac_check_allow_show_camps
{
    my (undef, $uid, $cids) = @_;

    return is_user_allowed_for_action_on_camps( 'show', $uid, $cids );
}

#======================================================================

=head2 is_user_allowed_for_action_on_camps

    $cid_result = is_user_allowed_for_action_on_camps($action, $uid, [$cid, ...]);

    $action => show / edit

    Result:
        $hash = { cid1 => 1, cid2 => 0, ... }

        1 - права есть
        0 - прав нет

=cut

sub is_user_allowed_for_action_on_camps
{
    my ( $action, $uid, $cids ) = @_;

    croak "Unknown action $action given!"
        unless $action eq 'show'
            || $action eq 'edit'
            || $action eq 'delete'
            || $action eq 'import_xls';

    return {} unless @$cids;

    $uid = rbac_replace_freelancer_by_client_via_cids(undef, $uid, $cids);

    my %ret = map { $_ => 0 } @$cids;

    my $perminfo = Rbac::get_perminfo(uid => $uid);
    return \%ret unless $perminfo && $perminfo->{role};

    my $mcc = { map { $_ => 1 } @{$perminfo->{mcc_client_ids} // []} };

    my $camps_info = get_hashes_hash_sql(PPC(cid => $cids), [
        'SELECT cid, ClientID, AgencyID, ManagerUID FROM campaigns',
        WHERE => {
            cid         => SHARD_IDS,
            statusEmpty => 'No',
        }
    ]);

    my @unique_camp_client_ids = uniq map { $_->{ClientID} } values %$camps_info;

    # к удаленным кампаниям нет доступа
    $cids = [ grep { exists $camps_info->{ $_ } } @$cids ];

    my $client_id = $perminfo->{ClientID};
    my $role = $perminfo->{role};
    if (   $role eq $ROLE_SUPER ) {
        # супер имеет полный доступ к кампаниям
        $ret{ $_ } = 1 for @$cids;
    }
    elsif ( $role eq $ROLE_SUPERREADER ) {
        # суперридер имеет доступ к кампаниям только на чтение
        my $right = ( $action eq 'show' ) ? 1 : 0;
        $ret{ $_ } = $right for @$cids;
    }
    elsif (    $role eq $ROLE_SUPPORT
            || $role eq $ROLE_PLACER
            || $role eq $ROLE_MEDIA
    ) {
        my $right = ( $action ne 'delete' ) ? 1 : 0;
        $ret{ $_ } = $right for @$cids;
    }
    elsif ( $role eq $ROLE_INTERNAL_AD_ADMIN ) {
        my $client_perminfo = Rbac::get_key2perminfo( ClientID => \@unique_camp_client_ids );
        for my $cid (@$cids) {
            my $camp_client_perminfo = $client_perminfo->{ $camps_info->{$cid}->{ClientID} };
            $ret{$cid} = has_perm( $camp_client_perminfo, 'internal_ad_product' );
        }
    }
    elsif ( $role eq $ROLE_INTERNAL_AD_MANAGER ) {
        my @camp_client_ids = map { $_->{ClientID} } values %$camps_info;
        my $client_perminfo = Rbac::get_key2perminfo( ClientID => \@camp_client_ids );
        my @product_cids;
        for my $cid (@$cids) {
            my $camp_client_perminfo = $client_perminfo->{ $camps_info->{$cid}->{ClientID} };
            if ( has_perm( $camp_client_perminfo, 'internal_ad_product' ) ) {
                push @product_cids, $cid;
            } else {
                $ret{$cid} = 0;
            }
        }

        my @product_camp_client_ids = map { $camps_info->{$_}->{ClientID} } @product_cids;
        my $access_types_by_client_id = InternalAdManagers::filter_accessible_client_ids( $client_id, \@product_camp_client_ids );
        for my $cid (@product_cids) {
            if ( ! exists $access_types_by_client_id->{ $camps_info->{$cid}->{ClientID} } ) {
                $ret{$cid} = 0;
            } else {
                my $access_type = $access_types_by_client_id->{ $camps_info->{$cid}->{ClientID} };
                if ( $access_type eq 'internal_ad_publisher' ) {
                    $ret{$cid} = 1;
                } else {
                    $ret{$cid} = ( $action eq 'show' ) ? 1 : 0;
                }
            }
        }
    }
    elsif ( $role eq $ROLE_INTERNAL_AD_SUPERREADER ) {
        my $client_perminfo = Rbac::get_key2perminfo( ClientID => \@unique_camp_client_ids );
        for my $cid (@$cids) {
            my $camp_client_perminfo = $client_perminfo->{ $camps_info->{$cid}->{ClientID} };
            $ret{$cid} = ( $action eq 'show' ) ? 1 : 0;
        }
    }
    else {
        my $subrole  = $perminfo->{subrole};
        my $rep_type = $perminfo->{rep_type};

        if ( $role eq $ROLE_MANAGER ) {

            my $owned_by_idm_client_ids_idx = {map {$_ => 1} @{rbac_filter_client_ids_by_manager_idm_access(undef, $client_id, \@unique_camp_client_ids)} };
            #Исключим из дальнейших проверок кампании к которым нашелся доступ через idm-группы
            if (keys %$owned_by_idm_client_ids_idx){
                my %tail;
                for my $cid ( keys %$camps_info ) {
                    if ($owned_by_idm_client_ids_idx->{$camps_info->{$cid}->{ClientID}}){
                        $ret{$cid} = 1;
                    } else {
                        $tail{$cid} = $camps_info->{$cid};
                    }
                }
                if ( keys %tail == 0 ) {
                    return \%ret;
                } else {
                    $camps_info = \%tail;
                }
            }

            my $manager_data = get_manager_data( $client_id ) // {};
            my $original_client_id = $client_id;

            # заменяем простого менеджера его тимлидером (если таковой есть)
            # аналог $uid = rbac_replace_manager_by_teamlead($rbac, $uid);
            if ( ! $subrole && $manager_data->{chiefs_cnt} ) {
                $client_id = $manager_data->{supervisor_client_id};
                $uid = $manager_data->{supervisor_uid};
                $manager_data = get_manager_data( $client_id );
            }

            # для проверки прав на сервисируемые кампании:
            #   - менеджер имеет право на просмотр и редактирование сервисируемых им же кампаний
            #   - менеджер имеет право на просмотр и редактирование кампаний "своих" агентств
            #
            # для проверки прав на просмотр агентских кампаний:
            #   - менеджер имеет право на просмотр и редактирование кампаний, сервисируемых своими подчиненными - менеджерами
            #   - менеджер имеет право на просмотр и редактирование кампаний агентств своих подчиненных - менеджеров

            my %manager_uids = ( $uid => 1 );
            if ( $manager_data->{subordinates_uid} ) {
                $manager_uids{ $_ } = 1 for @{ $manager_data->{subordinates_uid} };
            }

            my %agency_client_id;
            my @agency_id = uniq grep { defined } map { $_->{AgencyID} } values %$camps_info;
            if ( @agency_id ) {
                #Проверим доступ через группы idm к агенствам и добавим доступные агентства в индекс доступных через idm client_id
                foreach my $agency_client_id (@{rbac_filter_client_ids_by_manager_idm_access(undef, $original_client_id, \@agency_id)}) {
                    $owned_by_idm_client_ids_idx->{ $agency_client_id } = 1;
                }

                my $manager_uids_by_agency_id = rbac_get_all_managers_of_agencies_clientids( undef, \@agency_id );

                for my $agency_id ( keys %$manager_uids_by_agency_id ) {
                    my $muids = $manager_uids_by_agency_id->{ $agency_id };
                    $agency_client_id{ $agency_id } = 1 if any { exists $manager_uids{ $_ } } @$muids;
                }
            }

            for my $camp ( values %$camps_info ) {
                $ret{ $camp->{cid} } = 1 if $camp->{ManagerUID} && exists $manager_uids{ $camp->{ManagerUID} };
                $ret{ $camp->{cid} } = 1 if $camp->{AgencyID}   &&
                    ( $owned_by_idm_client_ids_idx->{ $camp->{AgencyID} } || exists $agency_client_id{ $camp->{AgencyID} });
            }
        }
        elsif ( $role eq $ROLE_AGENCY ) {

            # оператор - представитель агентства имеет доступ только к кампаниям этого агентства
            my @agency_camps = grep { $_->{AgencyID} && $_->{AgencyID} eq $client_id } values %$camps_info;

            if ( @agency_camps ) {

                # не лимитированные представители агентства (chief и main) могут смотреть и редактировать все кампании
                if ( $rep_type eq $REP_CHIEF || $rep_type eq $REP_MAIN ) {
                    $ret{ $_->{cid} } = 1 for @agency_camps;
                }
                else {
                    # лимитированный представитель агентства может смотреть и редактировать только привязанные к нему кампании
                    my $clients_perms = Rbac::get_key2perminfo( ClientID => [ uniq map { $_->{ClientID} } @agency_camps ] );

                    for my $camp ( @agency_camps ) {
                        my $agency_uid = $clients_perms->{ $camp->{ClientID} }->{agency_uid};
                        next unless ($agency_uid && ($agency_uid == $uid || rbac_has_lim_rep_access_to_client($uid, $camp->{ClientID})));
                        $ret{ $camp->{cid} } = 1;
                    }
                }
            }
        }
        elsif ( $role eq $ROLE_CLIENT ) {

            my $is_supersubclient = has_perm( $perminfo, $PERM_SUPER_SUBCLIENT );
            my $can_import_xls    = has_perm( $perminfo, $PERM_XLS_IMPORT );

            my $clients_perminfo = %$mcc ? Rbac::get_key2perminfo(ClientID => [ grep { $mcc->{$_} } @unique_camp_client_ids ]) : {};

            # если кампания принадлежит тому же клиенту, что и оператор, то:
            #   - оператор-клиент или оператор-субклиент может просматривать все кампании
            #   - оператор-клиент может редактировать кампании и загружать кампании из xls
            #   - оператор-субклиент может редактировать кампании если у него есть право super_subclient
            #       и загружать кампании из xls если у него есть право xls_import

            for my $camp ( values %$camps_info ) {
                if ( $camp->{ClientID} eq $client_id || $mcc->{$camp->{ClientID}} ) {
                    my $res = 0;
                    if ( $action eq 'show' ) {
                        $res = 1;
                    }
                    elsif ($perminfo->{rep_type} eq  $REP_READONLY) {
                        $res = 0;
                    }
                    elsif ( $action eq 'import_xls' ) {
                        if ($camp->{AgencyID}) {
                            if ($camp->{ClientID} eq $client_id) {
                                $res = $can_import_xls ? 1 : 0;
                            } else {
                                my $client_perminfo = $clients_perminfo->{$camp->{ClientID}};
                                $res = $client_perminfo && has_perm($client_perminfo, $PERM_XLS_IMPORT) ? 1 : 0;
                            }
                        } else {
                            $res = 1;
                        }
                    }
                    else { # 'edit' or 'delete'
                        if ($camp->{AgencyID}) {
                            if ($camp->{ClientID} eq $client_id) {
                                $res = $is_supersubclient ? 1 : 0;
                            } else {
                                my $client_perminfo = $clients_perminfo->{$camp->{ClientID}};
                                $res = $client_perminfo && has_perm($client_perminfo, $PERM_SUPER_SUBCLIENT) ? 1 : 0;
                            }
                        } else {
                            $res = 1;
                        }
                    }

                    $ret{ $camp->{cid} } = $res;
                }
            }
        }
        elsif ($role eq $ROLE_LIMITED_SUPPORT && ($action eq 'show' || $action eq 'import_xls')){
            my @camps = values %$camps_info;
            my $rows = get_all_sql(PPC(ClientID => [uniq map { $_->{ClientID} } @camps ]), [
                    q/SELECT client_id_to, client_id_from FROM clients_relations/,
                    WHERE => {client_id_to => SHARD_IDS, type => [qw/freelancer mcc/]}
            ]);

            my %related_fl_or_mcc;
            foreach my $row (@$rows) {
                push @{$related_fl_or_mcc{$row->{client_id_to}}}, $row->{client_id_from};
            }

            for my $camp ( @camps ) {
                my $client_id = $camp->{ClientID};

                $ret{ $camp->{cid} } = ($camp->{AgencyID} && rbac_is_related_client_support(undef, $uid, $camp->{AgencyID}))
                   || (any { rbac_is_related_client_support(undef, $uid, $_) } @{$related_fl_or_mcc{$client_id}})
                   || rbac_is_related_client_support(undef, $uid, $client_id);
            }
        }
    }

    return \%ret;
}
#======================================================================

=head2 rbac_cmd_user_allow_show_camps

  0 - OK
  1 - No rights

=cut

sub rbac_cmd_user_allow_show_camps
{
    my (undef, $vars, $result) = @_;

    # $vars->{cid} already splitted in DoCmd::FormCheck::check_required_params()
    my @cids = grep {$_} ref($vars->{cid}) eq 'ARRAY' ? @{ $vars->{cid} } : ($vars->{cid});
    return 0 unless @cids;

    my $res = is_user_allowed_for_action_on_camps('show', $vars->{UID}, \@cids);
    return 1 unless all {$res->{$_}} @cids;

    return 0;
}

=head2 rbac_allow_edit_description

  можно ли редактировать описание пользователя

  0 - forbidden
  1 - OK

=cut

sub rbac_allow_edit_description{
    my ($user_role, $login_rights) = @_;

    my $result = $user_role =~ /^(?:.*client|agency)$/ && ($login_rights->{super_control} || $login_rights->{support_control} || $login_rights->{manager_control} || $login_rights->{placer_control})
                                      ||
                                      $user_role =~ /^.*client$/ && $login_rights->{agency_control};
    return $result ? 1: 0;
}

=head2 rbac_check_freedom

    проверка свободы создавать самоходные/сервисируемые кампании у субклиента

    0 - forbidden
    1 - ok

=cut

sub rbac_check_freedom {
    my ( undef, $uid ) = @_;

    return 0 if ( rbac_has_agency( undef, $uid ) && ! get_allow_create_scamp_by_subclient( $uid ) );

    return 1; #ok
}

=head2 rbac_agency_can_edit_user_data

    Право агенства редактировать FIO/email/phone клиента
    не даем редактировать если у клиента > 1 типа обслуживания или обслуживание завершенно

=cut

sub rbac_agency_can_edit_user_data {
    my (undef, $agency_client_id, $client_client_ids) = @_;

    my $agency_client_relations = get_agency_client_relations($agency_client_id, $client_client_ids);
    my $res;
    foreach my $client_client_id (@$client_client_ids) {
        if (! $agency_client_relations->{$client_client_id}->{relation}
            || $agency_client_relations->{$client_client_id}->{servicing_types_count} > 1
            )
        {
            $res->{$client_client_id} = 0;
        } else {
            $res->{$client_client_id} = 1;
        }
    }
    return $res;
}

=head2 rbac_get_ownership_groups_for_clients

  $ownership_groups_hash_ref =  rbac_get_ownership_groups_of_clients(undef, $client_ids);

  Результат:
      { ClientID1 => [idm_group1, idm_group2], ClientID2 => [idm_group1], ClientID3 => [], ... }

=cut

sub rbac_get_ownership_groups_for_clients($$) :Cacheable(
        label => 'rbac_get_ownership_groups_for_clients',
        serialize_by => \&__srlz_numeric_array)
{
    my (undef, $client_ids) = @_;

    my $arr_client_ids = ref($client_ids) eq 'ARRAY' ? $client_ids : [$client_ids];

    # инициализируем результат
    my $result = { map { $_ => [] } @$arr_client_ids };

    my $idm_groups = get_all_sql(PPC(ClientID => $client_ids),
            ['SELECT subject_ClientID as ClientID, idm_group_id FROM idm_group_roles',
                WHERE => {subject_ClientID => SHARD_IDS}
            ]

    );

    foreach my $group (@$idm_groups){
        push @{$result->{$group->{ClientID}}}, $group->{idm_group_id};
    }

    return $result;
}

=head2 rbac_get_idm_groups_for_managers

  $idm_groups_hash_ref = rbac_get_idm_groups_for_managers(undef, $manager_uids);

  Результат:
      { manager_ClientID1 => [idm_group1, idm_group2], manager_ClientID2 => [idm_group1], manager_ClientID3 => [], ... }

=cut

sub rbac_get_idm_groups_for_managers($$) :Cacheable(
        label => 'rbac_get_idm_groups_for_managers',
        serialize_by => \&__srlz_numeric_array)
{
    my (undef, $manager_client_ids) = @_;

    my $arr_client_ids = ref($manager_client_ids) eq 'ARRAY' ? $manager_client_ids : [$manager_client_ids];

    # инициализируем результат
    my $result = { map { $_ => [] } @$arr_client_ids };

    my $idm_groups = get_all_sql(PPC(ClientID => $manager_client_ids),
            ['SELECT  ClientID, idm_group_id FROM idm_groups_members',
                WHERE => {ClientID => SHARD_IDS}
            ]

    );

    foreach my $group (@$idm_groups){
        push @{$result->{$group->{ClientID}}}, $group->{idm_group_id};
    }

    return $result;
}

=head2 __srlz_numeric_array
    Формирует ключ для кеширования результатов вызова метода у которого первый аргумент не влияет на результат,
    а второй представляет собой массив числовых величин

=cut

 sub __srlz_numeric_array {
    state $MAX_ITEMS = 50;
    my ($prefix, $args, $opts) = @_;
    my (undef, $items) = @$args;

    $items = [$items] unless ref $items;

    return undef if (@$items > $MAX_ITEMS);

    return join ':', ($prefix, sort {$a <=> $b} @$items);
}

=head2 has_idm_access_to_client

    Возвращает флаг имеет/нет оператор доступ к клиенту через idm-группы

=cut

sub has_idm_access_to_client($$) :Cacheable(
                label => 'has_idm_access_to_client')
{
    my ($operatorUid, $clientId) = @_;

    my $perminfo = Rbac::get_perminfo(uid => $operatorUid);

    my $allowed_clientIds = rbac_filter_client_ids_by_manager_idm_access(undef, $perminfo->{ClientID}, [$clientId]);

    return @$allowed_clientIds ? 1 : 0;
}


=head2 rbac_filter_client_ids_by_manager_idm_access
    Получает на вход uid менеджера и список ClientID клиентов.
    Возвращает те из входных ClientID, к которым менеджер имеет доступ.

    $accessable_client_ids = rbac_filter_client_ids_by_manager_idm_access(undef, $manager_client_id, $client_ids)

    Результат:
        [ClientID1, ClientId2, ...]

=cut

sub rbac_filter_client_ids_by_manager_idm_access($$$)
{
    my (undef, $manager_client_id, $client_ids) = @_;

    my %manager_groups_idx = map {$_ => 1} @{rbac_get_idm_groups_for_managers(undef, $manager_client_id)->{$manager_client_id}};

    return [] unless keys %manager_groups_idx;

    #Доступ по группе - клиент доступен менеджерам, входящим в группу, которая может управлять клиентом
    my $groups_by_client = rbac_get_ownership_groups_for_clients(undef, $client_ids);
    my @by_group = grep {_contains(\%manager_groups_idx, $groups_by_client->{$_})} @$client_ids;
    return \@by_group if @by_group == @$client_ids;

    #Доступ по принадлежности менеджеров одной и той же группе
    my $groups_by_manager =  rbac_get_idm_groups_for_managers(undef, $client_ids);
    my @by_same_group = grep {_contains(\%manager_groups_idx, $groups_by_manager->{$_})} @$client_ids;

    return [uniq @by_group, @by_same_group];
}

sub _contains {
    my ($idx, $values) = @_;
    return @$values && (any {exists $idx->{$_}} @$values);
}

=head2 rbac_get_managers_with_same_groups
    Получает на вход список client_id менеджеров.
    Для каждого из полученных client_id формирует список uid менеджеров, входящих в idm-группы, в которые входит исходный менеджер.
    Если исходный менеджер входит в несколько групп - входящие  них менеджеры объединяются.
    uid исходного менеджера в список менеджеров из его групп не входит.

    Результат:
        { manager_ClientID1 => [manager_uid1, manager_uid2], manager_ClientID2 => [manager_uid3], manager_ClientID3 => [], ... }

=cut

sub rbac_get_managers_with_same_groups {
    my ( undef, $manager_client_ids ) = @_;

    my $groups_by_manager = rbac_get_idm_groups_for_managers(undef, $manager_client_ids);

    my (%manager_subordinates_by_group, %manager_subordinates_by_manager);

    #Для всех входных идентификаторов менеджеров создадим анонимные "хеши-корзины".
    #В них позже будем собирать "подчиненных" менеджеров.
    foreach my $manager (@$manager_client_ids){
        $manager_subordinates_by_manager{$manager} //= {};
        #"Корзины" распределим по группам в которые входит менеджер.
        #Это позволит при обработке "подчиненных" менеджеров просто пройтись по всем "корзинам" группы.
        foreach my $group (@{ $groups_by_manager->{$manager} }) {
            push @{ $manager_subordinates_by_group{$group} }, $manager_subordinates_by_manager{$manager};
        }
    }

    my $subordinate_managers = get_all_sql(PPC(shard => 'all'),[
        qq/SELECT i.idm_group_id as group_id, i.ClientID as subordinate_client_id, c.chief_uid as subordinate_manager_uid
            FROM
                idm_groups_members i
                JOIN clients c on (i.ClientID = c.ClientID)
        /,
        WHERE => {'i.idm_group_id' => [keys %manager_subordinates_by_group]}
    ]);

    foreach my $rec (@$subordinate_managers) {
        #Добавляем uid менеджера в "корзины" всех менеджеров из входного списка, входящих в те же группы
        foreach my $manager_subordinates (@{$manager_subordinates_by_group{$rec->{group_id}}}){
            #Владельца "корзины" добавлять в нее не будем. Здесь сравниваются символьные представления hashref'ов
            next if ($manager_subordinates_by_manager{$rec->{subordinate_client_id}} // '') eq $manager_subordinates;
            #В группе владельца "корзины" найден менеджер отличный от него
            $manager_subordinates->{$rec->{subordinate_manager_uid}} = 1;
        }
    }

    return {map {$_ => [keys %{$manager_subordinates_by_manager{$_}}]} keys %manager_subordinates_by_manager};
}

=head2 rbac_has_lim_rep_access_to_client

    Получаем признак того, что ограниченный представитель имеет доступ к клиенту

=cut

sub rbac_has_lim_rep_access_to_client($$) :Cacheable(
    label => 'rbac_has_lim_rep_access_to_client')
{
    my ($lim_rep_uid, $client_id) = @_;

    return get_one_field_sql(PPC(ClientID => $client_id), [
            'SELECT 1 FROM agency_lim_rep_clients',
            WHERE => { ClientID => $client_id, agency_uid => $lim_rep_uid }
    ]) // 0;
}

=head2 rbac_has_lim_rep_access_to_other_rep

    Получаем признак того, что ограниченный представитель-тимлид имеет доступ к ограниченному представителю-менеджеру

=cut

sub rbac_has_lim_rep_access_to_other_rep($$) :Cacheable(
    label => 'rbac_has_lim_rep_access_to_other_rep')
{
    my ($lim_rep_uid, $rep_uid) = @_;

    return get_one_field_sql(PPC(uid => $lim_rep_uid), [
            'SELECT 1 FROM users_agency ua1 join users_agency ua2 on (ua2.group_id = ua1.group_id and ua2.lim_rep_type="chief")',
            WHERE => { 'ua1.uid' => $rep_uid, 'ua2.uid' => $lim_rep_uid }
    ]) // 0;
}

=head2 get_agency_clients_lim_reps($client_ids)

    Возвращаем хэш, uid'ов ограниченных представителей для заданных client_ids

=cut

sub get_agency_clients_lim_reps {
    my ($client_ids) = @_;
    return {} unless $client_ids && @$client_ids;

    my $rows = get_all_sql(PPC(ClientID => $client_ids), [
        'SELECT ClientID, agency_uid FROM agency_lim_rep_clients',
         WHERE => { ClientID => SHARD_IDS }
    ]);

    my $result = {};
    foreach my $row (@$rows) {
        push @{ $result->{ $row->{ClientID} } }, $row->{agency_uid};
    }

    return $result;
}

1;

