package RedisLock;

=encoding utf8

=head1 NAME

    RedisLock

=head1 SYNOPSIS

    my $redis_cluster = DirectRedis::get_redis();
    my ($locked, $lock_failed) = RedisLock::lock_multi($redis_cluster, ['hello', 'world'], 15);
    my ($unlocked, $unlock_failed) = RedisLock::unlock_multi($redis_cluster, $locked);

    my ($guard, $locked, $lock_failed) = RedisLock::lock_multi_guard($redis_cluster, ['hello', 'world'], 15);
    # ... do not worry ...

=cut

use Direct::Modern;

use Digest::CRC qw/crc/;
use Guard qw/guard/;
use List::MoreUtils qw/uniq/;
use Path::Tiny qw/path/;
use Redis::Script;

use Yandex::ListUtils qw/chunks/;

use Settings;

=head1 CONSTANTS
=cut
=head2 $MULTILOCK_BATCH_SIZE

    Максимальное количество локов, которое можно взять одним запросом в Redis.

=cut
our $MULTILOCK_BATCH_SIZE = 100;

# объекты Redis::Script для блокировки и разблокировки
my $lock_multi_script;
my $unlock_multi_script;


=head1 FUNCTIONS
=cut

=head2 lock_multi($redis, \@keys, $ttl)

    Берет Redis "лок" на ключи @keys на время $ttl в секундах.
    Пытается минимизировать количество походов в Redis путем группировки ключей
    по хеш слотам (числа от 0 до 16383).

    Чтобы несколько ключей гарантированно попали в один хешслот,
    у них должен быть одинаковый "тег" - строка между фигурными скобками.
    Например, wallet-1234{group1}1 и wallet-4567{group1}1 попадут в один слот.
    См. https://redis.io/topics/cluster-spec#keys-hash-tags

    Возвращает (\%locked, \@failed)
    %locked - словарик с ключами, на которые удалось взять лок.
    @failed - список ключей, на которые не получилось взять лок.

    Работа с Redis не в кластерной конфигурации не проверялась, но возможна.

=cut
sub lock_multi {
    my ($redis, $keys, $ttl) = @_;

    _init_scripts();

    my @uniq_keys = uniq @$keys;
    my $slot_groups = _group_keys_by_slot(\@uniq_keys);

    my %locked;
    my @failed;
    my $ttl_milli = $ttl * 1000;
    foreach my $slot_keys (@$slot_groups) {
        for my $chunk (chunks($slot_keys, $MULTILOCK_BATCH_SIZE)) {
            my @values = map { rand(2**48) } @$chunk;

            my $result = $lock_multi_script->eval($redis, $chunk, [@values, $ttl_milli]);
            for (my $i = 0; $i < @$chunk; ++$i) {
                my $key = $chunk->[$i];
                if ($result->[$i]) {
                    $locked{$key} = $values[$i];
                } else {
                    push @failed, $key;
                }
            }
        }
    }

    return (\%locked, \@failed);
}

=head2 unlock_multi($redis, \%locks)

    Снимает Redis "лок" с ключей %locks. Словарь %locks должен был быть
    получен функцией lock_multi.

    Возвращает (\@unlocked, \@failed)
    @unlocked - список ключей, с который удалось снять лок
    @failed - список ключей, с которых не удалось снять лок

=cut
sub unlock_multi {
    my ($redis, $locks) = @_;

    _init_scripts();

    my $slot_groups = _group_keys_by_slot([keys %$locks]);

    my @unlocked;
    my @failed;
    foreach my $slot_keys (@$slot_groups) {
        for my $chunk (chunks($slot_keys, $MULTILOCK_BATCH_SIZE)) {
            my @values = map { $locks->{$_} } @$chunk;

            my $result = $unlock_multi_script->eval($redis, $chunk, \@values);
            for (my $i = 0; $i < @$chunk; ++$i) {
                my $key = $chunk->[$i];
                if ($result->[$i]) {
                    push @unlocked, $key;
                } else {
                    push @failed, $key;
                }
            }
        }
    }

    return (\@unlocked, \@failed);
}

=head2 lock_multi_guard($redis, \@keys, $ttl)

    Берет лок с помощью lock_multi($redis, \@keys, $ttl).

    Возвращает ($guard, \%locked, \@failed)
    $guard - объект Guard, который сам вызовет unlock_multi
    %locked и @failed - из ответа lock_multi

    Используя эту функцию нельзя понять, с каких ключей не удалось снять лок.

=cut
sub lock_multi_guard {
    my ($redis, $keys, $ttl) = @_;

    my ($locked, $failed) = lock_multi($redis, $keys, $ttl);
    my $guard = guard { unlock_multi($redis, $locked) };
    return ($guard, $locked, $failed);
}

sub _print_slots_and_masters {
    my ($redis_cluster, $group_prefix, $num) = @_;

    my %seen_masters;
    for my $i (0..($num - 1)) {
        my $tag = "$group_prefix$i";
        my $slot = _get_slot_by_key($tag);
        my $master_address = $redis_cluster->get_master_by_key($tag)->{server};
        my $seen = $seen_masters{$master_address} ? 1 : 0;
        $seen_masters{$master_address} = 1;
        say "$tag: slot $slot, master $master_address".($seen ? '' : " first seen");
    }
}

=head2 _group_keys_by_slot(\@keys)

    Возвращает \@slot_groups
    @slot_group - списки ключей, сгруппированные по хешслотам.

=cut
sub _group_keys_by_slot {
    my ($keys) = @_;

    my %slot2keys;
    foreach my $key (@$keys) {
        my $slot = _get_slot_by_key($key);
        push @{$slot2keys{$slot}}, $key;
    }
    return [values %slot2keys];
}

=head2 _get_slot_by_key

    Вычисление хешслота: https://redis.io/topics/cluster-spec#keys-hash-tags
    Скопировано из Redis::Cluster

=cut
sub _get_slot_by_key {
    my ($key) = @_;

    # Hash tag (e.g. {tag}.key)
    my $tag = ($key =~ m/{([^}]+)}/ ? $1 : $key);

    # Redis-specific CRC16
    my $slot = crc($tag, 0x10, 0, 0, 0, 0x1021, 0, 0) & 0x3fff;

    return $slot;
}

=head2 _init_scripts()

    Инициализация package-private переменных с объектами Redis::Script,
    если это нужно.

=cut
sub _init_scripts {
    $lock_multi_script //= _make_script_from_file($Settings::ROOT.'/data/redis-lock/multilock.lua');
    $unlock_multi_script //= _make_script_from_file($Settings::ROOT.'/data/redis-lock/multiunlock.lua');
}

=head2 _make_script_from_file($filepath)

    Делает объект Redis::Script из файла $filepath.

=cut
sub _make_script_from_file {
    my ($filepath) = @_;

    my $text = path($filepath)->slurp();
    return Redis::Script->new(script => $text);
}

1;
