package Reports::Checks;
## no critic (TestingAndDebugging::RequireUseWarnings)

=head1 NAME
    
    Reports::Checks -- модуль для реализации разнообразных проверок входных данных для заказываемых отчетов

=head1 DESCRIPTION

    В модуле реализованы основные проверки, которые необходимы для заказа отчетов, поддерживаемых в Reports::Queue

=cut

use strict;
use utf8;

use Settings;
use Direct::Validation::Keywords qw/base_validate_keywords/;
use GeoTools qw/validate_geo/;
use PrimitivesIds;

use RBAC2::Extended;
use RBACDirect;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::Validate;
use Yandex::TimeCommon qw/check_mysql_date normalize_date/;
use Yandex::DateTime;
use DateTime;

my $CLIENT_POTENTIAL_MAX_EXPAND_PHRASES_COUNT = 100_000;
my $CLIENT_POTENTIAL_MAX_EXPAND_PHRASES_BYTES = 6*1024*1024;

sub check_period {
	my ($date_from, $date_to) = @_;

	my @errors = ();
    ($date_from, $date_to) = map { normalize_date($_) } ($date_from, $date_to);

    if ( check_mysql_date($date_from) ) {
        $date_from = date($date_from);
    } elsif( ($date_from||'') =~ m/^\s*$/ ) {
        $date_from = undef;     #   earliest possible date
    } else {
        push @errors, iget("Указан неверный формат даты: %s", $date_from||'');
    }

    if ( check_mysql_date($date_to) ) {
        $date_to = date($date_to);
    } elsif( ($date_to||'') =~ m/^\s*$/ ) {
        $date_to = undef;     #   earliest possible date
    } else {
        push @errors, iget("Указан неверный формат даты: %s", $date_to||'');
    }

    unless (@errors) {
        if ( $date_from && $date_to ) {
            push @errors, iget("Дата окончания не может быть меньше даты начала")
                if DateTime->compare($date_from, $date_to) > 0;
        }

        if ( $date_to) {
            # Очень желательно поменять на "Дата начала периода" ... date_from
            push @errors, iget("Дата конца периода не может быть больше \"Сегодня\"")
                if DateTime->compare($date_to, now()->truncate(to => 'day')) > 0;
        }
    }
    return @errors;
}

sub check_cids {
	my ($cids, $vars, %O) = @_;

	my @errors = ();
	my @camps = split /[\s,]+/, $cids;

    if ( @camps ) {

        my @invalid_cids = grep { !is_valid_id($_) } @camps;
        if (@invalid_cids) {
            push @errors, iget("Следующие номера кампаний некорректны: %s", join(', ', @invalid_cids));
        }
        
        @camps = grep { is_valid_id($_) } @camps;
        if (@camps) {
            $vars->{cids} = join(',', @camps);

            $vars->{camps} =  get_all_sql(PPC(cid => \@camps),  
                                           [ "SELECT cid, IF(archived = 'Yes', 1, 0) is_archived, name
                                                FROM campaigns", 
                                               WHERE => {
                                                    cid => SHARD_IDS,
                                                    statusEmpty => 'No',
                                                    # костыльный но рабочий способ запретить заказывать отчет на неподдерживаемые типы
                                                    # если будут жаловаться - сделат человекочитаемую ошибку
                                                    type__not_in => ['geo', 'mcb', 'wallet', 'billing_aggregate']
                                                }]);

            my %existed = map {$_->{cid} => 1} @{$vars->{camps}};
            if (my @not_existed = grep {!$existed{$_}} @camps) {
                #   Return error unless all the campaigns are accessible
                push @errors, iget("Следующие кампании не существуют: %s", join(', ', @not_existed));
            }
            if ($O{no_archive}) {
                if (my @archived = map {$_->{cid}} grep { $_->{is_archived} } @{$vars->{camps}}) {
                    push @errors, iget("Следующие кампании архивны: %s", join(', ', @archived));
                }
            }

            my $rbac = RBAC2::Extended->get_singleton(1);
            if ($O{check_rights} && keys %existed && !rbac_is_owner_of_camps($rbac, $vars->{operator_uid}, [keys %existed])) {
                push @errors, iget("У вас нет прав на некоторые из указанных кампаний");
            }
        }
    } else {
        ${$O{is_empty_flag}} = 1 if ref $O{is_empty_flag};
        push @errors, iget("Укажите хотя бы одну кампанию") if $O{is_required};
    }

    return @errors;
}

sub check_logins {
    my ($logins, $vars, %O) = @_;

    my @errors = ();
    my @logins = grep { length $_ } split /[\s,]+/, $logins;
    $vars->{logins} = join ',', @logins;

    if ( @logins ) {
        my $login2uid = get_login2uid(login => \@logins);

        if (my @not_existed = grep {!exists $login2uid->{$_}} @logins) {
            #   Return error unless all the campaigns are accessible
            push @errors, iget("Следующих логинов не существует: %s", join(', ', @not_existed));
        } elsif ($O{check_rights}) {
            my $rbac = RBAC2::Extended->get_singleton(1);
            push @errors, iget("У вас нет прав на некоторые из указанных логинов") if !rbac_mass_is_owner($rbac, $vars->{operator_uid}, [values %$login2uid]);
        }
    } else {
        ${$O{is_empty_flag}} = 1 if ref $O{is_empty_flag};
        push @errors, iget("Укажите хотя бы один логин") if $O{is_required};
    }

    return @errors;
}

sub check_phrases {
    my ($phrases, $vars, %O) = @_;
    my @errors = ();
    my @phrases = grep { length $_ } split /[\n\r\t,]+/, $phrases;
    $vars->{phrases} = join ',', @phrases;

    if ( @phrases ) {
        if (scalar @phrases > $CLIENT_POTENTIAL_MAX_EXPAND_PHRASES_COUNT) {
            push @errors, iget('Превышено максимальное количество фраз-расширений');
        }
        use bytes;
        if (length($vars->{phrases}) > $CLIENT_POTENTIAL_MAX_EXPAND_PHRASES_BYTES) {
            push @errors, iget('Превышена максимальная длина фраз-расширений');
        }
        no bytes;

        unless (@errors) {
            my $validation_result = base_validate_keywords(\@phrases, {advq_phrase => 1});
            unless ($validation_result->is_valid) {
                push @errors, @{$validation_result->one_error_description_by_objects // []};
            }
        }
        
    } else {
        ${$O{is_empty_flag}} = 1 if ref $O{is_empty_flag};
        push @errors, iget("Укажите хотя бы одну фразу-расширение") if $O{is_required};
    }

    return @errors;

}

sub check_geo {
    my ($geo_str, $vars, %O) = @_;
    
    #TODO: после выезда дерева api в транк - заменить ru на api
    return validate_geo($geo_str, undef, hash_merge({tree => 'ru'}, \%O)) // ();
}

1;
