package Reports::ClientPotential;

=head1 NAME

    Reports::ClientPotential -- модуль для обработки заказов на формирование отчетов "Потенциал клиента" (client_potential)

=head1 DESCRIPTION

    Модуль реализует работу с отчетами "Потенциал клиента" (client_potential)
    Для работы с очередью отчетов используется Reports::Queue

=cut

use Direct::Modern;

use feature qw/state/;

use Time::HiRes qw/gettimeofday tv_interval/;
use Date::Calc qw/check_date/;
use List::Util qw/maxstr minstr sum/;
use List::MoreUtils qw/none uniq all/;

use Settings;
use PrimitivesIds;
use Tools;
use Stat::CustomizedArray;
use Stat::Tools qw/suffix_list/;
use Stat::Const;
use Currencies;
use Currency::Rate;
use Client;
use Notification;
use Reports::Queue;
use Reports::Checks;
use Lang::Unglue;
use Forecast;
use Forecast::Autobudget;
use ADVQ6;
use MinusWords;
use MinusWordsTools;
use GeoTools;
use DirectCache;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Trace;
use Yandex::I18n;
use Yandex::ReportsXLSX;
use Yandex::HashUtils;
use Yandex::DateTime qw/date datetime now/;
use Yandex::ScalarUtils;
use Yandex::ListUtils qw/xminus/;
use Yandex::MyGoodWords;

our $REPORTS_ITERATION_LIMIT = 10; # максимальное кол-во обрабатываемых отчетов за одну итерацию
our $MAX_QUEUE_LENGTH = 5; # максимальное количество требующих обработки отчетов у одного пользователя
my $REPORT_TIMEOUT = 3600 * 12; # максимальное время генерации отчета, сек.

my $st;
# Логи ошибок использования системы заказа отчётов
{
    my $report_log;
    sub set_report_log($) {
        $report_log = shift;
    }
    sub report_log() {
        return $report_log;
    }
}

sub check_report_order {
    my ($vars) = @_;

    my @errors = ();

    my ($cids, $logins, $date_from, $date_to, $date_group, $client_name, $category_name, $report_date, 
        $ext_phrases, $geo, $geo_stat_flag, $ext_phrases_geo_flag, $ext_phrases_geo, $currency, $use_common_minus_words) = 
                    map { $vars->{$_} } qw/cids logins date_from date_to date_group client_name category_name report_date ext_phrases 
                                           geo geo_stat_flag ext_phrases_geo_flag ext_phrases_geo currency use_common_minus_words
                                           remove_phrases_operators remove_ext_phrases_operators/;

    push @errors, Reports::Checks::check_period($date_from, $date_to);

    unless ( defined $date_group and $date_group =~ m/^(none|day|week|month)$/) {
        push @errors, iget("Указан неверный период агрегации");
    }

    my ($cids_is_empty, $logins_is_empty, $phrases_is_empty);
    push @errors, Reports::Checks::check_cids($cids, $vars, is_required => 0, is_empty_flag => \$cids_is_empty, check_rights => 1, no_archive => 1 );
    push @errors, Reports::Checks::check_logins($logins, $vars, is_required => 0, is_empty_flag => \$logins_is_empty, check_rights => 1);
    push @errors, Reports::Checks::check_phrases($ext_phrases, $vars, is_required => 0, is_empty_flag => \$phrases_is_empty);
    if ( all { $_ } ($cids_is_empty, $logins_is_empty, $phrases_is_empty) ) {
        push @errors, iget('Пожалуйста, заполните хотя бы одно из полей: "ID кампаний", "Логины", "Расширения"');
    }

    push @errors, Reports::Checks::check_geo($geo, $vars);
    push @errors, Reports::Checks::check_geo($ext_phrases_geo, $vars) if $ext_phrases_geo_flag;

    push @errors, iget("Указана некорректная валюта") unless is_valid_currency($currency);

    my $queue_length = rqueue()->get_queue_length(operator_uid => $vars->{UID});
    if ($queue_length >= $MAX_QUEUE_LENGTH) {
        push @errors, iget("Невозможно заказать более %s отчётов одновременно. Пожалуйста, дождитесь готовности хотя бы одного отчёта.", $MAX_QUEUE_LENGTH);
    }

    return @errors;
}

=head2 make_report

    формирование данных для отчета

    вход - параметры отчета: { operator_uid - uid оператора
                               cids - кампании чрез разделитель
                               logins - логины через разделитель
                               date_from - начало периода
                               date_to - окончание периода
                               date_group - группировка по дате none|day|week|month|year
                               client_name 
                               category_name 
                               report_date 
                               ext_phrases 
                               geo 
                               geo_stat_flag 
                               ext_phrases_geo
                               currency
                               use_common_minus_words
                               remove_phrases_operators
                               remove_ext_phrases_operators
                            }

    выход - { error_code => <код ошибки>, error => <текст ошибки>} или
            { error_code => 0, data => <массив хешей с данными по строкам отчета>}

=cut

sub make_report($){
    my $opt = shift;
    my $profile = Yandex::Trace::new_profile('client_potential:make_report');

    $opt->{date_group} = 'none' unless $opt->{date_group};
    $opt->{geo} ||= 0;

    my $date_from = $opt->{date_from};
    $date_from ||= $BEGIN_OF_TIME_FOR_STAT;
    $date_from = maxstr $date_from, $BEGIN_OF_TIME_FOR_STAT;
    
    my $today = now()->ymd('');
    my $date_to = $opt->{date_to} || $today;
    report_log()->out("date_to: $date_to, today: $today");
    $date_to = minstr $date_to, $today;

    $date_from =~ s/\D//g;
    $date_to   =~ s/\D//g;

    my @cids = split /,+/, $opt->{cids} || '';
    my @logins = split /,+/, $opt->{logins} || '';
    if (!@cids && @logins) {
        @cids = @{ _get_cids_by_logins(\@logins) };
    }
    my $orderids = get_orderids(cid => \@cids);

    # выбираем статистику по фразам/позиции показа, из которой получаем список фраз и статистику по позициям показа
    report_log()->out('get real statistics');
    my $stats_opt = hash_cut $opt, qw/date_group geo_stat_flag geo currency/;
    hash_merge $stats_opt, {oids => $orderids,
                            date_from => $date_from,
                            date_to   => $date_to};
    my ($stat_phrases_raw_list, $stat_by_date, $stat_by_position, $total_stat) = get_real_statistics($stats_opt);

    # готовим списки фраз
    report_log()->out('prepare phrases list');
    $opt->{$_} ||= 0 for qw/remove_phrases_operators remove_ext_phrases_operators/;
    
    my $stat_phrases_list = Forecast::prepare_phrases_for_forecast(join ',', @$stat_phrases_raw_list);
    remove_phrases_operators($stat_phrases_list) if $opt->{remove_phrases_operators};
    @$stat_phrases_list = uniq @$stat_phrases_list;

    my $stat_and_ext_phrases_list = Forecast::prepare_phrases_for_forecast(join ',', @$stat_phrases_raw_list, $opt->{ext_phrases} || '');
    remove_phrases_operators($stat_and_ext_phrases_list) if $opt->{remove_ext_phrases_operators};
    @$stat_and_ext_phrases_list = uniq @$stat_and_ext_phrases_list;

    # выбираем общие минус-фразы на все фразы (путем объединения общих минус-фраз, указанных на каждой кампании)
    my $common_minus_words = [];
    if ($opt->{use_common_minus_words}) {
        my $camp_minus_words = get_one_column_sql(PPC(cid => \@cids), ["select minus_words from camp_options", where => { cid => SHARD_IDS,
                                                                                                                          minus_words__ne => '' }]);
        push @$common_minus_words, @{ MinusWordsTools::minus_words_str2array($_) } for @$camp_minus_words;
        $common_minus_words = polish_minus_words_array($common_minus_words);
    }

    # получаем прогноз по различным наборам фраз
    my %forecast_opt = (purpose => 'advanced_forecast', 
                        currency => $opt->{currency},
                        conv_unit_rate => 1, 
                        period => 'month',
                        consider_sitelinks_ctr => 0,
                        unglue => 1,
                        advq_timeout => 120,
                        minus_words => $common_minus_words,
                        lang => $opt->{lang},
    );

    # на каждый список фраз делаем отдельный запрос в прогноз, из соображений индивидуальной 
    # для каждого списка корректировки пересечений и удалению дублей
    # список фраз клиента (из статистики)
    report_log()->out('get forecast for stat phrases list ('.scalar(@$stat_phrases_list).' phrases)');
    my $dcache = new DirectCache(groups => ['lemmer','forecast_db_queries']);

    # увеличиваем таймаут для запросов в показометр (DIRECT-35414)
    local $Pokazometer::RPC::POKAZOMETER_TIMEOUT = 600;
    
    my $stat_phrases_forecast = get_moneymeter_data({   %forecast_opt,
                                                        phrases => $stat_phrases_list,
                                                        phrases_geo => GeoTools::modify_translocal_region_before_save($opt->{geo}, {tree => 'ru'}), });
    my $stat_phrases_list_corrected = [values %{$stat_phrases_forecast->{key2phrase}}];

    # расширенный список фраз (фразы из статистики + расширения)
    report_log()->out('get forecast for stat ext list ('.scalar(@$stat_and_ext_phrases_list).' phrases)');
    my $ext_phrases_geo = ($opt->{ext_phrases_geo} // $opt->{geo}) || 0;
    my $stat_and_ext_phrases_forecast;
    if (($opt->{ext_phrases} // '') !~ /\S+/ && $opt->{geo} eq $ext_phrases_geo && 
        $opt->{remove_phrases_operators} == $opt->{remove_ext_phrases_operators}) {
        $stat_and_ext_phrases_forecast = $stat_phrases_forecast;
    } else {
        $stat_and_ext_phrases_forecast = get_moneymeter_data({  %forecast_opt,
                                                                phrases => $stat_and_ext_phrases_list,
                                                                phrases_geo => GeoTools::modify_translocal_region_before_save($ext_phrases_geo, {tree => 'ru'}), });
    }
    my $stat_and_ext_phrases_list_corrected = [values %{$stat_and_ext_phrases_forecast->{key2phrase}}];

    # расширенный список без фраз клиента
    my $ext_wo_stat_dup_phrases_list = xminus_phrases_dup($stat_and_ext_phrases_list_corrected, $stat_phrases_list_corrected);
    
    # расширенный список без фраз клиента и фраз, которые являются вложенными к фразам клиента
    my $ext_wo_stat_sub_phrases_list = xminus_phrases_sub($stat_and_ext_phrases_list_corrected, $stat_phrases_list_corrected);

    # хроносрез из ADVQ
    report_log()->out('get avdq stat history');
    my %advq_hist_opts = (
        timeout => 180,
        function => 'time_hist',
        lang => $opt->{lang},
    );
    my ($stat_phrases_time_hist, $stat_and_ext_phrases_time_hist);
    if ((($ext_phrases_geo // '0') eq ($opt->{geo} // '0')) && (($opt->{remove_phrases_operators} // 0) == ($opt->{remove_ext_phrases_operators} // 0))) {
        my $all_phrases = [ @$stat_and_ext_phrases_list_corrected, @{ xminus $stat_phrases_list_corrected, $stat_and_ext_phrases_list_corrected } ];
        $stat_phrases_time_hist = $stat_and_ext_phrases_time_hist = advq_get_stat($all_phrases, $opt->{geo}, %advq_hist_opts);
    } else {
        $stat_phrases_time_hist = advq_get_stat($stat_phrases_list_corrected, $opt->{geo}, timeout => 180, function => "time_hist", lang => $opt->{lang});
        $stat_and_ext_phrases_time_hist = advq_get_stat($stat_and_ext_phrases_list_corrected, $ext_phrases_geo, %advq_hist_opts);
    }

    # формируем данные для выгрузки в отчет, на основе полученных статистических и прогнозных показателей
    report_log()->out('prepare data for export to xlsx');
    my ($task_sheet_data, $task_sheet_format) = prepare_task_sheet(hash_merge $opt, 
                                                                              {date_from => $date_from,
                                                                               date_to => $date_to,
                                                                               common_minus_words => $common_minus_words});

    my ($extention_sheet_data, $extention_sheet_format) = prepare_extention_sheet($stat_and_ext_phrases_forecast,
                                                                                  $ext_wo_stat_sub_phrases_list,
                                                                                  $ext_wo_stat_dup_phrases_list);

    my ($stat_history_sheet_data, $stat_history_sheet_format) = prepare_stat_history_sheet($stat_phrases_time_hist, 
                                                                                           $stat_phrases_list_corrected,
                                                                                           iget('Статистика по фразам (факт)'));

    my ($stat_history_ext_sheet_data, $stat_history_ext_sheet_format) = prepare_stat_history_sheet($stat_and_ext_phrases_time_hist, 
                                                                                                   $stat_and_ext_phrases_list_corrected,
                                                                                                   iget('Статистика по фразам (расширен)'));

    my ($stat_by_date_sheet_data, $stat_by_date_sheet_format) = prepare_stat_by_date_sheet($stat_by_date,
                                                                                           $total_stat,
                                                                                           {date_from => $date_from,
                                                                                            date_to => $date_to});

    my ($stat_by_position_sheet_data, $stat_by_position_sheet_format) = prepare_stat_by_position_sheet($stat_by_position);

    my ($mediaplan_sheet_data, $mediaplan_sheet_format) = prepare_mediaplan_sheet($stat_phrases_forecast, hash_cut($opt, 'currency'));

    my ($ext_mediaplan_sheet_data, $ext_mediaplan_sheet_format) = prepare_ext_mediaplan_sheet($stat_and_ext_phrases_forecast, hash_cut($opt, 'currency'));

    my @sheets_data = ($task_sheet_data, $extention_sheet_data, $stat_history_sheet_data, $stat_history_ext_sheet_data,
                       $stat_by_date_sheet_data, $stat_by_position_sheet_data, $mediaplan_sheet_data, $ext_mediaplan_sheet_data);
    my @sheets_format = ($task_sheet_format, $extention_sheet_format, $stat_history_sheet_format, $stat_history_ext_sheet_format,
                         $stat_by_date_sheet_format, $stat_by_position_sheet_format, $mediaplan_sheet_format, $ext_mediaplan_sheet_format);

    return {error_code => 0, sheets_data => \@sheets_data, sheets_format => \@sheets_format};
}   

=head2 get_real_statistics
    
    вытягивает статистику по заданным заказам, периоду, в контекста позиция/дата(период), а также возвращает список фраз из статистики

=cut

sub get_real_statistics {
    my $opt = shift;
    my @stat_phrases_list = ();
    my %stat_by_position = ();
    my %stat_by_date = ();
    my $total_stat = {};

    report_log()->out("start fetching statistics from database");
    foreach my $oid (@{$opt->{oids}}) {
        report_log()->out("fetching statistics for OrderID $oid");
        dbstat()->set_order_id($oid);
        my %geo_filter = $opt->{geo_stat_flag} && defined $opt->{geo} ? (geo => $opt->{geo}) : ();
        dbstat()->set_report_parameters(oid => $oid,
                                        start_date => $opt->{date_from},
                                        end_date   => $opt->{date_to},
                                        group_by   => [qw/phrase/, ($opt->{date_group} ne 'none' ? 'date' : ()), ($geo_filter{geo} ? 'geo': ())],
                                        filter     => {%geo_filter},
                                            $opt->{date_group} ne 'none' ? 
                                        (date_aggregation_by => $opt->{date_group}) : (),
                                        options    => {with_discount => 1,
                                                       with_nds => ($opt->{currency} eq 'YND_FIXED' ? 1 : 0),
                                                       countable_fields_by_targettype => 1,
                                                       single_currency => 1,
                                                       currency => $opt->{currency}},
                                        translocal_params => {tree => 'ru'});
        my $order_stat = dbstat()->generate();
        report_log()->out("fetched phrase/date statistics for OrderID $oid");
        
        push @stat_phrases_list, uniq map { $_->{phrase} } 
                                     grep { $_->{ContextType} == $Stat::Const::CONTEXT_TYPE_PHRASE } @{$order_stat->{data_array}};
        foreach my $row (@{$order_stat->{data_array}}) {
            my $date_key = ($row->{stat_date} && $opt->{date_group} ne 'none' ? date($row->{stat_date})->ymd('') : 'none');
            my $agr = $stat_by_date{$date_key} //= { date => $row->{date} };
            dbstat()->aggregation_stat($agr, $row);
        }

        dbstat()->set_report_parameters(oid => $oid,
                                        start_date => $opt->{date_from},
                                        end_date   => $opt->{date_to},
                                        group_by   => [qw/position page/, ($geo_filter{geo} ? 'geo' : ())],
                                        filter     => {%geo_filter,
                                                       page_target => 'search'},
                                        options    => {with_discount => 1,
                                                       with_nds => ($opt->{currency} eq 'YND_FIXED' ? 1 : 0),
                                                       countable_fields_by_targettype => 1,
                                                       single_currency => 1,
                                                       currency => $opt->{currency}},
                                        translocal_params => {tree => 'ru'});
        $order_stat = dbstat()->generate();
        report_log()->out("fetched position/page statistics for OrderID $oid");
        
        foreach my $row (@{$order_stat->{data_array}}) {
            # при текущем функционале нельзя сразу сгруппировать по площадкам, и добиться выборки полей с соотв. суффиксом - _0, _1
            my $suf = ($row->{TargetType} != $Stat::Const::CONTEXT_TARGET_TYPE ? '_0' : '_1');
            foreach my $field (Stat::Tools::field_list_sum()) {
                $row->{$field.$suf} = $row->{$field};
            }

            # по позициям показа нас интересуют только поисковые показы
            my $position = ($row->{position} == $Stat::Const::PRIME_TYPE_ID) ? $Stat::Const::PRIME_TYPE_ID : $Stat::Const::NON_PRIME_TYPE_ID;
            my $page_group = ($row->{page_group} =~ /^yandex\.\d+$/) ? 'yandex' : 'other';
            my $agr = $stat_by_position{$position}->{$page_group} //= {};
            dbstat()->aggregation_stat($agr, $row);
        }
    }

    foreach my $date (keys %stat_by_date) {
        dbstat()->calc_sec_stat($stat_by_date{$date});
        dbstat()->aggregation_stat($total_stat, $stat_by_date{$date});
    }
    dbstat()->calc_sec_stat($total_stat);

    foreach my $position (keys %stat_by_position) {
        foreach my $page (keys %{$stat_by_position{$position}}) {
            dbstat()->calc_sec_stat($stat_by_position{$position}->{$page});
        }
    }
    
    @stat_phrases_list = uniq @stat_phrases_list;

    report_log()->out("finish fetching statistics from database");

    return (\@stat_phrases_list, \%stat_by_date, \%stat_by_position, $total_stat);

}

sub prepare_task_sheet {
    my $report_opt = shift;

    my @merge_cells = (
            {row1 => 0, row2 => 3, col1 => 0, col2 => 5},
            # клиент
            {row1 => 6, row2 => 7, col1 => 1, col2 => 4},
            {row1 => 9, row2 => 9, col1 => 1, col2 => 4},
            # категория
            {row1 => 11, row2 => 12, col1 => 1, col2 => 4},
            {row1 => 14, row2 => 14, col1 => 1, col2 => 4},
            # валюта
            {row1 => 16, row2 => 17, col1 => 1, col2 => 4},
            {row1 => 19, row2 => 19, col1 => 1, col2 => 4},
            # логины
            {row1 => 22, row2 => 23, col1 => 1, col2 => 4},
            {row1 => 25, row2 => 25, col1 => 1, col2 => 4},
            # кампании
            {row1 => 27, row2 => 28, col1 => 1, col2 => 4},
            {row1 => 30, row2 => 30, col1 => 1, col2 => 4},
            # период
            {row1 => 32, row2 => 33, col1 => 1, col2 => 4},
            {row1 => 35, row2 => 35, col1 => 1, col2 => 4},
            # группировка
            {row1 => 37, row2 => 38, col1 => 1, col2 => 4},
            {row1 => 40, row2 => 40, col1 => 1, col2 => 4},
            # дата подготовки отчета
            {row1 => 42, row2 => 43, col1 => 1, col2 => 4},
            {row1 => 45, row2 => 45, col1 => 1, col2 => 4},

            # геотаргетинг
            {row1 => 6, row2 => 7, col1 => 6, col2 => 9},
            {row1 => 9, row2 => 9, col1 => 6, col2 => 9},
            # геотаргетинг для расширения
            {row1 => 11, row2 => 12, col1 => 6, col2 => 9},
            {row1 => 14, row2 => 14, col1 => 6, col2 => 9},
            # учет геотаргетинга для статистики
            {row1 => 16, row2 => 17, col1 => 6, col2 => 9},
            {row1 => 19, row2 => 19, col1 => 6, col2 => 9},
            # очищать от спецоператоров
            {row1 => 22, row2 => 22, col1 => 6, col2 => 9},
            {row1 => 25, row2 => 25, col1 => 6, col2 => 7},
            {row1 => 25, row2 => 25, col1 => 8, col2 => 9},

            # учитывать единые минус-фразы
            {row1 => 6, row2 => 7, col1 => 11, col2 => 14},
            {row1 => 9, row2 => 9, col1 => 11, col2 => 14},
            # список учтенных минус-фраз
            {row1 => 11, row2 => 12, col1 => 11, col2 => 14}, );

    my $xls_format = {
        sheetname=> iget("Задание"),

        set_column => [
            {col1 => 0, count => 15, width => 11.5},
        ],

        merge_cells => \@merge_cells,

        set_color => [
            [40, 226, 241, 246]
        ],
    };

    my $title_format = {align => 'center', valign => 'vcenter', bold => 1, size => 14, font => 'Calibri'};
    my $title_left_format = hash_merge {}, $title_format, {align => 'left'};
    my $text_format = {align => 'center', bg_color => 40, size => 12, font => 'Calibri'};
    my $text_left_format = hash_merge {}, $text_format, {align => 'left'};

    my @data = ();
    my $common_minus_words = $report_opt->{common_minus_words};
    for (0..45 + (@$common_minus_words > 32 ? scalar(@$common_minus_words)-32 : 0)) {
        push @data, [map {undef} (0..14)];
    }
    $data[0]->[0] = {data => iget('Потенциал'), format => hash_merge({}, $title_format, {size => 24})};

    $data[6]->[1] = {data => iget('Отчёт построен для:'), format => $title_format};
    $data[9]->[1] = {data => $report_opt->{client_name}, format => $text_format};

    $data[11]->[1] = {data => iget('Категория:'), format => $title_format};
    $data[14]->[1] = {data => $report_opt->{category_name}, format => $text_format};

    $data[16]->[1] = {data => iget('Валюта:'), format => $title_format};
    $data[19]->[1] = {data => iget(get_currency_constant($report_opt->{currency}, 'name')), format => $text_format};

    $data[22]->[1] = {data => iget('Логины:'), format => $title_format};
    $data[25]->[1] = {data => $report_opt->{logins}, format => $text_format};

    $data[27]->[1] = {data => iget('Кампании:'), format => $title_format};
    $data[30]->[1] = {data => $report_opt->{cids}, format => $text_format};

    $data[32]->[1] = {data => iget('Период для выгрузки статистики:'), format => $title_format};
    $data[35]->[1] = {data => join(' - ', map { Stat::Tools::format_date($report_opt->{$_}) } qw/date_from date_to/), format => $text_format};

    my %date_group_text = (day   => iget('по дням'),
                           week  => iget('по неделям'),
                           month => iget('по месяцам'),
                           none  => iget('суммарно за весь период'));
    $data[37]->[1] = {data => iget('Группировка:'), format => $title_format};
    $data[40]->[1] = {data => $date_group_text{$report_opt->{date_group}}, format => $text_format};

    $data[42]->[1] = {data => iget('Дата подготовки отчета:'), format => $title_format};
    $data[45]->[1] = {data => $report_opt->{report_date}, format => $text_format};

    $data[6]->[6] = {data => iget('Геотаргетинг:'), format => $title_format};
    $data[9]->[6] = {data => (length($report_opt->{geo}) ? get_geo_names($report_opt->{geo}) : ''), format => $text_format};

    $data[11]->[6] = {data => iget('Геотаргетинг для расширения:'), format => $title_format};
    $data[14]->[6] = {data => (length($report_opt->{ext_phrases_geo}) ? get_geo_names($report_opt->{ext_phrases_geo}) : ''), format => $text_format};

    $data[16]->[6] = {data => iget('Учет геотаргетинга для статистики клиента:'), format => $title_left_format};
    $data[19]->[6] = {data => ($report_opt->{geo_stat_flag} ? iget('да') : iget('нет')), format => $text_format};    

    $data[22]->[6] = {data => iget('Очищать от спецоператоров'), format => $title_format};
    $data[23]->[6] = {data => iget('Фактический список:'), format => $title_left_format};
    $data[23]->[8] = {data => iget('Расширенный список:'), format => $title_left_format};
    $data[25]->[6] = {data => ($report_opt->{remove_phrases_operators} ? iget('да') : iget('нет')), format => $text_format};    
    $data[25]->[8] = {data => ($report_opt->{remove_ext_phrases_operators} ? iget('да') : iget('нет')), format => $text_format};    

    $data[6]->[11] = {data => iget('Учитывать единые минус-фразы кампаний в прогнозе:'), format => $title_left_format};
    $data[9]->[11] = {data => ($report_opt->{use_common_minus_words} ? iget('да') : iget('нет')), format => $text_format};

    $data[11]->[11] = {data => iget('Список учтенных минус-фраз:'), format => $title_format};

    if (@$common_minus_words) {
        my $ind = 14;
        foreach (@$common_minus_words) {
            push @merge_cells, {row1 => $ind, row2 => $ind, col1 => 11, col2 => 14};
            $data[$ind]->[11] = {data => $_, format => $text_left_format};
            $ind++;
        }
    }

    return \@data, $xls_format;
}

{
    my @mediaplan_header = (
                  iget_noop('Предложенные фразы'),
                  iget_noop('Позиция'),
                  iget_noop('Прогноз показов в месяц'),
                  iget_noop('Клики (СР)'),
                  iget_noop('Клики (Гар)'),
                  iget_noop('CTR (СР)'),
                  iget_noop('CTR (Гар)'),
                  iget_noop('CPC (СР)'),
                  iget_noop('CPC (Гар)'),
                  iget_noop('Бюджет (СР), руб.'),
                  iget_noop('Бюджет (Гар), руб.'),
                  iget_noop('Клики (уточненная гарантия)'),
                  iget_noop('CPC (уточненная гарантия)'),
                  iget_noop('Бюджет (уточненная гарантия)'),
                  iget_noop('Клики (уточненное спецразмещение)'),
                  iget_noop('CPC (уточненное спецразмещение)'),
                  iget_noop('Бюджет (уточненное спецразмещение)') );

sub prepare_mediaplan_sheet {
    my ($forecast, $report_opt) = @_;

    my ($data, $xls_format) = forecast2mediaplan($forecast, iget('Шаг 3. Медиаплан клиента'), $report_opt);

    return $data, $xls_format;
}

sub prepare_ext_mediaplan_sheet {
    my ($forecast, $report_opt) = @_;

    my ($data, $xls_format) = forecast2mediaplan($forecast, iget('Шаг 4. Расширенный медиаплан'), $report_opt);

    return $data, $xls_format;
}

sub forecast2mediaplan {
    my ($forecast, $sheet_title, $report_opt) = @_;
    my @data = ();
    foreach my $row (sort { $b->{shows} <=> $a->{shows} } @{$forecast->{data_by_positions}}) {
        my @sheet_row = ();
        push @sheet_row, (  $forecast->{key2phrase}->{$row->{md5}},
                            undef,
                            $row->{shows},
                            $row->{premium}->{yandex}->{clicks},
                            $row->{std}->{yandex}->{clicks},
                            $row->{premium}->{yandex}->{ctr},
                            $row->{std}->{yandex}->{ctr},
                            $row->{premium}->{yandex}->{amnesty_price} / 1e6,
                            $row->{std}->{yandex}->{amnesty_price} / 1e6,
                            $row->{premium}->{yandex}->{sum},
                            $row->{std}->{yandex}->{sum}, );
        push @data, \@sheet_row;
    }
    
    my $xls_format = {
        sheetname => $sheet_title,

        set_column => [
            {col1 => 0, col2 => 0, width => 21},
            {col1 => 1, col2 => 1, width => 9.5},
            {col1 => 1, col2 => 1, width => 23},
            {col1 => 3, col2 => 8, width => 11.5},
            {col1 => 9, col2 => 10, width => 18},
            {col1 => 11, col2 => 16, width => 11.5},
        ],

        set_row => [
            (map { {row => $_, hidden => 1, level => 1} } (0..5)),
            {row => 6, hidden => 0, level => 0, collapsed => 1},
        ],

        merge_cells => [
            {row1 => 7, row2 => 7, col1 => 0, col2 => 16},
        ],

        set_color => [
            [40, 226, 241, 246],
            [41, 224, 224, 224],
            [42, 245, 227, 226]

        ],
    };

    my $title_format = {size => 12, font => 'Calibri'};
    my $title2_format = hash_merge {}, $title_format, {align => 'center', bg_color => 41};
    my $text_format = hash_merge {}, $title_format, {bg_color => 40, border => 1};
    my $number_format = hash_merge {}, $text_format, {num_format => '#,##0', align => 'default'};
    my $float_format = hash_merge {}, $text_format, {num_format => '#,##0.00', align => 'default'};
    my $macros_format = hash_merge {}, $title_format, {bg_color => 42, border => 1};

    my $curr_rate_to_rub = convert_currency(1, $report_opt->{currency}, 'RUB', with_nds => 0);

    my $r_num = 9;
    foreach my $row (@data) {
        $r_num ++;
        @$row = ((map { {data => $_, format => $text_format} } @{$row}[0..1]),
                 (map { {data => $_, format => $number_format, save_cell => 1} } @{$row}[2..4]),
                 (map { {data => $_, format => $float_format, save_cell => 1} } @{$row}[5..8]),
                 {formula => "=D$r_num*H$r_num*$curr_rate_to_rub", formula_value => $row->[3]*$row->[7]*$curr_rate_to_rub, format => $float_format},
                 {formula => "=E$r_num*I$r_num*$curr_rate_to_rub", formula_value => $row->[4]*$row->[8]*$curr_rate_to_rub, format => $float_format},
                 (map { {format => $macros_format} } @{$row}[11..16]));
                
    }

    unshift @data, ((map {[undef]} (0..5)),
                    [undef],
                    [{data => iget('Выгрузка из медиаплана'), format => $title2_format}],
                    [map { {data => iget($_), format => $title_format} } @mediaplan_header], );


    return (\@data, $xls_format);
}

}

{
    my @extention_header = (  iget_noop('Фраза'),
                              iget_noop('Показы'),
                              iget_noop('Клики (СР)'),
                              iget_noop('Клики (Гар)'), );

sub prepare_extention_sheet {
    my ($forecast, $phrases_sub, $phrases_dup) = @_;

    my $data_sub = forecast2extention($forecast, $phrases_sub);
    my $data_dup = forecast2extention($forecast, $phrases_dup);

    my $xls_format = {
        sheetname=> iget("Расширение"),

        set_column => [
            {col1 => 0, col2 => 0, width => 22},
            {col1 => 1, col2 => 4, width => 11.5},
            {col1 => 5, col2 => 5, width => 22},
            {col1 => 6, col2 => 8, width => 11.5},
        ],

        merge_cells => [
            {row1 => 0, row2 => 0, col1 => 0, col2 => 3},
            {row1 => 0, row2 => 0, col1 => 5, col2 => 8}, ],

        set_color => [
            [40, 226, 241, 246]
        ],
    };

    my $title_format = {size => 12, font => 'Calibri'};
    my $title_center_format = {align => 'center', size => 12, font => 'Calibri'};
    my $text_format = {bg_color => 40, size => 12, font => 'Calibri'};
    my $number_format = hash_merge {}, $text_format, {num_format => '#,##0', align => 'default'};

    my @table_sub = ([{data => iget('С учетом вложенности'), format => $title_center_format}],
                     [ map { {data => iget($_), format => $title_format} } @extention_header ]);
    foreach my $row (@$data_sub) {
        push @table_sub, [ (map { {data => $_, format => $text_format} } @$row[0]),
                           (map { {data => $_, format => $number_format} } @$row[1 .. $#{$row}]) ];
    }

    my @table_dup = ([{data => iget('Без учета вложенности'), format => $title_center_format}],
                     [ map { {data => iget($_), format => $title_format} } @extention_header ]);
    foreach my $row (@$data_dup) {
        push @table_dup, [ (map { {data => $_, format => $text_format} } @$row[0]),
                           (map { {data => $_, format => $number_format} } @$row[1 .. $#{$row}]) ];
    }

    my @data = ();
    foreach my $t ({data => \@table_sub, offset => [0,0]},
                   {data => \@table_dup, offset => [0,5]}) {
        my $tdata = $t->{data};
        foreach my $row (0 .. $t->{offset}->[0]-1) {
            $data[$row] //= [];
            foreach my $row (@{$data[$row]} .. $t->{offset}->[1]-1) {
                push @{$data[$row]}, undef;
            }
        }
        foreach my $row (0 .. $#{$tdata}) {
            my $data_row = $data[$row + $t->{offset}->[0]] //= [];
            foreach my $col (0 .. $#{$tdata->[$row]}) {
                $data_row->[$col + $t->{offset}->[1]] = $tdata->[$row]->[$col];
            }
        }
    }
    return \@data, $xls_format;
}

sub forecast2extention {
    my ($forecast, $phrases) = @_;
    my %process_md5 = map { exists $forecast->{phrase2key}->{$_}
                            ? ($forecast->{phrase2key}->{$_} => 1)
                            : () } @$phrases;

    my @data = ();
    foreach my $row (sort { $b->{shows} <=> $a->{shows} } 
                          grep { $process_md5{$_->{md5}} } @{$forecast->{data_by_positions}}) {
        my @sheet_row = ();
        push @sheet_row, (  $forecast->{key2phrase}->{$row->{md5}},
                            $row->{shows},
                            $row->{premium}->{yandex}->{clicks},
                            $row->{std}->{yandex}->{clicks}, );
        push @data, \@sheet_row;
    }
    return \@data;   
}

}

sub prepare_stat_history_sheet {
    my ($history, $phrases, $sheet_title) = @_;

    my %process_phrases = map { $_ => 1 } @$phrases;
    my @data = ();
    my @header = ();
    foreach my $ph (sort { $b->{current}->{total_count} <=> $a->{current}->{total_count} } 
                         grep { $process_phrases{$_->{req}} } @$history) {
        unless (@header) {
            push @header, '', map { sprintf '%d%02d', $_->{year}, $_->{month} } @{$ph->{hist}};
        }
        push @data, [ $ph->{req}, map { $_->{total_count} } @{$ph->{hist}} ];
    }

     my $xls_format = {
        sheetname=> $sheet_title,

        set_column => [
            {col1 => 0, col2 => 0, width => 22},
        ],

        set_color => [
            [40, 226, 241, 246]
        ],
    };

    my $title_format = {bg_color => 40, size => 12, font => 'Calibri', bold => 1, align => 'right'};
    my $text_format = {bg_color => 40, size => 12, font => 'Calibri'};
    my $number_format = hash_merge {}, $text_format, {num_format => '#,##0', align => 'default'};

    @header = map { { data => $_, format => $title_format } } @header;
    foreach my $row (@data) {
        @$row = ((map { { data => $_, format => $text_format } } @{$row}[0]),
                 (map { { data => $_, format => $number_format } } @{$row}[1 .. $#{$row}]));
    }
    unshift @data, \@header;

    return \@data, $xls_format;
}

sub prepare_stat_by_date_sheet {
    my ($stat_by_date, $total_stat, $report_opt) = @_;

    my @header1 = (iget('Дата'));
    my @header2 = ('');
    foreach my $field ( iget('Показы'),
                        iget('Клики'),
                        iget('CTR (%)'),
                        iget('Расход всего'),
                        iget('Ср. цена клика') ) {
        push @header1, ('', $field, '', '');
        push @header2, ('', iget('всего'), iget('поиск'), iget('контекст'));
    }

    my @data;
    foreach my $stat_date (sort keys %$stat_by_date) {
        my @row = ($stat_by_date->{$stat_date}->{date});
        foreach my $field (qw/shows clicks ctr sum av_sum/) {
            push @row, '';
            for my $suf ( suffix_list() ) {
                push @row, $stat_by_date->{$stat_date}->{$field.$suf};
            }
        }
        push @data, \@row;
    }

    # строка с суммарной статистикой
    my @total_row = (iget('Итого:'));
    foreach my $field (qw/shows clicks ctr sum av_sum/) {
        push @total_row, '';
        for my $suf ( suffix_list() ) {
            push @total_row, $total_stat->{$field.$suf};
        }
    }
    push @data, \@total_row;

    my $xls_format = {
        sheetname => iget('Шаг 1. Статистика'),

        set_column => [
            {col1 => 0, col2 => 0, width => 20},
            {col1 => 1, col2 => 20, width => 14.5},
        ],

        set_row => [
            (map { {row => $_, hidden => 1, level => 1} } (0..11)),
            {row => 12, hidden => 0, level => 0, collapsed => 1},
        ],

        merge_cells => [
            {row1 => 15, row2 => 15, col1 => 0, col2 => 20},
        ],

        set_color => [
            [40, 226, 241, 246],
            [41, 224, 224, 224]
        ],
    };

    my $title_format = {size => 12, font => 'Calibri'};
    my $title2_format = hash_merge {}, $title_format, {align => 'center', bg_color => 41};
    my $text_format = hash_merge {}, $title_format, {bg_color => 40, border => 1};
    my $number_format = hash_merge {}, $text_format, {num_format => '#,##0', align => 'default'};
    my $float_format = hash_merge {}, $text_format, {num_format => '#,##0.00', align => 'default'};

    my @sheet_data = (map {[undef]} (0..11));
    push @sheet_data, [undef],
                      [{data => iget('Период:'), format => $title_format},
                       {data => join(' - ', map { Stat::Tools::format_date($report_opt->{$_}) } qw/date_from date_to/), format => $text_format}, ],
                      [undef];

    push @sheet_data, [{data => 'Суммарная статистика по кампаниям', format => $title2_format}],
                      [map { {data => $_, format => $title_format} } @header1],
                      [map { {data => $_, format => $title_format} } @header2];
    while (my $row = shift @data) {
        push @sheet_data, [(map { {data => $_, format => $text_format} } @{$row}[0..1]),
                           (map { {data => $_, format => $number_format} } @{$row}[2..9]),
                           (map { {data => $_, format => $float_format} } @{$row}[10..20])];
    }

    return \@sheet_data, $xls_format;
}

sub prepare_stat_by_position_sheet {
    my $stat_by_position = shift;

    my @header =   (iget('Позиция'),
                    iget('Площадка'),
                    iget('Показы'),
                    iget('Клики'),
                    iget('CTR (%)'),
                    iget('Расход всего'),
                    iget('Ср. цена клика'),
                    '',
                    iget('Глубина (стр.)'),
                    iget('Конверсия (%)'),
                    iget('Цена цели'), );

    my %positions = (1 => iget('спецразмещение'),
                     2 => iget('прочее'));
    my %pages = (yandex => iget('Яндекс'),
                 other  => iget('партнеры'));
    my @data = ();

    foreach my $page (keys %pages) {
        foreach my $position (keys %positions) {
            my @row = ($positions{$position}, $pages{$page});
            if ($stat_by_position->{$position} && $stat_by_position->{$position}->{$page}) {
                my $st = $stat_by_position->{$position}->{$page};
                push @row, (map { $st->{$_} } qw/shows clicks ctr sum av_sum/), '', 
                           (map { $st->{$_} } qw/adepth aconv agoalcost/);
            } else {
                push @row, 0,0,0,0,0;
            }

            push @data, \@row;
        }
    }
    
    my $xls_format = {
        sheetname => iget('Шаг 2. Вычисление CTR'),

        set_column => [
            {col1 => 0, col2 => 10, width => 11.5},
        ],

        set_row => [
            (map { {row => $_, hidden => 1, level => 1} } (0..11)),
            {row => 12, hidden => 0, level => 0, collapsed => 1},
        ],

        merge_cells => [
            {row1 => 13, row2 => 13, col1 => 0, col2 => 10},
        ],

        set_color => [
            [40, 226, 241, 246],
            [41, 224, 224, 224]
        ],
    };

    my $title_format = {size => 12, font => 'Calibri'};
    my $title2_format = hash_merge {}, $title_format, {align => 'center', bg_color => 41};
    my $text_format = hash_merge {}, $title_format, {bg_color => 40, border => 1};
    my $number_format = hash_merge {}, $text_format, {num_format => '#,##0', align => 'default'};
    my $float_format = hash_merge {}, $text_format, {num_format => '#,##0.00', align => 'default'};

    foreach my $row (@data) {
        @$row = ((map { {data => $_, format => $text_format} } @{$row}[0..1]),
                 (map { {data => $_, format => $number_format} } @{$row}[2..3]),
                 (map { {data => $_, format => $float_format} } @{$row}[4..10]), );
    }

    unshift @data, ((map {[undef]} (0..11)),
                    [undef],
                    [{data => 'Статистика только по поисковым кампаниям', format => $title2_format}],
                    [map { {data => $_, format => $title_format} } @header], );


    return \@data, $xls_format;
}

=head2 create_report

    Создать заявку на отчёт и сохранить её в БД Директа в случае успешного создания

=cut

sub create_report($) {
    my ($opt) = @_;

    my $report_opt = hash_cut $opt, qw/uid operator_uid cids logins date_from date_to date_group client_name category_name report_date 
                                       ext_phrases geo geo_stat_flag ext_phrases_geo currency use_common_minus_words
                                       remove_phrases_operators remove_ext_phrases_operators lang/;
    $report_opt->{uid} = $report_opt->{operator_uid};
    $report_opt->{cids} //= '';
    delete $report_opt->{ext_phrases_geo} unless $opt->{ext_phrases_geo_flag};

    my $report_stats = $report_opt->{report_stats} //= {};
    my @cids = split /,+/, $report_opt->{cids} || '';
    my @logins = split /,+/, $report_opt->{logins} || '';
    if (!@cids && @logins) {
        @cids = @{ _get_cids_by_logins(\@logins) };
    }
    $report_stats->{bids_qty} = sum(0, @{get_one_column_sql(PPC(cid => \@cids), ["select count(*)
                                                                              from bids bi 
                                                                              join phrases p using(pid) ",
                                                                             where => {'p.cid' => SHARD_IDS} ])}) || 0;
    $report_stats->{bids_qty} += scalar(grep { $_ } split(/[,\n]+/, $report_opt->{ext_phrases})) if $report_opt->{ext_phrases};

    rqueue()->create_report($report_opt);
}

=head2 get_report
    
    Проверяет наличие готового отчёта на стороне Директа.

=cut

sub get_report($$;$) {
    my ($uid, $id) = @_;
    
    return wantarray ? (rqueue()->get_report($uid, $id, 0)) : rqueue()->get_report($uid, $id, 0);
}
        
=head2 save_report_data

    Сохранить отчёт

=cut

sub save_report_data($) {
    my ($opt) = @_; 

    rqueue()->save_report_data($opt);
}

=head2 make_reports

    Формируем отчеты

    $error_stat = make_reports($shard,
        thread => $thread,
        limit => $limit,
        login => $login,
        on_before_new_report => sub {
            my ($report) = @_;
            ...
        },
    );

    $error_stat => {
        cnt =>
        ok =>
        no_response =>
        other_errors =>
    }

=cut

sub make_reports($;%) {
    my $shard = shift || die "shard is not defined";
    my %O = @_;

    my $log = report_log();

    my $error_stat = { map {$_ => 0} qw/cnt ok no_response other_errors/};
    my $queue = rqueue(thread => $O{thread});
    $log->out('start make_reports');
    my @reports_to_process = $queue->get_report_list_to_process(shard => $shard, 
                                                                limit => $O{limit} // $REPORTS_ITERATION_LIMIT,
                                                                $O{login} ? (login => $O{login}) : () );
    $log->out("selected " . scalar(@reports_to_process) . " reports to process");
    foreach my $report_item (@reports_to_process) {
        my $report_id = $report_item->{id};

        my $msg_prefix = $log->msg_prefix();
        $log->msg_prefix(join ' ', grep {$_} ($msg_prefix, "report_$report_id: "));

        $log->out('Starting report processing');
        my $report = $queue->get_report_info($report_item->{uid}, $report_id);
        $log->out('Report info is:', $report);
        next unless $report && $queue->start_processing_report($report, $shard);

        if ($O{on_before_new_report}) {
            $O{on_before_new_report}->($report);
        }

        ++$error_stat->{cnt};

        $log->out('Making report', $report);
        my $result = eval {
            local $SIG{ALRM} = sub { die "alarm timeout!" };
            alarm $REPORT_TIMEOUT;
            make_report($report);
        };
        alarm 0;
        if ($@) {
            $log->out('Error processing report:', $@);
            $result = {error_code => 1};
        }

        if ( !$result->{error_code} ) {
            $log->out('Report successfully made');
            ++$error_stat->{ok};

            $log->out('Creating XLSX file from report result');
            my $xls_report = Yandex::ReportsXLSX->new(no_optimization => _set_xlsx_optimization($result->{sheets_data}) ? 0 : 1);
            my $t0 = [gettimeofday];
            $report->{report_data} = $xls_report->array2excel2scalar($result->{sheets_data}, $result->{sheets_format});
            $log->out("xlsx file created in " . tv_interval($t0) . 's');

            $log->out('Saving XLSX file data');
            my $report_data_format = 'xlsx';
            $queue->save_report_data({id => $report->{id}, 
                                      uid => $report->{uid},
                                      report_data_parts => [$report->{report_data}], 
                                      report_data_format => $report_data_format});
            $log->out("report created");

            my $mail_vars = { uid => $report->{uid},
                              operator_uid => $report->{operator_uid},
                              operator_login => get_login(uid => $report->{operator_uid}),
                              client_name => $report->{client_name},
                              category_name => $report->{category_name},
                              report_id => $report->{id},
                              report_filename => get_report_filename({ready_time => now(), report_data_format => $report_data_format}),
                             };

            $log->out('Sending notification with mail_vars:', $mail_vars);
            add_notification(undef, 'client_potential_report_ready', $mail_vars);
            $log->out('report notification sent');
        } else {
            ++$error_stat->{other_errors};
        }
        $log->out('report processing finished');

        $log->msg_prefix($msg_prefix);
    }
    return $error_stat;
}

sub get_report_filename {
    my $report_opt = shift;
    my $ready_time_ymd = '';
    if ($report_opt->{ready_time}) {
        my ($y, $m, $d) = $report_opt->{ready_time} =~ /^(\d{4})-?(\d\d)-?(\d\d)(?:\D|$)/;
        if (check_date($y, $m, $d)) {
            $ready_time_ymd = sprintf('%4d%02d%02d', $y, $m, $d);
        }
    }

    return "potential_client_$ready_time_ymd.$report_opt->{report_data_format}";
}

=head2 _set_xlsx_optimization

    На основе размера данных, которые нужно выгрузить в xlsx-файл, решаем необходимо ли включать оптимизацию

=cut

sub _set_xlsx_optimization {
    my ($sheets_data) = @_;
    my $total_rows = 0;

    foreach my $sheet ( @{$sheets_data // []} ) {
        $total_rows += scalar @{$sheet // []};
    }
    return $total_rows > 100_000 ? 1 : 0;
}

=head2 _get_cids_by_logins

    По списку логинов, возвращает список кампаний подходящих для обработке в отчете

=cut

sub _get_cids_by_logins {
    my $logins = shift;
    my $login2uid = get_login2uid(login => $logins);
    return get_one_column_sql(PPC(uid => [values %$login2uid]), ["select cid from campaigns", where => { uid => SHARD_IDS, statusEmpty => 'No', archived => 'No', type__not_in => ['wallet', 'mcb', 'geo', 'billing_aggregate']}]);
}

=head2 xminus_phrases_dup

    На вход принимает ссылки на два массива - набора фраз
    На выходе ссылка на первый список фраз за вычетом фраз, которые являются дубликатами фраз из второго набора

=cut

sub xminus_phrases_dup {
    my ($list1, $list2) = @_;

    my %norm_list2 = map { Yandex::MyGoodWords::norm_words($_) => 1 } @$list2;
    return [grep { !$norm_list2{ Yandex::MyGoodWords::norm_words($_) } } @$list1];
}

=head2 xminus_phrases_sub

    На вход принимает ссылки на два массива - набора фраз
    На выходе ссылка на первый список фраз за вычетом фраз, которые являются вложенными к фразам из второго набора (включает и дубликаты)

=cut

sub xminus_phrases_sub {
    my ($list1, $list2) = @_;
    my @list1_ext = map { Lang::Unglue::get_phrase_extended_info({phrase => $_}) } @$list1;
    my @list2_ext = map { Lang::Unglue::get_phrase_extended_info({phrase => $_}) } @$list2;

    my @result = ();
    for (my $i=0; $i<@list1_ext; $i++) {
        if (none { Lang::Unglue::is_subphrase($_, $list1_ext[$i]) } @list2_ext) {
            push @result, $list1->[$i];
        }
    }

    return \@result;
}

=head2 remove_phrases_operators

    убирает некоторые операторы из ключевых фраз
    минус-слова остаются без изменений

    На вход принимает ссылку на массив фраз, который модифицирует

=cut

sub remove_phrases_operators {
    my $phrases = shift;

    foreach my $ph (@$phrases) {
        $ph = join ' ', map { m/^-/ ? $_ : s/[!"\[\]+]//gr } (split /\s+/, $ph);
    }
}

sub rqueue {
    my %O = @_;
    state $rqueue = new Reports::Queue(type => 'client_potential', 
                                       log => report_log(),
                                       $O{thread} ? (thread => $O{thread}) : ());
    return $rqueue;
}

sub dbstat {
    state $dbstat = Stat::CustomizedArray->new();
    return $dbstat;
}

1;
