package Reports::DynamicBanners;

=head2 DESCRIPTION

Формирование отчёта по динамическим объявлениям

=cut

use Direct::Modern;
use Carp;

use Mouse;
use Mouse::Util::TypeConstraints;

use Yandex::TimeCommon qw/ str_round str_round_day /;
use Yandex::DBTools;
use Yandex::HashUtils qw/ hash_merge /;
use Digest::MD5 qw/ md5 md5_hex /;
use Encode qw/encode_utf8/;
use List::Util qw/first/;
use List::MoreUtils qw/zip uniq/;

use Settings;

use Currency::Rate qw/convert_currency/;


our @HASH_FIELDS = qw/ SearchQuery Title Body Url ParentExportID /;
our %HASH_FIELD_POS = map {($HASH_FIELDS[$_] => $_)} (0 .. $#HASH_FIELDS);
our @SUM_FIELDS = qw/ :count Shows Clicks Cost CostCur GoalsNum SessionNum SessionDepth /;
our %SUM_FIELD_POS = map {($SUM_FIELDS[$_] => $_)} (0 .. $#SUM_FIELDS);


has group_by => (is => 'ro', required => 1, isa => enum [qw/ day week month year /]);

has record_count => (is => 'rw', isa => 'Int', default => 0);
has record_info => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has record_sum => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has record_total_sum => (is => 'ro', isa => 'ArrayRef', default => sub {+[]});
has _record_dates_hash => (is => 'ro', isa => 'HashRef', default => sub {+{}});

has base_currency => (is => 'ro', isa => 'Str');
has data_by_bid => (is => 'ro', isa => 'HashRef', default => sub {+{}});


=head2 add_record($record)

Добавляем к отчёту одну запись, полученную от БК

=cut

sub add_record {
    my ($self, $record) = @_;

    # DIRECT-49711: пропускать строки с нулевыми показами/кликами в ДТО
    if ($record->{Shows} == 0 && $record->{Clicks} == 0) {
        # в данных БК могут быть строки без показов и кликов (как результат откатов)
        # пропускаем и в ответ не добавляем
        return;
    }

    # отдельно запоминаем значения полей
    my @hash_values = map {_filter_field($record->{$_})} @HASH_FIELDS;
    my $hash = md5(map {encode_utf8 $_ . "\t"} @hash_values);
    $self->record_info->{$hash} ||= \@hash_values;

    # суммируем счётчики
    $self->record_count($self->record_count + 1);
    my $date = str_round($record->{UpdateTime}, 'day');
    $self->_record_dates_hash->{$date} ++;
    my $base_date = $self->get_base_date($record->{UpdateTime});
    my $count = $self->record_sum->{$base_date}->{$hash} ||= [];
    for my $pos (0 .. $#SUM_FIELDS) {
        my $field = $SUM_FIELDS[$pos];
        if ($field eq ':count') {
            $count->[$pos] ++;
            $self->record_total_sum->[$pos] ++;
        }
        elsif ($field eq 'CostCur') {
            my $cur = $self->_data_by_bid($record->{ParentExportID})->{currency} || 'YND_FIXED';
            my $value = $cur eq 'YND_FIXED' ? $record->{Cost} : $record->{CostCur};
            # если задана целевая валюта - пересчитываем
            if ($self->base_currency && $cur ne $self->base_currency) {
                $value = $self->base_currency eq 'YND_FIXED'
                    ? $record->{Cost}
                    : convert_currency($value, $cur, $self->base_currency, date => $date, with_nds => 1);
            }
            $count->[$pos] += $value;
            $self->record_total_sum->[$pos] += $value;
        }
        else {
            $count->[$pos] += $record->{$field};
            $self->record_total_sum->[$pos] += $record->{$field};
        }
    }

    return;
}

# фильтруем поля от проблемных символов
sub _filter_field {
    my ($s) = @_;
    $s =~ s/[\p{Unassigned}\p{Noncharacter_Code_Point}]//gxms;
    return $s;
}


sub _data_by_bid {
    my ($self, $bid) = @_;

    my $data = $self->data_by_bid;

    my $record = $data->{$bid} ||= get_one_line_sql(PPC(bid => $bid), [
            'SELECT b.bid, cid, c.name as campaign_name, pid, p.group_name,
                IFNULL(c.currency, "YND_FIXED") as currency
            FROM banners b
            JOIN phrases p USING(pid,cid)
            JOIN campaigns c USING(cid)',
            WHERE => { 'b.bid' => $bid },
        ]) || {};
    return $record;
}



=head2 get_base_date($date)

    my $rep = Reports::DynamicBanners->new(group_by => 'week')
    my $week_start = $rep->get_base_date('2015-07-24');
    # now $week_start == '2015-07-20'

Возвращает базовую дату диапазона группировки

=cut

sub get_base_date {
    my ($self, $date) = @_;

    my $rounded_date = str_round($date, $self->group_by);
    return $rounded_date;
}


=head2 get_iterator()

    my $it = $rep->get_iterator();
    while (my $record = $it->()) {
        # do something with $record
        ...
    }

Возвращает функцию-итератор, которая отдаёт поочерёдно все проагрегированные записи

=cut

sub get_iterator {
    my ($self) = @_;

    my $current_count = $self->record_count;
    my $dates = $self->get_group_dates();

    my $ri = $self->record_info;
    my @hashes =
        map {$_->[0]}
        sort { $a->[1] cmp $b->[1] }
        map { my $h = $_; [ $h => join "\t", map {$ri->{$h}->[$_]} (0 .. $#HASH_FIELDS) ] }
        keys %$ri;
    my %hash_skey = map {($hashes[$_] => $_)} (0 .. $#hashes);

    my $date;
    my $sums_by_date;
    my @date_hashes;
    my $hash_pos;

    return sub {
        croak "Data was updated since iterator created"  if $self->record_count != $current_count;

        if (!defined $hash_pos || $hash_pos > $#date_hashes) {
            $date = shift @$dates;
            return if !defined $date;

            $hash_pos = 0;
            $sums_by_date = $self->record_sum->{$date};
            @date_hashes = sort {$hash_skey{$a} <=> $hash_skey{$b}} keys %$sums_by_date;
        }

        my $hash = $date_hashes[$hash_pos];
        my $sums = $sums_by_date->{$hash};
        $hash_pos ++;

        my %record = (
            DateInterval => $date,
            zip(@HASH_FIELDS, @{$ri->{$hash}}),
            zip(@SUM_FIELDS, @$sums),
        );

        $self->_add_calc_fields(\%record);
        hash_merge \%record, $self->_data_by_bid($record{ParentExportID});
        return \%record;
    }
}


=head2 get_totals()

    my $totals = $rep->get_totals();

Возвращает итоговые суммы

=cut

sub get_totals {
    my ($self) = @_;

    my %record = zip(@SUM_FIELDS, @{$self->record_total_sum});
    $self->_add_calc_fields(\%record);

    return \%record;
}

=head2 get_group_dates

    my $dates = $rep->get_group_dates();

Возвращает список сгруппированных дат, по которым есть данные

=cut

sub get_group_dates {
    my ($self) = @_;
    return [ sort keys %{ $self->record_sum } ];
}


=head2 get_record_dates

    my $dates = $rep->get_record_dates();

Возвращает список всех дат, по которым есть данные

=cut

sub get_record_dates {
    my ($self) = @_;

    return [sort keys %{$self->_record_dates_hash}];
}



sub _add_calc_fields {
    my ($self, $record) = @_;
    return $record if !$record->{':count'};

    $record->{Cost} /= 1_000_000;
    $record->{CostCur} /= 1_000_000;

    # eval to avoid division by zero
    $record->{CTR}          = eval { $record->{Clicks} / $record->{Shows} };
    $record->{AvgClickCost} = eval { $record->{CostCur} / $record->{Clicks} };
    $record->{AvgDepth}     = eval { $record->{SessionDepth} / $record->{SessionNum} };
    $record->{AvgGoalCost}  = eval { $record->{CostCur} / $record->{GoalsNum} };
    $record->{Conversion}   = eval { $record->{GoalsNum} / $record->{SessionNum} };

    return $record;
}

__PACKAGE__->meta->make_immutable;
1;

