package Reports::Offline;

=head2 DESCRIPTION

Обёртка для прозрачной работы с офлайн-отчётами

=cut

use Direct::Modern;
use Exporter;

use Date::Calc;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_merge/;

use Reports::Offline::Broadmatch;
use Reports::Offline::Dynamic;
use Reports::Offline::Pdf;
use Reports::Offline::Performance;
use Reports::Offline::Postview;
use Reports::Offline::Xls;
use Reports::Offline::Video;

use Settings;



our @EXPORT_OK = qw/
    validate_report_params
/;



=head2 %REPORT_CLASS

Классы обработчиков отчётов.

Id сложились исторически, пока не меняем

=cut

our %REPORT_CLASS = (
    bm => 'Reports::Offline::Broadmatch',
    dynamic => 'Reports::Offline::Dynamic',
    offline => 'Reports::Offline::Xls',
    performance => 'Reports::Offline::Performance',
    pdf => 'Reports::Offline::Pdf',
    video => 'Reports::Offline::Video',
    $Reports::Offline::Postview::POSTVIEW_REPORT_TYPE => 'Reports::Offline::Postview',
);



sub _get_report_object {
    my ($type, @params) = @_;
    my $class = $REPORT_CLASS{$type};
    croak "Unknown report type <$type>" if !$class;
    return $class->new();
}


=head2 validate_report_params

Проверяет параметры заказанного отчёта

Возвращает список ошибок

=cut

# copypasted from PdfReport::checkReportOrder
sub validate_report_params {
    my ($vars) = @_;

    my @errors;

    my ($uid, $cids, $date_from, $date_to, $group, $type, $target_type) = map { $vars->{$_} } qw/uid cids date_from date_to group type target_type/;

    my $date_re = '^(20\d{2})\-(\d{2})\-(\d{2})$';

    my @d_from;
    my @d_to;
    if ( ($date_from||'') =~ m/$date_re/ and Date::Calc::check_date($1,$2,$3) ) {
        @d_from = map {int $_} ($1,$2,$3);
        $vars->{date_from} = $date_from = sprintf "%04d%02d%02d", @d_from;
    } elsif( ($date_from||'') =~ m/^\s*$/ ) {
        $vars->{date_from} = $date_from = undef;     #   earliest possible date
    } else {
        push @errors, iget("Указан неверный формат даты: %s", $date_from||'');
    }

    if ( ($date_to||'') =~ m/$date_re/ and Date::Calc::check_date($1,$2,$3) ) {
        @d_to = map {int $_} ($1,$2,$3);
        $vars->{date_to} = $date_to = sprintf "%04d%02d%02d", @d_to;
    } elsif ( ($date_to||'') =~ m/^\s*$/ ) {
        $vars->{date_to} = $date_to = undef;       #   latest possible date
    } else {
        push @errors, iget( "Указан неверный формат даты: %s", $date_to||'');
    }

    if ( @d_to and @d_from ) {
        push @errors, iget("Дата окончания не может быть меньше даты начала")
            unless Date::Calc::Delta_Days(@d_from, @d_to) >= 0;
    }

    if ( @d_to) {
        # Очень желательно поменять на "Дата начала периода" ... date_from
        push @errors, iget("Дата конца периода не может быть больше \"Сегодня\"")
            unless Date::Calc::Delta_Days(@d_to, Date::Calc::Today) >= 0;
    }

    if (($type //'') eq 'video') {
        if (defined $group && $group !~ /(day|banner,)*(day|banner)/) {
            push @errors, iget("Указан неверный период агрегации");
        }
    }
    else {
        unless ( defined $group and $group =~ m/^(none|day|week|month|year)$/) {
            push @errors, iget("Указан неверный период агрегации");
        }
    }

    if ( $target_type and $target_type !~ m/^(search|theme)$/) {
            push @errors, iget("Указана неверная площадка");
    }

    my @cids = grep { m/^\d+$/ } split /\D+/, $cids;
    if (@cids) {
        $vars->{cids} = join(',', @cids);
        $vars->{camps} =  get_all_sql(PPC(cid => \@cids), [
            "SELECT cid, OrderID, type, name FROM campaigns", WHERE => {cid => SHARD_IDS}
        ]);

        my %existed = map { $_->{cid} => 1 } @{$vars->{camps}};
        if (my @not_existed = grep { !$existed{$_} } @cids) {
            #   Return error unless all the campaigns are accessible
            push @errors, iget("Следующие кампании не существуют: %s", join(', ', @not_existed));
        }
        my $str_type = $type // '';
        if ($str_type eq 'dynamic') {
            if (my @not_dynamic_cids = map { $_->{cid} } grep { $_->{type} ne 'dynamic' } @{$vars->{camps}}) {
                push @errors, iget('Следующие кампании не являются динамическими: %s', join(', ', @not_dynamic_cids));
            }
        } elsif ($str_type eq 'performance') {
            if (my @not_performance_cids = map { $_->{cid} } grep { $_->{type} ne 'performance' } @{$vars->{camps}}) {
                push @errors, iget('Для формирования этого отчета необходимо указывать только кампании с типом "смарт-баннер". Следующие идентификаторы кампаний не относятся к этому типу: %s',
                                   join(', ', @not_performance_cids),
                                   );
            }
        }
    } else {
        push @errors, iget("Укажите хотя бы одну кампанию");
    }

    # type-dependent validation
    if ( my $report_object = _get_report_object($type) ) {
        push @errors, $report_object->validate_params($vars);
    }

    return @errors;
}



=head2 place_report_order($vars)

Помещаем заказ на отчёт в правильную очередь

=cut

sub place_report_order {
    my ($vars) = @_;

    my $type = $vars->{type};
    my $report_object = _get_report_object($type);

    return $report_object->place_order($vars);
}


=head2 get_reports_list($uid, %opt)

Получаем список всех отчётов пользователя

Опции:
    type - тип или список типов, если не задан - отдаём все

=cut

sub get_reports_list {
    my ($uid, %opt) = @_;

    my %allowed_type = map {$_ => 1} ( ref $opt{type} ? @{$opt{type}} : $opt{type} || () );

    my @reports;

    for my $type (keys %REPORT_CLASS) {
        next if %allowed_type && !$allowed_type{$type};
        my $report_object = _get_report_object($type);
        my $type_reports = $report_object->get_list($uid);
        push @reports, map {hash_merge {type => $type}, $_} @$type_reports;
    }

    return \@reports;
}


=head2 get_report_data($uid, $type => $id)

Возвращает файл с отчётом по id запроса

Возврат:
    - mime-type
    - binary data

=cut

sub get_report_data {
    my ($uid, $type, $id) = @_;

    my $report_object = _get_report_object($type);
    return $report_object->get_data($uid, $id);
}


=head2 delete_report($uid, $type => $id)

Удаляет файл с отчётом из базы

=cut

sub delete_report {
    my ($uid, $type, $id) = @_;

    my $report_object = _get_report_object($type);
    return $report_object->delete_data($uid, $id);
}


=head2 requeue_report_order($uid, $type => $id)

Повторно запрашивает отчёт

=cut

sub requeue_report_order {
    my ($uid, $type => $id) = @_;

    my $report_object = _get_report_object($type);
    return $report_object->requeue_order($uid, $id);
}


1;
