package Reports::Offline::Dynamic;

=head2 DESCRIPTION

Отчёт по динамическим объявлениям

=cut



use Direct::Modern;
use Mouse;

use Yandex::I18n;
use Yandex::DateTime qw/ date now /;
use Yandex::TimeCommon qw/get_distinct_dates str_round_day/;
use Yandex::DBTools;
use Direct::Storage;
use Yandex::Trace;

use JSON;
use Stat::SearchQuery::Queue;
use BS::SearchQuery;
use Reports::DynamicBanners;
use Reports::Offline::TextWriter qw/buffered_write_lines/;
use Currencies;
use PrimitivesIds qw/get_clientid/;

use User qw/get_user_data/;
use Yandex::ReportsXLS;
use Yandex::ReportsXLSX;
use Stat::Tools;
use List::MoreUtils qw/none uniq/;
use List::Util qw/pairkeys pairvalues/;

use Settings;

use API::Settings qw//;
use HashingTools qw/sha256_hex/;
use Path::Tiny;

use Log::Any '$log';


our $REPORT_MAX_AGE = { months => 3 };

our $COLLECTION = 'report_dynamic';

# TODO: использовать MIME-типы из словаря %Reports::MIMETypes::type2mime
our %FILE_TYPE = (
    xls => {
        key => 'xls_filename',
        mime => 'application/xls',
    },
    xlsx => {
        mime => 'application/xlsx',
    },
    tsv => {
        key => 'tsv_filename',
        mime => 'text/tab-separated-values',
    },
);



=head2 validate_params

Проверка параметров перед созданием отчёта

=cut

sub validate_params {
    my ($self, $vars) = @_;

    my @errors;

    my $min_date = now()->subtract(%$REPORT_MAX_AGE)->ymd("");
    my $date_from = $vars->{date_from};
    push @errors, iget("Дата начала периода для отчёта по динамическим текстовым объявлениям не может быть меньше %s", date($min_date)->strftime("%d.%m.%Y")) if $date_from and $date_from lt $min_date;

    my $nobs_camps = join q{, }, map {$_->{cid}} grep {!$_->{OrderID}} @{$vars->{camps}};
    push @errors, iget("Кампании %s ещё не были запущены", $nobs_camps)  if $nobs_camps;

    push @errors, iget("Невалидный параметр группировки")  if none {$vars->{group} eq $_} qw/ day week month year /;

    push @errors, iget("Отчёт с такими параметрами уже заказан")  if @{$self->_get_request_doubles($vars)};

    return @errors;
}


sub _get_request_doubles {
    my ($self, $vars) = @_;

    my $requests = $self->get_list($vars->{uid});
    my $date_from = $vars->{date_from} ? str_round_day($vars->{date_from}) : q{};
    my $date_to = $vars->{date_to} ? str_round_day($vars->{date_to}) : q{};
    my @doubles = grep {
#            $_->{cids} eq join(q{,}, sort map {$_->{cid}} @{$vars->{camps}})
            $_->{cids} eq $vars->{cids}
            && ($_->{date_from} || q{}) eq $date_from
            && ($_->{date_to} || q{}) eq $date_to
            && ($_->{group} || q{}) eq ($vars->{group} || q{})
        }
        @$requests;

    return \@doubles;
}


=head2 place_order

Помещение отчёта в очередь на расчёт

Опции:

    lang

=cut

sub place_order {
    my ($self, $vars, %opt) = @_;

    die "Duplicate report"  if @{$self->_get_request_doubles($vars)};

    my $lang = $opt{lang} || Yandex::I18n::current_lang();

    my $params = {
        CampaignIds => [ map {$_->{cid}} @{$vars->{camps}} ],
        StartDate => $vars->{date_from},
        EndDate => $vars->{date_to},
        GroupBy => $vars->{group},
        Lang => $lang,
    };
    Stat::SearchQuery::Queue::add_to_queue($vars->{uid}, dynamic => $params);

    return;
}


=head2 requeue_order($uid, $id)

Перезапрос отчёта

=cut

sub requeue_order {
    my ($self, $uid, $id) = @_;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    return if !$request;

    Stat::SearchQuery::Queue::update_queue($id, rank => 0);

    return;
}


=head2 get_list($uid)

Возвращает список отчётов пользователя

=cut

sub get_list {
    my ($self, $uid) = @_;

    my $reports = get_all_sql(PPCDICT, [
            'SELECT id, status, options,
                timecreate as create_time,
                timeready as ready_time
            FROM api_queue_search_query',
            WHERE => {
                uid => $uid,
                type => 'dynamic',
                status => ['New', 'Process', 'Done'],
            },
        ]);

    for my $report (@$reports) {
        my $options = from_json delete $report->{options};
        $report->{cids} = join q{,}, sort @{$options->{CampaignIds}};
        $report->{date_from} = str_round_day($options->{StartDate} || q{});
        $report->{date_to} = str_round_day($options->{EndDate} || q{});
        $report->{group} = $options->{GroupBy};
        $report->{status_no_data} = $report->{status} eq 'Done' && ! $options->{xls_filename} ? 'Yes' : 'No';
        $report->{status} = {New=>'new', Process=>'processing', Done=>'ready'}->{$report->{status}};
        $report->{file_format} = ($options->{xls_filename} || q{}) =~ /\.xlsx$/ ? 'xlsx' : 'xls';
        $report->{is_fail} = 0;
    }

    return $reports;
}

=head2 get_report_options($uid, $id)

Возвращает параметры запроса отчёта пользователя

=cut

sub get_report_options {
    my ($self, $uid, $id) = @_;

    my $options_json = get_one_field_sql(PPCDICT, [
            'SELECT options
            FROM api_queue_search_query',
            WHERE => {
                uid => $uid,
                id => $id,
                type => 'dynamic',
                status => ['New', 'Process', 'Done'],
            },
        ]);

    my $options = {};
    if ($options_json) {
        $options = from_json($options_json);
    }

    return $options;
}

=head2 get_data($uid, $id)

Возвращает файл с отчётом

=cut


sub get_data {
    my ($self, $uid, $id, $type) = @_;

    $type ||= 'xls';
    my $key = $FILE_TYPE{$type}->{key};
    croak "Unsupported type <$type>"  if !$key;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    my $filename = $request->{options}->{$key};
    return if !$filename;

    my $storage = Direct::Storage->new();
    my $file = $storage->get_file($COLLECTION, filename => $filename, uid => $uid);
    return unless $file;

    $type = 'xlsx'  if $filename =~ /\.xlsx$/;
    my $mime = $FILE_TYPE{$type}->{mime};
    return $mime => $file->content;
}


=head2 delete_data($uid, $id)

Удаляет отчёт

=cut

sub delete_data {
    my ($self, $uid, $id) = @_;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    return if !$request;

    Stat::SearchQuery::Queue::remove_from_queue($id);

    for my $key (qw/ xls_filename tsv_filename /) {
        my $filename = $request->{options}->{$key};
        next if !$filename;

        $self->delete_report_file($filename, $uid);
    }
    return;
}


=head2 delete_report_file($filename)

Удаляет файл с отчётом из хранилища

=cut

sub delete_report_file {
    my ($self, $filename, $uid) = @_;

    my $storage = Direct::Storage->new();
    eval {
        $storage->delete_file($COLLECTION, filename => $filename, uid => $uid);
    };
    return;
}


=head2 generate_report($uid, $id)

Получает отчёт из БК и сохраняет в хранилище

=cut

sub generate_report {
    my ($self, $uid, $id) = @_;

    my $client_id = get_clientid(uid => $uid);
    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    my %report_opt = (group_by => $request->{options}->{GroupBy});

    my $lang = $request->{options}->{Lang} || 'ru';
    my $guard = Yandex::I18n::init_i18n_guard($lang);

    my $cids = $request->{options}->{CampaignIds};
    my $currencies = get_one_column_sql(PPC(shard => 'all'), [
            'SELECT IFNULL(currency, "YND_FIXED") FROM campaigns',
            WHERE => { cid => $cids },
        ]);
    my $report_currency = Currencies::get_dominant_currency($currencies);
    $report_opt{base_currency} = $report_currency  if uniq(@$currencies) > 1;

    $log->info("Fetching report from BS");
    my $report = $self->_bs_fetch_report($request->{bs_id}, %report_opt);

    $log->info("Generating XLS");
    my ($format, $file) = $self->_get_report_xls($request, $report, $report_currency);

    $log->info("Saving XLS");
    my $storage = Direct::Storage->new();
    if ($file) {
        $log->trace("Saving XLS to Storage");
        my $filename = $request->{options}->{xls_filename} = "report_$request->{id}.$format";
        $storage->save($COLLECTION, $file->slurp_raw(), filename => $filename, ClientID => $client_id);
        $log->trace("remove temporary XLS file");
        $file->remove();
    } else {
        $log->trace("Empty report, skip XLS saving");
    }

    $log->info("Generating TSV");
    $file = $self->_get_report_tsv($request, $report);
    my $hashed_name = sha256_hex(join("--", $uid, $request->{id}, $API::Settings::API_SECRET_PHRASE)); # для получения без (другой) аутентификации
    my $tsv_filename = $request->{options}->{tsv_filename} = "report_$hashed_name.tsv";
    $log->trace("Saving TSV to Storage");
    $storage->save($COLLECTION, $file->slurp_raw(), filename => $tsv_filename, ClientID => $client_id);
    $log->trace("remove temporary TSV file");
    $file->remove();

    $log->info("Updating queue");
    Stat::SearchQuery::Queue::update_queue($request->{id}, options => $request->{options});

    $log->info("Done");
    return;
}


sub _bs_fetch_report {
    my ($self, $bs_id, %opt) = @_;

    my $report = Reports::DynamicBanners->new(%opt);
    my $bs = BS::SearchQuery->new(ua_timeout => 300);
    my $status;
    my $chunk_cb = $bs->make_chunk_callback(
        status => \$status,
        record => sub { $report->add_record(@_) },
#        chunk => sub { print STDERR @_ },
    );
    my $error = $bs->get_chunked_data($bs_id, $chunk_cb);
    $error ||= "Status: $status"  if $status;

    croak "BS returned error: $error"  if $error;

    return $report;
}


sub _get_report_tsv {
    my ($self, $request, $report) = @_;
    
    my $profile = Yandex::Trace::new_profile('reports:offline:dynamic:_get_report_tsv');
    my @columns_definition = (
        # data_name     => report_name
        'date'          => 'Date',
        'campaign_name' => 'CampaignName',
        'cid'           => 'CampaignId',
        'group_name'    => 'AdGroupName',
        'pid'           => 'AdGroupId',
        'bid'           => 'AdId',
        'Title'         => 'Title',
        'Body'          => 'Text',
        'Url'           => 'Url',
        'SearchQuery'   => 'SearchQuery',
        'Shows'         => 'Impressions',
        'Clicks'        => 'Clicks',
        'CTR'           => 'CTR',
        'CostCur'       => 'Cost',
        'AvgClickCost'  => 'AvgClickCost',
        'AvgDepth'      => 'AvgDepth',
        'Conversion'    => 'Conversion',
        'AvgGoalCost'   => 'AvgGoalCost',
        'GoalsNum'      => 'GoalsNum',
    );
    my %col_names_translation = @columns_definition;
    my @cols = pairkeys @columns_definition;
    my @ext_col_names = pairvalues @columns_definition;

    my %fields = map {$_ => undef} @{$request->{options}->{FieldNames} || \@ext_col_names };

    my @cols_actual = grep {exists $fields{$col_names_translation{$_}}} @cols;
    my @ext_cols_actual = map {$col_names_translation{$_}} @cols_actual;

    $log->info("Preparing TSV data");
    my $it = $report->get_iterator();
    my $current_date = q{};
    my $row_gen = sub {
        my $record = $it->();
        return if !$record;
        if ($current_date ne $record->{DateInterval}) {
            $current_date = $record->{DateInterval};
            $log->info($current_date);
        }
        if (exists $fields{Date}) {
            $record->{date} = Stat::Tools::format_date($record->{DateInterval}, $request->{options}->{GroupBy});
        }

        return join "\t", map {$_ // ''} @{$record}{@cols_actual};
    };

    my $file = Path::Tiny->tempfile();
    $log->info("Writing to tmpfile $file");
    my $header = join "\t", @ext_cols_actual;
    buffered_write_lines($row_gen, $header, $file);
    return $file;
}

sub _get_report_xls {
    my ($self, $request, $report, $currency) = @_;

    my $profile = Yandex::Trace::new_profile('reports:offline:dynamic:_get_report_xls');
    my @xls_data;

    $log->info("Preparing XLS totals");
    my $user = get_user_data($request->{uid}, [qw/login fio/]);
    my $period_str = $request->{options}->{StartDate}
        ? iget("c %s по %s",
            date($request->{options}->{StartDate})->strftime('%d.%m.%Y'),
            date($request->{options}->{EndDate})->strftime('%d.%m.%Y')
        )
        : iget("за всё время");
    my $currency_name = get_currency_constant($currency, 'name');

    my $title = iget('Клиент %s (%s), %s',
        $user->{fio},
        $user->{login},
        $period_str,
    );
    push @xls_data, [{ data => $title, format => { bold => 1, size => 14 } }];

    my $avg_outcome_str = {
        day => iget("Ср.расход за день (%s)", $currency_name),
        week => iget("Ср.расход за неделю (%s)", $currency_name),
        month => iget("Ср.расход за месяц (%s)", $currency_name),
        year => iget("Ср.расход за год (%s)", $currency_name),
     }->{$request->{options}->{GroupBy}};

    push @xls_data, [
        map { $_ && {data=>$_, format=>{bold=>1}} }
            iget("Период"),
            $avg_outcome_str,
            iget("Показы"),
            iget("Клики"),
            iget("CTR (%)"),
            iget("Расход (%s)", $currency_name),
            iget("Ср. цена клика (%s)", $currency_name),
            iget("Глубина (стр.)"),
            iget("Конверсия (%)"),
            iget("Цена цели (%s)", $currency_name),
            iget("Конверсии"),
    ];

    my $totals = $report->get_totals();
    my $num_dates = @{$report->get_group_dates()};
    push @xls_data, [
        map { $_ && {data=>$_, format=>{bold=>1}} }
        $period_str,
        _format_float($num_dates ? $totals->{CostCur} / $num_dates : undef),
        $totals->{Shows},
        $totals->{Clicks},
        _format_float(($totals->{CTR} || 0) * 100),
        _format_float($totals->{CostCur}),
        _format_float($totals->{AvgClickCost}),
        _format_float($totals->{AvgDepth}),
        _format_float(($totals->{Conversion} || 0) * 100),
        _format_float($totals->{AvgGoalCost}),
        $totals->{GoalsNum},
    ];

    push @xls_data, [undef];

    $log->info("Preparing XLS data");
    push @xls_data, [
        map { $_ && {data=>$_, format=>{bold=>1}} }
            iget("Дата"),
            iget("Кампания"),
            iget("№ Кампании"),
            iget("Группа"),
            iget("№ Группы"),
            iget("№ Объявления"),
            iget("Заголовок объявления"),
            iget("Текст объявления"),
            iget("Ссылка объявления"),
            iget("Поисковый запрос"),
            iget("Показы"),
            iget("Клики"),
            iget("CTR (%)"),
            iget("Расход (%s)", $currency_name),
            iget("Ср. цена клика (%s)", $currency_name),
            iget("Глубина (стр.)"),
            iget("Конверсия (%)"),
            iget("Цена цели (%s)", $currency_name),
            iget("Конверсии"),
    ];

    my $it = $report->get_iterator();
    my $current_date = q();
    my $not_empty_stat;
    while (my $record = $it->()) {
        if ($current_date ne $record->{DateInterval}) {
            $current_date = $record->{DateInterval};
            $log->info($current_date);
        }
        $not_empty_stat //= 1;
        push @xls_data, [
            Stat::Tools::format_date($record->{DateInterval}, $request->{options}->{GroupBy}),
            $record->{campaign_name},
            { data => $record->{cid}, as_text => 1 },
            $record->{group_name},
            { data => $record->{pid}, as_text => 1 },
            iget('M-%d', $record->{bid}),
            $record->{Title},
            $record->{Body},
            { data => $record->{Url}, as_text => 1 },
            { data => $record->{SearchQuery}, as_text => 1 },
            $record->{Shows},
            $record->{Clicks},
            _format_float(($record->{CTR} || 0) * 100),
            _format_float($record->{CostCur}),
            _format_float($record->{AvgClickCost}),
            _format_float($record->{AvgDepth}),
            _format_float(($record->{Conversion} || 0) * 100),
            _format_float($record->{AvgGoalCost}),
            $record->{GoalsNum},
        ];
    }

    if (!$not_empty_stat) {
        # если нет данных - не создаем xls-файл
        return (undef, undef);
    }

    my $file = Path::Tiny->tempfile();
    $log->info("Writing to tmpfile $file");
    my $xls_format = {
        output => $file->openw_raw(),
        sheetname => iget("Динамические объявления"),
        set_row => [
            {row => 0, height => 20},
        ],
        set_column => [
            {col1 => 0, count => 0, width => 24},
        ],


    };

    my $format = @xls_data > 65000 ? 'xlsx' : 'xls';
    my $xls_writer = $format eq 'xlsx' ? Yandex::ReportsXLSX->new() : Yandex::ReportsXLS->new();

    $xls_writer->array2excel([\@xls_data], [$xls_format]);
    return $format => $file;
}


sub _format_float {
    my $num = shift;
    return "-"  if !$num;
    return 0 + sprintf("%.2f", $num);
}

__PACKAGE__->meta->make_immutable();
1;

