package Reports::Offline::Performance;

=head2 DESCRIPTION

Отчёт по динамическим медийным объявлениям

=cut

use Direct::Modern;

use JSON;
use List::MoreUtils qw/none uniq/;
use List::Util qw/pairkeys pairvalues/;
use Log::Any '$log';
use Mouse;
use Path::Tiny;

use Yandex::CSV;
use Yandex::DateTime qw/ date now /;
use Yandex::DBTools;
use Yandex::I18n;
use Direct::Storage;
use Yandex::ReportsXLSWriter;
use Yandex::ReportsXLSXWriter;
use Yandex::TimeCommon qw/str_round_day/;
use Yandex::Trace;

use API::Settings qw//;
use BS::SearchQuery;
use Currencies;
use HashingTools qw/sha256_hex/;
use Reports::MIMETypes;
use Reports::PerformanceBanners;
use Settings;
use Stat::SearchQuery::Queue;
use Stat::Tools;
use User qw/get_user_data/;
use PrimitivesIds qw/get_clientid/;


our $REPORT_MAX_AGE = { months => 3 };

our $COLLECTION = 'report_performance';

our %FILE_TYPE = (
    xls => 'xls_filename',
    xlsx => undef,
    csv => 'csv_filename',
);

my $MAX_CATEGORIES = 5;
my $BS_FETCH_TIMEOUT = 600;
my $MAX_XLS_ROWS = 2**16;
my $MAX_XLSX_STAT_ROWS_NUM = 1_000_000;
my $XLS_ITERATOR_CHUNK_SIZE = 10_000;


=head2 validate_params

Проверка параметров перед созданием отчёта

=cut

sub validate_params {
    my ($self, $vars) = @_;

    my @errors;

    my $min_date = now()->subtract(%$REPORT_MAX_AGE)->ymd("");
    my $date_from = $vars->{date_from};
    push @errors, iget("Дата начала периода для отчёта не может быть меньше %s", date($min_date)->strftime("%d.%m.%Y")) if $date_from and $date_from lt $min_date;

    my $nobs_camps = join q{, }, map {$_->{cid}} grep {!$_->{OrderID}} @{$vars->{camps}};
    push @errors, iget("Кампании %s ещё не были запущены", $nobs_camps)  if $nobs_camps;

    push @errors, iget("Невалидный параметр группировки")  if none {$vars->{group} eq $_} qw/ day week month year /;

    push @errors, iget("Отчёт с такими параметрами уже заказан")  if @{$self->_get_request_doubles($vars)};

    return @errors;
}


sub _get_request_doubles {
    my ($self, $vars) = @_;

    my $requests = $self->get_list($vars->{uid});
    my $date_from = $vars->{date_from} ? str_round_day($vars->{date_from}) : q{};
    my $date_to = $vars->{date_to} ? str_round_day($vars->{date_to}) : q{};
    my @doubles = grep {
            $_->{cids} eq $vars->{cids}
            && ($_->{date_from} || q{}) eq $date_from
            && ($_->{date_to} || q{}) eq $date_to
            && ($_->{group} || q{}) eq ($vars->{group} || q{})
            && ($_->{add_order_id} // 0) == ($vars->{add_order_id} // 0)
            && ($_->{add_categories} // 0) == ($vars->{add_categories} // 0)
            && ($_->{add_vendor} // 0) == ($vars->{add_vendor} // 0)
        }
        @$requests;

    return \@doubles;
}


=head2 $self->place_order($vars, %opt)

  Помещение отчёта в очередь на расчёт
  
  Опции:
    lang - язык генераци отчета

=cut

sub place_order {
    my ($self, $vars, %opt) = @_;

    die "Duplicate report"  if @{$self->_get_request_doubles($vars)};

    my $lang = $opt{lang} || Yandex::I18n::current_lang();
    my $params = {
        CampaignIds => [ map {$_->{cid}} @{$vars->{camps}} ],
        StartDate => $vars->{date_from},
        EndDate => $vars->{date_to},
        GroupBy => $vars->{group},
        Lang => $lang,
        AddOrderID => $vars->{add_order_id} // 0,
        AddCategories => $vars->{add_categories} // 0,
        AddVendor => $vars->{add_vendor} // 0,
    };
    Stat::SearchQuery::Queue::add_to_queue($vars->{uid}, performance => $params);

    return;
}


=head2 requeue_order($uid, $id)

Перезапрос отчёта

=cut

sub requeue_order {
    my ($self, $uid, $id) = @_;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    return if !$request;

    Stat::SearchQuery::Queue::update_queue($id, rank => 0);

    return;
}


=head2 get_list($uid)

Возвращает список отчётов пользователя

=cut

sub get_list {
    my ($self, $uid) = @_;

    my $all_reports = get_all_sql(PPCDICT, [
            'SELECT id, status, options,
                timecreate as create_time,
                timeready as ready_time
            FROM api_queue_search_query',
            WHERE => {
                uid => $uid,
                type => 'performance',
                status => ['New', 'Process', 'Done'],
            },
        ]);

    my $reports = [];
    for my $report (@$all_reports) {
        my $options = from_json delete $report->{options};

        $report->{cids} = join q{,}, sort @{$options->{CampaignIds}};
        $report->{date_from} = str_round_day($options->{StartDate} || q{});
        $report->{date_to} = str_round_day($options->{EndDate} || q{});
        $report->{group} = $options->{GroupBy};
        $report->{status_no_data} = $report->{status} eq 'Done' && ! $options->{xls_filename} ? 'Yes' : 'No';
        $report->{status} = {New=>'new', Process=>'processing', Done=>'ready'}->{$report->{status}};
        $report->{file_format} = ($options->{xls_filename} || q{}) =~ /\.xlsx$/ ? 'xlsx' : 'xls';
        $report->{is_fail} = 0;
        $report->{add_order_id} = $options->{AddOrderID};
        $report->{add_categories} = $options->{AddCategories};
        $report->{add_vendor} = $options->{AddVendor};
        push @$reports, $report;
    }

    return $reports;
}


=head2 get_data($uid, $id, $type)

Возвращает файл с отчётом

=cut


sub get_data {
    my ($self, $uid, $id, $type) = @_;

    $type ||= 'xls';
    my $key = $FILE_TYPE{$type};
    croak "Unsupported type <$type>"  if !$key;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    my $filename = $request->{options}->{$key};
    return if !$filename;

    my $storage = Direct::Storage->new();
    my $file = $storage->get_file($COLLECTION, filename => $filename, ClientID => get_clientid(uid => $uid));
    return unless $file;

    $type = 'xlsx'  if $filename =~ /\.xlsx$/;
    my $mime = $Reports::MIMETypes::type2mime{$type};
    return $mime => $file->content;
}


=head2 delete_data($uid, $id)

Удаляет отчёт

=cut

sub delete_data {
    my ($self, $uid, $id) = @_;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    return if !$request;

    Stat::SearchQuery::Queue::remove_from_queue($id);

    for my $key (qw/ xls_filename csv_filename /) {
        my $filename = $request->{options}->{$key};
        next if !$filename;

        $self->delete_report_file($filename, $uid);
    }
    return;
}


=head2 delete_report_file($filename, $uid)

Удаляет файл с отчётом из storage

=cut

sub delete_report_file {
    my ($self, $filename, $uid) = @_;

    my $storage = Direct::Storage->new();
    eval {
        $storage->delete_file($COLLECTION, filename => $filename, ClientID => get_clientid(uid => $uid));
    };
    return;
}

=head2 get_report_options($uid, $id)

Возвращает параметры запроса отчёта пользователя

=cut

sub get_report_options {
    my ($self, $uid, $id) = @_;

    my $options_json = get_one_field_sql(PPCDICT, [
            'SELECT options
            FROM api_queue_search_query',
            WHERE => {
                uid => $uid,
                id => $id,
                type => 'performance',
                status => ['New', 'Process', 'Done'],
            },
        ]);

    my $options = {};
    if ($options_json) {
        $options = from_json($options_json);
    }

    return $options;
}


=head2 generate_report($uid, $id)

Получает отчёт из БК и сохраняет в storage

=cut

sub generate_report {
    my ($self, $uid, $id) = @_;

    my $request = Stat::SearchQuery::Queue::get_from_queue($uid, $id);
    my $lang_guard = Yandex::I18n::init_i18n_guard($request->{options}->{Lang} || 'ru');
    my %report_opt = (group_by => $request->{options}->{GroupBy});
    my %pid2categories;

    my $cids = $request->{options}->{CampaignIds};
    my $currencies = get_one_column_sql(PPC(shard => 'all'), [
            'SELECT IFNULL(currency, "YND_FIXED") FROM campaigns',
            WHERE => { cid => $cids },
        ]);
    my $report_currency = Currencies::get_dominant_currency($currencies);
    $report_opt{base_currency} = $report_currency  if uniq(@$currencies) > 1;

    $log->trace("Fetching report from BS");
    my $report = $self->_bs_fetch_report($request->{bs_id}, %report_opt);

    $log->trace("Generating XLS");
    my ($format, $file) = $self->_get_report_xls($request, $report, $report_currency, \%pid2categories);
    if ($file) {
        my $filename = $request->{options}->{xls_filename} = "report_$request->{id}.$format";
        $log->trace("Saving XLS to Storage");
        my $storage = Direct::Storage->new();
        $storage->save($COLLECTION, \$file->slurp_raw, filename => $filename, ClientID => get_clientid(uid => $uid));
        $log->trace("remove temporary XLS file");
        $file->remove();
    } else {
        $log->trace("Empty report, skip XLS saving");
    }

    $log->trace("Generating CSV");
    $file = $self->_get_report_csv($request, $report, \%pid2categories);
    my $hashed_name = sha256_hex(join("--", $uid, $request->{id}, $API::Settings::API_SECRET_PHRASE)); # для получения без (другой) аутентификации
    my $csv_filename = $request->{options}->{csv_filename} = "report_$hashed_name.csv";
    $log->trace("Saving CSV to Storage");
    my $storage = Direct::Storage->new();
    $storage->save($COLLECTION, \$file->slurp_raw, filename => $csv_filename, ClientID => get_clientid(uid => $uid));
    $log->trace("remove temporary CSV file");
    $file->remove();

    $log->trace("Updating queue");
    Stat::SearchQuery::Queue::update_queue($request->{id}, options => $request->{options});

    $log->info("Done");
    return;
}


sub _bs_fetch_report {
    my ($self, $bs_id, %opt) = @_;

    my $report = Reports::PerformanceBanners->new(%opt);
    my $bs = BS::SearchQuery->new(ua_timeout => $BS_FETCH_TIMEOUT);
    my $status;
    my $chunk_cb = $bs->make_chunk_callback(
        status => \$status,
        record => sub { $report->add_record(@_) },
    );
    my $error = $bs->get_chunked_data($bs_id, $chunk_cb);
    $error ||= "Status: $status"  if $status;

    croak "BS returned error: $error"  if $error;

    return $report;
}

sub _get_report_csv {
    my ($self, $request, $report, $pid2categories) = @_;

    my $profile = Yandex::Trace::new_profile('reports:offline:performance:_get_report_csv');
    
    my @columns_definition = (
        # data_name     => report_name
        'date'          => 'Date',
        'cid'           => 'CampaignId',
        'campaign_name' => 'CampaignName',
        'OrderID'       => 'OrderID',
        'group_name'    => 'AdGroupName',
        'pid'           => 'AdGroupId',
        'bid'           => 'AdId',
        'OfferID'       => 'OfferID',
    );

    for my $c_num (1..$MAX_CATEGORIES) {
        push @columns_definition, ("Category$c_num" => "Category$c_num");
    }

    push @columns_definition, (
        'Vendor'        => 'Vendor',
        'MarketVendor'  => 'MarketVendor',
        'Title'         => 'Title',
        'Url'           => 'Url',
        'PhraseID'      => 'CriteriaId',
        'filter_name'   => 'Filter',
        'Shows'         => 'Shows',
        'Clicks'        => 'Clicks',
        'CTR'           => 'CTR',
        'CostCur'       => 'Cost',
        'AvgClickCost'  => 'AvgClickCost',
        'AvgDepth'      => 'AvgDepth',
        'Conversion'    => 'Conversion',
        'AvgGoalCost'   => 'AvgGoalCost',
        'GoalsNum'      => 'GoalsNum',
    );

    my %col_names_translation = @columns_definition;
    my @col_names = pairkeys @columns_definition;
    my @ext_col_names = pairvalues @columns_definition;

    my %fields;
    for my $col_name (@ext_col_names) {
        if ($col_name eq 'OrderID') {
            $fields{$col_name} = undef if $request->{options}->{AddOrderID};
        } elsif ($col_name =~ /^(?:Market)?Vendor$/) {
            $fields{$col_name} = undef if $request->{options}->{AddVendor};
        } elsif ($col_name =~ /^Category\d+$/) {
            $fields{$col_name} = undef if $request->{options}->{AddCategories};
        } else {
            $fields{$col_name} = undef;
        }
    }

    my @col_names_actual = grep {exists $fields{$col_names_translation{$_}}} @col_names;
    my @ext_cols_actual = map {$col_names_translation{$_}} @col_names_actual;

    $log->trace("Preparing CSV data");
    my $it = $report->get_iterator();
    my $current_date = q{};
    my $row_gen = sub {
        my $record = $it->();
        return if !$record;
        if ($current_date ne $record->{DateInterval}) {
            $current_date = $record->{DateInterval};
            $log->trace($current_date);
        }
        if (exists $fields{Date}) {
            $record->{date} = Stat::Tools::format_date($record->{DateInterval}, $request->{options}->{GroupBy});
        }
        if ($request->{options}->{AddCategories}) {
            my @category_names = _get_category_names($pid2categories, $record->{pid}, $record->{CategoryID});
            for my $c_num (1..$MAX_CATEGORIES) {
                $record->{"Category$c_num"} = $category_names[$c_num - 1];
            }
        }
        return @{$record}{@col_names_actual};
    };

    my $file = Path::Tiny->tempfile();
    $log->trace("Writing to tmpfile $file");
    data2csv($row_gen, {bom_header => 1, header_row => \@ext_cols_actual, output_file => "$file" });
    return $file;
}

sub _get_report_xls {
    my ($self, $request, $report, $currency, $pid2categories) = @_;

    my $profile = Yandex::Trace::new_profile('reports:offline:performance:_get_report_xls');
    my @xls_data;

    $log->trace("Preparing XLS totals");
    my $user = get_user_data($request->{uid}, [qw/login fio/]);
    my $period_str = $request->{options}->{StartDate}
        ? iget("c %s по %s",
            date($request->{options}->{StartDate})->strftime('%d.%m.%Y'),
            date($request->{options}->{EndDate})->strftime('%d.%m.%Y')
        )
        : iget("за всё время");
    my $currency_name = get_currency_constant($currency, 'name');
    my $title = iget('Клиент %s (%s), %s',
        $user->{fio},
        $user->{login},
        $period_str,
    );
    push @xls_data, [{ data => $title, format => { bold => 1, size => 14 } }];

    my $avg_outcome_str = {
        day => iget("Ср.расход за день (%s)", $currency_name),
        week => iget("Ср.расход за неделю (%s)", $currency_name),
        month => iget("Ср.расход за месяц (%s)", $currency_name),
        year => iget("Ср.расход за год (%s)", $currency_name),
     }->{$request->{options}->{GroupBy}};
    push @xls_data, [
        map { $_ && {data=>$_, format=>{bold=>1}} }
            iget("Период"),
            $avg_outcome_str,
            iget("Показы"),
            iget("Клики"),
            iget("CTR (%)"),
            iget("Расход (%s)", $currency_name),
            iget("Ср. цена клика (%s)", $currency_name),
            iget("Глубина (стр.)"),
            iget("Конверсии"),
            iget("Конверсия (%)"),
            iget("Цена цели (%s)", $currency_name),
    ];

    my $totals = $report->get_totals();
    my $num_dates = @{$report->get_group_dates()};
    push @xls_data, [
        map { $_ && {data=>$_, format=>{bold=>1}} }
        $period_str,
        _format_float($num_dates ? $totals->{CostCur} / $num_dates : undef),
        $totals->{Shows},
        $totals->{Clicks},
        _format_float(($totals->{CTR} || 0) * 100),
        _format_float($totals->{CostCur}),
        _format_float($totals->{AvgClickCost}),
        _format_float($totals->{AvgDepth}),
        $totals->{GoalsNum},
        _format_float(($totals->{Conversion} || 0) * 100),
        _format_float($totals->{AvgGoalCost}),
    ];

    push @xls_data, [undef];

    $log->trace("Preparing XLS data");

    my $header = _get_xls_header($request, $currency_name);
    push @xls_data, $header;

    my ($data_writer, $xls_worksheet, $format);

    my $file = Path::Tiny->tempfile();
    $log->trace("Writing to tmpfile $file");
    my $xls_format = {
        sheetname => iget('Статистика'),
        set_row => [
            {row => 0, height => 20},
        ],
        set_column => [
            {col1 => 0, count => 0, width => 24},
        ],
    };

    my $it = $report->get_iterator();
    my $current_date = q();
    my $not_empty_stat;
    my $stat_rows_num = 0;
    my $stat_rows_limit_exceeded = 0;

    # "лямбда" для более читаемой реализации манипуляций с xls/xlsx
    my $flush_rows = sub {
        my $is_last_flush = shift;
        if (!$data_writer && @xls_data && ($is_last_flush || @xls_data > $MAX_XLS_ROWS)) {
            if (@xls_data > $MAX_XLS_ROWS) {
                $data_writer = new Yandex::ReportsXLSXWriter($file->openw_raw(), no_optimization => 1);
                $format = 'xlsx';
            } else {
                $data_writer = new Yandex::ReportsXLSWriter($file->openw_raw());;
                $format = 'xls';
            }
            $xls_worksheet = $data_writer->add_worksheet($xls_format->{sheetname});
        }

        if ($data_writer && ($is_last_flush || @xls_data >= $XLS_ITERATOR_CHUNK_SIZE)) {
            $data_writer->add_data($xls_worksheet, $xls_format, \@xls_data);
            @xls_data = ();
        }
    };

    while (my $record = $it->()) {
        if ($current_date ne $record->{DateInterval}) {
            $current_date = $record->{DateInterval};
            $log->trace($current_date);
        }
        $not_empty_stat //= 1;

        if ($stat_rows_num < $MAX_XLSX_STAT_ROWS_NUM) {
            push @xls_data, _get_xls_row($record, $request, $pid2categories);
            $stat_rows_num++;
        } else {
            $stat_rows_limit_exceeded = 1;
            last;
        }

        $flush_rows->();
    }
    $flush_rows->('last');

    if ($stat_rows_limit_exceeded && $data_writer && $xls_worksheet) {
        # предупреждения в шапке прийдется добавлять низкоуровневыми средствами
        $xls_worksheet->{xls_worksheet}->write_string(0, 8, iget("Файл содержит первые %s строк отчета. Сократите период данных при заказе отчета, чтобы получить полные данные.", $MAX_XLSX_STAT_ROWS_NUM), $data_writer->make_format(color => 'red'));
    }
    $data_writer->close() if $data_writer;

    if (!$not_empty_stat) {
        # если нет данных - не создаем xls-файл
        return (undef, undef);
    }

    return $format => $file;
}


sub _get_xls_header {
    my ($request, $currency_name) = @_;

    my $xls_header = [];
    push @$xls_header, (
        iget( "Дата" ),
        iget( "Кампания" ),
        iget( "№ Кампании" ),
    );

    if ($request->{options}->{AddOrderID}) {
        push @$xls_header, iget( "№ Кампании в БК" );
    }

    push @$xls_header, (
        iget( "Группа" ),
        iget( "№ Группы" ),
        iget( "№ Объявления" ),
        iget( "id оффера" ),
    );

    if ($request->{options}->{AddCategories}) {
        for my $c_num (1..$MAX_CATEGORIES) {
            push @$xls_header, iget( "Категория %s", $c_num );
        }
    }

    if ($request->{options}->{AddVendor}) {
        push @$xls_header, iget( "Производитель" );
        push @$xls_header, iget( "Производитель по Маркету" );
    }

    push @$xls_header, (
        iget( "Заголовок" ),
        iget( "Ссылка" ),
        iget( "id условия показа" ),
        iget( "Условие показа" ),
        iget( "Показы" ),
        iget( "Клики" ),
        iget( "CTR (%)" ),
        iget( "Расход (%s)", $currency_name ),
        iget( "Ср. цена клика (%s)", $currency_name ),
        iget( "Глубина (стр.)" ),
        iget( "Конверсии" ),
        iget( "Конверсия (%)" ),
        iget( "Цена цели (%s)", $currency_name ),
    );

    return [map { $_ && {data=>$_, format=>{bold=>1}} } @$xls_header];
}


sub _get_xls_row {
    my ($record, $request, $pid2categories) = @_;

    my $xls_row = [];
    push @$xls_row, (
        Stat::Tools::format_date( $record->{DateInterval}, $request->{options}->{GroupBy} ),
        $record->{campaign_name},
        { data => $record->{cid}, as_text => 1 },
    );

    if ($request->{options}->{AddOrderID}) {
        push @$xls_row, { data => $record->{OrderID}, as_text => 1 };
    }

    push @$xls_row, (
        $record->{group_name},
        { data => $record->{pid}, as_text => 1 },
        iget( 'M-%d', $record->{bid} // 0 ),
        { data => $record->{OfferID}, as_text => 1 },
    );

    if ($request->{options}->{AddCategories}) {
        my @category_names = _get_category_names($pid2categories, $record->{pid}, $record->{CategoryID});
        for my $c_num (1..$MAX_CATEGORIES) {
            push @$xls_row, iget($category_names[$c_num - 1] // '-');
        }
    }

    if ($request->{options}->{AddVendor}) {
        push @$xls_row, iget($record->{Vendor} // '-');
        push @$xls_row, iget($record->{MarketVendor} // '-');
    }

    push @$xls_row, (
        { data => $record->{Title}, as_text => 1 },
        # NB! Url имеено с таким написанием, так от БК приходит
        { data => $record->{Url}, as_text => 1 },
        $record->{PhraseID},
        $record->{filter_name},
        $record->{Shows},
        $record->{Clicks},
        _format_float( ($record->{CTR} || 0) * 100 ),
        _format_float( $record->{CostCur} ),
        _format_float( $record->{AvgClickCost} ),
        _format_float( $record->{AvgDepth} ),
        $record->{GoalsNum},
        _format_float( ($record->{Conversion} || 0) * 100 ),
        _format_float( $record->{AvgGoalCost} ),
    );

    return $xls_row;
}


sub _get_category_names {
    my ($pid2categories, $pid, $category_id) = @_;
    my @category_names;
    my %found_ids;

    return @category_names unless $pid;

    if (! exists $pid2categories->{$pid}) {
        $pid2categories->{$pid} = get_hashes_hash_sql(PPC(pid => $pid), [
            "SELECT pf.category_id, pf.name, pf.parent_category_id FROM adgroups_performance a JOIN perf_feed_categories pf USING(feed_id)",
            WHERE => {"a.pid" => $pid},
        ]);
    }
    my $categories = $pid2categories->{$pid};

    if (defined $categories->{$category_id}->{category_names}) {
        return @{$categories->{$category_id}->{category_names}}
    }

    my $current_category_id = $category_id;
    while (defined $current_category_id &&
           exists $categories->{$current_category_id} &&
           exists $categories->{$current_category_id}->{category_id}) {
        $found_ids{$current_category_id} = 1;
        push @category_names, $categories->{$current_category_id}->{name};

        $current_category_id = $categories->{$current_category_id}->{parent_category_id} // 0;
        # Если находим циклическую ссылку - останавливаемся
        last if $found_ids{$current_category_id};
    }

    @category_names = reverse @category_names;
    @category_names = @category_names[0..$MAX_CATEGORIES-1];
    $categories->{$category_id}->{category_names} = \@category_names;
    return @category_names;
}


sub _format_float {
    my $num = shift;
    return "-"  if !$num;
    return 0 + sprintf("%.2f", $num);
}

__PACKAGE__->meta->make_immutable();
1;

