package Reports::Offline::Postview;

=head2 DESCRIPTION

Формализованная обёртка для Postview-отчётов.

=cut

use Direct::Modern;
use Mouse;

use Yandex::I18n;

use PrimitivesIds qw/get_clientid/;

use Reports::MIMETypes;
use JavaIntapi::CreatePostviewOfflineReport;
use JavaIntapi::ValidatePostviewOfflineReport;
use JavaIntapi::GetPostviewOfflineReportList;
use JavaIntapi::GetPostviewOfflineReport;
use JavaIntapi::DeletePostviewOfflineReport;
use JSON qw/to_json/;

our $POSTVIEW_REPORT_TYPE = 'postview';

my %STATUS_MAP = (
    NEW => 'new',
    GRABBED => 'processing',
    FINISHED => 'ready',
    FAILED => 'ready',
    REVOKED => 'ready',
);
my %FAIL_STATUSES = (
    FAILED => 1,
    REVOKED => 1,
);

our $PROCESSING_REPORTS_LIMIT = 3;
our %PROCESSING_STATUSES = (
    new => 1,
    processing => 1,
);

=head2 validate_params

Проверка параметров перед созданием отчёта

=cut

sub validate_params {
    my ($self, $vars) = @_;

    my @errors;

    my $result = JavaIntapi::ValidatePostviewOfflineReport->new(
        operator_uid => $vars->{UID},
        cids => [ split /\D+/, $vars->{cids} ],
        date_from => $vars->{date_from},
        date_to => $vars->{date_to},
    )->call();

    if (!$result->{success}) {
        if ($result->{message} eq 'report_exists') {
            push @errors, iget('Отчет уже есть в списке');
        } elsif ($result->{message} eq 'invalid_date_from') {
            push @errors, iget('Недопустимая дата начала отчета');
        } elsif ($result->{message} eq 'invalid_date_to') {
            push @errors, iget('Недопустимая дата окончания отчета');
        } elsif ($result->{message} eq 'processing_reports_limit_reached') {
            push @errors, iget('Достигнуто ограничение на количество формируемых отчетов');
        } else {
            push @errors, iget('Не удалось заказать отчет');
        }
    }

    return @errors;
}


=head2 place_order

Помещение отчёта в очередь на расчёт

=cut

sub place_order {
    my ($self, $vars) = @_;

    my $result = JavaIntapi::CreatePostviewOfflineReport->new(
        operator_uid => $vars->{UID},
        cids => [ split /\D+/, $vars->{cids} ],
        date_from => $vars->{date_from},
        date_to => $vars->{date_to},
    )->call();

    if (!$result->{success}) {
        die "failed to create postview report via java-intapi: ".to_json($result);
    }

    return;
}


=head2 requeue_order($uid, $id)

Перезапрос отчёта

=cut

sub requeue_order {
    my ($self, $uid, $id) = @_;
    croak "requeue is not supported";
}


=head2 get_list($uid)

Возвращает список отчётов пользователя

=cut

sub get_list {
    my ($self, $uid) = @_;

    my $reports = JavaIntapi::GetPostviewOfflineReportList->new(
        client_id => get_clientid(uid => $uid)
    )->call();
    if (!$reports) {
        warn "failed to get postview reports via java-intapi";
        return [];
    }

    my $results = [];
    foreach my $report (@$reports) {
        push $results, {
            id => $report->{report_id},
            cids => join(',', @{$report->{cids}}),
            create_time => $report->{create_time},
            date_from => $report->{date_from},
            date_to => $report->{date_to},
            ready_time => $report->{finish_time} // '0000-00-00 00:00:00',
            is_fail => $FAIL_STATUSES{$report->{status}} ? 1 : 0,
            status => $STATUS_MAP{$report->{status}},
            status_no_data => $report->{status} eq 'FINISHED' && !$report->{report_url} ? "Yes" : "No",
            group => "",
            rank => 0,
        };
    }

    return $results;
}


=head2 get_data($uid, $id)

Возвращает файл с отчётом

=cut

sub get_data {
    my ($self, $uid, $id) = @_;

    my $report = $self->_get_report($uid, $id);
    if ($report && $report->{report_url}) {
        my $report_file_name = $report->{report_url};
        my $extention = ($report_file_name =~ /\.(\w+)$/) ? $1 : 'xlsx';
        my $storage = Direct::Storage->new();
        my $file = $storage->get_file('offline_stat_reports', filename => $report_file_name, ClientID => get_clientid(uid => $uid));
        if ($file) {
            return($Reports::MIMETypes::type2mime{$extention} => $file->content);
        }
    }

    return;
}


sub _get_report {
    my ($self, $uid, $id) = @_;

    my $result = JavaIntapi::GetPostviewOfflineReport->new(
        client_id => get_clientid(uid => $uid),
        report_id => $id,
    )->call();

    if (!$result) {
        warn "failed to get postview report via java-intapi";
        return;
    }

    return $result;
}

=head2 get_report_options($uid, $id)

Возвращает параметры запроса отчёта пользователя

=cut

sub get_report_options {
    my ($self, $uid, $id) = @_;

    my $report = $self->_get_report($uid, $id);

    return $report ? { cids => $report->{cids} } : {};
}

=head2 delete_data($uid, $id)

Удаляет отчёт

=cut

sub delete_data {
    my ($self, $uid, $id) = @_;

    my $result = JavaIntapi::DeletePostviewOfflineReport->new(
        client_id => get_clientid(uid => $uid),
        report_id => $id,
    )->call();

    return;
}

__PACKAGE__->meta->make_immutable;
1;

