package Reports::PerformanceBanners;

=head2 DESCRIPTION

Формирование отчёта по динамическим медийным объявлениям

=cut

use Direct::Modern;

use Digest::MD5 qw/ md5 /;
use Encode qw/encode_utf8/;
use List::MoreUtils qw/zip/;
use Mouse::Util::TypeConstraints;
use Mouse;

use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::DBTools;
use Yandex::HashUtils qw/ hash_merge /;
use Yandex::TimeCommon qw/ str_round /;

use Currency::Rate qw/convert_currency/;
use Settings;

# можно не группировать по OrderID, так как ParentExportID (=bid) обеспечивает уникальность заказов
# но он нужен для того, чтобы не хранить данные по кампании на каждый bid
# без группировки по ContextType, так как кроме CT=8 статистики тут быть не должно
our @HASH_FIELDS = qw/ OrderID ParentExportID PhraseID OfferID Title Url CategoryID Vendor MarketVendor /;
our %HASH_FIELD_POS = map {($HASH_FIELDS[$_] => $_)} (0 .. $#HASH_FIELDS);
our @SUM_FIELDS = qw/ :count Shows Clicks Cost CostCur GoalsNum SessionNum SessionDepth /;
our %SUM_FIELD_POS = map {($SUM_FIELDS[$_] => $_)} (0 .. $#SUM_FIELDS);


has group_by => (is => 'ro', required => 1, isa => enum [qw/ day week month year /]);

has record_count => (is => 'rw', isa => 'Int', default => 0);
has record_info => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has record_sum => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has record_total_sum => (is => 'ro', isa => 'ArrayRef', default => sub {+[]});
has _record_dates_hash => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has _date_round_cache => (is => 'ro', isa => 'HashRef', default => sub {+{}});


has base_currency => (is => 'ro', isa => 'Str');
has data_by_ParentExportID => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has filter_name_by_PhraseID => (is => 'ro', isa => 'HashRef', default => sub {+{}});
has data_by_OrderID => (is => 'ro', isa => 'HashRef', default => sub {+{}});


=head2 add_record($record)

Добавляем к отчёту одну запись, полученную от БК

=cut

sub add_record {
    my ($self, $record) = @_;

    # DIRECT-49712: пропускать строки с нулевыми показами/кликами в ДМО
    if ($record->{Shows} == 0 && $record->{Clicks} == 0) {
        # в данных БК могут быть строки без показов и кликов (как результат откатов)
        # пропускаем и в ответ не добавляем
        return;
    }

    # отдельно запоминаем значения полей
    my @hash_values = map {_filter_field($record->{$_})} @HASH_FIELDS;
    my $hash = md5(map {encode_utf8 $_ . "\t"} @hash_values);
    $self->record_info->{$hash} ||= \@hash_values;

    # суммируем счётчики
    $self->record_count($self->record_count + 1);
    my $date = str_round($record->{UpdateTime}, 'day');
    $self->_record_dates_hash->{$date} ++;
    my $base_date = $self->get_base_date($record->{UpdateTime});
    my $count = $self->record_sum->{$base_date}->{$hash} ||= [];
    for my $pos (0 .. $#SUM_FIELDS) {
        my $field = $SUM_FIELDS[$pos];
        if ($field eq ':count') {
            $count->[$pos] ++;
            $self->record_total_sum->[$pos] ++;
        } elsif ($field eq 'CostCur') {
            my $currency = $self->_data_by_orderid($record->{OrderID})->{currency} || 'YND_FIXED';
            my $value = $currency eq 'YND_FIXED' ? $record->{Cost} : $record->{CostCur};
            # если задана целевая валюта - пересчитываем
            if ($self->base_currency && $currency ne $self->base_currency) {
                $value = $self->base_currency eq 'YND_FIXED'
                    ? $record->{Cost}
                    : convert_currency($value, $currency, $self->base_currency, date => $date, with_nds => 1);
            }
            $count->[$pos] += $value;
            $self->record_total_sum->[$pos] += $value;
        }
        else {
            $count->[$pos] += $record->{$field};
            $self->record_total_sum->[$pos] += $record->{$field};
        }
    }

    return;
}

# фильтруем поля от проблемных символов
sub _filter_field {
    my ($s) = @_;
    $s =~ s/[\p{Unassigned}\p{Noncharacter_Code_Point}]//gxms;
    return $s;
}

=head3 _data_by_orderid($OrderID)

    Получить по $OrderID хеш со следующими данными о кампании:
        cid
        campaign_name
        currency

=cut

sub _data_by_orderid {
    my ($self, $OrderID) = @_;

    my $data = $self->data_by_OrderID;

    my $record = $data->{$OrderID} //= get_one_line_sql(PPC(OrderID => $OrderID), [
            'SELECT c.cid, c.name AS campaign_name, IFNULL(c.currency, "YND_FIXED") AS currency',
            'FROM campaigns c',
            WHERE => { 'c.OrderID' => SHARD_IDS },
        ]) || {};

    return $record;
}

=head3 _filter_name_by_phraseid($PhraseID, $OrderID)

    Получить по $PhraseID название фильтра (bids_performance.name)
    $OrderID нужен для определени шарда/экономии на запросах в метабазу

=cut

sub _filter_name_by_phraseid {
    my ($self, $PhraseID, $OrderID) = @_;

    if (!exists $self->filter_name_by_PhraseID->{$PhraseID}) {
        $self->filter_name_by_PhraseID->{$PhraseID} = get_one_field_sql(PPC(OrderID => $OrderID), [
                'SELECT name FROM bids_performance',
                WHERE => { perf_filter_id => $PhraseID },
            ]);
    }

    return $self->filter_name_by_PhraseID->{$PhraseID};
}

=head3 _data_by_parent_export_id($ParentExportID, $OrderID)

    Получить по $ParentExportID хеш со следующими полями:
        bid
        pid
        group_name
    $OrderID нужен для определени шарда/экономии на запросах в метабазу

=cut

sub _data_by_parent_export_id {
    my ($self, $ParentExportID, $OrderID) = @_;

    my $data = $self->data_by_ParentExportID;

    my $record = $data->{$ParentExportID} ||= get_one_line_sql(PPC(OrderID => $OrderID), [
            'SELECT b.bid, b.pid, p.group_name',
            'FROM banners b',
            'JOIN phrases p ON p.pid = b.pid',
            WHERE => { 'b.bid' => $ParentExportID },
        ]) || {};

    return $record;
}

=head2 get_base_date($date)

    my $rep = Reports::PerformanceBanners->new(group_by => 'week')
    my $week_start = $rep->get_base_date('2015-07-24');
    # now $week_start == '2015-07-20'

Возвращает базовую дату диапазона группировки

=cut

sub get_base_date {
    my ($self, $date) = @_;

    my $cache = $self->_date_round_cache;

    my $rounded_date = $cache->{$date} //= str_round($date, $self->group_by);

    return $rounded_date;
}

=head2 get_iterator()

    my $it = $rep->get_iterator();
    while (my $record = $it->()) {
        # do something with $record
        ...
    }

Возвращает функцию-итератор, которая отдаёт поочерёдно все проагрегированные записи

=cut

sub get_iterator {
    my ($self) = @_;

    my $current_count = $self->record_count;
    my $dates = $self->get_group_dates();

    my $ri = $self->record_info;
    my @hashes =
        map {$_->[0]}
        sort { $a->[1] cmp $b->[1] }
        map { my $h = $_; [ $h => join "\t", map {$ri->{$h}->[$_]} (0 .. $#HASH_FIELDS) ] }
        keys %$ri;
    my %hash_skey = map {($hashes[$_] => $_)} (0 .. $#hashes);

    my $date;
    my $sums_by_date;
    my @date_hashes;
    my $hash_pos;

    return sub {
        croak "Data was updated since iterator created"  if $self->record_count != $current_count;

        if (!defined $hash_pos || $hash_pos > $#date_hashes) {
            $date = shift @$dates;
            return if !defined $date;

            $hash_pos = 0;
            $sums_by_date = $self->record_sum->{$date};
            @date_hashes = sort {$hash_skey{$a} <=> $hash_skey{$b}} keys %$sums_by_date;
        }

        my $hash = $date_hashes[$hash_pos];
        my $sums = $sums_by_date->{$hash};
        $hash_pos ++;

        my %record = (
            DateInterval => $date,
            zip(@HASH_FIELDS, @{$ri->{$hash}}),
            zip(@SUM_FIELDS, @$sums),
        );

        $self->_add_calc_fields(\%record);
        hash_merge \%record, $self->_data_by_orderid($record{OrderID});
        hash_merge \%record, $self->_data_by_parent_export_id($record{ParentExportID}, $record{OrderID});
        $record{filter_name} = $self->_filter_name_by_phraseid($record{PhraseID}, $record{OrderID});

        return \%record;
    }
}


=head2 get_totals()

    my $totals = $rep->get_totals();

Возвращает итоговые суммы

=cut

sub get_totals {
    my ($self) = @_;

    my %record = zip(@SUM_FIELDS, @{$self->record_total_sum});
    $self->_add_calc_fields(\%record);

    return \%record;
}

=head2 get_group_dates

    my $dates = $rep->get_group_dates();

Возвращает список сгруппированных дат, по которым есть данные

=cut

sub get_group_dates {
    my ($self) = @_;
    return [ sort keys %{ $self->record_sum } ];
}

=head2 get_record_dates

    my $dates = $rep->get_record_dates();

Возвращает список всех дат, по которым есть данные

=cut

sub get_record_dates {
    my ($self) = @_;

    return [sort keys %{$self->_record_dates_hash}];
}

sub _add_calc_fields {
    my ($self, $record) = @_;
    return $record if !$record->{':count'};

    $record->{Cost} /= 1_000_000;
    $record->{CostCur} /= 1_000_000;

    $record->{CTR}          = $record->{Shows}      ? ( $record->{Clicks} / $record->{Shows} )            : 0;
    $record->{AvgClickCost} = $record->{Clicks}     ? ( $record->{CostCur} / $record->{Clicks} )          : 0;
    $record->{AvgDepth}     = $record->{SessionNum} ? ( $record->{SessionDepth} / $record->{SessionNum} ) : 0;
    $record->{AvgGoalCost}  = $record->{GoalsNum}   ? ( $record->{CostCur} / $record->{GoalsNum} )        : 0;
    $record->{Conversion}   = $record->{SessionNum} ? ( $record->{GoalsNum} / $record->{SessionNum} )     : 0;

    return $record;
}

__PACKAGE__->meta->make_immutable;
1;

