package Reports::Queue;

=head1 NAME

    Reports::Queue -- модуль для работы с очередями оффлайновых отчетов

=head1 DESCRIPTION

    В модуле реализованы основные необходимые операции для работы с очередями разнообразных отчетов, и реализации жизненного цикла каждого отчета.
    В БД очереди хранятся в таблице stat_reports.
    На данный момент поддерживаются такие типы отчетов: stat_mp, client_potential.
    Модуль не содержит кода по реализации формирования каждого конкретного типа отчета.

=head1 CAVEAT

    Разделение по трёдам предполагается исключающее. Т.е. предполагается, что два разных трёда не могут получить один и тот же отчёт на обработку.

=cut

use Direct::Modern;

use JSON;
use Encode;

use Settings;
use Primitives qw/get_uid_by_login/;
use PrimitivesIds qw/get_clientid/;
use Tools ();
use Stat::Const;

use Yandex::DBTools;
use Yandex::DBShards;
use Direct::Storage;
use Yandex::Overshard;
use Yandex::Validate qw/is_valid_id/;
use Yandex::ListUtils qw/xminus/;
use Yandex::HashUtils qw/hash_cut hash_kmap hash_merge/;
use Yandex::DateTime qw/now date/;
use Yandex::Compress qw(inflate deflate);

use List::MoreUtils qw(any);

our %REPORT_TYPES = (stat_mp => {
                         max_attempts => 2,         # максимальное кол-во попыток построить отчет
                         max_report_lifetime => 90, # через сколько дней после создания удалять
                         process_iteration_limit => 10,  # сколько отчетов выбирать для обработки за одну итерацию
                     },
                     client_potential => {
                         max_attempts => 2,
                         max_report_lifetime => 365,
                         process_iteration_limit => 10,
                         threads => {easy =>  sub { return $_[0]->{bids_qty} <= 6_000 ? 1 : 0 },
                                     heavy => sub { return $_[0]->{bids_qty} > 6_000 ? 1 : 0 }, }
                     });

=head2 new

    Конструктор класса
    Возможные параметры:
      type - тип очереди (stat_mp|client_potential), обязательный
      thread - номер/название потока обработки очереди (если таковой поддерживается конкретным типом очереди)

=cut

sub new {
    my $class = shift;
    my %params = @_;
    die 'should be defined correct report type' unless $params{type} && exists $REPORT_TYPES{$params{type}};

    if (exists $params{thread}) {
        my $thread = _check_thread($params{type}, $params{thread});
        delete $params{thread} unless defined $thread;
    }

    report_log($params{log}) if $params{log};
    
    my $self = \%params;
    $self->{storage} = Direct::Storage->new();
    return bless $self, $class;
}

{
    my $report_log;
    sub report_log(;$) {
        my $report_log_name = shift || "static_report.log";
        return $report_log ||= (ref $report_log_name ? $report_log_name : Yandex::Log->new(log_file_name => $report_log_name));
    }
}

=head2 get_queue_length

    Получить длину очереди заказов для заданного uid, operator_uid или shard
        get_queue_length(<selector_type> => <selector>)
    Возможные селекторы:
        shard
        uid
        operator_uid
    Примеры вызова:
        get_queue_length(uid => 123)
        get_queue_length()
    
=cut

sub get_queue_length {
    my ($self, $selector_type, $selector) = @_;
    if ($selector_type) {
        die "Not correct selector type: $selector_type" if $selector_type !~ /^(uid|shard|operator_uid)$/;
        die "Not correct selector: $selector" if !is_valid_id($selector);
    }
    my ($shard_cond, $where_cond);
    if (!$selector_type) {
        $shard_cond = {shard => 'all'};
    } elsif ($selector_type eq 'shard') {
        $shard_cond = {shard => $selector};
    } elsif ($selector_type eq 'uid') {
        $where_cond = $shard_cond = {uid => $selector};
    } elsif ($selector_type eq 'operator_uid') {
        $shard_cond = {uid => $selector};
        $where_cond = {operator_uid => $selector};
    }

    my $reports = $self->_filter_by_thread(
                                    get_all_sql(PPC(%$shard_cond), [
                                           "SELECT id, uid, operator_uid, report_stats
                                              FROM stat_reports",
                                             where => { %{$where_cond // {}},
                                                        report_type => $self->{type},
                                                        status__not_in => [qw/ready failed/] }
                                              ]),
                                    allow_empty_thread => 1
                                );

    return scalar(@$reports);
}

=head2 get_oldest_report

    Получить самый старый необработанный заказ из очереди заказов для заданного шарда
    Примеры вызова:
        get_oldest_report(shard => 1)
        get_oldest_report()

=cut

sub get_oldest_report {
    my $self = shift;
    my %O = @_;

    my $reports = $self->_filter_by_thread(
                         overshard order => 'create_time_ts:num',
                                   get_all_sql(PPC(shard => $O{shard} || 'all'), ["
                                                SELECT id, uid, operator_uid, report_stats,
                                                       UNIX_TIMESTAMP(create_time) as create_time_ts
                                                  FROM stat_reports",
                                                 where => { report_type => $self->{type},
                                                            status__not_in => [qw/ready failed/] }, "
                                              ORDER BY create_time_ts"])
                    );

    return @$reports ? $reports->[0] : undef;
}

=head2 get_report_list_to_process

    Возвращает список отчетов, требующих обработки, в формате ({id => 11, uid => 22}, ...)
    Входные параметры:
        shard => 1 - обязательный

        limit => 10
        login => 'login1' - обрабатывать отчеты только этого клиента (по uid)

=cut

sub get_report_list_to_process {
    my $self = shift;
    my %O = @_;

    my %sql_filter = ();
    if ($O{login}) {
        my $uid = get_uid_by_login($O{login}) or report_log()->die("user not found: login '$O{login}'");
        $sql_filter{uid} = $uid;
    }
    my $reports = $self->_filter_by_thread(
                             get_all_sql( PPC(shard => $O{shard}), ["
                                select id, uid, report_stats
                                  from stat_reports",
                                 where => { status => [qw(new processing)],
                                            report_type => $self->{type},
                                            %sql_filter }, "
                              order by id"]) 
                        );
    my $limit =  $REPORT_TYPES{$self->{type}}->{process_iteration_limit} || $O{limit};
    @$reports = splice(@$reports, 0, $limit) if $limit;
    return @$reports;
}

=head2 start_processing_report

    Помечает отчет статусом о начале процесса обработки отчета

=cut

sub start_processing_report {
    my ($self, $report, $shard) = @_;

    my $status = $report->{status};
    return 0 unless any {$status eq $_} qw(new processing);

    my $id = $report->{id};
    my $attemps_cnt = $report->{process_attempts};
    my %report_cond = (id => $id, status => $status, process_attempts => $attemps_cnt);
    if ($attemps_cnt >= $REPORT_TYPES{$self->{type}}->{max_attempts}) {
        do_update_table(PPC(shard => $shard), 'stat_reports', { status => 'failed' }, where => \%report_cond);
        return 0; # всегда ноль, т.к. упавший отчёт нельзя строить
    } else {
        my $new_attemps_cnt = $attemps_cnt + 1;
        return do_update_table(PPC(shard => $shard), 'stat_reports', { status => 'processing', process_attempts => $new_attemps_cnt }, where => \%report_cond);
    }
}

=head2 create_report

    Создать заявку на отчёт и сохранить её в БД Директа в случае успешного создания
    вход - параметры отчета: { uid
                               operator_uid
                               cids - кампании чрез разделитель
                               date_from - начало периода
                               date_to - окончание периода
                               ...
                               другие опции, такие как:
                               show_phrases - по фразам
                               group - группировка по дате none|day|week|month|year
                               target_type - тип площадки undef|search|theme }
=cut

sub create_report {
    my $self = shift;
    my $opt = $_[0] // {};

    my @main_opts = qw/uid operator_uid cids date_from date_to report_stats/;
    my $extra_opt = hash_cut $opt, xminus [keys %$opt], \@main_opts;

    $opt->{date_from} = date($BEGIN_OF_TIME_FOR_STAT)->ymd() unless $opt->{date_from};
    $opt->{date_to} = now()->ymd() unless $opt->{date_to};

    $opt->{report_stats} = to_json($opt->{report_stats} // {});


    do_insert_into_table(PPC(uid => $opt->{uid}), 'stat_reports', {
        id => get_new_id('report_id'),
        report_type => $self->{type},
        %{hash_cut $opt, @main_opts},
        extra_opt_compressed => Tools::encode_json_and_compress($extra_opt),
    });
}

=head2 get_report
    
    Проверяет наличие готового отчёта на стороне Директа.
        uid - uid клиента (главного представителя) / operator_uid (для отчетов не привязанных к клиенту)
        id  - номер отчета

=cut

sub get_report {
    my ($self, $uid, $id, $part_id) = @_;
    
    $part_id ||= 1;

    my $report = $self->get_report_info($uid, $id);
    
    return undef unless $report && $report->{report_data_parts_qty};
    
    my $report_data;
    eval{
        my $file = $self->{storage}->get_file("offline_stat_reports",
            filename => $self->get_report_filename($report, $part_id),
            ClientID => get_clientid(uid => $uid),
        );
        return undef unless $file;
        $report_data = inflate $file->content;
        $report_data = Encode::decode_utf8 $report_data if $report->{report_data_format} eq 'csv';
    };

    if ($@) {
        warn($@);
        report_log()->out("Failed to load campaign $self->{type} report data: $@");
        return undef;
    }
    return wantarray ? ($report_data, $report->{report_data_format}) : $report_data;
}

=head2 repair_report

    отправить отчет на пересоздание

=cut

sub repair_report {
    my ($self, $uid, $id) = @_;

    my $updated = do_update_table(PPC(uid => $uid), 'stat_reports', {status => 'new', process_attempts => 0},
                                                           where => {id => $id, status => 'failed', uid => $uid, report_type => $self->{type}});
    $updated += 0;
    return $updated;
}

=head2 delete_report

    удалить отчет

=cut

sub delete_report {
    my ($self, $uid, $report) = @_;

    $report = $self->get_report_info($uid, $report) unless ref $report;

    my $deleted = do_sql( PPC(uid => $uid), 
                      "DELETE FROM stat_reports
                             WHERE uid=? AND id=? AND report_type = ?",
                                   $uid, $report->{id}, $self->{type});
    $deleted += 0;
    if ($deleted) {
        foreach (1 .. $report->{report_data_parts_qty}) {
            eval {
                $self->{storage}->delete_file("offline_stat_reports",
                    filename => $self->get_report_filename($report, $_),
                    ClientID => get_clientid(uid => $uid),
                );
            };
            report_log()->out($@) if $@;
        }
    }
    return $deleted;
}
        
=head2 save_report_data

    Сохранить отчёт

=cut

sub save_report_data {
    my ($self, $opt) = @_; 

    die "report id is not defined" unless $opt->{id};
    die "uid required" unless $opt->{uid};

    unless ($opt->{report_data_parts} && @{$opt->{report_data_parts}} ) {
        do_update_table(PPC(uid => $opt->{uid}), 'stat_reports', {report_data_parts_qty => 0, status=> 'ready', ready_time__dont_quote => 'NOW()'},
                                                 where => hash_cut($opt, qw/id/));
        return;
    }  
    my $part_id = 0;
    foreach my $data_part (@{$opt->{report_data_parts}}) {
        $part_id ++;
        my $zipped_data = deflate map { $opt->{report_data_format} eq 'csv' && (utf8::is_utf8($_) || !utf8::valid($_)) 
                                            ? Encode::encode_utf8 $_ 
                                            : $_ } $data_part;
        eval {
            $self->{storage}->save("offline_stat_reports", $zipped_data, filename => $self->get_report_filename($opt, $part_id), ClientID => get_clientid(uid => $opt->{uid}));
        };
        if ($@) {
            warn($@);
            report_log()->out("Failed to save $self->{type} report data: $@");

            # удаляем уже сохраненные файлы, если они есть
            foreach my $i (1 .. scalar(@{$opt->{report_data_parts}})) {
                $self->{storage}->delete_file("offline_stat_reports", filename => $self->get_report_filename($opt, $i), ClientID => get_clientid(uid => $opt->{uid}));
            }

            return undef;
        }
    }
    do_update_table(PPC(uid => $opt->{uid}), 'stat_reports', {  report_data_parts_qty => scalar(@{$opt->{report_data_parts}}),
                                                                report_data_format => $opt->{report_data_format},
                                                                status => 'ready',
                                                                ready_time__dont_quote => 'NOW()'},
                                                             where => hash_cut($opt, qw/id/));
}

=head2 clear_old_reports

    удаляет отчёты старше максимального периода жизни отчета

=cut

sub clear_old_reports {
    my $self = shift;
    my %O = @_;

    return unless defined $REPORT_TYPES{$self->{type}}->{max_report_lifetime};
    my $to_delete = get_all_sql(PPC(shard => $O{shard}), "select id, uid 
                                                            from stat_reports 
                                                           where create_time < NOW() - INTERVAL ? DAY AND report_type = ?",
                                                         $REPORT_TYPES{$self->{type}}->{max_report_lifetime}, $self->{type});
    foreach my $report (@$to_delete) {
        $self->delete_report($report->{uid}, $report->{id});
    }
}



=head2 get_report_filename

    по данным отчета и ID п/п файла , возвращает имя файла в GridFS

=cut

sub get_report_filename {
    my ($self, $report, $part_id) = @_;
    return sprintf('rep%d_part%d.%s', $report->{id}, int($part_id || 1), $report->{report_data_format});
}



=head2 get_report_info

    по ID отчета и uid клиента (или operator_uid для отчетов не привязанных к клиентам), возвращает информацию о нем (без собственно данных отчета)

=cut

sub get_report_info {
    my ($self, $uid, $id) = @_;
    my $report = get_one_line_sql(PPC(uid => $uid), "
        SELECT id,
               date_from, date_to,
               cids, uid, operator_uid, status,
               extra_opt_compressed,
               create_time, ready_time,
               report_data_format, report_data_parts_qty,
               process_attempts
          FROM stat_reports
         WHERE id=? AND report_type = ? AND uid = ?
            ", $id, $self->{type}, $uid);

    _parse_extra_opt($report);    
    return $report;
}

=head2 _parse_extra_opt

    на вход получает параметры отчета: {...
                                         extra_opt_compressed => <сжатый JSON с доп. параметрами отчета>
                                        ...}
    преобразовывает extra_opt в структуру перла (хеш)
    и все ключи из extra_opt, которые не пересекаются с основными параметрами отчета - дублируются рядом 
    с основными параметрами (для простоты работы с ними)

=cut
sub _parse_extra_opt {
    my $opt = shift;
    if ($opt && ref $opt eq 'HASH') {
        if ($opt->{extra_opt_compressed}) {
            $opt->{extra_opt} = Tools::decode_json_and_uncompress(delete $opt->{extra_opt_compressed});
        }

        if ($opt->{extra_opt}) {
            while (my ($k, $v) = each %{$opt->{extra_opt}}) {
                $opt->{$k} = $v unless exists $opt->{$k};
            }
        }
    }
}

=head2 _filter_by_thread

    отфильтровать отчеты по потоку обработки к которому они относятся

    Предполагается, что этот фильтр не может отдать один и тот же отчёт двум разным трёдам.

=cut

sub _filter_by_thread {
    my ($self, $items, %O) = @_;

    return $items if !$self->{thread} && $O{allow_empty_thread};

    _check_thread($self->{type}, $self->{thread});
    return $items unless $self->{thread};

    my $filtered_items = [grep {
                              my $stats = hash_merge {}, $_, ($_->{report_stats} ? from_json($_->{report_stats}) : {});
                              $REPORT_TYPES{$self->{type}}->{threads}->{$self->{thread}}->($stats);
                          } @$items];

    return $filtered_items;
}

=head2 _check_thread 

    проверка указанного потока обработки очереди на корректность

=cut

sub _check_thread
{
    my ($type, $thread) = @_;
    my $threads = $REPORT_TYPES{$type}->{threads};
    if ($threads && keys %$threads) {
        die 'not correct thread to process' unless $thread && exists $threads->{$thread};
    } else {
        return undef;
    }
    return 1;
}

1;
