package Retargeting;

=head1 NAME

Retargeting - работа с условиями ретаргетинга
https://jira.yandex-team.ru/browse/DIRECT-15172

TODO: хорошо бы иметь массовые версии всех функций (save_*, delete_*, ...), чтобы минимизировать число запросов в базу

=cut

# $Id$

use Direct::Modern;

use JSON;
use List::MoreUtils qw/part uniq any all/;
use List::Util qw/min first/;
use Carp qw/croak cluck/;
use Encode qw/decode/;

use Settings;
use ShardingTools;
use Tools;
use Primitives;
use PrimitivesIds;
use MailNotification;
use Currencies;
use LogTools qw/log_metrika_query/;
use SolomonTools;
use TextTools;
use JavaIntapi::GenerateObjectIds;

use Yandex::I18n;
use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HTTP;
use Yandex::ListUtils qw/xuniq chunks xflatten/;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::Trace;
use Yandex::TVM2;
use Yandex::Validate qw/is_valid_int is_valid_id/;
use Yandex::HashUtils;

use CampaignTools;
use Direct::Model::RetargetingCondition;
use Direct::Retargetings;
use Direct::RetargetingConditions;
use Direct::TargetInterests;
use Direct::Validation::RetargetingConditions qw//;

our $CONDITION_XLS_PREFIX = 'audience:';
our $CONDITION_XLS_PREFIX_OLD = 'retargeting:';
our $CONDITION_XLS_PREFIX_TARGET_INTEREST = 'interest:';

# текст ошибки на случай, если метрика ответила неизвестной ошибкой
our $request_access_to_metrika_common_error = iget_noop('Сервис временно недоступен. Попробуйте повторить запрос через некоторое время.');

our $request_access_to_metrika_errors = {
    ERR_INVALID_OBJECT_TYPE => iget_noop('К сожалению, сервис временно недоступен.'),
    ERR_INVALID_EMAIL => iget_noop('Неверно указан e-mail.'),
    ERR_INVALID_LOGIN => iget_noop('Логин владельца счетчика не найден.'),
    ERR_INVALID_REQUESTOR_LOGIN => iget_noop('Неверный логин, обратитесь в службу поддержки.'),
    ERR_COUNTER_NOT_FOUND => iget_noop('Указанный счетчик не найден. Убедитесь, что номер счетчика или домен указан правильно.'),
    ERR_ALREADY_EXISTS => iget_noop('Доступ к запрашиваемому счетчику уже есть.'),
    ERR_TEMPORARY_UNAVAILABLE => $request_access_to_metrika_common_error,
    ERR_COUNTER_IN_CONNECT => iget_noop('Доступ к запрашиваемому счетчику настраивается через Яндекс.Коннект и в настоящее время не может быть изменен.'),
    ERR_CONNECT_RESOURCE_NOT_FOUND => iget_noop('Счетчик отсутствует в Яндекс.Коннект'),
};

# поле name в таблице ppclog.monitor_targets (к нему нужно добавить номер конкретного шарда в конце)
our $BASE_TARGET_NAME_FOR_GOALS_NOT_ACCESSIBLE = 'ppcRetargetingCheckGoals.goals_made_not_accessible_shard_';

# --------------------------------------------------------------------

=head2 get_retargeting_conditions

Для клиента возвращаем все доступные условия ретаргетинга
Может использоваться с опциями:
    with_campaigns  -> для каждого ретаргетинга загружать также список кампаний, в которых он используется

  $vars->{all_retargeting_conditions} = get_retargeting_conditions(uid => $uid);
  $vars->{all_retargeting_conditions} = get_retargeting_conditions(ClientID => $ClientID);

В ответе хеш:
    {     '1' => { 'condition' => [{
                                    'goals' => [{ 'goal_id' => '32',
                                                  'time' => '1'
                                               }],
                                    'type' => 'or'
                                  },
                                  {
                                    'goals' => [{ 'goal_id' => '32',
                                                  'time' => '1'
                                                 },
                                                 { 'goal_id' => '32',
                                                   'time' => '1'
                                               }],
                                    'type' => 'or'
                                  }
                                ],
                   'condition_desc' => 'dewdew',
                   'condition_name' => 'dewdewd',
                   'ret_cond_id' => '1',
                   'is_accessible' => 1
                 },
          '2' => {
                   'condition' => [{'goals' => [{ 'goal_id' => '169283',
                                                  'time' => '1'
                                                 },
                                                 {
                                                   'goal_id' => '32',
                                                   'time' => '1'
                                                 }],
                                    'type' => 'all'
                                  },
                                  {'goals' => [{'goal_id' => '169284',
                                                'time' => '1'
                                              }],
                                    'type' => 'all'
                                  }],
                   'condition_desc' => 'edweww',
                   'condition_name' => 'dewdewd',
                   'ret_cond_id' => '2',
                   'is_accessible' => 0
                 },
    }

    'is_accessible' => 1 - все цели в условии доступны клиенту (0 - есть хотя бы одна цель которая не доступна)

=cut

sub get_retargeting_conditions($$;%) {
    my $type_of_id = shift;
    my $id = shift;
    my %OPTIONS = @_;

    die "get_retargeting_conditions: type is invalid" unless $type_of_id =~ /^(?:uid|ClientID)$/;
    my $ClientID = $type_of_id eq 'uid' ? get_clientid(uid => $id) : $id;
    die "get_retargeting_conditions: ClientID not found" unless $ClientID;

    return mass_get_retargeting_conditions_by_ClientIDS([$ClientID], %OPTIONS)->{$ClientID};
}

# --------------------------------------------------------------------

=head2 mass_get_retargeting_conditions_by_ClientIDS

По списку клиентов получаем все условия ретаргетинга
На входе список ClientID, и опции как в get_retargeting_conditions()
На выходе хеш:
    {
        ClientID1 => {условие как в get_retargeting_conditions()},
        ClientID2 => {...}, ...
    }

=cut

sub mass_get_retargeting_conditions_by_ClientIDS($;%) {
    my $clientids = shift;
    my %OPTIONS = @_;

    my ($get_by_key, $get_by_vals, %get_by_filter);

    if ($clientids && @$clientids) {
        ($get_by_key, $get_by_vals) = (client_id => $clientids);
    } elsif (@{$OPTIONS{ret_cond_id} // []}) {
        ($get_by_key, $get_by_vals) = (ret_cond_id => $OPTIONS{ret_cond_id});
    } else {
        die "mass_get_retargeting_conditions_by_ClientIDS: no where condition specified";
    }

    $get_by_filter{'rc.ret_cond_id'} = $OPTIONS{ret_cond_id} if @{$OPTIONS{ret_cond_id} // []};

    my $ret_conds = Direct::RetargetingConditions->get_by(
        $get_by_key, $get_by_vals, filter => \%get_by_filter,
        (defined $OPTIONS{type} ? (type => $OPTIONS{type}) : ())
    )->items;

    my $retargetings_result = {map {$_ => {}} @$clientids};
    return $retargetings_result if !@$ret_conds;

    for my $ret_cond (@$ret_conds) {
        my $x = ($retargetings_result->{$ret_cond->client_id}->{$ret_cond->id} = $ret_cond->to_db_hash);
        $x->{condition} = [map { $_->to_hash } @{$ret_cond->condition}];
    }

    if ($OPTIONS{with_campaigns}) {
        for my $clientid (keys %$retargetings_result) {
            my $ret_cond_ids = find_camps_used_ret_cond($clientid);
            for my $ret_cond_id(keys %$ret_cond_ids) {
                hash_merge $retargetings_result->{$clientid}->{$ret_cond_id},
                    { campaigns => $ret_cond_ids->{$ret_cond_id}};
            }
        }
    }

    for (map { values %$_ } values %$retargetings_result) {
        delete @{$_}{qw/condition_json is_deleted modtime/};
        delete $_->{is_accessible} if $OPTIONS{short};
    }

    return $retargetings_result;
}

# --------------------------------------------------------------------

=head2 save_retargeting_condition
=head2 prepare_user_ret_cond

Для клиента сохраняем условие ретаргетинга

  my $ret_cond_id = save_retargeting_condition(uid => $uid, $cond);

    1. кнопка "сохранить" при редактировании старого условия - пришел id "условия" - всегда обновляем это условие
    2. кнопка "сохранить" при создании нового или кнопка "сохранить как новое" при редактировании старого - id "условия" не приходит:
        2.1. если совпало или название или "условие", то возвращаем ошибку: "условие id" уже есть
        2.2. если не совпало и название и "условие", то добавляем, возвращаем новый id

=cut

sub prepare_user_ret_cond($) {
    my $user_ret_cond = shift;
    for my $goal (map { @{$_->{goals}} } @{$user_ret_cond->{condition}}) {
        my $goal_type = Primitives::get_goal_type_by_goal_id($goal->{goal_id});
        $goal->{time} = $Direct::Validation::RetargetingConditions::MAX_GOAL_REACH_TIME_DAYS if $goal_type eq 'audience';
    }
    return $user_ret_cond;
}

sub save_retargeting_condition($$$) {
    my $type_of_id = shift;
    my $id = shift;
    my $ret_cond_data = shift;

    die "save_retargeting_condition: type is invalid" unless $type_of_id =~ /^(?:uid|ClientID)$/;
    my $ClientID = $type_of_id eq 'uid' ? get_clientid(uid => $id) : $id;
    die "save_retargeting_condition: ClientID not found" unless $ClientID;

    prepare_user_ret_cond($ret_cond_data);

    my $ret_cond;
    if (my $ret_cond_id = $ret_cond_data->{ret_cond_id}) {
        $ret_cond = Direct::RetargetingConditions->get_by(id => $ret_cond_id)->items->[0];
        if ($ret_cond->client_id != $ClientID) {
            $ret_cond->id(0);
            $ret_cond->client_id($ClientID);
        }
    } else {
        $ret_cond = Direct::Model::RetargetingCondition->new(id => 0, client_id => $ClientID);
    }

    $ret_cond->condition_name($ret_cond_data->{condition_name});
    $ret_cond->condition_desc($ret_cond_data->{condition_desc} // '');
    $ret_cond->retargeting_conditions_type($ret_cond_data->{retargeting_conditions_type} // 'metrika_goals');
    $ret_cond->condition($ret_cond_data->{condition});
    $ret_cond->properties($ret_cond_data->{properties}) if $ret_cond_data->{properties};

    my $logic = Direct::RetargetingConditions->new([$ret_cond]);
    if (!$ret_cond->id) { $logic->create() } else { $logic->update() };

    return $ret_cond->id;
}

# --------------------------------------------------------------------

=head2 delete_retargeting_condition

Для клиента удаляем условие ретаргетинга. Проверка на то, что условие
не используется в каком-либо bids_retargeting, должна выполняться в
вызывающем коде.

  my $ret_cond_id = delete_retargeting_condition(uid => $uid, $ret_cond_id);
  Входные параметры:
    1. uid => $uid или ClientID => $ClientID
    2. $ret_cond_id

=cut
sub delete_retargeting_condition($$$) {
    my $type_of_id = shift;
    my $id = shift;
    my $ret_cond_id = shift;

    die "delete_retargeting_condition: type is invalid" unless $type_of_id =~ /^(?:uid|ClientID)$/;
    my $ClientID = $type_of_id eq 'uid' ? get_clientid(uid => $id) : $id;
    die "delete_retargeting_condition: ClientID not found" unless $ClientID;

    delete_retargeting_condition_by_ClientIDS([$ClientID], $ret_cond_id);
}

# --------------------------------------------------------------------

=head2 delete_retargeting_condition_by_ClientIDS

Для клиента удаляем условие ретаргетинга по ClientID

  my $ret_cond_id = delete_retargeting_condition([ClientID1, ClientID2, ...]);
  Входные параметры:
    1. Список ClientID
    2. $ret_cond_id

=cut

sub delete_retargeting_condition_by_ClientIDS($$) {
    my $clientids = shift;
    my $ret_cond_id = shift;

    croak "delete_retargeting_condition_by_ClientIDS: no `clientids` specified" if !$clientids;

    my $filter = $ret_cond_id ? {'rc.ret_cond_id' => $ret_cond_id} : undef;
    Direct::RetargetingConditions
        ->get_by(client_id => $clientids, filter => $filter, fields => [qw/id client_id/])
        ->delete();
}

# --------------------------------------------------------------------

=head2 guess_condition_by_name

Получает на входе список всех условий на аккаунте и имя условия

    получает:
        all_retargeting_conditions - хэш полученный при помощи Retargeting::get_retargeting_conditions
        condition_name - название условия, по которому пытаемся угадать ret_cond_id

    возвращает
        ret_cond_id условия, если оно есть, иначе undef

=cut
sub guess_condition_by_name
{
    my %params = @_;
    if(not exists $params{all_retargeting_conditions} || not exists $params{condition_name}){
        croak "not enough parameters";
    }

    my $conditions = $params{all_retargeting_conditions};
    for my $ret_cond_id (keys %$conditions){
        return $ret_cond_id if $conditions->{$ret_cond_id}->{condition_name} eq $params{condition_name};
    }

    return undef;
}

# --------------------------------------------------------------------

=head2 are_retargeting_conditions_exists($ClientID, $ret_cond_ids)

Проверка существования условия по ret_cond_id для ClientID

=cut
sub are_retargeting_conditions_exists {
    my ($ClientID, $ret_cond_ids) = @_;
    return {} unless @$ret_cond_ids;
    my $ids = get_one_column_sql(PPC(ClientID => $ClientID), ["select ret_cond_id from retargeting_conditions",
                where => { ret_cond_id => $ret_cond_ids, ClientID => $ClientID }]);
    return {map { $_ => 1 } @$ids};
}


# --------------------------------------------------------------------

=head2 validate_retargeting_condition_dublicate

Валидация условий ретаргетинга, на вход получает новое условие и готовый список условий клиента (используется в АПИ)
 - проверка на дубликаты по имени или самомму условию.
 - проверка на структуру условия, через вызов validate_retargeting_condition_without_db(), если не задан no_db => 1

my $error_code = validate_retargeting_condition_dublicate($condition, $exists_conditions, %OPT);

Опции:
    no_db => 0 -- вызывать validate_retargeting_condition_without_db(), 1 - не вызывать (странное название)

=cut

sub validate_retargeting_condition_dublicate {
    my ($cond, $exists_cond, %O) = @_;

    my $condition_json = to_json($cond->{condition}, {canonical => 1});

    # если сохраняется новое условие, то у клиента не должно быть условий с таким же названием или набором целей
    my $count_of_exists_same_cond = scalar(grep {
        # здесь может не быть json объекта, создаем
        if (!$_->{condition_json}) {$_->{condition_json} = to_json($_->{condition}, {canonical => 1})};

        ($cond->{ret_cond_id} || 0) != $_->{ret_cond_id}
        &&
        (lc($_->{condition_name}) eq lc($cond->{condition_name})
         || $_->{condition_json} eq $condition_json
        )
    } @$exists_cond);

    if ( $count_of_exists_same_cond > 0 ) {
        if ( $O{error_text_variant} && $O{error_text_variant} eq 'api4' ) {
            return iget('Уже есть условие с таким же названием или набором целей');
        } else {
            return iget('Уже есть условие с таким названием или набором целей/сегментов');
        }
    }

    my $count_of_exists_cond = scalar(grep {
        lc($_->{condition_name}) ne lc($cond->{condition_name})
        && $_->{condition_json} ne $condition_json
    } @$exists_cond);

    if (!$O{no_db}) {
        return validate_retargeting_condition_without_db($cond, count_of_exists_cond => $count_of_exists_cond);
    } else {
        return '';
    }
}

# --------------------------------------------------------------------

=head2 validate_retargeting_condition_without_db

Проверяем условие ретаргетинга, без обращения к БД, только ограничения и структуру,
вызывается из validate_retargeting_condition()

  my $error_code = validate_retargeting_condition_without_db($cond, count_of_exists_cond => $count_of_exists_cond);

=cut

sub validate_retargeting_condition_without_db($;%) {
    my $cond = shift;
    my %OPTIONS = @_;

    my ($cond_model, $new_model);

    eval {
        $new_model = Direct::Model::RetargetingCondition->new(
            id => $cond->{ret_cond_id} || 0,
            client_id => $cond->{ClientID},
            condition_name => $cond->{condition_name},
            condition_desc => $cond->{condition_desc},
            condition => $cond->{condition},
        );
    } or do {
        return iget('Неправильное условие подбора аудитории');
    };

    if (exists $cond->{old}) {
        $cond_model = Direct::Model::RetargetingCondition->new(
            id => $cond->{old}{ret_cond_id} || 0,
            client_id => $cond->{old}{ClientID},
            condition_name => $cond->{old}{condition_name},
            condition_desc => $cond->{old}{condition_desc},
            condition => $cond->{old}{condition},
        );
        $cond_model->merge_with($new_model);
    } else {
        $cond_model = $new_model;
    }

    my $vr = Direct::Validation::RetargetingConditions::validate_retargeting_conditions([$cond_model]);
    if (!$vr->is_valid) {
        $vr->process_objects_descriptions(Direct::RetargetingConditions->WEB_FIELD_NAMES);
        return $vr->get_error_descriptions->[0];
    }

    return iget('Количество условий на клиента превышено') if $OPTIONS{count_of_exists_cond} >= $Settings::MAX_RETARGETINGS_ON_CLIENT;

    return '';
}

=head2 get_metrika_ab_segments

=cut

sub get_metrika_ab_segments {
    my ($uids, %params) = @_;
    my $metrika_goals = get_metrika_goals_by_uid2($uids, 'segment_ab', %params);

    my @ab_segments = map {
        { section_id     => $_->{section_id},
            segment_id   => $_->{goal_id},
            section_name => $_->{section_name},
            segment_name => $_->{goal_name},
            counter_ids  => $_->{counter_ids},
            percent      => $_->{percent},
        }
    } grep {$_->{goal_type} eq 'ab_segment'} xuniq {$_->{goal_id}} map {@$_} values %$metrika_goals;
    return \@ab_segments;
}

=head2 get_ab_sections_by_id

=cut

sub get_ab_sections_by_id {
    my ($uids, %params) = @_;
    my $sections = {};
    my $ab_segments = get_metrika_ab_segments($uids, %params);
    for my $segment (@$ab_segments) {
        my $section = $sections->{$segment->{section_id}} ||={};
        $section->{section_id} = int $segment->{section_id};
        $section->{section_name} = $segment->{section_name};
        $section->{counter_ids} = $segment->{counter_ids};
        push @{$section->{segments} ||= []}, {
                segment_id   => $segment->{segment_id},
                segment_name => $segment->{segment_name},
                percent => $segment->{percent},
            };
    }
    return $sections;
}
# --------------------------------------------------------------------

=head2 get_metrika_goals_by_uid

Получаем по списку uid-ов, все доступные цели из метрики

  my $goals = get_metrika_goals_by_uid([$uid1, $uid2, ...]);
  my $goals = get_metrika_goals_by_uid($uid);
  $goals = get_metrika_goals_by_uid($uids, log => $log);

  в ответе:
  {
    uid1 => [
               {
                 'goal_id' => '32',
                 'goal_domain' => 'site.ru',
                 'goal_name' => "Корзина вариант 1",
                 'goal_type' => 'goal'
               },
               {
                 'goal_id' => '169227',
                 'goal_domain' => 'site.ru',
                 'goal_name' => "10 страниц",
                 'goal_type' => 'segment'
               },
            ],
    uid2 => [
               {
                 'goal_id' => '323',
                 'goal_domain' => 'site2.ru',
                 'goal_name' => "Goal name",
                 'goal_type' => 'goal'
               },
            ],
  }

    При ошибках падает (если не передали параметр skip_errors)

    perl -Mmy_inc=.. -MRetargeting -MDDP -e 'my $x = Retargeting::get_metrika_goals_by_uid([6138950]); p $x'

=cut

sub get_metrika_goals_by_uid {
    my $profile = Yandex::Trace::new_profile('retargeting:get_metrika_goals_by_uid');
    my ($uids, %params) = @_;

    $uids = [$uids] if ref($uids) ne 'ARRAY';

    my $method = "retargeting_conditions";
    my $solomon_labels = {external_system => "metrika", sub_system => "audience_internal_api", method => $method};
    my $URL = "${Settings::METRIKA_SUPPORT_AUDIENCE_HOST}/$method";
    my $request = { uid => to_json($uids) };
    my $response_headers;
    my $response_status_stat;

    my $ticket = eval{ Yandex::TVM2::get_ticket($Settings::METRIKA_AUDIENCE_TVM2_ID) }
        or die "Cannot get ticket for $Settings::METRIKA_AUDIENCE_TVM2_ID: $@";

    my $resp = eval {
        Yandex::HTTP::http_fetch('POST', $URL, $request,
            headers                  => {
                "Content-type"        => "application/x-www-form-urlencoded",
                'X-Ya-Service-Ticket' => $ticket
            },
            timeout                  => $params{timeout} // 10,
            log                      => $params{log},
            ipv6_prefer              => 1,
            handle_params            => { keepalive => 1 },
            response_headers_ref     => \$response_headers,
            response_status_stat_ref => \$response_status_stat,
            ($params{retry_fails} ? (num_attempts => 4) : ()),
        );
    };
    my $error;

    if ($@) {
        $error = $@;
        
        my %stat;
        for my $code (keys %$response_status_stat) {
            my $reqs = $response_status_stat->{$code};
            if ($code =~ /^2/) {
                # http_fetch может кинуть ошибку только на пустой ответ
                $stat{unparsable} += $reqs;
            } elsif ($code =~ /^5/) {
                $stat{server_error} += $reqs;
            } elsif ($code =~ /^4/) {
                $stat{client_error} += $reqs;
            } else {
                $stat{unknown} += $reqs;
            }
        }
        SolomonTools::send_requests_stat($solomon_labels, \%stat);
    }

    log_metrika_query({
        url => $URL,
        request => $request,
        ((!$error) ? (response => $resp) : (error => $error)),
    });

    if ($error) {
        if ($params{skip_errors}) {
            ${ $params{skip_errors} } = 1 if ref $params{skip_errors} eq 'SCALAR';
            LogTools::log_messages("skip_metrika_error", "get_metrika_goals_by_uid");
            return {};
        }
        die $error;
    }

    my $goals = eval { decode_json($resp); };
    if ($@) {
        SolomonTools::send_requests_stat($solomon_labels, {unparsable => 1});
        die $@;
    }

    # меняем на свой формат
    for my $uid (keys %$goals) {
        $goals->{$uid} = [] unless ref($goals->{$uid}) eq 'ARRAY';

        for my $goal (@{ $goals->{$uid} }) {
            $goal->{goal_name} = delete $goal->{name};
            $goal->{goal_domain} = delete $goal->{counter_domain};
            $goal->{goal_type} = delete $goal->{type};
            $goal->{goal_id} = delete $goal->{id};
            $goal->{allow_to_use} = 1;
        }
    }

    SolomonTools::send_requests_stat($solomon_labels, {success => 1});

    return $goals;
}

=head2 get_metrika_goals_by_uid2

Получаем по списку uid-ов, все доступные цели из метрики

  my $goals = get_metrika_goals_by_uid([$uid1, $uid2, ...]);
  my $goals = get_metrika_goals_by_uid($uid);
  $goals = get_metrika_goals_by_uid($uids, log => $log);

  goal_type:
    - segment_ab
=cut

sub get_metrika_goals_by_uid2 {
    my $profile = Yandex::Trace::new_profile('retargeting:get_metrika_goals_by_uid');
    my ($uids, $goal_type, %params) = @_;

    $uids = [$uids] if ref($uids) ne 'ARRAY';

    my $method = "direct/retargeting_conditions_by_uids2";
    my $solomon_labels = {external_system => "metrika", sub_system => "audience_internal_api", method => $method};
    my $URL = "${Settings::METRIKA_SUPPORT_AUDIENCE_HOST}/$method";
    my $request = { uid => to_json($uids), type => $goal_type };
    my $response_headers;
    my $response_status_stat;

    my $ticket = eval{ Yandex::TVM2::get_ticket($Settings::METRIKA_AUDIENCE_TVM2_ID) }
        or die "Cannot get ticket for $Settings::METRIKA_AUDIENCE_TVM2_ID: $@";

    my $resp = eval {
        Yandex::HTTP::http_fetch('POST', $URL, $request,
            headers => {
                "Content-type" => "application/x-www-form-urlencoded",
                'X-Ya-Service-Ticket' => $ticket
            },
            timeout => $params{timeout} // 10,
            log => $params{log},
            ipv6_prefer => 1,
            handle_params => {keepalive => 1},
            response_headers_ref => \$response_headers,
            response_status_stat_ref => \$response_status_stat,
            ($params{retry_fails} ? (num_attempts => 4) : ()),
        );
    };
    my $error;

    if ($@) {
        $error = $@;

        my %stat;
        for my $code (keys %$response_status_stat) {
            my $reqs = $response_status_stat->{$code};
            if ($code =~ /^2/) {
                # http_fetch может кинуть ошибку только на пустой ответ
                $stat{unparsable} += $reqs;
            } elsif ($code =~ /^5/) {
                $stat{server_error} += $reqs;
            } elsif ($code =~ /^4/) {
                $stat{client_error} += $reqs;
            } else {
                $stat{unknown} += $reqs;
            }
        }
        SolomonTools::send_requests_stat($solomon_labels, \%stat);
    }

    log_metrika_query({
        url => $URL,
        request => $request,
        ((!$error) ? (response => $resp) : (error => $error)),
    });

    if ($error) {
        if ($params{skip_errors}) {
            ${ $params{skip_errors} } = 1 if ref $params{skip_errors} eq 'SCALAR';
            LogTools::log_messages("skip_metrika_error", "get_metrika_goals_by_uid");
            return {};
        }
        die $error;
    }

    my $goals = eval { decode_json($resp); };
    $goals = eval { $goals->{owner_to_conditions} };
    if ($@) {
        SolomonTools::send_requests_stat($solomon_labels, {unparsable => 1});
        die $@;
    }

    # меняем на свой формат
    for my $uid (keys %$goals) {
        $goals->{$uid} = [] unless ref($goals->{$uid}) eq 'ARRAY';

        for my $goal (@{ $goals->{$uid} }) {
            $goal->{goal_name} = delete $goal->{name};
            $goal->{goal_domain} = delete $goal->{counter_domain};
            $goal->{goal_type} = delete $goal->{type};
            $goal->{goal_id} = delete $goal->{id};
            $goal->{allow_to_use} = 1;
        }
    }

    SolomonTools::send_requests_stat($solomon_labels, {success => 1});

    return $goals;
}


# --------------------------------------------------------------------

=head2 check_metrika_goals_access_by_uid

Получаем по списку uid-ов, все доступные цели из метрики

  my $goals = check_metrika_goals_access_by_uid([$uid1, $uid2, ...], [goal_id1, goal_id2]);

  в ответе:
  {
    uid1 => [32, 169227],
    uid2 => [323],
  }

    При ошибках падает (если не передали параметр skip_errors)

    perl -Mmy_inc=.. -MRetargeting -MDDP -e 'my $x = Retargeting::check_metrika_goals_access_by_uid([6138950], [123]); p $x'

=cut

sub check_metrika_goals_access_by_uid {
    my $profile = Yandex::Trace::new_profile('retargeting:check_metrika_goals_access_by_uid');
    my ($uids, $goal_ids, %params) = @_;

    $uids = [$uids] if ref($uids) ne 'ARRAY';
    $goal_ids = [$goal_ids] if ref($goal_ids) ne 'ARRAY';

    $uids = [map {$_ + 0} @$uids];
    $goal_ids = [map {$_ + 0} @$goal_ids];

    my $ticket = eval{ Yandex::TVM2::get_ticket($Settings::METRIKA_AUDIENCE_TVM2_ID) }
        or die "Cannot get ticket for $Settings::METRIKA_AUDIENCE_TVM2_ID: $@";

    my $method = "direct/retargeting_conditions_check_access";
    my $solomon_labels = {external_system => "metrika", sub_system => "audience_internal_api", method => $method};
    my $URL = "${Settings::METRIKA_SUPPORT_AUDIENCE_HOST}/$method";

    my $request = {
        uid => to_json($uids),
        ids => to_json($goal_ids)
    };
    my $response_headers;
    my $response_status_stat;
    my $resp = eval {
        Yandex::HTTP::http_fetch('GET', Yandex::HTTP::make_url($URL, $request), {},
            headers => {
            "Content-type" => "application/x-www-form-urlencoded",
                'X-Ya-Service-Ticket' => $ticket
            },
            timeout => $params{timeout} // 10,
            log => $params{log},
            ipv6_prefer => 1,
            handle_params => {keepalive => 1},
            response_headers_ref => \$response_headers,
            response_status_stat_ref => \$response_status_stat,
            ($params{retry_fails} ? (num_attempts => 4) : ()),
        );
    };
    my $error;

    if ($@) {
        $error = $@;

        my %stat;
        for my $code (keys %$response_status_stat) {
            my $reqs = $response_status_stat->{$code};
            if ($code =~ /^2/) {
                # http_fetch может кинуть ошибку только на пустой ответ
                $stat{unparsable} += $reqs;
            } elsif ($code =~ /^5/) {
                $stat{server_error} += $reqs;
            } elsif ($code =~ /^4/) {
                $stat{client_error} += $reqs;
            } else {
                $stat{unknown} += $reqs;
            }
        }
        SolomonTools::send_requests_stat($solomon_labels, \%stat);
    }

    log_metrika_query({
        url => $URL,
        request => $request,
        ((!$error) ? (response => $resp) : (error => $error)),
    });

    if ($error) {
        if ($params{skip_errors}) {
            ${ $params{skip_errors} } = 1 if ref $params{skip_errors} eq 'SCALAR';
            LogTools::log_messages("skip_metrika_error", "check_metrika_goals_access_by_uid");
            return {};
        }
        die $error;
    }

    my $uid_to_goal_ids = eval { decode_json($resp); };
    if ($@) {
        SolomonTools::send_requests_stat($solomon_labels, {unparsable => 1});
        die $@;
    }

    # меняем на свой формат
    for my $uid (keys %$uid_to_goal_ids) {
        $uid_to_goal_ids->{$uid} = [] unless ref($uid_to_goal_ids->{$uid}) eq 'ARRAY';
    }

    SolomonTools::send_requests_stat($solomon_labels, {success => 1});

    return $uid_to_goal_ids;
}

# --------------------------------------------------------------------

=head2 update_condition_goals_accessibility

    update_condition_goals_accessibility($is_accessible
        , $condition_id, \@goals_ids, $goal_type);

    $goal_type - необязателен
    Обновляет is_accessible=$is_accessible (0|1), goal_type (goal|segment) для @goals_ids,
    внутри заданного $condition_id

=cut
sub update_condition_goals_accessibility($$$;$) {
    my $status = shift; # 0|1
    my $condition_id = shift;
    my $goals_ids = shift;
    my $goals_type = shift;

    $goals_ids = [ $goals_ids ] unless ref $goals_ids eq 'ARRAY';

    _update_goal_accessibility($status, $condition_id, $goals_ids, $goals_type);
    _update_lals_accessibility($status, $condition_id, $goals_ids);
}

=head2 _update_lals_accessibility

    Дополнительно обновляет доступность LAL-сегментов по заданному списку
    родительских целей. Идентификаторы LAL'ов собирает из ppcdict.lal_segments,
    статусы обновляет в ppc.retargeting_goals.

=cut
sub _update_lals_accessibility($$$) {
    my $status = shift; # 0|1
    my $condition_id = shift;
    my $parent_ids = shift;

    my $lal_ids_raw = get_all_sql(PPCDICT(), [
        'SELECT lal_segment_id FROM lal_segments',
        WHERE => {parent_goal_id => $parent_ids}
    ]);
    my @lal_ids = map { $_->{lal_segment_id} } @$lal_ids_raw;

    if (scalar(@lal_ids) > 0) {
        _update_goal_accessibility($status, $condition_id, \@lal_ids, 'lal_segment');
    }
}

sub _update_goal_accessibility($$$;$) {
    my $status = shift; # 0|1
    my $condition_id = shift;
    my $goal_ids = shift;
    my $goals_type = shift;

    my $row = {is_accessible => $status};
    $row->{goal_type} = $goals_type if defined $goals_type;
    do_update_table(PPC(ret_cond_id => $condition_id),
       'retargeting_goals',
       $row,
       where => {
           goal_id => $goal_ids,
           ret_cond_id => $condition_id
       }
    );
}

# --------------------------------------------------------------------

=head2 get_metrika_goal_ids_by_uid

Аналог get_metrika_goals_by_uid, но возвращает простой список goal_id, вместо полной структуры целей
При ошибках умирает.

  my $goal_id_arrref = get_metrika_goal_ids_by_uid([$uid1, $uid2]);

=cut

sub get_metrika_goal_ids_by_uid($) {
    my $uids = shift;

    my $metrika_goals = Retargeting::get_metrika_goals_by_uid($uids);
    my $metrika_goals_for_uids = [xuniq {$_->{goal_id}} map {@$_} values %$metrika_goals];
    my $goals_ids = [map {$_->{goal_id}} @$metrika_goals_for_uids];
    return $goals_ids;
}

# --------------------------------------------------------------------

=head2 request_access_to_metrika_counters

Запрос доступа к счетчикам другого логина

    https://st.yandex-team.ru/DIRECT-122201

    Примеры запросов
    Для получение TVM-тикета (DIRECTKNOL-12):
    На тестовой: tvmknife get_service_ticket client_credentials -s 2000693 -d 2000269
    На продакшене: -s 2000270

    Тестовая:
    curl -s -H Content-type:application/json -H 'X-Ya-Service-Ticket:TVM_TICKET --data '{"requests":[{"object_type":"site", "object_id":"tehnika.ru", "send_email":"msa-c23@ya.ru", "permission":"RO", "comment":"comment text", "owner_login":"holodilnikru", "requestor_login":"msa-c23", "service_name":"direct"}]}' 'https://internalapi.test.metrika.yandex.net/internal/grant_requests' | json_xs
    curl -s -H Content-type:application/json -H 'X-Ya-Service-Ticket:TVM_TICKET' --data '{"requests":[{"object_type":"counter", "object_id":"1069", "send_email":"msa-c23@ya.ru", "permission":"RO", "comment":"comment text", "owner_login":"holodilnikru", "requestor_login":"msa-c23", "service_name":"direct"}]}' 'https://internalapi.test.metrika.yandex.net/internal/grant_requests' | json_xs
    Продакшен:
    curl -s -H Content-type:application/json 'X-Ya-Service-Ticket:TVM_TICKET' --data '{"requests":[{"object_type":"site", "object_id":"tehnika.ru", "send_email":"msa-c23@ya.ru", "permission":"RO", "comment":"comment text", "owner_login":"holodilnikru", "requestor_login":"msa-c23", "service_name":"direct"}]}' 'https://internalapi.metrika.yandex.ru/internal/grant_requests' | json_xs
    curl -s -H Content-type:application/json 'X-Ya-Service-Ticket:TVM_TICKET' --data '{"requests":[{"object_type":"counter", "object_id":"1069", "send_email":"msa-c23@ya.ru", "permission":"RO", "comment":"comment text", "owner_login":"holodilnikru", "requestor_login":"msa-c23", "service_name":"direct"}]}' 'https://internalapi.metrika.yandex.ru/internal/grant_requests' | json_xs
    Через функцию:
    /protected$ perl -Mmy_inc=.. -MRetargeting -ME -e 'p Retargeting::request_access_to_metrika_counters([ {object_type => "counter", object_id => 1069, send_email => "msa-c23\@ya.ru", permission => "RO", "comment" => "comment text", "owner_login" => "holodilnikru", "requestor_login" => "msa-c23", "service_name" => "direct"} ], 15035409);'
    /protected$ perl -Mmy_inc=.. -MRetargeting -ME -e 'p Retargeting::request_access_to_metrika_counters([ {object_type => "site", object_id => "tehnika.ru", send_email => "msa-c23\@ya.ru", permission => "RO", "comment" => "comment text", "owner_login" => "holodilnikru", "requestor_login" => "msa-c23", "service_name" => "direct"} ], 15035409);'

    Примеры ответов:
    {"response":[{"result":"ok","object_id":"1069","grants_affected":1}]}
    {"response":[{"result":"error","object_id":"tehn","error_text":"No counters with this site found for login","error_code":"ERR_COUNTER_NOT_FOUND"}]}';
    {"errors":[{"type":"invalid_json","message":"Could not read JSON"}],"code":400,"message":"Could not read JSON"}

=cut

sub request_access_to_metrika_counters($$) {
    my ($requests, $target_uid) = @_;

    my $json_requests = to_json( {requests => $requests} );

    my %req = ( 1 => {
        url => $Settings::METRIKA_GRANT_REQUESTS_URL,
        body => $json_requests,
    });
    my $ticket = eval{ Yandex::TVM2::get_ticket($Settings::METRIKA_TVM2_ID) }
        or die "Cannot get ticket for $Settings::METRIKA_TVM2_ID: $@";

    my $response = Yandex::HTTP::http_parallel_request(
        POST => \%req,
        headers => {
            'Content-Type' => 'application/json',
            'X-Ya-Service-Ticket' => $ticket
            },
        timeout => 15,
        num_attempts => 2,
    )->{1};


    my $result;
    my $object_for_log = {json_requests => $json_requests};
    if (!$response->{is_success}) {
        $object_for_log->{error} = 'request error';
        @{$object_for_log}{qw/response_content response_headers/} = @{$response}{qw/content headers/};

        $result = [{error_text => iget($request_access_to_metrika_common_error), result => 'error'}];
    } else {
        my $json_result = eval { from_json($response->{content}) };
        if ($@) {
            $object_for_log->{error} = 'decoding error';
            $object_for_log->{error_details} = $@;

            $result = [{error_text => iget($request_access_to_metrika_common_error), result => 'error'}];
        } elsif (!defined $json_result->{response}) {
            $object_for_log->{error} = 'undefined response';
            $object_for_log->{json_result} = $json_result;

            $result = [{error_text => iget($request_access_to_metrika_common_error), result => 'error'}];
        } else {
            $object_for_log->{json_result} = $json_result;

            $result = $json_result->{response};
            my (%counters_in_connect);
            for my $row (@$result) {
                if ($row->{result} eq 'error') {
                    if (exists $request_access_to_metrika_errors->{$row->{error_code}}) {
                        # если ошибка известная, то меняем на свое сообщение
                        $row->{error_text} = iget($request_access_to_metrika_errors->{$row->{error_code}});
                    } else {
                        $object_for_log->{warning} //= 'unknown error_code';
                        push @{ $object_for_log->{warning_data} }, $row;

                        # а если неизвестная  - все равно меняем сообщение,
                        # чтобы не показывать клиенту "пустую" ту же страницу, но без сообщения об успехе.
                        $row->{error_text} = iget($request_access_to_metrika_common_error);
                    }
                } elsif ($row->{result} eq 'counter_in_connect') {
                    my $object_id = $row->{object_id};
                    $row->{error_text} = iget($request_access_to_metrika_errors->{ERR_COUNTER_IN_CONNECT})
                        unless $object_id;
                    $counters_in_connect{$object_id} = $row;
                }
            }

            if (keys %counters_in_connect) {
                my $response = _request_access_to_metrika_counters_in_connect([keys %counters_in_connect], $target_uid);
                my $result;
                # 200 ОК означает успешное выполнение запроса, в этом случае Connect возвращает пустой json
                # Если же есть ошибки - он вернет 4XX, в code - текстовый код ошибки,
                # в params->resources - идентификаторы проблемных счетчиков через запятую, строкой
                unless ( $response->{is_success} ) {
                    my @counters;
                    my $error_text;
                    eval { $result = from_json($response->{content}) };
                    if ( $result && ref $result eq 'HASH' && $result->{code} && $result->{code} =~ 'resource_not_found') {
                        if (my $resources_string = $result->{params}->{resources}){
                            @counters = split /\s+,\s+/, $resources_string;
                            $error_text = iget($request_access_to_metrika_errors->{ERR_CONNECT_RESOURCE_NOT_FOUND});
                        }
                    } else {
                        $error_text = iget($request_access_to_metrika_errors->{ERR_COUNTER_IN_CONNECT});
                    }
                    @counters = keys %counters_in_connect unless @counters;
                    foreach my $counter (@counters) {
                        $counters_in_connect{$counter}->{error_text} = $error_text;
                    }
                }
            }
        }
    }
    log_metrika_query($object_for_log);
    return $result;
}

sub _request_access_to_metrika_counters_in_connect {
    my ($counter_ids, $target_uid) = @_;

    my $request_data;
    # id пользователя обязательно должен быть числом
    $request_data->{objects} = [{'type' => 'user', 'id' => 0 + $target_uid}];
    # а идентификаторы счетчиков - строками
    $request_data->{resource_ids} = [ map { ''.$_ } @$counter_ids ];
    $request_data->{comment} = 'grants for retargeting';
    my %req = ( 1 => {
        url => $Settings::CONNECT_METRIKA_GRANT_REQUESTS_URL,
        body => to_json($request_data)
    });

    my $response = Yandex::HTTP::http_parallel_request(
        POST => \%req,
        headers => {
            'Content-Type' => 'application/json',
            'x-uid' => $target_uid,
            'x-user-ip' => $ENV{REMOTE_ADDR} || '127.0.0.1',
            'X-Ya-Service-Ticket' => Yandex::TVM2::get_ticket($Settings::CONNECT_TVM2_ID),
        },
        timeout => 15,
        num_attempts => 2,
    )->{1};

    return $response;
}

# --------------------------------------------------------------------

=head2 get_group_retargeting

Достаем все условия на группу баннеров (из bids_retargeting)

    my $retargetings = Retargeting::get_group_retargeting(pid => [pid1, pid2]);
    my $retargetings = Retargeting::get_group_retargeting(ret_cond_id => [123, 3244]);

В результате хеш:
    $retargetings: {
        pid1 => [
            {
                ret_id => 1234567,
                ret_cond_id => 123,
                price_context => 0.77,
                autobudgetPriority => 1,
                is_suspended => 0
            },
            {...}
        ],
        pid2 => [{}, {}],
    }

=cut
sub get_group_retargeting {
    my %params = @_;

    my ($get_by_key, $get_by_vals, %get_by_filter, %get_by_params);

    if (exists $params{pid}) {
        ($get_by_key, $get_by_vals) = (adgroup_id => $params{pid}) if !$get_by_key;
        $get_by_filter{'g.pid'} = $params{pid};
    }
    if (exists $params{ret_cond_id}) {
        ($get_by_key, $get_by_vals) = (ret_cond_id => $params{ret_cond_id}) if !$get_by_key;
        $get_by_filter{'br.ret_cond_id'} = $params{ret_cond_id};
    }
    if (exists $params{ret_id}) {
        ($get_by_key, $get_by_vals) = (ret_id => $params{ret_id}) if !$get_by_key;
        $get_by_filter{'br.ret_id'} = $params{ret_id};
        $get_by_params{ClientID} = $params{ClientID};
    }
    if (exists $params{type}) {
        $get_by_params{type} = $params{type};
    }

    croak 'No selective conditions for get_group_retargeting' if !$get_by_key;

    $get_by_filter{'c.statusEmpty'} = 'No';

    my $retargetings = Direct::Retargetings->get_by($get_by_key, $get_by_vals, filter => \%get_by_filter, %get_by_params)->items;

    my $legacy_data = get_hashes_hash_sql(PPC(ret_cond_id => [map { $_->ret_cond_id } @$retargetings]), [
        "SELECT STRAIGHT_JOIN br.ret_id, br.bid, c.currency FROM bids_retargeting br JOIN phrases p on p.pid = br.pid JOIN campaigns c on c.cid = p.cid",
        WHERE => {'br.ret_cond_id' => SHARD_IDS},
    ]);

    my $result = {};
    push @{$result->{$_->adgroup_id}}, {%{$_->to_db_hash}, %{$legacy_data->{$_->id} // {}}} for @$retargetings;

    delete @{$_}{qw/modtime statusBsSynced/} for map { @$_ } values %$result;

    return $result;
}

# превращает хеш с условиями ретаргетинга на группу в хеш условий ретаргетинга
sub _group_retargeting_to_ret_id_hash {
    my $group_retargetings = shift;
    my %result;
    while (my ($pid, $retargeting) = each %$group_retargetings) {
        for my $ret (@$retargeting) {
            $result{$ret->{ret_id}} = $ret;
        }
    }
    return \%result;
}

=head2 get_group_retargeting_ret_id_hash

    Действует аналогично get_group_retargeting
    Возвращает хеш, в котором ключами явлются id ретарегетинга, а значениями - условия.
        { 1 => $ret1, 2 => $ret_2, ...}

=cut
sub get_group_retargeting_ret_id_hash {
    my $ret_by_pid = get_group_retargeting(@_);
    return _group_retargeting_to_ret_id_hash($ret_by_pid);
}

# --------------------------------------------------------------------

=head2 get_retargetings_into_groups

    get_retargetings_into_groups($groups);
    модифицирует каждую группу в $groups:
      - добавляем в него retargetings (если есть на группе)
      - расчитываем покрытие для каждого условия,
      - добавляеи hits_by_cost_context_table

=cut

sub get_retargetings_into_groups($) {
    my ($groups) = @_;

    my $retargetings = Retargeting::get_group_retargeting(pid => [map {$_->{pid}} @$groups]);

    for my $group (@$groups) {
        my ($pid, $geo) = ($group->{pid}, $group->{geo});
        my $group_retargetings = $retargetings->{$pid};
        if ($group_retargetings) {
            # пока решили не показывать охват по цене и прогноз охвата
            # for my $row (@$banner_retargetings) {
            #     $row->{context_coverage} = Pokazometer::get_coverage_by_cost_context($row->{price_context}, $geo);
            # }

            $group->{retargetings} = $group_retargetings;
            # пока решили не показывать охват по цене и прогноз охвата
            # $banner->{hits_by_cost_context_table} = Pokazometer::get_hits_by_cost_context_table($geo);
        }
    }
}

=head2 get_search_retargeting_into_groups_from_multipliers

    get_search_retargeting_into_groups_from_multipliers($groups);
    модифицирует каждую группу в $groups:
        - если группа содержит корректировку типа retargeting_filter в $group->{hierarchical_multipliers}, добавляем
        в неё search_retargetings с данными оттуда
=cut

sub get_search_retargeting_into_groups_from_multipliers($) {
    my ($groups) = @_;

    for my $group (@$groups) {
        my ($pid, $cid, $currency) = ($group->{pid}, $group->{cid}, $group->{currency} // 'YND_FIXED');
        my $min_price = get_currency_constant($currency, 'MIN_PRICE');

        for my $multiplier_type (keys %{ $group->{hierarchical_multipliers} || {} }) {
            if ($multiplier_type eq 'retargeting_filter') {
                my $group_search_retargetings = [];
                my $multiplier = $group->{hierarchical_multipliers}->{$multiplier_type};
                for my $condition_id (keys %{$multiplier->{conditions}}) {
                    my $search_retargeting = {
                        ret_cond_id => $condition_id,
                        pid => $pid,
                        cid => $cid,
                        currency => $currency, 
                        ret_id => $multiplier->{conditions}->{$condition_id}->{retargeting_multiplier_value_id},
                        
                        #дефолтные/пустые значения для остального
                        bid => "0",
                        price => $min_price,
                        price_context => $min_price,
                    };

                    push @{$group_search_retargetings}, $search_retargeting;
                }
                $group->{search_retargetings} = $group_search_retargetings;
            }
        }
    }
}

=head2 get_target_interests_into_groups

    get_target_interests_into_groups($groups);
    модифицирует каждую группу в $groups:
      - добавляем в него target_interests (если есть на группе)

=cut

sub get_target_interests_into_groups($) {
    my ($groups) = @_;

    my $adgroup_id2target_interests = Direct::TargetInterests->get_by(adgroup_id => [map {$_->{pid}} @$groups])->items_by('adgroup_id');
    return unless %$adgroup_id2target_interests;

    for my $group (@$groups) {
        $group->{target_interests} = [ map {$_->to_db_hash} @{ $adgroup_id2target_interests->{$group->{pid}} } ];
    }
}

# --------------------------------------------------------------------

=head2 get_retargetings_into_mediaplan_banners

    get_retargetings_into_mediaplan_banners($banners);
    модифицирует каждый баннер в $banners:
      - добавляем в него retargetings (если есть на баннере)
      - расчитываем покрытие для каждого условия,
      - добавляеи hits_by_cost_context_table

=cut

sub get_retargetings_into_mediaplan_banners($) {
    my ($banners) = @_;

    return unless @$banners;
    my $currency = $banners->[0]->{currency};
    # В медиапланах на данный момент предлагаемая цена на условие ретаргетинга равна минимально ставке. Одна на все.
    my $min_price = get_currency_constant($currency, 'MIN_PRICE');
    # хрупкая и не очень хорошая конструкция определения шарда,
    # т.к. нет разбивки mbid по cid и нельзя распределить запрос по шардам.
    # TODO: подумать, как бы улучшить (передавать ClientID извне?) при шардинге/рефакторинге медиапланов
    my $banner_retargetings_raw = get_all_sql(PPC(cid => [map {$_->{cid}} @$banners]),
        ["select mbid
               , ret_id
               , ret_cond_id
               , place
               , is_suspended
               , $min_price as price_context
          from mediaplan_bids_retargeting
         ", where => {mbid => [map {$_->{mbid}} @$banners]}
         ]
    );

    my $banner_retargetings = {};
    for my $row (@$banner_retargetings_raw) {
        push @{$banner_retargetings->{$row->{mbid}}}, $row;
    }

    for my $banner (@$banners) {
        $banner->{retargetings} = $banner_retargetings->{ $banner->{mbid} } if exists $banner_retargetings->{ $banner->{mbid} };
        # пока решили не показывать охват по цене и прогноз охвата
        # $banner->{hits_by_cost_context_table} = Pokazometer::get_hits_by_cost_context_table($banner->{geo});
    }
}

# --------------------------------------------------------------------

=head2 update_group_retargetings

    Сохраняем условие в группу.

    Перед сохранением вытягиваем текущие сохранённые в БД данные, и в зависимости от изменений
    принимаем решение о перепосылки в БК всей группы.

    Возвращает список bids_retargeting.ret_id, которые в результате оказались привязанными к группе.

    $new_banner - новый баннер

    %OPTIONS:
      insert_only => 1 -- только вставляем все условия из new_banner
                            (без удаления существующих в старом баннере, но не указанных в новом условий ретаргетинга)

=cut
sub update_group_retargetings {
    my ($new_banner, %OPTIONS) = @_;

    die "Nowhere to save retargetings - no pid or bid in banner" unless $new_banner->{pid} || $new_banner->{bid};

    my @data_for_log_price;
    my @mail_notifications;
    my $significant_change;

    my $pid = $new_banner->{pid} || get_pid(bid => $new_banner->{bid});
    my $old_retargetings = get_group_retargeting_ret_id_hash(pid => [$pid]);
    my $new_retargetings = {};

    # Собираем данные, которые собираемся записывать в bids_retargeting
    for my $row (@{ $new_banner->{retargetings} || [] }) {
        my ($autobudgetPriority, $price_context, $is_suspended);

        if ($row->{autobudgetPriority}) {
            # из нового баннера
            $autobudgetPriority = $row->{autobudgetPriority};
        } elsif ($row->{ret_id} && exists $old_retargetings->{ $row->{ret_id} }
                 && defined $old_retargetings->{ $row->{ret_id} }->{autobudgetPriority}
        ) {
            # из старого баннера
            $autobudgetPriority = $old_retargetings->{ $row->{ret_id} }->{autobudgetPriority};
        } else {
            # значение по-умолчанию
            $autobudgetPriority = 3;
        }

        if (($OPTIONS{camp_strategy} && $OPTIONS{camp_strategy}->{is_autobudget})
            || !$row->{price_context}
        ) {
            # если у кампании - автобюджетная стратегия или в новом баннере не определена цена - берем минимальную
            $price_context = get_currency_constant($new_banner->{currency}, 'MIN_PRICE');
        } else {
            $price_context = $row->{price_context};
        }

        if (defined $row->{is_suspended}) {
            # в новом баннере явно определено состояние ретаргетинга - берем его
            $is_suspended = $row->{is_suspended};
        } elsif (defined $row->{ret_id}
                 && exists $old_retargetings->{ $row->{ret_id} }
                 && defined $old_retargetings->{ $row->{ret_id} }->{is_suspended}
        ) {
            # состояние ретаргетинга есть в старом баннере - берем его
            $is_suspended = $old_retargetings->{ $row->{ret_id} }->{is_suspended};
        } else {
            # значение по-умолчанию - ретаргетинг включен
            $is_suspended = 0;
        }

        my $new_ret_id = $row->{ret_id} || get_new_id('ret_id');
        $new_retargetings->{$new_ret_id} = {
            ret_id => $new_ret_id,
            ret_cond_id => $row->{ret_cond_id},
            bid => $new_banner->{bid}, # TODO adgroup: удалить
            pid => $pid,
            cid => $new_banner->{cid},
            price_context => $price_context,
            autobudgetPriority => $autobudgetPriority,
            statusBsSynced => 'No',
            is_suspended => $is_suspended,
        };

        push @data_for_log_price, {
            cid => $new_banner->{cid},
            pid => $pid,
            bid => $new_banner->{bid},  # TODO adgroup: удалить
            id => $new_ret_id,
            type => $row->{ret_id} ? 'ret_update' : 'ret_add',
            price => 0,
            price_ctx => $price_context,
            currency    => $new_banner->{currency},
        };

        if ($OPTIONS{UID}) {
            push @mail_notifications, {
                object     => 'adgroup',
                event_type => 'b_retargeting',
                object_id  => $pid,
                old_text   => '',
                new_text   => $row->{ret_cond_id},
                uid        => $OPTIONS{UID},
            };
        }
    }

    my ($adgroup_changed, @insert_retargetings);
    # Смотрим, что поменялось на самом деле. Обновляем данные в БД, а
    # также вычисляем, нужно ли трогать statusBsSynced/LastChange
    # самой группы.
    while (my ($ret_id, $new) = each %$new_retargetings) {
        if (exists $old_retargetings->{$ret_id}) {
            my $old = delete $old_retargetings->{$ret_id};
            $adgroup_changed = 1 if $old->{ret_cond_id} != $new->{ret_cond_id} || $old->{is_suspended} != $new->{is_suspended};
            my $retargeting_changed =
                $old->{ret_cond_id} != $new->{ret_cond_id}
                || $old->{price_context} != $new->{price_context}
                || $old->{autobudgetPriority} != $new->{autobudgetPriority}
                || $old->{is_suspended} != $new->{is_suspended};
            if ($retargeting_changed) {
                do_update_table(PPC(pid => $pid), 'bids_retargeting', $new, where => {ret_id => $ret_id});
            }
        } else {
            $adgroup_changed = 1;
            push @insert_retargetings, $new;
        }
    }
    if (@insert_retargetings) {
        my @column_names = keys %{$insert_retargetings[0]};
        my $columns_str = join ", ", map { sql_quote_identifier($_) } @column_names;
        do_mass_insert_sql(PPC(pid => $pid), "insert into bids_retargeting($columns_str) values %s",
                           [ map { [ @{$_}{@column_names} ] } @insert_retargetings ]);
    }

    # удаляем отсутствующие
    unless ($OPTIONS{insert_only}) {
        # phrases.statusBsSynced и phrases.LastChange для удалённых условий ретаргетинга обрабатываются внутри этого вызова
        delete_group_retargetings([values %$old_retargetings], { $pid => $new_banner->{bid} }, UID => $OPTIONS{UID});
    }

    if ($adgroup_changed) {
        do_update_table(PPC(pid => $pid), 'phrases',
            { statusBsSynced => 'No', LastChange__dont_quote => 'now()'},
            where => { pid => $pid }
        );
    }

    LogTools::log_price(\@data_for_log_price);
    mass_mail_notification(\@mail_notifications) if @mail_notifications;
    return [keys %$new_retargetings];
}

=head2 delete_group_retargetings

    Удаляет ретаргетинги с группы.
    Принимает следующие параметры:
      позиционные
        ret_for_delete  - ссылка на массив условий ретаргетинга. обязательно должны содержать ret_cond_id !
        pid2bid         - хеш соответствия pid => bid - по нему определяется bid для почтовых уведомлений
      именованные
        UID             - uid, которому нужно отправить почтовые уведомления об удалении условий ретаргетинга

=cut
sub delete_group_retargetings {
    my ($ret_for_delete, $pid2bid, %OPTIONS) = @_;

    my @mail_notifications;
    my %changed_groups;

    return unless @$ret_for_delete;

    for my $row (@$ret_for_delete) {
        $changed_groups{$row->{pid}} = undef;
        die qq/No ret_cond_id specified for ret_id: $row->{ret_id}/ unless $row->{ret_cond_id};
    }

    do_in_transaction {
        do_delete_from_table(PPC(ret_cond_id => [map {$_->{ret_cond_id}} @$ret_for_delete]),
            'bids_retargeting',
            where => { ret_id => [map {$_->{ret_id}} @$ret_for_delete] }
        );

        if ($OPTIONS{UID}) {
            for my $row (@$ret_for_delete) {
                push @mail_notifications, {
                    object     => 'adgroup',
                    event_type => 'ret_delete',
                    object_id  => $pid2bid->{$row->{pid}} // $row->{bid},
                    old_text   => $row->{ret_cond_id},
                    new_text   => '',
                    uid        => $OPTIONS{UID},
                };
            }
        }

        if (%changed_groups) {
            my $adgroup_ids = [keys %changed_groups];
            do_update_table(PPC(pid => $adgroup_ids), 'phrases',
                { statusBsSynced => 'No', LastChange__dont_quote => 'NOW()'},
                where => {pid => SHARD_IDS}
            );
        }
    };

    mass_mail_notification(\@mail_notifications) if @mail_notifications;

    return
}

# --------------------------------------------------------------------

=head2 update_group_retargetings_params

Обновляем цены, отключенность, приоритет для условий в баннере
    update_group_retargetings_params($new_retargetings, $retargetings_to_delete, %OPTIONS);

    на входе 2 массива хешей - для обновления и для удаления
    [{
        ret_id => 12344,
        price_context => 0.01,
        autobudgetPriority => 0,
        is_suspended => 0,
        bid =>
        pid =>
    }, {}, ...]

    OPTIONS:
      cid       -- __обязательный параметр__ - номер кампании, которой принадлежат баннеры/условия ретаргетинга
      log_label -- если задана строка, то под такой меткой сохраянем логи цен (используется в кострукторе цен)

    # sharding: TODO - заменить cid на ClientID, если метод будет вызываться для баннеров более чем __одной__ кампании за раз, кроме того - обрабатывать все в цикле по шардам, предварительно определив, какие условия к каким относятся

=cut

sub update_group_retargetings_params {
    my ($new_retargetings, $retargetings_to_delete) = (shift, shift);
    my %OPTIONS = @_;

    my $cid = $OPTIONS{cid};
    die 'Valid cid must be specified' unless is_valid_int($cid, 1);

    my $strategy = $OPTIONS{strategy};
    return iget('Показы в сетях выключены, изменение условий ретаргетинга невозможно')
        if $strategy->{is_net_stop} && @{$new_retargetings // []};

    # обновление
    my (@data_for_log_price, @mail_notifications);
    my %changed_groups;

    # Выбираем старые цены для уведомлений об изменении
    my $old_retargetings = {};
    if (@$new_retargetings && $OPTIONS{UID}) {
        $old_retargetings = get_hash_sql(PPC(cid => $cid), [
               "SELECT ret_id, price_context FROM bids_retargeting",
               where => { ret_id => [map {$_->{ret_id}} @$new_retargetings] }
       ]);
    }

    for my $row (@$new_retargetings) {
        my $set = {};
        if (defined $row->{price_context} ) {
            return iget('Установлена автоматическая стратегия - изменение значений ставок невозможно')
                if $strategy->{is_autobudget};
            $set->{price_context} = $row->{price_context} if defined $row->{price_context};
        }
        if (defined $row->{autobudgetPriority}) {
            $set->{autobudgetPriority} = $row->{autobudgetPriority};
            $set->{price_context} = get_currency_constant($OPTIONS{currency}, 'MIN_PRICE');
        }

        if (defined $row->{is_suspended}) {
            $set->{is_suspended} = $row->{is_suspended};
        }

        next if ! %$set;

        $changed_groups{$row->{pid}} = 1;
        $set->{statusBsSynced} = 'No',

        my $where = {
            ret_id => $row->{ret_id}
        };
        do_update_table(PPC(cid => $cid), 'bids_retargeting', $set, where => $where);

        push @data_for_log_price, {
            cid => $row->{cid},
            bid => $row->{bid},
            pid => $row->{pid},
            id => $row->{ret_id},
            type => $OPTIONS{log_label} || 'ret_update_ajax',
            price => 0,
            price_ctx => $set->{price_context},
            currency    => $OPTIONS{currency},
        } if $set->{price_context};

        if ($OPTIONS{UID}) {
            push @mail_notifications, {
                object     => 'phrase',
                event_type => 'ph_price_ctx',
                object_id  => $row->{bid},
                old_text   => $old_retargetings->{ $row->{ret_id} },
                new_text   => $set->{price_context},
                uid        => $OPTIONS{UID},
            };
        }
    }
    if (%changed_groups) {
        do_update_table(PPC(cid => $cid), 'phrases',
            { statusBsSynced => 'No', LastChange__dont_quote => 'now()'},
            where => { pid => [keys %changed_groups] }
        );
    }

    if (@data_for_log_price) {
        LogTools::log_price(\@data_for_log_price);
    }

    mass_mail_notification(\@mail_notifications) if @mail_notifications;

    # удаление
    if (@{$retargetings_to_delete || []}) {
        my $ret_id_for_delete = [map {$_->{ret_id}} @$retargetings_to_delete];
        my $ret_cond_id_for_delete = get_all_sql(PPC(cid => $cid), [
            'SELECT bid, pid, ret_id, ret_cond_id FROM bids_retargeting',
            where => { ret_id => $ret_id_for_delete }
        ]);

        delete_group_retargetings($ret_cond_id_for_delete, {}, UID => $OPTIONS{UID});
    }

    return;
}

# --------------------------------------------------------------------

=head2 auto_price_for_retargetings

    Установка цен из конструктора цен
    $new_retargeting = auto_price_for_retargetings($retargeting, $params, $geo_ids);
    меняет price_context и возвращает новый $retargeting для update_group_retargetings_params()

    $retargeting -- строка из bids_retargeting для одного условия
    $params -- параметры конструктора
    $geo_ids -- строка гео-таргетинга
    $currency — валюта кампании

=cut

sub auto_price_for_retargetings {
    my ($retargeting, $params, $geo_ids, $currency) = @_;

    die 'no currency given' unless $currency;

    my $new_retargeting = {ret_id => $retargeting->{ret_id}};
    my $new_price_context = $new_retargeting->{price_context};

    if ($params->{single_price}) {
        $new_price_context = $params->{single_price};
    } elsif ($params->{for_different_places} && $params->{single} && $params->{single}->{price_ctx}) {
        $new_price_context = $params->{single}->{price_ctx};
    } else {
        # $new_price_context = Pokazometer::get_price_for_coverage_by_cost_context($params->{scope}, $geo_ids); # показометр пока не используем
        $new_price_context = get_currency_constant($currency, 'MIN_PRICE');
    }

    $new_price_context = currency_price_rounding($new_price_context, $currency, min_const => $params->{min_const}, max_const => $params->{max_const});

    if ($params->{max_price} && $new_price_context > $params->{max_price}) {
        $new_price_context = $params->{max_price};
    }

    $new_retargeting->{price_context} = $new_price_context;
    return $new_retargeting;
}

# --------------------------------------------------------------------

=head2 switch_retargetings_in_groups

Включение/выключение условий ретаргетинга на всех объявлениях

=cut

sub switch_retargetings_in_groups {
    my %params = @_;

    my $set = {is_suspended => $params{is_suspended} ? 1 : 0};

    my $where = {};
    $where->{"bids_retargeting.pid"} = $params{pid} if exists $params{pid};
    $where->{"bids_retargeting.ret_cond_id"} = $params{ret_cond_id} if exists $params{ret_cond_id};
    $where->{"bids_retargeting.ret_id"} = $params{ret_id} if exists $params{ret_id};

    die "switch_retargetings_in_groups: need bid or ret_cond_id param" unless keys %$where;
    # Умрет, если нет ни того ни другого
    my @shard = choose_shard_param(\%params, ['ClientID', 'pid']);

    do_update_table(PPC(@shard), 'bids_retargeting', $set, where => $where);
    do_sql(PPC(@shard), ["
            UPDATE phrases
                   JOIN bids_retargeting USING(pid)
               SET phrases.statusBsSynced = 'No'
        ", where => $where
    ]);
}

# --------------------------------------------------------------------

=head2 get_retargetings_form

По пришедшему из формы retargeting_conditions_id (строка через запятую id условий) модифицируем группу, поле retargetings
Добавляем hits_by_cost_context_table в группу для расчета нового охвата
Возвращает ошибку в случае ошибки.

    my $ret_error = get_retargetings_form(group => $group # группа со старыми условиями в поле retargetings
                                          , retargeting_conditions_id => $FORM{retargeting_conditions_id} # новые условия
                                          , all_retargeting_conditions => $all_retargeting_conditions # из get_retargeting_conditions()
                                          , geo => $FORM{geo}
                                          , MIN_PRICE => $MIN_PRICE
                                         );

=cut

sub get_retargetings_form {
    my %OPT = @_;
    my ($group, $old_retargetings, $new_retargetings, $all_retargeting_conditions) =
       ($OPT{group}, $OPT{old_retargetings}, $OPT{new_retargetings}, $OPT{all_retargeting_conditions});
    $OPT{MIN_PRICE} ||= 0.01;
    if (defined ($new_retargetings)) {
        if (all {is_valid_id($_)} map {$_->{ret_cond_id}} @$new_retargetings){
            return iget("Условие подбора аудитории недоступно") unless all { exists $all_retargeting_conditions->{$_->{ret_cond_id}} } @$new_retargetings;
            # пока решили не показывать охват по цене и прогноз охвата
            # my $default_price_context = Pokazometer::get_price_for_coverage_by_cost_context($Settings::DEFAULT_COVERAGE_FOR_RETARGETING, $OPT{geo});
            my %old_retargetings = map {$_->{ret_id} => $_} @$old_retargetings;
            # TODO : убрать эту специальную логику, когда редактирование CPM-условий будет в отдельных popup-ах, как в других местах
            if ($group->{adgroup_type} =~ 'cpm_banner') {
                $group->{retargetings} = [
                    map {$_->{ret_id} && exists $old_retargetings{$_->{ret_id}}
                        # Позволяем пользователю установить price_context
                        ? hash_merge $old_retargetings{$_->{ret_id}}, {
                            groups => $_->{groups},
                            type => $_->{type},
                            condition_name => $_->{condition_name},
                            condition_desc => $_->{condition_desc},
                            price_context => $_->{price_context},
                        }
                        : hash_merge { price_context => 0, autobudgetPriority => 3, }, $_
                    }
                    @$new_retargetings
                ];
            } else {
                $group->{retargetings} = [
                    map {$_->{ret_id} && exists $old_retargetings{$_->{ret_id}}
                        ? hash_merge { groups => $_->{groups}, type => $_->{type}, condition_name => $_->{condition_name}, condition_desc => $_->{condition_desc}, }, $old_retargetings{$_->{ret_id}}
                        : hash_merge { price_context => $OPT{MIN_PRICE}, autobudgetPriority => 3, }, $_
                    }
                    @$new_retargetings
                ];
            }
            if ($OPT{is_groups_copy_action}) {
                $_->{is_suspended} = 0 foreach @{$group->{retargetings}};
            }

          # перерасчитываем охват по новому geo
          # пока решили не показывать охват по цене и прогноз охвата
          # for my $row (@{ $group->{retargetings} }) {
          #     $row->{context_coverage} = Pokazometer::get_coverage_by_cost_context($row->{price_context}, $OPT{geo});
          # }
        } else {
            return iget("Ошибка в входных данных");
        }

    } elsif (defined $old_retargetings && @{$old_retargetings}) {
        # удалили все условия
        $group->{retargetings} = [];
    }

#    if (exists $group->{retargetings} && @{ $group->{retargetings} }) {
        # пока решили не показывать охват по цене и прогноз охвата
        # $group->{hits_by_cost_context_table} = Pokazometer::get_hits_by_cost_context_table($OPT{geo});
#    }

    return '';
}

# --------------------------------------------------------------------

=head2 get_target_interests_form

Обрабатываем пришедшие таргетинги по интересам
Возвращает ошибку в случае ошибки.

    my $ret_error = get_target_interests_form(
        $group # группа со старыми таргетингами по интересам в поле target_interests
      , $target_categories # справочник категорий (Direct::TargetingCategories->get_rmp_interests->items_by)
      , $old_target_interests # существующие таргетинги по интересам (из бд)
      , $min_price # ставка по умолчанию
      , $is_groups_copy_action # копирование ли это группы
    );

=cut

sub get_target_interests_form {
    my ($group, $target_categories, $old_target_interests, $min_price, $is_groups_copy_action) = @_;
    $old_target_interests //= [];
    $min_price //= 0.01;

    if (defined $group->{target_interests}) {
        return iget("Условие подбора аудитории (интерес) недоступно") unless all { exists $target_categories->{$_->{target_category_id}} } @{ $group->{target_interests} };

        my %old_target_interests = map {$_->{ret_id} => $_} @$old_target_interests;
        foreach my $target_interest (@{ $group->{target_interests} }) {
            if ($target_interest->{ret_id} && exists $old_target_interests{$target_interest->{ret_id}}) {
                $target_interest = $old_target_interests{$target_interest->{ret_id}};
            } else {
                $target_interest->{price_context} //= $min_price;
                $target_interest->{autobudgetPriority} //= 3;
            }
            $target_interest->{category_name} //= $target_categories->{$target_interest->{target_category_id}}->to_template_hash->{'name'};
            $target_interest->{is_suspended} = 0 if $is_groups_copy_action;
        }
    } elsif (defined $old_target_interests && @{$old_target_interests}) {
        $group->{target_interests} = [];
    }
    return '';
}

# --------------------------------------------------------------------

=head2 copy_retargetings_between_groups

    Копируем условия ретаргетинга с одного баннера на другой
    если новая кампания принадлежит другому клиенту, то копируем дополнительно сами условия

    Retargeting::copy_retargetings_between_groups(
        new_cid => $new_cid,
        old_cid => $old_cid,
        new_pid => $new_pid,
        old_pid => $old_pid,
        new_bid => $new_bid,
        copy_suspended_status => 1, # если нужно скопировать статус is_suspended
        price_convert_rate => - множитель для ставок на фразы и условия ретаргетина
                                используется при копировании в аккаунт с другой валютой и конвертации копированием
        new_currency => 'RUB'|'YND_FIXED'|'EUR'|... - новая валюта
    )

=cut

sub copy_retargetings_between_groups {
    my %params = @_;

    # TODO adgroup: удалить new_bid после полного перехода к bids_retargeting.pid
    my ($new_cid, $old_cid, $new_pid, $old_pid, $new_bid) = map {$params{$_} || die "no $_ given"} qw/new_cid old_cid new_pid old_pid new_bid/;
    die 'new cid or pid == old' if $new_cid == $old_cid || $new_pid == $old_pid;

    my $clients = get_cid2clientid(cid => [$new_cid, $old_cid]);

    my $old_pids_retargetings = Retargeting::get_group_retargeting(pid => $old_pid, type => [ qw/retargeting interest/ ]);
    return unless exists $old_pids_retargetings->{$old_pid}; # нет условий на исходном баннере

    my $old_pid2ret_cond_ids->{$old_pid} = [map {$_->{ret_cond_id}} @{$old_pids_retargetings->{$old_pid}}];
    my $ret_cond_ids_old2new = {};

    die 'no new_currency given' unless $params{new_currency};

    # В нормальных условиях этого не должно происходить - не для всех cid определился ClientID
    die 'Not all ClientIDs found for cids' unless scalar(keys %$clients) == 2;

    if ($clients->{$new_cid} == $clients->{$old_cid}) {
        # копирование между кампаниями одного клиента

        # уплощаем хеш со старыми ретаргетингами
        my $old_retargetings = _group_retargeting_to_ret_id_hash($old_pids_retargetings);

        # копируем условия по соцдему
        my $new_ret_cond_ids = _copy_interests_retargetings(
            $clients->{$old_cid},
            [map {$_->{ret_cond_id}} @{ $old_pids_retargetings->{$old_pid} }],
        );
        $ret_cond_ids_old2new = $new_ret_cond_ids;

        # получаем сразу пачку новых ret_id
        my $new_ret_ids = JavaIntapi::GenerateObjectIds->new(object_type => 'retargeting',
                count => scalar(keys %$old_retargetings))->call();

        my (@bids_to_insert, @log_price_data);
        for my $ret (values %$old_retargetings) {
            my $new_ret_id = shift @$new_ret_ids;
            my $new_price_context = currency_price_rounding(
                $ret->{price_context} * ($params{price_convert_rate} // 1),
                $params{new_currency},
                min_const => $params{min_const},
                max_const => $params{max_const}
            );
            push @bids_to_insert, [
                $new_ret_id,
                $ret_cond_ids_old2new->{$ret->{ret_cond_id}} // $ret->{ret_cond_id},
                $new_price_context,
                $ret->{autobudgetPriority},
                $new_cid,
                $new_pid,
                $new_bid,   # и здесь удалить
                ($params{copy_suspended_status} ? $ret->{is_suspended} : 0),
            ];
            push @log_price_data, {
                cid => $new_cid,
                pid => $new_pid,
                bid => $new_bid, # и здесь удалить
                id => $new_ret_id,
                price => 0,
                price_ctx => $new_price_context,
                type => 'ret_add',
                currency => $params{new_currency},
            };
        }

        if (@bids_to_insert) {
            LogTools::log_price(\@log_price_data);
            # TODO adgroup - удалить bid после полного перехода к bids_retargeting.pid
            do_mass_insert_sql(PPC(cid => $new_cid), 'INSERT INTO bids_retargeting (ret_id, ret_cond_id, price_context, autobudgetPriority, cid, pid, bid, is_suspended) VALUES %s', \@bids_to_insert);
        }
    } else {
        # копируем другому клиенту

        # отдельно копируем сами условия
        my $new_ret_cond_ids = Retargeting::copy_retargetings_between_clients(
            old_client_id => $clients->{$old_cid},
            new_client_id => $clients->{$new_cid},
            old_ret_cond_ids => [map {$_->{ret_cond_id}} @{ $old_pids_retargetings->{$old_pid} }],
        );
        $ret_cond_ids_old2new = $new_ret_cond_ids;

        my $new_banner = {
            cid => $new_cid,
            pid => $new_pid,
            retargetings => [],
            # TODO adgroup: удалить или подумать, что попадет в update_group_retargetings->data_for_log_price
            bid => $new_bid,
            currency => $params{new_currency},
        };

        for my $bid_ret_row (@{ $old_pids_retargetings->{$old_pid} }) {

            die "new ret_cond_id not found" unless exists $new_ret_cond_ids->{ $bid_ret_row->{ret_cond_id} };

            my $one_bids_retargeting_cond = {
                bid => $new_bid,    # TODO adgroup: удалить
                ret_cond_id => $new_ret_cond_ids->{ $bid_ret_row->{ret_cond_id} },
                price_context => currency_price_rounding($bid_ret_row->{price_context} * ($params{price_convert_rate} // 1), $params{new_currency}),
                autobudgetPriority => $bid_ret_row->{autobudgetPriority},
                (is_suspended => $params{copy_suspended_status} ? $bid_ret_row->{is_suspended} : 0 ),
            };

            push @{$new_banner->{retargetings}}, $one_bids_retargeting_cond;
        }

        Retargeting::update_group_retargetings($new_banner, insert_only => 1);
    }

    return { old_pid2ret_cond_ids => $old_pid2ret_cond_ids, ret_cond_ids_old2new => $ret_cond_ids_old2new };
}

# --------------------------------------------------------------------

sub _normalize_retargeting_name {
    my ($name) = @_;

    $name = smartstrip2(lc($name));
    $name =~ s/ё/е/g;

    return $name;
}

=head2 copy_retargetings_between_clients

Копирование условий между клиентами

    Retargeting::copy_retargetings_between_clients(
        old_client_id => $ClientID1,
        new_client_id => $ClientID2,
        old_ret_cond_ids => [123, 456, 789, ...], # ret_cond_id условий которые нужно скопировать
    );

=cut

sub copy_retargetings_between_clients {
    my %params = @_;
    my ($old_client_id, $new_client_id, $old_ret_cond_ids) =
       map {delete $params{$_} || die 'not enough param'}
       qw/old_client_id new_client_id old_ret_cond_ids/;
    die 'old_ret_cond_ids is wrong' unless ref($old_ret_cond_ids) eq 'ARRAY' && @$old_ret_cond_ids;

    return {map {$_ => $_} @$old_ret_cond_ids} if $old_client_id == $new_client_id;

    # не копируем одно и то же условие больше чем 1 раз
    $old_ret_cond_ids = [uniq @$old_ret_cond_ids];

    my $old_retargetings = Retargeting::get_retargeting_conditions(ClientID => $old_client_id, type => [ qw/retargeting interest/ ]);
    my $new_retargetings = Retargeting::get_retargeting_conditions(ClientID => $new_client_id, type => [ qw/retargeting interest/ ]);

    my $new_retargetings_by_json = {
        map {
            to_json($new_retargetings->{$_}->{condition}, {canonical => 1}) => $new_retargetings->{$_}
        } keys %$new_retargetings
    };
    my $new_retargetings_by_name = {
        map {
            _normalize_retargeting_name($new_retargetings->{$_}->{condition_name}) => undef
        } keys %$new_retargetings
    };

    my $result = {};
    for my $old_ret_cond_id (@$old_ret_cond_ids) {

        my $old_cond_json = to_json($old_retargetings->{$old_ret_cond_id}->{condition}, {canonical => 1});
        my $new_ret_cond_id;

        if (exists $new_retargetings_by_json->{$old_cond_json}) {
            $new_ret_cond_id = $new_retargetings_by_json->{$old_cond_json}->{ret_cond_id};
        } else {

            my $new_cond_name = $old_retargetings->{ $old_ret_cond_id }->{condition_name};
            my $is_interest = $old_retargetings->{ $old_ret_cond_id }->{properties} =~ 'interest';

            unless ($is_interest) {
                my $i = 0;
                while (++$i <= 10 && exists $new_retargetings_by_name->{ _normalize_retargeting_name($new_cond_name) }) {
                    $new_cond_name = "$new_cond_name " . iget("(копия)");
                }

                die "copy_retargetings_between_clients failed for $old_client_id => $new_client_id: $new_cond_name"
                    if exists $new_retargetings_by_name->{ _normalize_retargeting_name($new_cond_name) };
            }

            $new_ret_cond_id = Retargeting::save_retargeting_condition(
                ClientID => $new_client_id,
                {
                    condition      => $old_retargetings->{ $old_ret_cond_id }->{condition},
                    condition_name => $new_cond_name,
                    condition_desc => $old_retargetings->{ $old_ret_cond_id }->{condition_desc},
                    ($is_interest ? (properties => $old_retargetings->{ $old_ret_cond_id }->{properties}) : ()),
                }
            );
        }
        $result->{$old_ret_cond_id} = $new_ret_cond_id;
    }

    return $result;
}

sub _copy_interests_retargetings {
    my ($client_id, $old_ret_cond_ids) = @_;
    return {} unless @$old_ret_cond_ids;

    my $old_retargetings = Retargeting::get_retargeting_conditions(
        ClientID => $client_id,
        type => [ qw/retargeting/ ],
        ret_cond_id => $old_ret_cond_ids,
        );

    my $result = {};
    for my $old_ret_cond_id (@$old_ret_cond_ids) {
        my $old_retargeting = $old_retargetings->{ $old_ret_cond_id };
        next unless $old_retargeting
            && defined $old_retargeting->{retargeting_conditions_type}
            && $old_retargeting->{retargeting_conditions_type} eq 'interests';

        my $new_ret_cond_id = Retargeting::save_retargeting_condition(
            ClientID => $client_id,
            {
                condition      => $old_retargeting->{condition},
                condition_name => $old_retargeting->{condition_name},
                condition_desc => $old_retargeting->{condition_desc},
                properties => $old_retargeting->{properties},
                retargeting_conditions_type => $old_retargeting->{retargeting_conditions_type},
            }
        );

        $result->{$old_ret_cond_id} = $new_ret_cond_id;
    }

    return $result;
}

# --------------------------------------------------------------------

=head2 retargeting_condition_json_to_bs(\$condition_json, $for_internal_campaign)

    Преобразование структуры условия ретаргетинга в формат строки для БК
    Обертка над retargeting_condition_to_bs

    $phrase->{Expression} = Retargeting::retargeting_condition_json_to_bs(\$row->{phrase_condition_json});

    Параметры:
        $condition_json_ref - ссылка на строку с json-структурой условия ретаргетинга
        $for_internal_campaign - условие ретаргетинга конвертируется в контексе внутренних кампаний
        $project_param_conditions - условия таргетирования на параметры проектов
    Результат:
        $bs_retargeting - строка с условием ретаргетинга в формате для БК

=cut
sub retargeting_condition_json_to_bs($;$;$) {
    my ($condition_json_ref, $for_internal_campaign, $project_param_conditions) = @_;
    return retargeting_condition_to_bs(from_json($$condition_json_ref), $for_internal_campaign, $project_param_conditions);
}

=head2 retargeting_condition_to_bs

Преобразование структуры условия ретаргетинга в формат строки для БК

    $bs_retargeting = retargeting_condition_to_bs($retargeting_condition);

    perl -ME -MRetargeting -e 'my $cond = Retargeting::get_retargeting_conditions(ClientID => 15155); p Retargeting::retargeting_condition_to_bs($cond->{3}->{condition})'

    https://jira.yandex-team.ru/browse/YABS-23973
    https://jira.yandex-team.ru/browse/QABS-1165

=cut

sub retargeting_condition_to_bs($;$;$) {
    my ($retargeting_condition_param, $for_internal_campaign, $project_param_conditions) = @_;

    my $retargeting_condition = [grep { defined $_->{goals} && @{ $_->{goals} } } @$retargeting_condition_param];

    my ($metrika_rules, $crypta_rules) = part { Primitives::is_crypta_goal($_->{goals}->[0]) } @$retargeting_condition;

    my $bs_retargeting = join("&",
        map { convert_rule_to_bs($_) }
            (@{$metrika_rules // []}, @{$crypta_rules // []}) ## attention, order matters: metrika goes first
    );

    # Для кампаний внутренней рекламы не нужно выполнять замену сегментов дохода
    if (!defined $for_internal_campaign || !$for_internal_campaign) {
        # В первом этапе охватного продукта в интерфейсе будет всего 4е дохода.
        # Но в транспорте к БК нужно отдавать именно Средний доход как два значения.
        # https://wiki.yandex-team.ru/users/aliho/projects/direct/crypta/#diapazondljaobshhixpol/vozrast/doxod
        $bs_retargeting =~ s/1:0\@618/1:0\@618|2:0\@618/;
    }
    # В рамках охватного продукта пользователь на группе объявления может задать ограничение по ГЕО
    # и при этом не задать ограничение по крипте+-метрике (те ничего из Аудиторных условий).
    # В этом случае в транспорт необходимо будет отправить: спец ID на спец кейворде  1542:0@1
    if (!defined $bs_retargeting || $bs_retargeting eq '') {
        $bs_retargeting = '1542:0@1';
    }

    if (defined $project_param_conditions && @$project_param_conditions) {
        my @disjunctions = xflatten map {@{$_}} @$project_param_conditions;
        my @rules;
        for my $disjunction (@disjunctions) {
            my $conjunctions = $disjunction->{bbKeywords};
            my @goals = map {+{bb_keyword => $_->{keyword}, bb_keyword_value => $_->{value}, goal_id => $_->{value}}} @$conjunctions;
            push @rules, {goals => \@goals, type => 'all'};
        }
        my $project_param_expression = "(" . join("|", map {convert_rule_to_bs($_)} @rules ) . ")";
        $bs_retargeting = $bs_retargeting . "&" . $project_param_expression;
    }

    return $bs_retargeting;
}

=head2 convert_rule_to_bs($rule)

    Преобразование правила условия ретаргетинга в формат строки для БК

    Параметры:
        $rule -- структура, описывающая правило условия ретаргетинга с целями

    Результат:
        строка выражения для БК, описывающая правило

=cut

sub convert_rule_to_bs {
    my ($rule) = @_;

    my $goals = $rule->{goals};
    my $rule_type = $rule->{type} || "";
    my $rule_interest_type = $rule->{interest_type};

    if (any {defined $_->{crypta_goal_type} && $_->{crypta_goal_type} eq 'interests'} @$goals) {
        die 'interest_type is required for conditions containing crypta interests' unless defined $rule_interest_type;
    }

    my $rule_bs;

    my %goal_union_groups = map { $_->{goal_id} => [$_] } grep { !exists $_->{union_with_id} } @$goals;

    my @goals_with_union = grep { exists $_->{union_with_id} } @$goals;
    foreach my $goal_with_union (@goals_with_union) {
        die "goal with id=union_with_id is not found in rule" unless exists $goal_union_groups{$goal_with_union->{union_with_id}};
        push $goal_union_groups{$goal_with_union->{union_with_id}}, $goal_with_union;
    }

    my @goal_groups = map { exists $goal_union_groups{$_->{goal_id}} ? $goal_union_groups{$_->{goal_id}} : () } @$goals;

    if ($rule_type eq 'all') {
        $rule_bs = join("&", map {convert_goal_group_to_bs($_, $rule_interest_type, 0)} @goal_groups);
    }
    elsif ($rule_type eq 'or') {
        $rule_bs = "(" . join("|", map {convert_goal_group_to_bs($_, $rule_interest_type, 1)} @goal_groups) . ")";
    }
    elsif ($rule_type eq 'not') {
        $rule_bs = join("&", map { "~$_"} map { convert_goal_group_to_bs($_, $rule_interest_type, 0) } @goal_groups);
    }
    else {
        die "invalid retargeting group type";
    }
    return $rule_bs;
}

=head2 convert_goal_group_to_bs($goal_group, $rule_interest_type)

    Преобразование группы целей в формат строки для БК
    Группа целей - группа, состоящая из целей, которые объединяются по ИЛИ

    Параметры:
        $goal_group -- структура, описывающая группу целей, которые объединяются по ИЛИ
        $rule_interest_type -- значение свойства interest_type на правиле, где содержится группа целей $goal_group,
                               используется в convert_goal_to_bs
        $skip_parentheses -- если true, то не выставляем круглые скобки ни при каких обстоятельствах

    Результат:
        преобразованная в формат БК группа целей в виде строки

=cut

sub convert_goal_group_to_bs {
    my ($goal_group, $rule_interest_type, $skip_parentheses) = @_;

    my @converted_goals = xflatten map { convert_goal_to_bs($_, $rule_interest_type) } @$goal_group;
    my $converted_goal_group = join("|", @converted_goals);

    # в подавляющем большинстве случаев группа состоит из одного элемента, поэтому ставим скобки, только если нужно
    $converted_goal_group = "(" . $converted_goal_group . ")" if scalar(@converted_goals) > 1 && !$skip_parentheses;

    return $converted_goal_group;
}

=head2 convert_goal_to_bs($goal, $rule_interest_type)

    Преобразование цели в формат строки для БК

    Параметры:
        $goal -- структура, описывающая цель
        $rule_interest_type -- значение свойства interest_type на правиле, где содержится цель $goal

    Результат:
        список преобразованных в формат БК целей в виде строк -- с целью типа interest может выдавать
            две результирующие строки для короткого и длинного интереса

=cut

sub convert_goal_to_bs {
    my ($goal, $rule_interest_type) = @_;
    if (defined $goal->{crypta_goal_type} && $goal->{crypta_goal_type} eq 'interests' && defined $rule_interest_type) {
        my $long_term_expr = $rule_interest_type =~ /all|long-term/ && $goal->{interest_type} =~ /all|long-term/
            ? "$goal->{bb_keyword_value}:0\@$goal->{bb_keyword}" : undef;
        my $short_term_expr = $rule_interest_type =~ /all|short-term/ && $goal->{interest_type} =~ /all|short-term/
            ? "$goal->{bb_keyword_value_short}:0\@$goal->{bb_keyword_short}" : undef;
        return grep {defined $_} ($long_term_expr, $short_term_expr);
    } elsif (defined $goal->{bb_keyword}) {
        return "$goal->{bb_keyword_value}:0\@$goal->{bb_keyword}";
    } else {
        return convert_goal_to_bs_by_id($goal);
    }
}

=head2 convert_goal_to_bs_by_id($goal)

    Преобразование идентификатора цели в формат строки для БК. Используется для метричных целей, хостов и cdp_segment

    Параметры:
        $goal -- структура, описывающая цель

    Результат:
        строка выражения для БК, описывающая цель

=cut

sub convert_goal_to_bs_by_id {
    my ($goal) = @_;
    my $goal_type = Primitives::get_goal_type_by_goal_id($goal->{goal_id});
    my $goal_time;
    if ($goal_type eq 'cdp_segment') {
        $goal_time = '0';
    } elsif ($goal_type eq 'host') {
        $goal_time = '540';
    } else {
        $goal_time = $goal->{time};
    }
    return "$goal->{goal_id}:$goal_time".(get_goal_bs_keyword($goal_type));
}

=head2 get_goal_bs_keyword($goal_type)

    Возвращает bb_keyword по типу цел

    Параметры:
        $goal_type -- тип цель

    Результат:
        keyword для БК

=cut

use constant BS_KEYWORD_BY_GOAL_TYPE => {
    cdp_segment => '@1018',
    host        => '@1161'
};

sub get_goal_bs_keyword {
    my ($goal_type) = @_;
    return exists BS_KEYWORD_BY_GOAL_TYPE->{$goal_type} ? BS_KEYWORD_BY_GOAL_TYPE->{$goal_type} : '';
}


# --------------------------------------------------------------------

=head2 _get_exist_ret_cond_in_text($ret_cond_ids)

Возвращаем все ret_cond_id используемые в текстовых и РМП кампаниям

Параметры:
    $ret_cond_ids - id условия ретаргетинга [ret_cond_id1, ret_cond_id2, ...]

Результат:
    $ret_cond_ids - массив [] id условий ретаргетинга используемых в смарт-баннерах

=cut

sub _get_exist_ret_cond_in_text {
    my ($ret_cond_ids) = @_;
    return get_one_column_sql(PPC(ret_cond_id => $ret_cond_ids),
        ['SELECT DISTINCT br.ret_cond_id
            FROM bids_retargeting br
            JOIN phrases p on p.pid = br.pid
            JOIN campaigns c on c.cid = p.cid',
          WHERE => {ret_cond_id => SHARD_IDS, 'c.statusEmpty' => 'No'}]);
}

=head2 _get_exist_ret_cond_in_media_banners($ret_cond_ids)

Возвращаем все ret_cond_id используемые в медиаплана

Параметры:
    $ret_cond_ids - id условия ретаргетинга [ret_cond_id1, ret_cond_id2, ...]

Результат:
    $ret_cond_ids - массив [] id условий ретаргетинга используемых в смарт-баннерах

=cut

sub _get_exist_ret_cond_in_media_banners {
    my ($ret_cond_ids) = @_;
    return get_one_column_sql(PPC(ret_cond_id => $ret_cond_ids),
        ['SELECT DISTINCT ret_cond_id
            FROM mediaplan_bids_retargeting
            JOIN mediaplan_banners mb USING(mbid)
            JOIN campaigns c USING(cid)',
          WHERE => {ret_cond_id => SHARD_IDS, 'c.statusEmpty' => 'No'}]);
}

=head2 _get_exist_ret_cond_in_retargeting_multiplier($ret_cond_ids)

Возвращаем все ret_cond_id используемые в коэффициентах для групп

Параметры:
    $ret_cond_ids - id условия ретаргетинга [ret_cond_id1, ret_cond_id2, ...]
    %O - именованные опции
        cid => [qw/111 222/] - фильтровать коэффициенты по привязке к группам в заданных кампаниях

Результат:
    $ret_cond_ids - массив [] id условий ретаргетинга используемых в коэффициентах для групп
=cut

sub _get_exist_ret_cond_in_retargeting_multiplier {
    my ($ret_cond_ids, %O) = @_;
    return get_one_column_sql(PPC(ret_cond_id => $ret_cond_ids),
        ['SELECT DISTINCT ret_cond_id
          FROM retargeting_multiplier_values r
          JOIN hierarchical_multipliers h using (hierarchical_multiplier_id)
          JOIN campaigns c USING(cid)',
          WHERE => {ret_cond_id => SHARD_IDS,
                    'c.statusEmpty' => 'No',
                    ($O{cid} ? (cid => $O{cid}) : ())}]);
}

=head2 _get_exist_ret_cond_in_performance($ret_cond_ids)

Возвращаем все ret_cond_id используемые в смарт-баннерах

Параметры:
    $ret_cond_ids - id условия ретаргетинга [ret_cond_id1, ret_cond_id2, ...]

Результат:
    $ret_cond_ids - массив [] id условий ретаргетинга используемых в смарт-баннерах

=cut

sub _get_exist_ret_cond_in_performance {
    my ($ret_cond_ids) = @_;
    return get_one_column_sql(PPC(ret_cond_id => $ret_cond_ids),
            ['SELECT DISTINCT bp.ret_cond_id
                FROM bids_performance bp
                JOIN phrases p USING(pid)
                JOIN campaigns c USING(cid)',
                WHERE => {'bp.ret_cond_id__int' => SHARD_IDS, 'c.statusEmpty' => 'No'}])
}

=head2 get_used_ret_cond

Находит те условия из списка, которые используются хоть где-нибудь в системе.

    my $used = Retargeting::get_used_ret_cond([$ret_cond_id_1, $ret_cond_id_2, ...]);

где

    $used = { $ret_cond_id_1 => 1, ... };

=cut

sub get_used_ret_cond {
    my ($ret_cond_ids) = @_;

    my $used = {};
    my $text = _get_exist_ret_cond_in_text($ret_cond_ids);
    my $mediabanners = _get_exist_ret_cond_in_media_banners($ret_cond_ids);
    my $multiplier = _get_exist_ret_cond_in_retargeting_multiplier($ret_cond_ids);
    my $performance = _get_exist_ret_cond_in_performance($ret_cond_ids);
    for (@$text, @$mediabanners, @$multiplier, @$performance) {
        $used->{$_} = 1;
    }
    return $used;
}

=head2 find_camps_used_ret_cond($ClientID)

Кампании клиента использующие условия ретаргетинга

Результат:
    {ret_cond_id => [
    {cid => 1, name => ....},
    {cid => 2, name => ....}
    ]}

=cut

sub find_camps_used_ret_cond {
    my ($ClientID) = @_;
    my %ret_conds;
    my $camps = get_all_sql(PPC(ClientID => $ClientID),
                ['SELECT DISTINCT rc.ret_cond_id, p.cid, c.name
                    FROM retargeting_conditions rc
                    JOIN bids_retargeting br on br.ret_cond_id = rc.ret_cond_id
                    JOIN phrases p on p.pid = br.pid
                    JOIN campaigns c on c.cid = p.cid',
                    WHERE => {'rc.ClientID' => $ClientID, 'c.statusEmpty' => 'No'}]);
    for my $row(@$camps) {
        my %camp_name = ('cid' => $row->{cid}, 'name' => $row->{name});
        push @{$ret_conds{$row->{ret_cond_id}}}, \%camp_name;
    }
    $camps = get_all_sql(PPC(ClientID => $ClientID),
        ['SELECT DISTINCT rc.ret_cond_id, c.cid, c.name
                    FROM retargeting_conditions rc
                    JOIN retargeting_multiplier_values rmv USING(ret_cond_id)
                    JOIN hierarchical_multipliers hm USING(hierarchical_multiplier_id)
                    JOIN campaigns c USING(cid)',
            WHERE => {'rc.ClientID' => $ClientID, 'c.statusEmpty' => 'No'}]);
        for my $row(@$camps) {
            my %camp_name = ('cid' => $row->{cid}, 'name' => $row->{name});
            push @{$ret_conds{$row->{ret_cond_id}}}, \%camp_name;
        }
    $camps = get_all_sql(PPC(ClientID => $ClientID),
                ['SELECT DISTINCT rc.ret_cond_id, p.cid, c.name
                    FROM retargeting_conditions rc
                    JOIN bids_performance bp USING(ret_cond_id)
                    JOIN phrases p USING(pid)
                    JOIN campaigns c USING(cid)',
                    WHERE => {
                        'rc.ClientID' => $ClientID,
                        'c.statusEmpty' => 'No',
                        'bp.is_deleted' => 0
                    }
    ]);
        for my $row(@$camps) {
            my %camp_name = ('cid' => $row->{cid}, 'name' => $row->{name});
            push @{$ret_conds{$row->{ret_cond_id}}}, \%camp_name;
        }
    return \%ret_conds;
}

# --------------------------------------------------------------------

=head2 get_all_goals_by_uid

Выдаем для логина все доступные цели для ретаргетинга
за целями онлайново ходим в метрику

=cut
sub get_all_goals_by_uid($$) {
    my ($client_uids, $retargeting_conditions) = @_;

    my $metrika_goals = get_metrika_goals_by_uid($client_uids);
    my $metrika_goals_for_uids = [xuniq {$_->{goal_id}} map {@$_} values %$metrika_goals];
    my $exists_goals_ids = {map {$_->{goal_id} => 1} @$metrika_goals_for_uids};

    my %goals_in_ret;
    for my $goals_row (map {$_->{condition}} @$retargeting_conditions) {
        for my $goal_id (map {$_->{goal_id}} map {@{$_->{goals}}} @$goals_row) {
            next if $exists_goals_ids->{$goal_id};
            $goals_in_ret{$goal_id} = 1;
        }
    }

    my $goals_without_access = CampaignTools::get_metrika_goals(where => {goal_id => [keys %goals_in_ret]});

    for my $goal_id (keys %goals_in_ret) {
        push @$metrika_goals_for_uids, {
                goal_id => $goal_id,
                goal_name => $goals_without_access->{$goal_id}->{name},
                goal_domain => '',
                allow_to_use => 0,
                goal_type => Primitives::get_goal_type_by_goal_id($goal_id),
            };
    }

    return $metrika_goals_for_uids;
}

1;

