package Balance;

# $Id$

=head1 NAME

    Sandbox::Balance

=head1 DESCRIPTION

    Эмуляция блэкбокса для песочницы

    XMLRPC::Transport::HTTP::Apache хочет, чтобы модуль и пакет назывались именно Balance

=cut

use Direct::Modern;

use JSON;
use List::MoreUtils qw/uniq/;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HTTP qw/http_fetch/;
use Yandex::TVM2;
use TextTools;
use Client;
use Stat::Const;
use Campaign;
use Currency::Rate;
use PrimitivesIds;
use geo_regions ();

use Time::HiRes qw/time/;

my $FAKE_CONTRACT = '11111/00';

sub FindClient {
    my ($self, $params) = @_;

    my $where = {};

    if ($params->{PassportID}) {
        $where = {'r.uid' => $params->{PassportID}};
    } elsif ($params->{ClientID}) {
        $where = {'c.ClientID' => $params->{ClientID}};
    } elsif ($params->{Login}) {
        $where = {'r.login' => $params->{Login}};
    }

    my $clients = get_all_sql(FAKEBALANCE, ['select 
                                                c.ClientID as CLIENT_ID, 
                                                c.ClientTypeID as CLIENT_TYPE_ID, 
                                                c.name as NAME,
                                                c.email as EMAIL,
                                                c.phone as PHONE,
                                                c.fax as FAX,
                                                c.url as URL,
                                                IF(c.type = "agency", 1, 0) as IS_AGENCY,
                                                c.agency_clientid as AGENCY_ID
                                              from clients c left join reps r using (ClientID)', where => $where ]);

    foreach (@$clients) {
        if (!$_->{AGENCY_ID}) {
            delete $_->{AGENCY_ID};
        }
    }

    return (0,0,$clients);
}

sub CreateClient {
    my ($self, $opearator_uid, $params) = @_;

    my $res;

    if ($params->{CLIENT_ID}) {
        do_update_table(FAKEBALANCE, 'clients', 
                                {
                                    ClientTypeID => $params->{CLIENT_TYPE_ID},
                                    name => $params->{NAME},
                                    email => $params->{EMAIL},
                                    phone => $params->{PHONE},
                                    fax => $params->{FAX},
                                    url => $params->{URL},
                                    city => $params->{CITY},
                                    type => $params->{IS_AGENCY} ? 'agency' : 'client',
                                    agency_clientid => $params->{AGENCY_ID} || 0
                                }, where => {ClientID => $params->{CLIENT_ID}});
        $res = $params->{CLIENT_ID};
    } else {
        $res = do_insert_into_table(FAKEBALANCE, 'clients', 
                                {
                                    ClientTypeID => $params->{CLIENT_TYPE_ID},
                                    name => $params->{NAME},
                                    email => $params->{EMAIL},
                                    phone => $params->{PHONE},
                                    fax => $params->{FAX},
                                    url => $params->{URL},
                                    city => $params->{CITY},
                                    type => $params->{IS_AGENCY} ? 'agency' : 'client',
                                    agency_clientid => $params->{AGENCY_ID} || 0
                                });
    }

    return (0,0, $res+0);
}

sub ListClientPassports {
    my ($self, $opearator_uid, $client_id) = @_;

    my $res = get_all_sql(FAKEBALANCE, 'select uid as Uid, login as Login, name as Name,
                                               ClientID as ClientId, isMain as IsMain
                                          from reps where ClientID = ?', $client_id);
    return $res;
}

sub CreateUserClientAssociation {
    my ($self, $opearator_uid, $new_clid, $uid) = @_;

    my $cur_clid = get_one_field_sql(FAKEBALANCE, 'select ClientID from reps where uid = ?', $uid);

    if ($cur_clid && $cur_clid == $new_clid) {
        return (4006, 0);
    } elsif ($cur_clid && $cur_clid != $new_clid) {
        return (4008, 0);
    } elsif (! get_one_field_sql(FAKEBALANCE, 'select ClientID from clients where ClientID = ?', $new_clid)) {
        return (1003, 0);
    } else {
        my $user_info = get_one_line_sql(FAKEBLACKBOX, [ 'select `account_info.fio.uid` as fio, login from users', where => {uid => $uid} ]);

        do_insert_into_table(FAKEBALANCE, 'reps', {uid => $uid, ClientID => $new_clid, name => $user_info->{fio}, login => $user_info->{login}});

        if (get_one_field_sql(FAKEBALANCE, 'select count(*) from reps where ClientID = ? and uid = ?', $new_clid, $uid)) {
            return (0, 0);
        } else {
            return (4, 0);
        }
    }
}

sub RemoveUserClientAssociation {
    my ($self, $opearator_uid, $uid, $clid) = @_;

    return do_sql(FAKEBALANCE, [ 'delete from reps', where => {uid => $uid, ClientID => $clid} ]);
}

sub EditPassport {
    my ($self, $opearator_uid, $uid, $props) = @_;

    if ($props->{IsMain}) {
        do_update_table(FAKEBALANCE, 'reps', {isMain => $props->{IsMain}}, where => {uid => $uid});
    }

    if ($props->{ClientId}) {
        if ($props->{ClientId} == -1) {
            do_sql(FAKEBALANCE, 'delete from reps where uid = ?', $uid);
        } else {
            do_update_table(FAKEBALANCE, 'reps', {ClientID => $props->{ClientId}}, where => {uid => $uid});
        }
    }

    return GetPassportByUid($self, $opearator_uid, $uid);
}

sub GetPassportByUid {
    my ($self, $opearator_uid, $uid) = @_;

    my $res = get_one_line_sql(FAKEBALANCE, 'select uid as Uid, login as Login,  name as Name,
                                                    ClientID as ClientId, isMain as IsMain
                                               from reps where uid = ?', $uid);
    return $res;
}

sub CreateOrUpdateOrdersBatch {
    my ($self, $operator_uid, $orders) = @_;
    return [[0, 'ok']];
}

sub GetClientCreditLimits {
    return {
        LIMITS => [
            {
                CONTRACT_ID => 0,
                CONTRACT_EID => $FAKE_CONTRACT,
                LIMIT_SPENT => 0,
                LIMIT_TOTAL => 10000
            }],
        CURRENCY => 'RUR',
        PRODUCT_PRICE => 1
    };
}

sub CreateRequest {
    my ($self, $operator_uid, $client_id, $data) = @_;

    my $request_id = do_insert_into_table(FAKEBALANCE, 'requests', {ClientID => $client_id});

    if ($request_id) {
        foreach my $payment (@$data) {
            do_insert_into_table(FAKEBALANCE, 'payments', {request_id => $request_id, OrderID => $payment->{ServiceOrderID}, sum => $payment->{Qty}});
        }
        my $billing_url = 'https://passport.yandex.ru/passport?mode=subscribe&from=balance&retpath=http%3A%2F%2Fbalance.yandex.ru%2Fpaypreview.xml%3Frequest_id%3D'.$request_id.'%26ref_service_id%3D7%26ui_type%3Dstd';

        return 0, 'ok', $billing_url;
    }

    return;
}

sub CreateInvoice {
    my ($self, $opearator_uid, $data) = @_;

    my $payments = get_all_sql(FAKEBALANCE, 'select OrderID, sum from payments where request_id = ?', $data->{RequestID});
    my @cids = map { $_->{OrderID} } @$payments;
    my $camps = get_camp_info(\@cids, undef, short => 1);
    my %cid2camp = map { $_->{cid} => $_ } @$camps;
    my $tid = int(time() * 997);

    foreach my $payment (@$payments) {
        my $cid = $payment->{OrderID};
        my $camp = $cid2camp{$cid};
        my $new_sum = round2s($camp->{sum} + $payment->{sum});
        my $currency = $camp->{currency};
        my %notification_data = (
            ServiceID => $Settings::SERVICEID{direct},
            ServiceOrderID => $payment->{OrderID},
            Tid => $tid,
            Signal => 0,
            SignalDescription => 0,
            ConsumeQty => $new_sum,
            ConsumeAmount => $new_sum,
            ConsumeMoneyQty => ($currency eq 'YND_FIXED') ? convert_currency($new_sum, $currency, 'RUB', with_nds => 1) : $new_sum,
            _get_wallet_total_sum_record_for_notification(($camp->{type} eq 'wallet' ? $cid : undef), $cid, $currency, $new_sum),
            ConsumeCurrency => $currency,
            ProductCurrency => $currency,
            Certificate => 0,
        );
        my $ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";
        # если ошибка - интапи вернет не 200х код и http_fetch упадет
        my $content = http_fetch(POST => $Settings::NOTIFY_ORDER_URL
                                 , to_json([\%notification_data])
                                 , timeout => 20
                                 , headers => { 'Content-type' => 'application/json', 'X-Ya-Service-Ticket' => $ticket }
                                 );
    }

    return 1;
}

sub CreateTransferMultiple {
    my ($self, $opearator_uid, $from, $to) = @_;

    my $tid = int(time() * 997);

    my @cids = map { $_->{ServiceOrderID} } (@$from, @$to);
    my $camps = get_camp_info(\@cids, undef, short => 1);
    my %cid2camp = map { $_->{cid} => $_ } @$camps;

    my @currencies = uniq map { $_->{currency} } @$camps;
    die 'multiple currencies found: ' . join(', ', @currencies) if scalar(@currencies) > 1;

    foreach my $payment (@$from) {
        my $cid = $payment->{ServiceOrderID};
        my $camp = $cid2camp{$cid};
        my $new_sum = round2s($payment->{QtyNew});
        my $currency = $camp->{currency};
        my %notification_data = (
            ServiceID => $Settings::SERVICEID{direct},
            ServiceOrderID => $cid,
            Tid => $tid,
            Signal => 0,
            SignalDescription => 0,
            ConsumeQty => $new_sum,
            ConsumeAmount => $new_sum,
            ConsumeMoneyQty => ($currency eq 'YND_FIXED') ? convert_currency($new_sum, $currency, 'RUB', with_nds => 1) : $new_sum,
            _get_wallet_total_sum_record_for_notification(($camp->{type} eq 'wallet' ? $cid : undef), $cid, $currency, $new_sum),
            ConsumeCurrency => $currency,
            ProductCurrency => $currency,
            Certificate => 0,
        );
        my $ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";
        # если ошибка - интапи вернет не 200х код и http_fetch упадет
        my $content = http_fetch(POST => $Settings::NOTIFY_ORDER_URL
                                 , to_json([\%notification_data])
                                 , timeout => 20
                                 , headers => { 'Content-type' => 'application/json', 'X-Ya-Service-Ticket' => $ticket }
                                 );
    }

    foreach my $payment (@$to) {
        my $cid = $payment->{ServiceOrderID};
        my $camp = $cid2camp{$cid};
        my $new_sum = round2s($camp->{sum} + $payment->{QtyDelta});
        my $currency = $camp->{currency};
        my %notification_data = (
            ServiceID => $Settings::SERVICEID{direct},
            ServiceOrderID => $cid,
            Tid => $tid,
            Signal => 0,
            SignalDescription => 0,
            ConsumeQty => $new_sum,
            ConsumeAmount => $new_sum,
            ConsumeMoneyQty => ($currency eq 'YND_FIXED') ? convert_currency($new_sum, $currency, 'RUB', with_nds => 1) : $new_sum,
            _get_wallet_total_sum_record_for_notification(($camp->{type} eq 'wallet' ? $cid : undef), $cid, $currency, $new_sum),
            ConsumeCurrency => $currency,
            ProductCurrency => $currency,
            Certificate => 0,
        );

        my $ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";
        # если ошибка - интапи вернет не 200х код и http_fetch упадет
        my $content = http_fetch(POST => $Settings::NOTIFY_ORDER_URL
                                 , to_json([\%notification_data])
                                 , timeout => 20
                                 , headers => { 'Content-type' => 'application/json', 'X-Ya-Service-Ticket' => $ticket }
                                 );
    }

    return [0, 'ok'];
}

sub GetRequestChoices {
    my ($self, $params) = @_;
    if ($params->{ContractExternalID} && $params->{ContractExternalID} ne $FAKE_CONTRACT) {
        die 'CONTRACT_NOT_FOUND';
    } else {
        return { 
                    credits => {is_available => 1},
                    pcp_list => [ { contract => { external_id => $FAKE_CONTRACT, id => 0, person_id => 0 },
                                    paysyses => [{cc => 'ur', id => 1003,
                                                  payment_method => { cc => 'bank', id => 1001, name => 'Bank Payment' }}]
                                } ],
               }
    }
}

=head2 CreateFastPayment

    Все платежи в рамках одного запроса - в одной валюте, на это есть валидация.
    Валюты кампаний, в песочнице можно считать, что совпадают с валютой клиента.
    Валюта запроса должна совпадать с валютов одной из кампаний (в версии API4Live), либо быть YND_FIXED (в версии API4), на это тоже есть валидация
    Для версии API4 фишки автоматически будут переведены в валюту клиента см.  APICommon::create_pay_campaign_request
    Следовательно оплата тоже может быть только в валюте клиента

=cut

sub CreateFastPayment {
    my ($self, $params) = @_;

    my $tid = int(time() * 997);

    my @cids = map { $_->{service_order_id} } @{$params->{items}};
    my $cid2data = get_hashes_hash_sql(PPC(cid => \@cids), ['select cid, type, wallet_cid, sum, IFNULL(currency, "YND_FIXED") AS currency from campaigns', where => { cid => SHARD_IDS }] );

    my $client_id = get_clientid(uid => $params->{passport_id});
    my $client_currencies = get_client_currencies($client_id);
    my $client_working_currency = $client_currencies->{work_currency};

    my $total_sum = 0;
    foreach my $payment (@{$params->{items}}) {
        $total_sum += round2s($payment->{qty});
    }

    my $max_payment_sum_cc;
    {
        no warnings 'once';
        $max_payment_sum_cc = convert_currency($SandboxCommon::DEFAULT_OVERDRAFT, 'YND_FIXED', $client_working_currency, with_nds => 1);
    };

    if ( !$params->{overdraft} && $total_sum > $max_payment_sum_cc ) {
        return {status_code => 2, status_desc => 'Amount too big'};
    } elsif ($params->{overdraft}) {
        my $client_nds = get_client_NDS($client_id);
        my $client_discount = get_client_discount($client_id);

        my $overdraft_info = get_overdraft_info($client_id, client_discount => $client_discount, client_nds => $client_nds, client_currencies => $client_currencies);

        my $overdraft_rest = $overdraft_info->{overdraft_rest};
        if ($total_sum > $overdraft_rest) {
            return {status_code => 3, status_desc => 'Overdraft limit excedded', overdraft => {available_sum => $overdraft_rest}};
        }
    }

    foreach my $payment (@{$params->{items}}) {
        my $camp = $cid2data->{$payment->{service_order_id}};
        my $sum = round2s($camp->{sum} + $payment->{qty});
        my $currency = $camp->{currency};

        my $notify_params = {
            ServiceID => $Settings::SERVICEID{direct},
            ServiceOrderID => $payment->{service_order_id},
            Tid => $tid,
            ConsumeQty => $sum,
            ConsumeMoneyQty => convert_currency($sum, $client_working_currency, 'RUB', with_nds => 1),
            _get_wallet_total_sum_record_for_notification(($camp->{type} eq 'wallet' ? $camp->{cid} : undef), $camp->{cid}, $client_working_currency, $sum),
            ProductCurrency => $currency,
            # TODO: для сконвертированных без копирования надо уметь CompletionFixedMoneyQty и CompletionFixedQty
        };

        my $ticket = eval { Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi}) } or die "Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@";
        # если ошибка - интапи вернет не 200х код и http_fetch упадет
        my $content = http_fetch(POST => $Settings::NOTIFY_ORDER_URL
                                 , to_json([$notify_params])
                                 , timeout => 20
                                 , headers => { 'Content-type' => 'application/json', 'X-Ya-Service-Ticket' => $ticket }
                                 );
    }
    return {status_code => 0, status_desc => 'Success', invoice_id => '123', invoice_eid => 'Б-12456780–1'};
}

=head2 GetClientNDS

    Возвращает для всех существующих клиентов график НДС, в котором всегда 20%

    http://wiki.yandex-team.ru/balance/xmlrpc#balance.getclientnds

=cut

sub GetClientNDS {
    my ($self, $params) = @_;

    my $mod = $params->{Mod};
    my $rem = $params->{Rem};

    my $client_ids = get_one_column_sql(FAKEBALANCE, 'SELECT ClientID FROM clients WHERE ClientID % ? = ?', $params->{Mod}, $params->{Rem}) || [];
    my @data = map { +{CLIENT_ID => $_, DT => $BEGIN_OF_TIME_FOR_STAT, NDS_PCT => 20} } @$client_ids;
    my @fields = qw/CLIENT_ID DT NDS_PCT/;
    my @rows;
    push @rows, join("\t", @fields);
    for my $row (@data) {
        push @rows, join("\t", map {$row->{$_}} @fields);
    }
    return join("\n", @rows);
}

=head2 _get_wallet_total_sum_for_notification

    Выбирает из БД и возвращает общую сумму поступившую на кошелек и кампании под ним, в валюте кошелька
    Для кампании по которой идет нотификация подставляет новую сумму

=cut

sub _get_wallet_total_sum_record_for_notification {
    my ($wallet_cid, $notification_cid, $currency, $notification_cid_new_sum) = @_;
    
    return () unless $wallet_cid;

    my $total_sum_cur = get_one_field_sql(PPC(cid => $wallet_cid), "select sum(c.sum) 
                                                                      from campaigns wc
                                                                 left join campaigns c 
                                                                        on c.cid <> ? 
                                                                           and ((c.wallet_cid = wc.cid 
                                                                                  and IFNULL(c.currency, 'YND_FIXED') = IFNULL(wc.currency, 'YND_FIXED')
                                                                                  and c.uid = wc.uid)
                                                                                or c.cid = wc.cid)
                                                                     where wc.cid = ? and wc.currency = ?",
                                                                           $notification_cid, $wallet_cid, $currency) // 0;
    $total_sum_cur += $notification_cid_new_sum;
    return (TotalConsumeQty => $total_sum_cur);
}

=head2 GetFirmCountryCurrency

    https://wiki.yandex-team.ru/balance/xmlrpc/#balance.getfirmcountrycurrency

=cut

sub GetFirmCountryCurrency {
    my ($self, $params) = @_;

    my $client_id = $params->{client_id};

    # после изменения списка надо запускать ./protected/ppcFetchCountryCurrencies.pl в Песочнице
    my @result = (
        {
            region_id => $geo_regions::RUS,
            region_name_en => 'Russia',
            region_name => 'Россия',
            currency => 'RUB',
            resident => 1,
            firm_id => 1, # ООО Яндекс
            convert_type_modify => 1,
            agency => 0,
        },
        {
            region_id => $geo_regions::UKR,
            region_name_en => 'Ukraine',
            region_name => 'Украина',
            currency => 'UAH',
            resident => 1,
            firm_id => 2, # ООО Яндекс.Украина
            convert_type_modify => 0,
            agency => 0,
        },
        {
            region_id => 126, # Швейцария
            region_name_en => 'Switzerland',
            region_name => 'Швейцария',
            currency => 'EUR',
            resident => 1,
            firm_id => 7, # Yandex Europe AG
            convert_type_modify => 0,
            agency => 0,
        },
        {
            region_id => 126, # Швейцария
            region_name_en => 'Switzerland',
            region_name => 'Швейцария',
            currency => 'USD',
            resident => 1,
            firm_id => 7, # Yandex Europe AG
            convert_type_modify => 0,
            agency => 0,
        },
        {
            region_id => 126, # Швейцария
            region_name_en => 'Switzerland',
            region_name => 'Швейцария',
            currency => 'CHF',
            resident => 1,
            firm_id => 7, # Yandex Europe AG
            convert_type_modify => 0,
            agency => 0,
        },
        {
            region_id => $geo_regions::TR,
            region_name_en => 'Turkey',
            region_name => 'Турция',
            currency => 'TRY',
            resident => 1,
            firm_id => 8, # Yandex Turkey
            convert_type_modify => 0,
            agency => 0,
        },
        {
            region_id => $geo_regions::KAZ,
            region_name_en => 'Kazakhstan',
            region_name => 'Казахстан',
            currency => 'KZT',
            resident => 1,
            firm_id => 3, # КазНет Медиа
            convert_type_modify => 0,
            agency => 0,
        },
        {
            region_id => $geo_regions::BY,
            region_name_en => 'Belarus',
            region_name => 'Беларусь',
            currency => 'BYN',
            resident => 1,
            firm_id => 1, # ООО Яндекс
            convert_type_modify => 0,
            agency => 0,
        },


        {
            region_id => $geo_regions::RUS,
            region_name_en => 'Russia',
            region_name => 'Россия',
            currency => 'RUB',
            resident => 1,
            firm_id => 1, # ООО Яндекс
            convert_type_modify => 1,
            agency => 1,
        },
        {
            region_id => $geo_regions::UKR,
            region_name_en => 'Ukraine',
            region_name => 'Украина',
            currency => 'UAH',
            resident => 1,
            firm_id => 2, # ООО Яндекс.Украина
            convert_type_modify => 0,
            agency => 1,
        },
        {
            region_id => 126, # Швейцария
            region_name_en => 'Switzerland',
            region_name => 'Швейцария',
            currency => 'EUR',
            resident => 1,
            firm_id => 7, # Yandex Europe AG
            convert_type_modify => 0,
            agency => 1,
        },
        {
            region_id => 126, # Швейцария
            region_name_en => 'Switzerland',
            region_name => 'Швейцария',
            currency => 'USD',
            resident => 1,
            firm_id => 7, # Yandex Europe AG
            convert_type_modify => 0,
            agency => 1,
        },
        {
            region_id => 126, # Швейцария
            region_name_en => 'Switzerland',
            region_name => 'Швейцария',
            currency => 'CHF',
            resident => 1,
            firm_id => 7, # Yandex Europe AG
            convert_type_modify => 0,
            agency => 1,
        },
        {
            region_id => $geo_regions::TR,
            region_name_en => 'Turkey',
            region_name => 'Турция',
            currency => 'TRY',
            resident => 1,
            firm_id => 8, # Yandex Turkey
            convert_type_modify => 0,
            agency => 1,
        },
        {
            region_id => $geo_regions::KAZ,
            region_name_en => 'Kazakhstan',
            region_name => 'Казахстан',
            currency => 'KZT',
            resident => 1,
            firm_id => 3, # КазНет Медиа
            convert_type_modify => 0,
            agency => 1,
        },
        {
            region_id => $geo_regions::BY,
            region_name_en => 'Belarus',
            region_name => 'Беларусь',
            currency => 'BYN',
            resident => 1,
            firm_id => 1, # ООО Яндекс
            convert_type_modify => 0,
            agency => 1,
        },
    );

    if ($client_id) {
        my $currency = get_client_currencies($client_id)->{work_currency};
        if ($currency && $currency ne 'YND_FIXED') {
            @result = grep { $_->{currency} eq $currency } @result;
        }
    }

    return [0, 'SUCCESS', \@result];
}

=head2 GetOrdersInfo

    Стаб метода, нужен чтобы не ломалось рассервисирование в балансе при удалении агентстких кампаний

=cut

sub GetOrdersInfo {
    return [];
}

1;
