package Sandbox::SandboxService;

# $Id$

=head1 NAME

    Sandbox::SandboxService

=head1 DESCRIPTION

    Сервис для управления песочницей из Директа

    Для запуска команд этого контроллера из консоли можно использовать sandbox_service.pl

=cut

use Direct::Modern;

use Plack::Request;

use Settings;

use Rbac;
use RBAC2::Extended;
use RBACElementary qw/rbac_who_is/;
use PrimitivesIds qw/get_clientid/;
use Client;
use Yandex::HashUtils qw/hash_cut/;

use SandboxCommon;
use APICommon qw/drop_finance_ops_counter/;
use API::App::Access;

=head2 new

=cut

sub new {
    bless {};
}

sub _error_invalid_user_uid {
    return {code => 'ERROR', description => 'Invalid user uid'}
}

=head2 save_app_access($request)

    $request - заявка на доступ приложению

=cut

sub save_app_access {
    my ($self, $request) = @_;

    my $app_access_info = hash_cut($request->{data}, qw/application_id access_type manager_login comment check_time sync_time min_api_version_override/);

    if (API::App::Access::save_app_access_in_sandbox($app_access_info)) {
        return {code => 'OK'};
    } else {
        return {code => 'ERROR'};
    }
}

=head2 drop_fin_ops_counter($request)

    $request - хэш с полями uid (users.id)

    Сбрасывает счетчик фин. операций пользователя по $uid в песочнице

=cut

sub drop_fin_ops_counter {
    my ($self, $request) = @_;
    my $uid = $request->{uid};

    if ($uid && $uid =~ /^\d+$/) {
        if (drop_finance_ops_counter($uid)) {
            return {code => 'OK'};
        }
        else {
            return {code => 'ERROR'};
        }
    }

    return _error_invalid_user_uid();
}

=head2 drop_user($request)

    $request - хэш с полями uid (users.id)

    Удаляет данные пользователя по $uid в песочнице

=cut

sub drop_user {
    my ($self, $request) = @_;
    my $uid = $request->{uid};

    my $rbac = $self->_rbac();
    if ($uid && $uid =~ /^\d+$/) {
        if (drop_sandbox_user($rbac, $uid)) {
            return {code => 'OK'};
        }
        else {
            return {code => 'ERROR'};
        }
    }

    return _error_invalid_user_uid();
}

=head2 get_current_state($request)

    $request - хэш с полями uid (users.id)

    Возвращает статус пользователя в песочнице (существует или нет, какого типа, в какой валюте)

=cut

sub get_current_state {
    my ($self, $request) = @_;

    my $uid = $request->{uid};

    my $rbac = $self->_rbac();
    if ($uid && $uid =~ /^\d+$/) {
        return get_current_sandbox_user_state($rbac, $uid);
    }

    return _error_invalid_user_uid();
}

=head2 create_user($request)

    $request - хэш с полями currency email fio init_test_data enable_shared_account login type uid

    Создает пользователя в песочнице

=cut

sub create_user {
    my ($self, $request) = @_;

    # Кэш не расчитан на то, что одним зарпосом мы будем создавать и удалять клиента
    # А в песочнице это обычное дело
    local $Rbac::USE_CACHE = 0;

    my $rbac = $self->_rbac();
    my $params = hash_cut($request, qw/currency email fio init_test_data enable_shared_account login type uid/);

    # костыль для кнопок "очистить песочницу" и "создать тестовые данные"
    # при нажатии на которые интерфейс не передает валюту, ее надо взять из БД
    if(($params->{currency}//'') eq 'YND_FIXED') {
        if (my $client_id = get_clientid(uid => $params->{uid})) {
            # запоминаем параметры клиента
            $params->{currency} = get_client_currencies($client_id)->{work_currency};
        } else {
            return {code => 'ERROR', descr => qq~user uid: $params->{uid} doesn't exists in sandbox, while YDN_FIXED currency specified~};
        }
    }

    if (my $validate_error = validate_create_sandbox_user($params)) {
        return {code => 'ERROR', descr => $validate_error};
    }

    my $role = rbac_who_is($rbac, $params->{uid});

    drop_sandbox_user($rbac, $params->{uid});

    if ($params->{enable_shared_account} || ($params->{currency} && $params->{currency} ne 'YND_FIXED')) {
        # проставляем галочку, если пользователь смог прислать форму без неё
        $params->{init_test_data} = 1;
    }

    my $create_user_result = create_sandbox_user($rbac, $params);

    if ($create_user_result) {
        return {code => 'ERROR', descr => $create_user_result};
    } else {
        if ($params->{init_test_data}) {
            $SandboxCommon::plack_request = Plack::Request->new({ REMOTE_ADDR => $self->_remote_addr });
            my $create_test_data_result = create_test_data($rbac, $params);
        }
        return {code => 'OK'};
    }
}

sub _rbac { RBAC2::Extended->get_singleton(1) }

sub _remote_addr { $ENV{REMOTE_ADDR} || '127.0.0.1' }

1;
