package SandboxClient;

#  $Id$

=head1 NAME

    SandboxClient

=head1 DESCRIPTION

    Модуль для работы с сервисом песочницы

=cut

use Direct::Modern;

use JSON;
use JSON::RPC2::Client;

use Yandex::TVM2;
use Yandex::HTTP qw/http_fetch/;

use Settings;

use base qw/Exporter/;
our @EXPORT = qw/
    sandbox_get_user_status_request
    sandbox_create_user_request
    sandbox_drop_user_request
    sandbox_drop_finance_ops_counter_request
    sandbox_send_app_access
/;


=head2 execute_sandbox_method

    $method - название метода
    $params - параметры

    Выполнить json-rpc метод SandboxService-а

=cut

sub execute_sandbox_method {
    my ($method, $params) = @_;

    my $client = JSON::RPC2::Client->new();
    my $json_string = $client->call_named($method, %$params);
    my $ticket = Yandex::TVM2::get_ticket($Settings::SANDBOX_INTAPI_TVM2_ID);

    my $content = eval{ http_fetch('POST', $Settings::SANDBOX_SERVICE_URL, $json_string, timeout => 150, headers => {'X-Ya-Service-Ticket' => $ticket}) };
    if (!$content) {
        warn "execute_sandbox_method call failed: $@";
        return undef;
    }

    my ($failed, $result, $error) = $client->response($content);
    if (defined $failed) {
        warn "execute_sandbox_method call failed: $failed"
            . "\nContent: ". substr($content, 0, 1024);
    }
    warn "execute_sandbox_method call error: $error->{message}" if defined $error;

    return $result;
}

=head2 sandbox_get_user_status_request

    Возвращает текущий статуса пользователя в песочнице

=cut

sub sandbox_get_user_status_request
{
    my $uid = shift;

    local $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;

    my $service = Yandex::Trace::new_service('direct.sandbox', 'GetCurrentState');
    my $response = execute_sandbox_method(
        'get_current_state', { uid => $uid, master_reqid => Yandex::Trace::trace_id() });
    undef $service;

    if ($response) {
        return $response;
    }
    return undef;
}

=head2 sandbox_create_user_request

    Инициализировать нового пользователя в песочнице
    Параметры:
        fio
        login
        email
        type => [agency, client] - тип создаваемого клиента
        init_test_data => [0,1] - создать тестовые данные
        currency - валюта создаваемого клиента

=cut

sub sandbox_create_user_request
{
    my ($uid, $params) = @_;

    die 'no currency given' unless $params->{currency};

    my %request;

    $request{$_} = $params->{$_} foreach keys %$params;

    $request{uid} = $uid;
    $request{master_reqid} = Yandex::Trace::trace_id();

    local $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;
    my $service = Yandex::Trace::new_service('direct.sandbox', 'CreateSandboxUser');
    my $response = execute_sandbox_method('create_user', \%request);
    undef $service;

    if ($response) {
        return $response;
    }

    return {};
}

=head2 sandbox_drop_user_request

    Удалить пользователя из песочницы, со всеми кампаниями и субклиентами (для агентств)

=cut

sub sandbox_drop_user_request
{
    my ($uid) = @_;

    local $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;

    my $service = Yandex::Trace::new_service('direct.sandbox', 'DropSandboxUser');
    my $response = execute_sandbox_method(
        'drop_user', { uid => $uid, master_reqid => Yandex::Trace::trace_id() });
    undef $service;

    if ($response) {
        return $response;
    }

    return {};
}

=head2 sandbox_drop_finance_ops_counter_request

=cut

sub sandbox_drop_finance_ops_counter_request
{
    my ($uid) = @_;

    local $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;

    my $service = Yandex::Trace::new_service('direct.sandbox', 'DropSandboxFinOpsCounter');
    my $response = execute_sandbox_method(
        'drop_fin_ops_counter', { uid => $uid, master_reqid => Yandex::Trace::trace_id() });
    undef $service;

    if ($response) {
        return $response;
    }

    return {};
}


=head2 sandbox_send_app_access

    Отправляет доступ приложения к API в песочницу 

=cut

sub sandbox_send_app_access
{
    my $data = shift;

    local $ENV{PERL_LWP_SSL_VERIFY_HOSTNAME} = 0;

    my $service = Yandex::Trace::new_service('direct.sandbox', 'SaveSandboxAppAccess');
    my $response = execute_sandbox_method(
        'save_app_access', { data => $data, master_reqid => Yandex::Trace::trace_id() });
    undef $service;

    if ($response->{code} && ($response->{code} eq 'OK')) {
        return 1;
    } else {
        return 0;
    }
}

1;
