package ScriptHelper;

=head1 NAME

ScriptHelper - подключение полезных модулей для скриптов.

=head1 SYNOPSIS

  use ScriptHelper;

  # поключение с кастомными опциями для Yandex::Log и get_file_lock
  use ScriptHelper 'Yandex::Log' => [date_suf => "%Y"]
                  , get_file_lock => ['dont_die', 'file_for_lock.name']
                  , script_timer => ['super-scripts/me', $$]
                  , carp => 0
                  , umask => 0002
                  , sharded => 1
                  ;

=head1 DESCRIPTION

  что вообще происходит:
    - подменяется обработчик предупреждений, чтобы всё выводилось с стэк-трейсом (параметр carp) 
    - подменяется обработчик SIGTERM, для использования сигнала вместо стоп-файла (параметр sigterm)
    - устанавливается umask
    - создается глобальный объект Yandex::Log - $log с параметрами date_suf => "%Y%m%d", log_file_name => "script_file_name.log"
      это можно переопределить при подключении ScriptHelper
      !!если указано 'Yandex::Log' => 'messages', стандартный лог начинает писать в общий syslog-овый messages лог!!
    - умеет перенаправлять в него запись любых Log::Any-логов (параметр --log-any <category>)
    - умеет перенаправлять в него ворнинги (параметр log_warn или --log-warn)
    - при падении скрипта в лог пишется стектрейс (параметр log_die)
    - в начале скрипта вызывается LockTools::get_file_lock(), параметры можно определить при подключении ScriptHelper
    - по окончании вызывается LockTools::release_file_lock()
    - подключается Data::Dumper, Getopt::Long и Dumper(), extract_script_params() (обертка над GetOptions) экспортируются в скрипт
    - подключается Yandex::JugglerQueue и экспортируется juggler_event (обертка над queue_juggler_event, умеющая определять имя сервиса по имени скрипта), а также juggler_{ok,warn,crit}
    - таже экспортируется juggler_check, но не исходный, а его обертка. подробнее описано ниже.
    - если установлена переменная окружения DEBUG_SQL все запросы записываются в лог $script_name.sql
    - экспортирует переменную $SHARD, содержащую номер шарда, для которого скрипт запущен
    - sharded => 1 - скрипт должен быть шардирован (запущен с параметром --shard-id)

  если массив с параметрами не указан - используются параметры по-умолчанию
  если вместо массива с параметрами указан false (0|undef,''), то соответствующий блок не исполняется

Обрабатываются параметры командной строки:

    --help              вывести справку и выйти
    --shard-id <N>      задать номер шарда (для шардированных скриптов)
    --log-any <cat>     выводить события заданной категории Log::Any в лог скрипта
    --reject-run-interval   игнорировать параметр run_interval у итератора

=cut

use Data::Dumper;
use Getopt::Long;

use Yandex::DateTime qw/ now iso8601_2_datetime /;
use Yandex::DBTools qw//;
use Yandex::JugglerQueue qw/juggler_event juggler_ok juggler_warn juggler_crit/;
use Yandex::Log::Messages;
use Yandex::Log;
use Yandex::Trace;

use Direct::Modern;
use EnvTools qw//;
use LockTools;
use TvmChecker;

use Time::HiRes qw/ time sleep /;
use Property;

use Settings ();

use parent qw/Exporter/;

our $log; # global Yandex::Log object
our $trace;
our $script_name;
our $program_name_at_start;
our $SHARD;
our $is_sharded;
our $is_beta;
our %_options;
our @ORIGINAL_ARGV;

our $LOG_ANY_PREFIX = q/<cat>  >>  /;

our @EXPORT = qw/
    restart_tracing
    
    current_trace
    Dumper
    extract_script_params
    
    get_script_name
    smart_check_stop_file

    juggler_event
    juggler_check

    juggler_ok
    juggler_warn
    juggler_crit

    usage

    iterate

    $log
    $SHARD
/;

use constant SCRIPT_START_TIME => time();
use constant SIGTERM_REASON => 'SIGTERM received';
my $SCRIPT_START_DIRECT_VERSION;
my $SIGTERM_RECEIVED = 0;
my $REJECT_RUN_INTERVAL;

=head2 $Yandex::JugglerQueue::SERVICE_NAME_GENERATOR

    Задает способ формирования имен для сырых событий в Juggler

    Параметры:
        $params_str     - строка, которую нужно дописать после .working (перед .shard_$shard)
    Результат:
        $service_name   - строка вида scripts.script_name.working
                            или scripts.script_name.working.$params_str
                            или scripts.script_name.working.shard_$shard
                            или scripts.script_name.working.$params_str.shard_$shard

=cut

$Yandex::JugglerQueue::SERVICE_NAME_GENERATOR = sub {
    local *__ANON__ = 'service_name_generator_from_script_helper';
    my $params_str = shift;
    return sprintf('scripts.%s.working%s%s',
                   get_script_name(shardid => undef),
                   (defined $params_str ? ".$params_str": ''),
                   ($is_sharded ? ".shard_$SHARD": ''),
                   );
};


=head2 juggler_check

    Оборачиваем исходную версию, чтобы решить проблему с несрабатыванием проверок при большом ttl и "ок'ах" с множества хостов.
    Отправляем принудительно все события на один хост (и его же нужно указывать как raw_host в проверках).
    Чтобы события из разных окружений не затирали друг друга - дописываем в конец название конфигурации (его нужно дописывать к raw_event в проверках)
    Подробности почему так сделано можно спросить у ppalex@, когда-нибудь переделаем на более правильную/понятную/явную схему.

=cut

sub juggler_check {
    my %data = @_;

    if ($data{service}) {
        $data{service} .= ".$Settings::CONFIGURATION";
    } elsif ($data{service_suffix}) {
        $data{service_suffix} .= ".$Settings::CONFIGURATION";
    }

    # NB: он же в packages/yandex-direct/debian/rules в параметрах вызова pod2juggler: --auto-host=checks_auto.direct.yandex.ru
    $data{host} = $ScriptsMetadata::JUGGLER_CHECK_RAW_HOST;

    Yandex::JugglerQueue::juggler_check(%data);
}

#-----------------------------------------------------------
sub import {
    my $cls = shift @_;
    %_options = @_;

    if (grep {!/^(get_file_lock|script_timer|Yandex::Log|carp|log_die|log_warn|umask|sharded|sigterm|help)$/} keys %_options) {
        die "Incorrect ScriptHelper usage";
    }

    ScriptHelper->export_to_level(1, $cls, grep {$_ ne '$SHARD' || $_options{sharded}} @EXPORT);
}

=head2 usage

Дефолтный обработчик --help:
выводит pod-секции NAME и DESCRIPTION текущего скрипта

=cut

sub usage {
    my $base_cmd = "podselect -section NAME -section SYNOPSIS -section USAGE -section DESCRIPTION -section RUNNING $0 | /usr/local/bin/pod2text-utf8";

    if ( my $pager = $ENV{PAGER} ) {
        system("$base_cmd | $pager");
    } else {
        system($base_cmd);
    }

    exit(0);
}

=head2 extract_script_params(%params)

Получить параметры скрипта. Обёртка над GetOptions с падением при ошибках.

=cut

sub extract_script_params {
    my %options = @_;

    Getopt::Long::GetOptions(%options)
    or die "Getopt error: $!; stop";
     
    return 1;
}


=head2 _extract_builtin_script_params(%params)

Обработать специальные параметры скрипта (с игнорирование неизвестных).

=cut

sub _extract_builtin_script_params {
    my %options = @_;

    @ORIGINAL_ARGV = @ARGV  if !@ORIGINAL_ARGV;

    Getopt::Long::Configure('pass_through');
    Getopt::Long::GetOptions(%options);
    Getopt::Long::Configure('no_pass_through');

    return;
}


sub _is_correct_shard_id {
    my ($shard) = @_;
    return  if !defined $shard || $shard !~ m/^[0-9]+$/;
    return  if $shard < 1 || $shard > $Settings::SHARDS_NUM;
    return 1;
}


=head2 attach_log_any

Подключает в лог скрипта события Log::Any (в основном нужен для обработки параметра --log-any).
На входе - объект Y::L, категория и уровень (по умолчанию info)

    attach_log_any($log, 'Yandex::BMApi');            # точная категория
    attach_log_any($log, qr/Yandex::BM/, 'trace');    # регексп

Умеет разбирать хитрый формат для командной строки:

    attach_log_any($log, '/XLS/=debug');

=cut

sub attach_log_any {
    my ($ylog, $category, $level) = @_;

    # 'require' to avoid global dependency
    require Log::Any::Adapter;

    carp "Logging is disabled; unable to attach log" and return  if !$ylog;

    if (!$level && $category =~ m/=/) {
        ($category, $level) = split /=/, $category;
    }
    $level //= 'info';

    if (!ref $category && (my ($re) = $category =~ m#^/(.+)/$#)) {
        $category = qr/$re/;
    }

    Log::Any::Adapter->set({category => $category}, 'YandexLog', log => $ylog, severity => $level, extra_prefix => $LOG_ANY_PREFIX);
    return;
}


=head2 get_current_shard_id()

Возвращает текущий шард для скрипта; попутно выставляет переменную $SHARD

Использует параметры командной строки:

    --shard-id N

=cut

sub get_current_shard_id {

    return undef  if !$is_sharded;

    if (!defined $SHARD) {
        _extract_builtin_script_params(
            'shard-id=i' => \my $shard,
#            'stop' => \my $stop, # непонятно, зачем это нужно
        );

        if (!_is_correct_shard_id($shard)) {
            die sprintf('Incorrect shard number "%s" (usage: --shard-id N)', ($shard // ''));
        }

        $SHARD = $shard;
    }

    return $SHARD;
}


=head2 smart_check_stop_file

    "Умная" проверка "требуется ли перезапуск".
    Возвращает 0 (ложь), если перезапуск не требуется.
    Возвращает строку (истина) с описанием причины, по которой нужен перезапуск.

    Типичный пример использования:
    if (my $reason = smart_check_stop_file()) {
        $log->out("$reason. Exitinig!");
        exit(0);
    }

    Параметры: отсутствуют
    Результат: 0 или строка. Возможные варианты:
      # остановка не требуется:
        0 - обработка SIGTERM была отключена при импорте ScriptHelper'а
            скрипт не получал сигнал TERM
            хост, на котором запущен скрипт является разработческим (is_beta())
            версия пакета yandex-direct не изменялась с момента запуска
      # требуется остановка:
        'SIGTERM received'          - был получен сигнал TERM
        'Changed Direct version'    - версия пакета yandex-direct изменилась с момента запуска
                                      (проверяется только на !is_beta() хостах)

=cut
sub smart_check_stop_file {
    if ($SIGTERM_RECEIVED) {
        return SIGTERM_REASON;
    }
    if (!$is_beta && EnvTools::get_current_direct_version() ne $SCRIPT_START_DIRECT_VERSION) {
        return 'Changed Direct version';
    }
    return 0;
}

=head2 get_script_name

    get script name witout .pl and path and shard-id parameter if it exists
    print get_script_name()."\n"; # ppcMonitorYacontextPhrases
    return '' if exec from not .pl script

    Named options:
        shardid => '...' - use this value instead of $SHARD
        short => 1 | 0 - return only script name (without path and ext), do not add _shard_$SHARD

=cut

sub get_script_name
{
    my %options = @_;
    my $script_name = $program_name_at_start;
    
    return '' unless $script_name =~ /\.pl$/;

    $script_name =~ s/\.pl$//; # remove .pl
    $script_name =~ s|^.+/||;  # remove path

    if ($options{short}) {
        return $script_name;
    }

    my $shardid = exists $options{shardid} ? $options{shardid} : get_current_shard_id(); 
    return defined $shardid ? sprintf('%s_shard_%s', $script_name, $shardid) : $script_name;
}

=head2 restart_tracing

Создать объект трейсинга и настроить логгирование в trace.log

=cut

sub restart_tracing
{
    my $tags = shift;
    if (!$tags && $is_sharded) {
        # в умолчальном режиме работы добавляем шард в теги (если он задан)
        my $shard_for_tag = get_current_shard_id();
        if (defined $shard_for_tag) {
            $tags = sprintf("shard_%d", $shard_for_tag);
        }
    }
    undef $trace;
    $trace = Yandex::Trace->new(service => 'direct.script', method => get_script_name(short => 1), $tags ? (tags => $tags) : ());
}

=head3 _sig_term_handler

    Обработчик для сигнала TERM.
    Взводит флаг SIGTERM_RECEIVED, а также выводит в лог или на STDERR соответствующее сообщение.

=cut
sub _sig_term_handler {
    unless ($SIGTERM_RECEIVED) {
        my $msg = 'Received SIGTERM. Script will be stopped at the next smart_check_stop_file call';
        if ($log) {
            # Скрипт (вероятно) использует $log, созданный ScriptHelper'ом. Сообщим в него
            $log->out($msg);
        } else {
            # Скрипт не использует $log, сообщим на STDERR
            print STDERR $msg, "\n";
        }
        $SIGTERM_RECEIVED = 1;
    }
};

=head2 iterate

    iterate %params, sub { ... };

    iterate
        sleep_coef => 0.2,
        run_interval => { hours => 11 },
        sub { ... };

"Менеджер итераций" для скриптов.
Получает параметры и функцию, которую последовательно запускает.
На вход функции подаётся результат предыдущего запуска (для первого запуска - undef).
Когда функция вернёт пустой ответ, итерирование прекращается.

На данный момент умеет:
* отдыхать между итерациями (sleep_coef)
* соблюдать интервал между запусками (run_interval)
* обрабатывать падения

Параметры:
    sleep_coef
    run_interval - хеш в формате DateTime::Duration
    skip_failure

Ловит опции комадной строки:
    --reject-run-interval

=cut

sub iterate {
    my $sub = pop @_;
    my %opt = @_;

    # проверяем, нужно ли выполнять
    my $last_success_time_prop = Property->new(get_script_name() . "__last_success_time");
    if ($opt{run_interval} && !$REJECT_RUN_INTERVAL) {
        if (my $last_success_time = $last_success_time_prop->get()) {
            my $valid_start_time = iso8601_2_datetime($last_success_time)->add(%{$opt{run_interval}});
            if ($valid_start_time > now()) {
                $log->out("It's too early, wait until $valid_start_time") if $log;
                return;
            }
        }            
    }

    my $iteration_count = 0;
    my $iteration_result;

    ITERATION:
    while (1) {
        $iteration_count++;

        $log->out("Iteration #$iteration_count") if $log;
        my $start_time = time;
        my $succeed = eval { $iteration_result = $sub->($iteration_result); 1};

        if (!$succeed) {
            my $error = $@;
            $log->out("Iteration #$iteration_count failed: $error") if $log;
            next ITERATION if $opt{skip_failure};
            croak $error;
        }

        #todo: save last_success_iteration

        if (!defined $iteration_result) {
            $log->out("Iterating finished")  if $log;
            last ITERATION;
        }

        sleep $opt{sleep_coef} * (time-$start_time)  if $opt{sleep_coef};
    }

    if ($opt{run_interval}) {
        $last_success_time_prop->set(now()->datetime());
    }

    return;
}


#-----------------------------------------------------------
INIT {
    # кешируем путь до скрипта (т.к. $0 может меняться, например из foreach_shard_parallel)
    $program_name_at_start = $0;
    $is_sharded = $_options{sharded};
    $is_beta = EnvTools::is_beta();

    # выводим хелп: переданный снаружи, usage у скрипта или встроенную usage
    my $usage_sub = exists $_options{help}
        ? $_options{help}
        : $main::{usage} && $main::{usage}->{CODE} || \&usage;
    _extract_builtin_script_params('help' => $usage_sub) if $usage_sub;

    # тут неявно выставляется переменная $SHARD
    $script_name ||= get_script_name() || 'common';

    if (!exists $_options{carp} || $_options{carp}) {
        $SIG{__WARN__} = \&Carp::cluck;
        $SIG{__DIE__} = \&Carp::confess;
    }

    if (!exists $_options{script_timer} || $_options{script_timer}) {
        restart_tracing();
    }
    
    if (!exists $_options{umask} || defined $_options{umask}) {
        umask(defined $_options{umask} ? $_options{umask} : 0002);
    }

    if (!exists $_options{'Yandex::Log'} || $_options{'Yandex::Log'}) {
        if (!exists $_options{'Yandex::Log'} || $_options{'Yandex::Log'} eq 'messages') {
            $log = Yandex::Log::Messages->new();
            if ($is_sharded) {
                $log->msg_prefix("[shard_".get_current_shard_id()."]");
            }
        } else {
            my $yandex_log_options = {@{ $_options{'Yandex::Log'} }};
            $yandex_log_options->{log_file_name} = "$script_name.log" unless $yandex_log_options->{log_file_name};
            $log = Yandex::Log->new(%$yandex_log_options);
        }
    }

    if ($log && (!exists $_options{log_die} || $_options{log_die}) ) {
        my $old_sig_die = $SIG{__DIE__};
        $SIG{__DIE__} = sub {
            # пишем падения в лог,
            # если не внутри eval или $log->die,
            # и если в лог уже что-то писали
            if (defined $^S && !$^S && $log->num_records) {
                my $msg = Carp::longmess(@_);
                $log->out("CROAK: $msg")  if $msg !~ 'Yandex::Log::die';
            }

            local $Carp::CarpLevel = $Carp::CarpLevel + 1;
            return $old_sig_die->(@_)  if $old_sig_die;
        };
    }

    _extract_builtin_script_params('log-warn!' => \$_options{log_warn});
    if ($log && $_options{log_warn}) {
        my $old_sig_warn = $SIG{__WARN__};
        $SIG{__WARN__} = sub {
            # пишем ворнинги в лог,
            # если не внутри $log->warn,
            # и если в лог уже что-то писали
            if ($log->num_records) {
                my $msg = Carp::longmess(@_);
                $log->out("WARN: $msg")  if $msg !~ 'Yandex::Log::warn';
            }

            local $Carp::CarpLevel = $Carp::CarpLevel + 1;
            return $old_sig_warn ? $old_sig_warn->(@_) : warn @_;
        };
    }

    if ($ENV{LOG_TEE}) {
        binmode(STDERR, ':utf8');
    }

    if (!exists $_options{sigterm} || $_options{sigterm}) {
        $SIG{TERM} = \&_sig_term_handler;
    }

    if (!exists $_options{get_file_lock} || $_options{get_file_lock}) {
        my $get_file_lock_options;
        
        if ($_options{get_file_lock}) {
            die "Incorrect get_file_lock param, should be an array refernce" unless ref $_options{get_file_lock} eq 'ARRAY';

            if (scalar(@{ $_options{get_file_lock} }) == 1) {
                # В случае, если позвали с ['dont_die'], принудительно добавляем имя скрипта (попытка учесть шардинг)
                push @{ $_options{get_file_lock} }, $script_name;
            }
            $get_file_lock_options = $_options{get_file_lock};
        } else {
            $get_file_lock_options = [undef, $script_name];
        }

        get_file_lock(@$get_file_lock_options);
    }

    unless ($is_beta || $ENV{CI}) {
        $SCRIPT_START_DIRECT_VERSION = EnvTools::get_current_direct_version();
    }

    if ($ENV{DEBUG_SQL}) {
        $Yandex::DBTools::QUERIES_LOG = "$script_name.sql";
    }

    {
        no warnings 'once';
        $Yandex::TVM2::SECRET_PATH //= $Settings::TVM2_SECRET_PATH{scripts};
        $Yandex::TVM2::APP_ID //= $Settings::TVM2_APP_ID{scripts};
        $Yandex::Blackbox::BLACKBOX_USE_TVM_CHECKER = \&TvmChecker::use_tvm;
    }

    _extract_builtin_script_params(
        'reject-run-interval' => \$REJECT_RUN_INTERVAL,
        'log-any=s' => sub { shift; attach_log_any($log => @_) },
    );
}

#-----------------------------------------------------------
END {
    if (!exists $_options{get_file_lock} || $_options{get_file_lock}) {
        release_file_lock();
    }
    Yandex::DBTools::disconnect_all();
    undef $trace;
}

1;
