#!/usr/bin/perl

package SearchBanners;

=head1 SYNOPSIS

  use SearchBanners;
  
  search_banner ( { what => 'num' text_search => '1,3,4,5', limit => 0 } )

=cut

=head1 DESCRIPTION

    Поиск баннера по домену, фразе или номеру.

=cut

use strict;
use warnings;

use Common;
use Yandex::ListUtils qw/xuniq xisect/;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::Overshard;
use Yandex::Validate;
use Yandex::DBShards;
use Yandex::IDN qw/is_valid_domain/;
use VCards;
use User;
use Settings;
use Primitives;
use Campaign;
use Campaign::Types;
use BS::Search;
use ShardingTools;
use Direct::AdGroups2::MobileContent;
use Direct::BannersAdditions;
use Direct::Model::ImageFormat;
use Direct::Model::ImagePool;
use Direct::Model::Creative;
use Direct::Model::BannerCreative;
use Direct::Model::Creative::Factory;
use Direct::Model::TurboLanding;
use Sitelinks qw//;
use GeoTools qw/is_targeting_include_region/;
use PrimitivesIds qw/get_bids/;
use Stat::OrderStatDay;

use List::MoreUtils qw/any uniq none/;
use List::Util qw/min/;

use constant DEFAULT_LIMIT => 500;
use constant BANNERS_LIMIT => 7_000;
use constant GROUP_CAMP_LIMIT => 250_000;

use utf8;

use base qw/Exporter/;
our @EXPORT = qw/
                 search_banners
                 convert_old_style_params
                /;

sub _do_banners_search
{
    my ($params, $where_cond) = @_;

    $where_cond->{'c.type'} = get_camp_kind_types('web_edit_base');
    $where_cond->{'p.adgroup_type'} = [qw/base dynamic mobile_content performance mcbanner cpm_banner cpm_video cpm_outdoor cpm_yndx_frontpage 
      content_promotion_video cpm_indoor cpm_audio cpm_geoproduct cpm_geo_pin content_promotion/];

    my $sql_activeonly = {};

    if ( $params->{activeonly} ) {
        $sql_activeonly = {'b.statusModerate' => 'Yes',
                           'p.statusModerate' => 'Yes',
                           'c.statusShow' => 'Yes',
                           'b.statusShow' => 'Yes',
                           'c.OrderID__gt' => 0,
                           'b.BannerID__gt' => 0,
                           'c.archived' => 'No',
                           '_AND' => {_TEXT => 'c.sum - c.sum_spent + IF(wc.cid, wc.sum - wc.sum_spent, 0) > 0'},
                         };
    }
    
    my $sql_limit  =  $params->{limit}  ? " limit ".$params->{limit} : "";
    my $select_fields;

    if ($params->{ short }) {
        $select_fields="b.bid, p.cid, b.BannerID, p.geo";
    }
    else {
        my $vc_fields     = join ', ', map { "vc.$_" } @{$VCards::VCARD_FIELDS};

        $select_fields="u.phone as user_phone, u.uid, u.login, u.FIO, u.ClientID

                      , b.bid, b.BannerID, b.title, b.title_extension, b.body, b.flags, IFNULL(b.domain,d.domain) AS domain, b.href
                      , b.sitelinks_set_id
                      , if(b.banner_type in ('image_ad', 'mcbanner', 'cpm_banner', 'cpc_video', 'content_promotion'), b.banner_type, 'text') as ad_type
                      , b.banner_type

                      , $vc_fields

                      , bim.image_id
                      , bim.image_hash as image, bif.namespace, bif.mds_group_id, bif.avatars_host
                      , bim.statusModerate as image_statusModerate
                      , bimp.name as image_name
                      , bim.PriorityID as image_PriorityID
                      , bim.BannerID AS image_BannerID

                      , p.pid
                      , p.geo, '' AS yacontextCategories

                      , c.ManagerUID , c.AgencyUID

                      , c.cid, c.sum, c.sum_spent, c.sum_to_pay, DATE_FORMAT(c.start_time, '%Y%m%d000000') as start_time, c.finish_time
                      , c.wallet_cid, ifnull(wc.sum, 0) wallet_sum, ifnull(wc.sum_spent, 0) wallet_sum_spent
                      , c.OrderID, c.statusShow, c.statusActive, c.statusModerate
                      , c.statusBsSynced, c.timeTarget, c.timezone_id, c.name as camp_name, c.archived
                      , wc.day_budget as wallet_day_budget
                      , wco.day_budget_stop_time as wallet_day_budget_stop_time

                      , co.stopTime, co.statusPostModerate
                      , c.day_budget, c.day_budget_show_mode, co.day_budget_daily_change_count, co.day_budget_stop_time
                      , c.currencyConverted
                      , IFNULL(c.currency, 'YND_FIXED') AS currency
                      , c.type
                      , c.type as mediaType
                      , p.adgroup_type
                      , gmc.store_content_href
                      , gmc.mobile_content_id
                      , gmc.device_type_targeting
                      , gmc.network_targeting
                      , bmc.reflected_attrs
                      , bmc.primary_action
                      , aid.disclaimer_text AS disclaimer
                      , btl.tl_id as turbolanding_id
                      , cpv.video_preview_url as content_promotion_video_preview_url
                      , bpml.permalink
                      , ifnull(cpromo.preview_url,cpv.video_preview_url) as content_promotion_preview_url
                      "
                      . "," . Direct::Model::Image->get_db_columns(images => 'im', prefix => 'im_')
                      . "," . Direct::Model::ImageFormat->get_db_columns(banner_images_formats => 'bimf', prefix => 'bimf_')
                      . "," . Direct::Model::ImagePool->get_db_columns(banner_images_pool => 'imp', prefix => 'imp_')
                      . "," . Direct::Model::BannerCreative->get_db_columns(banners_performance => 'bp', prefix => 'bp_')
                      . "," . Direct::Model::Creative->get_db_columns(perf_creatives => 'perfc', prefix => 'perfc_')
                      ;
    }

    my $currency_archive_joins = '';
    my $currency_archive = {};
    if (!$params->{include_currency_archived_campaigns}) {
        $currency_archive_joins = 'LEFT JOIN clients cl ON u.ClientID = cl.ClientID';
        $currency_archive = {_TEXT => '(IFNULL(cl.work_currency,"YND_FIXED") = "YND_FIXED" OR IFNULL(c.currency, "YND_FIXED") <> "YND_FIXED")'};
    }

    my $where_str = Yandex::DBTools::sql_condition(hash_merge ($where_cond, $sql_activeonly, $currency_archive));
    my $tables = qq{  banners b
                      left join vcards vc on vc.vcard_id = b.vcard_id 
                      join phrases p on b.pid = p.pid
                      left join adgroups_dynamic gd on (gd.pid = p.pid)
                      left join domains d on (d.domain_id = gd.main_domain_id)
                      join campaigns c on p.cid = c.cid
                      left join campaigns wc on wc.cid = c.wallet_cid
                      left join camp_options co on co.cid = c.cid
                      left join camp_options wco on wco.cid = c.wallet_cid
                      join users u on c.uid = u.uid
                      left join banner_images bim on bim.bid = b.bid
                      left join banner_images_pool bimp on bimp.ClientID = u.ClientID AND bimp.image_hash = bim.image_hash
                      left join banner_images_formats bif on bif.image_hash = bim.image_hash
                      left join adgroups_mobile_content gmc on p.pid = gmc.pid
                      left join banners_mobile_content bmc on bmc.bid = b.bid
                      left join banners_performance bp on bp.bid = b.bid
                      left join perf_creatives perfc on perfc.creative_id = bp.creative_id
                      left join images im on im.bid = b.bid
                      left join banner_images_formats  bimf on bimf.image_hash = im.image_hash
                      left join banner_images_pool imp on imp.ClientID = u.ClientID AND imp.image_hash = bimf.image_hash
                      LEFT JOIN banners_additions bad ON (bad.bid=b.bid AND bad.additions_type="disclaimer")
                      LEFT JOIN additions_item_disclaimers aid ON (aid.additions_item_id = bad.additions_item_id)
                      LEFT JOIN banner_turbolandings btl ON (b.bid = btl.bid)
                      LEFT JOIN banners_content_promotion_video bcpv ON (b.bid = bcpv.bid) 
                      LEFT JOIN content_promotion_video cpv ON (bcpv.content_promotion_video_id = cpv.content_promotion_video_id)
                      LEFT JOIN banner_permalinks bpml ON bpml.bid = b.bid AND bpml.permalink_assign_type = 'manual'
                      LEFT JOIN banners_content_promotion bcpromo ON (bcpromo.bid = b.bid)
                      LEFT JOIN content_promotion cpromo ON (cpromo.id = bcpromo.content_promotion_id)
                      $currency_archive_joins
    };
    my $search_banners_sql = qq{select $select_fields from $tables where $where_str $sql_limit};

    my %shard = choose_shard_param($where_cond, [qw/uid pid bid cid/], allow_shard_all => 1, set_shard_ids => 1);
    my $banners;
    if ($params->{chunk_by_bid} && $where_cond->{'b.bid__int'}) { 
        my $bids = delete $where_cond->{'b.bid__int'};
        $bids = [sort {$a <=> $b} @$bids];
        $banners = [];

        if ($params->{compute_total_banners_count}) {
            $params->{total_banners_count} = 0;
            for my $chunk (sharded_chunks bid => $bids) {
                $params->{total_banners_count} += get_one_field_sql(PPC(shard => $chunk->{shard}),
                    ["SELECT count(*) cnt FROM $tables", WHERE => {'b.bid' => $chunk->{bid}, %$where_cond}]);
            }
        }

        my $seen_bids = {};
        for my $chunk (sharded_chunks bid => $bids, chunk_size => 1000) {
            my $shard = $chunk->{shard};
            my $bids_chunk = $chunk->{bid};

            my $rows = get_all_sql(PPC(shard => $shard), [
                qq{SELECT $select_fields FROM $tables},
                WHERE => {'b.bid__int' => $bids_chunk, %$where_cond}, 
               'ORDER BY b.bid'
            ]);

            if ($params->{limit} && (@$banners + @$rows > $params->{limit})) {
                splice(@$rows, $params->{limit} - @$banners);
                $seen_bids->{$_} = 1 for grep { @$rows && $_ <= $rows->[-1]->{bid} } @$bids_chunk;
                push @$banners, @$rows;
                $params->{found_more_rows_than_limit} = 1;
                last;
            } else {
                $seen_bids->{$_} = 1 for @$bids_chunk;
                push @$banners, @$rows;
            }
        }
        $params->{use_bansearch_bids} = [grep { !$seen_bids->{$_} } @{ $params->{use_bansearch_bids} }];

        if ($params->{group_camp}) {
            my $bids_by_cid = {};
            $bids_by_cid->{$_->{cid}}->{$_->{bid}}++ for @$banners;

            $banners = [xuniq {$_->{cid}} @$banners];
            $_->{bids} = join(',', keys %{ $bids_by_cid->{$_->{cid}} }) for @$banners;
        }
        $where_cond->{'b.bid__int'} = $bids;
    } elsif ($params->{group_camp}) {
        $search_banners_sql = "SELECT t.*, GROUP_CONCAT(t.bid SEPARATOR ',') AS bids FROM ($search_banners_sql) t GROUP BY cid order by null";
        $banners = get_all_sql(PPC(%shard), $search_banners_sql );
    } else {

        $banners = get_all_sql(PPC(%shard), $search_banners_sql );
        $banners = overshard (%{hash_cut $params, qw/limit offset/}, $banners);
    }

    unless ($params->{short}) {
        my %manager_or_agency;
        for my $banner (@$banners) {
            $manager_or_agency{$banner->{ManagerUID}} = 1 if $banner->{ManagerUID};
            $manager_or_agency{$banner->{AgencyUID}} = 1 if $banner->{AgencyUID};
        }

        my $users_data = get_users_data([keys %manager_or_agency], [qw/uid login fio email/]);

        enrich_data($banners, using => 'ManagerUID', map_fields => {login => 'mlogin', fio => 'mfio', email => 'memail'}, sub { return $users_data });
        enrich_data($banners, using => 'AgencyUID', map_fields => {login => 'alogin', fio => 'afio', email => 'aemail'}, sub { return $users_data });
    }

    my $mobile_content_banners = [grep {defined $_->{adgroup_type} && $_->{adgroup_type} eq 'mobile_content'} @$banners];
    if (@$mobile_content_banners) {
        my $mobile_content = Direct::AdGroups2::MobileContent->get_mobile_content_by(adgroup_id => [map {$_->{pid}} @$mobile_content_banners]);
        for my $banner (@$mobile_content_banners) {
            $banner->{reflected_attrs} = [split(/,/, $banner->{reflected_attrs})];
            $banner->{mobile_content} = exists $mobile_content->{$banner->{pid}}
                                        ? $mobile_content->{$banner->{pid}}->to_template_hash
                                        : {};
        }
    }

    my $_cache = {};
    for my $banner (@$banners) {
        if ( any { $banner->{banner_type} eq $_ } qw/image_ad mcbanner cpm_banner/ ) {
            if ($banner->{im_bid}) {
                # imagead/picture
                my $image = Direct::Model::Image->from_db_hash($banner, \$_cache, prefix => 'im_');
                $image->format(Direct::Model::ImageFormat->from_db_hash($banner, \$_cache, prefix => 'bimf_'));
                $image->pool(Direct::Model::ImagePool->from_db_hash($banner, \$_cache, prefix => 'imp_'));
                $banner->{image_ad} = $image->to_template_hash();
            } else {
                # canvas-creative
                my $creative = Direct::Model::BannerCreative->from_db_hash($banner, \$_cache, prefix => 'bp_');
                $creative->creative(Direct::Model::Creative::Factory->create($banner, \$_cache, prefix => 'perfc_'));
                $banner->{creative} = $creative->to_template_hash();
            }
        }

        if ($banner->{banner_type} =~ /performance|cpc_video|cpm_outdoor|cpm_indoor|cpm_audio|cpm_geoproduct|cpm_geo_pin/) {
            $banner->{creative} = Direct::Model::Creative::Factory->create($banner, \$_cache, prefix => 'perfc_')->to_template_hash();
        }

        if ($banner->{banner_type} =~ /text|mobile_content/ && $banner->{perfc_creative_id}) {
            $banner->{video_resources} = Direct::Model::Creative::Factory->create($banner, \$_cache, prefix => 'perfc_')->to_template_hash();
        }

        $banner->{cpm_banners_type} = $banner->{adgroup_type};
        $banner->{adgroup_type} = 'cpm_banner' if any {$banner->{adgroup_type} eq $_} qw/cpm_video cpm_outdoor cpm_yndx_frontpage cpm_indoor cpm_audio cpm_geoproduct cpm_geo_pin/;
        
        my @delete_fields = 
            (
                (map { 'bimf_'.$_ } Direct::Model::ImageFormat->get_db_columns_list('banner_images_formats')),
                (map { 'imp_'.$_ } Direct::Model::ImagePool->get_db_columns_list('banner_images_pool')),
                (map { 'perfc_'.$_ } Direct::Model::Creative->get_db_columns_list('perf_creatives')),
            );
        delete $banner->{$_} for @delete_fields;
    }

    return $banners;
} ## end sub _doBannersSearch

sub _do_media_banners_search
{
    my ( $params, $where_cond ) = @_;

    $where_cond->{'c.type'} = get_camp_kind_types('media');

    my $sql_activeonly = {};

    if ( $params->{activeonly} ) {
        $sql_activeonly = {'b.statusActive' => 'Yes',
                           'c.statusShow' => 'Yes',
                           'c.OrderID__gt' => 0,
                           'b.BannerID__gt' => 0,
                           'c.archived' => 'No',
                           '_AND' => {_TEXT => 'c.sum - c.sum_spent + IF(wc.cid, wc.sum - wc.sum_spent, 0) > 0'},
                         };
    }

    my $sql_limit = $params->{limit} ? " limit ".$params->{limit} : "";
    my $where_str = Yandex::DBTools::sql_condition(hash_merge ($where_cond, $sql_activeonly));

    my $search_banners_sql = qq{
              select b.mbid as bid
                       , u.phone as user_phone, u.uid, u.login, u.FIO
                       , b.mgid, b.alt, b.href, b.domain
                       , b.format_id, b.md5_flash, b.name_flash, b.md5_picture, b.name_picture
                       , b.name
                       , c.cid
                       , mg.geo
                       , c.ManagerUID
                       , c.AgencyUID
                       , c.sum, c.sum_spent
                       , c.wallet_cid, ifnull(wc.sum, 0) wallet_sum, ifnull(wc.sum_spent, 0) wallet_sum_spent
                       , c.sum_to_pay
                       , DATE_FORMAT(c.start_time, '%Y%m%d000000') as start_time
                       , c.OrderID
                       , c.statusShow
                       , c.statusActive
                       , c.statusModerate
                       , co.statusPostModerate
                       , c.statusBsSynced
                       , c.timeTarget
                       , c.name as camp_name
                       , c.archived
                       , co.stopTime
                       , caq.operation as delayed_arc
                from media_banners b
                      join media_groups mg on b.mgid = mg.mgid
                      join campaigns c on c.cid = mg.cid
                      left join campaigns wc on wc.cid = c.wallet_cid
                      left join camp_operations_queue caq on caq.cid=c.cid
                      left join camp_options co on co.cid = c.cid
                      join users u on c.uid = u.uid
                where
                      $where_str
                $sql_limit
                };
    my $mbanners;
    if ($params->{group_camp}) {
        $search_banners_sql = "SELECT * FROM ($search_banners_sql) t GROUP BY cid";
        $mbanners = get_all_sql(PPC(shard => 'all'), $search_banners_sql );
        $mbanners = overshard(group => 'cid', $mbanners);
    } else {
        #Пока не происходит заполнения таблицы shard_inc_mbid - используем все шарды.
        $mbanners = get_all_sql(PPC(shard => 'all'), $search_banners_sql );
        $mbanners = overshard (%{hash_cut $params, qw/limit/}, $mbanners);
    }
    
    my @manager_or_agency = xuniq {$_} map { $_->{ManagerUID} || $_->{AgencyUID} || () } @$mbanners;
    my $users_data = get_users_data(\@manager_or_agency, [qw/uid login fio email/]);

    enrich_data($mbanners, using => 'ManagerUID', map_fields => {login => 'mlogin', fio => 'mfio', email => 'memail'}, sub { return $users_data });
    enrich_data($mbanners, using => 'AgencyUID', map_fields => {login => 'alogin', fio => 'afio', email => 'aemail'}, sub { return $users_data });

} ## end sub _doMediaBannersSearch


=head3 _text_processor

    Получить список bid по списку текстов баннеров с помощью полнотекстового поиска БК
    и вернуть SQL условие фильтрации по этим bid

=cut

sub _text_processor {
    my ($values) = @_;

    my @bids;
    for my $value (@$values) {
        push @bids, BS::Search::fulltext_search($value);
    }

    return {'b.bid__int' => \@bids}
}

=head3 _domain_processor

    Получить SQL условие фильтрации по списку доменов

=cut

sub _domain_processor {
    my ($values, $params) = @_;

    my @reverse_domains = map { reverse_domain($_) } @$values;
    my $condition_field = 'reverse_domain';
    unless ($params->{exact_domain}) {
        $condition_field .= '__like';
        @reverse_domains = map {sql_quote_like_pattern($_) . '%'} @reverse_domains;
    }

    my @conds = map { ("b.${condition_field}" => $_) } @reverse_domains;
    if (!$params->{media}) {
        # Т.к. фильтровать в одном запросе по banners.reverse_domain и по сджоиненному полю domains.reverse_domain
        # не представляется возможным (нужно делать через UNION), то
        # выберем список динамических групп, содержащих искомый домен, отдельным запросом.
        my $domain_gids = get_one_column_sql(PPC(shard => 'all'), [
                "SELECT gd.pid FROM adgroups_dynamic gd JOIN domains d ON (d.domain_id = gd.main_domain_id)",
                WHERE => {_OR => [map { ("d.${condition_field}" => $_) } @reverse_domains]},
            ]);
        push @conds, "b.pid__int" => $domain_gids;
    }

    return {_OR => \@conds};
}

=head3 _phrase_processor

    Получить SQL условие фильтрации по списку фраз

=cut

sub _phrase_processor {
    my ($values, $params) = @_;

    if ($params->{use_bansearch}) {
        $params->{use_bansearch_bids} //= Common::get_bids_by_phrase_text($values);
        $params->{chunk_by_bid} = 1;
        return {'b.bid__int' => $params->{use_bansearch_bids}};
    } else {
        my $banners = Common::get_competitors($values, {get_all_ads => 1, strict_phrase => $params->{strict_phrase}}) || [];
 
        my $bid2price = { map {$_->{bid} => $_->{price}} @$banners };
        $params->{set_prices} = $bid2price;
 
        return {'b.bid__int' => [ map {$_->{bid}} @$banners ]}
    }
}

=head3 _banner_id_processor

    Получить SQL условие фильтрации по списку номеров объявления в БК

=cut

sub _banner_id_processor {
    my ($values, $params) = @_;

    my $bids = get_one_column_sql(PPC(shard => 'all'), [
        'SELECT b.bid FROM banners b', WHERE => {'b.BannerID' => $values},
        'UNION SELECT bim.bid FROM banner_images bim', WHERE => {'bim.BannerID' => $values},
    ]);

    return {'b.bid__int' => $bids}
}

=head3 _creative_nmb_processor

    Получить SQL условие фильтрации по списку номеров креативов (smart и canvas)

=cut

sub _creative_nmb_processor {
    my ($values, $params) = @_;

    my $bids = PrimitivesIds::get_bids(creative_id => $values);

    return {'b.bid__int' => $bids}
}

=head3 _is_not_empty

    Проверить что текст не пустой

=cut

sub _is_not_empty {
    my ($val) = @_;

    return ((defined $val && $val ne '') ? 1 : 0);
}

=head3

     Получить SQL условие фильтрации по списку id турболендингов

=cut

sub _turbolanding_id_processor {
    my ($values, $params) = @_;

    my $bids = _get_affected_bids_by_tl_params({'t.tl_id' => $values});

    return {'b.bid__int' => $bids};
}

=head3

    Получить SQL условие фильтрации по имени турболендинга или его части (синтаксис like)

=cut

sub _turbolanding_name_processor {
    my ($values, $params) = @_;

    my $name_part = $values->[0];
    $name_part =~ s/\*/%/g;
    my $bids = _get_affected_bids_by_tl_params({'t.name__like' => $name_part});

    return {'b.bid__int' => $bids};
}

=head3

    Получить SQL условие фильтрации по имени турболендинга или его url

=cut

sub _turbolanding_url_processor {
    my ($values, $params) = @_;

    my $url = $values->[0];
    #Ищем в url уникальный хеш турболендинга
    
    my ($hash) = $url =~ /text=[a-z\-]+(?:\/|%2F)?([0-9a-z]+)/;
    
    #В turbo_site_href url хранится без http/https.
    #Get-параметров там тоже быть не должно.
    my $url_without_scheme = $url =~ s/^https?:\/\/([^\?]+).*/$1/ir; 
    my @turbo_site_params = ('t.turbo_site_href__like' => '%'.$url_without_scheme.'%');
    
    my $bids = _get_affected_bids_by_tl_params(length($hash // '') > 0 ? 
            {'t.href__like' => 'https://yandex.ru/turbo%?text=%'.$hash.'%'} 
            : {_OR => {'t.href' => $url, @turbo_site_params}});

    return {'b.bid__int' => $bids};
}


=head3

    Получить идентификаторы баннеров, к которым привзяны удовлетворяющие условиям поиска турболендинги

=cut

sub _get_affected_bids_by_tl_params {
    my ($param) = @_;
    
    my $bids = get_one_column_sql(PPC(shard => 'all'), ['
        SELECT DISTINCT bid FROM (
            SELECT DISTINCT b.bid FROM
                banners b
                JOIN campaigns c ON (b.cid = c.cid)
                JOIN sitelinks_set_to_link sll ON (b.sitelinks_set_id=sll.sitelinks_set_id)
                JOIN sitelinks_links s ON (s.sl_id=sll.sl_id)
                JOIN turbolandings t on (c.ClientID = t.ClientID AND s.tl_id = t.tl_id)
            ',
            WHERE => $param,
            'UNION ALL 
                SELECT DISTINCT b.bid FROM
                    banner_turbolandings b
                    JOIN campaigns c ON (b.cid = c.cid)
                    JOIN turbolandings t on (c.ClientID = t.ClientID AND b.tl_id = t.tl_id)
            ',
            WHERE => $param,
        ') s'
    ]);

    return $bids;
}

=head3 SUPPORTED_SEARCH_KEYS

    Хеш с описанием всех поддерживаемых поиском полей. В качестве ключа выступает
    имя поля, значение - hashref с описанием принципов его обработки, который может
    содержать пять ключей:

    - supports_media - флаг, по умолчанию - ложный. Поддерживается ли поиск по этому ключу в Баяне
    - validator - coderef, обязательный. Функция-валидатор значения для ключа.
    - processor - coderef. Обязателен, если не задан query_key. Функция, которая
        преобразует переданные значения для поиска в условия для SQL-запроса.
        Имеет приоритет над query_key
        Функции передается arrayref значений для поиска и параметры вызова search_banners
    - query_key - текст. Обязателен, если не задан processor. Ключ для условия SQL-запроса
        для стандартных и медийных кампаний (последнее, если не задан media_query_key)
    - media_query_key - текст, опциональный. Ключ для условия SQL-запроса для
        медийных кампаний.

=cut

my %SUPPORTED_SEARCH_KEYS = (
    num => {
        supports_media => 1,
        validator => \&is_valid_id,
        query_key => 'b.bid__int',
        media_query_key => 'b.mbid__int',
    },
    campaign => {
        validator => \&is_valid_id,
        query_key => 'c.cid__int',
    },
    image_id => {
        validator => \&is_valid_id,
        query_key => 'bim.image_id__int',
    },
    group => {
        validator => \&is_valid_id,
        query_key => 'p.pid__int',
    },
    text => {
        validator => \&_is_not_empty,
        processor => \&_text_processor,
    },
    phrase => {
        validator => \&_is_not_empty,
        processor => \&_phrase_processor,
    },
    domain => {
        supports_media => 1,
        validator => \&is_valid_domain,
        processor => \&_domain_processor,
    },
    login => {
        supports_media => 1,
        validator => \&_is_not_empty,
        query_key => 'u.login',
    },
    banner_id => {
        validator => \&is_valid_id,
        processor => \&_banner_id_processor,
    },
    # Поиск по номеру креатива (в таблицe perf_creatives)
    # Это касается смартовых креативов и креативов из конструктора (canvas)
    creative_nmb => {
        validator => \&is_valid_id,
        processor => \&_creative_nmb_processor,
    },
    agency_id => {
        validator => \&is_valid_id,
        query_key => 'c.AgencyID__int',
    },
    agency_uid => {
        validator => \&is_valid_id,
        query_key => 'c.AgencyUID__int'
    },
    turbolanding_id => {
        validator => \&is_valid_id,
        processor => \&_turbolanding_id_processor,
    },
    turbolanding_name_part => {
        validator => \&_is_not_empty,
        processor => \&_turbolanding_name_processor,
    },
    turbolanding_url => {
        validator => \&_is_not_empty,
        processor => \&_turbolanding_url_processor,
    },
);


our @ALLOWED_PARAMS = qw/
    exact_domain
    group_camp
    strict_phrase
    limit
    offset
    short
    activeonly
    include_currency_archived_campaigns
    search_or_net
    broad_match_flag
    retargetings
    strategy
    geo
    media
    use_bansearch
    /;

=head2 search_banners($criteria, $params)

    Осуществляет поиск баннеров или медийных баннеров по (домену и/или номеру) или фразам
    Баннеры ищутся в БД, а фразы запрашиваются из крутилки.

    Принимает два аргумента:
        - criteria - arrayref, список хешей со значениями для поиска
        - params - дополнительные параметры поиска

    Агрумент criteria должен состоять из положительного количества хешей вида

        {key => $type, values => $values}

        Здесь key - это что-то задающее тип условия фильтрации. Может принимать следующие значения:

            - num - поиск по номеру баннера (bid)
            - domain - поиск по домену
            - campaign - поиск по номеру кампании (cid)
            - phrase - поиск по фразе
            - text - поиск по тексту объявления (используется полнотекстовый поиск в БК)
            - image_id - поиск по id картинки для баннера (banner_images.image_id)
            - group - поиск по номеру группы (pid)
            - login - поиск по номеру группы (pid)

        values - это arrayref со значениями для соответствующего условия

    Ключи не могут встречаться более одного раза. В случае указания более одного условия поиска,
    результатом будет пересечение этих условий.

    Аргумент params содержит флаги, ограничения, фильтры и специфичные для типов опции:

    Флаги:

        - media - если истина, производить поиск по медийным баннерам. В этом
            случае доступны для указания в качестве условий только num, domain, login

    Общие ограничения:

        - limit - сколько записей вернуть.
        - offset - смещение с которого надо начинать возвращать баннеры (не
            работает для медийных баннеров)
        - short - вернуть краткую информацию о баннре - bid, cid, BannerID
        - activeonly - вернуть только активные баннеры
        - include_currency_archived_campaigns - включить в результаты поиска
            кампании из специального архива (сконвертированные из у.е. в реальную валюту)
            по умолчанию, такие кампании включаются в результаты только при поиске
            по номеру баннера

    Группировки:

        - group_camp - группировать по номерам кампаний.

    Специфичные для отдельных условий параметры:

        Фразы:
            - strict_phrase - поиск по строгому соответствию фразы

        Домены:
            - exact_domain - поиск по строгому соответствию домена

    Поддерживаются следующие параметры фильтрации:
        - search_or_net
        - broad_match_flag
        - retargetings
        - strategy
        - geo - RegionID, оставить только баннеры, имеющие таргетинг на этот
            регион или вложенные в него регионы

    Возвращает hashref вида:

        {regions => $banners}

    где $banners - массив найденных баннеров

=cut

#TODO Для медийных баннеров нельзы указывать параметр offset
sub search_banners
{
    my ($criteria, $params) = @_;

    my $filters = hash_cut $params, qw/search_or_net broad_match_flag retargetings strategy geo/;

    my $is_add_filters = any {defined $params->{$_}} keys %$filters;
    my $is_media = $params->{media};
    my $is_domain_search = 0;

    my %used_keys;
    for my $item (@$criteria) {
        die "Empty search key" if !$item->{key};
        die "Unknown key" if !exists $SUPPORTED_SEARCH_KEYS{$item->{key}};
        die "Key $item->{key} is used more than once" if exists $used_keys{$item->{key}};
        die "Unsupported media key $item->{key}" if ($is_media && !$SUPPORTED_SEARCH_KEYS{$item->{key}}->{supports_media});
        die "Empty search values" if !($item->{values} && @{$item->{values}});

        my $validator = $SUPPORTED_SEARCH_KEYS{$item->{key}}->{validator};
        for my $value (@{$item->{values}}) {
            die "Found invalid value for key $item->{key}: $value" unless $validator->($value);
        }

        if ($item->{key} eq 'domain') {
            $is_domain_search = 1;
        }
        $used_keys{$item->{key}} = undef;
    }

    die "Geo should be an id" if (defined $params->{geo} && !is_valid_id($params->{geo}));

    $params->{limit} = undef if $params->{limit} && $params->{limit} !~ /^\d+$/;
    $params->{offset} = undef if $params->{offset} && $params->{offset} !~ /^\d+$/;
    if (!defined $params->{limit} && $params->{group_camp}) {
        $params->{limit} = GROUP_CAMP_LIMIT;
    }
    elsif (!defined $params->{limit}) {
        $params->{limit}= $is_domain_search && !$is_add_filters ? DEFAULT_LIMIT : BANNERS_LIMIT;
    }
    my $where_cond = {};

    for my $item (@$criteria) {
        my $processor = $SUPPORTED_SEARCH_KEYS{$item->{key}}->{processor};
        my $item_where_cond;

        if (defined $processor) {
            $item_where_cond = eval { $processor->($item->{values}, $params) };
            if ($@) {
                warn $@;
                return {banners => [], error => $@};
            }
        } else {
            my $query_key;
            if ($is_media) {
                $query_key = $SUPPORTED_SEARCH_KEYS{$item->{key}}->{media_query_key} // $SUPPORTED_SEARCH_KEYS{$item->{key}}->{query_key};
            } else {
                $query_key = $SUPPORTED_SEARCH_KEYS{$item->{key}}->{query_key};
            }
            die "No search parameters for $item->{key}" if ! defined $query_key;
            $item_where_cond->{$query_key} = $item->{values};
        }

        for my $key (keys %$item_where_cond) {
            if (exists $where_cond->{$key}) {
                # TODO: Сейчас это невозможно, но потенциально сдесь может начать мержиться два _OR
                # Выбирать пересечение в этом случае будет некорректно
                $where_cond->{$key} = xisect($where_cond->{$key}, $item_where_cond->{$key});
            } else {
                $where_cond->{$key} = $item_where_cond->{$key};
            }
        }
    }

    if (! %$where_cond) {
        return {banners => [], error => 'Error generating query'};
    }

    for my $key (keys %$where_cond) {
        if (ref $where_cond->{$key} eq 'ARRAY') {
            # Если при поиске какое-то условие схлопнулось - значит мы ничего не нашли
            return {banners => []} if !@{$where_cond->{$key}};
        }
    }
    $where_cond->{'c.statusEmpty'} = 'No';

    my $banners;
    if ( !$is_media ) {
        $banners = _do_banners_search( $params, $where_cond);

        # DIRECT-24853: если искали по номеру баннера и ничего не нашли - попытаемся поискать по номеру картиночного баннера
        # Работает только в случае, если производился поиск только по номеру баннера
        if (scalar @$criteria == 1 && $criteria->[0]->{key} eq 'num' && !$is_media) {
            my @bids = @{$where_cond->{'b.bid__int'} // []};
            unless (@$banners && @$banners == scalar @bids) {
                $where_cond->{'bim.image_id__int'} = delete $where_cond->{'b.bid__int'};
                push @$banners, @{_do_banners_search( $params, $where_cond) // []};
                $banners = [ xuniq { $_->{bid} } @$banners ];
            }
        }

        my @banners_with_day_budget = grep {$_->{day_budget} && $_->{day_budget} > 0 && $_->{OrderID} && $_->{OrderID} > 0} @$banners;
        my $spent_today = Stat::OrderStatDay::get_order_spent_today_multi([uniq map { $_->{OrderID} } @banners_with_day_budget]);
        $_->{spent_today} = $spent_today->{ $_->{OrderID} } for @banners_with_day_budget;

    } else {
        $banners = _do_media_banners_search($params, $where_cond);
    }

    if (! $params->{short}) {
        _add_info($banners, $params);
    }

    # если пришёл хотя бы один параметр по которому следует фильтровать
    if ($is_add_filters) {
        $banners = _filter_result($banners, $filters, is_domain_search => $is_domain_search);
    }

    if ($params->{short}) {
        $banners = [map {hash_cut $_, qw/bid cid BannerID/} @$banners];
    }

    return {banners => $banners};
} ## end sub searchBanners

=head3 _add_info

    Добавить к баннерам в переданном списке информацию об уточнениях и сайтлинках

=cut

sub _add_info {
    my ($banners, $params) = @_;
    # заполняем уточнения
    Direct::BannersAdditions::add_callouts_to_banners($banners);
    # заполняем сайтлинки
    my $sitelinks_sets = Sitelinks::get_sitelinks_by_set_id_multi([
            uniq
            map {$_->{sitelinks_set_id}}
            grep {defined $_->{sitelinks_set_id}}
            @$banners
        ]);
    my $turbolandings_by_id = _get_turbolandings($banners);
    for my $banner (@$banners) {
        $banner->{turbolanding} = $turbolandings_by_id->{$banner->{turbolanding_id}} if defined $banner->{turbolanding_id};

        if (exists $params->{set_prices} && exists $params->{set_prices}->{$banner->{bid}}) {
            $banner->{price} = $params->{set_prices}->{$banner->{bid}};
        }
        $banner->{sitelinks} = $banner->{sitelinks_set_id} ? $sitelinks_sets->{$banner->{sitelinks_set_id}} : [];
    }
}

sub _get_turbolandings {
    my ($banners) = @_;

    my $tl_id_by_client_id;
    foreach my $bnr (@$banners) {
        next unless $bnr->{turbolanding_id};
        push @{$tl_id_by_client_id->{$bnr->{ClientID}}}, $bnr->{turbolanding_id};
    }

    my $turbolandings_by_id = {};
    if (keys %$tl_id_by_client_id) {
        my $cache;
        my @columns = Direct::Model::TurboLanding->get_db_columns_list('turbolandings');

        foreach my $chunk (sharded_chunks(ClientID => [keys %$tl_id_by_client_id])){
            my @desired_tl_ids = map {@$_} @$tl_id_by_client_id{@{$chunk->{ClientID}}};
            my $turbolandings = get_all_sql(PPC(shard => $chunk->{shard}), [
                 'SELECT '.join(', ', @columns), FROM => 'turbolandings',
                 WHERE => { tl_id => \@desired_tl_ids }
            ]);

            $turbolandings_by_id->{$_->{tl_id}} =  Direct::Model::TurboLanding->from_db_hash($_, \$cache)->to_template_hash()
                foreach (@$turbolandings);
        }
    }

    return $turbolandings_by_id;
}

=head3 _filter_result

    Вернуть отфильтрованный список баннеров

=cut

sub _filter_result {
        my ($banners, $filters, %O) = @_;

        if (defined $filters->{geo}) {
            $banners = [grep {is_targeting_include_region($_->{geo}, $filters->{geo})} @$banners];
        }

        my @bids;
        foreach my $banner (@$banners) {
            my @ids = $banner->{bids} ? split(/,/, $banner->{bids}) : ($banner->{bid});
            push @bids, @ids;
            $banner->{bids} = \@ids;
        }

        # условие ретаргетинга берётся первое попавшееся, т.к. важно только то, что условий > 0
        my $select = qq{
            SELECT
                b.bid,
                co.cid,
                br.bid retargetings
            FROM
                banners b 
                join camp_options co on co.cid = b.cid
                left join bids_retargeting br on br.pid = b.pid
        };

        my %where;
        $where{'b.bid'} = SHARD_IDS;
        
        if(defined $filters->{broad_match_flag}){
            # будем выбирать баннеры только с ДРФ
            $where{'co.broad_match_flag'} = $filters->{broad_match_flag} ? 'Yes' : 'No';
        }
        
        my $db_banners = get_hashes_hash_sql(PPC(bid => \@bids), [$select, where => \%where, "group by b.bid"]);
        # если нужно, то учитываем количество условий ретаргетинга на баннере
        if(defined $filters->{retargetings}){
            for my $bid (keys %$db_banners) {
                if($filters->{retargetings} && !$db_banners->{$bid}->{retargetings}){
                    delete $db_banners->{$bid};
                } elsif(!$filters->{retargetings} && $db_banners->{$bid}->{retargetings}){
                    delete $db_banners->{$bid};
                }
            }
        }

        unless(%$db_banners){
            return [];
        }

        my $delete_cids = {};
        if ((defined $filters->{'strategy'} && $filters->{'strategy'} ne 'no') || $filters->{'search_or_net'}){

            my @cids = uniq map {$_->{cid}} values %$db_banners;
            my $sn = mass_campaign_strategy(\@cids);

            my %search_campaigns;
            if ($filters->{'search_or_net'} && $filters->{'search_or_net'} eq 'search') {
                # кампании только на поиске
                my $search_cids = get_one_column_sql(PPC(cid => \@cids),
                                    ["SELECT cid FROM campaigns", WHERE => {cid => SHARD_IDS, platform => 'search'}]);
                $search_campaigns{$_} = 1 foreach @$search_cids;
            }

            for my $cid (keys %$sn){

                # фильтрация только по поиску или только по РСЯ
                if ($filters->{'search_or_net'} && $filters->{'search_or_net'} eq 'net') {
                    # независимое управление (показы на поиске отключены)
                    if ($sn->{$cid}->{platform} ne 'context') {
                        $delete_cids->{$cid} = 1;
                        next;
                    }
                } elsif ($filters->{'search_or_net'} && $filters->{'search_or_net'} eq 'search') {
                    if (!$search_campaigns{$cid}) {
                        $delete_cids->{$cid} = 1;
                        next;
                    }
                }

                # фильтрация по названию стратегии
                if ($filters->{'strategy'} && $filters->{'strategy'} ne 'no') {
                    my $camp_strategy_name = campaign_strategy_to_strategy_name($sn->{$cid});

                    if($camp_strategy_name ne $filters->{strategy}){
                        $delete_cids->{$cid} = 1;
                    }
                }
            }
        }

        if(%$delete_cids){
            for my $bid (keys %$db_banners){
                if(exists $delete_cids->{$db_banners->{$bid}->{cid}}){
                    delete $db_banners->{$bid};
                }
            }
        }

        $banners = [grep { any { exists $db_banners->{$_} } @{$_->{bids}} } @$banners];
        if ($O{is_domain_search}) {
            $banners = [splice @$banners, 0, min(DEFAULT_LIMIT, scalar @$banners)];
        }

        return $banners;
}

=head2 convert_old_style_params($old_params)

    Сконвертировать параметры старого стиля search_banners в новый стиль

    Возвращает $criteria (условия поиска) и $params (параметры запуска)

=cut

sub convert_old_style_params {
    my ($old_params) = @_;

    my $advanced_params = hash_cut $old_params, @ALLOWED_PARAMS;
    my @values;
    if (none {$_ eq $old_params->{what}} qw/phrase text login domain domain_media turbolanding_name_part turbolanding_url/) {
        push @values, grep { $_ ne ''} split /\D+/, $old_params->{text_search};
    } else {
        push @values, grep { $_ ne ''} split/\s*,\s*/, $old_params->{text_search};
    }

    my $key = $old_params->{what};
    if ($old_params->{what} =~ /^(\w+)_media$/) {
        $advanced_params->{media} = 1;
        $key = $1;
    }
    my $criteria = [{
        key => $key,
        values => \@values
    }];

    return $criteria, $advanced_params;
}

1;
