use Direct::Modern;

=head1 NAME

SentryTools -- работа с Sentry (в реалиях Яндекса - с ErrorBooster)

=cut


package SentryTools;

use List::MoreUtils qw/any/;
use File::Basename;
use Devel::StackTrace;
use POSIX qw/floor/;
use Time::HiRes;

use Yandex::Trace;

use LogTools qw//;
use EnvTools qw//;
use Settings;

our $LAST_STACKTRACE;
our $LAST_ERROR;
our $LAST_CTX;

our @FINGERPRINTS = (
    qr/External templating error/,
    qr/in \S+: 2006, MySQL server has gone away/,
    qr/http_request\(http\S+/,
    qr/http_fetch\(\S+\s+=>\s+http\S+/,
    qr/error in call_intapi_java\('[^']+'\):/,
    qr/BB fatal error:/,
    qr/advq_get_time_coef error:/,
    );

my $UNKNOWN = 'unknown';


=head2 prepare_die_handler

    обнулить последний пойманный stacktrace, подготовить новый колбек для захвата stacktrace

=cut
sub prepare_die_handler {
    $LAST_STACKTRACE = undef;
    $LAST_ERROR = undef;
    $LAST_CTX = undef;
    return sub {
        $LAST_ERROR = shift;
        $LAST_STACKTRACE = Devel::StackTrace->new(ignore_package => ['SentryTools', 'Carp']);
        $LAST_CTX = LogTools::get_context();
    };
}


=head2 get_last_stacktrace

    получить последний пойманный stacktrace в текстовом виде

=cut
sub get_last_stacktrace {
    if ($LAST_STACKTRACE) {
        return $LAST_STACKTRACE->as_string;
    } else {
        return "";
    }
}


=head2 get_last_error

    получить сообщение от последнего пойманного die

=cut
sub get_last_error {
    return $LAST_ERROR // "";
}


=head2 send_last_exception

   записываем в лог событие с последним пойманным stacktrace

=cut
sub send_last_exception {
    my ($msg, %opt) = @_;

    state $logger;
    $logger //= Yandex::Log->new(
        use_syslog => 0,
        log_file_name => "error_booster-perl.log",
        no_date => 1,
        date_suf => '%Y%m%d',
        lock => 1,
        auto_rotate => 1,
    );

    my $trace = Yandex::Trace::current_trace();
    my $stacktrace = $LAST_STACKTRACE;

    my $ctx = $SentryTools::LAST_CTX;

    my $filtered_stacktrace = _filter_unwanted_frames($stacktrace, %opt);
    my $last_frame = @$filtered_stacktrace ? $filtered_stacktrace->[$#$filtered_stacktrace] : undef;

    my $fingerprint = _calc_fingerprint_for_error_booster($msg, $last_frame, $trace);
    my %rec = (
        timestamp   => floor(Time::HiRes::time() * 1000),
        host        => $Settings::BETA_HOST // $EnvTools::hostname,
        dc          => EnvTools::get_dc(),
        project     => 'direct',
        message     => $msg,
        reqid       => '' . ($trace ? $trace->span_id // 0 : 0),
        language    => 'perl',
        service     => ($trace ? $trace->service : $UNKNOWN),
        source      => _get_source($trace),
        method      => ($trace ? $trace->method : $UNKNOWN),
        #В parsed_stacktrace отдадим облегченный stacktrace, без утилитарных модулей и оберток.
        #Привычный perl'овый stacktrace положим в addtional
        parsed_stacktrace   => $stacktrace ? _prepare_stacktrace($filtered_stacktrace) : [],
        env         => _calc_environment(),
        fingerprint => $fingerprint,
        yandexuid   => '' . ($ctx ? $ctx->{UID} // 0 : 0) // 0,
        version     => _calc_direct_version(),
        additional  => _calc_additional($stacktrace, $fingerprint, $msg, $last_frame, $trace),
    );

    $logger->out(\%rec);
}

sub _get_source {
    my ($trace) = @_;

    my $source;

    if ($trace){
        $source = $trace->service // '';
        $source =~ s/^direct\.//;
    }
    if ($source eq '' || $source eq 'direct.script') {
        $source = basename($0);
    }

    return $source;
}

sub _prepare_stacktrace {
    my ( $stacktrace ) = @_;
    my @trace;
    foreach my $frame (@$stacktrace){
        my $data = {
            lineno => $frame->{line},
            filename => $frame->{filename},
        };
        $data->{function} = $frame->{subroutine} if ($frame->{subroutine} ne 'SentryTools::__ANON__');
        if ($data->{function} && $frame->{hasargs}) {
            $data->{function} .= '(' . join(', ', @{_get_normalized_args($frame->{args})}) . ')';
        }
        push @trace, $data;
    }
    return \@trace;
}

sub _get_normalized_args {
    my $args = shift // [];
    $args = [$args] unless ref $args;

    my @result;
    foreach my $arg (@$args) {
        my $r;
        if (!defined $arg) {
            $r = 'NULL';
        } elsif (ref $arg) {
            $r = ref $arg;
        } elsif ($arg =~ /^(\-)?\d+(\.\d+)?$/){
            $r = 'NUMBER';
        }
        else {
            $r = 'TEXT'
        }
        push @result, $r;
    }

    return \@result;
}

sub _calc_direct_version {
    my $version;

    $version = eval { EnvTools::get_current_direct_version() };

    return $version // '0.0';
}

sub _calc_additional {
    my ($stacktrace, $fingerprint, $msg, $last_frame, $trace) = @_;

    my $root_frame = $stacktrace ? $stacktrace->frame(0) : undef;
    my $additional = {
        fingerprint => $fingerprint,
        thrownClass => _extract_frame_package($last_frame),
        rootCauseClass => _extract_frame_package($root_frame),
        rootCauseMethod => _extract_root_method($stacktrace) || $UNKNOWN,
        rootCauseNormalizedMessage => _normalize_message($msg),
        classicStacktrace => $stacktrace ? $stacktrace->as_string : '',
    };

    if ($trace) {
        $additional->{trace_id}  = $trace->trace_id() if $trace->trace_id();
        $additional->{span_id}   = $trace->span_id()  if $trace->span_id();
        $additional->{parent_id} = $trace->parent_id() if $trace->parent_id();
    }

    return $additional;
}

sub _calc_environment {
    my $env = lc( $Settings::CONFIGURATION // $UNKNOWN );
    return $env eq 'test' ? 'testing' : $env;
}

sub _calc_fingerprint_for_error_booster {
    my ($msg, $last_frame, $trace) = @_;

    my $stacktrace_fingerprint = _extract_package_and_subroutine($last_frame, $trace ? $trace->method : '');

    return _normalize_message($msg).'|'.$stacktrace_fingerprint;
}

sub _extract_root_method {
    my ($stacktrace) = @_;

    return '' unless $stacktrace;
    my $method = $stacktrace->frame(0)->{subroutine} // '';

    #В нулевом фрейме subroutine обычно SentryTools::__ANON__,
    #в этом случае реальный метод в котором было падение будет в первом фрейме
    if ($method =~/^SentryTools/ && $stacktrace->frame_count() > 1){
        $method = $stacktrace->frame(1)->{subroutine} // '';
    }

    return $method;
}

sub _filter_unwanted_frames {
    my ($stacktrace, %opt) = @_;

    return [] unless $stacktrace;

    my @result;
    foreach my $frame ($stacktrace->frames()) {
        next if $opt{skip} && (any {$frame->{package} eq $_} @{$opt{skip}});
        next if $opt{skip_re} && (any {$frame->{package} =~ $_} @{$opt{skip_re}});
        push @result, $frame;
    }

    #Если stacktrace был непустой - хотя бы один фрейм вернем
    return @result ? \@result : [$stacktrace->frame(0)];
}

sub _extract_frame_package {
    my ($frame) = @_;

    return $UNKNOWN unless $frame;
    return $frame->{package} eq 'main' ? $frame->{filename} : $frame->{package};
}

sub _extract_package_and_subroutine {
    my ($frame, $default_method) = @_;

    my $package = _extract_frame_package($frame);
    my $subroutine = $frame->{subroutine} // '';
    $subroutine = '' if $subroutine =~/^SentryTools::/;
    $subroutine ||= $default_method;

    return $subroutine ne '' ? $subroutine.'@'.$package : $package;
}

sub _normalize_message {
    my ($msg) = @_;

    #убираем из сообщения текст, относящийся к месту падения
    $msg =~ s|\s+at\s+/\S+\s+line\s+\d+\.?\n?$||;

    for my $re (@FINGERPRINTS) {
        if ($msg =~ /($re)/) {
            return $1;
        }
    }

    $msg =~ s/\b0x[a-f0-9]{12}\b/0xHEX12/g;
    $msg =~ s/\b[a-f0-9]{16}\b/HEX16/g;
    $msg =~ s/\b[0-9]{4,}\b/NUM/g;
    $msg =~ s/_[0-9]{4,}_/_NUM_/g;
    $msg =~ s/Incorrect ids: (.*?) at /Incorrect ids: TEXT at /g;

    return $msg;
}

1;

package SentryTools::UserProcessor;

sub new {
    bless {};
}

=head2 process

    Дописать в событие UID оператора из последнего контекста.
    Если в событии есть поле $e->{extra}{customUserId}, в качестве ID оператора используется оно.
    Поле $e->{extra}{customUserId} при этом удаляется.

=cut
sub process {
    my ($self, $e) = @_;
    my $ctx = $SentryTools::LAST_CTX;
    if ($e->{extra}{customUserId}) {
        $e->{user} = {id => delete $e->{extra}{customUserId}};
    } elsif ($ctx && $ctx->{UID}) {
        $e->{user} = {id => $ctx->{UID}};
    }
    return $e;
}

1;
