use warnings;
use strict;

=head NAME

    ShardingTools -- простые функции про шардинг

=head1 DESCRIPTION 

=cut

package ShardingTools;

use parent qw/Exporter/;

use Carp;
use Data::Dumper;
use List::Util qw/sum/;
use List::MoreUtils qw/uniq all/;

use Yandex::ListUtils qw/weighted_xshuffle/;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::Ketama;

use Settings;

our @EXPORT = qw(
    get_new_available_shard
    ppc_shards
    choose_shard_param

    overshard_get_one_line
    overshard_sum

    foreach_shard_parallel_verbose
);

our @EXPORT_OK = qw(iterate_table_parallel);

=head2 $DEFAULT_MAPPING_SHARD_KEYS

    Список ключей шардинга, которые можно использовать для определения шарда.
        * отсортирован по убыванию предпочтительности использования

=cut
our $DEFAULT_MAPPING_SHARD_KEYS ||= [qw/
    ClientID
    uid
    cid
    OrderID
    pid
    bid
    mbid
    BannerID
    tag_id
    sitelinks_set_id
    ret_cond_id
/];


=head2 get_new_available_shard

    Возвращает номер шарда, в который лучше всего на данный момент записать нового клиента.

    Параметры:
    ClientID - id клиента
    operator_uid - uid оператора, создающего клиента (агентство, менеджер, или сам клиент),
                   нужно для более оптимального размещения клиента
                   может отсутствовать

=cut
sub _is_shard_alive($) {
    eval {get_dbh(PPC(shard => $_[0]))->ping()};
}
sub _choose_consistent_alive_shard {
    my ($shards, $weights, $key) = @_;
    return undef if ! @$shards;
    
    my $k = Yandex::Ketama->new(servers => [map {+{name => $_, weight => $weights->{$_}}} @$shards]);
    for my $shard ($k->find_all($key)) {
        return $shard if _is_shard_alive($shard);
    }

    return undef;
}
sub _get_new_available_shard_old($$) {
    my ($ClientID, $operator_uid) = @_;

    my %weights = map {$_ => get_db_config(PPC(shard => $_))->{weight} || 0} ppc_shards();

    # пытаемся поместить клиента в шард оператора, если он не закрыт на запись (вес 0)
    my ($operator_client_id, $operator_shard);
    if ($operator_uid) {
        $operator_client_id = get_shard(uid => $operator_uid, 'ClientID');
        $operator_shard = get_shard(ClientID => $operator_client_id);
        return $operator_shard if $operator_shard && $weights{$operator_shard};
    }

    my $consistent_key = $operator_client_id || $operator_uid || $ClientID;

    # пытаемся поместить в живой шард, открытый на запись
    # пропорционально весу и используя consistent-hashing алгоритм по consistent_key
    my $shard = _choose_consistent_alive_shard([grep {$weights{$_}} ppc_shards()], \%weights, $consistent_key);
    return $shard if defined $shard;

    # если нет живых шардов, открытых на запись - возвращаемcя к идее поместить клиента в шард оператора
    return $operator_shard if defined $operator_shard;

    # закрытый, но живой шард
    $shard = _choose_consistent_alive_shard([grep {!$weights{$_}} ppc_shards()], \%weights, $consistent_key);
    return $shard if defined $shard;    

    # всё пропало - выбираем случайных шард с учётом веса
    return [weighted_xshuffle {$weights{$_}} ppc_shards()]->[0];
}
sub get_new_available_shard($$) {
    my ($ClientID, $operator_uid) = @_;

    # на оператора смотреть вредно, это рождает перекосы
    my %weights = map {$_ => get_db_config(PPC(shard => $_))->{weight} || 0} ppc_shards();
    return _choose_consistent_alive_shard([grep {$weights{$_}} ppc_shards()], \%weights, $ClientID)
        // [weighted_xshuffle {$weights{$_}} ppc_shards()]->[0];
}


=head2 ppc_shards

    Получить список доступных номеров шардов PPC

=cut
sub ppc_shards {
    return 1..$Settings::SHARDS_NUM;
}


=head2 choose_shard_param

    Выбрать из хеша %$params первый непустой параметр, в порядке, указанном вторым параметром.
        Функция понимает расширенные имена параметров (имя_таблицы.имя_параметра)
        Например ключ 'p.cid' из %$params будет воспринят как 'cid'
        Если второй параметр не определен, будет использован список $DEFAULT_MAPPING_SHARD_KEYS
    Именованные параметры:
        allow_shard_all - в случае, если шард не был вычислен, то возвращать shard => 'all'
        set_shard_ids - изменить $params (==$where), установить выбранное поле равное SHARD_IDS

    my @shard = choose_shard_param(\%params, [qw/cid bid pid uid/], allow_shard_all => 1);
    get_all_sql(PPC(@shard), [ "select ...", where => \%params ]);
    my @shard2 = choose_shard_param(\%params2, undef, allow_shard_all => 0);

=cut
sub choose_shard_param {
    my ($params, $keys, %O) = @_;
    $keys ||= $DEFAULT_MAPPING_SHARD_KEYS;
    my %is_key_for_sharding = map { $_ => 1 } @$keys;

    # делаем из { 'b.bid' => 123,  '`c`.`cid`' => 456 } => { bid => 123, cid => 456 }
    my %aliases;    # хеш вида bid => 'b.bid'
    foreach my $param (keys %$params) {
        my $name = $param;
        $name =~ s/^.*\.//; $name =~ s/\`//g;
        if (!exists $aliases{$name}) {
            $aliases{$name} = $param;
        } else {
            if ($is_key_for_sharding{$name}) {
                die "Key $name in params is ambiguous: ".(join ",", keys %$params);
            }
        }
    }

    foreach my $key (@$keys) {
        if ($aliases{$key} && $params->{ $aliases{$key} }) {
            my @shard = ( $key => $params->{ $aliases{$key} } );
            if ($O{set_shard_ids}) {
                $params->{ $aliases{$key} } = SHARD_IDS;
            }
            return @shard;
        }
    }

    if ($O{allow_shard_all}) {
        return (shard => 'all');
    } else {
        die Carp::longmess('Can not choose shard param. Where: ' . Dumper($params) . "\nkeys: " . (join ',', @$keys));
    }

}


=head2 overshard_get_one_line(\%overshard_opt, @get_all_sql_params)

    Выполнить запрос в базу, произвести над результатом overshard и вернуть первую запись
    my $min_max = overshard_get_one_line({ group => '', min => 'min_bid', max => 'max_bid' },
        PPC(shard => 'all'),
        "select min(bid) as min_bid, max_bid as max_bid from banners"
    );
    # $min_max = { min_bid => ..., max_bid => ... }

=cut
sub overshard_get_one_line
{
    my ($overshard_opt, @sql_param) = @_;
    return ( overshard %$overshard_opt,  get_all_sql( @sql_param ) )->[0];
}


=head2 overshard_sum

    Просуммировать значения по всем шардам:
    my $count = overshard_sum(PPC(shard=>'all'), "select count(*) from users");

=cut
sub overshard_sum
{
    return sum @{get_one_column_sql( @_ ) || []};
}

=head2 foreach_shard_parallel_verbose

ДОПУСТИМО ИСПОЛЬЗОВАТЬ ТОЛЬКО ИЗ СКРИПТОВ (т.к. недопустимо форкать apache)

Обычный foreach_shard_parallel тихо глотает ошибки возникшие при обработке шарда, в это обёртке мы их явно
логгируем.

    my $logger = Yandex::Log->new(...);
    foreach_shard_parallel_verbose($logger, sub {
        my ($shard) = @_;
    });

=cut
sub foreach_shard_parallel_verbose {
    my ($logger, $shard_sub) = @_;

    my $result = foreach_shard_parallel(
        shard => [ppc_shards()],
        sub {
            my $shard = shift;
            my $ret = eval {
                $logger->msg_prefix("[shard $shard] ");
                $shard_sub->($shard);
                return 1;
            };
            unless ($ret) {
                my $err = $@;
                $logger->out($err);
                die $err;
            }
        }
    );

    my @shards_with_error = grep { ! all { $_ } @{$result->{$_}} } keys %$result;

    if (@shards_with_error) {
        my $msg = "Error in shards: " . join ', ', @shards_with_error;
        $logger->die($msg);
        die $msg;
    }
}


1;
