package Sitelinks;

# $Id$

=head1 NAME
    
    Sitelinks 

=head1 DESCRIPTION

    модуль для функций, относящихся к сайтлинкам (блок из не более чем $SITELINKS_NUMBER ссылок для спецразмещения)
    сайтлинки объединяются в сеты по 1-$SITELINKS_NUMBER штуки, баннер имеет привязку именно к сету

=cut

use strict;
use warnings;
use feature 'state';

use List::MoreUtils qw/all any/;
use List::Util qw/sum/;
use JSON;

use Yandex::HashUtils;
use Yandex::ScalarUtils qw/str/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Overshard;
use Yandex::I18n;
use Yandex::URL;
use Settings;
use HashingTools;
use URLDomain;

use Direct::Model::SitelinksSet;
use Direct::Model::Sitelink;
use Direct::Model::TurboLanding::Sitelink;

use Direct::Validation::SitelinksSets qw/$SITELINKS_NUMBER $SITELINKS_MAX_LENGTH $ONE_SITELINK_MAX_LENGTH $ONE_SITELINK_DESC_MAX_LENGTH/;

use base qw/Exporter/;
our @EXPORT_OK = qw/$SITELINKS_NUMBER $SITELINKS_MAX_LENGTH $ONE_SITELINK_MAX_LENGTH $ONE_SITELINK_DESC_MAX_LENGTH get_sitelinks_by_set_id_multi/;

use utf8;

=head2 get_sitelinks_by_set_id

    возвращаем сайтлинки в сете:
    get_sitelinks_by_set_id( $set_id )
    [ 
        { 
            sl_id  => ...,
            title  => ...,
            description => ...,
            href   => ...,
            hash   => ...,
        },
    ]

=cut

sub get_sitelinks_by_set_id
{
    my $set_id = shift;
    return [] unless $set_id;
    return get_sitelinks_by_set_id_multi([$set_id])->{ $set_id || '' } || [];
}

{
my $sql_table_alias = 'stl';
my $sql_table_key = 'sitelinks_set_id';
my $sql_select = qq!SELECT $sql_table_alias.$sql_table_key, $sql_table_alias.order_num, $sql_table_alias.sl_id, title, description, sl.href, hash,
                    tl.tl_id as tl_tl_id, tl.name as tl_name, tl.href as tl_href
                    FROM
                        sitelinks_set_to_link $sql_table_alias
                        JOIN sitelinks_links sl ON ( sl.sl_id = $sql_table_alias.sl_id )
                        LEFT JOIN turbolandings tl ON ( sl.tl_id = tl.tl_id)
                    !;
my $where_key = "${sql_table_alias}.${sql_table_key}__int";
my @sql_order_by = qw/sitelinks_set_id order_num/;

=head2 get_sitelinks_by_set_id_multi(\@set_ids)

    возвращаем сайтлинки в сете:
    get_sitelinks_by_set_id_multi( [$set_id1, $set_id2, ...] )
    { 
        $set_id1 => [
            {
                sl_id  => ...,
                title  => ...,
                description => ...,
                href   => ...,
                hash   => ...,
            },
            ...
        ],
        ...
    }

=cut

sub get_sitelinks_by_set_id_multi
{
    my $set_ids = shift;
    return {} if !@$set_ids;

    my $links = overshard order => \@sql_order_by,
        get_all_sql(PPC($sql_table_key => $set_ids), [
            $sql_select,
            WHERE => { $where_key => SHARD_IDS },
        ]);

    return _get_sitelinks_dict_from_db_array($links);
}

=head2 get_sitelinks_by_set_id_multi_without_sharding($dbh, \@set_ids)

    Возвращает сайтлинки в сете, в формате как get_sitelinks_by_set_id_multi
    Но в отличие от нее не использует метабазу, и делает выборку данных только из $dbh

=cut

sub get_sitelinks_by_set_id_multi_without_sharding {
    my ($dbh, $set_ids) = @_;
    return {} if !@$set_ids;

    my $links = get_all_sql($dbh, [$sql_select, WHERE => { $where_key => $set_ids }, 'ORDER BY', join(',', @sql_order_by)]);
    return _get_sitelinks_dict_from_db_array($links);
}

=head3 _get_sitelinks_dict_from_db_array

    Внутренняя функция, предназначенная для приведения массива выбранных из базы линков в словарь сетов сайтлинков

=cut

sub _get_sitelinks_dict_from_db_array {
    my $links = shift;
    my %ret;
    my $cache;
    for my $link (@$links) {
        $link->{href} = clear_banner_href($link->{href});
        my $sl = hash_cut $link, qw/sl_id title description href hash/;
        if ($link->{tl_tl_id}) {
            $sl->{turbolanding} = Direct::Model::TurboLanding::Sitelink->from_db_hash(
                $link, \$cache, prefix => 'tl_'
            )->to_template_hash();
        }
        
        push @{$ret{$link->{sitelinks_set_id}}}, $sl;
    }
    return \%ret;
}
}

=head2 save_sitelinks_set

    сохраняем сайтлинки с данными параметрами
    save_sitelinks_set( [ { title => ..., description => ..., href => ..., }, ... ], $client_id )

    ищем, есть ли уже такой же set, если нет, то создаем

    возвращаем sitelinks_set_id или undef, если не смогли сохранить

=cut

sub save_sitelinks_set
{
    my ($sitelinks_set, $client_id) = @_;

    my @sitelinks_ids = map { _save_sitelink($_, $client_id) } @$sitelinks_set;
    if (my $sitelinks_set_id = _get_sitelinks_set_id(\@sitelinks_ids, $client_id)) {
        return $sitelinks_set_id;
    } else {
        my $sitelinks_set_id = get_new_id('sitelinks_set_id', ClientID => $client_id);
        do_insert_into_table(PPC(ClientID => $client_id), 'sitelinks_sets', 
                             {sitelinks_set_id => $sitelinks_set_id, ClientID => $client_id, links_hash => _calc_sitelinks_set_links_hash(\@sitelinks_ids)});

        my $order_num = 0;
        do_mass_insert_sql(PPC(ClientID => $client_id)
                           , 'INSERT IGNORE INTO sitelinks_set_to_link (sitelinks_set_id, sl_id, order_num) VALUES %s'
                           , [map { [$sitelinks_set_id, $_, $order_num++] } @sitelinks_ids]
                          );
        return $sitelinks_set_id;
    }
}


=head2 need_save_sitelinks_set

    проверяем, надо ли сохранить сет (были ли введены данные)
    
    need_save_sitelinks_set( [ { title => ..., description => ..., href => ..., }, ... ] )

=cut

sub need_save_sitelinks_set($)
{
    my $sitelinks_set = shift;
    return ref($sitelinks_set) eq 'ARRAY'
           && scalar(@$sitelinks_set) > 0
           && ( all {
                   (defined $_->{title} && $_->{title} ne '')
                   && (defined $_->{href} && $_->{href} ne '')
               } @$sitelinks_set
              )
           ? 1
           : 0;
}

=head2 validate_sitelinks_set

    @errors = validate_sitelinks_set($sitelinks_set, $banner_href);
    $sitelinks_set -- ссылка на массив сайтлинков [{href => '...', title => '...', description => '...'}, ...]
    $banner_href -- основная ссылка баннера

    при валидации сайтлинков проверяем: 
        все сайтлинки заполнены (либо все пустые, в этом случае игнорируем)
        суммарная длина заголовков не превышает $SITELINKS_MAX_LENGTH символов
        корректность формата ссылок
        длина url ссылок не превышает допустимую
        в тексте ссылок используются только буквы латинского, русского, украинского или казахского алфавита, цифры, дефис
        запрещаем дубликаты в текстах и адресах ссылок
    то, что все ссылки ведут на основной сайт, пока не проверяем

=cut

sub validate_sitelinks_set($$;%) {
    my $sitelinks_set = shift;
    my $href = shift; # основная ссылка объявления
    my %OPT = @_;
    my $banner_turbolanding = $OPT{banner_turbolanding};
    my $client_id = $OPT{ClientID};
    # Список ошибок
    my @result;

    # Валидация структуры
    for my $sitelink (@$sitelinks_set) {
        if (defined $sitelink->{title} && length($sitelink->{title}) > $ONE_SITELINK_MAX_LENGTH) {
            push @result, iget("Превышена допустимая длина текста одной быстрой ссылки в %s символов", $ONE_SITELINK_MAX_LENGTH);
        } elsif (!defined $sitelink->{title}) {
            push @result, iget('Не указан заголовок быстрой ссылки');
        }

        if (defined $sitelink->{href} && length($sitelink->{href}) > 1024) {
            push @result, iget("Превышена допустимая длина адреса одной быстрой ссылки в %s символов", 1024);
        } elsif (!defined $sitelink->{href}) {
            push @result, iget('Не указан адрес быстрой ссылки') if !$sitelink->{turbolanding};
        }

        if (defined $sitelink->{description} && $sitelink->{description} =~ /^\s*$/) {
            push @result, iget("Не указан текст описания быстрой ссылки");
        } elsif (defined $sitelink->{description} && length($sitelink->{description}) > $ONE_SITELINK_DESC_MAX_LENGTH) {
            push @result, iget("Превышена допустимая длина описания одной быстрой ссылки в %s символов", $ONE_SITELINK_DESC_MAX_LENGTH);
        }
    }

    # Для валидации данных нужна валидная структура
    return @result if @result;

    my $sitelinks_set_model = Direct::Model::SitelinksSet->new(
        client_id => $client_id,
        links => [map { Direct::Model::Sitelink->new(
            title => $_->{title},
            href => $_->{href},
            description => $_->{description},
            ($_->{turbolanding} ? (turbolanding => {%{$_->{turbolanding}}, client_id => $client_id}) : ()),
        ) } @$sitelinks_set],
    );

    my $validation = Direct::Validation::SitelinksSets::validate_sitelinks_sets([$sitelinks_set_model]);

    push @result, @{$validation->get_error_descriptions} if !$validation->is_valid;

    return @result;
}


=head2 get_diff_string

    Получаем строку для вычисления диффов и использования в письме

=cut

sub get_diff_string
{
    my $sitelinks_set = shift;

    my @result = ();
    for my $sitelink (@$sitelinks_set) {
        my $description = defined $sitelink->{description} ? "($sitelink->{description}) " : "";
        push @result, iget("Доп. ссылка: ") . $sitelink->{title} . " $description" . $sitelink->{href} if $sitelink->{href};
    }
    return join "\n", @result;
}

=head2 get_clean_sitelinks_list

    Возвращает структуру сайтлинков только с информационными полями

=cut
sub get_clean_sitelinks_list($) {
    my $list = shift or return undef;
    
    my @result;
    
    foreach my $link (@$list) {
        next unless all {defined $link->{$_} && $link->{$_} ne ''} qw/title href/;
        my $clean_link = hash_cut $link, qw/title href description/;
        foreach  (qw/tl_id turbolanding/){
            next unless defined $link->{$_};
            #переводим все значения в текстовые, чтобы в json не было разницы между "0" и 0.
            $clean_link->{$_} = ref $link->{$_} ? $link->{$_} : ''.$link->{$_};
        }
        push @result, $clean_link;
    }
        
    return \@result;
}

=head2 serialize

    Обратимое (по информационным полям) преобразование в строку

=cut
sub serialize($) {
    my $sitelinks_set = shift // [];

    state $json = JSON->new->canonical->allow_nonref;
    return $json->encode(get_clean_sitelinks_list($sitelinks_set));
}


=head2 compare_sitelinks($vcard1, $vcard2)
    
    Сравнить 2 набора быстрых ссылок.

    Возвращает 
        0 Быстрые ссылки одинаковы
        1 Быстрые ссылки разные

=cut
sub compare_sitelinks($$) {
    my ($sitelink1, $sitelink2) = @_;

    return serialize($sitelink1) ne serialize($sitelink2);
}

=head2 get_sitelinks_form($form, $suffix)

    Извлекаем из формы структуру для сета
    В форме ищем поля 
        'sitelinks_0_title' . $suffix
        'sitelinks_0_href'  . $suffix
        'sitelinks_0_description'  . $suffix
        'sitelinks_1_title' . $suffix
        'sitelinks_1_href'  . $suffix
        'sitelinks_2_title' . $suffix
        'sitelinks_2_href'  . $suffix
    Отдаем [ { title => ..., href => ..., description => ...}, ... ]

=cut

sub get_sitelinks_form
{
    my $form   = shift || return;
    my $suffix = shift || '';

    my $sitelinks = [
        map {
            {
            title => $form->{'sitelinks_' . $_ . '_title' . $suffix},
            href => clear_banner_href($form->{'sitelinks_' . $_ . '_href' . $suffix}, $form->{'sitelinks_' . $_ . '_url_protocol' . $suffix}),
            defined $form->{'sitelinks_' . $_ . '_description' . $suffix}
            ? (description => $form->{'sitelinks_' . $_ . '_description' . $suffix})
            : (),
            defined $form->{'sitelinks_' . $_ . '_tl_id' . $suffix}
            ? (tl_id => $form->{'sitelinks_' . $_ . '_tl_id' . $suffix})
            :(),
        }}
        sort {$a <=> $b}
        map {m/^sitelinks_(\d)_href(-\d+)?$/ && str($2) eq $suffix ? $1 : ()}
        keys %$form
    ];

    foreach my $sitelink (@$sitelinks) {
        $sitelink->{'title'} ||= '';
        $sitelink->{title} =~ s!^\s+|\s$!!g;
        if (defined $sitelink->{description}) {
            $sitelink->{description} =~ s/^\s+|\s$//g;
        }
    }

    return $sitelinks;
}

=head2 divide_sitelinks_href

    Разделяет ссылку сайтлинки на два поля: ссылку без протокола и протокол.
    Используется в контроллерах для показа формы редактирования.

=cut
sub divide_sitelinks_href($) {
    my $sitelinks_set = shift;
    
    for my $sitelink (@$sitelinks_set) {
        next if $sitelink->{url_protocol};
        hash_merge $sitelink, divide_href_protocol($sitelink->{href});
    }
    
    return $sitelinks_set;    
}

=head2 clear_sitelinks_href

    Разделяет ссылку сайтлинки на два поля: ссылку без протокола и протокол.
    Используется в контроллерах для показа формы редактирования.

=cut
sub clear_sitelinks_href($) {
    my $sitelinks_set = shift;
    for my $sitelink (@$sitelinks_set) {
        $sitelink->{url_protocol} = clear_protocol($sitelink->{url_protocol});
        $sitelink->{href} = clear_banner_href($sitelink->{href}, $sitelink->{url_protocol});
        $sitelink->{'title'} ||= '';
    }
    
    return $sitelinks_set;    
}

=head2 _save_sitelink

    сохраняем сайтлинк с данными параметрами
    save_sitelink( { title => ..., description => ..., href => ..., } )

    ищем, есть ли уже такой же sitelink, если нет, то создаем

    возвращаем sitelink_id

=cut

sub _save_sitelink
{
    my ($sitelink, $client_id) = @_;

    $sitelink->{href} = clear_banner_href($sitelink->{href}) if (defined $sitelink->{href});
    my $sitelink_id = _get_sitelink_id($sitelink, $client_id);
    my $tl_id;
    if (exists $sitelink->{turbolanding}){
        $tl_id = $sitelink->{turbolanding}->{id};
    }
    elsif(exists $sitelink->{tl_id}){
        $tl_id = $sitelink->{tl_id};
    }

    unless ($sitelink_id) {
        $sitelink_id = get_new_id('sl_id');
        do_insert_into_table(PPC(ClientID => $client_id), 'sitelinks_links',
            {
                sl_id  => $sitelink_id,
                title  => $sitelink->{title},
                description => (defined $sitelink->{description} && length($sitelink->{description}) > 0 ? $sitelink->{description} : undef),
                href   => $sitelink->{href},
                hash   => _calc_sitelink_hash($sitelink),
                tl_id  => $tl_id,
            }
        );
    }
    return $sitelink_id;
}

=head2 _get_sitelink_id

    ищем сохраненный сайтлинк
    возвращаем sitelink_id или undef

=cut

sub _get_sitelink_id
{
    my ($sitelink, $client_id) = @_;

    my $tl_id;
    if ( exists  $sitelink->{turbolanding}) {
        $tl_id = $sitelink->{turbolanding}->{id};
    }
    else {
        $tl_id = $sitelink->{tl_id};
    }
    
    my $sl_id = get_one_field_sql(PPC(ClientID => $client_id), '
        SELECT sl_id
        FROM sitelinks_links
        WHERE hash  = ?
          AND title = ?
          AND href  = ?
          AND tl_id = ?
        ',
        _calc_sitelink_hash($sitelink),
        $sitelink->{title},
        $sitelink->{href}, 
        $tl_id,
    );
    return $sl_id;
}

=head2 _get_sitelinks_set_id

    ищем сохраненный сет сайтлинков по их id
    возвращаем sitelinks_set_id или undef

=cut

sub _get_sitelinks_set_id
{
    my ($sitelinks_ids, $client_id) = @_;
    die "wrong set content" unless scalar(@$sitelinks_ids) > 0 && scalar(@$sitelinks_ids) <= $SITELINKS_NUMBER;
    return get_one_field_sql(PPC(ClientID => $client_id), ['SELECT sitelinks_set_id FROM sitelinks_sets', 
                            where=>{links_hash => _calc_sitelinks_set_links_hash($sitelinks_ids), ClientID=>$client_id}, "limit 1"]);
}


=head2 _calc_sitelink_hash

    хэш от сайтлинка для быстрого поиска дублей

=cut

sub _calc_sitelink_hash
{
    my $sitelink = shift;

    my ($tl_id) = ('');
    if (exists  $sitelink->{turbolanding}){
        $tl_id = $sitelink->{turbolanding}->{id};
    }

    return url_hash_utf8($sitelink->{title} . $sitelink->{href} . ($sitelink->{description} // '') . $tl_id);
}


=head2 _calc_sitelinks_set_links_hash

    хэш от сета для быстрого поиска дублей

=cut

sub _calc_sitelinks_set_links_hash
{
    my $sitelinks_ids = shift;
    return url_hash_utf8(join(',', @$sitelinks_ids));
}


=head2 delete_sitelinks_mod_versions 

    Удаляет по заданным bid-ам версии модерации сайтлинков
    Используется при отвязке сайтлинков от баннера

=cut

sub delete_sitelinks_mod_versions {
    my $bids = shift;

    do_delete_from_table(PPC(bid => $bids), 'mod_object_version', where => { obj_id => SHARD_IDS, obj_type => 'sitelinks_set'});
}

1;
