package Staff;

# $Id$

=head1 NAME
    
    Staff

=head1 DESCRIPTION

    Функции для работы с ролями внутренних пользователей

=cut

use strict;
use warnings;
use utf8;   

our @ISA = qw(Exporter);
our @EXPORT = qw(
    check_create_internal_user
    save_edited_user
    save_edited_user_API_settings
    save_subclients_rights
    save_status_not_resident
);

use JSON qw/to_json/;

use Direct::AgencyOptions;
use Direct::Validation::Domains qw/validate_disabled_domains/;
use Yandex::DBTools;
use List::Util qw/max min sum/;
use IpTools;
use EnvTools;
use TextTools;
use RBACElementary;
use RBACDirect;
use Rbac qw/:const/;
use Client;
use Client::ConvertToRealMoney ();
use Client::CustomOptions;
use User;
use BannerImages::Pool;
use Notification;
use BalanceQueue;
use BS::ResyncQueue;

use APIUnits;
use Client::CurrencyTeaserData;
use Client::NDSDiscountSchedule ();

use Currencies;

use Primitives;
use PrimitivesIds;
use Yandex::I18n;

use Settings;

use Yandex::HashUtils;
use Yandex::Staff3 qw( get_staff_info );
use Yandex::IDN qw(is_valid_email is_valid_domain);
use Yandex::Validate qw/is_valid_phone is_valid_int is_valid_id/;
use Yandex::DBShards;
use Yandex::URL;

use List::MoreUtils qw/any uniq none/;
use API::ClientOptions;

=head2 check_create_internal_user
  
  Проверки валидности, при создании нового внутреннего пользователя.
  Если форма заполнена правильно возвращается 0, если найдена ошибка возвращается ее текст

=cut

sub check_create_internal_user {

    my $checks = $_[0]->{checks};
    my $data = $_[0]->{data};

    my %FORM = %{$data->{form}};

    # проверка на заполненность полей
    if ($checks->{fullness}) {

        if( any { ! defined $FORM{$_} || length($FORM{$_}) == 0 } 
                qw/login fio phone email domain_login/  )
        {
            return iget('не введено одно из значений: ФИО, телефон, email, доменный логин или login');
        }
    }

    # заполнено ли поле "логин"
    if ($checks->{filled_login}) {
        if (! defined $FORM{login}    ||
            length($FORM{login}) == 0)
        {
            return iget('не введено одно из значений: login');
        }
    }

    # менеджер ли пользователь
    if ($checks->{is_manager}) {
        # странная проверка, точно такая была в функциях cmd_createTeamleader и cmd_createSuperTeamleader
        if (get_one_field_sql(PPC(uid => $data->{uid}), "select count(*) from users where uid = ?", $data->{uid}) == 0) {
            return iget('Ошибка: данный логин должен быть менеджером');
        }
    }

    # корректный ли email
    if ($checks->{valid_email}) {
        unless (is_valid_email($FORM{email})) {
            return iget('не правильный формат email');
        }
    }

    # существует ли данынй логин
    if ($checks->{login_exists}) {
        if ( ! defined $data->{uid} ) {
            return iget('такой логин не существует');
        }
    }

    # проверки для поля domain_login, корректность и существование
    if ($checks->{domain_login}) {
        $FORM{domain_login} =~ s/^\s+//;
        $FORM{domain_login} =~ s/\s+$//;

        if ( ! validate_login($FORM{domain_login}) ) {
            return iget("Некорректный доменный логин");
        }

        #check whether domain_login exists on staff.yandex.ru
        if ( ! %{get_staff_info($FORM{domain_login})}) {
            return iget('Доменный логин введен неверно или не существует');
        }
    }

    # проверка сущесвтования ClientID у пользователя
    if ($checks->{client_id}) {
        if (get_clientid(uid => $data->{uid})) {
            return iget('логин уже имеет ClientID');
        }
    }

    # указан ли офис менеджера
    if ($checks->{managers_office}) {
        # check yandex office for manager
        return iget("Не указан офис") if ! $FORM{manager_office_id} || $FORM{manager_office_id} !~ /^\d+$/;
        return iget("Не верно указан офис") if ! grep {$_->{office_id} == $FORM{manager_office_id}} @{$data->{yandex_offices}};
    }

    return 0;
}

=head2 save_edited_user

 Сохраняем отредактированные параметры пользователей

=cut

sub save_edited_user {
    
        my ($UID, $uid, $rbac, $login_rights, $vars, $subclients_rights, $agency_client_id, $client_client_id) = @{$_[0]}{
          qw/UID   uid   RBAC   LOGIN_RIGHTS   VARS   SUBCLIENTS_RIGHTS   AGENCY_CLIENT_ID   CLIENT_CLIENT_ID/};
        my %FORM = %{$_[0]{FORM}};

        my $user_role = $vars->{user_role};
        my $client_id = $vars->{ClientID};

        my $error;

        # $user_data for table users, $user_options_data for table users_options;
        my $user_data;

        # проверяем может ли пользователь исправлять данные у Агентства и geo_id агентства 
        my $geo_id_ok_flag = 1;
        if (($user_role eq 'agency' || ($vars->{user_is_any_client} && $vars->{serviced})) && ($login_rights->{super_control} || $login_rights->{manager_control})) {
            if(! defined $FORM{geo_id} || $FORM{geo_id} !~ /^\d+$/){
                $geo_id_ok_flag = 0;
            }
        }

        $FORM{FIO} =~ s/^\s*//;
        $FORM{FIO} =~ s/\s*$//;

        return iget("поле Имя не заполнено") unless length($FORM{FIO});
        return iget("нельзя включить тизер мультивалютности клиенту с установленным флагом 'Не конвертировать в валюту'")
            if $FORM{force_multicurrency_teaser} && $vars->{not_convert_to_currency};
        return iget("нельзя разблокировать клиента с установленным флагом 'cant_unblock'")
            if (($FORM{blocked_status} // '') ne 'Yes') && ($FORM{save} eq 'yes') && $vars->{cant_unblock};

        if (defined $FORM{FIO} &&
            defined $FORM{email} &&
            $geo_id_ok_flag
           )
        {
            # validate email
            return iget("неправильный формат адреса: %s", $FORM{email}) unless is_valid_email($FORM{email});

            # validate phone
            return iget("неправильный формат номера телефона: %s", $FORM{phone}) if ! $vars->{dont_edit_fio_email_by_agency} && ! is_valid_phone($FORM{phone});
            
            my (%client_data_updates, %client_data_updates_balance);

            if (($login_rights->{support_control} || $login_rights->{placer_control} || $login_rights->{super_control}) && $FORM{report_name}) {
                $client_data_updates{report_name} = $FORM{report_name};
            }

            if ($login_rights->{support_control} || $login_rights->{placer_control} || $login_rights->{super_control}) {
                $client_data_updates{social_advertising} = $FORM{social_advertising} ? 1 : 0;
            }

            if (! $vars->{dont_edit_fio_email_by_agency}) {
                $user_data = hash_merge( $user_data, hash_cut(\%FORM, qw/phone email/ ) );
                $user_data->{fio} = $FORM{FIO};
            }

            # set not_resident status
            save_status_not_resident($rbac, 
                                        serviced => $vars->{serviced}, 
                                        client_client_id => $client_client_id, 
                                        agency_client_id => $agency_client_id, 
                                        not_resident => ($FORM{not_resident} ? 'Yes' : 'No'),
                                        user_role => $user_role,
                                        login_rights => $login_rights,
                                    );
            # save description
            if ($vars->{allow_edit_description} && ! $login_rights->{agency_control}) {
                my $user_description = $FORM{description};
                $user_description = substr($user_description, 0, $Client::MAX_USER_DESCRIPTION_LENGTH) if $user_description && length($user_description) > $Client::MAX_USER_DESCRIPTION_LENGTH;
                $user_data->{description} = $user_description;
            }

            # save city geo
            # сохранять гео может только тимлидер или супер
            if (($user_role eq 'agency' || $vars->{user_is_any_client})
                && ($vars->{can_edit_subregion})
                && defined $FORM{geo_id} && $FORM{geo_id} =~ /^\d+$/
                && $vars->{subregion_id} != $FORM{geo_id} # Убедимся, что значение geo_id изменилось
            ) {
                if ($user_role eq 'agency') {
                    my @uids_for_update_geo_id;

                    # Получаем всех представителей агентства
                    push @uids_for_update_geo_id, @{ rbac_get_main_reps_of_agency($agency_client_id) }, @{ rbac_get_limited_reps_of_agency($agency_client_id) };
                    @uids_for_update_geo_id = uniq grep { defined } @uids_for_update_geo_id;

                    # Получаем их ClientID
                    my @clienids_for_update_geo_id = @{get_clientids(uid => \@uids_for_update_geo_id)};

                    # Получаем всех субклиентов агентства
                    push @clienids_for_update_geo_id, @{ rbac_get_subclients_clientids($rbac, $uid) };
                    @clienids_for_update_geo_id = uniq grep { defined } @clienids_for_update_geo_id;

                    # Прописываем "Город" на полученных ClientID
                    if (@clienids_for_update_geo_id) {
                        foreach_shard ClientID => \@clienids_for_update_geo_id, chunk_size => 500, sub {
                            my ($shard, $client_id_chunk) = @_;
                            do_update_table(PPC(shard => $shard), 'clients_options',
                                {subregion_id => $FORM{geo_id}}, where => {ClientID => $client_id_chunk});
                        };
                    }
                }
            }

            # суппорт и суперпользователь могут менять показ тизера для первой помощи и значение плохой паспорт спам-кармы
            if ($login_rights->{support_control} || $login_rights->{super_control}) {
                $user_data->{show_fa_teaser} = $FORM{show_fa_teaser} ? 'Yes' : 'No';
            }

            if ($login_rights->{support_control} || $login_rights->{super_control} || $login_rights->{manager_control} || $login_rights->{placer_control}) {
                if (!$vars->{is_bad_passport_karma} && $FORM{is_bad_passport_karma}) {
                    $user_data->{passport_karma} = 101;
                } elsif ($vars->{is_bad_passport_karma} && !$FORM{is_bad_passport_karma}) {
                    $user_data->{passport_karma} = -1;
                    do_delete_from_table(PPCDICT, 'bad_karma_clients', where => { uid => $uid } );
                }

                # менять настройки дефолтного списка запрещенных PageID
                my @default_disallowed_page_ids = uniq grep { $_ ne "" } map { smartstrip($_) } split /[\s,]/, $FORM{default_disallowed_page_ids};
                if (scalar(grep {!is_valid_id($_)} @default_disallowed_page_ids)) {
                    return iget("PageID может содержать только числа. Исправьте, пожалуйста.");
                }
                $client_data_updates{default_disallowed_page_ids} = to_json(\@default_disallowed_page_ids);

                # менять настройки дефолтного белого списка доменов
                my @default_allowed_domains = uniq grep { $_ ne "" } map { smartstrip($_) } split /[\s,]/, $FORM{default_allowed_domains};
                if (scalar(grep {!validate_disabled_domains([ $_ ])->is_valid} @default_allowed_domains)) {
                    return iget("Список разрешенных доменов содержит домены неправильного формата. Исправьте, пожалуйста.");
                }
                $client_data_updates{default_allowed_domains} = to_json(\@default_allowed_domains);

            }

            if ($login_rights->{support_control} || $login_rights->{super_control}) {
                
                if ($vars->{user_is_any_client}) {
                    my %limits;
                    foreach (keys %Client::CLIENT_LIMIT_FIELDS) {
                        $limits{$_} = $FORM{$_} || 0  if defined $FORM{$_}
                    }
                    
                    if (keys %limits) {
                        my @errors = validate_client_limits(\%limits);
                        return join "\n", @errors if @errors;
    
                        set_client_limits($client_client_id, \%limits);
                    }
                }
            }

            BalanceQueue::add_to_balance_info_queue($UID, 'uid', $uid, BalanceQueue::PRIORITY_USER_ON_MODIFY_USER);

            # set subclients rights
            if ($vars->{user_is_any_client} && ref($vars->{client_by_agency_rights}) eq 'ARRAY' && $login_rights->{ModifySubClient}) {
                if (my $error = save_subclients_rights($rbac, $uid, $UID, $login_rights, $client_client_id, $subclients_rights, \%FORM, save_description => 1)) {
                    return $error;
                }
            }

            # save hidden status
            if ($login_rights->{support_control} || $login_rights->{placer_control} || $login_rights->{super_control}) {
                if (!validate_networks($FORM{allowed_ips})) {
                    return iget("Неверно указаны разрешённые ip");
                }

                $user_data = hash_merge($user_data, {
                    hidden => $FORM{hidden_status} ? 'Yes' : 'No',
                    allowed_ips => $FORM{allowed_ips} || '',
                });

                if (is_beta() || $vars->{user_is_any_client} || $user_role eq 'agency') { # DIRECT-30326, DIRECT-31059 - reason for is_beta()

                    # Менять статус блокировки support не может, поэтому флаг blocked_status не передается
                    if (exists $FORM{blocked_status} || !$login_rights->{support_control}) {
                        $user_data->{statusBlocked} = $FORM{blocked_status} ? 'Yes' : 'No';
                    } else {
                        $user_data->{statusBlocked} = defined $vars->{statusBlocked} && $vars->{statusBlocked} eq 'Yes' ? 'Yes' : 'No';
                    }

                    my $client_id = get_clientid(uid => $uid);
                    if ($FORM{blocked_status}) {
                        # клиентов в фишках дополнительно помечаем флагом cant_unblock
                        my $client_data = get_client_data($client_client_id, [qw/work_currency/]);
                        $client_data_updates{cant_unblock} = 1 if $client_data->{work_currency} eq 'YND_FIXED';
                    }
                }
            }

            # save AllowAgencyMassADVQ
            if ($user_role eq 'agency' && ($login_rights->{manager_control} || $login_rights->{super_control})) {
                my $allow_agency_mass_advq = $FORM{AllowAgencyMassADVQ} || 0;
                if ($allow_agency_mass_advq != $vars->{AllowAgencyMassADVQ}) {
                    my $error_code = rbac_set_allow_agency_mass_advq($rbac, $uid, $allow_agency_mass_advq);
                    if ($error_code) {
                        return iget('не удалось запретить/разрешить возможность для агентства массово запрашивать ADVQ');
                    }
                }
            }
            
            if ($user_role eq 'agency' && ($login_rights->{manager_control} && $UID == $vars->{ManagerOfAgencyUID} || $login_rights->{super_control}) ) {
                # update all reps of agency
                for my $agency_uid (@{ Rbac::get_reps(ClientID => $agency_client_id, role => $ROLE_AGENCY) }) {
                    create_update_user(
                                $agency_uid, 
                                {
                                    sendAgencyMcbLetters => ( $FORM{sendAgencyMcbLetters} ? 'Yes' : 'No' ), 
                                    sendAgencyDirectLetters => ($FORM{sendAgencyDirectLetters} ? 'Yes' : 'No')
                                }
                              );
                }
            }

            # (un)create SuperPlacer
            if ($user_role eq 'placer' && $login_rights->{super_control}) {
                my $create_super_placer = $FORM{isSuperPlacer} || 0;
                if ($create_super_placer != $vars->{isSuperPlacer}) {

                    my $error_code;
                    if ($create_super_placer) {
                        create_client_in_balance($UID, $uid, %{hash_cut \%FORM, qw/FIO phone email initial_country initial_currency/}) or die "create_client_in_balance error for UID:$UID, uid:$uid";
                        $error_code = rbac_create_superplacer($rbac, $uid);
                        Client::update_role($client_client_id, 'placer', 'superplacer');
                    } else {
                        $error_code = rbac_drop_superplacer($rbac, $uid);
                        Client::update_role($client_client_id, 'placer', undef);
                    }

                    if ($error_code) {
                        return iget('не удалось запретить/разрешить возможность принимать импортируемые баннеры без модерации');
                    }
                }
            }
            
            # (un)create SuperMedia
            if ($user_role eq 'media' && $login_rights->{super_control}) {
                my $create_super_media = $FORM{isSuperMedia} || 0;
                if ($create_super_media != $vars->{isSuperMedia}) {

                    my $error_code;
                    if ($create_super_media) {
                        create_client_in_balance($UID, $uid, %{hash_cut \%FORM, qw/FIO phone email initial_country initial_currency/}) or die "create_client_in_balance error for UID:$UID, uid:$uid";
                        $error_code = rbac_create_supermedia($rbac, $uid);
                        Client::update_role(get_clientid(uid => $uid), 'media', 'supermedia');
                    } else {
                        $error_code = rbac_drop_supermedia($rbac, $uid);
                        Client::update_role(get_clientid(uid => $uid), 'media', undef);
                    }

                    if ($error_code) {
                         return iget('не удалось создать/удалить супер медиапланнера');
                    }
                }
            }

            # save captcha_freq
            if ($login_rights->{super_control} && defined $FORM{captcha_freq} && $FORM{captcha_freq} =~ /^\d+$/) {
                $user_data->{captcha_freq} = $FORM{captcha_freq};
            }

            # save autoban_bonus
            if (($login_rights->{super_control} || $login_rights->{manager_control} || $login_rights->{support_control})
                && $vars->{user_is_any_client}) {
                my $autoban_bonus = int($FORM{autoban_bonus});
                if ($autoban_bonus != $vars->{autoban_bonus}) {
                    do_insert_into_table(PPC(uid => $uid), 'clients_autoban', {ClientID => $vars->{ClientID}, autoban_bonus => $autoban_bonus}, 
                                         on_duplicate_key_update => 1);
                }
            }

            # save statusYandexAdv for simple client
            if ($vars->{user_is_any_client} && $login_rights->{super_control}) {
                my $new_status_context = $FORM{statusYandexAdv} ? 'Yes' : 'No';
                my $new_status_yandex  = $FORM{showOnYandexOnly}  ? 'Yes' : 'No';
                if ($new_status_context ne $vars->{statusYandexAdv} or $new_status_yandex ne $vars->{showOnYandexOnly}) {
                    $user_data = hash_merge( $user_data, { statusYandexAdv=>$new_status_context, showOnYandexOnly=>$new_status_yandex } );
                    do_update_table(PPC(uid => $uid), 'campaigns', {statusBsSynced => 'No'}, where => {uid => $uid});
                    do_sql(PPC(uid => $uid), qq{UPDATE campaigns c join phrases p using (cid) SET p.LastChange=p.LastChange, p.statusBsSynced = 'No' WHERE c.uid = ?}, $uid );
                }
            }

            if ($login_rights->{super_control} || $login_rights->{manager_control}) {
                $user_data->{statusPostmoderate} = $FORM{isGoodUser} ? 'Yes' : 'No';
            }

            # для менеджера обновляем офис/CRM
            if ($user_role eq 'manager') {
                $user_data->{manager_use_crm} = $FORM{manager_use_crm} ? 'Yes' : 'No';

                return iget("неправильный формат адреса: %s", $FORM{manager_private_email}) if $FORM{manager_private_email} && ! is_valid_email($FORM{manager_private_email});
                $user_data->{manager_private_email} = $FORM{manager_private_email};

                if ($login_rights->{super_control} || $login_rights->{support_control} || $login_rights->{manager_control}) {
                    my $office_id = $FORM{manager_office_id};
                    return "Не указан офис" if !$office_id || $office_id !~ /^\d+$/;
                    return "Не верно указан офис" if !grep {$_->{office_id} == $office_id} @{$vars->{yandex_offices}};
                    $user_data->{manager_office_id} = $office_id;
                }

            }

            # для не-шефов агентств сохраняем галку "Отсылать копии писем главному представителю"
            if ($user_role eq 'agency' && $vars->{agency_rep_type} ne 'chief') {
                $user_data->{agency_email_to_chief} = $FORM{agency_email_to_chief} ? 'Yes' : 'No';
            }

            # обновляем разрешение пользователям использовать финансовые операции
            # обновляем дефолтного плательщика

            if (defined $FORM{agency_name}) {
                $FORM{agency_name} =~ s/^\s*//;
                $FORM{agency_name} =~ s/\s*$//;
            }

            my $agency_uids_to_refresh_clients;

            if ($user_role eq 'agency' && ($login_rights->{super_control} || $login_rights->{manager_control} || $login_rights->{agency_control})) {
                my ($lim_rep_group_id, $lim_rep_type);
                if ($FORM{ag_role} && $FORM{ag_role} =~ s/^(limited)_(chief|main)$/$1/) {
                    $lim_rep_type = $2;
                }

                if (Client::ClientFeatures::has_new_lim_rep_schema_feature($agency_client_id)) {
                    if ($FORM{chief_lim_rep_uid} && $lim_rep_type) {
                        $lim_rep_group_id = Agency::get_lim_rep_group_id($FORM{chief_lim_rep_uid});
                    }

                    my $same_lim_rep_type = ($lim_rep_type // '') eq ($vars->{agency_lim_rep_type} // '') ? 1 : 0;
                    my $same_lim_rep_group_id =  ($lim_rep_group_id // 0) == ($vars->{agency_group_id} // 0) ? 1 : 0;
                    my @uids_to_refresh_clients;

                    if ($vars->{agency_lim_rep_type} && $vars->{agency_lim_rep_type} eq $LIM_REP_CHIEF) {
                        unless ($same_lim_rep_type && $same_lim_rep_group_id) {
                            if (Agency::has_lim_rep_group_main_reps($uid)) {
                                return iget('Нельзя поменять тимлида в группе с менеджерами');
                            }

                            if ($same_lim_rep_group_id && (($lim_rep_type // '') eq 'main')) {
                                return iget('Нельзя тимлида сделать менеджером в той же группе');
                            }

                            if ($same_lim_rep_type && !$same_lim_rep_group_id) {
                                return iget('Нельзя сделать тимлидом другой группы');
                            }

                            if ($FORM{ag_role} eq $REP_MAIN) {
                                $user_data->{rep_type} = $REP_MAIN;
                                do_delete_from_table(PPC(uid => $uid), 'users_agency', where=>{uid => $uid});
                                Agency::unlink_all_clients_from_lim_rep($uid, rbac_get_chief_rep_of_agency_rep($uid));
                                $agency_uids_to_refresh_clients = [$uid];
                            } elsif (!$same_lim_rep_type) { # тимлида делаем менеджером
                                Agency::relink_lim_rep_clients_old_schema($uid);
                                Agency::link_lim_rep_clients_to_other_lim_rep($uid, $FORM{chief_lim_rep_uid});
                                $agency_uids_to_refresh_clients = [$FORM{chief_lim_rep_uid}];
                            }
                            Direct::TurboLandings::refresh_metrika_grants_for_agency_clients($agency_client_id);
                        }
                    } elsif ($vars->{agency_lim_rep_type} && $vars->{agency_lim_rep_type} eq $LIM_REP_MAIN) {
                        unless ($same_lim_rep_type && $same_lim_rep_group_id) { # меняем тип представителя или переносим в другую группу
                            if ($lim_rep_type && $lim_rep_type eq $LIM_REP_CHIEF) {
                                $lim_rep_group_id = get_new_id('agency_lim_rep_group_id');
                                Agency::relink_lim_rep_clients_old_schema($uid);
                            }

                            if ($FORM{ag_role} eq $REP_MAIN) {
                                $user_data->{rep_type} = $REP_MAIN;
                                do_delete_from_table(PPC(uid => $uid), 'users_agency', where=>{uid => $uid});
                                Agency::unlink_all_clients_from_lim_rep($uid, rbac_get_chief_rep_of_agency_rep($uid));
                            } elsif ($same_lim_rep_type && !$same_lim_rep_group_id) { # переносим менеджера в другую группу
                                Agency::relink_lim_rep_clients_old_schema($uid);
                                Agency::link_lim_rep_clients_to_other_lim_rep($uid, $FORM{chief_lim_rep_uid});
                                push @uids_to_refresh_clients, $FORM{chief_lim_rep_uid};
                            }

                            $agency_uids_to_refresh_clients = [$uid, @uids_to_refresh_clients];
                            Direct::TurboLandings::refresh_metrika_grants_for_agency_clients($agency_client_id);
                        }
                    } elsif ($vars->{agency_rep_type} eq $REP_MAIN || $vars->{agency_rep_type} eq $REP_LIMITED) {
                        if ($lim_rep_type) {
                            if ($lim_rep_type eq $LIM_REP_MAIN && !$FORM{chief_lim_rep_uid}) {
                                return iget('Нельзя сделать менеджером в пустой группе');
                            }

                            $user_data->{rep_type} = $REP_LIMITED;

                            if ($vars->{agency_rep_type} eq $REP_LIMITED) {
                                Agency::relink_lim_rep_clients_old_schema($uid);
                            }

                            if ($lim_rep_type eq $LIM_REP_CHIEF) {
                                $lim_rep_group_id = get_new_id('agency_lim_rep_group_id');
                                $agency_uids_to_refresh_clients = [$uid];
                            } else {
                                Agency::relink_lim_rep_clients_old_schema($uid);
                                Agency::link_lim_rep_clients_to_other_lim_rep($uid, $FORM{chief_lim_rep_uid});
                                $agency_uids_to_refresh_clients = [$uid, $FORM{chief_lim_rep_uid}];
                                Direct::TurboLandings::refresh_metrika_grants_for_agency_clients($agency_client_id);
                            }
                        }
                    }
                } else {
                    $lim_rep_type = $vars->{agency_lim_rep_type};
                    $lim_rep_group_id = $vars->{agency_group_id};
                }

                if (($FORM{ag_role} && $FORM{ag_role} eq $REP_LIMITED) || $vars->{agency_rep_type} eq $REP_LIMITED) {
                    do_insert_into_table(PPC(uid => $uid), 'users_agency', 
                                     { uid => $uid, 
                                       is_no_pay => ($FORM{agency_allow_pay}) ? 0 : 1,
                                       disallow_money_transfer => ($FORM{agency_disallow_money_transfer}) ? 0 : 1,
                                       lim_rep_type => $lim_rep_type,
                                       group_id => $lim_rep_group_id,
                                     }, 
                                     on_duplicate_key_update => 1, key => 'uid');
                }
            }

            # save name and url agency into ppc.clients + mass convert permitted
            if ($user_role eq 'agency' && ($login_rights->{manager_control} || $login_rights->{super_control} || $login_rights->{support_control})) {
                my $agency_status = $FORM{agency_status} && $FORM{agency_status} =~ /^(?:SA|AA|HY|WC|NR|ABU)$/ ? $FORM{agency_status} : undef;

                if (defined $FORM{agency_url} && ! is_valid_domain(get_host($FORM{agency_url}))) {
                    return iget("неправильный формат URL агентства");
                }

                create_update_client({
                                      client_data => {
                                            ClientID => $agency_client_id, 
                                            name => $FORM{agency_name}, 
                                            agency_url => $FORM{agency_url},
                                            agency_status => $agency_status
                                      }
                                    });
            }

            # смотреть/задавать свободу создавать самоходные/сервисируемые кампании субклиенту может только супер
            if ($vars->{user_is_any_client} && $login_rights->{super_control} && ref($vars->{client_by_agency_rights}) eq 'ARRAY' && $login_rights->{ModifySubClient}) {
                my $old_allow_create_scamp_by_subclient = $vars->{allow_create_scamp_by_subclient} ? 'Yes' : 'No';
                my $new_allow_create_scamp_by_subclient = $FORM{allow_create_scamp_by_subclient} ? 'Yes' : 'No';
                if ($old_allow_create_scamp_by_subclient ne $new_allow_create_scamp_by_subclient) {

                    if ($new_allow_create_scamp_by_subclient eq 'Yes') {
                        # EXPIRES 2013-12-01 multicurrency-agency-nds
                        # не даём получить мультивалютного свободного клиента
                        # подробности смотри в Client::mass_get_client_NDS
                        my $client_currencies = get_client_currencies($vars->{ClientID});
                        if ($client_currencies->{work_currency} ne 'YND_FIXED') {
                            return iget('Мультивалютным клиентам нельзя получать право создавать самоходные/сервисируемые кампании');
                        }
                    }

                    $client_data_updates{allow_create_scamp_by_subclient} = $new_allow_create_scamp_by_subclient;

                    # при снятии галочки запрещаем оплату/перенос на всех сервисируемых/самоходных кампаниях клиента (DIRECT-16615)
                    if ($new_allow_create_scamp_by_subclient eq 'No') {
                        do_update_table(PPC(ClientID => $vars->{ClientID}), 'campaigns c join users u using(uid)', {statusNoPay => 'Yes'},
                                        where => {'u.ClientID' => $vars->{ClientID}, 
                                                  _TEXT => 'IFNULL(c.AgencyUID, 0) = 0',
                                                  'c.statusNoPay' => 'No',
                                                  'c.type' => [qw/text mcb/]});
                    }
                }
            }

            #обрабатываем признак "Запретить фавиконки на выдаче"
            my $is_favicon_blocked = $FORM{is_favicon_blocked} ? 1 : 0;
            if ($vars->{user_is_any_client} && ($login_rights->{super_control} || $login_rights->{placer_control} || $login_rights->{support_control}) &&
                ((! defined $vars->{is_favicon_blocked}) || $is_favicon_blocked != $vars->{is_favicon_blocked})) {
                $client_data_updates{is_favicon_blocked} = $is_favicon_blocked;
                do_update_table(PPC(ClientID => $vars->{ClientID}), 'campaigns c join users u using(uid)', {statusBsSynced => 'No'},
                                where => {'u.ClientID' => $vars->{ClientID}, 'c.archived' => 'No'});
            }

            if ($user_role eq 'superreader' && $login_rights->{super_control}) {
                # сохраняем роль "супер-менеджер" для суперридеров
                $user_data->{is_super_manager} = $FORM{is_super_manager} ? 1 : 0;
            }

            # сохраняем язык для уведомлений
            if ($FORM{user_lang}) {
                $user_data->{lang} = Yandex::I18n::is_valid_lang($FORM{user_lang}) ? $FORM{user_lang} : Yandex::I18n::default_lang();
            }

            my $agency_options;
            if ($user_role eq 'agency') {
                # Сохраняем флажок "Получать уведомления о добавлении новых доменов"
                $user_data->{notify_about_new_domains} = $FORM{notify_about_new_domains} ? 1 : 0;

                # Сохраняем настройки агентства
                $agency_options = $vars->{agency_options};
                agency_options_apply_form($UID, $uid, $login_rights, $vars, $agency_options, \%FORM);
            }

            # Обновляем пользователя
            create_update_user( $uid, $user_data );
            my %user_options = ();
            my %user_custom_options = ();

            if ($agency_uids_to_refresh_clients) {
                Agency::refresh_clients_of_agency_limited_reps($agency_uids_to_refresh_clients, $UID);
            }

            # Настройка таргетинга на устройства
            if ($login_rights->{super_control} && $user_role =~ m/client$/) {
                # Плохо выглядит отдельный "еще один" апдейт, но с другой стороны
                # получилась бы полная копипаста функции update_user_options
                $user_options{can_have_device_targeting} = $FORM{allow_device_targeting} ? 1 : 0;
            }

            if ($login_rights->{super_control} || $login_rights->{support_control} 
                || $login_rights->{manager_control} || $login_rights->{is_teamleader} )
            {
                $user_custom_options{disallow_money_transfer} = $FORM{disallow_money_transfer} ? 1 : 0;
            }

            if (keys %user_options) {
                update_user_options($uid, \%user_options);
            }
            if (keys %user_custom_options) {
                set_user_custom_options($vars->{ClientID}, \%user_custom_options);
            }

            # ниже по коду get_client_first_agency может вызываться трижды, хорошо бы вызывать только один раз
            # но при этом не вызывать совсем, если эти данные не нужны

            # сохраняем галку "Принудительно показывать тизер перехода на мультивалютность"
            if ($login_rights->{super_control}) {
                if ($FORM{force_multicurrency_teaser} && !$vars->{force_multicurrency_teaser}) {
                    if (none {$_ == $vars->{ClientID}} @Client::ConvertToRealMoney::KNOWN_BAD_CLIENTIDS) {
                        do_insert_into_table(PPC(ClientID => $vars->{ClientID}), 'clients_to_fetch_nds', {ClientID => $vars->{ClientID}}, ignore => 1);
                        do_insert_into_table(PPC(ClientID => $vars->{ClientID}), 'clients_to_force_multicurrency_teaser', {ClientID => $vars->{ClientID}, modify_convert_allowed => 1}, ignore => 1);
                        # обновляем валюты/страны для клиента не дожидаясь ночи. если он подпадает под остальные условия, то тизер у него появится сразу после простановки галки
                        my $agency_id = Primitives::get_client_first_agency($vars->{ClientID});
                        Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($vars->{ClientID}, $agency_id);
                    } else {
                        return iget('Клиент входит в список плохих, включать ему тизер нельзя');
                    }
                } elsif (!$FORM{force_multicurrency_teaser} && $vars->{force_multicurrency_teaser}) {
                    do_delete_from_table(PPC(ClientID => $vars->{ClientID}), 'clients_to_force_multicurrency_teaser', where => {ClientID => $vars->{ClientID}});
                }

                if ($FORM{force_multicurrency_teaser} && ($FORM{modify_convert_allowed} ? 1 : 0) != ($vars->{modify_convert_allowed} ? 1 : 0)) {
                    do_update_table(PPC(ClientID => $vars->{ClientID}), 'clients_to_force_multicurrency_teaser', {modify_convert_allowed => ($FORM{modify_convert_allowed} ? 1 : 0)}, where => {ClientID => $vars->{ClientID}});
                }
            }

            # сохраняем страну/валюту
            if ($vars->{can_edit_country_currency} && $FORM{edit_country_currency}) {
                my $country = $FORM{country};

                my $country_currencies;
                my $agency_id = Primitives::get_client_first_agency($vars->{ClientID});
                if ($user_role eq 'agency') {
                    # для агентств - берем агентсткие страны/валюты
                    $country_currencies = Currencies::get_currencies_by_country_hash_for_agency($uid, undef);
                } elsif($agency_id) {
                    # для субклиентов - берем агентсткие страны/валюты
                    $country_currencies = Currencies::get_currencies_by_country_hash_for_agency(undef, $agency_id);
                } else {
                    # для остальных (самостоятельные, сервисируемые) - неагентсткие
                    $country_currencies = Currencies::get_currencies_by_country_hash_not_for_agency($uid);
                }

                if (!defined $country || ($country > 0 && !exists $country_currencies->{$country})) {
                    return iget("Недопустимая страна: %s", $country);
                }
                $client_data_updates{country_region_id} = $country;
                if ($country) {
                    $client_data_updates_balance{REGION_ID} = $country;
                }

                my $currencies = $country_currencies->{$country};
                my $currency = $FORM{currency};
                if (!$currency || ((!is_valid_currency($currency) || none {$_ eq $currency} @$currencies) && $currency ne 'YND_FIXED')) {
                    return iget("Недопустимая валюта: %s", $currency);
                }
                $client_data_updates{work_currency} = $currency;
                if ($currency ne 'YND_FIXED') {
                    $client_data_updates_balance{CURRENCY} = $currency;
                    $client_data_updates_balance{MIGRATE_TO_CURRENCY} = Client::get_client_migrate_to_currency($vars->{ClientID});
                }
            }

            my $need_bs_resync_banners;
            if ($client_client_id && ($login_rights->{super_control} || $login_rights->{placer_control} || $login_rights->{support_control})) {
                my $no_display_hrefs = !!$FORM{no_display_hrefs};
                $client_data_updates{no_display_hrefs} = $no_display_hrefs;
                my $old_client_data = get_client_data($client_client_id, [qw/no_display_hrefs/]);
                $need_bs_resync_banners = 1  if $old_client_data->{no_display_hrefs} != $no_display_hrefs;
            }

            if ($user_role eq 'agency' && ($login_rights->{manager_control} || $login_rights->{super_control})) {
                $client_data_updates{can_copy_ctr} = $FORM{can_copy_ctr} ? 1 : 0;
            }

            # Сохраняем город клиента
            if ($vars->{can_edit_subregion} && $FORM{geo_id} && ($FORM{geo_id} != $vars->{subregion_id})) {
                $client_data_updates{subregion_id} = $FORM{geo_id};
                $client_data_updates_balance{SUBREGION_ID} = $FORM{geo_id};
            }

            if ($user_role eq 'client'
                && ( $login_rights->{support_control} || $login_rights->{super_control} || $login_rights->{placer_control} )
            ) {
                $client_data_updates{suspend_video} = $FORM{suspend_video} ? 1 : 0;
            }

            if (($user_role eq 'manager' || $user_role eq 'support') && $login_rights->{super_control}) {
                if ($FORM{can_manage_price_packages} && $FORM{can_approve_price_packages}) {
                    return iget("Недопустимое сочетание прав на пакеты");
                }
                $client_data_updates{can_manage_price_packages} = $FORM{can_manage_price_packages} ? 1 : 0;
                $client_data_updates{can_approve_price_packages} = $FORM{can_approve_price_packages} ? 1 : 0;
            }

            # сохраняем изменения в клиента
            if (%client_data_updates) {
                create_update_client({client_data => {ClientID => $vars->{ClientID}, %client_data_updates}});

                if (($vars->{suspend_video} // 0) != ($client_data_updates{suspend_video} // 0)) {
                    Direct::VideoAdditions->resend_to_bs($vars->{ClientID}, BS::ResyncQueue::PRIORITY_RESEND_SUSPENDED_VIDEO);
                }
            }
            if ($agency_options) {
                Direct::AgencyOptions->new([$agency_options])->update();
            }

            if (%client_data_updates_balance) {
                my $is_error = update_client_id($UID, $vars->{ClientID}, \%client_data_updates_balance);
                die "Error updating ClientID $vars->{ClientID} in Balance" if $is_error;
                # изменение страны в Балансе влечёт за собой изменение доступных стран-валют и графика НДС
                if ($vars->{ClientID} && $vars->{ClientID} > 0) {
                    my $agency_id = Primitives::get_client_first_agency($vars->{ClientID});
                    Client::CurrencyTeaserData::fetch_client_multicurrency_teaser_data($vars->{ClientID}, $agency_id);
                    Client::NDSDiscountSchedule::sync_nds_schedule_for_clients([$vars->{ClientID}], rbac => $rbac);
                }
            }

            if ($need_bs_resync_banners) {
                # ситуация очень редкая, поэтому запрашиваем всё сразу
                my $bids = get_all_sql(PPC(uid => $uid), [
                        'select c.cid, b.bid
                        from users u
                        join campaigns c on c.uid = u.uid
                        join banners b on b.cid = c.cid
                        join banner_display_hrefs bdh on bdh.bid = b.bid',
                        WHERE => {'u.ClientID' => $client_client_id},
                    ]);
                BS::ResyncQueue::bs_resync($bids) if @$bids;
            }

            return;

        } else {
            $error = iget('Нужно ввести все поля ') . ' ' . ($user_role eq 'agency' ? iget('(скидка от 0 до 99%)') : '');
        }

        return $error;
}

=head2 agency_options_apply_form($UID, $uid, $login_rights, $vars, $agency_options, \%FORM);

    Проверить права оператора/агентства и, если можно, обновить настройки агентства $agency_options данными из формы
    %FORM

=cut

sub agency_options_apply_form {
    my ($UID, $uid, $login_rights, $vars, $agency_options, $FORM) = @_;

    my $is_login_internal =
        $login_rights->{manager_control} ||
        $login_rights->{super_control} ||
        $login_rights->{support_control};
    my $is_login_chief =
        $login_rights->{is_agency_chief} && $UID == $uid;

    my $front_agency_options = $FORM->{json_agency_options};

    if ($is_login_internal) {
        $agency_options->allow_clients_without_wallet(
            front_bool2int($front_agency_options->{allow_clients_without_wallet})
        );
    }

    if ($agency_options->allow_clients_without_wallet() &&
        ($is_login_internal || $vars->{agency_rep_type} eq 'chief' && $is_login_chief)
    ) {
        $agency_options->default_clients_with_wallet(
            front_bool2int($front_agency_options->{default_clients_with_wallet})
        );
    }
}

=head2 save_edited_user_API_settings

  Сохранение отредактированных параметров API пользователей

=cut

sub save_edited_user_API_settings {
    my ($uid, $login_rights, $client_id) = @{$_[0]}{qw/uid LOGIN_RIGHTS CLIENT_ID/};
    my %FORM = %{$_[0]{FORM}};

    my $user_data = {};

    if ($login_rights->{super_control} || $login_rights->{support_control}) {
        return iget("Incorrect api_units_daily") if $FORM{api_units_daily} && $FORM{api_units_daily} !~ /^\d+$/;
        $user_data = hash_merge($user_data, {api_units_daily => $FORM{api_units_daily}});

        for my $scheme (qw/XLS API/) {
            if (defined $FORM{"${scheme}_units"}) {
                if ($FORM{"${scheme}_units"} !~ m/^\d+$/) {
                    return (iget('Неправильный формат количества баллов %s: %s', "${scheme}_units", defined $FORM{"${scheme}_units"} ? $FORM{"${scheme}_units"}:"undefined" ));
                }
                my $uhost = new APIUnits({scheme => $scheme});
                $uhost->update_user_daily_units($uid, $FORM{"${scheme}_units"});
            }
        }

        if (defined $FORM{json_method_limits} && $client_id) {
            # обновляем ограничения числа ежедневных вызовов
            my %limits = %{$FORM{json_method_limits}};
            while (my ($key, $value) = each %limits) {
                set_special_user_option($client_id, $key.'_calls_limit', $value);
            }
        }
        
        if ($client_id) {
            my $api_enabled = $FORM{api_enabled} =~ /(Yes|No|Default)/ ? $FORM{api_enabled} : 'Default';
            my $client_options = {'api_enabled' => $api_enabled};
            if (defined $FORM{API5_units_limit}) {
                my $api5_today_units_limit = $FORM{API5_units_limit};
                if ($api5_today_units_limit !~ m/^\d+$/) {
                    return iget('Неправильный формат количества баллов %s: %s', "API5_units_limit", $api5_today_units_limit);
                }
                $client_options->{api5_units_daily} = $api5_today_units_limit;
            }

            API::ClientOptions::add($client_id, $client_options);

            # обновляем ограничения числа одновременных вызовов
            set_special_user_option($client_id, 'concurrent_calls', $FORM{'concurrent_calls'} || '');

            if ($FORM{image_pool_limit} != $BannerImages::Pool::MINIMUM_IMAGES_IN_POOL) {
                set_special_user_option($client_id, 'image_pool_limit', $FORM{'image_pool_limit'});
            } else {
                set_special_user_option($client_id, 'image_pool_limit', '');
            }
        }
        # обновляем разрешение доступа к заблокированным версиям API
        $user_data->{api_allow_old_versions} = $FORM{api_allow_old_versions} ? 'Yes' : 'No';

        if (exists $FORM{'advq_report_queue_priority'}) {
            my $priority = $FORM{'advq_report_queue_priority'};
            # пустое значение удаляет
            if (!defined $priority || is_valid_int($priority, 0)) {
                set_special_user_option($client_id, 'advq_report_queue_priority', $priority);
            }
        }
    }

    if ($login_rights->{manager_control} || $login_rights->{super_control} || $login_rights->{support_control}) {
        # обновляем лимит advq запросов для api пользователей
        if (is_valid_int($FORM{advq_queries_lim})) {
            $user_data->{advq_queries_lim} = $FORM{advq_queries_lim};
        } else {
            $user_data->{advq_queries_lim} = undef;
        }
        $user_data->{excel_rows_limit} = $FORM{excel_rows_limit} && $FORM{excel_rows_limit} =~ /^\d+$/ && $FORM{excel_rows_limit} > 0
            ? $FORM{excel_rows_limit} : 0;

        set_special_user_option($client_id, "api_reports_limit",
            is_valid_int($FORM{api_reports_limit})
                ? $FORM{api_reports_limit} : undef
        );

    }

    if ($login_rights->{super_control}) {
        # обновляем разрешение управлять кампаниями, созданными Геоконтекстом, через API
        $user_data->{api_geo_allowed} = $FORM{api_geo_allowed} ? 'Yes' : 'No';
    }
    if ($login_rights->{super_control} || $login_rights->{support_control}) {
        # обновляем разрешение пользователям создавать клиентов через API
        $user_data->{allow_create_subclients} = $FORM{allow_create_subclients} ? 'Yes' : 'No';
    }

    # Обновляем пользователя
    create_update_user($uid, $user_data);

    return;
}

=head2 save_subclients_rights

    Вынесено из save_edited_user, сохраняет права и описание пользователя

=cut

sub save_subclients_rights {

    my ($rbac, $uid, $UID, $login_rights, $client_client_id, $subclients_rights, $data, %O) = @_;

    for my $agency_client_id (keys %$subclients_rights) {

        my $already_removed_ssr; # уже отобрали права super-subclient-а, не нужно отбирать права на xls

        # make super|simple subclient
        if ($subclients_rights->{$agency_client_id}->{isSuperSubClient} && ! $data->{"isSuperSubClient_$agency_client_id"}) {
            # make simple subclient from super
            return iget("не удалось преобразовать супер-субклиента в простого") if rbac_make_simple_subclient($rbac, $agency_client_id, $uid);
            $already_removed_ssr = 1;
        } elsif (! $subclients_rights->{$agency_client_id}->{isSuperSubClient} && $data->{"isSuperSubClient_$agency_client_id"}) {
            # make super subclient from simple
            if (rbac_make_super_subclient($rbac, $agency_client_id, $uid)) {
                return iget("не удалось преобразовать простого субклиента в супер-субклиента");
            } else {
                my $agency_info = get_user_info($login_rights->{agency_control} ? $UID : rbac_get_chief_rep_of_agency($agency_client_id));
                my $client_info = get_user_info($uid);
                #TODO: для camps использовать Common::get_agency_camps(), предварительно сделав ее более универсальной. с добавлением флага short
                my $mailvars = {
                    agency_fio   => $agency_info->{fio},
                    agency_uid   => $agency_info->{uid},
                    client_fio   => $client_info->{fio},
                    client_login => $client_info->{login},
                    client_email => $client_info->{email},
                    camps        => get_all_sql(PPC(uid => $uid), 
                                   "SELECT cid, name FROM campaigns WHERE statusEmpty = 'No' AND uid = ? AND AgencyID = ?", 
                                   $uid, $agency_info->{ClientID}),
                };

                add_notification($rbac, 'make_supersubclient_to_agency', $mailvars) if defined $agency_info->{email};
            }
        }

        # save allowImportXLS
        my $new_allow_import_xls = $data->{"allowImportXLS_$agency_client_id"} ? 1 : 0;
        if ($subclients_rights->{$agency_client_id}->{allowImportXLS} != $new_allow_import_xls
            && ! ($already_removed_ssr && $new_allow_import_xls == 0) # уже отобрали права на редактирование, права на xls отобрались автоматически, вызов не нужен
        )
        {
            my $error_code = rbac_set_allow_import_xls($rbac, $agency_client_id, $uid, $new_allow_import_xls);
            if ($error_code) {
                return iget('не удалось запретить/разрешить право суперсубклиенту импортировать из excel-файла');
            }
        }

        # save allowTransferMoney
        my $new_allow_transfer_money = $data->{"allowTransferMoney_$agency_client_id"} ? 1 : 0;
        if ($subclients_rights->{$agency_client_id}->{allowTransferMoney} != $new_allow_transfer_money) {
            my $error_code = rbac_set_allow_transfer_money($rbac, $agency_client_id, $uid, $new_allow_transfer_money);
            if ($error_code) {
                return iget('не удалось запретить/разрешить право суперсубклиенту переносить средства с кампании на кампанию.');
            }
        }

        if ($O{save_description}) {
            # save client descriptions
            my $user_description = $data->{"client_description_$agency_client_id"};

            $user_description = substr($user_description, 0, $Client::MAX_USER_DESCRIPTION_LENGTH) if $user_description && length($user_description) > $Client::MAX_USER_DESCRIPTION_LENGTH;
            my $new_client_description = {
                agency_client_id => $agency_client_id
                , client_client_id => $client_client_id
                , client_description => $data->{"client_description_$agency_client_id"}
            };
            create_update_client({ agency_client_relations_data => $new_client_description} );
        }

    }
    return 0;
}

=head2 save_status_not_resident

    Сохраняет статус NotResident
    Параметры именованные:
        serviced - является ли клиент сервесируемым
        client_client_id - client_id клиента (если клиент)
        agency_client_id - client_id агенства (если агентство)
        not_resident - Yes/No

=cut

sub save_status_not_resident
{
    my ($rbac, %O) = @_;

    if ( ( $O{login_rights}{manager_control} || $O{login_rights}{support_control} || $O{login_rights}{super_control} )
        && ($O{serviced} || ($O{user_role} eq 'agency' && $O{login_rights}{SetAgencyNotResident})) ) {
        my @uids;
        if ($O{serviced}) {
            @uids = @{ rbac_get_client_uids_by_clientid($O{client_client_id}) } if $O{client_client_id};
        } else {
            @uids = @{ Rbac::get_reps(ClientID => $O{agency_client_id}, role => $ROLE_AGENCY) };
        }

        do_update_table(PPC(uid => \@uids), "users", {not_resident => $O{not_resident}}, where => {uid => \@uids});

        return 1;
    } else {
        return 0;
    }
}

1;
