package Stat::AddedPhrasesCache;

=head1 NAME

    Stat::AddedPhrasesCache

=head1 DESCRIPTION

    Модуль для упрощения работы с кешем добавленных фраз ДРФ и ПЗ.

=cut

use Direct::Modern;
use Digest::MD5;

use Yandex::ListUtils;
use Yandex::DBTools;
use Yandex::DBShards;

use ShardingTools;

use Settings;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    get_phrases_from_cache_by_key
/;

# Количество дней, которое мы будем хранить
my $CACHE_STORAGE_DAYS = 3;
# Количество кампании в чанке на удаление устаревших фраз
my $DELETE_CHUNK_SIZE = 100;

# Доступные типы добавленных фраз
my $TYPE_PLUS_PHRASE = 'plus';
my $TYPE_MINUS_PHRASE = 'minus';

=head3 get_phrases_from_cache_by_key($key => $ids, %O)

	Получить из кеша записи по номеру заказа или номеру группы.
	Принимает два обязательных аргумента:

	    - key - поле, в котором нужно искать. Сейчас поддерживает только значения "cid" и "OrderID".
	    - ids - список идентификаторов, соответствующих полю

	Так же есть возможность задать фильтрацию по типу:

	    - type => "plus" - если нужно получить только добавленные ключевые фразы;
	    - type => "minus" - если нужно получить только добавленные минус-фразы;

	Пример вызова:

	my $cached_phrases = get_phrases_from_cache_by_key(cid => [1, 2, 3, 4], type => 'minus');

	Возвращает следующую структуру:

	{
	    OrderID1 => {
	        pid1 => {
	            plus => {
	                0 => [123, 321, 234],
	                1 => [213],
	            },
	            minus => [...],
	        }
	        pid2 => {...},
	        ...
	    },
	    OrderID2 => {...},
	    ...
	}

=cut

sub get_phrases_from_cache_by_key {
    my ($key, $ids, %O) = @_;

    die "Invalid key field: $key" if ($key !~ /^(OrderID|cid)$/);
    die "Invalid type: $O{type}" if ($O{type} && $O{type} ne $TYPE_MINUS_PHRASE && $O{type} ne $TYPE_PLUS_PHRASE);

    my $phrases = get_all_sql(PPC($key => $ids),
        ["SELECT c.OrderID, apc.pid, apc.type, apc.phrase_hash,
                 IF(apc.bids_id > 0 AND bi.id IS NULL AND apc.type = 'plus', 0, 1) AS bs_add_state
            FROM added_phrases_cache apc
                JOIN campaigns c ON c.cid = apc.cid
                LEFT JOIN bids bi ON (apc.bids_id != 0 AND bi.id = apc.bids_id)",
          WHERE => {
                "c.$key" => SHARD_IDS,
                "c.OrderID__ne" => 0,
                ($O{type} ? ("apc.type" => $O{type}): ())
            }]);

    my %id2phrases;
    for my $p (@$phrases) {
        next if !$p->{OrderID};
        push @{ $id2phrases{$p->{OrderID}}->{$p->{pid}}->{$p->{type}}->{$p->{bs_add_state}} }, $p->{phrase_hash};
    }

    return \%id2phrases;
}


=head2 delete_old_phrases_from_cache(%O)

	Удалить все записи старше заданного возраста из таблиц с кешем.
	Принимает два необязательных именованных аргумента:

	    - shard - номер шарда, в котором нужно производить очистку. Если не указан, очистка производится во всех шардах.
	    - max_days_in_storage - возраст в днях, объекты старше которого нужно удалять. По умолчанию $CACHE_STORAGE_DAYS

	Пример вызова:

	Stat::AddedPhrasesCache::delete_old_phrases_from_cache(shard => 1, max_days_in_storage => 2);

	Возвращает количество удаленных строк.

=cut

sub delete_old_phrases_from_cache {
    my (%O) = @_;

    my $max_days_in_storage = $O{max_days_in_storage} || $CACHE_STORAGE_DAYS;
    die "Invalid max_days_in_storage value: $max_days_in_storage" if $max_days_in_storage !~ /^\d+$/;

    my @shards;
    if ($O{shard}) {
        push @shards, $O{shard};
    } else {
        push @shards, ppc_shards();
    }

    my $cnt = 0;
    for my $shard (@shards) {
        my $ids_to_delete = get_one_column_sql(PPC(shard => $shard),
            "SELECT DISTINCT cid FROM added_phrases_cache WHERE add_date < DATE_SUB(NOW(), INTERVAL $max_days_in_storage DAY)"
        );

        for my $chunk (chunks($ids_to_delete, $DELETE_CHUNK_SIZE)) {
            $cnt += do_delete_from_table(PPC(shard => $shard), "added_phrases_cache",
                where => {
                    cid => $chunk,
                    add_date__lt__dont_quote => "DATE_SUB(NOW(), INTERVAL $max_days_in_storage DAY)",
                }
            );
        }
    }
    return $cnt;
}

1;
