package Stat::CustomizedArray;

=pod

    $Id$

=head1 NAME

    Stat::CustomizedArray

=head1 DESCRIPTION

    Автоматически выбирает между быстрой и обычной версией отчета, по сути
    являясь фабрикой между:
      Stat::CustomizedArray::DBStat - вспомогательные методы, детальную статистику НЕ умеет
      Stat::CustomizedArray::StreamedExt - МОЛ,

=head2 Интферфейс
    new - конструктор
    set_report_parameters - параметры отчета для генерации
    generate - генерирует и возвращает отчет
    is_stream - истина если отчет с текущими параметрами быстрый

    все остальные методы внутренние, либо для классов наследников
=cut

use Direct::Modern;

use base qw/Stat::CustomizedArray::DBStat 
            Stat::CustomizedArray::StreamedExt/;

# настройки для разных методов проверки доступности, выгрузки статистики
my %streamed_checkers = (is_streamed_ext => {generator => 'generate_streamed_ext_with_simple_params', 
                                             iterator  => 'generate_streamed_ext_iterator_with_simple_params', 
                                             flag => 'streamed_ext_choosen'},
                         );

=head2 generate
    Отдает статистику с самыми обобщёнными группировками, фильтрами и
    лимитом. Фильтр работает только на уровне SQL-запроса

    Не умеет фильтровать на лету, не формирует общую статистику по
    всему запросу. Поддерживает лимиты.

    На лету переключается на быстрые отчеты там, где это возможно.
=cut

sub generate {
    my ($self, $opts) = @_;
    
    return $self->_generate($opts, 'is_streamed_ext');

}

=head2 generate_with_metrika
    Аналогично generate, но проверяем возможность формирования отчета с условием получения данных из метрики
=cut

sub generate_with_metrika {
    my ($self, $opts) = @_;

    return $self->_generate($opts, 'is_streamed_ext');
}

=head2 generate_iterator
    Аналогичен методу generate, только возвращается итератор, позволяющий получать статистику почанково
    методы итератора:
        headers - содержит такие данные как totals, row_num,...
        next_chunk - возвращает очередную пачку данных, [...] или undef
=cut

sub generate_iterator {
    my ($self, $opts) = @_;
    
    return $self->_generate_iterator($opts, 'is_streamed_ext');

}

sub _generate {
    my $self = shift;
    my $opts = shift;
    my @streamed_checker_names = @_; # список методов в порядке убывания приоритета которыми 
                                          # проверяем возможность использования соответствующих быстрых отчетов

    if (my $checker_name = $self->_choose_streamed_checker(@streamed_checker_names)) {
        my $generator = $streamed_checkers{$checker_name}->{generator};
        return $self->$generator($opts);
    }

    # генерируем обычную статистику если быстрая недоступна для текущих
    # параметров отчета
    return $self->generate_dbstat;
}

sub _generate_iterator {
    my $self = shift;
    my $opts = shift;
    my @streamed_checker_names = @_; # список методов в порядке убывания приоритета которыми 
                                          # проверяем возможность использования соответствующих быстрых отчетов

    if (my $checker_name = $self->_choose_streamed_checker(@streamed_checker_names)) {
        my $iterator = $streamed_checkers{$checker_name}->{iterator};
        return $self->$iterator($opts);
    }

    return $self->generate_dbstat_iterator;
}

sub _choose_streamed_checker {
    my $self = shift;
    my @streamed_checker_names = @_; # список методов в порядке убывания приоритета которыми 
                                     # проверяем возможность использования соответствующих быстрых отчетов

    for my $checker_name (@streamed_checker_names) {
        next unless $streamed_checkers{$checker_name};

        if ($self->$checker_name) {
            $self->{$streamed_checkers{$checker_name}->{flag}} = 1;
            return $checker_name;
        }
    }

    return undef;
}

1;

