package Stat::CustomizedArray::Base;

=pod

    $Id$

=head1 NAME

    Stat::CustomizedArray::Base

=head1 DESCRIPTION

    Базовый класс для
    Stat::CustomizedArray::DBStat
    Stat::CustomizedArray::StreamedExt

=cut

use strict;
use warnings;

use utf8;

use List::MoreUtils qw/uniq any/;
use JSON; # for to_json

use Settings;

use YaCatalogApi;
use Stat::Tools qw/periods_suffix_list/;
use Stat::Const qw/:base :enums/;
use DirectCache;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::Validate qw/is_valid_int is_valid_float/;
use Yandex::HashUtils qw/hash_cut hash_merge/;
use Yandex::SendMail qw/send_alert/;

use Direct::PhraseTools qw/get_norm_phrase unquoted_key_phrase_from_text/;

use CampaignTools;

my @REPORT_PARAMETERS = (qw/oid group_by date_aggregation_by filter limits stat_type/,
                         map { ("start_date$_", "end_date$_") } ('', periods_suffix_list()) );

=head2 new(OrderID => 123, dbh => $dbh, %report_parameters)
    Конструктор класса, совместим с DBStat->new (сохраняет в созданный объект свойства которые используются в DBStat)
    Возможные параметры
        Для DBStat->new(OrderID => 1234, dbh => $dbh)
          OrderID - база данных будет выбрана автоматически
          dbh - будет использован указанный dbh

    %report_parameters
        Параметры уникальные для Stat::CustomizedArray->new(..), задают
        опции самого запроса, описание см. в set_report_parameters
        Эти же опции можно задать отдельно вызвав set_report_parameters
=cut

sub new {
    my $class = shift;
    my %params = @_;
    my $base_params = hash_cut \%params, qw/OrderID dbh/;

    delete $params{$_} foreach (keys %$base_params);

    my $self = bless $base_params, $class;
    
    $self->set_report_parameters(%params) if scalar keys %params;
    return $self;
}

sub report_parameters { @REPORT_PARAMETERS }
sub oid { shift->{oid} }
sub start_date {
    my ($self, $suf) = @_;
    return $self->{"start_date".($suf // '')};
}
sub end_date {
    my ($self, $suf) = @_;
    return $self->{"end_date".($suf // '')};
}

sub date_aggregation_by { shift->{date_aggregation_by} }

sub group_by { shift->{group_by} || [] }

=head2 group_by_hashref
    возвращает по заданным ранее параметрам отчета детали группировки в виде ссылки на хеш
=cut

sub group_by_hashref {
    my $self = shift;
    return Stat::Tools::get_group_by_as_hash(
        $self->group_by, $self->date_aggregation_by
    );
}
sub filter { shift->{filter}    || {} }
sub limits { shift->{limits}    || {} }
sub options{ shift->{options}   || {} }
sub order_by { shift->limits->{order_by}    || [] }
sub limit { shift->limits->{limit} }
sub offset { shift->limits->{offset} }

sub option_use_page_id { shift->options->{use_page_id} }
sub option_with_discount { shift->_option_or_null('with_discount') }
sub option_with_nds { shift->_option_or_null('with_nds') }
sub option_no_spec_and_all_prefix { shift->_option_or_null('no_spec_and_all_prefix') }

=head2 option_with_reach_stat

    получить статистику с данными по охватным кампаниям

=cut

sub option_with_reach_stat { shift->_option_or_null('with_reach_stat') }

=head2 option_compare_periods

=cut

sub option_compare_periods { shift->_option_or_null('compare_periods') }

sub _option_or_null {
    my $self = shift;
    my $option_name = shift;
    return defined $self->options->{$option_name}
        ? $self->options->{$option_name}
        : 0;
}

sub _master_cid_or_cid {
    my $master_cid_by_cid = shift;
    my $cid = shift;
    return $master_cid_by_cid->{$cid} // $cid;
}


=head2 set_report_parameters
    Задает опции отчета
        oid -- № заказа в БК
        start_date(''|_a|_b) -- дата "с"
        end_date(''|_a|_b) -- дата "по"
        group_by -- массив, по элементам которого делаются группировки
                    [qw/date
                        banner
                        adgroup
                        image
                        page
                        geo
                        phrase
                        retargeting
                        dynamic
                        phraseid
                        retargeting_coef # поддерживается только в МОЛ
                        contexttype
                        position
                        tag
                        device_type
                        age
                        gender
                        detailed_device_type
                        connection_type /]
        date_aggregation_by -- способ групировке по дате, возможные
                                значения week month quarter year day
        filter -- условия фильтрации
                   { page_target_type => 3 # или, например, '!3'
                     banner => [qw/123 321/], # BannerID
                     image => (with_img|without_img),
                     position => (prime|non-prime),
                     tag_ids => [qw/123 321/],
                     adgroup => [qw/111 222/],
                     phrase => 'трактор, комбайн нива',
                     phraseid => [qw/333 444/], # PhraseID
                     retargeting => 'мое условие ретаргетинга',
                     dynamic => 'мое условие нацеливания',
                     retargeting_coef => [qw/111 222/], # ret_cond_id
                     geo => '225,-1'
                     page => 'yandex.ru', # pages.Name
                     page_group => 'yandex.0', # pages.group_nick
                     page_id => '1001,1002',
                     goal_id => 123,
                     device_type => 'mobile', # или [qw/mobile tablet desktop/]
                     age => '0-17', # или [qw/undefined 0-17 18-24 25-34 35-44 45-/]
                     gender => 'male', # или [qw/undefined male female/]
                     detailed_device_type => 'ios', # или [qw/undefined other ios android/]
                     connection_type => 'mobile', # или [qw/undefined stationary mobile/]
                     template_id => '3131',
                   }
        limits -- дополнительные ограничения и параметры
                   { order_by => ['field1', 'field2', ...],
                     limit => 100,
                     offset => 20,
                   }
        options -- ещё одни дополнительные параметры :)
                    with_discount => 1,
                    with_nds => 1,
                    use_page_id => 1,
                    single_currency => 1, # перевести все суммы в другую
                                          # валюту (рубли)
                    currency => 'CURRENCY_NAME'
                                          # сконвертировать суммы в
                                          # указанную валюту (на текущий день).
                                          # Имеет смысл при
                                          # single_currency => 1,
                                          # по-умолчанию значения
                                          # приводятся к рублям
                    # настройки ниже нужны для МОЛ, с целью ускорения запросов в БК и экономии памяти как в БК так и в Директе
                    countable_fields_by_targettype => 0|1 (нужно ли измеримые поля разбивать на поиск/контекст, используется в МОЛ)
                    without_totals => 0|1 (не нужны тоталзы - возвращаются нули, используется в МОЛ)
                    partial_total_rows => 0|1 (возвращается не честный row_num, но достаточный для пейджинга в интерфейсе МОЛ)
                    with_winrate => 0|1 (нужна ли информация по "% полученных показов", используется в МОЛ)
                    no_spec_and_all_prefix => 0|1, по-умолчанию 0, если 1 то не отдаем поля со spec_, all_ префиксами (работает опция пока не во всех источниках)
                    totals_separate => 0|1, по-умолчанию 0, если 1 то тоталы и вычислимые от них поля отдаются в отдальном хеше totals, и без префикса "t"
                    compare_periods => 0|1 (показывает необходимость сравнения статистики из двух периодов, поддерживается только в МОЛ)
                    without_round   => 0|1, по-умолчанию 0, если 1 то значения не округляются в рамках Stat::Tools::calc_avg 
                                            (стоит использовать если значения проходят дальнейшую обработку/агрегацию, и округляются позже)
                    consumer => api|web_ui|default, потребитель статистики (работает в МОЛ, влияет на наименования значений в словарных полях)
                    filter_by_consumer_values => 0|1, если 1 значит значения в фильтрах по словарным полям заданы из набора потребителя
                    external_countable_fields_override => [...], поля, которые нужно подставить в countable_fields после перевода в формат БК
        translocal_params -- данные для определения правильного геодерева (транслокальности)
                    Возможные параметры хеша описаны в GeoTools::get_translocal_georeg
        ClientID_for_stat_experiments - ClientID для которого запрашивается статистика, может
                    использоваться при выборе хоста для запроса БК (прод/пре-прод)

    Все параеметры доступны по одноименному ассессору
=cut

sub set_report_parameters {
    my ($self, %parameters) = @_;
    my @plist = $self->report_parameters;
    @{$self}{ @plist } = @parameters{ @plist };
    $self->{OrderID} = $self->oid if !ref($self->{oid}) && defined $self->oid;
    $self->{options} = $parameters{options}||{};
    $self->{translocal_params} = $parameters{translocal_params}||{};
    $self->{ClientID_for_stat_experiments} = $parameters{ClientID_for_stat_experiments};

    if ( $self->{group_by} ) {
        $self->{group_by} = [ uniq @{ $self->{group_by} } ];
    }
}

=head2 add_group_data(\@rows)
    Добавляет в отчет информацию для группировки
=cut

sub add_group_data {
    my $self = shift;
    my $rows = shift;

    for my $group (@{$self->group_by}) {
        $self->_group_by($group, $rows);
    }
}

=head2 _group_by($group_by, \@rows)
    Коллекция методов группировки для статистики
    $group_by - поле группировки
    @rows - массив строк статистики
=cut

sub _group_by {
    my $self = shift;
    my $group_param = shift;
    my $rows = shift;

    if($group_param eq 'banner') {
        $self->group_by_banner($rows);
    } elsif (any { $group_param eq $_ } @Stat::Const::ANY_PHRASES_GROUP_BY, qw/contextcond contextcond_ext contextcond_orig contextcond_mod contextcond_ext_mod contextcond_orig_mod/) {
        $self->group_by_phrase($rows);
    } elsif ($group_param eq 'position') {
        $self->group_by_position($rows);
    } elsif (any { $group_param eq $_ } qw/geo region/) {
        $self->group_by_geo($rows);
    } elsif (any { $group_param eq $_ } qw/physical_region/) {
        $self->group_by_physical_geo($rows);
    } elsif ($group_param eq 'date') {
        $self->group_by_date($rows);
    } elsif (any { $group_param eq $_ } qw/page page_name/) {
        $self->group_by_page($rows);
    } elsif (any { $group_param eq $_ } qw/tag tags/) {
        $self->group_by_tag($rows);
    } elsif ($group_param eq 'adgroup') {
        $self->group_by_adgroup($rows);
    } elsif ($group_param eq 'age') {
        $self->group_by_age($rows);
    } elsif ($group_param eq 'gender') {
        $self->group_by_gender($rows);
    } elsif ($group_param eq 'campaign') {
        $self->group_by_campaign($rows);
    } elsif ($group_param eq 'retargeting_coef') {
        $self->group_by_retargeting_coef($rows);
    } elsif ($group_param eq 'search_query_status') {
        $self->group_by_search_query_status($rows);
    } elsif ($group_param eq 'ext_phrase_status') {
        $self->group_by_ext_phrase_status($rows);
    } elsif ($group_param eq 'deal') {
        $self->group_by_deal($rows);
    } elsif ($group_param eq 'targeting_category') {
        $self->group_targeting_category($rows);
    } elsif ($group_param eq 'prisman_income_grade') {
        $self->group_prisma_income_grade($rows);
    }
}

=head2 group_by_banner(\@rows)
    Группирует данные по баннерам
=cut

sub group_by_banner {
    my $self = shift;
    my $rows = shift;
    # хэш для баннеров, не используем хэширование в get_banner_info
    # т.к. правим title
    my $banner_key = $self->options->{banner_key} || 'BannerID';
    my $banners_info = Stat::Tools::mass_get_banner_info($banner_key => [uniq map { $_->{$banner_key} } @$rows], 
                                                         translocal_params => $self->{translocal_params},
                                                         sharding_params => {OrderID => $self->oid});
    foreach my $bn_info (values %$banners_info) {
        next unless $bn_info->{title};
        $bn_info->{title} =~ s/&amp;/&/g;
        $bn_info->{title} =~ s/&quot;/\"/g;
    }

    my @cids = uniq grep { $_ } map { $_->{cid} } values %$banners_info;
    my $master_cid_by_cid = get_hash_sql(PPC(cid => \@cids), ['SELECT sc.cid, sc.master_cid
                                                               FROM subcampaigns sc',
                                                               WHERE => {'sc.cid' => SHARD_IDS}]);

    foreach my $row (@$rows) {
        if (!is_valid_int($row->{$banner_key}, 1)) {
            my $msg = sprintf "Skipping bad $banner_key (%s) for OrderID %s and dates between %s and %s: %s",
                                $row->{$banner_key}, $self->oid, $self->start_date, $self->end_date, to_json($row);
            send_alert(Carp::longmess($msg), 'get_stat_customized_array error');
            next;
        }
        # Работаем с баннером
        my $banner = $banners_info->{$row->{$banner_key}};
        $row->{title} = $banner->{title};
        $row->{bid} = $banner->{bid};
        $row->{BannerID} //= $banner->{BannerID};
        $row->{adgroup_id} //= $banner->{pid};
        $row->{cid} = _master_cid_or_cid($master_cid_by_cid, $row->{cid} // $banner->{cid});
        $row->{ClientID} //= $banner->{ClientID};
    }
}


=head2 group_by_phrase(\@rows)
    Группирует данные статистики по ключевым словам или условиям ретаргетинга
=cut

sub group_by_phrase {
    my $self = shift;
    my $rows = shift;

    my (%phrase_ids, %ret_ids, %dyn_ids, %perf_ids);
    foreach my $row (@$rows) {
        if (any { $row->{ContextType} == $_ } ($CONTEXT_TYPE_PHRASE, $CONTEXT_TYPE_BM_SYNONYM)) {
            $phrase_ids{$row->{PhraseID}} = undef;
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_RET) {
            $ret_ids{$row->{PhraseID}} = undef;
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_DYNAMIC) {
            $dyn_ids{$row->{PhraseID}} = undef;
        } elsif ($row->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
            $perf_ids{$row->{PhraseID}} = undef;
        }
    }

    my $retargetings_data = %ret_ids
        ? Stat::Tools::get_retargetings_data([keys %ret_ids], OrderID => $self->oid)
        : {};

    my $dynamic_data = Stat::Tools::get_dynamic_data([keys %dyn_ids],
                                                     OrderID => $self->oid,
                                                     with_condition => (defined $self->options->{dynamic_data_ref} ? 1 : 0),
                                                     );
    if (defined $self->options->{dynamic_data_ref}) {
        ${ $self->options->{dynamic_data_ref} } = $dynamic_data;
    }

    my $performance_text = Stat::Tools::get_performance_text([keys %perf_ids],
                                                             OrderID => $self->oid,
                                                             );

    foreach my $ph (@$rows) {
        if (any { $ph->{ContextType} == $_ } ($CONTEXT_TYPE_PHRASE, $CONTEXT_TYPE_BM_SYNONYM)) {
            $ph->{phrase} //= '';
            if (defined $ph->{PhraseID} && $ph->{PhraseID} == $BROADMATCH_PHRASE_ID &&
                ($ph->{phrase} eq '' || $ph->{phrase} eq $Stat::Tools::BROADMATCH_PHRASE)) {
                $ph->{phrase} = Stat::Tools::special_phrase_title($BROADMATCH_PHRASE_ID);
            }
        } elsif ($ph->{ContextType} == $CONTEXT_TYPE_RET) {
            $ph->{phrase} = $retargetings_data->{$ph->{PhraseID}}->{name} || '';
            $ph->{is_interest} = ($retargetings_data->{$ph->{PhraseID}}->{type} || '') eq 'interest' ? 1 : 0;
        } elsif ($ph->{ContextType} == $CONTEXT_TYPE_DYNAMIC) {
            $ph->{phrase} = $dynamic_data->{$ph->{PhraseID}}->{name} || '';
        } elsif ($ph->{ContextType} == $CONTEXT_TYPE_PERFORMANCE) {
            $ph->{phrase} = $performance_text->{$ph->{PhraseID}} || '';
        } elsif ($ph->{ContextType} == $CONTEXT_TYPE_BROADMATCH && 
                 defined $ph->{PhraseID} && 
                 ($ph->{PhraseID} == $BROADMATCH_PHRASE_ID ||
                    $ph->{PhraseID} == $BROADMATCH_EXT_PHRASE_ID && $ph->{phrase} eq '')
                ) {
            $ph->{PhraseID} = $BROADMATCH_PHRASE_ID;
            $ph->{phrase} = Stat::Tools::special_phrase_title($BROADMATCH_PHRASE_ID);
        } elsif ($ph->{ContextType} == $CONTEXT_TYPE_RELEVANCE_MATCH &&
                  defined $ph->{PhraseID} &&
                  $ph->{PhraseID} == $RELEVANCE_MATCH_PHRASE_ID
                ) {
            $ph->{phrase} = iget($RELEVANCE_MATCH_PHRASE_PLACEHOLDER);
        } else {
            # фразы могу придти уже заполненые, поэтому не затираем, особенно актуально для
            # PhraseID = 1
            $ph->{phrase} ||= '';
        }

        if ($ph->{phrase} =~ /^\@(\d+)/) {
            $ph->{category_id} = $1;
            $ph->{phrase} = get_category_name($1);
        }
    }
}

=head2 group_by_position(\@rows)
    Группирует данные статистики по позициям
=cut

sub group_by_position {
    my $self = shift;
    my $rows = shift;

    foreach my $row (@$rows) {
        next if defined $row->{position};
        
        my $typeid = delete $row->{TypeID};
        if ( any { $typeid == $_} ($Stat::Const::PRIME_TYPE_ID, $Stat::Const::NON_PRIME_TYPE_ID, $Stat::Const::ALONE_TYPE_ID, $Stat::Const::SUGGEST_TYPE_ID, $Stat::Const::ADV_GALLERY_TYPE_ID) ) {
            $row->{position} = $typeid;
        } else {
            $row->{position} = 0;
        }
    }
}

=head2 group_by_geo(\@rows)
    Группирует данные статистики по регионам
=cut

sub group_by_geo {
    my $self = shift;
    my $rows = shift;

    foreach my $row (@$rows) {
        next if defined $row->{region_name};

        hash_merge $row, Stat::Tools::get_region_info($row->{region}, 'region');
    }
}

=head2 group_by_physical_geo(\@rows)
    Группирует данные статистики по регионам 
=cut

sub group_by_physical_geo {
    my $self = shift;
    my $rows = shift;

    foreach my $row (@$rows) {
        next if defined $row->{physical_region_name};

        hash_merge $row, Stat::Tools::get_region_info($row->{physical_region}, 'physical_region');
    }
}

=head2 group_by_date (\@rows)
    Группирует данные статистики по датам
=cut

sub group_by_date {
    my $self = shift;
    my $rows = shift;

    my $direct_cache = new DirectCache(groups => ['stat_date_format']);
    foreach my $row (@$rows) {
        for my $suf ('', $self->option_compare_periods ? periods_suffix_list() : ()) {
            my $date = $row->{"stat_date$suf"};
            next unless $date;
            $row->{"date$suf"} = Stat::Tools::format_date( $date, $self->date_aggregation_by);
        }
    }
}

=head2 group_by_page(\@rows)
    Группирует данные статистики по площадкам
=cut

sub group_by_page {
    my $self = shift;
    my $rows = shift;

    foreach my $row (@$rows) {
        $row->{page_type} =
            $row->{TargetType} && $row->{TargetType} == 3
                ? iget('сети')
                : iget('поиск');
    }
}

=head2 group_by_tag(\@rows)
    Группирует данные статистики по тэгам
=cut

sub group_by_tag {
    my $self = shift;
    my $rows = shift;

    my @tags = ();
    foreach my $row (@$rows) {
        $row->{tags} = [ split /\s+/, $row->{tags} ];
        push @tags, @{$row->{tags}};
    }

    if (@tags) {
        my $tags = Tag::get_tags_by_tag_ids(uniq \@tags);
        foreach (@$rows) {
            $_->{tag_names} = join ', ', sort map { $self->options->{lc_tags} ? lc($tags->{$_}) : $tags->{$_} } @{$_->{tags}};
        }
    }
}

=head2 group_by_adgroup(\@rows)

    Группирует данные статистики по группам баннеров (баннеры без группы группируются в одну общую "группу")

=cut

sub group_by_adgroup {
    my $self = shift;
    my $rows = shift;

    my @adgroup_ids = uniq grep { $_ } map { $_->{adgroup_id} } @$rows;
    if (@adgroup_ids) {
        my $adgroups_info = Stat::Tools::get_adgroups_info(pid => \@adgroup_ids);
        my @cids = uniq grep { $_ } map { $_->{cid} } values %$adgroups_info;
        my $master_cid_by_cid = get_hash_sql(PPC(cid => \@cids), ['SELECT sc.cid, sc.master_cid
                                                                   FROM subcampaigns sc',
                                                                   WHERE => {'sc.cid' => SHARD_IDS}]);
        foreach my $row (@$rows) {
            my $adgroup_info = $adgroups_info->{ $row->{adgroup_id} };
            $row->{adgroup_name} = $adgroup_info->{group_name};
            $row->{adgroup_type} = $adgroup_info->{adgroup_type};
            $row->{adgroup_banners_count} = $adgroup_info->{banners_quantity};
            $row->{adgroup_archived_banners_count} = $adgroup_info->{banners_arch_quantity};
            $row->{cid} = _master_cid_or_cid($master_cid_by_cid, $row->{cid} // $adgroup_info->{cid});
        }
    }
}

=head2 group_by_age(\@rows)
    Группирует данные статистики по возрастным меткам
=cut

sub group_by_age {
    my $self = shift;
    my $rows = shift;

    my $AGES_REVERSE = Stat::Tools::reverse_consumer_dict( Stat::Tools::consumer_dict_by_name('AGES', $self->options->{consumer}) );
    foreach my $row (@$rows) {
        $row->{age} //= $AGES_REVERSE->{$row->{age_id}} if defined $row->{age_id};
    }
}

=head2 group_by_gender(\@rows)
    Группирует данные статистики по полу
=cut

sub group_by_gender {
    my $self = shift;
    my $rows = shift;

    my $GENDERS_REVERSE = Stat::Tools::reverse_consumer_dict( Stat::Tools::consumer_dict_by_name('GENDERS', $self->options->{consumer}) );
    foreach my $row (@$rows) {
        $row->{gender} //= $GENDERS_REVERSE->{$row->{gender_id}} if defined $row->{gender_id};
    }
}

=head2 group_by_campaign(\@rows)

    Группирует данные статистики по кампаниям

=cut

sub group_by_campaign {
    my $self = shift;
    my $rows = shift;

    my @cids = uniq grep { $_ } map { $_->{cid} } @$rows;
    if (@cids) {
        my $campaigns = get_hashes_hash_sql(PPC(cid => \@cids), [
            'select cid, name, href from campaigns left join camp_additional_data using (cid)',
            where => {cid => SHARD_IDS}
        ]);
        foreach my $row (@$rows) {
            next unless $row->{cid};
            $row->{camp_name} = $campaigns->{$row->{cid}}->{name};
            $row->{camp_url_path} = $campaigns->{$row->{cid}}->{href};
        }
    }
}

=head2 group_by_retargeting_coef(\@rows)

    Группирует данные статистики по корректировкам для условий подбора аудитории

=cut

sub group_by_retargeting_coef {
    my $self = shift;
    my $rows = shift;

    my %ret_ids;
    foreach my $row (@$rows) {
        $ret_ids{$row->{coef_ret_cond_id}} = undef if $row->{coef_ret_cond_id};
    }

    my $retargetings_text = %ret_ids
        ? Stat::Tools::get_retargetings_text([keys %ret_ids], OrderID => $self->oid, without_text_suffix => 1)
        : {};

    foreach my $ph (@$rows) {
        if ($ph->{coef_ret_cond_id}) {
            $ph->{coef_ret_cond_name} = $retargetings_text->{$ph->{coef_ret_cond_id}};
        }
    }
}

=head2 group_by_search_query_status(\@rows)

    Добавляет в данные статистики по статусу ПЗ нормированный поисковый запрос (с учетом фиксирования стоп-слов и игнорирования кавычек)

=cut

sub group_by_search_query_status {
    my $self = shift;
    my $rows = shift;
    my $source_type = shift // 'search_query';

    my %norm_phrases_cache = ();
    for my $row (@$rows) {
        my $source_text = $source_type eq 'search_query' ? $row->{search_query} : $row->{phrase};
        $norm_phrases_cache{$source_text} //= get_norm_phrase( unquoted_key_phrase_from_text( $source_text ) );
        $row->{"norm_$source_type"} = $norm_phrases_cache{$source_text};
    }
}

=head2 group_by_ext_phrase_status(\@rows)

    Добавляет в данные статистики по статусу АДФ нормированную добавленную фразу (с учетом фиксирования стоп-слов и игнорирования кавычек)

=cut

sub group_by_ext_phrase_status {
    my $self = shift;
    my $rows = shift;

    $self->group_by_search_query_status($rows, 'ext_phrase');
}

=head2 group_by_deal(\@rows)

    Группирует данные статистики по сделкам

=cut

sub group_by_deal {
    my $self = shift;
    my $rows = shift;

    my @deal_ids = uniq grep { $_ } map { $_->{deal_export_id} } @$rows;
    if (@deal_ids) {
        my $deals = CampaignTools::get_deals_by_deal_ids(\@deal_ids);
        my $deals_hash = {map { $_->{deal_id} => $_->{name} } @$deals};
        foreach my $row (@$rows) {
            next unless $row->{deal_export_id};
            $row->{deal_name} = $deals_hash->{$row->{deal_export_id}};
        }
    }
}

=head2 group_targeting_category(\@rows)

    Группирует данные статистики по категориям таргетинга

=cut

sub group_targeting_category {
    my $self = shift;
    my $rows = shift;

    my $TARGETING_CATEGORIES_REVERSE = Stat::Tools::reverse_consumer_dict( Stat::Tools::consumer_dict_by_name('TARGETING_CATEGORIES', $self->options->{consumer}) );
    foreach my $row (@$rows) {
        $row->{targeting_category} //= $TARGETING_CATEGORIES_REVERSE->{$row->{targeting_category}} if defined $row->{targeting_category};
    }
}

=head2 group_prisma_income_grade(\@rows)

    Группирует данные статистики по уровню дохода из призмы

=cut

sub group_prisma_income_grade {
    my $self = shift;
    my $rows = shift;

    my $PRISMA_INCOME_GRADES_REVERSE = Stat::Tools::reverse_consumer_dict( Stat::Tools::consumer_dict_by_name('PRISMA_INCOME_GRADES', $self->options->{consumer}) );
    foreach my $row (@$rows) {
        $row->{prisma_income_grade} //= $PRISMA_INCOME_GRADES_REVERSE->{$row->{prisma_income_grade}} if defined $row->{prisma_income_grade};
    }
}

1;
