package Stat::CustomizedArray::Iterator;

=pod

    $Id$

=head1 NAME

    Stat::CustomizedArray::Iterator

=head1 DESCRIPTION

    Почанково тянет данные из итератора источника статистики, "обогащает" их доп. даными (для группировочных ключей, и расчет разнообразных "формульных" полей),
    и отдает также почанково
    
    Обязательные методы:
        headers - возвращает накопленные общие данные по всей полученной статистике (все кроме основного массива со строками статистики)
        next_chunk(chunk_size => 123) - возвращает очередную пачку строк статистики, размер пачки можно регулировать параметром chunk_size

=cut

use Direct::Modern;

use Yandex::Trace;
use Yandex::Clone;
use Yandex::HashUtils;

use Settings;
use DirectCache;

# дефолтное количество строк, отдаваемое методом next_chunk итератора
our $ITERATOR_DEFAULT_CHUNK_SIZE = 50_000;

=head2 new()

    Конструктор класса
    Возможные параметры
        source - источника статистики: итератор (объект с методами headers и next_chunk) или ARRAYREF (обязателен)
        ca - ссылка на объект Stat::CustomizedArray, или его производные (обязателен)
        _process_data_ca_method_name($stat_data_chunk) - имя метода Stat::CustomizedArray для обработки пачки с данными
        _process_totals_ca_method_name($self->headers) - имя метода Stat::CustomizedArray для обработки тоталов
        _aggregate_headers_ca_method_name($self->headers, $stat_data_chunk) - имя метода Stat::CustomizedArray для досборки хедеров по ходу обработки данных
        _finalize_headers_ca_method_name($self->{headers}, $self->headers) - имя метода Stat::CustomizedArray для финализации хедеров после обработки всех данных,
                                                                             в параметрах $self->{headers} - хеш куда можно класть рассчитанные финальные данные
        headers - различные заголовки, полученные извне (не из итератора источника статистики)
            totals
            row_num
            stat_stream_ts
            days_num
            periods_num
            stat_source

=cut

sub new {
    my $class = shift;
    my %params = @_;

    my $self = hash_cut \%params, qw/source ca/;

    for (qw/source ca/) {
        die "$_ is required parameter" unless $self->{$_};
    }
    $self->{headers} = hash_merge {}, $params{headers};

    hash_copy $self, \%params, qw/_process_data_ca_method_name
                                  _process_totals_ca_method_name
                                  _aggregate_headers_ca_method_name
                                  _finalize_headers_ca_method_name
                                 /;

    $self->{_direct_cache} = new DirectCache(groups => ['stat_date_format']);

    bless $self, $class;
    return $self;
}

=head2 headers

    Возвращает ссылку на хеш с уже полученными/вычисленными заголовками полученной статистики (totals, stat_stream_ts ...)

=cut

sub headers {
    my $self = shift;
    
    # Клонируем headers, так как $self->{_process_totals_ca_method_name} модифицирует переданный хэш
    my $headers = yclone $self->{headers};
    hash_merge $headers, yclone($self->{source}->headers) unless $self->_source_is_array;

    if (my $method_name = $self->{_process_totals_ca_method_name}) {
        $self->{ca}->$method_name($headers);
    }

    return $headers;
}

=head2 next_chunk

    Возвращает очередную пачку строк статистики или undef
    входные параметры:
    %O - именованные параметры
        chunk_size => 1000 - переопределяем размер пачки запрашиваемых данных

=cut

sub next_chunk {
    my ($self, %O) = @_;
    my $profile = Yandex::Trace::new_profile('stat_ca_iterator:next_chunk');

    my $chunk_size = int($O{chunk_size} // 0) || $ITERATOR_DEFAULT_CHUNK_SIZE;
    if (my $chunk = $self->_source_next_chunk(chunk_size => $chunk_size)) {
        if (my $method_name = $self->{_process_data_ca_method_name}) {
            $self->{ca}->$method_name($chunk);
        }
        if (my $method_name = $self->{_aggregate_headers_ca_method_name}) {
            $self->{ca}->$method_name($self->{headers}, $chunk);
        }
        return $chunk;
    } else {
        if (my $method_name = $self->{_finalize_headers_ca_method_name}) {
            $self->{ca}->$method_name($self->{headers}, $self->headers);
        }
        return undef;
    }
}

=head2 _source_next_chunk

    Возвращает очередную пачку строк из источника статистики или undef
    входные параметры:
    %O - именованные параметры
        chunk_size => 1000 - определяем размер пачки запрашиваемых данных (обязательный параметр)

=cut

sub _source_next_chunk {
    my ($self, %O) = @_;

    die 'chunk_size is required parameter' unless $O{chunk_size};

    if ($self->_source_is_array) {
        if (scalar(@{$self->{source}})) {
            return [ splice(@{$self->{source}}, 0, $O{chunk_size}) ];
        } else {
            return undef;
        }
    } else {
        return $self->{source}->next_chunk(chunk_size => $O{chunk_size});
    }
}

=head2 _source_is_array

    Возвращает 1 если источник статистики - ссылка на массив (а не итератор)

=cut

sub _source_is_array {
    my $self = shift;

    return ref $self->{source} eq 'ARRAY' ? 1 : 0;
}

1;
