package Stat::CustomizedArray::StreamedExt;

=pod

    $Id$

=head1 NAME

    Stat::CustomizedArray::StreamedExt;

=head1 DESCRIPTION

    Версия Stat::CustomizedArray использующая только расширенные быстрые отчеты Stat::StreamExtended из БК

=head2 Интферфейс

    new - конструктор
    set_report_parameters - параметры отчета для генерации
    generate_streamed_ext - генерирует и возвращает отчет

    все остальные методы внутренние, либо для классов наследников

    см. Stat::CustomizedArray::Base

    NB: при прикрутке данного класса к Stat::CustomizedArray учесть 
        что фильтрация в Stat::StreamedExtended идет по bid, а не BannerID

=cut

use strict; 
use warnings;

use List::MoreUtils qw/uniq all any/;

use Settings;
use Stat::StreamExtended;
use Stat::Fields;
use Stat::Tools qw/suffix_list periods_suffix_list periods_full_suffix_list field_list_sum/;
use Stat::Const qw/:base/;
use Stat::CustomizedArray::Iterator;
use Stat::OrderStatDay;
use Campaign::Types;
use Client::ClientFeatures;
use Yandex::I18n;
use Yandex::TimeCommon;
use Yandex::ListUtils qw/xisect xflatten xuniq xminus/;
use Yandex::HashUtils;
use Yandex::DateTime qw/date/;
use Yandex::DBTools;
use Yandex::Clone;

use base qw/Stat::CustomizedArray::Base/;

sub _get_streamed_ext_source_iterator {
    my $self = shift;
    my $opts = shift // {};

    return Stat::StreamExtended::get_stream_stat(
        $self->stat_stream_ext_parameters(hash_cut($opts, qw/simple_params/)),
        four_digits_precision => $self->options->{four_digits_precision},
        is_mcc_mode => $self->options->{is_mcc_mode},
    );
}

=head2 generate_streamed_ext_iterator

    Возвращает расширенную быструю статистику по параметрам отчета в формате итератора, для почанковой обработки и использования

=cut

sub generate_streamed_ext_iterator {
     my $self = shift;
    my $opts = shift // {};    

    my $source_iterator = $self->_get_streamed_ext_source_iterator($opts);
    my %headers = (stat_stream_ts => $source_iterator->headers->{stat_time} ? mysql2unix($source_iterator->headers->{stat_time}) : 0,
                   stat_source => 'stream_ext', );

    my @cp_suffix_list = $self->_cp_suffix_list;
    if (!$opts->{no_need_days_num}) {
        # Считаем статистику за период
        for my $suf (@cp_suffix_list) {
            $headers{"days_num$suf"} = Stat::OrderStatDay::get_order_days_num(
                $self->oid, $self->start_date($suf), $self->end_date($suf));
        }
    }

    my $ca_iterator = Stat::CustomizedArray::Iterator->new(
                                    source => $source_iterator,
                                    ca => $self,
                                    _process_data_ca_method_name      => '_process_streamed_ext_stat_data',
                                    _process_totals_ca_method_name    => '_process_streamed_ext_stat_totals',
                                    _aggregate_headers_ca_method_name => '_aggregate_streamed_ext_stat_totals_from_data',
                                    _finalize_headers_ca_method_name  => '_finalize_streamed_ext_stat_totals',
                                    headers => \%headers);
    return $ca_iterator;
}

=head2 generate_streamed_ext_iterator_with_simple_params

    Возвращает расширенную быструю статистику по стандартным для Stat::CustomizedArray параметрам отчета в формате итератора, для почанковой обработки и использования

=cut

sub generate_streamed_ext_iterator_with_simple_params {
    my $self = shift;
    my $opts = shift // {};

    $opts->{simple_params} = 1;

    return $self->generate_streamed_ext_iterator($opts);
}

=head2 generate_streamed_ext

    Возвращает расширенную быструю статистику по параметрам отчета

=cut

sub generate_streamed_ext {
    my $self = shift;
    my $opts = shift // {};

    my $iterator = $self->generate_streamed_ext_iterator($opts);
    my @data_array = ();
    while (my $chunk = $iterator->next_chunk) {
        push @data_array, @$chunk;
    }

    my $headers = $iterator->headers;
    my $res = {
        data_array => \@data_array,
    };
    hash_merge $res, hash_cut($headers, qw/row_num stat_stream_ts stat_source/);
    for my $f (keys %$headers) {
        $res->{$f} = $headers->{$f} if $f =~ /^(days_num|periods_num)/;
    }

    if ($self->options->{totals_separate} || $self->option_compare_periods) {
        $res->{totals} = $headers->{totals};
    } else {
        hash_merge $res, $headers->{totals};
        Stat::Tools::rename_field($res);
        for ($self->_tt_suffix_list) {
            Stat::Tools::calc_average_campaign_stat($res, $_, $res->{days_num}, $res->{periods_num}, undef,
                spec_and_all_prefix => $self->option_no_spec_and_all_prefix ? 0 : 1,
                preserve_existing => 1,
                four_digits_precision => $self->options->{four_digits_precision},
            );
        }
    } 

    return $res;
}

sub generate_streamed_ext_with_simple_params {
    my $self = shift;
    my $opts = shift // {};

    $opts->{simple_params} = 1;

    return $self->generate_streamed_ext($opts);
}


sub _process_streamed_ext_stat_totals {
    my ($self, $headers) = @_;

    my @tt_suffix_list = $self->_tt_suffix_list;
    my @cp_suffix_list = $self->_cp_suffix_list;
    my $attribution_model = $self->filter->{attribution_models} || $self->filter->{attribution_model} || {};

    my %options = (
        spec_and_all_prefix => $self->option_no_spec_and_all_prefix ? 0 : 1,
        preserve_existing => 1,
        four_digits_precision => $self->options->{four_digits_precision},
        attribution_model => (ref $attribution_model eq 'HASH') ? $attribution_model->{eq} : $attribution_model
    );

    if ($self->option_compare_periods) {
        for (@cp_suffix_list) {
            Stat::Tools::calc_average_campaign_stat($headers->{totals}, $_, $headers->{"days_num$_"}, $headers->{"periods_num$_"}, '', %options);
        }
        Stat::Tools::calc_avg_compare_periods($headers->{totals},
            total_av => 1,
            %options
        );
    } else {
        for (@tt_suffix_list) {
            Stat::Tools::calc_average_campaign_stat($headers->{totals}, $_, $headers->{days_num}, $headers->{periods_num}, '', %options);
        }
    }
}

sub _process_streamed_ext_stat_data {
    my ($self, $stat_data_array) = @_;

    $self->options->{lc_tags} = 1;
    $self->options->{banner_key} = 'bid';

    $self->add_group_data($stat_data_array);

    my @tt_suffix_list = $self->_tt_suffix_list;
    my @cp_suffix_list = $self->_cp_suffix_list;

    my $attribution_model = $self->filter->{attribution_models} || $self->filter->{attribution_model} || {};
    my %options = (
        spec_and_all_prefix => $self->option_no_spec_and_all_prefix ? 0 : 1,
        without_round => $self->options->{without_round} ? 1 : 0,
        allow_null_shows => 1,
        four_digits_precision => $self->options->{four_digits_precision},
        attribution_model => $attribution_model,
    );
    if ($self->option_compare_periods) {
        foreach my $row (@$stat_data_array) {
            Stat::Tools::calc_avg_compare_periods($row, %options);
        }
    } else {
        foreach my $suf (@tt_suffix_list) {
            foreach my $row (@$stat_data_array) {
                Stat::Tools::calc_avg($row, $suf, %options);
            }
        }
    }
}

sub _aggregate_streamed_ext_stat_totals_from_data {
    my ($self, $headers, $stat_data_array) = @_;

    my @cp_suffix_list = $self->_cp_suffix_list;
    # если понадобится periods_num, с заданными лимитами - 
    # прийдется добавлять его со стороны БК
    if ($self->group_by_hashref->{date}) {
        unless ($self->limits->{limit} || $self->limits->{offset}) {
            for my $suf (@cp_suffix_list) {
                $headers->{"_uniq_periods$suf"} //= {};
                $headers->{"_uniq_periods$suf"}->{$_} = undef for map { $_->{"stat_date$suf"} } @{$stat_data_array};
            }
        }
    }
}

sub _finalize_streamed_ext_stat_totals {
    my ($self, $headers, $src_headers) = @_;

    my @cp_suffix_list = $self->_cp_suffix_list;
    for my $suf (@cp_suffix_list) {
        if ($headers->{"_uniq_periods$suf"}) {
            $headers->{"periods_num$suf"} = scalar(keys %{ $headers->{"_uniq_periods$suf"} });
            delete $headers->{"_uniq_periods$suf"};
        }
    }
    $headers->{row_num} = $src_headers->{total_rows};
}

sub _tt_suffix_list {
    my $self = shift;

    my @tt_suffix_list = '';
    if ($self->options->{countable_fields_by_targettype} &&
            (ref $self->options->{countable_fields_by_targettype} ne 'ARRAY' ||
             @{ xminus($self->options->{countable_fields_by_targettype}, [field_list_sum()]) }) ) {
        # в списке полей требующих разбиения по типу площадки есть неаггрегируемые поля
        @tt_suffix_list = suffix_list();
    }

    return @tt_suffix_list;
}

sub _cp_suffix_list {
    my $self = shift;

    return $self->option_compare_periods ? periods_suffix_list() : '';
}

=head2 set_report_parameters

    переопределение родительского метода

    группировки поддерживаемые в StreamedExt:
    group_by -- массив, по элементам которого делаются группировки
                [qw/date
                    campaign 
                    banner 
                    adgroup 
                    contextcond
                    contextcond_ext
                    contextcond_orig 
                    retargeting_coef
                    contexttype
                    contexttype_orig 
                    region 
                    physical_region
                    page 
                    targettype 
                    position 
                    tags 
                    has_image 
                    device_type 
                    click_place 
                    gender 
                    age
                    detailed_device_type
                    connection_type
                    search_query
                    search_query_status
                    ext_phrase_status
                    broadmatch_type
                    sim_distance
                    matched_phrase
                    match_type/]

    фильтры, поддерживаемые в StreamedExt:
    формат: adgroup => {eq => [123,456]},
            ctr     => {gt => 2, lt => 3}
    в целом возможные операции: eq, ne, lt, gt, starts_with, not_starts_with,
                                contains, not_contains, contains_in_plus_part, not_contains_in_plus_part
    возможные фильтры:
        banner      - bid
        adgroup     - pid
        phrase      - тексты фраз
        phrase_ext  - тексты фраз (в том числе тексты ДРФ-фраз)
        phrase_orig - тексты фраз (клиентские фразы, в том числе соответствующие ДРФ-показам)
        phraseid    - PhraseID
        retargeting - ret_cond_id
        dynamic     - dyn_cond_id
        matched_phrase - тексты фраз используемых при матчинге (ДРФ-фразы)
        match_type  - (none|rkw|syn|experiment)
        retargeting_coef - ret_cond_id
        contexttype - (phrases|retargeting|auto-added-phrases|dynamic|performance|relevance-match)
        contexttype_orig - (phrases|retargeting|dynamic|performance|relevance-match)
        region      - region_id
        physical_region - region_id
        page        - PageID
        targettype  - (search|context)
        position    - (prime|non-prime)
        tags        - текст меток
        tag_ids     - id меток
        has_image   - (0|1)
        device_type - (desktop|mobile|tablet)
        click_place - (title|sitelink1|sitelink2|sitelink3|sitelink4|sitelink5|sitelink6|sitelink7|sitelink8|vcard)
        gender      - (undefined|male|female)
        age         - (undefined|0-17|18-24|25-34|35-44|45-)
        detailed_device_type - (undefined|other|ios|android)
        connection_type      - (undefined|stationary|mobile)
        search_query - текст поискового запроса
        broadmatch_type - (undefined|adf|drf|experiment)

        goal_id     - GoalID (единичный)
        + измеримые поля

=cut

sub set_report_parameters {
    my ($self, %parameters) = @_;

    return $self->SUPER::set_report_parameters(%parameters);
}

=head2 stat_stream_ext_parameters

    Параметры отчета приготовленные для отправки в
    Stat::StreamExtended::get_stream_stat

=cut

sub stat_stream_ext_parameters {
    my $self = shift;
    my $opts = shift // {};

    # Поля неподдерживаемые в БК, дополняются на стороне Директа в Stat::CustomizedArray::Base
    # Список должен быть пустой
    my %only_local_fields = map { $_ => 1 } qw//;

    my $ext_params = $opts->{simple_params} 
                        ? $self->converted_simple_to_ext_params
                        : {group_by => $self->group_by,
                           filter => $self->filter,
                           order_by => $self->order_by};

    if (any { $only_local_fields{$_}} ( @{$ext_params->{group_by}}, 
                                        keys %{$ext_params->{filter}},
                                        map { $_->{field} } @{$ext_params->{order_by}} )) {
        $ext_params = yclone $ext_params;
        $ext_params->{group_by} = xminus($ext_params->{group_by}, [keys %only_local_fields]);
        @{$ext_params->{order_by}} = grep { !$only_local_fields{$_->{field}} } @{$ext_params->{order_by}};
    }

    my %date_params = map {(
            "date_from$_" => $self->start_date($_),
            "date_to$_" => $self->end_date($_),
        )} ($self->option_compare_periods ? periods_suffix_list() : '');
    my $client_id = $self->{ClientID_for_stat_experiments};
    my $operator_client_id = $self->options->{operator_ClientID} // $client_id;

    my %lang_aliases = (ua => 'uk');
    my $lang = Yandex::I18n::current_lang;
    $lang = $lang_aliases{$lang} || $lang;
    my $using_mol_page_name = 0;
    if (Client::ClientFeatures::has_mol_page_name_support($client_id)
        || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
    ) {
        # сообщаем Stat::StreamExtended::_process_row, что название площадки приходит из МОЛ напрямую,
        # - это влияет на способ локализации пейджа "yandex" в "Яндекс"
        $using_mol_page_name = 1;
    }

    my %parameters = (
        currency => $self->options->{currency} || 'YND_FIXED',
        order_ids => ref $self->oid ? $self->oid : [ $self->oid ] ,
        %date_params,
        with_discount => $self->option_with_discount,
        with_nds => $self->option_with_nds,
        group_by => $self->group_by_stat_stream_ext($ext_params->{group_by}, $client_id, $operator_client_id),
        filters => _translate_magic_page_name_filters($ext_params->{filter}, $lang),
        order_by => _order_by_stat_stream_ext($ext_params->{order_by}, $client_id, $operator_client_id),
        limits => {limit => $self->limit,
                   offset => $self->offset},
        extra_countable_fields => $self->options->{extra_countable_fields},
        countable_fields_by_targettype => $self->options->{countable_fields_by_targettype},
        without_totals => $self->options->{without_totals},
        partial_total_rows => $self->options->{partial_total_rows},
        with_winrate => $self->options->{with_winrate},
        compare_periods => $self->option_compare_periods,
        consumer => $self->options->{consumer},
        stat_type => $self->{stat_type},
        filter_by_consumer_values => $self->options->{filter_by_consumer_values},
        region_level => $self->options->{region_level},
        lang => $lang,
        ClientID_for_stat_experiments => $client_id,
        operator_ClientID => $operator_client_id,
        using_mol_page_name => $using_mol_page_name,
    );

    if ($self->option_with_reach_stat) {
        $parameters{extra_countable_fields} = [ uniq(@{$parameters{extra_countable_fields} // []}, qw/avg_cpm avg_view_freq uniq_viewers/) ];
    }

    if ($self->options->{external_countable_fields_override}) {
        $parameters{external_countable_fields_override} = $self->options->{external_countable_fields_override};
    }

    if (my $group_by_date = $self->group_by_hashref->{date}) {
        $parameters{group_by_date} = $group_by_date;
    }

    if (Client::ClientFeatures::creative_free_interface_enabled($client_id)) {
        $parameters{creative_free} = 1;
    }

    return %parameters;
}

=head2 group_by_stat_stream_ext

    group_by в формате понятном Stat::StreamExtended::get_stream_stat

=cut

sub group_by_stat_stream_ext {
    my $self = shift;
    my $group_by = shift // $self->group_by;
    my $client_id = shift;
    my $operator_client_id = shift;

    my @new_group_by;
    for my $field (@$group_by) {
        if ($field eq 'date') {
            next;
        }
        if ($field eq 'page') {
            my $new_page_group = 'page_group';
            my $new_page_detailed_group = 'page';
            if (Client::ClientFeatures::has_mol_page_name_support($client_id)
                || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
            ) {
                $new_page_group = 'page_name';
                $new_page_detailed_group = 'page_name_detailed';
            }
            push @new_group_by, $self->option_use_page_id ? $new_page_detailed_group : $new_page_group;
        } elsif ($field eq 'page_group') {
            my $new_field = $field;
            if (Client::ClientFeatures::has_mol_page_name_support($client_id)
                || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
            ) {
                $new_field = 'page_name';
            }
            push @new_group_by, $new_field;
        } elsif ($field eq 'page_name') {
            push @new_group_by, ($self->option_use_page_id ? 'page_name_detailed' : 'page_name');
        } else {
            push @new_group_by, $field;
        }
    }
    return \@new_group_by;
}

=head2 _order_by_stat_stream_ext($order_by, $client_id, $operator_client_id)

    Преобразование сортировки из легаси формата в понятный Stat::StreamExtended

=cut
sub _order_by_stat_stream_ext {
    my ($order_by, $client_id, $operator_client_id) = @_;

    my @new_order_by;
    for my $item (@$order_by) {
        my $new_item = hash_cut($item, qw/field dir/);
        if (Client::ClientFeatures::has_mol_page_name_support($client_id)
            || Client::ClientFeatures::has_mol_page_name_support($operator_client_id)
        ) {
            # конфуз: статистика по площадкам (не подробная) запрашивает группировку по `page_group` но
            # сортировку по `page`. Сортировка по `page` мапится в `PageSortName, PageID`, но `PageID` нет в группировке
            # по `page_group`.
            # Лишний `PageID` отрезается в Stat::StreamExtended::_add_orderby_opts
            if ($new_item->{field} eq 'page') {
                $new_item->{field} = 'page_name_detailed';
            }
            if ($new_item->{field} eq 'page_group') {
                $new_item->{field} = 'page_name';
            }
        }
        push @new_order_by, $new_item;
    }
    return \@new_order_by;
}

=head2 is_streamed_ext

    статистику возможно получить из МОЛ

=cut

sub is_streamed_ext {
    my $self = shift;

    my $df = date($self->start_date)->ymd('');
    my $dt = date($self->end_date)->ymd('');

    my %supported_filters = map { $_ => 1 } qw/
        banner image position page_target_type page_target tag_ids adgroup phrase phraseid retargeting
        retargeting_coef page page_group page_name geo page_id goal_id single_goal_id device_type
        age gender detailed_device_type connection_type banner_image_type attribution_model
    /;
    for (Stat::Fields::get_countable_filter_fields()) {
        $supported_filters{$_} = 1;
    }
    return 0 if any { !$supported_filters{$_} } grep { $self->filter->{$_} } keys %{ $self->filter };

    my %supported_group_by = map { $_ => 1 } (qw/
        campaign image position page page_name geo banner phraseid retargeting_coef contexttype tag
        adgroup device_type age gender date detailed_device_type connection_type banner_image_type statgoals agoalnum
        /, @Stat::Const::ANY_PHRASES_GROUP_BY);
    return 0 if any { !$supported_group_by{$_} } @{ $self->group_by };

    return 0 if !camp_kind_in(OrderID => $self->oid, 'stat_stream_ext');

    return 1;
}

=head2 converted_simple_to_ext_params

    конвертирует фильтры, группировки, сортировки принятые в DBStat и Stat::Stream в параметры принятые в Stat::StreamExtended

=cut

sub converted_simple_to_ext_params {
    my $self = shift;

    # для api4 используем срезы/фильтры отличные от тех которые используем везде в интерфейсах (старые)

    my $phrase_group = 'contextcond_orig';
    my $phrase_filter = 'phrase_orig';
    my $phraseid = 'phraseid_orig';
    my $contexttype = 'contexttype_orig';
    my $contexttype_dict = \%Stat::Const::CONTEXT_COND_TYPES;
    if ($self->options->{consumer} && $self->options->{consumer} eq 'api4') {
        $phrase_group = 'contextcond';
        $phrase_filter = 'phrase';
        $phraseid = 'phraseid';
        $contexttype = 'contexttype';
        $contexttype_dict = \%Stat::Const::CONTEXT_TYPES;
    } elsif ($self->option_with_reach_stat) {
        $phrase_group = 'criterion_cond';
        $phraseid = 'criterion_id';
        $contexttype = 'criterion_type';
        $contexttype_dict = \%Stat::Const::CRITERION_TYPES;
    }

    # фильтрация
    my $sfilter = $self->filter;
    $sfilter->{page_target_type} = Stat::Tools::page_target_type_number_from_filter($sfilter);
    delete $sfilter->{page_target};

    my $efilter = {};
    for my $field (grep { $sfilter->{$_} } keys %$sfilter) {
        if ($field eq 'banner') {
            $efilter->{banner} = {eq => _get_bids_by_bannerid($sfilter->{banner}, $self->oid)}
        } elsif ($field eq 'image') {
            $efilter->{has_image} =  {eq => $sfilter->{image} eq 'with_img' ? 1 : 0};
        } elsif ($field eq 'position') {
            $efilter->{position} =  {eq => $sfilter->{position}};
        } elsif ($field eq 'page_target_type') {
            $efilter->{targettype} =  {eq => $sfilter->{page_target_type} eq '3' ? 'context' : 'search'};
        } elsif ($field eq 'tag_ids') {
            $efilter->{tag_ids} =  {eq => $sfilter->{tag_ids}};
        } elsif ($field eq 'adgroup') {
            $efilter->{adgroup} =  {eq => $sfilter->{adgroup}};
        } elsif ($field eq 'phrase') {
            $efilter->{$phrase_filter} =  {starts_with => [grep { $_ } split /\s*,+\s*/, $sfilter->{phrase}]};
        } elsif ($field eq 'phraseid') {
            $efilter->{$phraseid} =  {eq => $sfilter->{phraseid}};
        } elsif ($field eq 'retargeting') {
            my $filter_ret_cond_ids = Stat::Tools::get_ret_cond_ids_by_name($sfilter->{retargeting}, $self->oid);
            $efilter->{retargeting} =  {eq => $filter_ret_cond_ids};
        } 
        elsif ($field eq 'dynamic') {
            my $filter_dyn_cond_ids = Stat::Tools::get_dyn_cond_ids_by_names($sfilter->{dynamic}, OrderID => $self->oid);
            $efilter->{dynamic} =  {eq => $filter_dyn_cond_ids};
        }
        elsif ($field eq 'geo') {
            my $geo_ids;
            if (ref $sfilter->{geo}) {
                $geo_ids = $sfilter->{geo};
            } else {
                $geo_ids = Stat::Tools::get_geo_ids_by_geo_text($sfilter->{geo});
            }
            $efilter->{region}  = {eq =>  Stat::Tools::get_plus_regions_by_geo($geo_ids, $self->{translocal_params})};
        } elsif ($field eq 'page') {
            $efilter->{page}  = {eq =>  Stat::Tools::get_page_id_by_name([split /,/, $sfilter->{page}])};
        } elsif ($field eq 'page_group') {
            $efilter->{page}  = {eq =>  Stat::Tools::get_page_id_by_group_nick($sfilter->{page_group})};
        } elsif ($field eq 'page_id') {
            $efilter->{page}  = {eq =>  $sfilter->{page_id}};
        } elsif ($field eq 'page_name') {
            $efilter->{page_name} = {eq => [$sfilter->{page_name}]};
        } elsif ($field eq 'goal_id') {
            $efilter->{goal_id}  = {eq =>  $sfilter->{goal_id}};
        } elsif ($field eq 'single_goal_id') {
            $efilter->{single_goal_id}  = {eq =>  $sfilter->{single_goal_id}};
        } elsif (any { $field eq $_ } qw/retargeting_coef device_type age gender detailed_device_type connection_type banner_image_type/) {
            $efilter->{$field} =  {eq => $sfilter->{$field}};
        } elsif ($field eq 'attribution_model') {
            $efilter->{attribution_model}  = {eq => $sfilter->{attribution_model}};
        } else {
            die "Unsupported filter field: $field";
        }
    }

    # группировка
    my $group_hash = $self->group_by_hashref();

    my @egroup_by = ();

    if (any { $group_hash->{$_} } @Stat::Const::ANY_PHRASES_GROUP_BY) {
        my %context_types = (phrase => [qw/phrases auto-added-phrases synonym/],
                             retargeting => 'retargeting',
                             dynamic => 'dynamic',
                             performance_filter => 'performance',
                             );
        push @egroup_by, $phrase_group;
        unless (all { $group_hash->{$_} } @Stat::Const::ANY_PHRASES_GROUP_BY) {
            $efilter->{$contexttype} //= {};

            my @selected_context_types = grep { exists $contexttype_dict->{$_} } map { xflatten($context_types{$_} // ()) } grep { $group_hash->{$_} } @Stat::Const::ANY_PHRASES_GROUP_BY;
            if ($efilter->{$contexttype}->{eq}) {
                $efilter->{$contexttype}->{eq} = xisect $efilter->{$contexttype}->{eq}, \@selected_context_types;
            } else {
                $efilter->{$contexttype}->{eq} = \@selected_context_types;
            }
            
        }

        delete $group_hash->{$_} for @Stat::Const::ANY_PHRASES_GROUP_BY;
    }
    delete $group_hash->{statgoals};
    delete $group_hash->{agoalnum};
    
    my %group_by_ext = (image => 'has_image',
                        position => 'position',
                        # постепенно переходит на page_name в group_by_stat_stream_ext
                        page => 'page',
                        banner => 'banner',
                        phraseid => $phrase_group,
                        contexttype => $contexttype,
                        geo => 'region',
                        tag => 'tags',
                        adgroup => 'adgroup',
                        device_type => 'device_type',
                        age => 'age',
                        gender => 'gender',
                        date => 'date',
                        detailed_device_type => 'detailed_device_type',
                        connection_type => 'connection_type',
                        campaign => 'campaign',
                        retargeting_coef => 'retargeting_coef',
                        banner_image_type => 'banner_image_type',
                        );
    for my $field (keys %$group_hash) {
        if ($group_by_ext{$field}) {
            push @egroup_by, $group_by_ext{$field};
        } else {
            die "Unsupported group by field: $field"
        }
    }
    @egroup_by = uniq @egroup_by;

    # сортировка
    my %order_by_ext = (phrase_id => $phraseid,
                        geo => 'region',
                        tag => 'tags',
                        image => 'has_image',
                        page_name => $self->option_use_page_id ? 'page' : 'page_group',
                        (map { $_ => $phrase_group} @Stat::Const::ANY_PHRASES_GROUP_BY),
                        (map { $_ => $_} qw/date strategy_id banner region_id page position adgroup device_type age gender detailed_device_type connection_type banner_image_type
                                            retargeting_coef coef_ret_cond_id
                                            shows clicks sum ctr av_sum adepth aconv agoalcost agoalnum agoalroi agoalcrr agoalincome fp_shows_avg_pos fp_clicks_avg_pos
                                            pv_adepth pv_aconv pv_agoalcost pv_agoalnum pv_agoalroi pv_agoalcrr pv_agoalincome/),
                       );
    my @order_by = ();
    for my $o (xuniq { $_->{field} } grep {$_->{field} ne 'statgoals'} @{$self->order_by}) {
        if ($order_by_ext{$o->{field}}) {
            push @order_by, {field => $order_by_ext{$o->{field}}, dir => $o->{reverse} ? 'desc' : 'asc'};
        } else {
            warn("unsupported field for order by: $o->{field}");
        }
    }

    return {group_by => \@egroup_by,
            filter => $efilter,
            order_by => \@order_by};
}

=head2 _get_bids_by_bannerid 

    по BannerID и OrderID (для шардинга) возвращает список bid

=cut

sub _get_bids_by_bannerid {
    my ($banner_ids, $order_ids) = @_;

    return get_one_column_sql(PPC(OrderID => $order_ids), ["select bid from banners", where => {BannerID__gt => 0, BannerID => $banner_ids}]);
}

=head2 _translate_magic_page_name_filters($filters, $lang)

    Преобразование фильтров по названию площадки для того чтобы немного сгладить проблемы
    от локализации "магических" площадок (МОЛ присылает нам строку "yandex", мы рисуем её как "Яндекс" или "Yandex")
    Локализация производится в Stat::StreamExtended::_process_row

    Если пользователь фильтрует по названию "декс", нужно отправить в МОЛ фильтр "декс" и "yandex", чтобы строка
    "Яндекс" не пропала.

    С фильтрацией по локализованным площадкам остаются проблемы, которые невозможно исправить на стороне Директа:
    https://st.yandex-team.ru/DIRECT-123355

=cut

sub _translate_magic_page_name_filters {
    my ($filters, $lang) = @_;

    # мапа "локализованное название" -> "магическая строчка"
    my $name_to_magic = Stat::Tools::get_page_name_to_magic_translations($lang);
    my $new_filters = {};
    for my $field (keys %$filters) {
        $new_filters->{$field} = $filters->{$field};

        if ($field eq 'page_name') {
            for my $op (keys %{$filters->{$field}}) {
                my %add;
                for my $value (@{$filters->{$field}{$op}}) {
                    for my $page_name (keys %$name_to_magic) {
                        my $magic_name = $name_to_magic->{$page_name};
                        if (lc($magic_name) eq lc($page_name)) {
                            next;
                        }
                        # если значение фильтра входит в или совпадает с локализованным названием, добавляем к фильтру
                        # дополнительно магическую строчку
                        if (($op eq 'contains' || $op eq 'not_contains') && index(lc($page_name), lc($value)) != -1) {
                            $add{$magic_name} = 1;
                        }
                        if (($op eq 'eq' || $op eq 'ne') && lc($page_name) eq lc($value)) {
                            $add{$magic_name} = 1;
                        }
                    }
                }
                for my $magic_name (keys %add) {
                    if ($op eq 'contains' || $op eq 'eq' || $op eq 'ne') {
                        # "contains":["декс"] превратится в "contains":["декс", "yandex"],
                        # чтобы в выдаче были и неяндекс.рф и "Яндекс"

                        # в операторах "eq"/"ne" в принципе можно было бы заменять "Яндекс" на "yandex",
                        # но хуже от добавления не станет
                        push @{$new_filters->{$field}{$op}}, $magic_name;
                    } elsif ($op eq 'not_contains') {
                        # "not_contains":["декс"] превратится в "not_contains":["декс"], "ne":["yandex"]
                        # чтобы в выдаче не было ни неяндекс.рф ни "Яндекс", но остались zen.yandex.ru и прочие
                        push @{$new_filters->{$field}{'ne'}}, $magic_name;
                    }
                }
            }
        }
    }
    return $new_filters;
}

1;
