package Stat::Fields;

use Direct::Modern;

use Yandex::I18n;

use List::MoreUtils qw/uniq/;

=head2 @COUNTABLE_FIELDS

    Описание countable полей мастера отчетов с указанием различных управляющих атрибутов.

    name        - внутреннее имя поля, которое используется для общения с фронтом Директа.

    bs_name     - имя поля в мастере отчетов

    kind        - размерность значений поля, используется в Stat::Plot

    xls_title   - человекочитаемое название поля для XLS отчетов.
                  Должно быть обёрнуто iget_noop.
                  Для денежных полей должно содержать %s в начале и конце строки, и быть
                  готовым к использованию в первом параметре sprintf

    bs_required_when_filtering  - если выставлено правдивое значение, то при фильтрации по этому полю,
                                  оно будет неявно добавлено в список запрашиваемых полей.
                                  Иногда без этого запрос падает в БК со словами типа
                                  "Unknown identifier: SumVideoFirstQuartileAdjusted"

    floating_point - если выставлено правдивое значение, поле будет округляться
                     до 2-х знаков в Stat::Tools.
                     Иначе поле будет считаться целочисленным.

    extended_precision - если выставлено правдивое значение, поле будет округляться до 4-х знаков.

=cut

my @COUNTABLE_FIELDS = (
    {
        name => 'video_first_quartile',
        bs_name => 'VideoFirstQuartileAdjusted',
        kind => 'points',
        xls_title => iget_noop('Проигрывание 25% ролика'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_midpoint',
        bs_name => 'VideoMidpointAdjusted',
        kind => 'points',
        xls_title => iget_noop('Проигрывание 50% ролика'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_third_quartile',
        bs_name => 'VideoThirdQuartileAdjusted',
        kind => 'points',
        xls_title => iget_noop('Проигрывание 75% ролика'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_complete',
        bs_name => 'VideoCompleteAdjusted',
        kind => 'points',
        xls_title => iget_noop('Проигрывание 100% ролика'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_true_view',
        bs_name => 'VideoTrueViewAdjusted',
        kind => 'points',
        xls_title => iget_noop('Просмотры'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_true_view_rate',
        bs_name => 'VideoTrueViewRate',
        kind => 'percent',
        xls_title => iget_noop('Доля просмотров (%)'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_avg_true_view_cost',
        bs_name => 'AvgVideoTrueViewsCost',
        kind => 'money',
        xls_title => iget_noop('%sСр. цена просмотра ролика%s'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_first_quartile_rate',
        bs_name => 'VideoFirstQuartileRate',
        kind => 'percent',
        xls_title => iget_noop('Доля проигрываний 25% ролика (%)'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_midpoint_rate',
        bs_name => 'VideoMidpointRate',
        kind => 'percent',
        xls_title => iget_noop('Доля проигрываний 50% ролика (%)'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_third_quartile_rate',
        bs_name => 'VideoThirdQuartileRate',
        kind => 'percent',
        xls_title => iget_noop('Доля проигрываний 75% ролика (%)'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_complete_rate',
        bs_name => 'VideoCompleteRate',
        kind => 'percent',
        xls_title => iget_noop('Доля проигрываний 100% ролика (%)'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'cpcv',
        bs_name => 'CPV',
        kind => 'money',
        xls_title => iget_noop('%sСр. цена досмотра 100%% ролика%s'),
        floating_point => 1,
        extended_precision => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_mute',
        bs_name => 'VideoMuteAdjusted',
        kind => 'points',
        xls_title => iget_noop('Клики выключить звук'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'video_unmute',
        bs_name => 'VideoUnmuteAdjusted',
        kind => 'points',
        xls_title => iget_noop('Клики включить звук'),
        bs_required_when_filtering => 1,
    },
    {
        name    => 'uniq_completed_viewers',
        bs_name => 'CompletedReach',
        kind => 'points',
        xls_title => iget_noop('Достроенный охват'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'ad_site_clicks',
        bs_name => 'AdSiteClicks',
        kind => 'points',
        xls_title => iget_noop('Показ пэкшота'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'close_clicks',
        bs_name => 'CloseClicks',
        kind => 'points',
        xls_title => iget_noop('Клики закрытия'),
        bs_required_when_filtering => 1,
    },
    {
        name => 'close_ctr',
        bs_name => 'CloseCtr',
        kind => 'percent',
        xls_title => iget_noop('CTR закрытия (%)'),
        floating_point => 1,
        bs_required_when_filtering => 1,
    },
    {
        name => 'viewable_impressions_mrc',
        bs_name => 'ViewableImpressionsMRC',
        kind => 'points',
        xls_title => iget_noop('Видимые показы, стандарт MRC'),
        floating_point => 0,
        bs_required_when_filtering => 0,
    },
    {
        name => 'nonviewable_impressions_mrc',
        bs_name => 'NonViewableImpressionsMRC',
        kind => 'points',
        xls_title => iget_noop('Невидимые показы, стандарт MRC'),
        floating_point => 0,
        bs_required_when_filtering => 0,
    },
    {
        name => 'undetermined_impressions_mrc',
        bs_name => 'UndeterminedImpressionsMRC',
        kind => 'points',
        xls_title => iget_noop('Неустановленные показы, стандарт MRC'),
        floating_point => 0,
        bs_required_when_filtering => 0,
    },
    {
        name => 'measured_rate_mrc',
        bs_name => 'MeasuredRateMRC',
        kind => 'percent',
        xls_title => iget_noop('Доля измеряемых показов, стандарт MRC (%)'),
        floating_point => 1,
        bs_required_when_filtering => 0,
    },
    {
        name => 'viewable_rate_mrc',
        bs_name => 'ViewableRateMRC',
        kind => 'percent',
        xls_title => iget_noop('Доля видимых показов, стандарт MRC (%)'),
        floating_point => 1,
        bs_required_when_filtering => 0,
    },
    {
        name => 'avg_nshow',
        bs_name => 'AvgNShow',
        kind => 'percent',
        xls_title => iget_noop('Заметность показов (%)'),
        floating_point => 1,        
        bs_required_when_filtering => 1,
    },
    {
        name => 'avg_nshow_complete',
        bs_name => 'AvgNShowComplete',
        kind => 'percent',
        xls_title => iget_noop('Заметность досмотров (%)'),
        floating_point => 1,        
        bs_required_when_filtering => 1,
    },
);

my $common_countable_fields = [qw/shows clicks sum aseslen asesnum asesnumlim agoalnum agoalincome/];
my $campdate_countable_fields = [qw/shows clicks sum aseslen asesnum asesnumlim agoalnum/];
my %countable_fields_by_type = (
    common => $common_countable_fields,
    phrasedate => $common_countable_fields,
    geo => $common_countable_fields,
    pages => $common_countable_fields,
    campdate => $campdate_countable_fields,
    by_agency_clients => $campdate_countable_fields,
    by_agencies => $campdate_countable_fields,
    by_clients => $campdate_countable_fields,
    by_managers => $campdate_countable_fields,
);

=head2 countable_fields_dependencies

Большинство полей мы запрашиваем напрямую из БК, некоторые вычисляем или перевычисляем из других,
некоторые отдаём только вместе, некоторые запрашиваем только для вычисления других и не показываем напрямую.

Правила вычисления описаны в Stat::Tools.

Здесь отмечаем, какие поля нужно реально запрашивать из БК, используя внутрениие названия полей Директа до перевода в названия полей БК.
Название в API определяется в API::Reports::DataRules, в XLS — здесь выше или в StatXLS.

=cut

my %countable_fields_dependencies = (
#   внутренне название                 от чего зависит                            название в API         название в XLS или описание для API
    aconv                           => [qw/agoalnum clicks/],                   # ConversionRate         Конверсия (%)
    ad_site_clicks                  => [qw/ad_site_clicks/],                    # —                      Визиты
    adepth                          => [qw/aseslen asesnum asesnumlim clicks/], # AvgPageviews           Глубина (стр.)
    agoalcost                       => [qw/agoalcost/],                         # CostPerConversion      Цена цели
    agoalcrr                        => [qw/agoalcrr/],                          # —                      Доля рекламных расходов
    agoalincome                     => [qw/agoalincome/],                       # Revenue                Доход
    agoalnum                        => [qw/agoalnum clicks/],                   # Conversions            Конверсии
    agoalroi                        => [qw/agoalincome sum/],                   # GoalsRoi               Рентабельность
    agoals_profit                   => [qw/agoals_profit/],                     # Profit                 Прибыль
    aprgoodmultigoal                => [qw/aprgoodmultigoal/],                  # —                      Сумма ВC
    aprgoodmultigoal_conv_rate      => [qw/aprgoodmultigoal_conv_rate/],        # —                      Доля ВC (%)
    aprgoodmultigoal_cpa            => [qw/aprgoodmultigoal_cpa/],              # —                      Цена ВC
    aseslen                         => [qw/aseslen/],                           # —                      ¯\_(ツ)_/¯
    asesnum                         => [qw/asesnum/],                           # Sessions               «Количество визитов»
    asesnumlim                      => [qw/asesnumlim/],                        # —                      ¯\_(ツ)_/¯
    auction_hits                    => [qw/auction_hits/],                      # AuctionHits            Участие в аукционах
    auction_win_rate                => [qw/auction_win_rate/],                  # AuctionWinRate         Доля выигрышей (%)
    auction_wins                    => [qw/auction_wins/],                      # AuctionWins            Выигрыши в аукционах
    av_day                          => [qw/sum/],                               # —                      Ср. расход за день
    av_day_shows                    => [qw/shows/],                             # —                      —
    av_grouping                     => [qw/sum/],                               # —                      —
    av_sum                          => [qw/clicks sum/],                        # AvgCpc                 Ср. цена клика
    avg_bid                         => [qw/avg_bid/],                           # AvgEffectiveBid        Ср. ставка за клик
    avg_cpm                         => [qw/avg_cpm sum shows/],                 # AvgCpm                 Ср. цена тыс. показов
    avg_cpm_bid                     => [qw/avg_cpm_bid/],                       # AvgEffectiveCpmBid     Ср. ставка за тыс. показов
    avg_nshow                       => [qw/avg_nshow/],                         # AvgNShow               Заметность показов.
    avg_nshow_complete              => [qw/avg_nshow_complete/],                # AvgNShowComplete       Заметность досмотров.
    avg_time_to_conv                => [qw/avg_time_to_conv/],                  # AvgTimeToConversion    Ср. время до конверсии, час.
    avg_view_freq                   => [qw/avg_view_freq/],                     # AvgImpressionFrequency Ср. частота показов
    avg_x                           => [qw/eshows shows/],                      # AvgTrafficVolume       Ср. объём трафика
    bounce_ratio                    => [qw/bounce_ratio/],                      # BounceRate             «Доля отказов»
    bounces                         => [qw/bounces/],                           # Bounces                «Количество отказов»
    clicks                          => [qw/clicks/],                            # Clicks                 Клики
    close_clicks                    => [qw/close_clicks/],                      # —                      Клики закрытия
    close_ctr                       => [qw/close_ctr/],                         # —                      CTR закрытия (%)
    cpcv                            => [qw/cpcv/],                              # —                      Ср. цена досмотра 100% ролика
    ctr                             => [qw/clicks shows/],                      # Ctr                    CTR (%)
    ectr                            => [qw/clicks eshows/],                     # WeightedCtr            wCTR (%)
    eshows                          => [qw/eshows/],                            # WeightedImpressions    Взвешенные показы
    fp_clicks                       => [qw/fp_clicks/],                         # —                      ¯\_(ツ)_/¯
    fp_clicks_avg_pos               => [qw/fp_clicks fp_clicks_pos/],           # AvgClickPosition       Ср. позиция кликов
    fp_clicks_pos                   => [qw/fp_clicks_pos/],                     # —                      ¯\_(ツ)_/¯
    fp_shows                        => [qw/fp_shows/],                          # —                      ¯\_(ツ)_/¯
    fp_shows_avg_pos                => [qw/fp_shows fp_shows_pos/],             # AvgImpressionPosition  Ср. позиция показов
    fp_shows_pos                    => [qw/fp_shows_pos/],                      # —                      ¯\_(ツ)_/¯
    imp_reach_rate                  => [qw/imp_reach_rate/],                    # ImpReachRate           SOV (%)
    imp_to_win_rate                 => [qw/imp_to_win_rate/],                   # ImpToWinRate           Доля видимых показов (%)
    measured_rate_mrc               => [qw/measured_rate_mrc/],                 # —                      Доля измеряемых показов, стандарт MRC (%)
    nonviewable_impressions_mrc     => [qw/nonviewable_impressions_mrc/],       # —                      Невидимые показы, стандарт MRC
    pv_aconv                        => [qw/pv_agoalnum shows/],                 # -                      Post-view Конверсия (%)
    pv_aseslen                      => [qw/pv_aseslen/],                        # —                      ¯\_(ツ)_/¯
    pv_asesnum                      => [qw/pv_asesnum/],                        # -                      ¯\_(ツ)_/¯
    pv_asesnumlim                   => [qw/pv_asesnumlim/],                     # —                      ¯\_(ツ)_/¯
    pv_adepth                       => [qw/pv_aseslen pv_asesnum pv_asesnumlim shows/],  # -             Post-view: Глубина (стр.)
    pv_agoalcost                    => [qw/pv_agoalnum sum/],                   # -                      Post-view: Цена цели
    pv_agoalcrr                     => [qw/pv_agoalcrr/],                       # —                      Post-view: Доля рекламных расходов
    pv_agoalincome                  => [qw/pv_agoalincome/],                    # -                      Post-view: Доход
    pv_agoalnum                     => [qw/pv_agoalnum shows/],                 # -                      Post-view: Конверсии
    pv_agoalroi                     => [qw/pv_agoalincome sum/],                # -                      Post-view: Рентабельность
    pv_agoals_profit                => [qw/pv_agoals_profit/],                  # -                      Post-view: Прибыль    
    pv_aprgoodmultigoal             => [qw/pv_aprgoodmultigoal/],               # —                      Post-view: Сумма ВC
    pv_aprgoodmultigoal_conv_rate   => [qw/pv_aprgoodmultigoal_conv_rate/],     # —                      Post-view: Доля ВC (%)
    pv_aprgoodmultigoal_cpa         => [qw/pv_aprgoodmultigoal_cpa/],           # —                      Post-view: Цена ВC
    pv_avg_time_to_conv             => [qw/pv_avg_time_to_conv/],               # -                      Post-view: Ср. время до конверсии, час.
    pv_bounce_ratio                 => [qw/pv_bounce_ratio/],                   # -                      Post-view: Отказы (%)
    served_impressions              => [qw/served_impressions/],                # —                      Выходы
    shows                           => [qw/shows/],                             # Impressions            Показы
    sum                             => [qw/sum/],                               # Cost                   Расход
    undetermined_impressions_mrc    => [qw/undetermined_impressions_mrc/],      # —                      Неустановленные показы, стандарт MRC
    uniq_completed_viewers          => [qw/uniq_completed_viewers/],            # —                      Достроенный охват
    uniq_viewers                    => [qw/uniq_viewers/],                      # ImpressionReach        Охват
    video_avg_true_view_cost        => [qw/video_avg_true_view_cost/],          # —                      Ср. цена просмотра ролика
    video_complete                  => [qw/video_complete/],                    # —                      Проигрывание 100% ролика
    video_complete_rate             => [qw/video_complete_rate/],               # —                      Доля проигрываний 100% ролика (%)
    video_first_quartile            => [qw/video_first_quartile/],              # —                      Проигрывание 25% ролика
    video_first_quartile_rate       => [qw/video_first_quartile_rate/],         # —                      Доля проигрываний 25% ролика (%)
    video_midpoint                  => [qw/video_midpoint/],                    # —                      Проигрывание 50% ролика
    video_midpoint_rate             => [qw/video_midpoint_rate/],               # —                      Доля проигрываний 50% ролика (%)
    video_mute                      => [qw/video_mute/],                        # —                      Клики выключить звук
    video_third_quartile            => [qw/video_third_quartile/],              # —                      Проигрывание 75% ролика
    video_third_quartile_rate       => [qw/video_third_quartile_rate/],         # —                      Доля проигрываний 75% ролика (%)
    video_true_view                 => [qw/video_true_view/],                   # —                      Просмотры
    video_true_view_rate            => [qw/video_true_view_rate/],              # —                      Доля просмотров (%)
    video_unmute                    => [qw/video_unmute/],                      # —                      Клики включить звук
    viewable_impressions_mrc        => [qw/viewable_impressions_mrc/],          # —                      Видимые показы, стандарт MRC
    viewable_rate_mrc               => [qw/viewable_rate_mrc/],                 # —                      Доля видимых показов, стандарт MRC (%)
    winrate                         => [qw/winrate/],                           # ImpressionShare        Процент полученных показов (%)
);

my @other_money_fields = qw/sum bonus agoalincome pv_agoalincome av_sum agoalcost pv_agoalcost avg_cpm 
                            aprgoodmultigoal_cpa avg_bid avg_cpm_bid agoals_profit pv_agoals_profit/;

=head2 get_all_money_fields

Получить список денежных полей (без суффиксов)

=cut

sub get_all_money_fields {
    return @other_money_fields, get_money_fields();
}

=head2 get_columns_for_plot

    Возвращает все countable поля с указанием размерности
    для использования в Stat::Plot

=cut
sub get_columns_for_plot {
    return map { $_->{name} => $_->{kind} } @COUNTABLE_FIELDS;
}

=head2 get_columns_for_xls

    Возвращает все countable поля для использования в StatXLS

    Параметр start_num обозначает номер, начиная с котороог нужно
    добавлять поля в хешик %columns

=cut
sub get_columns_for_xls {
    my (%O) = @_;
    my $start_num = $O{start_num};

    my %columns;

    for my $f (@COUNTABLE_FIELDS) {
        my $col = [$f->{name}, $f->{xls_title}];
        if ($f->{kind} eq 'money') {
            $col->[2] = { currency => 1 };
        }
        $columns{$start_num} = $col;
        ++$start_num;
    }

    return %columns;
}

=head2 get_extra_countable_fields

    Возвращает список полей, которые нужно добавить в список необязательных
    поддерживаемых полей в мастере отчетов.

=cut
sub get_extra_countable_fields {
    return map { $_->{name} } @COUNTABLE_FIELDS;
}

=head2 get_countable_filter_fields

    Возвращает список полей, которые можно использовать для фильтрации в мастере отчетов.

=cut
sub get_countable_filter_fields {
    return map { $_->{name} } @COUNTABLE_FIELDS;
}

=head2 get_countable_direct2bs

    Возвращает соответствие имен полей в Директе и мастере отчетов.

=cut
sub get_countable_direct2bs {
    return map { $_->{name} => $_->{bs_name} } @COUNTABLE_FIELDS;
}

=head2 get_money_fields

    Возвращает списк всех денежных полей.

=cut
sub get_money_fields {
    return map { $_->{name} } grep { $_->{kind} eq 'money' } @COUNTABLE_FIELDS;
}

=head2 get_countable_integer_fields

    Возвращает список всех целочисленных полей.

=cut
sub get_countable_integer_fields {
    return map { $_->{name} } grep { !$_->{floating_point} } @COUNTABLE_FIELDS;
}

=head2 get_countable_floating_point_fields

    Возвращает список всех полей с нецелочисленными значениями.

=cut
sub get_countable_floating_point_fields {
    return map { $_->{name} } grep { $_->{floating_point} } @COUNTABLE_FIELDS;
}

=head2 get_bs_fields_required_when_filtering

    Возвращает список имен полей в мастере отчетов, которые нужно неявно добавлять
    в список запрашиваемых полей в случае, если мы запрашиваем по ним фильтрацию.

=cut
sub get_bs_fields_required_when_filtering {
    return map { $_->{bs_name} } grep { $_->{bs_required_when_filtering} } @COUNTABLE_FIELDS;
}

=head2 get_countable_extended_precision_fields

    Возвращает список полей, которые нужно выводить с расширенной точностью.

=cut
sub get_countable_extended_precision_fields {
    return map { $_->{name} } grep { $_->{extended_precision} } @COUNTABLE_FIELDS;
}


=head2 get_countable_to_skip_in_camp_stat_xls

    Возвращает список полей, которые не нужно показывать при выгрузке в XLS
    статистики по кампаниям (контроллер showCampStat)

=cut
sub get_countable_to_skip_in_camp_stat_xls {
    return map { $_->{name} } @COUNTABLE_FIELDS;
}

my @multigoal_fields = qw/agoalnum agoalcost agoalroi agoalcrr agoalincome aconv pv_agoalnum pv_agoalcost pv_agoalroi pv_agoalcrr pv_agoalincome pv_aconv/;

=head2 get_multigoal_field_names

Возвращает список имён полей, которые размножаются при запросе мультицелей

=cut

sub get_multigoal_field_names {
    return @multigoal_fields;
}

=head2 get_countable_fields_for_override

По типу отчёта возвращает массив рассчитываемых полей (внутренние названия, до конвертации в формат БК)
для подстановки вместо умолчания при запросе в БК.

=cut

sub get_countable_fields_for_override {
    my $type = shift;
    return $countable_fields_by_type{$type};
}

sub _get_countable_fields_dependencies {
    return \%countable_fields_dependencies;
}

=head2 get_countable_fields_for_override_by_fields_array

По массиву нужных полей возвращает массив полей во внутреннем формате, которые нужно запросить из БК

=cut

sub get_countable_fields_for_override_by_fields_array {
    my $fields = shift;
    my @response;
    for my $field (@$fields) {
        if (exists $countable_fields_dependencies{$field}) {
            push @response, @{$countable_fields_dependencies{$field}};
        }
    }
    return [uniq @response];
}

my @post_view_fields = qw/pv_adepth pv_agoalnum pv_aconv pv_agoalcost pv_agoalroi pv_agoalcrr pv_agoalincome pv_agoals_profit pv_aprgoodmultigoal
                          pv_aprgoodmultigoal_cpa pv_aprgoodmultigoal_conv_rate pv_avg_time_to_conv all_agoalnum/;

=head2 get_post_view_fields_to_skip_in_camp_stat_xls

    Возвращает список Post View полей, которые не нужно показывать при выгрузке в XLS
    статистики по кампаниям (контроллер showCampStat)

=cut
sub get_post_view_fields_to_skip_in_camp_stat_xls {
    return {map { $_ => 1 } @post_view_fields};
}

1;
