package Stat::OrderStatDay;

=pod

=head1 NAME

    Stat::OrderStatDay

=head1 DESCRIPTION

    Вызовы java intapi ручки получения статистики из динамической таблицы в YT

=cut

use Direct::Modern;
use Date::Calc;

use Yandex::DateTime;
use Yandex::TimeCommon;
use Yandex::Validate qw/is_valid_int is_valid_id/;
use Yandex::DateTime qw/now/;

use Settings;
use EnvTools qw/is_sandbox/;
use Stat::Const qw/BS_STAT_START_DATE/;

use JavaIntapi::GetFraudClicks;
use JavaIntapi::GetRoughForecast;
use JavaIntapi::GetOrderSessionsNumForWeekByOrderId qw//;
use JavaIntapi::GetOrdersFirstLastDay;
use JavaIntapi::GetOrdersDaysNum;
use JavaIntapi::GetOrderClicksToday;
use JavaIntapi::GetOrdersSpentToday;
use JavaIntapi::GetOrdersSumSpent;

use Stat::Const qw/:base :enums/;
use List::Util qw/max min maxstr minstr reduce sum/;

=head2 order_sessions_num_for_week

    Возвращает статистику по количеству сессий за последнюю неделю для указанного заказа

    $order_id - id заказа в БК

=cut
sub order_sessions_num_for_week {
    my $order_id = shift;

    return JavaIntapi::GetOrderSessionsNumForWeekByOrderId->new(order_id => $order_id)->call();
}

=head2 order_clicks_today

    Возвращает количество кликов за сегодня для указанного заказа.
    Id заказа в БК может быть равным нулю для кампаний которые не были в БК.
    В этом случае всегда возвращаем ноль кликов.

    $order_id - id заказа в БК

=cut
sub order_clicks_today {
    my $order_id = shift;
    return is_valid_id($order_id) ? JavaIntapi::GetOrderClicksToday->new(order_id => $order_id)->call() : 0;
}

=head2 get_fraud_clicks_total

    Вовращает данные по подаренным и недействительным кликам для набора заказов из java-intapi ручки

=cut
sub get_fraud_clicks_total {
    my ($order_ids, $start_date, $end_date) = @_;

    if (is_sandbox()) {
        return {
            fraud_clicks => 0,
            fraud_gift_clicks => 0,
            fraud_shows_general => 0,
            fraud_shows_sophisticated => 0,
        };
    }

    return JavaIntapi::GetFraudClicks->new(
        order_ids => ($order_ids // []),
        start_date => $start_date,
        end_date => $end_date
    )->call();
}

=head2 get_camp_bsstat_forecast

    Возвращает грубый прогноз трат по кампании как максимум из потраченых денег за последние 7 дней
    Для Песочницы API всегда возвращаются нули

=cut

sub get_camp_bsstat_forecast {
    my ($campaign_ids, $currency) = @_;

    my $result;
    if (is_sandbox()) {
        $result = { map { $_ => 0 } @$campaign_ids };
    } else {
        $result = JavaIntapi::GetRoughForecast->new(
            campaign_ids => $campaign_ids,
            currency => $currency
        )->call();
    }
    return $result;
}

=head2 get_orderids_start_last_day

    Принимает на вход ссылку на масссив order_id - идентификаторов заказов в БК
    Возвращает ссылку на хеш заказ - начальная и конечная даты, на которые есть статистика
    или undef, если статистики по заказу нет:
    {
        123 => {
            firstDate => '2019-01-01',
            lastDate => '2019-02-10'
        },
        456 => undef,
        ...
    }

=cut
sub get_orderids_start_last_day {
    my $order_ids = shift;
    return JavaIntapi::GetOrdersFirstLastDay->new(order_ids => $order_ids)->call();
}

=head2 get_orderids_with_stat_and_dates

    Принимает на вход ссылку на масссив order_id - идентификаторов заказов в БК
    Возвращает три значения:
    $result - хеш с заказами у которых есть статистика
    $min_date_from - минимальная дата начиная с которой есть статистика
    $max_date_to - максимальная дата заканчивая которой есть статистика

=cut
sub get_orderids_with_stat_and_dates {
    my $order_ids = shift;

    my ($result, $min_date_from, $max_date_to);

    my $orderid2date = get_orderids_start_last_day($order_ids);
    for my $oid (keys %$orderid2date) {
        if (defined $orderid2date->{$oid}->{'firstDate'} && defined $orderid2date->{$oid}->{'lastDate'}) {
            $min_date_from = minstr($min_date_from // $orderid2date->{$oid}->{'firstDate'}, $orderid2date->{$oid}->{'firstDate'});
            $max_date_to = maxstr($max_date_to // $orderid2date->{$oid}->{'lastDate'}, $orderid2date->{$oid}->{'lastDate'});
            $result->{$oid} = 1;
        }
    }
    ($min_date_from, $max_date_to) = calc_min_date_from_max_date_to($min_date_from, $max_date_to);

    return ($result, $min_date_from, $max_date_to);
}

=head2 get_orders_days_num

    Возвращает количество дней за которые есть статистика в указанном диапазоне дат

    $date_from, $date_to - начальная и конечная дата в формате YYYY-MM-DD
    $order_ids - ссылка на массив OrderID

=cut
sub get_orders_days_num {
    my ($date_from, $date_to, $order_ids) = @_;

    if (defined $date_from && defined $date_to) {
        return JavaIntapi::GetOrdersDaysNum->new(order_ids => $order_ids, start_date => $date_from, end_date => $date_to)->call() // 0;
    } else {
        return 0;
    }
}

=head2 get_order_days_num

    Возвращает количество дней за которые есть статистика в указанном диапазоне дат для одного заказа
    NB! $order_id может приходить как в виде скалярного значения, так и в виде ссылки на массив с одним элементом

=cut
sub get_order_days_num {
    my ($order_id, $date_from, $date_to) = @_;

    # даты могут приходить в формате YYYYMMDD, приводим их к формату YYYY-MM-DD,
    # чтобы не упасть в get_orders_days_num при вызове java ручки
    # также ограничиваем дату в прошлом тремя годами (BS_STAT_START_DATE) - раньше этой даты статистики все равно нет
    $date_from = maxstr(str_round_day($date_from), str_round_day(BS_STAT_START_DATE));
    $date_to = maxstr(str_round_day($date_to), str_round_day(BS_STAT_START_DATE));

    my $result;
    if (is_sandbox()) {
        $result = 0;
    } else {
        $result = get_orders_days_num($date_from, $date_to, ref($order_id) eq 'ARRAY' ? $order_id : [$order_id]);
    }
    return $result;
}

=head2 get_orders_info

    Возвращает ссылку на хеш со статистикой для списка заказов БК для указанного клиента и валюты
    Параметры:
        $client_id - идентификатор клиента
        $order_ids - ссылка на массив OrderID
        $currency  - трехбуквенный код валюты

    $result = {
        avg_1month - средние траты по заказам за месяц
        avg_3month - средние траты по заказам за три месяца
        first_show_date - дата первого показа
    }

=cut
sub get_orders_info {
    my ($client_id, $order_ids, $currency) = @_;

    my $periods = {
        sum_1month => {from => now()->subtract( months => 1 )->ymd(), to => now()->ymd()},
        sum_3month => {from => now()->subtract( months => 3 )->ymd(), to => now()->ymd()},
    };
    my $orders_sum_spent = JavaIntapi::GetOrdersSumSpent->new(order_ids => $order_ids, periods => $periods, currency => $currency)->call();
    my (undef, $min_date_from) = get_orderids_with_stat_and_dates($order_ids);

    my $result = {
        avg_1month => $orders_sum_spent->{sum_1month},
        avg_3month => $orders_sum_spent->{sum_3month} / 3,
        first_show_date => $min_date_from,
    };

    return $result;
}

=head2 calc_min_date_from_max_date_to

    Вычисляем правильные начальные и конечные даты. Если они не заданы, то подставляется сегодняшняя дата.
    Если не задана одна из дат, то для обоих используется заданная.

=cut
sub calc_min_date_from_max_date_to {
    my ($min_date_from, $max_date_to) = @_;

    $min_date_from ||= $max_date_to ? $max_date_to : sprintf "%04d-%02d-%02d", Date::Calc::Today;
    $max_date_to ||= $min_date_from;

    return ($min_date_from, $max_date_to);
}

=head2 get_order_spent_today

    Вызов get_order_spent_today_multi для одной кампании

=cut

sub get_order_spent_today {
    my ($order_id, %O) = @_;

    die 'no valid OrderID given' unless $order_id && is_valid_int($order_id, 0);
    return get_order_spent_today_multi([$order_id], %O)->{$order_id};
}

=head2 get_order_spent_today_multi

    Получаем сумму, истраченную заказами за сегодня из mysql или yt в зависимости от проперти

=cut

sub get_order_spent_today_multi {
    my ($order_ids, %O) = @_;

    return {}  if !@$order_ids;

    if (!is_sandbox()) {
        return JavaIntapi::GetOrdersSpentToday->new(
            order_ids => $order_ids,
            with_nds => ($O{with_nds} || 0)
        )->call();
    } else {
        my %sb_result = map { $_ => 0 } @$order_ids;
        return \%sb_result;
    }
}

=head2 get_orders_sum_spent

    Получаем сумму, истраченную заказами за переданные периоды из mysql или yt в зависимости от проперти

=cut

sub get_orders_sum_spent {
    my ($order_ids, $periods, $currency) = @_;

    return {}  if !@$order_ids;

    return JavaIntapi::GetOrdersSumSpent->new(
        order_ids => $order_ids,
        periods => $periods,
        currency => $currency
    )->call();
}

1;
