package Stat::Plot;

=pod

    $Id:

=head1 NAME

    Stat::Plot

=head1 DESCRIPTION

    Обработка статистики для выдачи в JSON-формате для построения графиков фронтом

=cut

use Direct::Modern;

use Carp;

use Yandex::HashUtils;

use Stat::Fields;

# Количество элементов, которые будут получать очки за нахождение в топе
my $MAX_POINTED_ITEMS_IN_SLICED_TOP = 100;

# Колонки, которые мы умеем показывать и их размерность
my %columns = (
    shows => 'points',
    eshows => 'points',
    clicks => 'points',
    ctr => 'percent',
    ectr => 'percent',
    sum => 'money',
    av_sum => 'money',
    avg_x => 'points',
    fp_shows_avg_pos => 'points',
    fp_clicks_avg_pos => 'points',
    bounce_ratio => 'percent',
    adepth => 'points',
    aconv => 'percent',
    agoalcost => 'money',
    agoalnum => 'points',
    agoalroi => 'points',
    agoalcrr => 'percent',
    agoalincome => 'money',
    winrate => 'percent',
    aprgoodmultigoal => 'points',
    uniq_viewers => 'points',
    avg_view_freq => 'points',
    avg_cpm => 'money',
    aprgoodmultigoal_cpa => 'money',
    aprgoodmultigoal_conv_rate => 'percent',
    avg_bid => 'money',
    avg_cpm_bid => 'money',
    avg_time_to_conv => 'points',
    agoals_profit => 'money',
    auction_hits => 'points',
    auction_wins => 'points',
    auction_win_rate => 'percent',
    imp_to_win_rate => 'percent',
    imp_reach_rate => 'percent',
    served_impressions => 'points',
    pv_bounce_ratio => 'percent',
    pv_adepth => 'points',
    pv_aconv => 'percent',
    pv_agoalcost => 'money',
    pv_agoalnum => 'points',
    pv_agoalroi => 'points',
    pv_agoalcrr => 'percent',
    pv_agoalincome => 'money',
    pv_agoals_profit => 'money',
    pv_aprgoodmultigoal => 'points',
    pv_aprgoodmultigoal_cpa => 'money',
    pv_aprgoodmultigoal_conv_rate => 'percent',
    pv_avg_time_to_conv => 'points',
    Stat::Fields::get_columns_for_plot,
);

# Группировки, которые мы умеем показывать и ключ, описывающий их в результате. undef - если ключ в результате такой же
my %slices_real_names = (
    tags => undef,
    adgroup => 'adgroup_id',
    banner => 'bid',
    banner_type => undef,
    contextcond => 'phrase',
    contextcond_orig => 'phrase',
    contexttype_orig => 'contexttype',
    retargeting_coef => 'coef_ret_cond_id',
    targettype => undef,
    page_group => undef,
    page => 'page_name',
    ssp => undef,
    region => undef,
    physical_region => undef,
    position => undef,
    click_place => undef,
    banner_image_type => undef,
    image_size => undef,
    device_type => undef,
    gender => undef,
    age => undef,
    campaign_type => undef,
    campaign => 'cid',
    detailed_device_type => undef,
    connection_type => undef,
    deal => 'deal_export_id',
    content_targeting => 'content_targeting',
);

# Ключ группировки в результате и список полей, которые нужно добавить к выдаче
my %slice_additional_fields = (
    adgroup_id => ['adgroup_name'],
    bid => ['title'],
    phrase => ['contexttype', 'contexttype_orig'],
    page_group => ['page_domain', 'page_name', 'page_type'],
    page_name => ['page_domain', 'page_type'],
    region => ['region_name'],
    physical_region => ['physical_region_name'],
    cid => ['camp_name'],
    coef_ret_cond_id => ['coef_ret_cond_name'],
    tags => ['tag_names'],
    deal_export_id => ['deal_name'],
    client_id => ['internal_product_name'],
    content_targeting => ['content_targeting_name'],
);

# Поля с описанием периодов, за которые дана статистика. Добавляем к ответу всегда
my @additional_date_fields = qw/stat_date date stat_date_end/;
# Основной ключ для разбора статистики по умолчанию
my $PRIMARY_KEY = 'stat_date';

my %slices_handlers = (
    tags => sub {
        my ($tags) = @_;

        return join ', ', sort @$tags;
    },
);


=head2 get_all_used_dimensions($columns_list)

    Получить список измерений, которые используются в переданных функции колонках

=cut

sub get_all_used_dimensions {
    my ($used_columns) = @_;

    my %used_dimensions;
    for my $column (@$used_columns) {
        die "Unknown column $column" unless exists $columns{$column};

        $used_dimensions{$columns{$column}} = undef;
    }
    return [keys %used_dimensions];
}

sub _sort_func {
    my ($sa, $sb, $column, $slice_key, $totals) = @_;

    my $sa_key = $sa->{$slice_key};
    my $sb_key = $sb->{$slice_key};

    return $sb->{$column} == $sa->{$column} ?
        ($totals->{$sb_key} == $totals->{$sa_key} ?
            $sb_key cmp $sa_key :
            $totals->{$sb_key} <=> $totals->{$sa_key}) :
        $sb->{$column} <=> $sa->{$column};
}

=head2 _get_slices_order($slice_key, $column, $date2slices)

    Получить список значений срезов отсортированный в порядке убывания значений колонки $column.

    Принцип сортировки следующий: чем больше значение колонки попадает в топ-пять в каждом периоде, тем выше оно будет в
    общем списке. Если количество попаданий для двух срезов равное, сравнивается абсолютное суммарное значение колонки в
    каждом из срезов

=cut

sub _get_slices_order {
    my ($slice_key, $column, $date2slices) = @_;

    return [] if $slice_key eq $PRIMARY_KEY;

    my %total_score;
    for my $stat_date (keys %$date2slices) {
        for my $val (values %{$date2slices->{$stat_date}}) {
            $total_score{$val->{$slice_key}} += $val->{$column};
        }
    }

    my %points_score;
    for my $stat_date (keys %$date2slices) {
        my @sortded_date = map {$_->{$slice_key}}
            sort {_sort_func($a, $b, $column, $slice_key, \%total_score)} values %{$date2slices->{$stat_date}};
        my $i = 0;
        for my $val (@sortded_date) {
            $points_score{$val} += $MAX_POINTED_ITEMS_IN_SLICED_TOP - $i > 0 ? $MAX_POINTED_ITEMS_IN_SLICED_TOP - $i : 0;
            $i++;
        }
    }

    my @ordered_vals = sort {$points_score{$b} == $points_score{$a} ?
        ($total_score{$b} == $total_score{$a} ? $b cmp $a : $total_score{$b} <=> $total_score{$a}) :
        $points_score{$b} <=> $points_score{$a}} keys %points_score;

    return \@ordered_vals;
}

=head2 generate_plot_data($stat, $report_options)

    Сгенерировать хеш с данными для построения графика

=cut

sub generate_plot_data {
    my ($stat, $report_options) = @_;

    my ($plot_type, $slice_key);
    if (scalar @{$report_options->{group_by}} == 1) {
        $slice_key = $report_options->{group_by}->[0];
        $slice_key = $slices_real_names{$slice_key} ? $slices_real_names{$slice_key} : $slice_key;
        $plot_type = 'slices';
    } elsif (scalar @{$report_options->{group_by}} == 0) {
        $slice_key = $PRIMARY_KEY;
        $plot_type = 'columns';
    } else {
        croak "Too much slices";
    }
    my @used_columns = grep {exists $columns{$_}} @{$report_options->{columns}};
    my $additional_fields = $slice_additional_fields{$slice_key} // [];

    my %date2slices;
    my %dates_descriptions;
    for my $item (@{$stat->{data_array}}) {
        my %row = map {$_ => $item->{$_}} $slice_key, @used_columns, @$additional_fields;
        if (exists $slices_handlers{$slice_key}) {
            $row{$slice_key} = $slices_handlers{$slice_key}->($row{$slice_key});
        }

        $date2slices{$item->{$PRIMARY_KEY}}->{$row{$slice_key}} = \%row;
        if (! exists $dates_descriptions{$item->{$PRIMARY_KEY}}) {
            $dates_descriptions{$item->{$PRIMARY_KEY}} = hash_cut($item, @additional_date_fields);
        }
    }

    my $slice_values_order = _get_slices_order($slice_key, $report_options->{columns}->[0], \%date2slices);

    my @data;
    for my $stat_date (sort keys %date2slices) {
        my %slices;
        if ($slice_key eq $PRIMARY_KEY) {
            %slices = %{$date2slices{$stat_date}->{$stat_date}};
        } else {
            for my $val (@$slice_values_order) {
                if ($date2slices{$stat_date}->{$val}) {
                    $slices{$val} = $date2slices{$stat_date}->{$val};
                }
            }
        }

        my $row = $dates_descriptions{$stat_date};
        $row->{slices} = \%slices;
        push @data, $row;
    }

    return {
        type => $plot_type,
        slice_values => $slice_values_order,
        stat => \@data,
        date_from => $report_options->{date_from},
        date_to => $report_options->{date_to},
    };
}

1;
